// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { getLevelByName, getLevelName, LogLevels } from "./levels.ts";
export class LoggerConfig {
  level;
  handlers;
}
/**
 * An object that encapsulates provided message and arguments as well some
 * metadata that can be later used when formatting a message.
 */ export class LogRecord {
  msg;
  #args;
  #datetime;
  level;
  levelName;
  loggerName;
  constructor(options){
    this.msg = options.msg;
    this.#args = [
      ...options.args
    ];
    this.level = options.level;
    this.loggerName = options.loggerName;
    this.#datetime = new Date();
    this.levelName = getLevelName(options.level);
  }
  get args() {
    return [
      ...this.#args
    ];
  }
  get datetime() {
    return new Date(this.#datetime.getTime());
  }
}
export class Logger {
  #level;
  handlers;
  #loggerName;
  constructor(loggerName, levelName, options = {}){
    this.#loggerName = loggerName;
    this.#level = getLevelByName(levelName);
    this.handlers = options.handlers || [];
  }
  /** Use this to retrieve the current numeric log level. */ get level() {
    return this.#level;
  }
  /** Use this to set the numeric log level. */ set level(level) {
    try {
      this.#level = getLevelByName(getLevelName(level));
    } catch (_) {
      throw new TypeError(`Invalid log level: ${level}`);
    }
  }
  get levelName() {
    return getLevelName(this.#level);
  }
  set levelName(levelName) {
    this.#level = getLevelByName(levelName);
  }
  get loggerName() {
    return this.#loggerName;
  }
  /**
   * If the level of the logger is greater than the level to log, then nothing
   * is logged, otherwise a log record is passed to each log handler.  `msg` data
   * passed in is returned.  If a function is passed in, it is only evaluated
   * if the msg will be logged and the return value will be the result of the
   * function, not the function itself, unless the function isn't called, in which
   * case undefined is returned.  All types are coerced to strings for logging.
   */ #log(level, msg, ...args) {
    if (this.level > level) {
      return msg instanceof Function ? undefined : msg;
    }
    let fnResult;
    let logMessage;
    if (msg instanceof Function) {
      fnResult = msg();
      logMessage = this.asString(fnResult);
    } else {
      logMessage = this.asString(msg);
    }
    const record = new LogRecord({
      msg: logMessage,
      args: args,
      level: level,
      loggerName: this.loggerName
    });
    this.handlers.forEach((handler)=>{
      handler.handle(record);
    });
    return msg instanceof Function ? fnResult : msg;
  }
  asString(data, isProperty = false) {
    if (typeof data === "string") {
      if (isProperty) return `"${data}"`;
      return data;
    } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
      return String(data);
    } else if (data instanceof Error) {
      return data.stack;
    } else if (typeof data === "object") {
      return `{${Object.entries(data).map(([k, v])=>`"${k}":${this.asString(v, true)}`).join(",")}}`;
    }
    return "undefined";
  }
  debug(msg, ...args) {
    return this.#log(LogLevels.DEBUG, msg, ...args);
  }
  info(msg, ...args) {
    return this.#log(LogLevels.INFO, msg, ...args);
  }
  warn(msg, ...args) {
    return this.#log(LogLevels.WARN, msg, ...args);
  }
  error(msg, ...args) {
    return this.#log(LogLevels.ERROR, msg, ...args);
  }
  critical(msg, ...args) {
    return this.#log(LogLevels.CRITICAL, msg, ...args);
  }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjIxNy4wL2xvZy9sb2dnZXIudHMiXSwic291cmNlc0NvbnRlbnQiOlsiLy8gQ29weXJpZ2h0IDIwMTgtMjAyNCB0aGUgRGVubyBhdXRob3JzLiBBbGwgcmlnaHRzIHJlc2VydmVkLiBNSVQgbGljZW5zZS5cbmltcG9ydCB7IGdldExldmVsQnlOYW1lLCBnZXRMZXZlbE5hbWUsIExvZ0xldmVscyB9IGZyb20gXCIuL2xldmVscy50c1wiO1xuaW1wb3J0IHR5cGUgeyBMZXZlbE5hbWUsIExvZ0xldmVsIH0gZnJvbSBcIi4vbGV2ZWxzLnRzXCI7XG5pbXBvcnQgdHlwZSB7IEJhc2VIYW5kbGVyIH0gZnJvbSBcIi4vYmFzZV9oYW5kbGVyLnRzXCI7XG5cbi8vIGRlbm8tbGludC1pZ25vcmUgbm8tZXhwbGljaXQtYW55XG5leHBvcnQgdHlwZSBHZW5lcmljRnVuY3Rpb24gPSAoLi4uYXJnczogYW55W10pID0+IGFueTtcblxuZXhwb3J0IGludGVyZmFjZSBMb2dSZWNvcmRPcHRpb25zIHtcbiAgbXNnOiBzdHJpbmc7XG4gIGFyZ3M6IHVua25vd25bXTtcbiAgbGV2ZWw6IExvZ0xldmVsO1xuICBsb2dnZXJOYW1lOiBzdHJpbmc7XG59XG5cbmV4cG9ydCBjbGFzcyBMb2dnZXJDb25maWcge1xuICBsZXZlbD86IExldmVsTmFtZTtcbiAgaGFuZGxlcnM/OiBzdHJpbmdbXTtcbn1cblxuZXhwb3J0IGludGVyZmFjZSBMb2dDb25maWcge1xuICBoYW5kbGVycz86IHtcbiAgICBbbmFtZTogc3RyaW5nXTogQmFzZUhhbmRsZXI7XG4gIH07XG4gIGxvZ2dlcnM/OiB7XG4gICAgW25hbWU6IHN0cmluZ106IExvZ2dlckNvbmZpZztcbiAgfTtcbn1cblxuLyoqXG4gKiBBbiBvYmplY3QgdGhhdCBlbmNhcHN1bGF0ZXMgcHJvdmlkZWQgbWVzc2FnZSBhbmQgYXJndW1lbnRzIGFzIHdlbGwgc29tZVxuICogbWV0YWRhdGEgdGhhdCBjYW4gYmUgbGF0ZXIgdXNlZCB3aGVuIGZvcm1hdHRpbmcgYSBtZXNzYWdlLlxuICovXG5leHBvcnQgY2xhc3MgTG9nUmVjb3JkIHtcbiAgcmVhZG9ubHkgbXNnOiBzdHJpbmc7XG4gICNhcmdzOiB1bmtub3duW107XG4gICNkYXRldGltZTogRGF0ZTtcbiAgcmVhZG9ubHkgbGV2ZWw6IG51bWJlcjtcbiAgcmVhZG9ubHkgbGV2ZWxOYW1lOiBzdHJpbmc7XG4gIHJlYWRvbmx5IGxvZ2dlck5hbWU6IHN0cmluZztcblxuICBjb25zdHJ1Y3RvcihvcHRpb25zOiBMb2dSZWNvcmRPcHRpb25zKSB7XG4gICAgdGhpcy5tc2cgPSBvcHRpb25zLm1zZztcbiAgICB0aGlzLiNhcmdzID0gWy4uLm9wdGlvbnMuYXJnc107XG4gICAgdGhpcy5sZXZlbCA9IG9wdGlvbnMubGV2ZWw7XG4gICAgdGhpcy5sb2dnZXJOYW1lID0gb3B0aW9ucy5sb2dnZXJOYW1lO1xuICAgIHRoaXMuI2RhdGV0aW1lID0gbmV3IERhdGUoKTtcbiAgICB0aGlzLmxldmVsTmFtZSA9IGdldExldmVsTmFtZShvcHRpb25zLmxldmVsKTtcbiAgfVxuICBnZXQgYXJncygpOiB1bmtub3duW10ge1xuICAgIHJldHVybiBbLi4udGhpcy4jYXJnc107XG4gIH1cbiAgZ2V0IGRhdGV0aW1lKCk6IERhdGUge1xuICAgIHJldHVybiBuZXcgRGF0ZSh0aGlzLiNkYXRldGltZS5nZXRUaW1lKCkpO1xuICB9XG59XG5cbmV4cG9ydCBpbnRlcmZhY2UgTG9nZ2VyT3B0aW9ucyB7XG4gIGhhbmRsZXJzPzogQmFzZUhhbmRsZXJbXTtcbn1cblxuZXhwb3J0IGNsYXNzIExvZ2dlciB7XG4gICNsZXZlbDogTG9nTGV2ZWw7XG4gIGhhbmRsZXJzOiBCYXNlSGFuZGxlcltdO1xuICByZWFkb25seSAjbG9nZ2VyTmFtZTogc3RyaW5nO1xuXG4gIGNvbnN0cnVjdG9yKFxuICAgIGxvZ2dlck5hbWU6IHN0cmluZyxcbiAgICBsZXZlbE5hbWU6IExldmVsTmFtZSxcbiAgICBvcHRpb25zOiBMb2dnZXJPcHRpb25zID0ge30sXG4gICkge1xuICAgIHRoaXMuI2xvZ2dlck5hbWUgPSBsb2dnZXJOYW1lO1xuICAgIHRoaXMuI2xldmVsID0gZ2V0TGV2ZWxCeU5hbWUobGV2ZWxOYW1lKTtcbiAgICB0aGlzLmhhbmRsZXJzID0gb3B0aW9ucy5oYW5kbGVycyB8fCBbXTtcbiAgfVxuXG4gIC8qKiBVc2UgdGhpcyB0byByZXRyaWV2ZSB0aGUgY3VycmVudCBudW1lcmljIGxvZyBsZXZlbC4gKi9cbiAgZ2V0IGxldmVsKCk6IExvZ0xldmVsIHtcbiAgICByZXR1cm4gdGhpcy4jbGV2ZWw7XG4gIH1cblxuICAvKiogVXNlIHRoaXMgdG8gc2V0IHRoZSBudW1lcmljIGxvZyBsZXZlbC4gKi9cbiAgc2V0IGxldmVsKGxldmVsOiBMb2dMZXZlbCkge1xuICAgIHRyeSB7XG4gICAgICB0aGlzLiNsZXZlbCA9IGdldExldmVsQnlOYW1lKGdldExldmVsTmFtZShsZXZlbCkpO1xuICAgIH0gY2F0Y2ggKF8pIHtcbiAgICAgIHRocm93IG5ldyBUeXBlRXJyb3IoYEludmFsaWQgbG9nIGxldmVsOiAke2xldmVsfWApO1xuICAgIH1cbiAgfVxuXG4gIGdldCBsZXZlbE5hbWUoKTogTGV2ZWxOYW1lIHtcbiAgICByZXR1cm4gZ2V0TGV2ZWxOYW1lKHRoaXMuI2xldmVsKTtcbiAgfVxuICBzZXQgbGV2ZWxOYW1lKGxldmVsTmFtZTogTGV2ZWxOYW1lKSB7XG4gICAgdGhpcy4jbGV2ZWwgPSBnZXRMZXZlbEJ5TmFtZShsZXZlbE5hbWUpO1xuICB9XG5cbiAgZ2V0IGxvZ2dlck5hbWUoKTogc3RyaW5nIHtcbiAgICByZXR1cm4gdGhpcy4jbG9nZ2VyTmFtZTtcbiAgfVxuXG4gIC8qKlxuICAgKiBJZiB0aGUgbGV2ZWwgb2YgdGhlIGxvZ2dlciBpcyBncmVhdGVyIHRoYW4gdGhlIGxldmVsIHRvIGxvZywgdGhlbiBub3RoaW5nXG4gICAqIGlzIGxvZ2dlZCwgb3RoZXJ3aXNlIGEgbG9nIHJlY29yZCBpcyBwYXNzZWQgdG8gZWFjaCBsb2cgaGFuZGxlci4gIGBtc2dgIGRhdGFcbiAgICogcGFzc2VkIGluIGlzIHJldHVybmVkLiAgSWYgYSBmdW5jdGlvbiBpcyBwYXNzZWQgaW4sIGl0IGlzIG9ubHkgZXZhbHVhdGVkXG4gICAqIGlmIHRoZSBtc2cgd2lsbCBiZSBsb2dnZWQgYW5kIHRoZSByZXR1cm4gdmFsdWUgd2lsbCBiZSB0aGUgcmVzdWx0IG9mIHRoZVxuICAgKiBmdW5jdGlvbiwgbm90IHRoZSBmdW5jdGlvbiBpdHNlbGYsIHVubGVzcyB0aGUgZnVuY3Rpb24gaXNuJ3QgY2FsbGVkLCBpbiB3aGljaFxuICAgKiBjYXNlIHVuZGVmaW5lZCBpcyByZXR1cm5lZC4gIEFsbCB0eXBlcyBhcmUgY29lcmNlZCB0byBzdHJpbmdzIGZvciBsb2dnaW5nLlxuICAgKi9cbiAgI2xvZzxUPihcbiAgICBsZXZlbDogTG9nTGV2ZWwsXG4gICAgbXNnOiAoVCBleHRlbmRzIEdlbmVyaWNGdW5jdGlvbiA/IG5ldmVyIDogVCkgfCAoKCkgPT4gVCksXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQgfCB1bmRlZmluZWQge1xuICAgIGlmICh0aGlzLmxldmVsID4gbGV2ZWwpIHtcbiAgICAgIHJldHVybiBtc2cgaW5zdGFuY2VvZiBGdW5jdGlvbiA/IHVuZGVmaW5lZCA6IG1zZztcbiAgICB9XG5cbiAgICBsZXQgZm5SZXN1bHQ6IFQgfCB1bmRlZmluZWQ7XG4gICAgbGV0IGxvZ01lc3NhZ2U6IHN0cmluZztcbiAgICBpZiAobXNnIGluc3RhbmNlb2YgRnVuY3Rpb24pIHtcbiAgICAgIGZuUmVzdWx0ID0gbXNnKCk7XG4gICAgICBsb2dNZXNzYWdlID0gdGhpcy5hc1N0cmluZyhmblJlc3VsdCk7XG4gICAgfSBlbHNlIHtcbiAgICAgIGxvZ01lc3NhZ2UgPSB0aGlzLmFzU3RyaW5nKG1zZyk7XG4gICAgfVxuICAgIGNvbnN0IHJlY29yZDogTG9nUmVjb3JkID0gbmV3IExvZ1JlY29yZCh7XG4gICAgICBtc2c6IGxvZ01lc3NhZ2UsXG4gICAgICBhcmdzOiBhcmdzLFxuICAgICAgbGV2ZWw6IGxldmVsLFxuICAgICAgbG9nZ2VyTmFtZTogdGhpcy5sb2dnZXJOYW1lLFxuICAgIH0pO1xuXG4gICAgdGhpcy5oYW5kbGVycy5mb3JFYWNoKChoYW5kbGVyKSA9PiB7XG4gICAgICBoYW5kbGVyLmhhbmRsZShyZWNvcmQpO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuIG1zZyBpbnN0YW5jZW9mIEZ1bmN0aW9uID8gZm5SZXN1bHQgOiBtc2c7XG4gIH1cblxuICBhc1N0cmluZyhkYXRhOiB1bmtub3duLCBpc1Byb3BlcnR5ID0gZmFsc2UpOiBzdHJpbmcge1xuICAgIGlmICh0eXBlb2YgZGF0YSA9PT0gXCJzdHJpbmdcIikge1xuICAgICAgaWYgKGlzUHJvcGVydHkpIHJldHVybiBgXCIke2RhdGF9XCJgO1xuICAgICAgcmV0dXJuIGRhdGE7XG4gICAgfSBlbHNlIGlmIChcbiAgICAgIGRhdGEgPT09IG51bGwgfHxcbiAgICAgIHR5cGVvZiBkYXRhID09PSBcIm51bWJlclwiIHx8XG4gICAgICB0eXBlb2YgZGF0YSA9PT0gXCJiaWdpbnRcIiB8fFxuICAgICAgdHlwZW9mIGRhdGEgPT09IFwiYm9vbGVhblwiIHx8XG4gICAgICB0eXBlb2YgZGF0YSA9PT0gXCJ1bmRlZmluZWRcIiB8fFxuICAgICAgdHlwZW9mIGRhdGEgPT09IFwic3ltYm9sXCJcbiAgICApIHtcbiAgICAgIHJldHVybiBTdHJpbmcoZGF0YSk7XG4gICAgfSBlbHNlIGlmIChkYXRhIGluc3RhbmNlb2YgRXJyb3IpIHtcbiAgICAgIHJldHVybiBkYXRhLnN0YWNrITtcbiAgICB9IGVsc2UgaWYgKHR5cGVvZiBkYXRhID09PSBcIm9iamVjdFwiKSB7XG4gICAgICByZXR1cm4gYHske1xuICAgICAgICBPYmplY3QuZW50cmllcyhkYXRhKVxuICAgICAgICAgIC5tYXAoKFtrLCB2XSkgPT4gYFwiJHtrfVwiOiR7dGhpcy5hc1N0cmluZyh2LCB0cnVlKX1gKVxuICAgICAgICAgIC5qb2luKFwiLFwiKVxuICAgICAgfX1gO1xuICAgIH1cbiAgICByZXR1cm4gXCJ1bmRlZmluZWRcIjtcbiAgfVxuXG4gIGRlYnVnPFQ+KG1zZzogKCkgPT4gVCwgLi4uYXJnczogdW5rbm93bltdKTogVCB8IHVuZGVmaW5lZDtcbiAgZGVidWc8VD4obXNnOiBUIGV4dGVuZHMgR2VuZXJpY0Z1bmN0aW9uID8gbmV2ZXIgOiBULCAuLi5hcmdzOiB1bmtub3duW10pOiBUO1xuICBkZWJ1ZzxUPihcbiAgICBtc2c6IChUIGV4dGVuZHMgR2VuZXJpY0Z1bmN0aW9uID8gbmV2ZXIgOiBUKSB8ICgoKSA9PiBUKSxcbiAgICAuLi5hcmdzOiB1bmtub3duW11cbiAgKTogVCB8IHVuZGVmaW5lZCB7XG4gICAgcmV0dXJuIHRoaXMuI2xvZyhMb2dMZXZlbHMuREVCVUcsIG1zZywgLi4uYXJncyk7XG4gIH1cblxuICBpbmZvPFQ+KG1zZzogKCkgPT4gVCwgLi4uYXJnczogdW5rbm93bltdKTogVCB8IHVuZGVmaW5lZDtcbiAgaW5mbzxUPihtc2c6IFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQsIC4uLmFyZ3M6IHVua25vd25bXSk6IFQ7XG4gIGluZm88VD4oXG4gICAgbXNnOiAoVCBleHRlbmRzIEdlbmVyaWNGdW5jdGlvbiA/IG5ldmVyIDogVCkgfCAoKCkgPT4gVCksXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLiNsb2coTG9nTGV2ZWxzLklORk8sIG1zZywgLi4uYXJncyk7XG4gIH1cblxuICB3YXJuPFQ+KG1zZzogKCkgPT4gVCwgLi4uYXJnczogdW5rbm93bltdKTogVCB8IHVuZGVmaW5lZDtcbiAgd2FybjxUPihtc2c6IFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQsIC4uLmFyZ3M6IHVua25vd25bXSk6IFQ7XG4gIHdhcm48VD4oXG4gICAgbXNnOiAoVCBleHRlbmRzIEdlbmVyaWNGdW5jdGlvbiA/IG5ldmVyIDogVCkgfCAoKCkgPT4gVCksXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLiNsb2coTG9nTGV2ZWxzLldBUk4sIG1zZywgLi4uYXJncyk7XG4gIH1cblxuICBlcnJvcjxUPihtc2c6ICgpID0+IFQsIC4uLmFyZ3M6IHVua25vd25bXSk6IFQgfCB1bmRlZmluZWQ7XG4gIGVycm9yPFQ+KG1zZzogVCBleHRlbmRzIEdlbmVyaWNGdW5jdGlvbiA/IG5ldmVyIDogVCwgLi4uYXJnczogdW5rbm93bltdKTogVDtcbiAgZXJyb3I8VD4oXG4gICAgbXNnOiAoVCBleHRlbmRzIEdlbmVyaWNGdW5jdGlvbiA/IG5ldmVyIDogVCkgfCAoKCkgPT4gVCksXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQgfCB1bmRlZmluZWQge1xuICAgIHJldHVybiB0aGlzLiNsb2coTG9nTGV2ZWxzLkVSUk9SLCBtc2csIC4uLmFyZ3MpO1xuICB9XG5cbiAgY3JpdGljYWw8VD4obXNnOiAoKSA9PiBULCAuLi5hcmdzOiB1bmtub3duW10pOiBUIHwgdW5kZWZpbmVkO1xuICBjcml0aWNhbDxUPihcbiAgICBtc2c6IFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQsXG4gICAgLi4uYXJnczogdW5rbm93bltdXG4gICk6IFQ7XG4gIGNyaXRpY2FsPFQ+KFxuICAgIG1zZzogKFQgZXh0ZW5kcyBHZW5lcmljRnVuY3Rpb24gPyBuZXZlciA6IFQpIHwgKCgpID0+IFQpLFxuICAgIC4uLmFyZ3M6IHVua25vd25bXVxuICApOiBUIHwgdW5kZWZpbmVkIHtcbiAgICByZXR1cm4gdGhpcy4jbG9nKExvZ0xldmVscy5DUklUSUNBTCwgbXNnLCAuLi5hcmdzKTtcbiAgfVxufVxuIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiJBQUFBLDBFQUEwRTtBQUMxRSxTQUFTLGNBQWMsRUFBRSxZQUFZLEVBQUUsU0FBUyxRQUFRLGNBQWM7QUFjdEUsT0FBTyxNQUFNO0VBQ1gsTUFBa0I7RUFDbEIsU0FBb0I7QUFDdEI7QUFXQTs7O0NBR0MsR0FDRCxPQUFPLE1BQU07RUFDRixJQUFZO0VBQ3JCLENBQUMsSUFBSSxDQUFZO0VBQ2pCLENBQUMsUUFBUSxDQUFPO0VBQ1AsTUFBYztFQUNkLFVBQWtCO0VBQ2xCLFdBQW1CO0VBRTVCLFlBQVksT0FBeUIsQ0FBRTtJQUNyQyxJQUFJLENBQUMsR0FBRyxHQUFHLFFBQVEsR0FBRztJQUN0QixJQUFJLENBQUMsQ0FBQyxJQUFJLEdBQUc7U0FBSSxRQUFRLElBQUk7S0FBQztJQUM5QixJQUFJLENBQUMsS0FBSyxHQUFHLFFBQVEsS0FBSztJQUMxQixJQUFJLENBQUMsVUFBVSxHQUFHLFFBQVEsVUFBVTtJQUNwQyxJQUFJLENBQUMsQ0FBQyxRQUFRLEdBQUcsSUFBSTtJQUNyQixJQUFJLENBQUMsU0FBUyxHQUFHLGFBQWEsUUFBUSxLQUFLO0VBQzdDO0VBQ0EsSUFBSSxPQUFrQjtJQUNwQixPQUFPO1NBQUksSUFBSSxDQUFDLENBQUMsSUFBSTtLQUFDO0VBQ3hCO0VBQ0EsSUFBSSxXQUFpQjtJQUNuQixPQUFPLElBQUksS0FBSyxJQUFJLENBQUMsQ0FBQyxRQUFRLENBQUMsT0FBTztFQUN4QztBQUNGO0FBTUEsT0FBTyxNQUFNO0VBQ1gsQ0FBQyxLQUFLLENBQVc7RUFDakIsU0FBd0I7RUFDZixDQUFDLFVBQVUsQ0FBUztFQUU3QixZQUNFLFVBQWtCLEVBQ2xCLFNBQW9CLEVBQ3BCLFVBQXlCLENBQUMsQ0FBQyxDQUMzQjtJQUNBLElBQUksQ0FBQyxDQUFDLFVBQVUsR0FBRztJQUNuQixJQUFJLENBQUMsQ0FBQyxLQUFLLEdBQUcsZUFBZTtJQUM3QixJQUFJLENBQUMsUUFBUSxHQUFHLFFBQVEsUUFBUSxJQUFJLEVBQUU7RUFDeEM7RUFFQSx3REFBd0QsR0FDeEQsSUFBSSxRQUFrQjtJQUNwQixPQUFPLElBQUksQ0FBQyxDQUFDLEtBQUs7RUFDcEI7RUFFQSwyQ0FBMkMsR0FDM0MsSUFBSSxNQUFNLEtBQWUsRUFBRTtJQUN6QixJQUFJO01BQ0YsSUFBSSxDQUFDLENBQUMsS0FBSyxHQUFHLGVBQWUsYUFBYTtJQUM1QyxFQUFFLE9BQU8sR0FBRztNQUNWLE1BQU0sSUFBSSxVQUFVLENBQUMsbUJBQW1CLEVBQUUsTUFBTSxDQUFDO0lBQ25EO0VBQ0Y7RUFFQSxJQUFJLFlBQXVCO0lBQ3pCLE9BQU8sYUFBYSxJQUFJLENBQUMsQ0FBQyxLQUFLO0VBQ2pDO0VBQ0EsSUFBSSxVQUFVLFNBQW9CLEVBQUU7SUFDbEMsSUFBSSxDQUFDLENBQUMsS0FBSyxHQUFHLGVBQWU7RUFDL0I7RUFFQSxJQUFJLGFBQXFCO0lBQ3ZCLE9BQU8sSUFBSSxDQUFDLENBQUMsVUFBVTtFQUN6QjtFQUVBOzs7Ozs7O0dBT0MsR0FDRCxDQUFDLEdBQUcsQ0FDRixLQUFlLEVBQ2YsR0FBd0QsRUFDeEQsR0FBRyxJQUFlO0lBRWxCLElBQUksSUFBSSxDQUFDLEtBQUssR0FBRyxPQUFPO01BQ3RCLE9BQU8sZUFBZSxXQUFXLFlBQVk7SUFDL0M7SUFFQSxJQUFJO0lBQ0osSUFBSTtJQUNKLElBQUksZUFBZSxVQUFVO01BQzNCLFdBQVc7TUFDWCxhQUFhLElBQUksQ0FBQyxRQUFRLENBQUM7SUFDN0IsT0FBTztNQUNMLGFBQWEsSUFBSSxDQUFDLFFBQVEsQ0FBQztJQUM3QjtJQUNBLE1BQU0sU0FBb0IsSUFBSSxVQUFVO01BQ3RDLEtBQUs7TUFDTCxNQUFNO01BQ04sT0FBTztNQUNQLFlBQVksSUFBSSxDQUFDLFVBQVU7SUFDN0I7SUFFQSxJQUFJLENBQUMsUUFBUSxDQUFDLE9BQU8sQ0FBQyxDQUFDO01BQ3JCLFFBQVEsTUFBTSxDQUFDO0lBQ2pCO0lBRUEsT0FBTyxlQUFlLFdBQVcsV0FBVztFQUM5QztFQUVBLFNBQVMsSUFBYSxFQUFFLGFBQWEsS0FBSyxFQUFVO0lBQ2xELElBQUksT0FBTyxTQUFTLFVBQVU7TUFDNUIsSUFBSSxZQUFZLE9BQU8sQ0FBQyxDQUFDLEVBQUUsS0FBSyxDQUFDLENBQUM7TUFDbEMsT0FBTztJQUNULE9BQU8sSUFDTCxTQUFTLFFBQ1QsT0FBTyxTQUFTLFlBQ2hCLE9BQU8sU0FBUyxZQUNoQixPQUFPLFNBQVMsYUFDaEIsT0FBTyxTQUFTLGVBQ2hCLE9BQU8sU0FBUyxVQUNoQjtNQUNBLE9BQU8sT0FBTztJQUNoQixPQUFPLElBQUksZ0JBQWdCLE9BQU87TUFDaEMsT0FBTyxLQUFLLEtBQUs7SUFDbkIsT0FBTyxJQUFJLE9BQU8sU0FBUyxVQUFVO01BQ25DLE9BQU8sQ0FBQyxDQUFDLEVBQ1AsT0FBTyxPQUFPLENBQUMsTUFDWixHQUFHLENBQUMsQ0FBQyxDQUFDLEdBQUcsRUFBRSxHQUFLLENBQUMsQ0FBQyxFQUFFLEVBQUUsRUFBRSxFQUFFLElBQUksQ0FBQyxRQUFRLENBQUMsR0FBRyxNQUFNLENBQUMsRUFDbEQsSUFBSSxDQUFDLEtBQ1QsQ0FBQyxDQUFDO0lBQ0w7SUFDQSxPQUFPO0VBQ1Q7RUFJQSxNQUNFLEdBQXdELEVBQ3hELEdBQUcsSUFBZSxFQUNIO0lBQ2YsT0FBTyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsVUFBVSxLQUFLLEVBQUUsUUFBUTtFQUM1QztFQUlBLEtBQ0UsR0FBd0QsRUFDeEQsR0FBRyxJQUFlLEVBQ0g7SUFDZixPQUFPLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxVQUFVLElBQUksRUFBRSxRQUFRO0VBQzNDO0VBSUEsS0FDRSxHQUF3RCxFQUN4RCxHQUFHLElBQWUsRUFDSDtJQUNmLE9BQU8sSUFBSSxDQUFDLENBQUMsR0FBRyxDQUFDLFVBQVUsSUFBSSxFQUFFLFFBQVE7RUFDM0M7RUFJQSxNQUNFLEdBQXdELEVBQ3hELEdBQUcsSUFBZSxFQUNIO0lBQ2YsT0FBTyxJQUFJLENBQUMsQ0FBQyxHQUFHLENBQUMsVUFBVSxLQUFLLEVBQUUsUUFBUTtFQUM1QztFQU9BLFNBQ0UsR0FBd0QsRUFDeEQsR0FBRyxJQUFlLEVBQ0g7SUFDZixPQUFPLElBQUksQ0FBQyxDQUFDLEdBQUcsQ0FBQyxVQUFVLFFBQVEsRUFBRSxRQUFRO0VBQy9DO0FBQ0YifQ==