// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { deepMerge } from "../collections/deep_merge.ts";
export class TOMLParseError extends Error {
}
export class Scanner {
  source;
  #whitespace;
  #position;
  constructor(source){
    this.source = source;
    this.#whitespace = /[ \t]/;
    this.#position = 0;
  }
  /**
   * Get current character
   * @param index - relative index from current position
   */ char(index = 0) {
    return this.source[this.#position + index] ?? "";
  }
  /**
   * Get sliced string
   * @param start - start position relative from current position
   * @param end - end position relative from current position
   */ slice(start, end) {
    return this.source.slice(this.#position + start, this.#position + end);
  }
  /**
   * Move position to next
   */ next(count) {
    if (typeof count === "number") {
      for(let i = 0; i < count; i++){
        this.#position++;
      }
    } else {
      this.#position++;
    }
  }
  /**
   * Move position until current char is not a whitespace, EOL, or comment.
   * @param options.inline - skip only whitespaces
   */ nextUntilChar(options = {
    comment: true
  }) {
    if (options.inline) {
      while(this.#whitespace.test(this.char()) && !this.eof()){
        this.next();
      }
    } else {
      while(!this.eof()){
        const char = this.char();
        if (this.#whitespace.test(char) || this.isCurrentCharEOL()) {
          this.next();
        } else if (options.comment && this.char() === "#") {
          // entering comment
          while(!this.isCurrentCharEOL() && !this.eof()){
            this.next();
          }
        } else {
          break;
        }
      }
    }
    // Invalid if current char is other kinds of whitespace
    if (!this.isCurrentCharEOL() && /\s/.test(this.char())) {
      const escaped = "\\u" + this.char().charCodeAt(0).toString(16);
      throw new TOMLParseError(`Contains invalid whitespaces: \`${escaped}\``);
    }
  }
  /**
   * Position reached EOF or not
   */ eof() {
    return this.position() >= this.source.length;
  }
  /**
   * Get current position
   */ position() {
    return this.#position;
  }
  isCurrentCharEOL() {
    return this.char() === "\n" || this.slice(0, 2) === "\r\n";
  }
}
// -----------------------
// Utilities
// -----------------------
function success(body) {
  return {
    ok: true,
    body
  };
}
function failure() {
  return {
    ok: false
  };
}
export const Utils = {
  unflat (keys, values = {}, cObj) {
    const out = {};
    if (keys.length === 0) {
      return cObj;
    } else {
      if (!cObj) {
        cObj = values;
      }
      const key = keys[keys.length - 1];
      if (typeof key === "string") {
        out[key] = cObj;
      }
      return this.unflat(keys.slice(0, -1), values, out);
    }
  },
  deepAssignWithTable (target, table) {
    if (table.key.length === 0 || table.key[0] == null) {
      throw new Error("Unexpected key length");
    }
    const value = target[table.key[0]];
    if (typeof value === "undefined") {
      Object.assign(target, this.unflat(table.key, table.type === "Table" ? table.value : [
        table.value
      ]));
    } else if (Array.isArray(value)) {
      if (table.type === "TableArray" && table.key.length === 1) {
        value.push(table.value);
      } else {
        const last = value[value.length - 1];
        Utils.deepAssignWithTable(last, {
          type: table.type,
          key: table.key.slice(1),
          value: table.value
        });
      }
    } else if (typeof value === "object" && value !== null) {
      Utils.deepAssignWithTable(value, {
        type: table.type,
        key: table.key.slice(1),
        value: table.value
      });
    } else {
      throw new Error("Unexpected assign");
    }
  }
};
// ---------------------------------
// Parser combinators and generators
// ---------------------------------
function or(parsers) {
  return function Or(scanner) {
    for (const parse of parsers){
      const result = parse(scanner);
      if (result.ok) {
        return result;
      }
    }
    return failure();
  };
}
function join(parser, separator) {
  const Separator = character(separator);
  return function Join(scanner) {
    const first = parser(scanner);
    if (!first.ok) {
      return failure();
    }
    const out = [
      first.body
    ];
    while(!scanner.eof()){
      if (!Separator(scanner).ok) {
        break;
      }
      const result = parser(scanner);
      if (result.ok) {
        out.push(result.body);
      } else {
        throw new TOMLParseError(`Invalid token after "${separator}"`);
      }
    }
    return success(out);
  };
}
function kv(keyParser, separator, valueParser) {
  const Separator = character(separator);
  return function Kv(scanner) {
    const key = keyParser(scanner);
    if (!key.ok) {
      return failure();
    }
    const sep = Separator(scanner);
    if (!sep.ok) {
      throw new TOMLParseError(`key/value pair doesn't have "${separator}"`);
    }
    const value = valueParser(scanner);
    if (!value.ok) {
      throw new TOMLParseError(`Value of key/value pair is invalid data format`);
    }
    return success(Utils.unflat(key.body, value.body));
  };
}
function merge(parser) {
  return function Merge(scanner) {
    const result = parser(scanner);
    if (!result.ok) {
      return failure();
    }
    let body = {};
    for (const record of result.body){
      if (typeof body === "object" && body !== null) {
        // deno-lint-ignore no-explicit-any
        body = deepMerge(body, record);
      }
    }
    return success(body);
  };
}
function repeat(parser) {
  return function Repeat(scanner) {
    const body = [];
    while(!scanner.eof()){
      const result = parser(scanner);
      if (result.ok) {
        body.push(result.body);
      } else {
        break;
      }
      scanner.nextUntilChar();
    }
    if (body.length === 0) {
      return failure();
    }
    return success(body);
  };
}
function surround(left, parser, right) {
  const Left = character(left);
  const Right = character(right);
  return function Surround(scanner) {
    if (!Left(scanner).ok) {
      return failure();
    }
    const result = parser(scanner);
    if (!result.ok) {
      throw new TOMLParseError(`Invalid token after "${left}"`);
    }
    if (!Right(scanner).ok) {
      throw new TOMLParseError(`Not closed by "${right}" after started with "${left}"`);
    }
    return success(result.body);
  };
}
function character(str) {
  return function character(scanner) {
    scanner.nextUntilChar({
      inline: true
    });
    if (scanner.slice(0, str.length) === str) {
      scanner.next(str.length);
    } else {
      return failure();
    }
    scanner.nextUntilChar({
      inline: true
    });
    return success(undefined);
  };
}
// -----------------------
// Parser components
// -----------------------
const Patterns = {
  BARE_KEY: /[A-Za-z0-9_-]/,
  FLOAT: /[0-9_\.e+\-]/i,
  END_OF_VALUE: /[ \t\r\n#,}\]]/
};
export function BareKey(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  if (!scanner.char() || !Patterns.BARE_KEY.test(scanner.char())) {
    return failure();
  }
  const acc = [];
  while(scanner.char() && Patterns.BARE_KEY.test(scanner.char())){
    acc.push(scanner.char());
    scanner.next();
  }
  const key = acc.join("");
  return success(key);
}
function EscapeSequence(scanner) {
  if (scanner.char() === "\\") {
    scanner.next();
    // See https://toml.io/en/v1.0.0-rc.3#string
    switch(scanner.char()){
      case "b":
        scanner.next();
        return success("\b");
      case "t":
        scanner.next();
        return success("\t");
      case "n":
        scanner.next();
        return success("\n");
      case "f":
        scanner.next();
        return success("\f");
      case "r":
        scanner.next();
        return success("\r");
      case "u":
      case "U":
        {
          // Unicode character
          const codePointLen = scanner.char() === "u" ? 4 : 6;
          const codePoint = parseInt("0x" + scanner.slice(1, 1 + codePointLen), 16);
          const str = String.fromCodePoint(codePoint);
          scanner.next(codePointLen + 1);
          return success(str);
        }
      case '"':
        scanner.next();
        return success('"');
      case "\\":
        scanner.next();
        return success("\\");
      default:
        throw new TOMLParseError(`Invalid escape sequence: \\${scanner.char()}`);
    }
  } else {
    return failure();
  }
}
export function BasicString(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  if (scanner.char() === '"') {
    scanner.next();
  } else {
    return failure();
  }
  const acc = [];
  while(scanner.char() !== '"' && !scanner.eof()){
    if (scanner.char() === "\n") {
      throw new TOMLParseError("Single-line string cannot contain EOL");
    }
    const escapedChar = EscapeSequence(scanner);
    if (escapedChar.ok) {
      acc.push(escapedChar.body);
    } else {
      acc.push(scanner.char());
      scanner.next();
    }
  }
  if (scanner.eof()) {
    throw new TOMLParseError(`Single-line string is not closed:\n${acc.join("")}`);
  }
  scanner.next(); // skip last '""
  return success(acc.join(""));
}
export function LiteralString(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  if (scanner.char() === "'") {
    scanner.next();
  } else {
    return failure();
  }
  const acc = [];
  while(scanner.char() !== "'" && !scanner.eof()){
    if (scanner.char() === "\n") {
      throw new TOMLParseError("Single-line string cannot contain EOL");
    }
    acc.push(scanner.char());
    scanner.next();
  }
  if (scanner.eof()) {
    throw new TOMLParseError(`Single-line string is not closed:\n${acc.join("")}`);
  }
  scanner.next(); // skip last "'"
  return success(acc.join(""));
}
export function MultilineBasicString(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  if (scanner.slice(0, 3) === '"""') {
    scanner.next(3);
  } else {
    return failure();
  }
  if (scanner.char() === "\n") {
    // The first newline (LF) is trimmed
    scanner.next();
  } else if (scanner.slice(0, 2) === "\r\n") {
    // The first newline (CRLF) is trimmed
    scanner.next(2);
  }
  const acc = [];
  while(scanner.slice(0, 3) !== '"""' && !scanner.eof()){
    // line ending backslash
    if (scanner.slice(0, 2) === "\\\n") {
      scanner.next();
      scanner.nextUntilChar({
        comment: false
      });
      continue;
    } else if (scanner.slice(0, 3) === "\\\r\n") {
      scanner.next();
      scanner.nextUntilChar({
        comment: false
      });
      continue;
    }
    const escapedChar = EscapeSequence(scanner);
    if (escapedChar.ok) {
      acc.push(escapedChar.body);
    } else {
      acc.push(scanner.char());
      scanner.next();
    }
  }
  if (scanner.eof()) {
    throw new TOMLParseError(`Multi-line string is not closed:\n${acc.join("")}`);
  }
  // if ends with 4 `"`, push the fist `"` to string
  if (scanner.char(3) === '"') {
    acc.push('"');
    scanner.next();
  }
  scanner.next(3); // skip last '""""
  return success(acc.join(""));
}
export function MultilineLiteralString(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  if (scanner.slice(0, 3) === "'''") {
    scanner.next(3);
  } else {
    return failure();
  }
  if (scanner.char() === "\n") {
    // The first newline (LF) is trimmed
    scanner.next();
  } else if (scanner.slice(0, 2) === "\r\n") {
    // The first newline (CRLF) is trimmed
    scanner.next(2);
  }
  const acc = [];
  while(scanner.slice(0, 3) !== "'''" && !scanner.eof()){
    acc.push(scanner.char());
    scanner.next();
  }
  if (scanner.eof()) {
    throw new TOMLParseError(`Multi-line string is not closed:\n${acc.join("")}`);
  }
  // if ends with 4 `'`, push the fist `'` to string
  if (scanner.char(3) === "'") {
    acc.push("'");
    scanner.next();
  }
  scanner.next(3); // skip last "'''"
  return success(acc.join(""));
}
const symbolPairs = [
  [
    "true",
    true
  ],
  [
    "false",
    false
  ],
  [
    "inf",
    Infinity
  ],
  [
    "+inf",
    Infinity
  ],
  [
    "-inf",
    -Infinity
  ],
  [
    "nan",
    NaN
  ],
  [
    "+nan",
    NaN
  ],
  [
    "-nan",
    NaN
  ]
];
export function Symbols(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  const found = symbolPairs.find(([str])=>scanner.slice(0, str.length) === str);
  if (!found) {
    return failure();
  }
  const [str, value] = found;
  scanner.next(str.length);
  return success(value);
}
export const DottedKey = join(or([
  BareKey,
  BasicString,
  LiteralString
]), ".");
export function Integer(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  // If binary / octal / hex
  const first2 = scanner.slice(0, 2);
  if (first2.length === 2 && /0(?:x|o|b)/i.test(first2)) {
    scanner.next(2);
    const acc = [
      first2
    ];
    while(/[0-9a-f_]/i.test(scanner.char()) && !scanner.eof()){
      acc.push(scanner.char());
      scanner.next();
    }
    if (acc.length === 1) {
      return failure();
    }
    return success(acc.join(""));
  }
  const acc = [];
  if (/[+-]/.test(scanner.char())) {
    acc.push(scanner.char());
    scanner.next();
  }
  while(/[0-9_]/.test(scanner.char()) && !scanner.eof()){
    acc.push(scanner.char());
    scanner.next();
  }
  if (acc.length === 0 || acc.length === 1 && /[+-]/.test(acc[0])) {
    return failure();
  }
  const int = parseInt(acc.filter((char)=>char !== "_").join(""));
  return success(int);
}
export function Float(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  // lookahead validation is needed for integer value is similar to float
  let position = 0;
  while(scanner.char(position) && !Patterns.END_OF_VALUE.test(scanner.char(position))){
    if (!Patterns.FLOAT.test(scanner.char(position))) {
      return failure();
    }
    position++;
  }
  const acc = [];
  if (/[+-]/.test(scanner.char())) {
    acc.push(scanner.char());
    scanner.next();
  }
  while(Patterns.FLOAT.test(scanner.char()) && !scanner.eof()){
    acc.push(scanner.char());
    scanner.next();
  }
  if (acc.length === 0) {
    return failure();
  }
  const float = parseFloat(acc.filter((char)=>char !== "_").join(""));
  if (isNaN(float)) {
    return failure();
  }
  return success(float);
}
export function DateTime(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  let dateStr = scanner.slice(0, 10);
  // example: 1979-05-27
  if (/^\d{4}-\d{2}-\d{2}/.test(dateStr)) {
    scanner.next(10);
  } else {
    return failure();
  }
  const acc = [];
  // example: 1979-05-27T00:32:00Z
  while(/[ 0-9TZ.:-]/.test(scanner.char()) && !scanner.eof()){
    acc.push(scanner.char());
    scanner.next();
  }
  dateStr += acc.join("");
  const date = new Date(dateStr.trim());
  // invalid date
  if (isNaN(date.getTime())) {
    throw new TOMLParseError(`Invalid date string "${dateStr}"`);
  }
  return success(date);
}
export function LocalTime(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  let timeStr = scanner.slice(0, 8);
  if (/^(\d{2}):(\d{2}):(\d{2})/.test(timeStr)) {
    scanner.next(8);
  } else {
    return failure();
  }
  const acc = [];
  if (scanner.char() === ".") {
    acc.push(scanner.char());
    scanner.next();
  } else {
    return success(timeStr);
  }
  while(/[0-9]/.test(scanner.char()) && !scanner.eof()){
    acc.push(scanner.char());
    scanner.next();
  }
  timeStr += acc.join("");
  return success(timeStr);
}
export function ArrayValue(scanner) {
  scanner.nextUntilChar({
    inline: true
  });
  if (scanner.char() === "[") {
    scanner.next();
  } else {
    return failure();
  }
  const array = [];
  while(!scanner.eof()){
    scanner.nextUntilChar();
    const result = Value(scanner);
    if (result.ok) {
      array.push(result.body);
    } else {
      break;
    }
    scanner.nextUntilChar({
      inline: true
    });
    // may have a next item, but trailing comma is allowed at array
    if (scanner.char() === ",") {
      scanner.next();
    } else {
      break;
    }
  }
  scanner.nextUntilChar();
  if (scanner.char() === "]") {
    scanner.next();
  } else {
    throw new TOMLParseError("Array is not closed");
  }
  return success(array);
}
export function InlineTable(scanner) {
  scanner.nextUntilChar();
  if (scanner.char(1) === "}") {
    scanner.next(2);
    return success({});
  }
  const pairs = surround("{", join(Pair, ","), "}")(scanner);
  if (!pairs.ok) {
    return failure();
  }
  let table = {};
  for (const pair of pairs.body){
    table = deepMerge(table, pair);
  }
  return success(table);
}
export const Value = or([
  MultilineBasicString,
  MultilineLiteralString,
  BasicString,
  LiteralString,
  Symbols,
  DateTime,
  LocalTime,
  Float,
  Integer,
  ArrayValue,
  InlineTable
]);
export const Pair = kv(DottedKey, "=", Value);
export function Block(scanner) {
  scanner.nextUntilChar();
  const result = merge(repeat(Pair))(scanner);
  if (result.ok) {
    return success({
      type: "Block",
      value: result.body
    });
  } else {
    return failure();
  }
}
export const TableHeader = surround("[", DottedKey, "]");
export function Table(scanner) {
  scanner.nextUntilChar();
  const header = TableHeader(scanner);
  if (!header.ok) {
    return failure();
  }
  scanner.nextUntilChar();
  const block = Block(scanner);
  return success({
    type: "Table",
    key: header.body,
    value: block.ok ? block.body.value : {}
  });
}
export const TableArrayHeader = surround("[[", DottedKey, "]]");
export function TableArray(scanner) {
  scanner.nextUntilChar();
  const header = TableArrayHeader(scanner);
  if (!header.ok) {
    return failure();
  }
  scanner.nextUntilChar();
  const block = Block(scanner);
  return success({
    type: "TableArray",
    key: header.body,
    value: block.ok ? block.body.value : {}
  });
}
export function Toml(scanner) {
  const blocks = repeat(or([
    Block,
    TableArray,
    Table
  ]))(scanner);
  if (!blocks.ok) {
    return failure();
  }
  let body = {};
  for (const block of blocks.body){
    switch(block.type){
      case "Block":
        {
          body = deepMerge(body, block.value);
          break;
        }
      case "Table":
        {
          Utils.deepAssignWithTable(body, block);
          break;
        }
      case "TableArray":
        {
          Utils.deepAssignWithTable(body, block);
          break;
        }
    }
  }
  return success(body);
}
export function ParserFactory(parser) {
  return function parse(tomlString) {
    const scanner = new Scanner(tomlString);
    let parsed = null;
    let err = null;
    try {
      parsed = parser(scanner);
    } catch (e) {
      err = e instanceof Error ? e : new Error("[non-error thrown]");
    }
    if (err || !parsed || !parsed.ok || !scanner.eof()) {
      const position = scanner.position();
      const subStr = tomlString.slice(0, position);
      const lines = subStr.split("\n");
      const row = lines.length;
      const column = (()=>{
        let count = subStr.length;
        for (const line of lines){
          if (count > line.length) {
            count -= line.length + 1;
          } else {
            break;
          }
        }
        return count;
      })();
      const message = `Parse error on line ${row}, column ${column}: ${err ? err.message : `Unexpected character: "${scanner.char()}"`}`;
      throw new TOMLParseError(message);
    }
    return parsed.body;
  };
}
//# sourceMappingURL=data:application/json;base64,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