// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// Copyright (c) 2014 Jameson Little. MIT License.
// This module is browser compatible.
/**
 * Utilities for
 * {@link https://datatracker.ietf.org/doc/html/rfc4648#section-6 | base32}
 * encoding and decoding.
 *
 * Modified from {@link https://github.com/beatgammit/base64-js}.
 *
 * This module is browser compatible.
 *
 * @module
 */ import { validateBinaryLike } from "./_util.ts";
const lookup = [];
const revLookup = [];
// RFC4648 base32
const code = "ABCDEFGHIJKLMNOPQRSTUVWXYZ234567";
for(let i = 0, len = code.length; i < len; ++i){
  lookup[i] = code[i];
  revLookup[code.charCodeAt(i)] = i;
}
const placeHolderPadLookup = [
  0,
  1,
  ,
  2,
  3,
  ,
  4
];
function _getPadLen(placeHoldersLen) {
  const maybeLen = placeHolderPadLookup[placeHoldersLen];
  if (typeof maybeLen !== "number") {
    throw new Error("Invalid pad length");
  }
  return maybeLen;
}
function getLens(b32) {
  const len = b32.length;
  if (len % 8 > 0) {
    throw new Error("Invalid string. Length must be a multiple of 8");
  }
  let validLen = b32.indexOf("=");
  if (validLen === -1) validLen = len;
  const placeHoldersLen = validLen === len ? 0 : 8 - validLen % 8;
  return [
    validLen,
    placeHoldersLen
  ];
}
function _byteLength(validLen, placeHoldersLen) {
  return (validLen + placeHoldersLen) * 5 / 8 - _getPadLen(placeHoldersLen);
}
/**
 * Decodes a base32-encoded string.
 *
 * @see {@link https://datatracker.ietf.org/doc/html/rfc4648#section-6}
 *
 * @example
 * ```ts
 * import { decodeBase32 } from "https://deno.land/std@$STD_VERSION/encoding/base32.ts";
 *
 * decodeBase32("NRQMA==="); // Uint8Array(3) [ 108, 96, 192 ]
 * ```
 */ export function decodeBase32(b32) {
  let tmp;
  const [validLen, placeHoldersLen] = getLens(b32);
  const arr = new Uint8Array(_byteLength(validLen, placeHoldersLen));
  let curByte = 0;
  // if there are placeholders, only get up to the last complete 8 chars
  const len = placeHoldersLen > 0 ? validLen - 8 : validLen;
  let i;
  for(i = 0; i < len; i += 8){
    tmp = revLookup[b32.charCodeAt(i)] << 20 | revLookup[b32.charCodeAt(i + 1)] << 15 | revLookup[b32.charCodeAt(i + 2)] << 10 | revLookup[b32.charCodeAt(i + 3)] << 5 | revLookup[b32.charCodeAt(i + 4)];
    arr[curByte++] = tmp >> 17 & 0xff;
    arr[curByte++] = tmp >> 9 & 0xff;
    arr[curByte++] = tmp >> 1 & 0xff;
    tmp = (tmp & 1) << 15 | revLookup[b32.charCodeAt(i + 5)] << 10 | revLookup[b32.charCodeAt(i + 6)] << 5 | revLookup[b32.charCodeAt(i + 7)];
    arr[curByte++] = tmp >> 8 & 0xff;
    arr[curByte++] = tmp & 0xff;
  }
  if (placeHoldersLen === 1) {
    tmp = revLookup[b32.charCodeAt(i)] << 20 | revLookup[b32.charCodeAt(i + 1)] << 15 | revLookup[b32.charCodeAt(i + 2)] << 10 | revLookup[b32.charCodeAt(i + 3)] << 5 | revLookup[b32.charCodeAt(i + 4)];
    arr[curByte++] = tmp >> 17 & 0xff;
    arr[curByte++] = tmp >> 9 & 0xff;
    arr[curByte++] = tmp >> 1 & 0xff;
    tmp = (tmp & 1) << 7 | revLookup[b32.charCodeAt(i + 5)] << 2 | revLookup[b32.charCodeAt(i + 6)] >> 3;
    arr[curByte++] = tmp & 0xff;
  } else if (placeHoldersLen === 3) {
    tmp = revLookup[b32.charCodeAt(i)] << 19 | revLookup[b32.charCodeAt(i + 1)] << 14 | revLookup[b32.charCodeAt(i + 2)] << 9 | revLookup[b32.charCodeAt(i + 3)] << 4 | revLookup[b32.charCodeAt(i + 4)] >> 1;
    arr[curByte++] = tmp >> 16 & 0xff;
    arr[curByte++] = tmp >> 8 & 0xff;
    arr[curByte++] = tmp & 0xff;
  } else if (placeHoldersLen === 4) {
    tmp = revLookup[b32.charCodeAt(i)] << 11 | revLookup[b32.charCodeAt(i + 1)] << 6 | revLookup[b32.charCodeAt(i + 2)] << 1 | revLookup[b32.charCodeAt(i + 3)] >> 4;
    arr[curByte++] = tmp >> 8 & 0xff;
    arr[curByte++] = tmp & 0xff;
  } else if (placeHoldersLen === 6) {
    tmp = revLookup[b32.charCodeAt(i)] << 3 | revLookup[b32.charCodeAt(i + 1)] >> 2;
    arr[curByte++] = tmp & 0xff;
  }
  return arr;
}
function encodeChunk(uint8, start, end) {
  let tmp;
  const output = [];
  for(let i = start; i < end; i += 5){
    tmp = uint8[i] << 16 & 0xff0000 | uint8[i + 1] << 8 & 0xff00 | uint8[i + 2] & 0xff;
    output.push(lookup[tmp >> 19 & 0x1f]);
    output.push(lookup[tmp >> 14 & 0x1f]);
    output.push(lookup[tmp >> 9 & 0x1f]);
    output.push(lookup[tmp >> 4 & 0x1f]);
    tmp = (tmp & 0xf) << 16 | uint8[i + 3] << 8 & 0xff00 | uint8[i + 4] & 0xff;
    output.push(lookup[tmp >> 15 & 0x1f]);
    output.push(lookup[tmp >> 10 & 0x1f]);
    output.push(lookup[tmp >> 5 & 0x1f]);
    output.push(lookup[tmp & 0x1f]);
  }
  return output.join("");
}
/**
 * Converts data to a base32-encoded string.
 *
 * @see {@link https://datatracker.ietf.org/doc/html/rfc4648#section-6}
 *
 * @example
 * ```ts
 * import { encodeBase32 } from "https://deno.land/std@$STD_VERSION/encoding/base32.ts";
 *
 * encodeBase32("6c60c0"); // "NRQMA==="
 * ```
 */ export function encodeBase32(data) {
  const uint8 = validateBinaryLike(data);
  let tmp;
  const len = uint8.length;
  const extraBytes = len % 5;
  const parts = [];
  const maxChunkLength = 16385; // must be multiple of 5
  const len2 = len - extraBytes;
  // go through the array every 5 bytes, we'll deal with trailing stuff later
  for(let i = 0; i < len2; i += maxChunkLength){
    parts.push(encodeChunk(uint8, i, i + maxChunkLength > len2 ? len2 : i + maxChunkLength));
  }
  // pad the end with zeros, but make sure to not forget the extra bytes
  if (extraBytes === 4) {
    tmp = (uint8[len2] & 0xff) << 16 | (uint8[len2 + 1] & 0xff) << 8 | uint8[len2 + 2] & 0xff;
    parts.push(lookup[tmp >> 19 & 0x1f]);
    parts.push(lookup[tmp >> 14 & 0x1f]);
    parts.push(lookup[tmp >> 9 & 0x1f]);
    parts.push(lookup[tmp >> 4 & 0x1f]);
    tmp = (tmp & 0xf) << 11 | uint8[len2 + 3] << 3;
    parts.push(lookup[tmp >> 10 & 0x1f]);
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("=");
  } else if (extraBytes === 3) {
    tmp = (uint8[len2] & 0xff) << 17 | (uint8[len2 + 1] & 0xff) << 9 | (uint8[len2 + 2] & 0xff) << 1;
    parts.push(lookup[tmp >> 20 & 0x1f]);
    parts.push(lookup[tmp >> 15 & 0x1f]);
    parts.push(lookup[tmp >> 10 & 0x1f]);
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("===");
  } else if (extraBytes === 2) {
    tmp = (uint8[len2] & 0xff) << 12 | (uint8[len2 + 1] & 0xff) << 4;
    parts.push(lookup[tmp >> 15 & 0x1f]);
    parts.push(lookup[tmp >> 10 & 0x1f]);
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("====");
  } else if (extraBytes === 1) {
    tmp = (uint8[len2] & 0xff) << 2;
    parts.push(lookup[tmp >> 5 & 0x1f]);
    parts.push(lookup[tmp & 0x1f]);
    parts.push("======");
  }
  return parts.join("");
}
//# sourceMappingURL=data:application/json;base64,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