// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { decodeHex, encodeHex } from "../encoding/hex.ts";
const encoder = new TextEncoder();
function splitByLast(value, separator) {
  const index = value.lastIndexOf(separator);
  return index === -1 ? [
    value,
    ""
  ] : [
    value.slice(0, index),
    value.slice(index + 1)
  ];
}
/**
 * Returns a promise with the signed cookie value from the given cryptographic
 * key.
 *
 * @example
 * ```ts
 * import { signCookie } from "https://deno.land/std@$STD_VERSION/http/unstable_signed_cookie.ts";
 * import { setCookie } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const key = await crypto.subtle.generateKey(
 *   { name: "HMAC", hash: "SHA-256" },
 *   true,
 *   ["sign", "verify"],
 * );
 * const value = await signCookie("my-cookie-value", key);
 *
 * const headers = new Headers();
 * setCookie(headers, {
 *   name: "my-cookie-name",
 *   value,
 * });
 *
 * const cookieHeader = headers.get("set-cookie");
 * ```
 */ export async function signCookie(value, key) {
  const data = encoder.encode(value);
  const signature = await crypto.subtle.sign("HMAC", key, data);
  const signatureHex = encodeHex(signature);
  return `${value}.${signatureHex}`;
}
/**
 * Returns a promise of a boolean indicating whether the signed cookie is valid.
 *
 * @example
 * ```ts
 * import { verifyCookie } from "https://deno.land/std@$STD_VERSION/http/unstable_signed_cookie.ts";
 * import { getCookies } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const key = await crypto.subtle.generateKey(
 *   { name: "HMAC", hash: "SHA-256" },
 *   true,
 *   ["sign", "verify"],
 * );
 *
 * const headers = new Headers({
 *   Cookie: "location=tokyo.37f7481039762eef5cd46669f93c0a3214dfecba7d0cdc0b0dc40036063fb22e",
 * });
 * const signedCookie = getCookies(headers)["location"];
 * await verifyCookie(signedCookie, key);
 * ```
 */ export async function verifyCookie(signedCookie, key) {
  const [value, signatureHex] = splitByLast(signedCookie, ".");
  if (!value || !signatureHex) return false;
  const data = encoder.encode(value);
  const signature = decodeHex(signatureHex);
  return await crypto.subtle.verify("HMAC", key, signature, data);
}
/**
 * Parses a signed cookie to get its value.
 *
 * Important: always verify the cookie using {@linkcode verifyCookie} first.
 *
 * @example
 * ```ts
 * import { verifyCookie, parseSignedCookie } from "https://deno.land/std@$STD_VERSION/http/unstable_signed_cookie.ts";
 * import { getCookies } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const key = await crypto.subtle.generateKey(
 *   { name: "HMAC", hash: "SHA-256" },
 *   true,
 *   ["sign", "verify"],
 * );
 *
 * const headers = new Headers({
 *   Cookie: "location=tokyo.37f7481039762eef5cd46669f93c0a3214dfecba7d0cdc0b0dc40036063fb22e",
 * });
 * const signedCookie = getCookies(headers)["location"];
 * await verifyCookie(signedCookie, key);
 * const cookie = parseSignedCookie(signedCookie);
 * ```
 */ export function parseSignedCookie(signedCookie) {
  return splitByLast(signedCookie, ".")[0];
}
//# sourceMappingURL=data:application/json;base64,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