// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { concat } from "../bytes/concat.ts";
/**
 * Converts a {@linkcode ReadableStream} of {@linkcode Uint8Array}s to an
 * {@linkcode ArrayBuffer}. Works the same as{@linkcode Response.arrayBuffer}.
 *
 * @example
 * ```ts
 * import { toArrayBuffer } from "https://deno.land/std@$STD_VERSION/streams/to_array_buffer.ts";
 *
 * const stream = ReadableStream.from([
 *   new Uint8Array([1, 2]),
 *   new Uint8Array([3, 4]),
 * ]);
 * await toArrayBuffer(stream); // ArrayBuffer { [Uint8Contents]: <01 02 03 04>, byteLength: 4 }
 * ```
 */ export async function toArrayBuffer(readableStream) {
  const reader = readableStream.getReader();
  const chunks = [];
  while(true){
    const { done, value } = await reader.read();
    if (done) {
      break;
    }
    chunks.push(value);
  }
  return concat(chunks).buffer;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjIxNy4wL3N0cmVhbXMvdG9fYXJyYXlfYnVmZmVyLnRzIl0sInNvdXJjZXNDb250ZW50IjpbIi8vIENvcHlyaWdodCAyMDE4LTIwMjQgdGhlIERlbm8gYXV0aG9ycy4gQWxsIHJpZ2h0cyByZXNlcnZlZC4gTUlUIGxpY2Vuc2UuXG4vLyBUaGlzIG1vZHVsZSBpcyBicm93c2VyIGNvbXBhdGlibGUuXG5cbmltcG9ydCB7IGNvbmNhdCB9IGZyb20gXCIuLi9ieXRlcy9jb25jYXQudHNcIjtcblxuLyoqXG4gKiBDb252ZXJ0cyBhIHtAbGlua2NvZGUgUmVhZGFibGVTdHJlYW19IG9mIHtAbGlua2NvZGUgVWludDhBcnJheX1zIHRvIGFuXG4gKiB7QGxpbmtjb2RlIEFycmF5QnVmZmVyfS4gV29ya3MgdGhlIHNhbWUgYXN7QGxpbmtjb2RlIFJlc3BvbnNlLmFycmF5QnVmZmVyfS5cbiAqXG4gKiBAZXhhbXBsZVxuICogYGBgdHNcbiAqIGltcG9ydCB7IHRvQXJyYXlCdWZmZXIgfSBmcm9tIFwiaHR0cHM6Ly9kZW5vLmxhbmQvc3RkQCRTVERfVkVSU0lPTi9zdHJlYW1zL3RvX2FycmF5X2J1ZmZlci50c1wiO1xuICpcbiAqIGNvbnN0IHN0cmVhbSA9IFJlYWRhYmxlU3RyZWFtLmZyb20oW1xuICogICBuZXcgVWludDhBcnJheShbMSwgMl0pLFxuICogICBuZXcgVWludDhBcnJheShbMywgNF0pLFxuICogXSk7XG4gKiBhd2FpdCB0b0FycmF5QnVmZmVyKHN0cmVhbSk7IC8vIEFycmF5QnVmZmVyIHsgW1VpbnQ4Q29udGVudHNdOiA8MDEgMDIgMDMgMDQ+LCBieXRlTGVuZ3RoOiA0IH1cbiAqIGBgYFxuICovXG5leHBvcnQgYXN5bmMgZnVuY3Rpb24gdG9BcnJheUJ1ZmZlcihcbiAgcmVhZGFibGVTdHJlYW06IFJlYWRhYmxlU3RyZWFtPFVpbnQ4QXJyYXk+LFxuKTogUHJvbWlzZTxBcnJheUJ1ZmZlcj4ge1xuICBjb25zdCByZWFkZXIgPSByZWFkYWJsZVN0cmVhbS5nZXRSZWFkZXIoKTtcbiAgY29uc3QgY2h1bmtzOiBVaW50OEFycmF5W10gPSBbXTtcblxuICB3aGlsZSAodHJ1ZSkge1xuICAgIGNvbnN0IHsgZG9uZSwgdmFsdWUgfSA9IGF3YWl0IHJlYWRlci5yZWFkKCk7XG5cbiAgICBpZiAoZG9uZSkge1xuICAgICAgYnJlYWs7XG4gICAgfVxuXG4gICAgY2h1bmtzLnB1c2godmFsdWUpO1xuICB9XG5cbiAgcmV0dXJuIGNvbmNhdChjaHVua3MpLmJ1ZmZlcjtcbn1cbiJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiQUFBQSwwRUFBMEU7QUFDMUUscUNBQXFDO0FBRXJDLFNBQVMsTUFBTSxRQUFRLHFCQUFxQjtBQUU1Qzs7Ozs7Ozs7Ozs7Ozs7Q0FjQyxHQUNELE9BQU8sZUFBZSxjQUNwQixjQUEwQztFQUUxQyxNQUFNLFNBQVMsZUFBZSxTQUFTO0VBQ3ZDLE1BQU0sU0FBdUIsRUFBRTtFQUUvQixNQUFPLEtBQU07SUFDWCxNQUFNLEVBQUUsSUFBSSxFQUFFLEtBQUssRUFBRSxHQUFHLE1BQU0sT0FBTyxJQUFJO0lBRXpDLElBQUksTUFBTTtNQUNSO0lBQ0Y7SUFFQSxPQUFPLElBQUksQ0FBQztFQUNkO0VBRUEsT0FBTyxPQUFPLFFBQVEsTUFBTTtBQUM5QiJ9