// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Utilities for working with {@link https://en.wikipedia.org/wiki/Ascii85 | ascii85} encoding.
 *
 * This module is browser compatible.
 *
 * ## Specifying a standard and delimiter
 *
 * By default, all functions are using the most popular Adobe version of ascii85
 * and not adding any delimiter. However, there are three more standards
 * supported - btoa (different delimiter and additional compression of 4 bytes
 * equal to 32), {@link https://rfc.zeromq.org/spec/32/ | Z85} and
 * {@link https://tools.ietf.org/html/rfc1924 | RFC 1924}. It's possible to use a
 * different encoding by specifying it in `options` object as a second parameter.
 *
 * Similarly, it's possible to make `encode` add a delimiter (`<~` and `~>` for
 * Adobe, `xbtoa Begin` and `xbtoa End` with newlines between the delimiters and
 * encoded data for btoa. Checksums for btoa are not supported. Delimiters are not
 * supported by other encodings.)
 *
 * @module
 */ import { validateBinaryLike } from "./_util.ts";
const rfc1924 = "0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz!#$%&()*+-;<=>?@^_`{|}~";
const Z85 = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-:+=^!/*?&<>()[]{}@%$#";
/**
 * Converts data into an ascii58-encoded string.
 *
 * @example
 * ```ts
 * import { encodeAscii85 } from "https://deno.land/std@$STD_VERSION/encoding/ascii85.ts";
 *
 * encodeAscii85("Hello world!"); // => "87cURD]j7BEbo80"
 * ```
 */ export function encodeAscii85(data, options) {
  let uint8 = validateBinaryLike(data);
  const standard = options?.standard ?? "Adobe";
  let output = [], v, n = 0, difference = 0;
  if (uint8.length % 4 !== 0) {
    const tmp = uint8;
    difference = 4 - tmp.length % 4;
    uint8 = new Uint8Array(tmp.length + difference);
    uint8.set(tmp);
  }
  const view = new DataView(uint8.buffer, uint8.byteOffset, uint8.byteLength);
  for(let i = 0, len = uint8.length; i < len; i += 4){
    v = view.getUint32(i);
    // Adobe and btoa standards compress 4 zeroes to single "z" character
    if ((standard === "Adobe" || standard === "btoa") && v === 0 && i < len - difference - 3) {
      output[n++] = "z";
      continue;
    }
    // btoa compresses 4 spaces - that is, bytes equal to 32 - into single "y" character
    if (standard === "btoa" && v === 538976288) {
      output[n++] = "y";
      continue;
    }
    for(let j = 4; j >= 0; j--){
      output[n + j] = String.fromCharCode(v % 85 + 33);
      v = Math.trunc(v / 85);
    }
    n += 5;
  }
  switch(standard){
    case "Adobe":
      if (options?.delimiter) {
        return `<~${output.slice(0, output.length - difference).join("")}~>`;
      }
      break;
    case "btoa":
      if (options?.delimiter) {
        return `xbtoa Begin\n${output.slice(0, output.length - difference).join("")}\nxbtoa End`;
      }
      break;
    case "RFC 1924":
      output = output.map((val)=>rfc1924[val.charCodeAt(0) - 33]);
      break;
    case "Z85":
      output = output.map((val)=>Z85[val.charCodeAt(0) - 33]);
      break;
  }
  return output.slice(0, output.length - difference).join("");
}
/**
 * Decodes a given ascii85-encoded string.
 *
 * @example
 * ```ts
 * import { decodeAscii85 } from "https://deno.land/std@$STD_VERSION/encoding/ascii85.ts";
 *
 * decodeAscii85("87cURD]j7BEbo80"); // => Uint8Array [ 72, 101, 108, 108, 111, 32, 119, 111, 114, 108, 100, 33 ]
 * ```
 */ export function decodeAscii85(ascii85, options) {
  const encoding = options?.standard ?? "Adobe";
  // translate all encodings to most basic adobe/btoa one and decompress some special characters ("z" and "y")
  switch(encoding){
    case "Adobe":
      ascii85 = ascii85.replaceAll(/(<~|~>)/g, "").replaceAll("z", "!!!!!");
      break;
    case "btoa":
      ascii85 = ascii85.replaceAll(/(xbtoa Begin|xbtoa End|\n)/g, "").replaceAll("z", "!!!!!").replaceAll("y", "+<VdL");
      break;
    case "RFC 1924":
      ascii85 = ascii85.replaceAll(/./g, (match)=>String.fromCharCode(rfc1924.indexOf(match) + 33));
      break;
    case "Z85":
      ascii85 = ascii85.replaceAll(/./g, (match)=>String.fromCharCode(Z85.indexOf(match) + 33));
      break;
  }
  //remove all invalid characters
  ascii85 = ascii85.replaceAll(/[^!-u]/g, "");
  const len = ascii85.length, output = new Uint8Array(len + 4 - len % 4);
  const view = new DataView(output.buffer);
  let v = 0, n = 0, max = 0;
  for(let i = 0; i < len;){
    for(max += 5; i < max; i++){
      v = v * 85 + (i < len ? ascii85.charCodeAt(i) : 117) - 33;
    }
    view.setUint32(n, v);
    v = 0;
    n += 4;
  }
  return output.slice(0, Math.trunc(len * 0.8));
}
//# sourceMappingURL=data:application/json;base64,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