// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Returns a tuple of two arrays with the first one containing all elements in
 * the given array that match the given predicate and the second one containing
 * all that do not.
 *
 * @example
 * ```ts
 * import { partition } from "https://deno.land/std@$STD_VERSION/collections/partition.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/assert/assert_equals.ts";
 *
 * const numbers = [5, 6, 7, 8, 9];
 * const [even, odd] = partition(numbers, (it) => it % 2 === 0);
 *
 * assertEquals(even, [6, 8]);
 * assertEquals(odd, [5, 7, 9]);
 * ```
 */ export function partition(array, predicate) {
  const matches = [];
  const rest = [];
  for (const element of array){
    if (predicate(element)) {
      matches.push(element);
    } else {
      rest.push(element);
    }
  }
  return [
    matches,
    rest
  ];
}
//# sourceMappingURL=data:application/json;base64,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