// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { DateTimeFormatter } from "./_common.ts";
/**
 * Takes an input `string` and a `formatString` to parse to a `date`.
 *
 * The following symbols from
 * {@link https://www.unicode.org/reports/tr35/tr35-dates.html#Date_Field_Symbol_Table | unicode LDML}
 * are supported:
 *
 * - `yyyy` - numeric year.
 * - `yy` - 2-digit year.
 * - `M` - numeric month.
 * - `MM` - 2-digit month.
 * - `d` - numeric day.
 * - `dd` - 2-digit day.
 *
 * - `H` - numeric hour (0-23 hours).
 * - `HH` - 2-digit hour (00-23 hours).
 * - `h` - numeric hour (1-12 hours).
 * - `hh` - 2-digit hour (01-12 hours).
 * - `m` - numeric minute.
 * - `mm` - 2-digit minute.
 * - `s` - numeric second.
 * - `ss` - 2-digit second.
 * - `S` - 1-digit fractionalSecond.
 * - `SS` - 2-digit fractionalSecond.
 * - `SSS` - 3-digit fractionalSecond.
 *
 * - `a` - dayPeriod, either `AM` or `PM`.
 *
 * - `'foo'` - quoted literal.
 * - `./-` - unquoted literal.
 *
 * @example
 * ```ts
 * import { parse } from "https://deno.land/std@$STD_VERSION/datetime/parse.ts";
 *
 * parse("20-01-2019", "dd-MM-yyyy"); // output : new Date(2019, 0, 20)
 * parse("2019-01-20", "yyyy-MM-dd"); // output : new Date(2019, 0, 20)
 * parse("20.01.2019", "dd.MM.yyyy"); // output : new Date(2019, 0, 20)
 * parse("01-20-2019 16:34", "MM-dd-yyyy HH:mm"); // output : new Date(2019, 0, 20, 16, 34)
 * parse("01-20-2019 04:34 PM", "MM-dd-yyyy hh:mm a"); // output : new Date(2019, 0, 20, 16, 34)
 * parse("16:34 01-20-2019", "HH:mm MM-dd-yyyy"); // output : new Date(2019, 0, 20, 16, 34)
 * parse("01-20-2019 16:34:23.123", "MM-dd-yyyy HH:mm:ss.SSS"); // output : new Date(2019, 0, 20, 16, 34, 23, 123)
 * ```
 *
 * @param dateString Date string
 * @param formatString Format string
 * @return Parsed date
 */ export function parse(dateString, formatString) {
  const formatter = new DateTimeFormatter(formatString);
  const parts = formatter.parseToParts(dateString);
  const sortParts = formatter.sortDateTimeFormatPart(parts);
  return formatter.partsToDate(sortParts);
}
//# sourceMappingURL=data:application/json;base64,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