// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { bytesToUuid } from "./_common.ts";
const UUID_RE = /^[0-9a-f]{8}-[0-9a-f]{4}-1[0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}$/i;
/**
 * Validates the UUID v1.
 *
 * @example
 * ```ts
 * import { validate } from "https://deno.land/std@$STD_VERSION/uuid/v1.ts";
 *
 * validate("ea71fc60-a713-11ee-af61-8349da24f689");  // true
 * validate("fac8c1e0-ad1a-4204-a0d0-8126ae84495d");  // false
 * ```
 *
 * @param id UUID value.
 */ export function validate(id) {
  return UUID_RE.test(id);
}
let _nodeId;
let _clockseq;
let _lastMSecs = 0;
let _lastNSecs = 0;
/**
 * Generates a RFC4122 v1 UUID (time-based).
 *
 * @example
 * ```ts
 * import { generate } from "https://deno.land/std@$STD_VERSION/uuid/v1.ts";
 *
 * const options = {
 *   node: [0x01, 0x23, 0x45, 0x67, 0x89, 0xab],
 *   clockseq: 0x1234,
 *   msecs: new Date("2011-11-01").getTime(),
 *   nsecs: 5678,
 * };
 *
 * generate(options); // "710b962e-041c-11e1-9234-0123456789ab"
 * ```
 *
 * @param options Can use RFC time sequence values as overwrites.
 * @param buf Can allow the UUID to be written in byte-form starting at the offset.
 * @param offset Index to start writing on the UUID bytes in buffer.
 */ export function generate(options, buf, offset) {
  let i = buf && offset || 0;
  const b = buf ?? [];
  options ??= {};
  let { node = _nodeId, clockseq = _clockseq } = options;
  if (node === undefined || clockseq === undefined) {
    // deno-lint-ignore no-explicit-any
    const seedBytes = options.random ?? options.rng ?? crypto.getRandomValues(new Uint8Array(16));
    if (node === undefined) {
      node = _nodeId = [
        seedBytes[0] | 0x01,
        seedBytes[1],
        seedBytes[2],
        seedBytes[3],
        seedBytes[4],
        seedBytes[5]
      ];
    }
    if (clockseq === undefined) {
      clockseq = _clockseq = (seedBytes[6] << 8 | seedBytes[7]) & 0x3fff;
    }
  }
  let { msecs = new Date().getTime(), nsecs = _lastNSecs + 1 } = options;
  const dt = msecs - _lastMSecs + (nsecs - _lastNSecs) / 10000;
  if (dt < 0 && options.clockseq === undefined) {
    clockseq = clockseq + 1 & 0x3fff;
  }
  if ((dt < 0 || msecs > _lastMSecs) && options.nsecs === undefined) {
    nsecs = 0;
  }
  if (nsecs > 10000) {
    throw new Error("Can't create more than 10M uuids/sec");
  }
  _lastMSecs = msecs;
  _lastNSecs = nsecs;
  _clockseq = clockseq;
  // We have to add this value because "msecs" here is the number of
  // milliseconds since January 1, 1970, not since October 15, 1582.
  // This is also the milliseconds from October 15, 1582 to January 1, 1970.
  msecs += 12219292800000;
  const tl = ((msecs & 0xfffffff) * 10000 + nsecs) % 0x100000000;
  b[i++] = tl >>> 24 & 0xff;
  b[i++] = tl >>> 16 & 0xff;
  b[i++] = tl >>> 8 & 0xff;
  b[i++] = tl & 0xff;
  const tmh = msecs / 0x100000000 * 10000 & 0xfffffff;
  b[i++] = tmh >>> 8 & 0xff;
  b[i++] = tmh & 0xff;
  b[i++] = tmh >>> 24 & 0xf | 0x10;
  b[i++] = tmh >>> 16 & 0xff;
  b[i++] = clockseq >>> 8 | 0x80;
  b[i++] = clockseq & 0xff;
  for(let n = 0; n < 6; ++n){
    b[i + n] = node[n];
  }
  return buf ?? bytesToUuid(b);
}
//# sourceMappingURL=data:application/json;base64,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