// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { DAY, HOUR, MINUTE, SECOND, WEEK } from "./constants.ts";
function calculateMonthsDifference(from, to) {
  let months = (to.getFullYear() - from.getFullYear()) * 12 + (to.getMonth() - from.getMonth());
  if (to.getDate() < from.getDate()) {
    months--;
  }
  return months;
}
/**
 * Returns the difference of the 2 given dates in the given units. If the units
 * are omitted, it returns the difference in the all available units.
 *
 * @example
 * ```ts
 * import { difference } from "https://deno.land/std@$STD_VERSION/datetime/difference.ts";
 *
 * const date0 = new Date("2018-05-14");
 * const date1 = new Date("2020-05-13");
 *
 * difference(date0, date1, { units: ["days", "months", "years"] });
 * // => returns { days: 730, months: 23, years: 1 }
 *
 * difference(date0, date1);
 * // => returns {
 * //   milliseconds: 63072000000,
 * //   seconds: 63072000,
 * //   minutes: 1051200,
 * //   hours: 17520,
 * //   days: 730,
 * //   weeks: 104,
 * //   months: 23,
 * //   quarters: 7,
 * //   years: 1
 * // }
 * ```
 *
 * @param from Year to calculate difference
 * @param to Year to calculate difference with
 * @param options Options for determining how to respond
 */ export function difference(from, to, options) {
  [from, to] = from < to ? [
    from,
    to
  ] : [
    to,
    from
  ];
  const uniqueUnits = options?.units ? [
    ...new Set(options?.units)
  ] : [
    "milliseconds",
    "seconds",
    "minutes",
    "hours",
    "days",
    "weeks",
    "months",
    "quarters",
    "years"
  ];
  const differenceInMs = Math.abs(from.getTime() - to.getTime());
  const differences = {};
  for (const uniqueUnit of uniqueUnits){
    switch(uniqueUnit){
      case "milliseconds":
        differences.milliseconds = differenceInMs;
        break;
      case "seconds":
        differences.seconds = Math.floor(differenceInMs / SECOND);
        break;
      case "minutes":
        differences.minutes = Math.floor(differenceInMs / MINUTE);
        break;
      case "hours":
        differences.hours = Math.floor(differenceInMs / HOUR);
        break;
      case "days":
        differences.days = Math.floor(differenceInMs / DAY);
        break;
      case "weeks":
        differences.weeks = Math.floor(differenceInMs / WEEK);
        break;
      case "months":
        differences.months = calculateMonthsDifference(from, to);
        break;
      case "quarters":
        differences.quarters = Math.floor(differences.months !== undefined && differences.months / 3 || calculateMonthsDifference(from, to) / 3);
        break;
      case "years":
        differences.years = Math.floor(differences.months !== undefined && differences.months / 12 || calculateMonthsDifference(from, to) / 12);
        break;
    }
  }
  return differences;
}
//# sourceMappingURL=data:application/json;base64,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