// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { DAY, WEEK } from "./constants.ts";
const DAYS_PER_WEEK = 7;
const Day = {
  Sun: 0,
  Mon: 1,
  Tue: 2,
  Wed: 3,
  Thu: 4,
  Fri: 5,
  Sat: 6
};
/**
 * Returns the ISO week number of the provided date (1-53).
 *
 * @example
 * ```ts
 * import { weekOfYear } from "https://deno.land/std@$STD_VERSION/datetime/week_of_year.ts";
 *
 * weekOfYear(new Date("2020-12-28T03:24:00")); // Returns 53
 * ```
 *
 * @return Number of the week in year
 */ export function weekOfYear(date) {
  const workingDate = new Date(Date.UTC(date.getFullYear(), date.getMonth(), date.getDate()));
  const day = workingDate.getUTCDay();
  const nearestThursday = workingDate.getUTCDate() + Day.Thu - (day === Day.Sun ? DAYS_PER_WEEK : day);
  workingDate.setUTCDate(nearestThursday);
  // Get first day of year
  const yearStart = new Date(Date.UTC(workingDate.getUTCFullYear(), 0, 1));
  // return the calculated full weeks to nearest Thursday
  return Math.ceil((workingDate.getTime() - yearStart.getTime() + DAY) / WEEK);
}
//# sourceMappingURL=data:application/json;base64,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