// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
// This module was heavily inspired by ua-parser-js
// (https://www.npmjs.com/package/ua-parser-js) which is MIT licensed and
// Copyright (c) 2012-2024 Faisal Salman <f@faisalman.com>
/** Provides {@linkcode UserAgent} and related types to be able to provide a
 * structured understanding of a user agent string.
 *
 * @module
 */ var _computedKey, _computedKey1;
import { assert } from "../assert/assert.ts";
const ARCHITECTURE = "architecture";
const MODEL = "model";
const NAME = "name";
const TYPE = "type";
const VENDOR = "vendor";
const VERSION = "version";
const EMPTY = "";
const CONSOLE = "console";
const EMBEDDED = "embedded";
const MOBILE = "mobile";
const TABLET = "tablet";
const SMARTTV = "smarttv";
const WEARABLE = "wearable";
const PREFIX_MOBILE = "Mobile ";
const SUFFIX_BROWSER = " Browser";
const AMAZON = "Amazon";
const APPLE = "Apple";
const ASUS = "ASUS";
const BLACKBERRY = "BlackBerry";
const CHROME = "Chrome";
const EDGE = "Edge";
const FACEBOOK = "Facebook";
const FIREFOX = "Firefox";
const GOOGLE = "Google";
const HUAWEI = "Huawei";
const LG = "LG";
const MICROSOFT = "Microsoft";
const MOTOROLA = "Motorola";
const OPERA = "Opera";
const SAMSUNG = "Samsung";
const SHARP = "Sharp";
const SONY = "Sony";
const WINDOWS = "Windows";
const XIAOMI = "Xiaomi";
const ZEBRA = "Zebra";
function lowerize(str) {
  return str.toLowerCase();
}
function majorize(str) {
  return str ? str.replace(/[^\d\.]/g, EMPTY).split(".")[0] : undefined;
}
function trim(str) {
  return str.trimStart();
}
/** A map where the key is the common Windows version and the value is a string
 * or array of strings of potential values parsed from the user-agent string. */ const windowsVersionMap = new Map([
  [
    "ME",
    "4.90"
  ],
  [
    "NT 3.11",
    "NT3.51"
  ],
  [
    "NT 4.0",
    "NT4.0"
  ],
  [
    "2000",
    "NT 5.0"
  ],
  [
    "XP",
    [
      "NT 5.1",
      "NT 5.2"
    ]
  ],
  [
    "Vista",
    "NT 6.0"
  ],
  [
    "7",
    "NT 6.1"
  ],
  [
    "8",
    "NT 6.2"
  ],
  [
    "8.1",
    "NT 6.3"
  ],
  [
    "10",
    [
      "NT 6.4",
      "NT 10.0"
    ]
  ],
  [
    "RT",
    "ARM"
  ]
]);
function has(str1, str2) {
  if (Array.isArray(str1)) {
    for (const el of str1){
      if (lowerize(el) === lowerize(str2)) {
        return true;
      }
    }
    return false;
  }
  return lowerize(str2).indexOf(lowerize(str1)) !== -1;
}
function mapWinVer(str) {
  for (const [key, value] of windowsVersionMap){
    if (Array.isArray(value)) {
      for (const v of value){
        if (has(v, str)) {
          return key;
        }
      }
    } else if (has(value, str)) {
      return key;
    }
  }
  return str || undefined;
}
function mapper(// deno-lint-ignore no-explicit-any
target, ua, tuples) {
  let matches = null;
  for (const [matchers, processors] of tuples){
    let j = 0;
    let k = 0;
    while(j < matchers.length && !matches){
      if (!matchers[j]) {
        break;
      }
      matches = matchers[j++].exec(ua);
      if (matches) {
        for (const processor of processors){
          const match = matches[++k];
          if (Array.isArray(processor)) {
            if (processor.length === 2) {
              const [prop, value] = processor;
              if (typeof value === "function") {
                target[prop] = value.call(target, match);
              } else {
                target[prop] = value;
              }
            } else if (processor.length === 3) {
              const [prop, re, value] = processor;
              target[prop] = match ? match.replace(re, value) : undefined;
            } else {
              const [prop, re, value, fn] = processor;
              assert(fn);
              target[prop] = match ? fn.call(prop, match.replace(re, value)) : undefined;
            }
          } else {
            target[processor] = match ? match : undefined;
          }
        }
      }
    }
  }
}
/** An object with properties that are arrays of tuples which provide match
 * patterns and configuration on how to interpret the capture groups. */ const matchers = {
  browser: [
    [
      [
        /\b(?:crmo|crios)\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${PREFIX_MOBILE}${CHROME}`
        ]
      ]
    ],
    [
      [
        /edg(?:e|ios|a)?\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Edge"
        ]
      ]
    ],
    // Presto based
    [
      [
        /(opera mini)\/([-\w\.]+)/i,
        /(opera [mobiletab]{3,6})\b.+version\/([-\w\.]+)/i,
        /(opera)(?:.+version\/|[\/ ]+)([\w\.]+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /opios[\/ ]+([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${OPERA} Mini`
        ]
      ]
    ],
    [
      [
        /\bopr\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          OPERA
        ]
      ]
    ],
    [
      [
        // Mixed
        /(kindle)\/([\w\.]+)/i,
        /(lunascape|maxthon|netfront|jasmine|blazer)[\/ ]?([\w\.]*)/i,
        // Trident based
        /(avant |iemobile|slim)(?:browser)?[\/ ]?([\w\.]*)/i,
        /(ba?idubrowser)[\/ ]?([\w\.]+)/i,
        /(?:ms|\()(ie) ([\w\.]+)/i,
        // Webkit/KHTML based
        // Flock/RockMelt/Midori/Epiphany/Silk/Skyfire/Bolt/Iron/Iridium/PhantomJS/Bowser/QupZilla/Falkon/Rekonq/Puffin/Brave/Whale/QQBrowserLite/QQ//Vivaldi/DuckDuckGo
        /(flock|rockmelt|midori|epiphany|silk|skyfire|ovibrowser|bolt|iron|vivaldi|iridium|phantomjs|bowser|quark|qupzilla|falkon|rekonq|puffin|brave|whale(?!.+naver)|qqbrowserlite|qq|duckduckgo)\/([-\w\.]+)/i,
        /(heytap|ovi)browser\/([\d\.]+)/i,
        /(weibo)__([\d\.]+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /(?:\buc? ?browser|(?:juc.+)ucweb)[\/ ]?([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "UCBrowser"
        ]
      ]
    ],
    [
      [
        /microm.+\bqbcore\/([\w\.]+)/i,
        /\bqbcore\/([\w\.]+).+microm/i
      ],
      [
        VERSION,
        [
          NAME,
          "WeChat(Win) Desktop"
        ]
      ]
    ],
    [
      [
        /micromessenger\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "WeChat"
        ]
      ]
    ],
    [
      [
        /konqueror\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Konqueror"
        ]
      ]
    ],
    [
      [
        /trident.+rv[: ]([\w\.]{1,9})\b.+like gecko/i
      ],
      [
        VERSION,
        [
          NAME,
          "IE"
        ]
      ]
    ],
    [
      [
        /ya(?:search)?browser\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Yandex"
        ]
      ]
    ],
    [
      [
        /(avast|avg)\/([\w\.]+)/i
      ],
      [
        [
          NAME,
          /(.+)/,
          `$1 Secure${SUFFIX_BROWSER}`
        ],
        VERSION
      ]
    ],
    [
      [
        /\bfocus\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${FIREFOX} Focus`
        ]
      ]
    ],
    [
      [
        /\bopt\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${OPERA} Touch`
        ]
      ]
    ],
    [
      [
        /coc_coc\w+\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Coc Coc"
        ]
      ]
    ],
    [
      [
        /dolfin\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Dolphin"
        ]
      ]
    ],
    [
      [
        /coast\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${OPERA} Coast`
        ]
      ]
    ],
    [
      [
        /miuibrowser\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `MIUI${SUFFIX_BROWSER}`
        ]
      ]
    ],
    [
      [
        /fxios\/([\w\.-]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${PREFIX_MOBILE}${FIREFOX}`
        ]
      ]
    ],
    [
      [
        /\bqihu|(qi?ho?o?|360)browser/i
      ],
      [
        [
          NAME,
          `360${SUFFIX_BROWSER}`
        ]
      ]
    ],
    [
      [
        /(oculus|samsung|sailfish|huawei)browser\/([\w\.]+)/i
      ],
      [
        [
          NAME,
          /(.+)/,
          "$1" + SUFFIX_BROWSER
        ],
        VERSION
      ]
    ],
    [
      [
        /(comodo_dragon)\/([\w\.]+)/i
      ],
      [
        [
          NAME,
          /_/g,
          " "
        ],
        VERSION
      ]
    ],
    [
      [
        /(electron)\/([\w\.]+) safari/i,
        /(tesla)(?: qtcarbrowser|\/(20\d\d\.[-\w\.]+))/i,
        /m?(qqbrowser|baiduboxapp|2345Explorer)[\/ ]?([\w\.]+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /(metasr)[\/ ]?([\w\.]+)/i,
        /(lbbrowser)/i,
        /\[(linkedin)app\]/i
      ],
      [
        NAME
      ]
    ],
    [
      [
        /((?:fban\/fbios|fb_iab\/fb4a)(?!.+fbav)|;fbav\/([\w\.]+);)/i
      ],
      [
        [
          NAME,
          FACEBOOK
        ],
        VERSION
      ]
    ],
    [
      [
        /(kakao(?:talk|story))[\/ ]([\w\.]+)/i,
        /(naver)\(.*?(\d+\.[\w\.]+).*\)/i,
        /safari (line)\/([\w\.]+)/i,
        /\b(line)\/([\w\.]+)\/iab/i,
        /(chromium|instagram)[\/ ]([-\w\.]+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /\bgsa\/([\w\.]+) .*safari\//i
      ],
      [
        VERSION,
        [
          NAME,
          "GSA"
        ]
      ]
    ],
    [
      [
        /musical_ly(?:.+app_?version\/|_)([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "TikTok"
        ]
      ]
    ],
    [
      [
        /headlesschrome(?:\/([\w\.]+)| )/i
      ],
      [
        VERSION,
        [
          NAME,
          `${CHROME} Headless`
        ]
      ]
    ],
    [
      [
        / wv\).+(chrome)\/([\w\.]+)/i
      ],
      [
        [
          NAME,
          `${CHROME} WebView`
        ],
        VERSION
      ]
    ],
    [
      [
        /droid.+ version\/([\w\.]+)\b.+(?:mobile safari|safari)/i
      ],
      [
        VERSION,
        [
          NAME,
          `Android${SUFFIX_BROWSER}`
        ]
      ]
    ],
    [
      [
        /chrome\/([\w\.]+) mobile/i
      ],
      [
        VERSION,
        [
          NAME,
          `${PREFIX_MOBILE}${CHROME}`
        ]
      ]
    ],
    [
      [
        /(chrome|omniweb|arora|[tizenoka]{5} ?browser)\/v?([\w\.]+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /version\/([\w\.\,]+) .*mobile(?:\/\w+ | ?)safari/i
      ],
      [
        VERSION,
        [
          NAME,
          `${PREFIX_MOBILE}Safari`
        ]
      ]
    ],
    [
      [
        /iphone .*mobile(?:\/\w+ | ?)safari/i
      ],
      [
        [
          NAME,
          `${PREFIX_MOBILE}Safari`
        ]
      ]
    ],
    [
      [
        /version\/([\w\.\,]+) .*(safari)/i
      ],
      [
        VERSION,
        NAME
      ]
    ],
    [
      [
        /webkit.+?(mobile ?safari|safari)(\/[\w\.]+)/i
      ],
      [
        NAME,
        [
          VERSION,
          "1"
        ]
      ]
    ],
    [
      [
        /(webkit|khtml)\/([\w\.]+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /(?:mobile|tablet);.*(firefox)\/([\w\.-]+)/i
      ],
      [
        [
          NAME,
          `${PREFIX_MOBILE}${FIREFOX}`
        ],
        VERSION
      ]
    ],
    [
      [
        /(navigator|netscape\d?)\/([-\w\.]+)/i
      ],
      [
        [
          NAME,
          "Netscape"
        ],
        VERSION
      ]
    ],
    [
      [
        /mobile vr; rv:([\w\.]+)\).+firefox/i
      ],
      [
        VERSION,
        [
          NAME,
          `${FIREFOX} Reality`
        ]
      ]
    ],
    [
      [
        /ekiohf.+(flow)\/([\w\.]+)/i,
        /(swiftfox)/i,
        /(icedragon|iceweasel|camino|chimera|fennec|maemo browser|minimo|conkeror|klar)[\/ ]?([\w\.\+]+)/i,
        // IceDragon/Iceweasel/Camino/Chimera/Fennec/Maemo/Minimo/Conkeror/Klar
        /(seamonkey|k-meleon|icecat|iceape|firebird|phoenix|palemoon|basilisk|waterfox)\/([-\w\.]+)$/i,
        // Firefox/SeaMonkey/K-Meleon/IceCat/IceApe/Firebird/Phoenix
        /(firefox)\/([\w\.]+)/i,
        /(mozilla)\/([\w\.]+) .+rv\:.+gecko\/\d+/i,
        // Other
        /(polaris|lynx|dillo|icab|doris|amaya|w3m|netsurf|sleipnir|obigo|mosaic|(?:go|ice|up)[\. ]?browser)[-\/ ]?v?([\w\.]+)/i,
        // Polaris/Lynx/Dillo/iCab/Doris/Amaya/w3m/NetSurf/Sleipnir/Obigo/Mosaic/Go/ICE/UP.Browser
        /(links) \(([\w\.]+)/i,
        /panasonic;(viera)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /(cobalt)\/([\w\.]+)/i
      ],
      [
        NAME,
        [
          VERSION,
          /[^\d\.]+./,
          EMPTY
        ]
      ]
    ]
  ],
  cpu: [
    [
      [
        /\b(?:(amd|x|x86[-_]?|wow|win)64)\b/i
      ],
      [
        [
          ARCHITECTURE,
          "amd64"
        ]
      ]
    ],
    [
      [
        /(ia32(?=;))/i,
        /((?:i[346]|x)86)[;\)]/i
      ],
      [
        [
          ARCHITECTURE,
          "ia32"
        ]
      ]
    ],
    [
      [
        /\b(aarch64|arm(v?8e?l?|_?64))\b/i
      ],
      [
        [
          ARCHITECTURE,
          "arm64"
        ]
      ]
    ],
    [
      [
        /windows (ce|mobile); ppc;/i
      ],
      [
        [
          ARCHITECTURE,
          "arm"
        ]
      ]
    ],
    [
      [
        /((?:ppc|powerpc)(?:64)?)(?: mac|;|\))/i
      ],
      [
        [
          ARCHITECTURE,
          /ower/,
          EMPTY,
          lowerize
        ]
      ]
    ],
    [
      [
        /(sun4\w)[;\)]/i
      ],
      [
        [
          ARCHITECTURE,
          "sparc"
        ]
      ]
    ],
    [
      [
        /((?:avr32|ia64(?=;))|68k(?=\))|\barm(?=v(?:[1-7]|[5-7]1)l?|;|eabi)|(?=atmel )avr|(?:irix|mips|sparc)(?:64)?\b|pa-risc)/i
      ],
      [
        [
          ARCHITECTURE,
          lowerize
        ]
      ]
    ]
  ],
  device: [
    [
      [
        /\b(sch-i[89]0\d|shw-m380s|sm-[ptx]\w{2,4}|gt-[pn]\d{2,4}|sgh-t8[56]9|nexus 10)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SAMSUNG
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /\b((?:s[cgp]h|gt|sm)-\w+|sc[g-]?[\d]+a?|galaxy nexus)/i,
        /samsung[- ]([-\w]+)/i,
        /sec-(sgh\w+)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SAMSUNG
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(?:\/|\()(ip(?:hone|od)[\w, ]*)(?:\/|;)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          APPLE
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\((ipad);[-\w\),; ]+apple/i,
        /applecoremedia\/[\w\.]+ \((ipad)/i,
        /\b(ipad)\d\d?,\d\d?[;\]].+ios/i
      ],
      [
        MODEL,
        [
          VENDOR,
          APPLE
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(macintosh);/i
      ],
      [
        MODEL,
        [
          VENDOR,
          APPLE
        ]
      ]
    ],
    [
      [
        /\b(sh-?[altvz]?\d\d[a-ekm]?)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SHARP
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\b((?:ag[rs][23]?|bah2?|sht?|btv)-a?[lw]\d{2})\b(?!.+d\/s)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          HUAWEI
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(?:huawei|honor)([-\w ]+)[;\)]/i,
        /\b(nexus 6p|\w{2,4}e?-[atu]?[ln][\dx][012359c][adn]?)\b(?!.+d\/s)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          HUAWEI
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\b(poco[\w ]+|m2\d{3}j\d\d[a-z]{2})(?: bui|\))/i,
        /\b; (\w+) build\/hm\1/i,
        /\b(hm[-_ ]?note?[_ ]?(?:\d\w)?) bui/i,
        /\b(redmi[\-_ ]?(?:note|k)?[\w_ ]+)(?: bui|\))/i,
        /\b(mi[-_ ]?(?:a\d|one|one[_ ]plus|note lte|max|cc)?[_ ]?(?:\d?\w?)[_ ]?(?:plus|se|lite)?)(?: bui|\))/i
      ],
      [
        [
          MODEL,
          /_/g,
          " "
        ],
        [
          VENDOR,
          XIAOMI
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\b(mi[-_ ]?(?:pad)(?:[\w_ ]+))(?: bui|\))/i
      ],
      [
        [
          MODEL,
          /_/g,
          " "
        ],
        [
          VENDOR,
          XIAOMI
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /; (\w+) bui.+ oppo/i,
        /\b(cph[12]\d{3}|p(?:af|c[al]|d\w|e[ar])[mt]\d0|x9007|a101op)\b/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "OPPO"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /vivo (\w+)(?: bui|\))/i,
        /\b(v[12]\d{3}\w?[at])(?: bui|;)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Vivo"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\b(rmx[12]\d{3})(?: bui|;|\))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Realme"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\b(milestone|droid(?:[2-4x]| (?:bionic|x2|pro|razr))?:?( 4g)?)\b[\w ]+build\//i,
        /\bmot(?:orola)?[- ](\w*)/i,
        /((?:moto[\w\(\) ]+|xt\d{3,4}|nexus 6)(?= bui|\)))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          MOTOROLA
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /\b(mz60\d|xoom[2 ]{0,2}) build\//i
      ],
      [
        MODEL,
        [
          VENDOR,
          MOTOROLA
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /((?=lg)?[vl]k\-?\d{3}) bui| 3\.[-\w; ]{10}lg?-([06cv9]{3,4})/i
      ],
      [
        MODEL,
        [
          VENDOR,
          LG
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(lm(?:-?f100[nv]?|-[\w\.]+)(?= bui|\))|nexus [45])/i,
        /\blg[-e;\/ ]+((?!browser|netcast|android tv)\w+)/i,
        /\blg-?([\d\w]+) bui/i
      ],
      [
        MODEL,
        [
          VENDOR,
          LG
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(ideatab[-\w ]+)/i,
        /lenovo ?(s[56]000[-\w]+|tab(?:[\w ]+)|yt[-\d\w]{6}|tb[-\d\w]{6})/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Lenovo"
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(?:maemo|nokia).*(n900|lumia \d+)/i,
        /nokia[-_ ]?([-\w\.]*)/i
      ],
      [
        [
          MODEL,
          /_/g,
          " "
        ],
        [
          VENDOR,
          "Nokia"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(pixel c)\b/i
      ],
      [
        MODEL,
        [
          VENDOR,
          GOOGLE
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /droid.+; (pixel[\daxl ]{0,6})(?: bui|\))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          GOOGLE
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /droid.+ (a?\d[0-2]{2}so|[c-g]\d{4}|so[-gl]\w+|xq-a\w[4-7][12])(?= bui|\).+chrome\/(?![1-6]{0,1}\d\.))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SONY
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /sony tablet [ps]/i,
        /\b(?:sony)?sgp\w+(?: bui|\))/i
      ],
      [
        [
          MODEL,
          "Xperia Tablet"
        ],
        [
          VENDOR,
          SONY
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        / (kb2005|in20[12]5|be20[12][59])\b/i,
        /(?:one)?(?:plus)? (a\d0\d\d)(?: b|\))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "OnePlus"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(alexa)webm/i,
        /(kf[a-z]{2}wi|aeo[c-r]{2})( bui|\))/i,
        /(kf[a-z]+)( bui|\)).+silk\//i
      ],
      [
        MODEL,
        [
          VENDOR,
          AMAZON
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /((?:sd|kf)[0349hijorstuw]+)( bui|\)).+silk\//i
      ],
      [
        [
          MODEL,
          /(.+)/g,
          "Fire Phone $1"
        ],
        [
          VENDOR,
          AMAZON
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(playbook);[-\w\),; ]+(rim)/i
      ],
      [
        MODEL,
        VENDOR,
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /\b((?:bb[a-f]|st[hv])100-\d)/i,
        /\(bb10; (\w+)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          BLACKBERRY
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(?:\b|asus_)(transfo[prime ]{4,10} \w+|eeepc|slider \w+|nexus 7|padfone|p00[cj])/i
      ],
      [
        MODEL,
        [
          VENDOR,
          ASUS
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        / (z[bes]6[027][012][km][ls]|zenfone \d\w?)\b/i
      ],
      [
        MODEL,
        [
          VENDOR,
          ASUS
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(nexus 9)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "HTC"
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(htc)[-;_ ]{1,2}([\w ]+(?=\)| bui)|\w+)/i,
        /(zte)[- ]([\w ]+?)(?: bui|\/|\))/i,
        /(alcatel|geeksphone|nexian|panasonic(?!(?:;|\.))|sony(?!-bra))[-_ ]?([-\w]*)/i
      ],
      [
        VENDOR,
        [
          MODEL,
          /_/g,
          " "
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /droid.+; ([ab][1-7]-?[0178a]\d\d?)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Acer"
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /droid.+; (m[1-5] note) bui/i,
        /\bmz-([-\w]{2,})/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Meizu"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(blackberry|benq|palm(?=\-)|sonyericsson|acer|asus|dell|meizu|motorola|polytron|infinix|tecno)[-_ ]?([-\w]*)/i,
        // BlackBerry/BenQ/Palm/Sony-Ericsson/Acer/Asus/Dell/Meizu/Motorola/Polytron
        /(hp) ([\w ]+\w)/i,
        /(asus)-?(\w+)/i,
        /(microsoft); (lumia[\w ]+)/i,
        /(lenovo)[-_ ]?([-\w]+)/i,
        /(jolla)/i,
        /(oppo) ?([\w ]+) bui/i
      ],
      [
        VENDOR,
        MODEL,
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(kobo)\s(ereader|touch)/i,
        /(archos) (gamepad2?)/i,
        /(hp).+(touchpad(?!.+tablet)|tablet)/i,
        /(kindle)\/([\w\.]+)/i
      ],
      [
        VENDOR,
        MODEL,
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(surface duo)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          MICROSOFT
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /droid [\d\.]+; (fp\du?)(?: b|\))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Fairphone"
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(shield[\w ]+) b/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Nvidia"
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(sprint) (\w+)/i
      ],
      [
        VENDOR,
        MODEL,
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(kin\.[onetw]{3})/i
      ],
      [
        [
          MODEL,
          /\./g,
          " "
        ],
        [
          VENDOR,
          MICROSOFT
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /droid.+; ([c6]+|et5[16]|mc[239][23]x?|vc8[03]x?)\)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          ZEBRA
        ],
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /droid.+; (ec30|ps20|tc[2-8]\d[kx])\)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          ZEBRA
        ],
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /smart-tv.+(samsung)/i
      ],
      [
        VENDOR,
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /hbbtv.+maple;(\d+)/i
      ],
      [
        [
          MODEL,
          /^/,
          "SmartTV"
        ],
        [
          VENDOR,
          SAMSUNG
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /(nux; netcast.+smarttv|lg (netcast\.tv-201\d|android tv))/i
      ],
      [
        [
          VENDOR,
          LG
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /(apple) ?tv/i
      ],
      [
        VENDOR,
        [
          MODEL,
          `${APPLE} TV`
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /crkey/i
      ],
      [
        [
          MODEL,
          `${CHROME}cast`
        ],
        [
          VENDOR,
          GOOGLE
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /droid.+aft(\w)( bui|\))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          AMAZON
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /\(dtv[\);].+(aquos)/i,
        /(aquos-tv[\w ]+)\)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SHARP
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /(bravia[\w ]+)( bui|\))/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SONY
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /(mitv-\w{5}) bui/i
      ],
      [
        MODEL,
        [
          VENDOR,
          XIAOMI
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /Hbbtv.*(technisat) (.*);/i
      ],
      [
        VENDOR,
        MODEL,
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /\b(roku)[\dx]*[\)\/]((?:dvp-)?[\d\.]*)/i,
        /hbbtv\/\d+\.\d+\.\d+ +\([\w\+ ]*; *([\w\d][^;]*);([^;]*)/i
      ],
      [
        [
          VENDOR,
          trim
        ],
        [
          MODEL,
          trim
        ],
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /\b(android tv|smart[- ]?tv|opera tv|tv; rv:)\b/i
      ],
      [
        [
          TYPE,
          SMARTTV
        ]
      ]
    ],
    [
      [
        /(ouya)/i,
        /(nintendo) (\w+)/i
      ],
      [
        VENDOR,
        MODEL,
        [
          TYPE,
          CONSOLE
        ]
      ]
    ],
    [
      [
        /droid.+; (shield) bui/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Nvidia"
        ],
        [
          TYPE,
          CONSOLE
        ]
      ]
    ],
    [
      [
        /(playstation \w+)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          SONY
        ],
        [
          TYPE,
          CONSOLE
        ]
      ]
    ],
    [
      [
        /\b(xbox(?: one)?(?!; xbox))[\); ]/i
      ],
      [
        MODEL,
        [
          VENDOR,
          MICROSOFT
        ],
        [
          TYPE,
          CONSOLE
        ]
      ]
    ],
    [
      [
        /((pebble))app/i
      ],
      [
        VENDOR,
        MODEL,
        [
          TYPE,
          WEARABLE
        ]
      ]
    ],
    [
      [
        /(watch)(?: ?os[,\/]|\d,\d\/)[\d\.]+/i
      ],
      [
        MODEL,
        [
          VENDOR,
          APPLE
        ],
        [
          TYPE,
          WEARABLE
        ]
      ]
    ],
    [
      [
        /droid.+; (glass) \d/i
      ],
      [
        MODEL,
        [
          VENDOR,
          GOOGLE
        ],
        [
          TYPE,
          WEARABLE
        ]
      ]
    ],
    [
      [
        /droid.+; (wt63?0{2,3})\)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          ZEBRA
        ],
        [
          TYPE,
          WEARABLE
        ]
      ]
    ],
    [
      [
        /(quest( 2| pro)?)/i
      ],
      [
        MODEL,
        [
          VENDOR,
          FACEBOOK
        ],
        [
          TYPE,
          WEARABLE
        ]
      ]
    ],
    [
      [
        /(tesla)(?: qtcarbrowser|\/[-\w\.]+)/i
      ],
      [
        VENDOR,
        [
          TYPE,
          EMBEDDED
        ]
      ]
    ],
    [
      [
        /(aeobc)\b/i
      ],
      [
        MODEL,
        [
          VENDOR,
          AMAZON
        ],
        [
          TYPE,
          EMBEDDED
        ]
      ]
    ],
    [
      [
        /droid .+?; ([^;]+?)(?: bui|\) applew).+? mobile safari/i
      ],
      [
        MODEL,
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /droid .+?; ([^;]+?)(?: bui|\) applew).+?(?! mobile) safari/i
      ],
      [
        MODEL,
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /\b((tablet|tab)[;\/]|focus\/\d(?!.+mobile))/i
      ],
      [
        [
          TYPE,
          TABLET
        ]
      ]
    ],
    [
      [
        /(phone|mobile(?:[;\/]| [ \w\/\.]*safari)|pda(?=.+windows ce))/i
      ],
      [
        [
          TYPE,
          MOBILE
        ]
      ]
    ],
    [
      [
        /(android[-\w\. ]{0,9});.+buil/i
      ],
      [
        MODEL,
        [
          VENDOR,
          "Generic"
        ]
      ]
    ]
  ],
  engine: [
    [
      [
        /windows.+ edge\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${EDGE}HTML`
        ]
      ]
    ],
    [
      [
        /webkit\/537\.36.+chrome\/(?!27)([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Blink"
        ]
      ]
    ],
    [
      [
        /(presto)\/([\w\.]+)/i,
        /(webkit|trident|netfront|netsurf|amaya|lynx|w3m|goanna)\/([\w\.]+)/i,
        /ekioh(flow)\/([\w\.]+)/i,
        /(khtml|tasman|links)[\/ ]\(?([\w\.]+)/i,
        /(icab)[\/ ]([23]\.[\d\.]+)/i,
        /\b(libweb)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /rv\:([\w\.]{1,9})\b.+(gecko)/i
      ],
      [
        VERSION,
        NAME
      ]
    ]
  ],
  os: [
    [
      [
        /microsoft (windows) (vista|xp)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /(windows) nt 6\.2; (arm)/i,
        /(windows (?:phone(?: os)?|mobile))[\/ ]?([\d\.\w ]*)/i,
        /(windows)[\/ ]?([ntce\d\. ]+\w)(?!.+xbox)/i
      ],
      [
        NAME,
        [
          VERSION,
          mapWinVer
        ]
      ]
    ],
    [
      [
        /(win(?=3|9|n)|win 9x )([nt\d\.]+)/i
      ],
      [
        [
          NAME,
          WINDOWS
        ],
        [
          VERSION,
          mapWinVer
        ]
      ]
    ],
    [
      [
        /ip[honead]{2,4}\b(?:.*os ([\w]+) like mac|; opera)/i,
        /(?:ios;fbsv\/|iphone.+ios[\/ ])([\d\.]+)/i,
        /cfnetwork\/.+darwin/i
      ],
      [
        [
          VERSION,
          /_/g,
          "."
        ],
        [
          NAME,
          "iOS"
        ]
      ]
    ],
    [
      [
        /(mac os x) ?([\w\. ]*)/i,
        /(macintosh|mac_powerpc\b)(?!.+haiku)/i
      ],
      [
        [
          NAME,
          "macOS"
        ],
        [
          VERSION,
          /_/g,
          "."
        ]
      ]
    ],
    [
      [
        /droid ([\w\.]+)\b.+(android[- ]x86|harmonyos)/i
      ],
      [
        VERSION,
        NAME
      ]
    ],
    [
      [
        /(android|webos|qnx|bada|rim tablet os|maemo|meego|sailfish)[-\/ ]?([\w\.]*)/i,
        /(blackberry)\w*\/([\w\.]*)/i,
        /(tizen|kaios)[\/ ]([\w\.]+)/i,
        /\((series40);/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /\(bb(10);/i
      ],
      [
        VERSION,
        [
          NAME,
          BLACKBERRY
        ]
      ]
    ],
    [
      [
        /(?:symbian ?os|symbos|s60(?=;)|series60)[-\/ ]?([\w\.]*)/i
      ],
      [
        VERSION,
        [
          NAME,
          "Symbian"
        ]
      ]
    ],
    [
      [
        /mozilla\/[\d\.]+ \((?:mobile|tablet|tv|mobile; [\w ]+); rv:.+ gecko\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${FIREFOX} OS`
        ]
      ]
    ],
    [
      [
        /web0s;.+rt(tv)/i,
        /\b(?:hp)?wos(?:browser)?\/([\w\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "webOS"
        ]
      ]
    ],
    [
      [
        /watch(?: ?os[,\/]|\d,\d\/)([\d\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          "watchOS"
        ]
      ]
    ],
    [
      [
        /crkey\/([\d\.]+)/i
      ],
      [
        VERSION,
        [
          NAME,
          `${CHROME}cast`
        ]
      ]
    ],
    [
      [
        /(cros) [\w]+(?:\)| ([\w\.]+)\b)/i
      ],
      [
        [
          NAME,
          "Chrome OS"
        ],
        VERSION
      ]
    ],
    [
      [
        /panasonic;(viera)/i,
        /(netrange)mmh/i,
        /(nettv)\/(\d+\.[\w\.]+)/i,
        // Console
        /(nintendo|playstation) (\w+)/i,
        /(xbox); +xbox ([^\);]+)/i,
        // Other
        /\b(joli|palm)\b ?(?:os)?\/?([\w\.]*)/i,
        /(mint)[\/\(\) ]?(\w*)/i,
        /(mageia|vectorlinux)[; ]/i,
        /([kxln]?ubuntu|debian|suse|opensuse|gentoo|arch(?= linux)|slackware|fedora|mandriva|centos|pclinuxos|red ?hat|zenwalk|linpus|raspbian|plan 9|minix|risc os|contiki|deepin|manjaro|elementary os|sabayon|linspire)(?: gnu\/linux)?(?: enterprise)?(?:[- ]linux)?(?:-gnu)?[-\/ ]?(?!chrom|package)([-\w\.]*)/i,
        // Ubuntu/Debian/SUSE/Gentoo/Arch/Slackware/Fedora/Mandriva/CentOS/PCLinuxOS/RedHat/Zenwalk/Linpus/Raspbian/Plan9/Minix/RISCOS/Contiki/Deepin/Manjaro/elementary/Sabayon/Linspire
        /(hurd|linux) ?([\w\.]*)/i,
        /(gnu) ?([\w\.]*)/i,
        /\b([-frentopcghs]{0,5}bsd|dragonfly)[\/ ]?(?!amd|[ix346]{1,2}86)([\w\.]*)/i,
        /(haiku) (\w+)/i
      ],
      [
        NAME,
        VERSION
      ]
    ],
    [
      [
        /(sunos) ?([\w\.\d]*)/i
      ],
      [
        [
          NAME,
          "Solaris"
        ],
        VERSION
      ]
    ],
    [
      [
        /((?:open)?solaris)[-\/ ]?([\w\.]*)/i,
        /(aix) ((\d)(?=\.|\)| )[\w\.])*/i,
        /\b(beos|os\/2|amigaos|morphos|openvms|fuchsia|hp-ux|serenityos)/i,
        /(unix) ?([\w\.]*)/i
      ],
      [
        NAME,
        VERSION
      ]
    ]
  ]
};
_computedKey = Symbol.for("Deno.customInspect"), _computedKey1 = Symbol.for("nodejs.util.inspect.custom");
/**
 * A representation of user agent string, which can be used to determine
 * environmental information represented by the string. All properties are
 * determined lazily.
 */ export class UserAgent {
  #browser;
  #cpu;
  #device;
  #engine;
  #os;
  #ua;
  /**
   * Constructs a new instance.
   *
   * @example
   * ```ts
   * import { UserAgent } from "https://deno.land/std@$STD_VERSION/http/user_agent.ts";
   *
   * Deno.serve((req) => {
   *   const userAgent = new UserAgent(req.headers.get("user-agent") ?? "");
   *   return new Response(`Hello, ${userAgent.browser.name}
   *     on ${userAgent.os.name} ${userAgent.os.version}!`);
   * });
   * ```
   */ constructor(ua){
    this.#ua = ua ?? "";
  }
  /**
   * The name and version of the browser extracted from the user agent
   * string.
   */ get browser() {
    if (!this.#browser) {
      this.#browser = {
        name: undefined,
        version: undefined,
        major: undefined
      };
      mapper(this.#browser, this.#ua, matchers.browser);
      // deno-lint-ignore no-explicit-any
      this.#browser.major = majorize(this.#browser.version);
      Object.freeze(this.#browser);
    }
    return this.#browser;
  }
  /** The architecture of the CPU extracted from the user agent string. */ get cpu() {
    if (!this.#cpu) {
      this.#cpu = {
        architecture: undefined
      };
      mapper(this.#cpu, this.#ua, matchers.cpu);
      Object.freeze(this.#cpu);
    }
    return this.#cpu;
  }
  /**
   * The model, type, and vendor of a device if present in a user agent
   * string.
   */ get device() {
    if (!this.#device) {
      this.#device = {
        model: undefined,
        type: undefined,
        vendor: undefined
      };
      mapper(this.#device, this.#ua, matchers.device);
      Object.freeze(this.#device);
    }
    return this.#device;
  }
  /** The name and version of the browser engine in a user agent string. */ get engine() {
    if (!this.#engine) {
      this.#engine = {
        name: undefined,
        version: undefined
      };
      mapper(this.#engine, this.#ua, matchers.engine);
      Object.freeze(this.#engine);
    }
    return this.#engine;
  }
  /** The name and version of the operating system in a user agent string. */ get os() {
    if (!this.#os) {
      this.#os = {
        name: undefined,
        version: undefined
      };
      mapper(this.#os, this.#ua, matchers.os);
      Object.freeze(this.#os);
    }
    return this.#os;
  }
  /** A read only version of the user agent string related to the instance. */ get ua() {
    return this.#ua;
  }
  /** Converts the current instance to a JSON representation. */ toJSON() {
    const { browser, cpu, device, engine, os, ua } = this;
    return {
      browser,
      cpu,
      device,
      engine,
      os,
      ua
    };
  }
  /** Converts the current instance to a string. */ toString() {
    return this.#ua;
  }
  /** Custom output for {@linkcode Deno.inspect}. */ [_computedKey](inspect) {
    const { browser, cpu, device, engine, os, ua } = this;
    return `${this.constructor.name} ${inspect({
      browser,
      cpu,
      device,
      engine,
      os,
      ua
    })}`;
  }
  /**
   * Custom output for Node's
   * {@linkcode https://nodejs.org/api/util.html#utilinspectobject-options | util.inspect}.
   */ [_computedKey1](depth, // deno-lint-ignore no-explicit-any
  options, inspect) {
    if (depth < 0) {
      return options.stylize(`[${this.constructor.name}]`, "special");
    }
    const newOptions = Object.assign({}, options, {
      depth: options.depth === null ? null : options.depth - 1
    });
    const { browser, cpu, device, engine, os, ua } = this;
    return `${options.stylize(this.constructor.name, "special")} ${inspect({
      browser,
      cpu,
      device,
      engine,
      os,
      ua
    }, newOptions)}`;
  }
}
//# sourceMappingURL=data:application/json;base64,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