// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { ALL } from "./constants.ts";
import { OPERATOR_XRANGE_REGEXP, XRANGE } from "./_shared.ts";
import { parseComparator } from "./_parse_comparator.ts";
import { parseBuild, parsePrerelease } from "./_shared.ts";
function isWildcard(id) {
  return !id || id.toLowerCase() === "x" || id === "*";
}
function parseHyphenRange(range) {
  // remove spaces between comparator and groups
  range = range.replace(/(?<=<|>|=) +/, "");
  const leftMatch = range.match(new RegExp(`^${XRANGE}`));
  const leftGroup = leftMatch?.groups;
  if (!leftGroup) return range.split(/\s+/);
  const leftLength = leftMatch[0].length;
  const hyphenMatch = range.slice(leftLength).match(/^\s+-\s+/);
  if (!hyphenMatch) return range.split(/\s+/);
  const hyphenLength = hyphenMatch[0].length;
  const rightMatch = range.slice(leftLength + hyphenLength).match(new RegExp(`^${XRANGE}\\s*$`));
  const rightGroups = rightMatch?.groups;
  if (!rightGroups) return range.split(/\s+/);
  let from = leftMatch[0];
  let to = rightMatch[0];
  if (isWildcard(leftGroup.major)) {
    from = "";
  } else if (isWildcard(leftGroup.minor)) {
    from = `>=${leftGroup.major}.0.0`;
  } else if (isWildcard(leftGroup.patch)) {
    from = `>=${leftGroup.major}.${leftGroup.minor}.0`;
  } else {
    from = `>=${from}`;
  }
  if (isWildcard(rightGroups.major)) {
    to = "";
  } else if (isWildcard(rightGroups.minor)) {
    to = `<${+rightGroups.major + 1}.0.0`;
  } else if (isWildcard(rightGroups.patch)) {
    to = `<${rightGroups.major}.${+rightGroups.minor + 1}.0`;
  } else if (rightGroups.prerelease) {
    to = `<=${rightGroups.major}.${rightGroups.minor}.${rightGroups.patch}-${rightGroups.prerelease}`;
  } else {
    to = `<=${to}`;
  }
  return [
    from,
    to
  ];
}
function handleCaretOperator(groups) {
  const majorIsWildcard = isWildcard(groups.major);
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (majorIsWildcard) return [
    ALL
  ];
  if (minorIsWildcard) {
    return [
      {
        operator: ">=",
        major,
        minor: 0,
        patch: 0
      },
      {
        operator: "<",
        major: major + 1,
        minor: 0,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) {
    if (major === 0) {
      return [
        {
          operator: ">=",
          major,
          minor,
          patch: 0
        },
        {
          operator: "<",
          major,
          minor: minor + 1,
          patch: 0
        }
      ];
    }
    return [
      {
        operator: ">=",
        major,
        minor,
        patch: 0
      },
      {
        operator: "<",
        major: major + 1,
        minor: 0,
        patch: 0
      }
    ];
  }
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  if (major === 0) {
    if (minor === 0) {
      return [
        {
          operator: ">=",
          major,
          minor,
          patch,
          prerelease
        },
        {
          operator: "<",
          major,
          minor,
          patch: patch + 1
        }
      ];
    }
    return [
      {
        operator: ">=",
        major,
        minor,
        patch,
        prerelease
      },
      {
        operator: "<",
        major,
        minor: minor + 1,
        patch: 0
      }
    ];
  }
  return [
    {
      operator: ">=",
      major,
      minor,
      patch,
      prerelease
    },
    {
      operator: "<",
      major: major + 1,
      minor: 0,
      patch: 0
    }
  ];
}
function handleTildeOperator(groups) {
  const majorIsWildcard = isWildcard(groups.major);
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (majorIsWildcard) return [
    ALL
  ];
  if (minorIsWildcard) {
    return [
      {
        operator: ">=",
        major,
        minor: 0,
        patch: 0
      },
      {
        operator: "<",
        major: major + 1,
        minor: 0,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) {
    return [
      {
        operator: ">=",
        major,
        minor,
        patch: 0
      },
      {
        operator: "<",
        major,
        minor: minor + 1,
        patch: 0
      }
    ];
  }
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  return [
    {
      operator: ">=",
      major,
      minor,
      patch,
      prerelease
    },
    {
      operator: "<",
      major,
      minor: minor + 1,
      patch: 0
    }
  ];
}
function handleLessThanOperator(groups) {
  const majorIsWildcard = isWildcard(groups.major);
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (majorIsWildcard) return [
    {
      operator: "<",
      major: 0,
      minor: 0,
      patch: 0
    }
  ];
  if (minorIsWildcard) {
    if (patchIsWildcard) return [
      {
        operator: "<",
        major,
        minor: 0,
        patch: 0
      }
    ];
    return [
      {
        operator: "<",
        major,
        minor,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) return [
    {
      operator: "<",
      major,
      minor,
      patch: 0
    }
  ];
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  const build = parseBuild(groups.build ?? "");
  return [
    {
      operator: "<",
      major,
      minor,
      patch,
      prerelease,
      build
    }
  ];
}
function handleLessThanOrEqualOperator(groups) {
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (minorIsWildcard) {
    if (patchIsWildcard) {
      return [
        {
          operator: "<",
          major: major + 1,
          minor: 0,
          patch: 0
        }
      ];
    }
    return [
      {
        operator: "<",
        major,
        minor: minor + 1,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) {
    return [
      {
        operator: "<",
        major,
        minor: minor + 1,
        patch: 0
      }
    ];
  }
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  const build = parseBuild(groups.build ?? "");
  return [
    {
      operator: "<=",
      major,
      minor,
      patch,
      prerelease,
      build
    }
  ];
}
function handleGreaterThanOperator(groups) {
  const majorIsWildcard = isWildcard(groups.major);
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (majorIsWildcard) return [
    {
      operator: "<",
      major: 0,
      minor: 0,
      patch: 0
    }
  ];
  if (minorIsWildcard) {
    return [
      {
        operator: ">=",
        major: major + 1,
        minor: 0,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) {
    return [
      {
        operator: ">=",
        major,
        minor: minor + 1,
        patch: 0
      }
    ];
  }
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  const build = parseBuild(groups.build ?? "");
  return [
    {
      operator: ">",
      major,
      minor,
      patch,
      prerelease,
      build
    }
  ];
}
function handleGreaterOrEqualOperator(groups) {
  const majorIsWildcard = isWildcard(groups.major);
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (majorIsWildcard) return [
    ALL
  ];
  if (minorIsWildcard) {
    if (patchIsWildcard) return [
      {
        operator: ">=",
        major,
        minor: 0,
        patch: 0
      }
    ];
    return [
      {
        operator: ">=",
        major,
        minor,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) return [
    {
      operator: ">=",
      major,
      minor,
      patch: 0
    }
  ];
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  const build = parseBuild(groups.build ?? "");
  return [
    {
      operator: ">=",
      major,
      minor,
      patch,
      prerelease,
      build
    }
  ];
}
function handleEqualOperator(groups) {
  const majorIsWildcard = isWildcard(groups.major);
  const minorIsWildcard = isWildcard(groups.minor);
  const patchIsWildcard = isWildcard(groups.patch);
  const major = +groups.major;
  const minor = +groups.minor;
  const patch = +groups.patch;
  if (majorIsWildcard) return [
    ALL
  ];
  if (minorIsWildcard) {
    return [
      {
        operator: ">=",
        major,
        minor: 0,
        patch: 0
      },
      {
        operator: "<",
        major: major + 1,
        minor: 0,
        patch: 0
      }
    ];
  }
  if (patchIsWildcard) {
    return [
      {
        operator: ">=",
        major,
        minor,
        patch: 0
      },
      {
        operator: "<",
        major,
        minor: minor + 1,
        patch: 0
      }
    ];
  }
  const prerelease = parsePrerelease(groups.prerelease ?? "");
  const build = parseBuild(groups.build ?? "");
  return [
    {
      operator: undefined,
      major,
      minor,
      patch,
      prerelease,
      build
    }
  ];
}
function parseRangeString(string) {
  const groups = string.match(OPERATOR_XRANGE_REGEXP)?.groups;
  if (!groups) return parseComparator(string);
  switch(groups.operator){
    case "^":
      return handleCaretOperator(groups);
    case "~":
    case "~>":
      return handleTildeOperator(groups);
    case "<":
      return handleLessThanOperator(groups);
    case "<=":
      return handleLessThanOrEqualOperator(groups);
    case ">":
      return handleGreaterThanOperator(groups);
    case ">=":
      return handleGreaterOrEqualOperator(groups);
    case "=":
    case "":
      return handleEqualOperator(groups);
    default:
      throw new Error(`'${groups.operator}' is not a valid operator.`);
  }
}
/**
 * Parses a range string into a Range object or throws a TypeError.
 * @param range The range set string
 * @returns A valid semantic range
 */ export function parseRange(range) {
  const ranges = range.split(/\s*\|\|\s*/).map((range)=>parseHyphenRange(range).flatMap(parseRangeString));
  Object.defineProperty(ranges, "ranges", {
    value: ranges
  });
  return ranges;
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjIxNy4wL3NlbXZlci9wYXJzZV9yYW5nZS50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgMjAxOC0yMDI0IHRoZSBEZW5vIGF1dGhvcnMuIEFsbCByaWdodHMgcmVzZXJ2ZWQuIE1JVCBsaWNlbnNlLlxuaW1wb3J0IHsgQUxMIH0gZnJvbSBcIi4vY29uc3RhbnRzLnRzXCI7XG5pbXBvcnQgdHlwZSB7IENvbXBhcmF0b3IsIFJhbmdlIH0gZnJvbSBcIi4vdHlwZXMudHNcIjtcbmltcG9ydCB7IE9QRVJBVE9SX1hSQU5HRV9SRUdFWFAsIFhSQU5HRSB9IGZyb20gXCIuL19zaGFyZWQudHNcIjtcbmltcG9ydCB7IHBhcnNlQ29tcGFyYXRvciB9IGZyb20gXCIuL19wYXJzZV9jb21wYXJhdG9yLnRzXCI7XG5pbXBvcnQgeyBwYXJzZUJ1aWxkLCBwYXJzZVByZXJlbGVhc2UgfSBmcm9tIFwiLi9fc2hhcmVkLnRzXCI7XG5cbmZ1bmN0aW9uIGlzV2lsZGNhcmQoaWQ/OiBzdHJpbmcpOiBib29sZWFuIHtcbiAgcmV0dXJuICFpZCB8fCBpZC50b0xvd2VyQ2FzZSgpID09PSBcInhcIiB8fCBpZCA9PT0gXCIqXCI7XG59XG5cbnR5cGUgUmVnRXhwR3JvdXBzID0ge1xuICBvcGVyYXRvcjogc3RyaW5nO1xuICBtYWpvcjogc3RyaW5nO1xuICBtaW5vcjogc3RyaW5nO1xuICBwYXRjaDogc3RyaW5nO1xuICBwcmVyZWxlYXNlPzogc3RyaW5nO1xuICBidWlsZD86IHN0cmluZztcbn07XG5cbmZ1bmN0aW9uIHBhcnNlSHlwaGVuUmFuZ2UocmFuZ2U6IHN0cmluZykge1xuICAvLyByZW1vdmUgc3BhY2VzIGJldHdlZW4gY29tcGFyYXRvciBhbmQgZ3JvdXBzXG4gIHJhbmdlID0gcmFuZ2UucmVwbGFjZSgvKD88PTx8Pnw9KSArLywgXCJcIik7XG5cbiAgY29uc3QgbGVmdE1hdGNoID0gcmFuZ2UubWF0Y2gobmV3IFJlZ0V4cChgXiR7WFJBTkdFfWApKTtcbiAgY29uc3QgbGVmdEdyb3VwID0gbGVmdE1hdGNoPy5ncm91cHM7XG4gIGlmICghbGVmdEdyb3VwKSByZXR1cm4gcmFuZ2Uuc3BsaXQoL1xccysvKTtcbiAgY29uc3QgbGVmdExlbmd0aCA9IGxlZnRNYXRjaFswXS5sZW5ndGg7XG4gIGNvbnN0IGh5cGhlbk1hdGNoID0gcmFuZ2Uuc2xpY2UobGVmdExlbmd0aCkubWF0Y2goL15cXHMrLVxccysvKTtcbiAgaWYgKCFoeXBoZW5NYXRjaCkgcmV0dXJuIHJhbmdlLnNwbGl0KC9cXHMrLyk7XG4gIGNvbnN0IGh5cGhlbkxlbmd0aCA9IGh5cGhlbk1hdGNoWzBdLmxlbmd0aDtcbiAgY29uc3QgcmlnaHRNYXRjaCA9IHJhbmdlLnNsaWNlKGxlZnRMZW5ndGggKyBoeXBoZW5MZW5ndGgpLm1hdGNoKFxuICAgIG5ldyBSZWdFeHAoYF4ke1hSQU5HRX1cXFxccyokYCksXG4gICk7XG4gIGNvbnN0IHJpZ2h0R3JvdXBzID0gcmlnaHRNYXRjaD8uZ3JvdXBzO1xuICBpZiAoIXJpZ2h0R3JvdXBzKSByZXR1cm4gcmFuZ2Uuc3BsaXQoL1xccysvKTtcbiAgbGV0IGZyb20gPSBsZWZ0TWF0Y2hbMF07XG4gIGxldCB0byA9IHJpZ2h0TWF0Y2hbMF07XG5cbiAgaWYgKGlzV2lsZGNhcmQobGVmdEdyb3VwLm1ham9yKSkge1xuICAgIGZyb20gPSBcIlwiO1xuICB9IGVsc2UgaWYgKGlzV2lsZGNhcmQobGVmdEdyb3VwLm1pbm9yKSkge1xuICAgIGZyb20gPSBgPj0ke2xlZnRHcm91cC5tYWpvcn0uMC4wYDtcbiAgfSBlbHNlIGlmIChpc1dpbGRjYXJkKGxlZnRHcm91cC5wYXRjaCkpIHtcbiAgICBmcm9tID0gYD49JHtsZWZ0R3JvdXAubWFqb3J9LiR7bGVmdEdyb3VwLm1pbm9yfS4wYDtcbiAgfSBlbHNlIHtcbiAgICBmcm9tID0gYD49JHtmcm9tfWA7XG4gIH1cblxuICBpZiAoaXNXaWxkY2FyZChyaWdodEdyb3Vwcy5tYWpvcikpIHtcbiAgICB0byA9IFwiXCI7XG4gIH0gZWxzZSBpZiAoaXNXaWxkY2FyZChyaWdodEdyb3Vwcy5taW5vcikpIHtcbiAgICB0byA9IGA8JHsrcmlnaHRHcm91cHMubWFqb3IhICsgMX0uMC4wYDtcbiAgfSBlbHNlIGlmIChpc1dpbGRjYXJkKHJpZ2h0R3JvdXBzLnBhdGNoKSkge1xuICAgIHRvID0gYDwke3JpZ2h0R3JvdXBzLm1ham9yfS4keytyaWdodEdyb3Vwcy5taW5vciEgKyAxfS4wYDtcbiAgfSBlbHNlIGlmIChyaWdodEdyb3Vwcy5wcmVyZWxlYXNlKSB7XG4gICAgdG8gPVxuICAgICAgYDw9JHtyaWdodEdyb3Vwcy5tYWpvcn0uJHtyaWdodEdyb3Vwcy5taW5vcn0uJHtyaWdodEdyb3Vwcy5wYXRjaH0tJHtyaWdodEdyb3Vwcy5wcmVyZWxlYXNlfWA7XG4gIH0gZWxzZSB7XG4gICAgdG8gPSBgPD0ke3RvfWA7XG4gIH1cblxuICByZXR1cm4gW2Zyb20sIHRvXTtcbn1cbmZ1bmN0aW9uIGhhbmRsZUNhcmV0T3BlcmF0b3IoZ3JvdXBzOiBSZWdFeHBHcm91cHMpOiBDb21wYXJhdG9yW10ge1xuICBjb25zdCBtYWpvcklzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5tYWpvcik7XG4gIGNvbnN0IG1pbm9ySXNXaWxkY2FyZCA9IGlzV2lsZGNhcmQoZ3JvdXBzLm1pbm9yKTtcbiAgY29uc3QgcGF0Y2hJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMucGF0Y2gpO1xuXG4gIGNvbnN0IG1ham9yID0gK2dyb3Vwcy5tYWpvcjtcbiAgY29uc3QgbWlub3IgPSArZ3JvdXBzLm1pbm9yO1xuICBjb25zdCBwYXRjaCA9ICtncm91cHMucGF0Y2g7XG5cbiAgaWYgKG1ham9ySXNXaWxkY2FyZCkgcmV0dXJuIFtBTExdO1xuICBpZiAobWlub3JJc1dpbGRjYXJkKSB7XG4gICAgcmV0dXJuIFtcbiAgICAgIHsgb3BlcmF0b3I6IFwiPj1cIiwgbWFqb3IsIG1pbm9yOiAwLCBwYXRjaDogMCB9LFxuICAgICAgeyBvcGVyYXRvcjogXCI8XCIsIG1ham9yOiBtYWpvciArIDEsIG1pbm9yOiAwLCBwYXRjaDogMCB9LFxuICAgIF07XG4gIH1cbiAgaWYgKHBhdGNoSXNXaWxkY2FyZCkge1xuICAgIGlmIChtYWpvciA9PT0gMCkge1xuICAgICAgcmV0dXJuIFtcbiAgICAgICAgeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3IsIHBhdGNoOiAwIH0sXG4gICAgICAgIHsgb3BlcmF0b3I6IFwiPFwiLCBtYWpvciwgbWlub3I6IG1pbm9yICsgMSwgcGF0Y2g6IDAgfSxcbiAgICAgIF07XG4gICAgfVxuICAgIHJldHVybiBbXG4gICAgICB7IG9wZXJhdG9yOiBcIj49XCIsIG1ham9yLCBtaW5vciwgcGF0Y2g6IDAgfSxcbiAgICAgIHsgb3BlcmF0b3I6IFwiPFwiLCBtYWpvcjogbWFqb3IgKyAxLCBtaW5vcjogMCwgcGF0Y2g6IDAgfSxcbiAgICBdO1xuICB9XG5cbiAgY29uc3QgcHJlcmVsZWFzZSA9IHBhcnNlUHJlcmVsZWFzZShncm91cHMucHJlcmVsZWFzZSA/PyBcIlwiKTtcbiAgaWYgKG1ham9yID09PSAwKSB7XG4gICAgaWYgKG1pbm9yID09PSAwKSB7XG4gICAgICByZXR1cm4gW1xuICAgICAgICB7IG9wZXJhdG9yOiBcIj49XCIsIG1ham9yLCBtaW5vciwgcGF0Y2gsIHByZXJlbGVhc2UgfSxcbiAgICAgICAgeyBvcGVyYXRvcjogXCI8XCIsIG1ham9yLCBtaW5vciwgcGF0Y2g6IHBhdGNoICsgMSB9LFxuICAgICAgXTtcbiAgICB9XG4gICAgcmV0dXJuIFtcbiAgICAgIHsgb3BlcmF0b3I6IFwiPj1cIiwgbWFqb3IsIG1pbm9yLCBwYXRjaCwgcHJlcmVsZWFzZSB9LFxuICAgICAgeyBvcGVyYXRvcjogXCI8XCIsIG1ham9yLCBtaW5vcjogbWlub3IgKyAxLCBwYXRjaDogMCB9LFxuICAgIF07XG4gIH1cbiAgcmV0dXJuIFtcbiAgICB7IG9wZXJhdG9yOiBcIj49XCIsIG1ham9yLCBtaW5vciwgcGF0Y2gsIHByZXJlbGVhc2UgfSxcbiAgICB7IG9wZXJhdG9yOiBcIjxcIiwgbWFqb3I6IG1ham9yICsgMSwgbWlub3I6IDAsIHBhdGNoOiAwIH0sXG4gIF07XG59XG5mdW5jdGlvbiBoYW5kbGVUaWxkZU9wZXJhdG9yKGdyb3VwczogUmVnRXhwR3JvdXBzKTogQ29tcGFyYXRvcltdIHtcbiAgY29uc3QgbWFqb3JJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMubWFqb3IpO1xuICBjb25zdCBtaW5vcklzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5taW5vcik7XG4gIGNvbnN0IHBhdGNoSXNXaWxkY2FyZCA9IGlzV2lsZGNhcmQoZ3JvdXBzLnBhdGNoKTtcblxuICBjb25zdCBtYWpvciA9ICtncm91cHMubWFqb3I7XG4gIGNvbnN0IG1pbm9yID0gK2dyb3Vwcy5taW5vcjtcbiAgY29uc3QgcGF0Y2ggPSArZ3JvdXBzLnBhdGNoO1xuXG4gIGlmIChtYWpvcklzV2lsZGNhcmQpIHJldHVybiBbQUxMXTtcbiAgaWYgKG1pbm9ySXNXaWxkY2FyZCkge1xuICAgIHJldHVybiBbXG4gICAgICB7IG9wZXJhdG9yOiBcIj49XCIsIG1ham9yLCBtaW5vcjogMCwgcGF0Y2g6IDAgfSxcbiAgICAgIHsgb3BlcmF0b3I6IFwiPFwiLCBtYWpvcjogbWFqb3IgKyAxLCBtaW5vcjogMCwgcGF0Y2g6IDAgfSxcbiAgICBdO1xuICB9XG4gIGlmIChwYXRjaElzV2lsZGNhcmQpIHtcbiAgICByZXR1cm4gW1xuICAgICAgeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3IsIHBhdGNoOiAwIH0sXG4gICAgICB7IG9wZXJhdG9yOiBcIjxcIiwgbWFqb3IsIG1pbm9yOiBtaW5vciArIDEsIHBhdGNoOiAwIH0sXG4gICAgXTtcbiAgfVxuICBjb25zdCBwcmVyZWxlYXNlID0gcGFyc2VQcmVyZWxlYXNlKGdyb3Vwcy5wcmVyZWxlYXNlID8/IFwiXCIpO1xuICByZXR1cm4gW1xuICAgIHsgb3BlcmF0b3I6IFwiPj1cIiwgbWFqb3IsIG1pbm9yLCBwYXRjaCwgcHJlcmVsZWFzZSB9LFxuICAgIHsgb3BlcmF0b3I6IFwiPFwiLCBtYWpvciwgbWlub3I6IG1pbm9yICsgMSwgcGF0Y2g6IDAgfSxcbiAgXTtcbn1cbmZ1bmN0aW9uIGhhbmRsZUxlc3NUaGFuT3BlcmF0b3IoZ3JvdXBzOiBSZWdFeHBHcm91cHMpOiBDb21wYXJhdG9yW10ge1xuICBjb25zdCBtYWpvcklzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5tYWpvcik7XG4gIGNvbnN0IG1pbm9ySXNXaWxkY2FyZCA9IGlzV2lsZGNhcmQoZ3JvdXBzLm1pbm9yKTtcbiAgY29uc3QgcGF0Y2hJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMucGF0Y2gpO1xuXG4gIGNvbnN0IG1ham9yID0gK2dyb3Vwcy5tYWpvcjtcbiAgY29uc3QgbWlub3IgPSArZ3JvdXBzLm1pbm9yO1xuICBjb25zdCBwYXRjaCA9ICtncm91cHMucGF0Y2g7XG5cbiAgaWYgKG1ham9ySXNXaWxkY2FyZCkgcmV0dXJuIFt7IG9wZXJhdG9yOiBcIjxcIiwgbWFqb3I6IDAsIG1pbm9yOiAwLCBwYXRjaDogMCB9XTtcbiAgaWYgKG1pbm9ySXNXaWxkY2FyZCkge1xuICAgIGlmIChwYXRjaElzV2lsZGNhcmQpIHJldHVybiBbeyBvcGVyYXRvcjogXCI8XCIsIG1ham9yLCBtaW5vcjogMCwgcGF0Y2g6IDAgfV07XG4gICAgcmV0dXJuIFt7IG9wZXJhdG9yOiBcIjxcIiwgbWFqb3IsIG1pbm9yLCBwYXRjaDogMCB9XTtcbiAgfVxuICBpZiAocGF0Y2hJc1dpbGRjYXJkKSByZXR1cm4gW3sgb3BlcmF0b3I6IFwiPFwiLCBtYWpvciwgbWlub3IsIHBhdGNoOiAwIH1dO1xuICBjb25zdCBwcmVyZWxlYXNlID0gcGFyc2VQcmVyZWxlYXNlKGdyb3Vwcy5wcmVyZWxlYXNlID8/IFwiXCIpO1xuICBjb25zdCBidWlsZCA9IHBhcnNlQnVpbGQoZ3JvdXBzLmJ1aWxkID8/IFwiXCIpO1xuICByZXR1cm4gW3sgb3BlcmF0b3I6IFwiPFwiLCBtYWpvciwgbWlub3IsIHBhdGNoLCBwcmVyZWxlYXNlLCBidWlsZCB9XTtcbn1cbmZ1bmN0aW9uIGhhbmRsZUxlc3NUaGFuT3JFcXVhbE9wZXJhdG9yKGdyb3VwczogUmVnRXhwR3JvdXBzKTogQ29tcGFyYXRvcltdIHtcbiAgY29uc3QgbWlub3JJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMubWlub3IpO1xuICBjb25zdCBwYXRjaElzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5wYXRjaCk7XG5cbiAgY29uc3QgbWFqb3IgPSArZ3JvdXBzLm1ham9yO1xuICBjb25zdCBtaW5vciA9ICtncm91cHMubWlub3I7XG4gIGNvbnN0IHBhdGNoID0gK2dyb3Vwcy5wYXRjaDtcblxuICBpZiAobWlub3JJc1dpbGRjYXJkKSB7XG4gICAgaWYgKHBhdGNoSXNXaWxkY2FyZCkge1xuICAgICAgcmV0dXJuIFt7IG9wZXJhdG9yOiBcIjxcIiwgbWFqb3I6IG1ham9yICsgMSwgbWlub3I6IDAsIHBhdGNoOiAwIH1dO1xuICAgIH1cbiAgICByZXR1cm4gW3sgb3BlcmF0b3I6IFwiPFwiLCBtYWpvciwgbWlub3I6IG1pbm9yICsgMSwgcGF0Y2g6IDAgfV07XG4gIH1cbiAgaWYgKHBhdGNoSXNXaWxkY2FyZCkge1xuICAgIHJldHVybiBbeyBvcGVyYXRvcjogXCI8XCIsIG1ham9yLCBtaW5vcjogbWlub3IgKyAxLCBwYXRjaDogMCB9XTtcbiAgfVxuICBjb25zdCBwcmVyZWxlYXNlID0gcGFyc2VQcmVyZWxlYXNlKGdyb3Vwcy5wcmVyZWxlYXNlID8/IFwiXCIpO1xuICBjb25zdCBidWlsZCA9IHBhcnNlQnVpbGQoZ3JvdXBzLmJ1aWxkID8/IFwiXCIpO1xuICByZXR1cm4gW3sgb3BlcmF0b3I6IFwiPD1cIiwgbWFqb3IsIG1pbm9yLCBwYXRjaCwgcHJlcmVsZWFzZSwgYnVpbGQgfV07XG59XG5mdW5jdGlvbiBoYW5kbGVHcmVhdGVyVGhhbk9wZXJhdG9yKGdyb3VwczogUmVnRXhwR3JvdXBzKTogQ29tcGFyYXRvcltdIHtcbiAgY29uc3QgbWFqb3JJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMubWFqb3IpO1xuICBjb25zdCBtaW5vcklzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5taW5vcik7XG4gIGNvbnN0IHBhdGNoSXNXaWxkY2FyZCA9IGlzV2lsZGNhcmQoZ3JvdXBzLnBhdGNoKTtcblxuICBjb25zdCBtYWpvciA9ICtncm91cHMubWFqb3I7XG4gIGNvbnN0IG1pbm9yID0gK2dyb3Vwcy5taW5vcjtcbiAgY29uc3QgcGF0Y2ggPSArZ3JvdXBzLnBhdGNoO1xuXG4gIGlmIChtYWpvcklzV2lsZGNhcmQpIHJldHVybiBbeyBvcGVyYXRvcjogXCI8XCIsIG1ham9yOiAwLCBtaW5vcjogMCwgcGF0Y2g6IDAgfV07XG5cbiAgaWYgKG1pbm9ySXNXaWxkY2FyZCkge1xuICAgIHJldHVybiBbeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvcjogbWFqb3IgKyAxLCBtaW5vcjogMCwgcGF0Y2g6IDAgfV07XG4gIH1cbiAgaWYgKHBhdGNoSXNXaWxkY2FyZCkge1xuICAgIHJldHVybiBbeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3I6IG1pbm9yICsgMSwgcGF0Y2g6IDAgfV07XG4gIH1cbiAgY29uc3QgcHJlcmVsZWFzZSA9IHBhcnNlUHJlcmVsZWFzZShncm91cHMucHJlcmVsZWFzZSA/PyBcIlwiKTtcbiAgY29uc3QgYnVpbGQgPSBwYXJzZUJ1aWxkKGdyb3Vwcy5idWlsZCA/PyBcIlwiKTtcbiAgcmV0dXJuIFt7IG9wZXJhdG9yOiBcIj5cIiwgbWFqb3IsIG1pbm9yLCBwYXRjaCwgcHJlcmVsZWFzZSwgYnVpbGQgfV07XG59XG5mdW5jdGlvbiBoYW5kbGVHcmVhdGVyT3JFcXVhbE9wZXJhdG9yKGdyb3VwczogUmVnRXhwR3JvdXBzKTogQ29tcGFyYXRvcltdIHtcbiAgY29uc3QgbWFqb3JJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMubWFqb3IpO1xuICBjb25zdCBtaW5vcklzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5taW5vcik7XG4gIGNvbnN0IHBhdGNoSXNXaWxkY2FyZCA9IGlzV2lsZGNhcmQoZ3JvdXBzLnBhdGNoKTtcblxuICBjb25zdCBtYWpvciA9ICtncm91cHMubWFqb3I7XG4gIGNvbnN0IG1pbm9yID0gK2dyb3Vwcy5taW5vcjtcbiAgY29uc3QgcGF0Y2ggPSArZ3JvdXBzLnBhdGNoO1xuXG4gIGlmIChtYWpvcklzV2lsZGNhcmQpIHJldHVybiBbQUxMXTtcbiAgaWYgKG1pbm9ySXNXaWxkY2FyZCkge1xuICAgIGlmIChwYXRjaElzV2lsZGNhcmQpIHJldHVybiBbeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3I6IDAsIHBhdGNoOiAwIH1dO1xuICAgIHJldHVybiBbeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3IsIHBhdGNoOiAwIH1dO1xuICB9XG4gIGlmIChwYXRjaElzV2lsZGNhcmQpIHJldHVybiBbeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3IsIHBhdGNoOiAwIH1dO1xuICBjb25zdCBwcmVyZWxlYXNlID0gcGFyc2VQcmVyZWxlYXNlKGdyb3Vwcy5wcmVyZWxlYXNlID8/IFwiXCIpO1xuICBjb25zdCBidWlsZCA9IHBhcnNlQnVpbGQoZ3JvdXBzLmJ1aWxkID8/IFwiXCIpO1xuICByZXR1cm4gW3sgb3BlcmF0b3I6IFwiPj1cIiwgbWFqb3IsIG1pbm9yLCBwYXRjaCwgcHJlcmVsZWFzZSwgYnVpbGQgfV07XG59XG5mdW5jdGlvbiBoYW5kbGVFcXVhbE9wZXJhdG9yKGdyb3VwczogUmVnRXhwR3JvdXBzKTogQ29tcGFyYXRvcltdIHtcbiAgY29uc3QgbWFqb3JJc1dpbGRjYXJkID0gaXNXaWxkY2FyZChncm91cHMubWFqb3IpO1xuICBjb25zdCBtaW5vcklzV2lsZGNhcmQgPSBpc1dpbGRjYXJkKGdyb3Vwcy5taW5vcik7XG4gIGNvbnN0IHBhdGNoSXNXaWxkY2FyZCA9IGlzV2lsZGNhcmQoZ3JvdXBzLnBhdGNoKTtcblxuICBjb25zdCBtYWpvciA9ICtncm91cHMubWFqb3I7XG4gIGNvbnN0IG1pbm9yID0gK2dyb3Vwcy5taW5vcjtcbiAgY29uc3QgcGF0Y2ggPSArZ3JvdXBzLnBhdGNoO1xuXG4gIGlmIChtYWpvcklzV2lsZGNhcmQpIHJldHVybiBbQUxMXTtcbiAgaWYgKG1pbm9ySXNXaWxkY2FyZCkge1xuICAgIHJldHVybiBbXG4gICAgICB7IG9wZXJhdG9yOiBcIj49XCIsIG1ham9yLCBtaW5vcjogMCwgcGF0Y2g6IDAgfSxcbiAgICAgIHsgb3BlcmF0b3I6IFwiPFwiLCBtYWpvcjogbWFqb3IgKyAxLCBtaW5vcjogMCwgcGF0Y2g6IDAgfSxcbiAgICBdO1xuICB9XG4gIGlmIChwYXRjaElzV2lsZGNhcmQpIHtcbiAgICByZXR1cm4gW1xuICAgICAgeyBvcGVyYXRvcjogXCI+PVwiLCBtYWpvciwgbWlub3IsIHBhdGNoOiAwIH0sXG4gICAgICB7IG9wZXJhdG9yOiBcIjxcIiwgbWFqb3IsIG1pbm9yOiBtaW5vciArIDEsIHBhdGNoOiAwIH0sXG4gICAgXTtcbiAgfVxuICBjb25zdCBwcmVyZWxlYXNlID0gcGFyc2VQcmVyZWxlYXNlKGdyb3Vwcy5wcmVyZWxlYXNlID8/IFwiXCIpO1xuICBjb25zdCBidWlsZCA9IHBhcnNlQnVpbGQoZ3JvdXBzLmJ1aWxkID8/IFwiXCIpO1xuICByZXR1cm4gW3sgb3BlcmF0b3I6IHVuZGVmaW5lZCwgbWFqb3IsIG1pbm9yLCBwYXRjaCwgcHJlcmVsZWFzZSwgYnVpbGQgfV07XG59XG5cbmZ1bmN0aW9uIHBhcnNlUmFuZ2VTdHJpbmcoc3RyaW5nOiBzdHJpbmcpIHtcbiAgY29uc3QgZ3JvdXBzID0gc3RyaW5nLm1hdGNoKE9QRVJBVE9SX1hSQU5HRV9SRUdFWFApPy5ncm91cHMgYXMgUmVnRXhwR3JvdXBzO1xuICBpZiAoIWdyb3VwcykgcmV0dXJuIHBhcnNlQ29tcGFyYXRvcihzdHJpbmcpO1xuXG4gIHN3aXRjaCAoZ3JvdXBzLm9wZXJhdG9yKSB7XG4gICAgY2FzZSBcIl5cIjpcbiAgICAgIHJldHVybiBoYW5kbGVDYXJldE9wZXJhdG9yKGdyb3Vwcyk7XG4gICAgY2FzZSBcIn5cIjpcbiAgICBjYXNlIFwifj5cIjpcbiAgICAgIHJldHVybiBoYW5kbGVUaWxkZU9wZXJhdG9yKGdyb3Vwcyk7XG4gICAgY2FzZSBcIjxcIjpcbiAgICAgIHJldHVybiBoYW5kbGVMZXNzVGhhbk9wZXJhdG9yKGdyb3Vwcyk7XG4gICAgY2FzZSBcIjw9XCI6XG4gICAgICByZXR1cm4gaGFuZGxlTGVzc1RoYW5PckVxdWFsT3BlcmF0b3IoZ3JvdXBzKTtcbiAgICBjYXNlIFwiPlwiOlxuICAgICAgcmV0dXJuIGhhbmRsZUdyZWF0ZXJUaGFuT3BlcmF0b3IoZ3JvdXBzKTtcbiAgICBjYXNlIFwiPj1cIjpcbiAgICAgIHJldHVybiBoYW5kbGVHcmVhdGVyT3JFcXVhbE9wZXJhdG9yKGdyb3Vwcyk7XG4gICAgY2FzZSBcIj1cIjpcbiAgICBjYXNlIFwiXCI6XG4gICAgICByZXR1cm4gaGFuZGxlRXF1YWxPcGVyYXRvcihncm91cHMpO1xuICAgIGRlZmF1bHQ6XG4gICAgICB0aHJvdyBuZXcgRXJyb3IoYCcke2dyb3Vwcy5vcGVyYXRvcn0nIGlzIG5vdCBhIHZhbGlkIG9wZXJhdG9yLmApO1xuICB9XG59XG5cbi8qKlxuICogUGFyc2VzIGEgcmFuZ2Ugc3RyaW5nIGludG8gYSBSYW5nZSBvYmplY3Qgb3IgdGhyb3dzIGEgVHlwZUVycm9yLlxuICogQHBhcmFtIHJhbmdlIFRoZSByYW5nZSBzZXQgc3RyaW5nXG4gKiBAcmV0dXJucyBBIHZhbGlkIHNlbWFudGljIHJhbmdlXG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBwYXJzZVJhbmdlKHJhbmdlOiBzdHJpbmcpOiBSYW5nZSB7XG4gIGNvbnN0IHJhbmdlcyA9IHJhbmdlXG4gICAgLnNwbGl0KC9cXHMqXFx8XFx8XFxzKi8pXG4gICAgLm1hcCgocmFuZ2UpID0+IHBhcnNlSHlwaGVuUmFuZ2UocmFuZ2UpLmZsYXRNYXAocGFyc2VSYW5nZVN0cmluZykpO1xuICBPYmplY3QuZGVmaW5lUHJvcGVydHkocmFuZ2VzLCBcInJhbmdlc1wiLCB7IHZhbHVlOiByYW5nZXMgfSk7XG4gIHJldHVybiByYW5nZXMgYXMgUmFuZ2U7XG59XG4iXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsMEVBQTBFO0FBQzFFLFNBQVMsR0FBRyxRQUFRLGlCQUFpQjtBQUVyQyxTQUFTLHNCQUFzQixFQUFFLE1BQU0sUUFBUSxlQUFlO0FBQzlELFNBQVMsZUFBZSxRQUFRLHlCQUF5QjtBQUN6RCxTQUFTLFVBQVUsRUFBRSxlQUFlLFFBQVEsZUFBZTtBQUUzRCxTQUFTLFdBQVcsRUFBVztFQUM3QixPQUFPLENBQUMsTUFBTSxHQUFHLFdBQVcsT0FBTyxPQUFPLE9BQU87QUFDbkQ7QUFXQSxTQUFTLGlCQUFpQixLQUFhO0VBQ3JDLDhDQUE4QztFQUM5QyxRQUFRLE1BQU0sT0FBTyxDQUFDLGdCQUFnQjtFQUV0QyxNQUFNLFlBQVksTUFBTSxLQUFLLENBQUMsSUFBSSxPQUFPLENBQUMsQ0FBQyxFQUFFLE9BQU8sQ0FBQztFQUNyRCxNQUFNLFlBQVksV0FBVztFQUM3QixJQUFJLENBQUMsV0FBVyxPQUFPLE1BQU0sS0FBSyxDQUFDO0VBQ25DLE1BQU0sYUFBYSxTQUFTLENBQUMsRUFBRSxDQUFDLE1BQU07RUFDdEMsTUFBTSxjQUFjLE1BQU0sS0FBSyxDQUFDLFlBQVksS0FBSyxDQUFDO0VBQ2xELElBQUksQ0FBQyxhQUFhLE9BQU8sTUFBTSxLQUFLLENBQUM7RUFDckMsTUFBTSxlQUFlLFdBQVcsQ0FBQyxFQUFFLENBQUMsTUFBTTtFQUMxQyxNQUFNLGFBQWEsTUFBTSxLQUFLLENBQUMsYUFBYSxjQUFjLEtBQUssQ0FDN0QsSUFBSSxPQUFPLENBQUMsQ0FBQyxFQUFFLE9BQU8sS0FBSyxDQUFDO0VBRTlCLE1BQU0sY0FBYyxZQUFZO0VBQ2hDLElBQUksQ0FBQyxhQUFhLE9BQU8sTUFBTSxLQUFLLENBQUM7RUFDckMsSUFBSSxPQUFPLFNBQVMsQ0FBQyxFQUFFO0VBQ3ZCLElBQUksS0FBSyxVQUFVLENBQUMsRUFBRTtFQUV0QixJQUFJLFdBQVcsVUFBVSxLQUFLLEdBQUc7SUFDL0IsT0FBTztFQUNULE9BQU8sSUFBSSxXQUFXLFVBQVUsS0FBSyxHQUFHO0lBQ3RDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsVUFBVSxLQUFLLENBQUMsSUFBSSxDQUFDO0VBQ25DLE9BQU8sSUFBSSxXQUFXLFVBQVUsS0FBSyxHQUFHO0lBQ3RDLE9BQU8sQ0FBQyxFQUFFLEVBQUUsVUFBVSxLQUFLLENBQUMsQ0FBQyxFQUFFLFVBQVUsS0FBSyxDQUFDLEVBQUUsQ0FBQztFQUNwRCxPQUFPO0lBQ0wsT0FBTyxDQUFDLEVBQUUsRUFBRSxLQUFLLENBQUM7RUFDcEI7RUFFQSxJQUFJLFdBQVcsWUFBWSxLQUFLLEdBQUc7SUFDakMsS0FBSztFQUNQLE9BQU8sSUFBSSxXQUFXLFlBQVksS0FBSyxHQUFHO0lBQ3hDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxZQUFZLEtBQUssR0FBSSxFQUFFLElBQUksQ0FBQztFQUN4QyxPQUFPLElBQUksV0FBVyxZQUFZLEtBQUssR0FBRztJQUN4QyxLQUFLLENBQUMsQ0FBQyxFQUFFLFlBQVksS0FBSyxDQUFDLENBQUMsRUFBRSxDQUFDLFlBQVksS0FBSyxHQUFJLEVBQUUsRUFBRSxDQUFDO0VBQzNELE9BQU8sSUFBSSxZQUFZLFVBQVUsRUFBRTtJQUNqQyxLQUNFLENBQUMsRUFBRSxFQUFFLFlBQVksS0FBSyxDQUFDLENBQUMsRUFBRSxZQUFZLEtBQUssQ0FBQyxDQUFDLEVBQUUsWUFBWSxLQUFLLENBQUMsQ0FBQyxFQUFFLFlBQVksVUFBVSxDQUFDLENBQUM7RUFDaEcsT0FBTztJQUNMLEtBQUssQ0FBQyxFQUFFLEVBQUUsR0FBRyxDQUFDO0VBQ2hCO0VBRUEsT0FBTztJQUFDO0lBQU07R0FBRztBQUNuQjtBQUNBLFNBQVMsb0JBQW9CLE1BQW9CO0VBQy9DLE1BQU0sa0JBQWtCLFdBQVcsT0FBTyxLQUFLO0VBQy9DLE1BQU0sa0JBQWtCLFdBQVcsT0FBTyxLQUFLO0VBQy9DLE1BQU0sa0JBQWtCLFdBQVcsT0FBTyxLQUFLO0VBRS9DLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFDM0IsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBRTNCLElBQUksaUJBQWlCLE9BQU87SUFBQztHQUFJO0VBQ2pDLElBQUksaUJBQWlCO0lBQ25CLE9BQU87TUFDTDtRQUFFLFVBQVU7UUFBTTtRQUFPLE9BQU87UUFBRyxPQUFPO01BQUU7TUFDNUM7UUFBRSxVQUFVO1FBQUssT0FBTyxRQUFRO1FBQUcsT0FBTztRQUFHLE9BQU87TUFBRTtLQUN2RDtFQUNIO0VBQ0EsSUFBSSxpQkFBaUI7SUFDbkIsSUFBSSxVQUFVLEdBQUc7TUFDZixPQUFPO1FBQ0w7VUFBRSxVQUFVO1VBQU07VUFBTztVQUFPLE9BQU87UUFBRTtRQUN6QztVQUFFLFVBQVU7VUFBSztVQUFPLE9BQU8sUUFBUTtVQUFHLE9BQU87UUFBRTtPQUNwRDtJQUNIO0lBQ0EsT0FBTztNQUNMO1FBQUUsVUFBVTtRQUFNO1FBQU87UUFBTyxPQUFPO01BQUU7TUFDekM7UUFBRSxVQUFVO1FBQUssT0FBTyxRQUFRO1FBQUcsT0FBTztRQUFHLE9BQU87TUFBRTtLQUN2RDtFQUNIO0VBRUEsTUFBTSxhQUFhLGdCQUFnQixPQUFPLFVBQVUsSUFBSTtFQUN4RCxJQUFJLFVBQVUsR0FBRztJQUNmLElBQUksVUFBVSxHQUFHO01BQ2YsT0FBTztRQUNMO1VBQUUsVUFBVTtVQUFNO1VBQU87VUFBTztVQUFPO1FBQVc7UUFDbEQ7VUFBRSxVQUFVO1VBQUs7VUFBTztVQUFPLE9BQU8sUUFBUTtRQUFFO09BQ2pEO0lBQ0g7SUFDQSxPQUFPO01BQ0w7UUFBRSxVQUFVO1FBQU07UUFBTztRQUFPO1FBQU87TUFBVztNQUNsRDtRQUFFLFVBQVU7UUFBSztRQUFPLE9BQU8sUUFBUTtRQUFHLE9BQU87TUFBRTtLQUNwRDtFQUNIO0VBQ0EsT0FBTztJQUNMO01BQUUsVUFBVTtNQUFNO01BQU87TUFBTztNQUFPO0lBQVc7SUFDbEQ7TUFBRSxVQUFVO01BQUssT0FBTyxRQUFRO01BQUcsT0FBTztNQUFHLE9BQU87SUFBRTtHQUN2RDtBQUNIO0FBQ0EsU0FBUyxvQkFBb0IsTUFBb0I7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFFL0MsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBQzNCLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFFM0IsSUFBSSxpQkFBaUIsT0FBTztJQUFDO0dBQUk7RUFDakMsSUFBSSxpQkFBaUI7SUFDbkIsT0FBTztNQUNMO1FBQUUsVUFBVTtRQUFNO1FBQU8sT0FBTztRQUFHLE9BQU87TUFBRTtNQUM1QztRQUFFLFVBQVU7UUFBSyxPQUFPLFFBQVE7UUFBRyxPQUFPO1FBQUcsT0FBTztNQUFFO0tBQ3ZEO0VBQ0g7RUFDQSxJQUFJLGlCQUFpQjtJQUNuQixPQUFPO01BQ0w7UUFBRSxVQUFVO1FBQU07UUFBTztRQUFPLE9BQU87TUFBRTtNQUN6QztRQUFFLFVBQVU7UUFBSztRQUFPLE9BQU8sUUFBUTtRQUFHLE9BQU87TUFBRTtLQUNwRDtFQUNIO0VBQ0EsTUFBTSxhQUFhLGdCQUFnQixPQUFPLFVBQVUsSUFBSTtFQUN4RCxPQUFPO0lBQ0w7TUFBRSxVQUFVO01BQU07TUFBTztNQUFPO01BQU87SUFBVztJQUNsRDtNQUFFLFVBQVU7TUFBSztNQUFPLE9BQU8sUUFBUTtNQUFHLE9BQU87SUFBRTtHQUNwRDtBQUNIO0FBQ0EsU0FBUyx1QkFBdUIsTUFBb0I7RUFDbEQsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFFL0MsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBQzNCLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFFM0IsSUFBSSxpQkFBaUIsT0FBTztJQUFDO01BQUUsVUFBVTtNQUFLLE9BQU87TUFBRyxPQUFPO01BQUcsT0FBTztJQUFFO0dBQUU7RUFDN0UsSUFBSSxpQkFBaUI7SUFDbkIsSUFBSSxpQkFBaUIsT0FBTztNQUFDO1FBQUUsVUFBVTtRQUFLO1FBQU8sT0FBTztRQUFHLE9BQU87TUFBRTtLQUFFO0lBQzFFLE9BQU87TUFBQztRQUFFLFVBQVU7UUFBSztRQUFPO1FBQU8sT0FBTztNQUFFO0tBQUU7RUFDcEQ7RUFDQSxJQUFJLGlCQUFpQixPQUFPO0lBQUM7TUFBRSxVQUFVO01BQUs7TUFBTztNQUFPLE9BQU87SUFBRTtHQUFFO0VBQ3ZFLE1BQU0sYUFBYSxnQkFBZ0IsT0FBTyxVQUFVLElBQUk7RUFDeEQsTUFBTSxRQUFRLFdBQVcsT0FBTyxLQUFLLElBQUk7RUFDekMsT0FBTztJQUFDO01BQUUsVUFBVTtNQUFLO01BQU87TUFBTztNQUFPO01BQVk7SUFBTTtHQUFFO0FBQ3BFO0FBQ0EsU0FBUyw4QkFBOEIsTUFBb0I7RUFDekQsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFFL0MsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBQzNCLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFFM0IsSUFBSSxpQkFBaUI7SUFDbkIsSUFBSSxpQkFBaUI7TUFDbkIsT0FBTztRQUFDO1VBQUUsVUFBVTtVQUFLLE9BQU8sUUFBUTtVQUFHLE9BQU87VUFBRyxPQUFPO1FBQUU7T0FBRTtJQUNsRTtJQUNBLE9BQU87TUFBQztRQUFFLFVBQVU7UUFBSztRQUFPLE9BQU8sUUFBUTtRQUFHLE9BQU87TUFBRTtLQUFFO0VBQy9EO0VBQ0EsSUFBSSxpQkFBaUI7SUFDbkIsT0FBTztNQUFDO1FBQUUsVUFBVTtRQUFLO1FBQU8sT0FBTyxRQUFRO1FBQUcsT0FBTztNQUFFO0tBQUU7RUFDL0Q7RUFDQSxNQUFNLGFBQWEsZ0JBQWdCLE9BQU8sVUFBVSxJQUFJO0VBQ3hELE1BQU0sUUFBUSxXQUFXLE9BQU8sS0FBSyxJQUFJO0VBQ3pDLE9BQU87SUFBQztNQUFFLFVBQVU7TUFBTTtNQUFPO01BQU87TUFBTztNQUFZO0lBQU07R0FBRTtBQUNyRTtBQUNBLFNBQVMsMEJBQTBCLE1BQW9CO0VBQ3JELE1BQU0sa0JBQWtCLFdBQVcsT0FBTyxLQUFLO0VBQy9DLE1BQU0sa0JBQWtCLFdBQVcsT0FBTyxLQUFLO0VBQy9DLE1BQU0sa0JBQWtCLFdBQVcsT0FBTyxLQUFLO0VBRS9DLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFDM0IsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBRTNCLElBQUksaUJBQWlCLE9BQU87SUFBQztNQUFFLFVBQVU7TUFBSyxPQUFPO01BQUcsT0FBTztNQUFHLE9BQU87SUFBRTtHQUFFO0VBRTdFLElBQUksaUJBQWlCO0lBQ25CLE9BQU87TUFBQztRQUFFLFVBQVU7UUFBTSxPQUFPLFFBQVE7UUFBRyxPQUFPO1FBQUcsT0FBTztNQUFFO0tBQUU7RUFDbkU7RUFDQSxJQUFJLGlCQUFpQjtJQUNuQixPQUFPO01BQUM7UUFBRSxVQUFVO1FBQU07UUFBTyxPQUFPLFFBQVE7UUFBRyxPQUFPO01BQUU7S0FBRTtFQUNoRTtFQUNBLE1BQU0sYUFBYSxnQkFBZ0IsT0FBTyxVQUFVLElBQUk7RUFDeEQsTUFBTSxRQUFRLFdBQVcsT0FBTyxLQUFLLElBQUk7RUFDekMsT0FBTztJQUFDO01BQUUsVUFBVTtNQUFLO01BQU87TUFBTztNQUFPO01BQVk7SUFBTTtHQUFFO0FBQ3BFO0FBQ0EsU0FBUyw2QkFBNkIsTUFBb0I7RUFDeEQsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFFL0MsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBQzNCLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFFM0IsSUFBSSxpQkFBaUIsT0FBTztJQUFDO0dBQUk7RUFDakMsSUFBSSxpQkFBaUI7SUFDbkIsSUFBSSxpQkFBaUIsT0FBTztNQUFDO1FBQUUsVUFBVTtRQUFNO1FBQU8sT0FBTztRQUFHLE9BQU87TUFBRTtLQUFFO0lBQzNFLE9BQU87TUFBQztRQUFFLFVBQVU7UUFBTTtRQUFPO1FBQU8sT0FBTztNQUFFO0tBQUU7RUFDckQ7RUFDQSxJQUFJLGlCQUFpQixPQUFPO0lBQUM7TUFBRSxVQUFVO01BQU07TUFBTztNQUFPLE9BQU87SUFBRTtHQUFFO0VBQ3hFLE1BQU0sYUFBYSxnQkFBZ0IsT0FBTyxVQUFVLElBQUk7RUFDeEQsTUFBTSxRQUFRLFdBQVcsT0FBTyxLQUFLLElBQUk7RUFDekMsT0FBTztJQUFDO01BQUUsVUFBVTtNQUFNO01BQU87TUFBTztNQUFPO01BQVk7SUFBTTtHQUFFO0FBQ3JFO0FBQ0EsU0FBUyxvQkFBb0IsTUFBb0I7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFDL0MsTUFBTSxrQkFBa0IsV0FBVyxPQUFPLEtBQUs7RUFFL0MsTUFBTSxRQUFRLENBQUMsT0FBTyxLQUFLO0VBQzNCLE1BQU0sUUFBUSxDQUFDLE9BQU8sS0FBSztFQUMzQixNQUFNLFFBQVEsQ0FBQyxPQUFPLEtBQUs7RUFFM0IsSUFBSSxpQkFBaUIsT0FBTztJQUFDO0dBQUk7RUFDakMsSUFBSSxpQkFBaUI7SUFDbkIsT0FBTztNQUNMO1FBQUUsVUFBVTtRQUFNO1FBQU8sT0FBTztRQUFHLE9BQU87TUFBRTtNQUM1QztRQUFFLFVBQVU7UUFBSyxPQUFPLFFBQVE7UUFBRyxPQUFPO1FBQUcsT0FBTztNQUFFO0tBQ3ZEO0VBQ0g7RUFDQSxJQUFJLGlCQUFpQjtJQUNuQixPQUFPO01BQ0w7UUFBRSxVQUFVO1FBQU07UUFBTztRQUFPLE9BQU87TUFBRTtNQUN6QztRQUFFLFVBQVU7UUFBSztRQUFPLE9BQU8sUUFBUTtRQUFHLE9BQU87TUFBRTtLQUNwRDtFQUNIO0VBQ0EsTUFBTSxhQUFhLGdCQUFnQixPQUFPLFVBQVUsSUFBSTtFQUN4RCxNQUFNLFFBQVEsV0FBVyxPQUFPLEtBQUssSUFBSTtFQUN6QyxPQUFPO0lBQUM7TUFBRSxVQUFVO01BQVc7TUFBTztNQUFPO01BQU87TUFBWTtJQUFNO0dBQUU7QUFDMUU7QUFFQSxTQUFTLGlCQUFpQixNQUFjO0VBQ3RDLE1BQU0sU0FBUyxPQUFPLEtBQUssQ0FBQyx5QkFBeUI7RUFDckQsSUFBSSxDQUFDLFFBQVEsT0FBTyxnQkFBZ0I7RUFFcEMsT0FBUSxPQUFPLFFBQVE7SUFDckIsS0FBSztNQUNILE9BQU8sb0JBQW9CO0lBQzdCLEtBQUs7SUFDTCxLQUFLO01BQ0gsT0FBTyxvQkFBb0I7SUFDN0IsS0FBSztNQUNILE9BQU8sdUJBQXVCO0lBQ2hDLEtBQUs7TUFDSCxPQUFPLDhCQUE4QjtJQUN2QyxLQUFLO01BQ0gsT0FBTywwQkFBMEI7SUFDbkMsS0FBSztNQUNILE9BQU8sNkJBQTZCO0lBQ3RDLEtBQUs7SUFDTCxLQUFLO01BQ0gsT0FBTyxvQkFBb0I7SUFDN0I7TUFDRSxNQUFNLElBQUksTUFBTSxDQUFDLENBQUMsRUFBRSxPQUFPLFFBQVEsQ0FBQywwQkFBMEIsQ0FBQztFQUNuRTtBQUNGO0FBRUE7Ozs7Q0FJQyxHQUNELE9BQU8sU0FBUyxXQUFXLEtBQWE7RUFDdEMsTUFBTSxTQUFTLE1BQ1osS0FBSyxDQUFDLGNBQ04sR0FBRyxDQUFDLENBQUMsUUFBVSxpQkFBaUIsT0FBTyxPQUFPLENBQUM7RUFDbEQsT0FBTyxjQUFjLENBQUMsUUFBUSxVQUFVO0lBQUUsT0FBTztFQUFPO0VBQ3hELE9BQU87QUFDVCJ9