// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { MAP_FORMAT_TO_EXTRACTOR_RX } from "./_formats.ts";
/**
 * Tests if a string has valid front matter. Supports YAML, TOML and JSON.
 *
 * @param str String to test.
 * @param formats A list of formats to test for. Defaults to all supported formats.
 *
 * ```ts
 * import { test } from "https://deno.land/std@$STD_VERSION/front_matter/mod.ts";
 * import { assert } from "https://deno.land/std@$STD_VERSION/assert/assert.ts";
 *
 * assert(test("---\ntitle: Three dashes marks the spot\n---\n"));
 * assert(test("---toml\ntitle = 'Three dashes followed by format marks the spot'\n---\n"));
 * assert(test("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\n"));
 *
 * assert(!test("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\n", ["yaml"]));
 * ```
 */ export function test(str, formats) {
  if (!formats) {
    formats = Object.keys(MAP_FORMAT_TO_EXTRACTOR_RX);
  }
  for (const format of formats){
    if (format === "unknown") {
      throw new TypeError("Unable to test for unknown front matter format");
    }
    const match = MAP_FORMAT_TO_EXTRACTOR_RX[format].exec(str);
    if (match?.index === 0) {
      return true;
    }
  }
  return false;
}
//# sourceMappingURL=data:application/json;base64,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