// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Returns the first element that is the smallest value of the given function or
 * undefined if there are no elements
 *
 * @example
 * ```ts
 * import { minBy } from "https://deno.land/std@$STD_VERSION/collections/min_by.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/assert/assert_equals.ts";
 *
 * const people = [
 *   { name: "Anna", age: 34 },
 *   { name: "Kim", age: 42 },
 *   { name: "John", age: 23 },
 * ];
 *
 * const personWithMinAge = minBy(people, (i) => i.age);
 *
 * assertEquals(personWithMinAge, { name: "John", age: 23 });
 * ```
 */ export function minBy(array, selector) {
  let min = undefined;
  let minValue = undefined;
  for (const current of array){
    const currentValue = selector(current);
    if (minValue === undefined || currentValue < minValue) {
      min = current;
      minValue = currentValue;
    }
  }
  return min;
}
//# sourceMappingURL=data:application/json;base64,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