// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Error message emitted from the thrown error while mapping. */ export const ERROR_WHILE_MAPPING_MESSAGE = "Threw while mapping.";
/**
 * pooledMap transforms values from an (async) iterable into another async
 * iterable. The transforms are done concurrently, with a max concurrency
 * defined by the poolLimit.
 *
 * If an error is thrown from `iterableFn`, no new transformations will begin.
 * All currently executing transformations are allowed to finish and still
 * yielded on success. After that, the rejections among them are gathered and
 * thrown by the iterator in an `AggregateError`.
 *
 * @example
 * ```ts
 * import { pooledMap } from "https://deno.land/std@$STD_VERSION/async/pool.ts";
 *
 * const results = pooledMap(
 *   2,
 *   [1, 2, 3],
 *   (i) => new Promise((r) => setTimeout(() => r(i), 1000)),
 * );
 *
 * for await (const value of results) {
 *   // ...
 * }
 * ```
 *
 * @param poolLimit The maximum count of items being processed concurrently.
 * @param array The input array for mapping.
 * @param iteratorFn The function to call for every item of the array.
 */ export function pooledMap(poolLimit, array, iteratorFn) {
  // Create the async iterable that is returned from this function.
  const res = new TransformStream({
    async transform (p, controller) {
      try {
        const s = await p;
        controller.enqueue(s);
      } catch (e) {
        if (e instanceof AggregateError && e.message === ERROR_WHILE_MAPPING_MESSAGE) {
          controller.error(e);
        }
      }
    }
  });
  // Start processing items from the iterator
  (async ()=>{
    const writer = res.writable.getWriter();
    const executing = [];
    try {
      for await (const item of array){
        const p = Promise.resolve().then(()=>iteratorFn(item));
        // Only write on success. If we `writer.write()` a rejected promise,
        // that will end the iteration. We don't want that yet. Instead let it
        // fail the race, taking us to the catch block where all currently
        // executing jobs are allowed to finish and all rejections among them
        // can be reported together.
        writer.write(p);
        const e = p.then(()=>executing.splice(executing.indexOf(e), 1));
        executing.push(e);
        if (executing.length >= poolLimit) {
          await Promise.race(executing);
        }
      }
      // Wait until all ongoing events have processed, then close the writer.
      await Promise.all(executing);
      writer.close();
    } catch  {
      const errors = [];
      for (const result of (await Promise.allSettled(executing))){
        if (result.status === "rejected") {
          errors.push(result.reason);
        }
      }
      writer.write(Promise.reject(new AggregateError(errors, ERROR_WHILE_MAPPING_MESSAGE))).catch(()=>{});
    }
  })();
  // Feature test until browser coverage is adequate
  return Symbol.asyncIterator in res.readable && typeof res.readable[Symbol.asyncIterator] === "function" ? res.readable[Symbol.asyncIterator]() : async function*() {
    const reader = res.readable.getReader();
    while(true){
      const { done, value } = await reader.read();
      if (done) break;
      yield value;
    }
    reader.releaseLock();
  }();
}
//# sourceMappingURL=data:application/json;base64,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