// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * {@linkcode parse} and {@linkcode stringify} for handling
 * {@link https://toml.io/en/latest | TOML} encoded data. Be sure to read the supported
 * types as not every spec is supported at the moment and the handling in
 * TypeScript side is a bit different.
 *
 * ## Supported types and handling
 *
 * - [x] [Keys](https://toml.io/en/latest#keys)
 * - [ ] [String](https://toml.io/en/latest#string)
 * - [x] [Multiline String](https://toml.io/en/latest#string)
 * - [x] [Literal String](https://toml.io/en/latest#string)
 * - [ ] [Integer](https://toml.io/en/latest#integer)
 * - [x] [Float](https://toml.io/en/latest#float)
 * - [x] [Boolean](https://toml.io/en/latest#boolean)
 * - [x] [Offset Date-time](https://toml.io/en/latest#offset-date-time)
 * - [x] [Local Date-time](https://toml.io/en/latest#local-date-time)
 * - [x] [Local Date](https://toml.io/en/latest#local-date)
 * - [ ] [Local Time](https://toml.io/en/latest#local-time)
 * - [x] [Table](https://toml.io/en/latest#table)
 * - [x] [Inline Table](https://toml.io/en/latest#inline-table)
 * - [ ] [Array of Tables](https://toml.io/en/latest#array-of-tables)
 *
 * _Supported with warnings see [Warning](#Warning)._
 *
 * ### Warning
 *
 * #### String
 *
 * - Regex: Due to the spec, there is no flag to detect regex properly in a TOML
 *   declaration. So the regex is stored as string.
 *
 * #### Integer
 *
 * For **Binary** / **Octal** / **Hexadecimal** numbers, they are stored as string
 * to be not interpreted as Decimal.
 *
 * #### Local Time
 *
 * Because local time does not exist in JavaScript, the local time is stored as a
 * string.
 *
 * #### Inline Table
 *
 * Inline tables are supported. See below:
 *
 * ```toml
 * animal = { type = { name = "pug" } }
 * ## Output { animal: { type: { name: "pug" } } }
 * animal = { type.name = "pug" }
 * ## Output { animal: { type : { name : "pug" } }
 * animal.as.leaders = "tosin"
 * ## Output { animal: { as: { leaders: "tosin" } } }
 * "tosin.abasi" = "guitarist"
 * ## Output { tosin.abasi: "guitarist" }
 * ```
 *
 * #### Array of Tables
 *
 * At the moment only simple declarations like below are supported:
 *
 * ```toml
 * [[bin]]
 * name = "deno"
 * path = "cli/main.rs"
 *
 * [[bin]]
 * name = "deno_core"
 * path = "src/foo.rs"
 *
 * [[nib]]
 * name = "node"
 * path = "not_found"
 * ```
 *
 * will output:
 *
 * ```json
 * {
 *   "bin": [
 *     { "name": "deno", "path": "cli/main.rs" },
 *     { "name": "deno_core", "path": "src/foo.rs" }
 *   ],
 *   "nib": [{ "name": "node", "path": "not_found" }]
 * }
 * ```
 *
 * This module is browser compatible.
 *
 * @example
 * ```ts
 * import {
 *   parse,
 *   stringify,
 * } from "https://deno.land/std@$STD_VERSION/toml/mod.ts";
 * const obj = {
 *   bin: [
 *     { name: "deno", path: "cli/main.rs" },
 *     { name: "deno_core", path: "src/foo.rs" },
 *   ],
 *   nib: [{ name: "node", path: "not_found" }],
 * };
 * const tomlString = stringify(obj);
 * console.log(tomlString);
 *
 * // =>
 * // [[bin]]
 * // name = "deno"
 * // path = "cli/main.rs"
 *
 * // [[bin]]
 * // name = "deno_core"
 * // path = "src/foo.rs"
 *
 * // [[nib]]
 * // name = "node"
 * // path = "not_found"
 *
 * const tomlObject = parse(tomlString);
 * console.log(tomlObject);
 *
 * // =>
 * // {
 * //   bin: [
 * //     { name: "deno", path: "cli/main.rs" },
 * //     { name: "deno_core", path: "src/foo.rs" }
 * //   ],
 * //   nib: [ { name: "node", path: "not_found" } ]
 * // }
 * ```
 *
 * @module
 */ export * from "./stringify.ts";
export * from "./parse.ts";
//# sourceMappingURL=data:application/json;base64,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