#!/usr/bin/env -S deno run --allow-net --allow-read
// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This program serves files in the current directory over HTTP.
// TODO(bartlomieju): Add tests like these:
// https://github.com/indexzero/http-server/blob/master/test/http-server-test.js
/**
 * Contains functions {@linkcode serveDir} and {@linkcode serveFile} for building a static file server.
 *
 * This module can also be used as a cli. If you want to run directly:
 *
 * ```shell
 * > # start server
 * > deno run --allow-net --allow-read https://deno.land/std@$STD_VERSION/http/file_server.ts
 * > # show help
 * > deno run --allow-net --allow-read https://deno.land/std@$STD_VERSION/http/file_server.ts --help
 * ```
 *
 * If you want to install and run:
 *
 * ```shell
 * > # install
 * > deno install --allow-net --allow-read https://deno.land/std@$STD_VERSION/http/file_server.ts
 * > # start server
 * > file_server
 * > # show help
 * > file_server --help
 * ```
 *
 * @module
 */ import { join as posixJoin } from "../path/posix/join.ts";
import { normalize as posixNormalize } from "../path/posix/normalize.ts";
import { extname } from "../path/extname.ts";
import { join } from "../path/join.ts";
import { relative } from "../path/relative.ts";
import { resolve } from "../path/resolve.ts";
import { SEPARATOR_PATTERN } from "../path/constants.ts";
import { contentType } from "../media_types/content_type.ts";
import { calculate, ifNoneMatch } from "./etag.ts";
import { isRedirectStatus, STATUS_CODE, STATUS_TEXT } from "./status.ts";
import { ByteSliceStream } from "../streams/byte_slice_stream.ts";
import { parseArgs } from "../cli/parse_args.ts";
import { red } from "../fmt/colors.ts";
import { VERSION } from "../version.ts";
import { format as formatBytes } from "../fmt/bytes.ts";
const ENV_PERM_STATUS = Deno.permissions.querySync?.({
  name: "env",
  variable: "DENO_DEPLOYMENT_ID"
}).state ?? "granted"; // for deno deploy
const DENO_DEPLOYMENT_ID = ENV_PERM_STATUS === "granted" ? Deno.env.get("DENO_DEPLOYMENT_ID") : undefined;
const HASHED_DENO_DEPLOYMENT_ID = DENO_DEPLOYMENT_ID ? calculate(DENO_DEPLOYMENT_ID, {
  weak: true
}) : undefined;
function modeToString(isDir, maybeMode) {
  const modeMap = [
    "---",
    "--x",
    "-w-",
    "-wx",
    "r--",
    "r-x",
    "rw-",
    "rwx"
  ];
  if (maybeMode === null) {
    return "(unknown mode)";
  }
  const mode = maybeMode.toString(8);
  if (mode.length < 3) {
    return "(unknown mode)";
  }
  let output = "";
  mode.split("").reverse().slice(0, 3).forEach((v)=>{
    output = `${modeMap[+v]} ${output}`;
  });
  output = `${isDir ? "d" : "-"} ${output}`;
  return output;
}
function createStandardResponse(status, init) {
  const statusText = STATUS_TEXT[status];
  return new Response(statusText, {
    status,
    statusText,
    ...init
  });
}
/**
 * parse range header.
 *
 * ```ts ignore
 * parseRangeHeader("bytes=0-100",   500); // => { start: 0, end: 100 }
 * parseRangeHeader("bytes=0-",      500); // => { start: 0, end: 499 }
 * parseRangeHeader("bytes=-100",    500); // => { start: 400, end: 499 }
 * parseRangeHeader("bytes=invalid", 500); // => null
 * ```
 *
 * Note: Currently, no support for multiple Ranges (e.g. `bytes=0-10, 20-30`)
 */ function parseRangeHeader(rangeValue, fileSize) {
  const rangeRegex = /bytes=(?<start>\d+)?-(?<end>\d+)?$/u;
  const parsed = rangeValue.match(rangeRegex);
  if (!parsed || !parsed.groups) {
    // failed to parse range header
    return null;
  }
  const { start, end } = parsed.groups;
  if (start !== undefined) {
    if (end !== undefined) {
      return {
        start: +start,
        end: +end
      };
    } else {
      return {
        start: +start,
        end: fileSize - 1
      };
    }
  } else {
    if (end !== undefined) {
      // example: `bytes=-100` means the last 100 bytes.
      return {
        start: fileSize - +end,
        end: fileSize - 1
      };
    } else {
      // failed to parse range header
      return null;
    }
  }
}
/**
 * Returns an HTTP Response with the requested file as the body.
 * @param req The server request context used to cleanup the file handle.
 * @param filePath Path of the file to serve.
 */ export async function serveFile(req, filePath, { etagAlgorithm: algorithm, fileInfo } = {}) {
  try {
    fileInfo ??= await Deno.stat(filePath);
  } catch (error) {
    if (error instanceof Deno.errors.NotFound) {
      await req.body?.cancel();
      return createStandardResponse(STATUS_CODE.NotFound);
    } else {
      throw error;
    }
  }
  if (fileInfo.isDirectory) {
    await req.body?.cancel();
    return createStandardResponse(STATUS_CODE.NotFound);
  }
  const headers = createBaseHeaders();
  // Set date header if access timestamp is available
  if (fileInfo.atime) {
    headers.set("date", fileInfo.atime.toUTCString());
  }
  const etag = fileInfo.mtime ? await calculate(fileInfo, {
    algorithm
  }) : await HASHED_DENO_DEPLOYMENT_ID;
  // Set last modified header if last modification timestamp is available
  if (fileInfo.mtime) {
    headers.set("last-modified", fileInfo.mtime.toUTCString());
  }
  if (etag) {
    headers.set("etag", etag);
  }
  if (etag || fileInfo.mtime) {
    // If a `if-none-match` header is present and the value matches the tag or
    // if a `if-modified-since` header is present and the value is bigger than
    // the access timestamp value, then return 304
    const ifNoneMatchValue = req.headers.get("if-none-match");
    const ifModifiedSinceValue = req.headers.get("if-modified-since");
    if (!ifNoneMatch(ifNoneMatchValue, etag) || ifNoneMatchValue === null && fileInfo.mtime && ifModifiedSinceValue && fileInfo.mtime.getTime() < new Date(ifModifiedSinceValue).getTime() + 1000) {
      const status = STATUS_CODE.NotModified;
      return new Response(null, {
        status,
        statusText: STATUS_TEXT[status],
        headers
      });
    }
  }
  // Set mime-type using the file extension in filePath
  const contentTypeValue = contentType(extname(filePath));
  if (contentTypeValue) {
    headers.set("content-type", contentTypeValue);
  }
  const fileSize = fileInfo.size;
  const rangeValue = req.headers.get("range");
  // handle range request
  // Note: Some clients add a Range header to all requests to limit the size of the response.
  // If the file is empty, ignore the range header and respond with a 200 rather than a 416.
  // https://github.com/golang/go/blob/0d347544cbca0f42b160424f6bc2458ebcc7b3fc/src/net/http/fs.go#L273-L276
  if (rangeValue && 0 < fileSize) {
    const parsed = parseRangeHeader(rangeValue, fileSize);
    // Returns 200 OK if parsing the range header fails
    if (!parsed) {
      // Set content length
      headers.set("content-length", `${fileSize}`);
      const file = await Deno.open(filePath);
      const status = STATUS_CODE.OK;
      return new Response(file.readable, {
        status,
        statusText: STATUS_TEXT[status],
        headers
      });
    }
    // Return 416 Range Not Satisfiable if invalid range header value
    if (parsed.end < 0 || parsed.end < parsed.start || fileSize <= parsed.start) {
      // Set the "Content-range" header
      headers.set("content-range", `bytes */${fileSize}`);
      return createStandardResponse(STATUS_CODE.RangeNotSatisfiable, {
        headers
      });
    }
    // clamps the range header value
    const start = Math.max(0, parsed.start);
    const end = Math.min(parsed.end, fileSize - 1);
    // Set the "Content-range" header
    headers.set("content-range", `bytes ${start}-${end}/${fileSize}`);
    // Set content length
    const contentLength = end - start + 1;
    headers.set("content-length", `${contentLength}`);
    // Return 206 Partial Content
    const file = await Deno.open(filePath);
    await file.seek(start, Deno.SeekMode.Start);
    const sliced = file.readable.pipeThrough(new ByteSliceStream(0, contentLength - 1));
    const status = STATUS_CODE.PartialContent;
    return new Response(sliced, {
      status,
      statusText: STATUS_TEXT[status],
      headers
    });
  }
  // Set content length
  headers.set("content-length", `${fileSize}`);
  const file = await Deno.open(filePath);
  const status = STATUS_CODE.OK;
  return new Response(file.readable, {
    status,
    statusText: STATUS_TEXT[status],
    headers
  });
}
async function serveDirIndex(dirPath, options) {
  const { showDotfiles } = options;
  const urlRoot = options.urlRoot ? "/" + options.urlRoot : "";
  const dirUrl = `/${relative(options.target, dirPath).replaceAll(new RegExp(SEPARATOR_PATTERN, "g"), "/")}`;
  const listEntryPromise = [];
  // if ".." makes sense
  if (dirUrl !== "/") {
    const prevPath = join(dirPath, "..");
    const entryInfo = Deno.stat(prevPath).then((fileInfo)=>({
        mode: modeToString(true, fileInfo.mode),
        size: "",
        name: "../",
        url: `${urlRoot}${posixJoin(dirUrl, "..")}`
      }));
    listEntryPromise.push(entryInfo);
  }
  // Read fileInfo in parallel
  for await (const entry of Deno.readDir(dirPath)){
    if (!showDotfiles && entry.name[0] === ".") {
      continue;
    }
    const filePath = join(dirPath, entry.name);
    const fileUrl = encodeURIComponent(posixJoin(dirUrl, entry.name)).replaceAll("%2F", "/");
    listEntryPromise.push((async ()=>{
      try {
        const fileInfo = await Deno.stat(filePath);
        return {
          mode: modeToString(entry.isDirectory, fileInfo.mode),
          size: entry.isFile ? formatBytes(fileInfo.size ?? 0) : "",
          name: `${entry.name}${entry.isDirectory ? "/" : ""}`,
          url: `${urlRoot}${fileUrl}${entry.isDirectory ? "/" : ""}`
        };
      } catch (error) {
        // Note: Deno.stat for windows system files may be rejected with os error 32.
        if (!options.quiet) logError(error);
        return {
          mode: "(unknown mode)",
          size: "",
          name: `${entry.name}${entry.isDirectory ? "/" : ""}`,
          url: `${urlRoot}${fileUrl}${entry.isDirectory ? "/" : ""}`
        };
      }
    })());
  }
  const listEntry = await Promise.all(listEntryPromise);
  listEntry.sort((a, b)=>a.name.toLowerCase() > b.name.toLowerCase() ? 1 : -1);
  const formattedDirUrl = `${dirUrl.replace(/\/$/, "")}/`;
  const page = dirViewerTemplate(formattedDirUrl, listEntry);
  const headers = createBaseHeaders();
  headers.set("content-type", "text/html; charset=UTF-8");
  const status = STATUS_CODE.OK;
  return new Response(page, {
    status,
    statusText: STATUS_TEXT[status],
    headers
  });
}
function serveFallback(maybeError) {
  if (maybeError instanceof URIError) {
    return createStandardResponse(STATUS_CODE.BadRequest);
  }
  if (maybeError instanceof Deno.errors.NotFound) {
    return createStandardResponse(STATUS_CODE.NotFound);
  }
  return createStandardResponse(STATUS_CODE.InternalServerError);
}
function serverLog(req, status) {
  const d = new Date().toISOString();
  const dateFmt = `[${d.slice(0, 10)} ${d.slice(11, 19)}]`;
  const url = new URL(req.url);
  const s = `${dateFmt} [${req.method}] ${url.pathname}${url.search} ${status}`;
  // using console.debug instead of console.log so chrome inspect users can hide request logs
  console.debug(s);
}
function createBaseHeaders() {
  return new Headers({
    server: "deno",
    // Set "accept-ranges" so that the client knows it can make range requests on future requests
    "accept-ranges": "bytes"
  });
}
function dirViewerTemplate(dirname, entries) {
  const paths = dirname.split("/");
  return `
    <!DOCTYPE html>
    <html lang="en">
      <head>
        <meta charset="UTF-8" />
        <meta name="viewport" content="width=device-width, initial-scale=1.0" />
        <meta http-equiv="X-UA-Compatible" content="ie=edge" />
        <title>Deno File Server</title>
        <style>
          :root {
            --background-color: #fafafa;
            --color: rgba(0, 0, 0, 0.87);
          }
          @media (prefers-color-scheme: dark) {
            :root {
              --background-color: #292929;
              --color: #fff;
            }
            thead {
              color: #7f7f7f;
            }
          }
          @media (min-width: 960px) {
            main {
              max-width: 960px;
            }
            body {
              padding-left: 32px;
              padding-right: 32px;
            }
          }
          @media (min-width: 600px) {
            main {
              padding-left: 24px;
              padding-right: 24px;
            }
          }
          body {
            background: var(--background-color);
            color: var(--color);
            font-family: "Roboto", "Helvetica", "Arial", sans-serif;
            font-weight: 400;
            line-height: 1.43;
            font-size: 0.875rem;
          }
          a {
            color: #2196f3;
            text-decoration: none;
          }
          a:hover {
            text-decoration: underline;
          }
          thead {
            text-align: left;
          }
          thead th {
            padding-bottom: 12px;
          }
          table td {
            padding: 6px 36px 6px 0px;
          }
          .size {
            text-align: right;
            padding: 6px 12px 6px 24px;
          }
          .mode {
            font-family: monospace, monospace;
          }
        </style>
      </head>
      <body>
        <main>
          <h1>Index of
          <a href="/">home</a>${paths.map((path, index, array)=>{
    if (path === "") return "";
    const link = array.slice(0, index + 1).join("/");
    return `<a href="${link}">${path}</a>`;
  }).join("/")}
          </h1>
          <table>
            <thead>
              <tr>
                <th>Mode</th>
                <th>Size</th>
                <th>Name</th>
              </tr>
            </thead>
            ${entries.map((entry)=>`
                  <tr>
                    <td class="mode">
                      ${entry.mode}
                    </td>
                    <td class="size">
                      ${entry.size}
                    </td>
                    <td>
                      <a href="${entry.url}">${entry.name}</a>
                    </td>
                  </tr>
                `).join("")}
          </table>
        </main>
      </body>
    </html>
  `;
}
/**
 * Serves the files under the given directory root (opts.fsRoot).
 *
 * ```ts
 * import { serveDir } from "https://deno.land/std@$STD_VERSION/http/file_server.ts";
 *
 * Deno.serve((req) => {
 *   const pathname = new URL(req.url).pathname;
 *   if (pathname.startsWith("/static")) {
 *     return serveDir(req, {
 *       fsRoot: "path/to/static/files/dir",
 *     });
 *   }
 *   // Do dynamic responses
 *   return new Response();
 * });
 * ```
 *
 * Optionally you can pass `urlRoot` option. If it's specified that part is stripped from the beginning of the requested pathname.
 *
 * ```ts
 * import { serveDir } from "https://deno.land/std@$STD_VERSION/http/file_server.ts";
 *
 * // ...
 * serveDir(new Request("http://localhost/static/path/to/file"), {
 *   fsRoot: "public",
 *   urlRoot: "static",
 * });
 * ```
 *
 * The above example serves `./public/path/to/file` for the request to `/static/path/to/file`.
 *
 * @param req The request to handle
 */ export async function serveDir(req, opts = {}) {
  let response;
  try {
    response = await createServeDirResponse(req, opts);
  } catch (error) {
    if (!opts.quiet) logError(error);
    response = serveFallback(error);
  }
  // Do not update the header if the response is a 301 redirect.
  const isRedirectResponse = isRedirectStatus(response.status);
  if (opts.enableCors && !isRedirectResponse) {
    response.headers.append("access-control-allow-origin", "*");
    response.headers.append("access-control-allow-headers", "Origin, X-Requested-With, Content-Type, Accept, Range");
  }
  if (!opts.quiet) serverLog(req, response.status);
  if (opts.headers && !isRedirectResponse) {
    for (const header of opts.headers){
      const headerSplit = header.split(":");
      const name = headerSplit[0];
      const value = headerSplit.slice(1).join(":");
      response.headers.append(name, value);
    }
  }
  return response;
}
async function createServeDirResponse(req, opts) {
  const target = opts.fsRoot || ".";
  const urlRoot = opts.urlRoot;
  const showIndex = opts.showIndex ?? true;
  const showDotfiles = opts.showDotfiles || false;
  const { etagAlgorithm, showDirListing, quiet } = opts;
  const url = new URL(req.url);
  const decodedUrl = decodeURIComponent(url.pathname);
  let normalizedPath = posixNormalize(decodedUrl);
  if (urlRoot && !normalizedPath.startsWith("/" + urlRoot)) {
    return createStandardResponse(STATUS_CODE.NotFound);
  }
  // Redirect paths like `/foo////bar` and `/foo/bar/////` to normalized paths.
  if (normalizedPath !== decodedUrl) {
    url.pathname = normalizedPath;
    return Response.redirect(url, 301);
  }
  if (urlRoot) {
    normalizedPath = normalizedPath.replace(urlRoot, "");
  }
  // Remove trailing slashes to avoid ENOENT errors
  // when accessing a path to a file with a trailing slash.
  if (normalizedPath.endsWith("/")) {
    normalizedPath = normalizedPath.slice(0, -1);
  }
  const fsPath = join(target, normalizedPath);
  const fileInfo = await Deno.stat(fsPath);
  // For files, remove the trailing slash from the path.
  if (fileInfo.isFile && url.pathname.endsWith("/")) {
    url.pathname = url.pathname.slice(0, -1);
    return Response.redirect(url, 301);
  }
  // For directories, the path must have a trailing slash.
  if (fileInfo.isDirectory && !url.pathname.endsWith("/")) {
    // On directory listing pages,
    // if the current URL's pathname doesn't end with a slash, any
    // relative URLs in the index file will resolve against the parent
    // directory, rather than the current directory. To prevent that, we
    // return a 301 redirect to the URL with a slash.
    url.pathname += "/";
    return Response.redirect(url, 301);
  }
  // if target is file, serve file.
  if (!fileInfo.isDirectory) {
    return serveFile(req, fsPath, {
      etagAlgorithm,
      fileInfo
    });
  }
  // if target is directory, serve index or dir listing.
  if (showIndex) {
    const indexPath = join(fsPath, "index.html");
    let indexFileInfo;
    try {
      indexFileInfo = await Deno.lstat(indexPath);
    } catch (error) {
      if (!(error instanceof Deno.errors.NotFound)) {
        throw error;
      }
    // skip Not Found error
    }
    if (indexFileInfo?.isFile) {
      return serveFile(req, indexPath, {
        etagAlgorithm,
        fileInfo: indexFileInfo
      });
    }
  }
  if (showDirListing) {
    return serveDirIndex(fsPath, {
      urlRoot,
      showDotfiles,
      target,
      quiet
    });
  }
  return createStandardResponse(STATUS_CODE.NotFound);
}
function logError(error) {
  console.error(red(error instanceof Error ? error.message : `${error}`));
}
function main() {
  const serverArgs = parseArgs(Deno.args, {
    string: [
      "port",
      "host",
      "cert",
      "key",
      "header"
    ],
    boolean: [
      "help",
      "dir-listing",
      "dotfiles",
      "cors",
      "verbose",
      "version"
    ],
    negatable: [
      "dir-listing",
      "dotfiles",
      "cors"
    ],
    collect: [
      "header"
    ],
    default: {
      "dir-listing": true,
      dotfiles: true,
      cors: true,
      verbose: false,
      version: false,
      host: "0.0.0.0",
      port: "4507",
      cert: "",
      key: ""
    },
    alias: {
      p: "port",
      c: "cert",
      k: "key",
      h: "help",
      v: "verbose",
      V: "version",
      H: "header"
    }
  });
  const port = Number(serverArgs.port);
  const headers = serverArgs.header || [];
  const host = serverArgs.host;
  const certFile = serverArgs.cert;
  const keyFile = serverArgs.key;
  if (serverArgs.help) {
    printUsage();
    Deno.exit();
  }
  if (serverArgs.version) {
    console.log(`Deno File Server ${VERSION}`);
    Deno.exit();
  }
  if (keyFile || certFile) {
    if (keyFile === "" || certFile === "") {
      console.log("--key and --cert are required for TLS");
      printUsage();
      Deno.exit(1);
    }
  }
  const wild = serverArgs._;
  const target = resolve(wild[0] ?? "");
  const handler = (req)=>{
    return serveDir(req, {
      fsRoot: target,
      showDirListing: serverArgs["dir-listing"],
      showDotfiles: serverArgs.dotfiles,
      enableCors: serverArgs.cors,
      quiet: !serverArgs.verbose,
      headers
    });
  };
  const useTls = !!(keyFile && certFile);
  if (useTls) {
    Deno.serve({
      port,
      hostname: host,
      cert: Deno.readTextFileSync(certFile),
      key: Deno.readTextFileSync(keyFile)
    }, handler);
  } else {
    Deno.serve({
      port,
      hostname: host
    }, handler);
  }
}
function printUsage() {
  console.log(`Deno File Server ${VERSION}
  Serves a local directory in HTTP.

INSTALL:
  deno install --allow-net --allow-read https://deno.land/std/http/file_server.ts

USAGE:
  file_server [path] [options]

OPTIONS:
  -h, --help            Prints help information
  -p, --port <PORT>     Set port
  --cors                Enable CORS via the "Access-Control-Allow-Origin" header
  --host     <HOST>     Hostname (default is 0.0.0.0)
  -c, --cert <FILE>     TLS certificate file (enables TLS)
  -k, --key  <FILE>     TLS key file (enables TLS)
  -H, --header <HEADER> Sets a header on every request.
                        (e.g. --header "Cache-Control: no-cache")
                        This option can be specified multiple times.
  --no-dir-listing      Disable directory listing
  --no-dotfiles         Do not show dotfiles
  --no-cors             Disable cross-origin resource sharing
  -v, --verbose         Print request level logs
  -V, --version         Print version information

  All TLS options are required when one is provided.`);
}
if (import.meta.main) {
  main();
}
//# sourceMappingURL=data:application/json;base64,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