// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Utility for representing n-tuple. Used in {@linkcode tee}. */ class Queue {
  #source;
  #queue;
  head;
  done;
  constructor(iterable){
    this.#source = iterable[Symbol.asyncIterator]();
    this.#queue = {
      value: undefined,
      next: undefined
    };
    this.head = this.#queue;
    this.done = false;
  }
  async next() {
    const result = await this.#source.next();
    if (!result.done) {
      const nextNode = {
        value: result.value,
        next: undefined
      };
      this.#queue.next = nextNode;
      this.#queue = nextNode;
    } else {
      this.done = true;
    }
  }
}
/**
 * Branches the given async iterable into the `n` branches.
 *
 * @example
 * ```ts
 * import { tee } from "https://deno.land/std@$STD_VERSION/async/tee.ts";
 *
 * const gen = async function* gen() {
 *   yield 1;
 *   yield 2;
 *   yield 3;
 * };
 *
 * const [branch1, branch2] = tee(gen());
 *
 * for await (const n of branch1) {
 *   console.log(n); // => 1, 2, 3
 * }
 *
 * for await (const n of branch2) {
 *   console.log(n); // => 1, 2, 3
 * }
 * ```
 */ export function tee(iterable, n = 2) {
  const queue = new Queue(iterable);
  async function* generator() {
    let buffer = queue.head;
    while(true){
      if (buffer.next) {
        buffer = buffer.next;
        yield buffer.value;
      } else if (queue.done) {
        return;
      } else {
        await queue.next();
      }
    }
  }
  return Array.from({
    length: n
  }).map(()=>generator());
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjIxNy4wL2FzeW5jL3RlZS50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgMjAxOC0yMDI0IHRoZSBEZW5vIGF1dGhvcnMuIEFsbCByaWdodHMgcmVzZXJ2ZWQuIE1JVCBsaWNlbnNlLlxuLy8gVGhpcyBtb2R1bGUgaXMgYnJvd3NlciBjb21wYXRpYmxlLlxuXG4vKiogVXRpbGl0eSBmb3IgcmVwcmVzZW50aW5nIG4tdHVwbGUuIFVzZWQgaW4ge0BsaW5rY29kZSB0ZWV9LiAqL1xuZXhwb3J0IHR5cGUgVHVwbGU8VCwgTiBleHRlbmRzIG51bWJlcj4gPSBOIGV4dGVuZHMgTlxuICA/IG51bWJlciBleHRlbmRzIE4gPyBUW10gOiBUdXBsZU9mPFQsIE4sIFtdPlxuICA6IG5ldmVyO1xuXG4vKiogVXRpbGl0eSBmb3IgcmVwcmVzZW50aW5nIG4tdHVwbGUgb2YuIFVzZWQgaW4ge0BsaW5rY29kZSBUdXBsZX0uICovXG5leHBvcnQgdHlwZSBUdXBsZU9mPFQsIE4gZXh0ZW5kcyBudW1iZXIsIFIgZXh0ZW5kcyB1bmtub3duW10+ID1cbiAgUltcImxlbmd0aFwiXSBleHRlbmRzIE4gPyBSXG4gICAgOiBUdXBsZU9mPFQsIE4sIFtULCAuLi5SXT47XG5cbmludGVyZmFjZSBRdWV1ZU5vZGU8VD4ge1xuICB2YWx1ZTogVDtcbiAgbmV4dDogUXVldWVOb2RlPFQ+IHwgdW5kZWZpbmVkO1xufVxuXG5jbGFzcyBRdWV1ZTxUPiB7XG4gICNzb3VyY2U6IEFzeW5jSXRlcmF0b3I8VD47XG4gICNxdWV1ZTogUXVldWVOb2RlPFQ+O1xuICBoZWFkOiBRdWV1ZU5vZGU8VD47XG5cbiAgZG9uZTogYm9vbGVhbjtcblxuICBjb25zdHJ1Y3RvcihpdGVyYWJsZTogQXN5bmNJdGVyYWJsZTxUPikge1xuICAgIHRoaXMuI3NvdXJjZSA9IGl0ZXJhYmxlW1N5bWJvbC5hc3luY0l0ZXJhdG9yXSgpO1xuICAgIHRoaXMuI3F1ZXVlID0ge1xuICAgICAgdmFsdWU6IHVuZGVmaW5lZCEsXG4gICAgICBuZXh0OiB1bmRlZmluZWQsXG4gICAgfTtcbiAgICB0aGlzLmhlYWQgPSB0aGlzLiNxdWV1ZTtcbiAgICB0aGlzLmRvbmUgPSBmYWxzZTtcbiAgfVxuXG4gIGFzeW5jIG5leHQoKSB7XG4gICAgY29uc3QgcmVzdWx0ID0gYXdhaXQgdGhpcy4jc291cmNlLm5leHQoKTtcbiAgICBpZiAoIXJlc3VsdC5kb25lKSB7XG4gICAgICBjb25zdCBuZXh0Tm9kZTogUXVldWVOb2RlPFQ+ID0ge1xuICAgICAgICB2YWx1ZTogcmVzdWx0LnZhbHVlLFxuICAgICAgICBuZXh0OiB1bmRlZmluZWQsXG4gICAgICB9O1xuICAgICAgdGhpcy4jcXVldWUubmV4dCA9IG5leHROb2RlO1xuICAgICAgdGhpcy4jcXVldWUgPSBuZXh0Tm9kZTtcbiAgICB9IGVsc2Uge1xuICAgICAgdGhpcy5kb25lID0gdHJ1ZTtcbiAgICB9XG4gIH1cbn1cblxuLyoqXG4gKiBCcmFuY2hlcyB0aGUgZ2l2ZW4gYXN5bmMgaXRlcmFibGUgaW50byB0aGUgYG5gIGJyYW5jaGVzLlxuICpcbiAqIEBleGFtcGxlXG4gKiBgYGB0c1xuICogaW1wb3J0IHsgdGVlIH0gZnJvbSBcImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAkU1REX1ZFUlNJT04vYXN5bmMvdGVlLnRzXCI7XG4gKlxuICogY29uc3QgZ2VuID0gYXN5bmMgZnVuY3Rpb24qIGdlbigpIHtcbiAqICAgeWllbGQgMTtcbiAqICAgeWllbGQgMjtcbiAqICAgeWllbGQgMztcbiAqIH07XG4gKlxuICogY29uc3QgW2JyYW5jaDEsIGJyYW5jaDJdID0gdGVlKGdlbigpKTtcbiAqXG4gKiBmb3IgYXdhaXQgKGNvbnN0IG4gb2YgYnJhbmNoMSkge1xuICogICBjb25zb2xlLmxvZyhuKTsgLy8gPT4gMSwgMiwgM1xuICogfVxuICpcbiAqIGZvciBhd2FpdCAoY29uc3QgbiBvZiBicmFuY2gyKSB7XG4gKiAgIGNvbnNvbGUubG9nKG4pOyAvLyA9PiAxLCAyLCAzXG4gKiB9XG4gKiBgYGBcbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHRlZTxULCBOIGV4dGVuZHMgbnVtYmVyID0gMj4oXG4gIGl0ZXJhYmxlOiBBc3luY0l0ZXJhYmxlPFQ+LFxuICBuOiBOID0gMiBhcyBOLFxuKTogVHVwbGU8QXN5bmNJdGVyYWJsZTxUPiwgTj4ge1xuICBjb25zdCBxdWV1ZSA9IG5ldyBRdWV1ZTxUPihpdGVyYWJsZSk7XG5cbiAgYXN5bmMgZnVuY3Rpb24qIGdlbmVyYXRvcigpOiBBc3luY0dlbmVyYXRvcjxUPiB7XG4gICAgbGV0IGJ1ZmZlciA9IHF1ZXVlLmhlYWQ7XG4gICAgd2hpbGUgKHRydWUpIHtcbiAgICAgIGlmIChidWZmZXIubmV4dCkge1xuICAgICAgICBidWZmZXIgPSBidWZmZXIubmV4dDtcbiAgICAgICAgeWllbGQgYnVmZmVyLnZhbHVlO1xuICAgICAgfSBlbHNlIGlmIChxdWV1ZS5kb25lKSB7XG4gICAgICAgIHJldHVybjtcbiAgICAgIH0gZWxzZSB7XG4gICAgICAgIGF3YWl0IHF1ZXVlLm5leHQoKTtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICByZXR1cm4gQXJyYXkuZnJvbSh7IGxlbmd0aDogbiB9KS5tYXAoXG4gICAgKCkgPT4gZ2VuZXJhdG9yKCksXG4gICkgYXMgVHVwbGU8XG4gICAgQXN5bmNJdGVyYWJsZTxUPixcbiAgICBOXG4gID47XG59XG4iXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsMEVBQTBFO0FBQzFFLHFDQUFxQztBQUVyQywrREFBK0QsR0FlL0QsTUFBTTtFQUNKLENBQUMsTUFBTSxDQUFtQjtFQUMxQixDQUFDLEtBQUssQ0FBZTtFQUNyQixLQUFtQjtFQUVuQixLQUFjO0VBRWQsWUFBWSxRQUEwQixDQUFFO0lBQ3RDLElBQUksQ0FBQyxDQUFDLE1BQU0sR0FBRyxRQUFRLENBQUMsT0FBTyxhQUFhLENBQUM7SUFDN0MsSUFBSSxDQUFDLENBQUMsS0FBSyxHQUFHO01BQ1osT0FBTztNQUNQLE1BQU07SUFDUjtJQUNBLElBQUksQ0FBQyxJQUFJLEdBQUcsSUFBSSxDQUFDLENBQUMsS0FBSztJQUN2QixJQUFJLENBQUMsSUFBSSxHQUFHO0VBQ2Q7RUFFQSxNQUFNLE9BQU87SUFDWCxNQUFNLFNBQVMsTUFBTSxJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUMsSUFBSTtJQUN0QyxJQUFJLENBQUMsT0FBTyxJQUFJLEVBQUU7TUFDaEIsTUFBTSxXQUF5QjtRQUM3QixPQUFPLE9BQU8sS0FBSztRQUNuQixNQUFNO01BQ1I7TUFDQSxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUMsSUFBSSxHQUFHO01BQ25CLElBQUksQ0FBQyxDQUFDLEtBQUssR0FBRztJQUNoQixPQUFPO01BQ0wsSUFBSSxDQUFDLElBQUksR0FBRztJQUNkO0VBQ0Y7QUFDRjtBQUVBOzs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztDQXVCQyxHQUNELE9BQU8sU0FBUyxJQUNkLFFBQTBCLEVBQzFCLElBQU8sQ0FBTTtFQUViLE1BQU0sUUFBUSxJQUFJLE1BQVM7RUFFM0IsZ0JBQWdCO0lBQ2QsSUFBSSxTQUFTLE1BQU0sSUFBSTtJQUN2QixNQUFPLEtBQU07TUFDWCxJQUFJLE9BQU8sSUFBSSxFQUFFO1FBQ2YsU0FBUyxPQUFPLElBQUk7UUFDcEIsTUFBTSxPQUFPLEtBQUs7TUFDcEIsT0FBTyxJQUFJLE1BQU0sSUFBSSxFQUFFO1FBQ3JCO01BQ0YsT0FBTztRQUNMLE1BQU0sTUFBTSxJQUFJO01BQ2xCO0lBQ0Y7RUFDRjtFQUVBLE9BQU8sTUFBTSxJQUFJLENBQUM7SUFBRSxRQUFRO0VBQUUsR0FBRyxHQUFHLENBQ2xDLElBQU07QUFLViJ9