// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { DateTimeFormatter } from "./_common.ts";
/**
 * Takes an input `date` and a `formatString` to format to a `string`.
 *
 * @example
 * ```ts
 * import { format } from "https://deno.land/std@$STD_VERSION/datetime/format.ts";
 *
 * format(new Date(2019, 0, 20), "dd-MM-yyyy"); // output : "20-01-2019"
 * format(new Date(2019, 0, 20), "yyyy-MM-dd"); // output : "2019-01-20"
 * format(new Date(2019, 0, 20), "dd.MM.yyyy"); // output : "20.01.2019"
 * format(new Date(2019, 0, 20, 16, 34), "MM-dd-yyyy HH:mm"); // output : "01-20-2019 16:34"
 * format(new Date(2019, 0, 20, 16, 34), "MM-dd-yyyy hh:mm a"); // output : "01-20-2019 04:34 PM"
 * format(new Date(2019, 0, 20, 16, 34), "HH:mm MM-dd-yyyy"); // output : "16:34 01-20-2019"
 * format(new Date(2019, 0, 20, 16, 34, 23, 123), "MM-dd-yyyy HH:mm:ss.SSS"); // output : "01-20-2019 16:34:23.123"
 * format(new Date(2019, 0, 20), "'today:' yyyy-MM-dd"); // output : "today: 2019-01-20"
 * format(new Date("2019-01-20T16:34:23:123-05:00"), "yyyy-MM-dd HH:mm:ss", { utc: true });
 * // output : "2019-01-20 21:34:23"
 * ```
 *
 * @param date The date to be formatted.
 * @param formatString The date time string format.
 * @param options The options to customize the formatting of the date.
 * @return The formatted date string.
 */ export function format(date, formatString, options = {}) {
  const formatter = new DateTimeFormatter(formatString);
  return formatter.format(date, options.utc ? {
    timeZone: "UTC"
  } : undefined);
}
//# sourceMappingURL=data:application/json;base64,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