// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { compare } from "./compare.ts";
import { isWildcardComparator } from "./_shared.ts";
function testComparator(version, comparator) {
  if (isWildcardComparator(comparator)) {
    return true;
  }
  const cmp = compare(version, comparator.semver ?? comparator);
  switch(comparator.operator){
    case "":
    case "=":
    case "==":
    case "===":
    case undefined:
      {
        return cmp === 0;
      }
    case "!=":
    case "!==":
      {
        return cmp !== 0;
      }
    case ">":
      {
        return cmp > 0;
      }
    case "<":
      {
        return cmp < 0;
      }
    case ">=":
      {
        return cmp >= 0;
      }
    case "<=":
      {
        return cmp <= 0;
      }
  }
}
function testComparatorSet(version, set) {
  for (const comparator of set){
    if (!testComparator(version, comparator)) {
      return false;
    }
  }
  if (version.prerelease && version.prerelease.length > 0) {
    // Find the comparator that is allowed to have prereleases
    // For example, ^1.2.3-pr.1 desugars to >=1.2.3-pr.1 <2.0.0
    // That should allow `1.2.3-pr.2` to pass.
    // However, `1.2.4-alpha.notready` should NOT be allowed,
    // even though it's within the range set by the comparators.
    for (const comparator of set){
      if (isWildcardComparator(comparator)) {
        continue;
      }
      const { prerelease } = comparator.semver ?? comparator;
      if (prerelease && prerelease.length > 0) {
        const major = comparator.semver?.major ?? comparator.major;
        const minor = comparator.semver?.minor ?? comparator.minor;
        const patch = comparator.semver?.patch ?? comparator.patch;
        if (version.major === major && version.minor === minor && version.patch === patch) {
          return true;
        }
      }
    }
    return false;
  }
  return true;
}
/**
 * Test to see if the version satisfies the range.
 * @param version The version to test
 * @param range The range to check
 * @returns true if the version is in the range
 */ export function testRange(version, range) {
  return range.some((set)=>testComparatorSet(version, set));
}
//# sourceMappingURL=data:application/json;base64,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