// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Merge multiple streams into a single one, taking order into account, and
 * each stream will wait for a chunk to enqueue before the next stream can
 * append another chunk. If a stream ends before other ones, the others will
 * continue adding data in order, and the finished one will not add any more
 * data.
 *
 * @example
 * ```ts
 * import { zipReadableStreams } from "https://deno.land/std@$STD_VERSION/streams/zip_readable_streams.ts";
 *
 * const stream1 = ReadableStream.from(["1", "2", "3"]);
 * const stream2 = ReadableStream.from(["a", "b", "c"]);
 * const zippedStream = zipReadableStreams(stream1, stream2);
 *
 * await Array.fromAsync(zippedStream); // ["1", "a", "2", "b", "3", "c"];
 * ```
 */ export function zipReadableStreams(...streams) {
  const readers = new Set(streams.map((s)=>s.getReader()));
  return new ReadableStream({
    async start (controller) {
      try {
        let resolved = 0;
        while(resolved !== streams.length){
          for (const reader of readers){
            const { value, done } = await reader.read();
            if (!done) {
              controller.enqueue(value);
            } else {
              resolved++;
              readers.delete(reader);
            }
          }
        }
        controller.close();
      } catch (e) {
        controller.error(e);
      }
    }
  });
}
//# sourceMappingURL=data:application/json;base64,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