// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Convert the generator function into a {@linkcode TransformStream}.
 *
 * @example
 * ```ts
 * import { toTransformStream } from "https://deno.land/std@$STD_VERSION/streams/to_transform_stream.ts";
 *
 * const readable = ReadableStream.from([0, 1, 2])
 *   .pipeThrough(toTransformStream(async function* (src) {
 *     for await (const chunk of src) {
 *       yield chunk * 100;
 *     }
 *   }));
 *
 * for await (const chunk of readable) {
 *   console.log(chunk);
 * }
 * // output: 0, 100, 200
 * ```
 *
 * @param transformer A function to transform.
 * @param writableStrategy An object that optionally defines a queuing strategy for the stream.
 * @param readableStrategy An object that optionally defines a queuing strategy for the stream.
 */ export function toTransformStream(transformer, writableStrategy, readableStrategy) {
  const { writable, readable } = new TransformStream(undefined, writableStrategy);
  const iterable = transformer(readable);
  const iterator = iterable[Symbol.asyncIterator]?.() ?? iterable[Symbol.iterator]?.();
  return {
    writable,
    readable: new ReadableStream({
      async pull (controller) {
        let result;
        try {
          result = await iterator.next();
        } catch (error) {
          // Propagate error to stream from iterator
          // If the stream status is "errored", it will be thrown, but ignore.
          await readable.cancel(error).catch(()=>{});
          controller.error(error);
          return;
        }
        if (result.done) {
          controller.close();
          return;
        }
        controller.enqueue(result.value);
      },
      async cancel (reason) {
        // Propagate cancellation to readable and iterator
        if (typeof iterator.throw === "function") {
          try {
            await iterator.throw(reason);
          } catch  {
          /* `iterator.throw()` always throws on site. We catch it. */ }
        }
        await readable.cancel(reason);
      }
    }, readableStrategy)
  };
}
//# sourceMappingURL=data:application/json;base64,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