// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * A {@linkcode TransformStream} that will only read & enqueue `size` amount of
 * bytes. This operation is chunk based and not BYOB based, and as such will
 * read more than needed.
 *
 * If `options.error` is set, then instead of terminating the stream,
 * an error will be thrown.
 *
 * @example
 * ```ts
 * import { LimitedBytesTransformStream } from "https://deno.land/std@$STD_VERSION/streams/limited_bytes_transform_stream.ts";
 *
 * const res = await fetch("https://example.com");
 * const parts = res.body!
 *   .pipeThrough(new LimitedBytesTransformStream(512 * 1024));
 * ```
 */ export class LimitedBytesTransformStream extends TransformStream {
  #read = 0;
  /** Constructs a new instance. */ constructor(size, options = {}){
    super({
      transform: (chunk, controller)=>{
        if (this.#read + chunk.byteLength > size) {
          if (options.error) {
            throw new RangeError(`Exceeded byte size limit of '${size}'`);
          } else {
            controller.terminate();
          }
        } else {
          this.#read += chunk.byteLength;
          controller.enqueue(chunk);
        }
      }
    });
  }
}
//# sourceMappingURL=data:application/json;base64,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