// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Reads and writes comma-separated values (CSV) files.
 *
 * There are many kinds of CSV files; this module supports the format described
 * in {@link https://tools.ietf.org/html/rfc4180 | RFC 4180}.
 *
 * A csv file contains zero or more records of one or more fields per record.
 * Each record is separated by the newline character. The final record may
 * optionally be followed by a newline character.
 *
 * ```csv
 * field1,field2,field3
 * ```
 *
 * White space is considered part of a field.
 *
 * Carriage returns before newline characters are silently removed.
 *
 * Blank lines are ignored. A line with only whitespace characters (excluding
 * the ending newline character) is not considered a blank line.
 *
 * Fields which start and stop with the quote character " are called
 * quoted-fields. The beginning and ending quote are not part of the field.
 *
 * The source:
 *
 * ```csv
 * normal string,"quoted-field"
 * ```
 *
 * results in the fields
 *
 * ```ts
 * [`normal string`, `quoted-field`]
 * ```
 *
 * Within a quoted-field a quote character followed by a second quote character is considered a single quote.
 *
 * ```csv
 * "the ""word"" is true","a ""quoted-field"""
 * ```
 *
 * results in
 *
 * [`the "word" is true`, `a "quoted-field"`]
 *
 * Newlines and commas may be included in a quoted-field
 *
 * ```csv
 * "Multi-line
 * field","comma is ,"
 * ```
 *
 * results in
 *
 * ```ts
 * [`Multi-line
 * field`, `comma is ,`]
 * ```
 *
 * @module
 */ export * from "./stringify.ts";
export * from "./parse.ts";
export * from "./csv_parse_stream.ts";
export * from "./csv_stringify_stream.ts";
//# sourceMappingURL=data:application/json;base64,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