// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/*!
 * Ported and modified from: https://github.com/beatgammit/tar-js and
 * licensed as:
 *
 * (The MIT License)
 *
 * Copyright (c) 2011 T. Jameson Little
 * Copyright (c) 2019 Jun Kato
 * Copyright (c) 2018-2024 the Deno authors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
 * THE SOFTWARE.
 */ import { FileTypes, ustarStructure } from "./_common.ts";
import { MultiReader } from "../io/multi_reader.ts";
import { Buffer } from "../io/buffer.ts";
import { assert } from "../assert/assert.ts";
import { HEADER_LENGTH } from "./_common.ts";
const USTAR_MAGIC_HEADER = "ustar\u000000";
/**
 * Simple file reader
 */ class FileReader {
  filePath;
  #file;
  constructor(filePath){
    this.filePath = filePath;
  }
  async read(p) {
    if (!this.#file) {
      this.#file = await Deno.open(this.filePath, {
        read: true
      });
    }
    const res = await this.#file.read(p);
    if (res === null) {
      this.#file.close();
      this.#file = undefined;
    }
    return res;
  }
}
/**
 * Initialize Uint8Array of the specified length filled with 0
 * @param length
 */ function clean(length) {
  const buffer = new Uint8Array(length);
  return buffer;
}
function pad(num, bytes, base = 8) {
  const numString = num.toString(base);
  return "000000000000".slice(numString.length + 12 - bytes) + numString;
}
/**
 * Create header for a file in a tar archive
 */ function formatHeader(data) {
  const encoder = new TextEncoder();
  const buffer = clean(HEADER_LENGTH);
  let offset = 0;
  ustarStructure.forEach(function(value) {
    const entry = encoder.encode(data[value.field] || "");
    buffer.set(entry, offset);
    offset += value.length; // space it out with nulls
  });
  return buffer;
}
/**
 * ### Overview
 * A class to create a tar archive.  Tar archives allow for storing multiple files in a
 * single file (called an archive, or sometimes a tarball).  These archives typically
 * have the '.tar' extension.
 *
 * ### Usage
 * The workflow is to create a Tar instance, append files to it, and then write the
 * tar archive to the filesystem (or other output stream).  See the worked example
 * below for details.
 *
 * ### Compression
 * Tar archives are not compressed by default.  If you want to compress the archive,
 * you may compress the tar archive after creation, but this capability is not provided
 * here.
 *
 * ### File format and limitations
 *
 * The ustar file format is used for creating the archive file.
 * While this format is compatible with most tar readers,
 * the format has several limitations, including:
 * * Files must be smaller than 8GiB
 * * Filenames (including path) must be shorter than 256 characters
 * * Filenames (including path) cannot contain non-ASCII characters
 * * Sparse files are not supported
 *
 * @example
 * ```ts
 * import { Tar } from "https://deno.land/std@$STD_VERSION/archive/tar.ts";
 * import { Buffer } from "https://deno.land/std@$STD_VERSION/io/buffer.ts";
 * import { copy } from "https://deno.land/std@$STD_VERSION/io/copy.ts";
 *
 * const tar = new Tar();
 *
 * // Now that we've created our tar, let's add some files to it:
 *
 * const content = new TextEncoder().encode("Some arbitrary content");
 * await tar.append("deno.txt", {
 *   reader: new Buffer(content),
 *   contentSize: content.byteLength,
 * });
 *
 * // This file is sourced from the filesystem (and renamed in the archive)
 * await tar.append("filename_in_archive.txt", {
 *   filePath: "./filename_on_filesystem.txt",
 * });
 *
 * // Now let's write the tar (with it's two files) to the filesystem
 * // use tar.getReader() to read the contents.
 *
 * const writer = await Deno.open("./out.tar", { write: true, create: true });
 * await copy(tar.getReader(), writer);
 * writer.close();
 * ```
 */ export class Tar {
  data;
  constructor(){
    this.data = [];
  }
  /**
   * Append a file or reader of arbitrary content to this tar archive. Directories
   * appended to the archive append only the directory itself to the archive, not
   * its contents.  To add a directory and its contents, recursively append the
   * directory's contents.  Directories and subdirectories will be created automatically
   * in the archive as required.
   *
   * @param filenameInArchive file name of the content in the archive
   *                 e.g., test.txt; use slash for directory separators
   * @param source details of the source of the content including the
   *               reference to the content itself and potentially any
   *               related metadata.
   */ async append(filenameInArchive, source) {
    if (typeof filenameInArchive !== "string") {
      throw new Error("file name not specified");
    }
    let fileName = filenameInArchive;
    /**
     * Ustar format has a limitation of file name length.  Specifically:
     * 1. File names can contain at most 255 bytes.
     * 2. File names longer than 100 bytes must be split at a directory separator in two parts,
     * the first being at most 155 bytes long. So, in most cases file names must be a bit shorter
     * than 255 bytes.
     */ // separate file name into two parts if needed
    let fileNamePrefix;
    if (fileName.length > 100) {
      let i = fileName.length;
      while(i >= 0){
        i = fileName.lastIndexOf("/", i);
        if (i <= 155) {
          fileNamePrefix = fileName.slice(0, i);
          fileName = fileName.slice(i + 1);
          break;
        }
        i--;
      }
      const errMsg = "ustar format does not allow a long file name (length of [file name" + "prefix] + / + [file name] must be shorter than 256 bytes)";
      if (i < 0 || fileName.length > 100) {
        throw new Error(errMsg);
      } else {
        assert(fileNamePrefix !== undefined);
        if (fileNamePrefix.length > 155) {
          throw new Error(errMsg);
        }
      }
    }
    source = source || {};
    // set meta data
    let info;
    if (source.filePath) {
      info = await Deno.stat(source.filePath);
      if (info.isDirectory) {
        info.size = 0;
        source.reader = new Buffer();
      }
    }
    const mode = source.fileMode || info && info.mode || parseInt("777", 8) & 0xfff /* 511 */ ;
    const mtime = Math.floor(source.mtime ?? (info?.mtime ?? new Date()).valueOf() / 1000);
    const uid = source.uid || 0;
    const gid = source.gid || 0;
    if (typeof source.owner === "string" && source.owner.length >= 32) {
      throw new Error("ustar format does not allow owner name length >= 32 bytes");
    }
    if (typeof source.group === "string" && source.group.length >= 32) {
      throw new Error("ustar format does not allow group name length >= 32 bytes");
    }
    const fileSize = info?.size ?? source.contentSize;
    assert(fileSize !== undefined, "fileSize must be set");
    const type = source.type ? FileTypes[source.type] : info?.isDirectory ? FileTypes.directory : FileTypes.file;
    const tarData = {
      fileName,
      fileNamePrefix,
      fileMode: pad(mode, 7),
      uid: pad(uid, 7),
      gid: pad(gid, 7),
      fileSize: pad(fileSize, 11),
      mtime: pad(mtime, 11),
      checksum: "        ",
      type: type.toString(),
      ustar: USTAR_MAGIC_HEADER,
      owner: source.owner || "",
      group: source.group || "",
      filePath: source.filePath,
      reader: source.reader
    };
    // calculate the checksum
    let checksum = 0;
    const encoder = new TextEncoder();
    Object.keys(tarData).filter((key)=>[
        "filePath",
        "reader"
      ].indexOf(key) < 0).forEach(function(key) {
      checksum += encoder.encode(tarData[key]).reduce((p, c)=>p + c, 0);
    });
    tarData.checksum = pad(checksum, 6) + "\u0000 ";
    this.data.push(tarData);
  }
  /**
   * Get a Reader instance for this tar archive.
   */ getReader() {
    const readers = [];
    this.data.forEach((tarData)=>{
      let { reader } = tarData;
      const { filePath } = tarData;
      const headerArr = formatHeader(tarData);
      readers.push(new Buffer(headerArr));
      if (!reader) {
        assert(filePath !== undefined);
        reader = new FileReader(filePath);
      }
      readers.push(reader);
      // to the nearest multiple of recordSize
      assert(tarData.fileSize !== undefined, "fileSize must be set");
      readers.push(new Buffer(clean(HEADER_LENGTH - (parseInt(tarData.fileSize, 8) % HEADER_LENGTH || HEADER_LENGTH))));
    });
    // append 2 empty records
    readers.push(new Buffer(clean(HEADER_LENGTH * 2)));
    return new MultiReader(readers);
  }
}
//# sourceMappingURL=data:application/json;base64,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