// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
const NEWLINE_REGEXP = /\r\n|\r|\n/;
const encoder = new TextEncoder();
function assertHasNoNewline(value, varName) {
  if (value.match(NEWLINE_REGEXP) !== null) {
    throw new RangeError(`${varName} cannot contain a newline`);
  }
}
/**
 * Converts a server-sent message object into a string for the client.
 *
 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/Server-sent_events/Using_server-sent_events#event_stream_format}
 */ function stringify(message) {
  const lines = [];
  if (message.comment) {
    assertHasNoNewline(message.comment, "`message.comment`");
    lines.push(`:${message.comment}`);
  }
  if (message.event) {
    assertHasNoNewline(message.event, "`message.event`");
    lines.push(`event:${message.event}`);
  }
  if (message.data) {
    message.data.split(NEWLINE_REGEXP).forEach((line)=>lines.push(`data:${line}`));
  }
  if (message.id) {
    assertHasNoNewline(message.id.toString(), "`message.id`");
    lines.push(`id:${message.id}`);
  }
  if (message.retry) lines.push(`retry:${message.retry}`);
  return encoder.encode(lines.join("\n") + "\n\n");
}
/**
 * Transforms server-sent message objects into strings for the client.
 *
 * @see {@link https://developer.mozilla.org/en-US/docs/Web/API/Server-sent_events/Using_server-sent_events}
 *
 * @example
 * ```ts
 * import {
 *   type ServerSentEventMessage,
 *   ServerSentEventStream,
 * } from "https://deno.land/std@$STD_VERSION/http/server_sent_event_stream.ts";
 *
 * const stream = ReadableStream.from<ServerSentEventMessage>([
 *   { data: "hello there" }
 * ]).pipeThrough(new ServerSentEventStream());
 * new Response(stream, {
 *   headers: {
 *     "content-type": "text/event-stream",
 *     "cache-control": "no-cache",
 *   },
 * });
 * ```
 */ export class ServerSentEventStream extends TransformStream {
  constructor(){
    super({
      transform: (message, controller)=>{
        controller.enqueue(stringify(message));
      }
    });
  }
}
//# sourceMappingURL=data:application/json;base64,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