// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { compare } from "./compare.ts";
import { testRange } from "./test_range.ts";
import { isWildcardComparator } from "./_shared.ts";
/**
 * Returns true if the range of possible versions intersects with the other comparators set of possible versions
 * @param c0 The left side comparator
 * @param c1 The right side comparator
 * @returns True if any part of the comparators intersect
 */ export function comparatorIntersects(c0, c1) {
  const op0 = c0.operator;
  const op1 = c1.operator;
  if (op0 === "" || op0 === undefined) {
    // if c0 is empty comparator, then returns true
    if (isWildcardComparator(c0)) {
      return true;
    }
    return testRange(c0, [
      [
        c1
      ]
    ]);
  } else if (op1 === "" || op1 === undefined) {
    if (isWildcardComparator(c1)) {
      return true;
    }
    return testRange(c1, [
      [
        c0
      ]
    ]);
  }
  const cmp = compare(c0, c1);
  const sameDirectionIncreasing = (op0 === ">=" || op0 === ">") && (op1 === ">=" || op1 === ">");
  const sameDirectionDecreasing = (op0 === "<=" || op0 === "<") && (op1 === "<=" || op1 === "<");
  const sameSemVer = cmp === 0;
  const differentDirectionsInclusive = (op0 === ">=" || op0 === "<=") && (op1 === ">=" || op1 === "<=");
  const oppositeDirectionsLessThan = cmp === -1 && (op0 === ">=" || op0 === ">") && (op1 === "<=" || op1 === "<");
  const oppositeDirectionsGreaterThan = cmp === 1 && (op0 === "<=" || op0 === "<") && (op1 === ">=" || op1 === ">");
  return sameDirectionIncreasing || sameDirectionDecreasing || sameSemVer && differentDirectionsInclusive || oppositeDirectionsLessThan || oppositeDirectionsGreaterThan;
}
//# sourceMappingURL=data:application/json;base64,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