// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { LogLevels } from "./levels.ts";
import { BaseHandler } from "./base_handler.ts";
const PAGE_SIZE = 4096;
/**
 * This handler will output to a file using an optional mode (default is `a`,
 * e.g. append). The file will grow indefinitely. It uses a buffer for writing
 * to file. Logs can be manually flushed with `fileHandler.flush()`. Log
 * messages with a log level greater than error are immediately flushed. Logs
 * are also flushed on process completion.
 *
 * Behavior of the log modes is as follows:
 *
 * - `'a'` - Default mode. Appends new log messages to the end of an existing log
 *   file, or create a new log file if none exists.
 * - `'w'` - Upon creation of the handler, any existing log file will be removed
 *   and a new one created.
 * - `'x'` - This will create a new log file and throw an error if one already
 *   exists.
 *
 * This handler requires `--allow-write` permission on the log file.
 */ export class FileHandler extends BaseHandler {
  _file;
  _buf = new Uint8Array(PAGE_SIZE);
  _pointer = 0;
  _filename;
  _mode;
  _openOptions;
  _encoder = new TextEncoder();
  #unloadCallback = (()=>{
    this.destroy();
  }).bind(this);
  constructor(levelName, options){
    super(levelName, options);
    this._filename = options.filename;
    // default to append mode, write only
    this._mode = options.mode ? options.mode : "a";
    this._openOptions = {
      createNew: this._mode === "x",
      create: this._mode !== "x",
      append: this._mode === "a",
      truncate: this._mode !== "a",
      write: true
    };
  }
  setup() {
    this._file = Deno.openSync(this._filename, this._openOptions);
    this.#resetBuffer();
    addEventListener("unload", this.#unloadCallback);
  }
  handle(logRecord) {
    super.handle(logRecord);
    // Immediately flush if log level is higher than ERROR
    if (logRecord.level > LogLevels.ERROR) {
      this.flush();
    }
  }
  log(msg) {
    const bytes = this._encoder.encode(msg + "\n");
    if (bytes.byteLength > this._buf.byteLength - this._pointer) {
      this.flush();
    }
    this._buf.set(bytes, this._pointer);
    this._pointer += bytes.byteLength;
  }
  flush() {
    if (this._pointer > 0 && this._file) {
      let written = 0;
      while(written < this._pointer){
        written += this._file.writeSync(this._buf.subarray(written, this._pointer));
      }
      this.#resetBuffer();
    }
  }
  #resetBuffer() {
    this._pointer = 0;
  }
  destroy() {
    this.flush();
    this._file?.close();
    this._file = undefined;
    removeEventListener("unload", this.#unloadCallback);
  }
}
//# sourceMappingURL=data:application/json;base64,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