// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { existsSync } from "../fs/exists.ts";
import { FileHandler } from "./file_handler.ts";
/**
 * This handler extends the functionality of the {@linkcode FileHandler} by
 * "rotating" the log file when it reaches a certain size. `maxBytes` specifies
 * the maximum size in bytes that the log file can grow to before rolling over
 * to a new one. If the size of the new log message plus the current log file
 * size exceeds `maxBytes` then a roll-over is triggered. When a roll-over
 * occurs, before the log message is written, the log file is renamed and
 * appended with `.1`. If a `.1` version already existed, it would have been
 * renamed `.2` first and so on. The maximum number of log files to keep is
 * specified by `maxBackupCount`. After the renames are complete the log message
 * is written to the original, now blank, file.
 *
 * Example: Given `log.txt`, `log.txt.1`, `log.txt.2` and `log.txt.3`, a
 * `maxBackupCount` of 3 and a new log message which would cause `log.txt` to
 * exceed `maxBytes`, then `log.txt.2` would be renamed to `log.txt.3` (thereby
 * discarding the original contents of `log.txt.3` since 3 is the maximum number
 * of backups to keep), `log.txt.1` would be renamed to `log.txt.2`, `log.txt`
 * would be renamed to `log.txt.1` and finally `log.txt` would be created from
 * scratch where the new log message would be written.
 *
 * This handler uses a buffer for writing log messages to file. Logs can be
 * manually flushed with `fileHandler.flush()`. Log messages with a log level
 * greater than ERROR are immediately flushed. Logs are also flushed on process
 * completion.
 *
 * Additional notes on `mode` as described above:
 *
 * - `'a'` Default mode. As above, this will pick up where the logs left off in
 *   rotation, or create a new log file if it doesn't exist.
 * - `'w'` in addition to starting with a clean `filename`, this mode will also
 *   cause any existing backups (up to `maxBackupCount`) to be deleted on setup
 *   giving a fully clean slate.
 * - `'x'` requires that neither `filename`, nor any backups (up to
 *   `maxBackupCount`), exist before setup.
 *
 * This handler requires both `--allow-read` and `--allow-write` permissions on
 * the log files.
 */ export class RotatingFileHandler extends FileHandler {
  #maxBytes;
  #maxBackupCount;
  #currentFileSize = 0;
  constructor(levelName, options){
    super(levelName, options);
    this.#maxBytes = options.maxBytes;
    this.#maxBackupCount = options.maxBackupCount;
  }
  setup() {
    if (this.#maxBytes < 1) {
      this.destroy();
      throw new Error("maxBytes cannot be less than 1");
    }
    if (this.#maxBackupCount < 1) {
      this.destroy();
      throw new Error("maxBackupCount cannot be less than 1");
    }
    super.setup();
    if (this._mode === "w") {
      // Remove old backups too as it doesn't make sense to start with a clean
      // log file, but old backups
      for(let i = 1; i <= this.#maxBackupCount; i++){
        try {
          Deno.removeSync(this._filename + "." + i);
        } catch (error) {
          if (!(error instanceof Deno.errors.NotFound)) {
            throw error;
          }
        }
      }
    } else if (this._mode === "x") {
      // Throw if any backups also exist
      for(let i = 1; i <= this.#maxBackupCount; i++){
        if (existsSync(this._filename + "." + i)) {
          this.destroy();
          throw new Deno.errors.AlreadyExists("Backup log file " + this._filename + "." + i + " already exists");
        }
      }
    } else {
      this.#currentFileSize = Deno.statSync(this._filename).size;
    }
  }
  log(msg) {
    const msgByteLength = this._encoder.encode(msg).byteLength + 1;
    if (this.#currentFileSize + msgByteLength > this.#maxBytes) {
      this.rotateLogFiles();
      this.#currentFileSize = 0;
    }
    super.log(msg);
    this.#currentFileSize += msgByteLength;
  }
  rotateLogFiles() {
    this.flush();
    this._file.close();
    for(let i = this.#maxBackupCount - 1; i >= 0; i--){
      const source = this._filename + (i === 0 ? "" : "." + i);
      const dest = this._filename + "." + (i + 1);
      if (existsSync(source)) {
        Deno.renameSync(source, dest);
      }
    }
    this._file = Deno.openSync(this._filename, this._openOptions);
  }
}
//# sourceMappingURL=data:application/json;base64,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