// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
import { MAP_FORMAT_TO_EXTRACTOR_RX, MAP_FORMAT_TO_RECOGNIZER_RX } from "./_formats.ts";
function _extract(str, rx, parse) {
  const match = rx.exec(str);
  if (!match || match.index !== 0) {
    throw new TypeError("Unexpected end of input");
  }
  const frontMatter = match.at(-1)?.replace(/^\s+|\s+$/g, "") || "";
  const attrs = parse(frontMatter);
  const body = str.replace(match[0], "");
  return {
    frontMatter,
    body,
    attrs
  };
}
/**
 * Recognizes the format of the front matter in a string. Supports YAML, TOML and JSON.
 *
 * @param str String to recognize.
 * @param formats A list of formats to recognize. Defaults to all supported formats.
 *
 * ```ts
 * import { recognize } from "https://deno.land/std@$STD_VERSION/front_matter/mod.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/assert/assert_equals.ts";
 *
 * assertEquals(recognize("---\ntitle: Three dashes marks the spot\n---\n"), "yaml");
 * assertEquals(recognize("---toml\ntitle = 'Three dashes followed by format marks the spot'\n---\n"), "toml");
 * assertEquals(recognize("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\n"), "json");
 * assertEquals(recognize("---xml\n<title>Three dashes marks the spot</title>\n---\n"), "unknown");
 *
 * assertEquals(recognize("---json\n<title>Three dashes marks the spot</title>\n---\n", ["yaml"]), "unknown");
 */ function recognize(str, formats) {
  if (!formats) {
    formats = Object.keys(MAP_FORMAT_TO_RECOGNIZER_RX);
  }
  const [firstLine] = str.split(/(\r?\n)/);
  for (const format of formats){
    if (format === "unknown") {
      continue;
    }
    if (MAP_FORMAT_TO_RECOGNIZER_RX[format].test(firstLine)) {
      return format;
    }
  }
  return "unknown";
}
/**
 * Factory that creates a function that extracts front matter from a string with the given parsers.
 * Supports YAML, TOML and JSON.
 *
 * @param formats A descriptor containing Format-parser pairs to use for each format.
 * @returns A function that extracts front matter from a string with the given parsers.
 *
 * ```ts
 * import { createExtractor, Parser } from "https://deno.land/std@$STD_VERSION/front_matter/mod.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/assert/assert_equals.ts";
 * import { parse as parseYAML } from "https://deno.land/std@$STD_VERSION/yaml/parse.ts";
 * import { parse as parseTOML } from "https://deno.land/std@$STD_VERSION/toml/parse.ts";
 * const extractYAML = createExtractor({ yaml: parseYAML as Parser });
 * const extractTOML = createExtractor({ toml: parseTOML as Parser });
 * const extractJSON = createExtractor({ json: JSON.parse as Parser });
 * const extractYAMLOrJSON = createExtractor({
 *     yaml: parseYAML as Parser,
 *     json: JSON.parse as Parser,
 * });
 *
 * let { attrs, body, frontMatter } = extractYAML<{ title: string }>("---\ntitle: Three dashes marks the spot\n---\nferret");
 * assertEquals(attrs.title, "Three dashes marks the spot");
 * assertEquals(body, "ferret");
 * assertEquals(frontMatter, "title: Three dashes marks the spot");
 *
 * ({ attrs, body, frontMatter } = extractTOML<{ title: string }>("---toml\ntitle = 'Three dashes followed by format marks the spot'\n---\n"));
 * assertEquals(attrs.title, "Three dashes followed by format marks the spot");
 * assertEquals(body, "");
 * assertEquals(frontMatter, "title = 'Three dashes followed by format marks the spot'");
 *
 * ({ attrs, body, frontMatter } = extractJSON<{ title: string }>("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\ngoat"));
 * assertEquals(attrs.title, "Three dashes followed by format marks the spot");
 * assertEquals(body, "goat");
 * assertEquals(frontMatter, "{\"title\": \"Three dashes followed by format marks the spot\"}");
 *
 * ({ attrs, body, frontMatter } = extractYAMLOrJSON<{ title: string }>("---\ntitle: Three dashes marks the spot\n---\nferret"));
 * assertEquals(attrs.title, "Three dashes marks the spot");
 * assertEquals(body, "ferret");
 * assertEquals(frontMatter, "title: Three dashes marks the spot");
 *
 * ({ attrs, body, frontMatter } = extractYAMLOrJSON<{ title: string }>("---json\n{\"title\": \"Three dashes followed by format marks the spot\"}\n---\ngoat"));
 * assertEquals(attrs.title, "Three dashes followed by format marks the spot");
 * assertEquals(body, "goat");
 * assertEquals(frontMatter, "{\"title\": \"Three dashes followed by format marks the spot\"}");
 * ```
 */ export function createExtractor(formats) {
  const formatKeys = Object.keys(formats);
  return function extract(str) {
    const format = recognize(str, formatKeys);
    const parser = formats[format];
    if (format === "unknown" || !parser) {
      throw new TypeError(`Unsupported front matter format`);
    }
    return _extract(str, MAP_FORMAT_TO_EXTRACTOR_RX[format], parser);
  };
}
//# sourceMappingURL=data:application/json;base64,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