// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
export function compareNumber(a, b) {
  if (isNaN(a) || isNaN(b)) throw new Error("Comparison against non-numbers");
  return a === b ? 0 : a < b ? -1 : 1;
}
export function checkIdentifier(v1 = [], v2 = []) {
  // NOT having a prerelease is > having one
  // But NOT having a build is < having one
  if (v1.length && !v2.length) return -1;
  if (!v1.length && v2.length) return 1;
  return 0;
}
export function compareIdentifier(v1 = [], v2 = []) {
  const length = Math.max(v1.length, v2.length);
  for(let i = 0; i < length; i++){
    const a = v1[i];
    const b = v2[i];
    // same length is equal
    if (a === undefined && b === undefined) return 0;
    // longer > shorter
    if (b === undefined) return 1;
    // shorter < longer
    if (a === undefined) return -1;
    // string > number
    if (typeof a === "string" && typeof b === "number") return 1;
    // number < string
    if (typeof a === "number" && typeof b === "string") return -1;
    if (a < b) return -1;
    if (a > b) return 1;
  // If they're equal, continue comparing segments.
  }
  return 0;
}
/**
 * A single `0`, or a non-zero digit followed by zero or more digits.
 */ const NUMERIC_IDENTIFIER = "0|[1-9]\\d*";
/**
 * Zero or more digits, followed by a letter or hyphen, and then zero or more letters, digits, or hyphens.
 */ const NON_NUMERIC_IDENTIFIER = "\\d*[a-zA-Z-][a-zA-Z0-9-]*";
/**
 * Three dot-separated numeric identifiers.
 */ const VERSION_CORE = `(?<major>${NUMERIC_IDENTIFIER})\\.(?<minor>${NUMERIC_IDENTIFIER})\\.(?<patch>${NUMERIC_IDENTIFIER})`;
/**
 * A numeric identifier, or a non-numeric identifier.
 */ const PRERELEASE_IDENTIFIER = `(?:${NUMERIC_IDENTIFIER}|${NON_NUMERIC_IDENTIFIER})`;
/**
 * A hyphen, followed by one or more dot-separated pre-release version identifiers.
 * @example "-pre.release"
 */ const PRERELEASE = `(?:-(?<prerelease>${PRERELEASE_IDENTIFIER}(?:\\.${PRERELEASE_IDENTIFIER})*))`;
/**
 * Any combination of digits, letters, or hyphens.
 */ const BUILD_IDENTIFIER = "[0-9A-Za-z-]+";
/**
 * A plus sign, followed by one or more period-separated build metadata identifiers.
 * @example "+build.meta"
 */ const BUILD = `(?:\\+(?<buildmetadata>${BUILD_IDENTIFIER}(?:\\.${BUILD_IDENTIFIER})*))`;
/**
 * A version, followed optionally by a pre-release version and build metadata.
 */ const FULL_VERSION = `v?${VERSION_CORE}${PRERELEASE}?${BUILD}?`;
export const FULL_REGEXP = new RegExp(`^${FULL_VERSION}$`);
/**
 * A comparator.
 * @example `=`, `<`, `<=`, `>`, `>=`
 */ const COMPARATOR = "(?:<|>)?=?";
/**
 * A wildcard identifier.
 * @example "x", "X", "*"
 */ const WILDCARD_IDENTIFIER = `x|X|\\*`;
const XRANGE_IDENTIFIER = `${NUMERIC_IDENTIFIER}|${WILDCARD_IDENTIFIER}`;
/**
 * A version that can contain wildcards.
 * @example "x", "1.x", "1.x.x", "1.2.x", "*", "1.*", "1.*.*", "1.2.*"
 */ export const XRANGE = `[v=\\s]*(?<major>${XRANGE_IDENTIFIER})(?:\\.(?<minor>${XRANGE_IDENTIFIER})(?:\\.(?<patch>${XRANGE_IDENTIFIER})${PRERELEASE}?${BUILD}?)?)?`;
/**
 * An operator (`~`, `~>`, `^`, `=`, `<`, `<=`, `>`, or `>=`), followed by an x-range.
 * @example "~1.x.x", "^1.2.*", ">=1.2.3"
 */ export const OPERATOR_XRANGE_REGEXP = new RegExp(`^(?<operator>~>?|\\^|${COMPARATOR})\\s*${XRANGE}$`);
/**
 * An empty string or a comparator (`=`, `<`, `<=`, `>`, or `>=`), followed by a version.
 * @example ">1.2.3"
 */ export const COMPARATOR_REGEXP = new RegExp(`^(?<operator>${COMPARATOR})\\s*(${FULL_VERSION})$|^$`);
/**
 * Returns true if the value is a valid SemVer number.
 *
 * Must be a number. Must not be NaN. Can be positive or negative infinity.
 * Can be between 0 and MAX_SAFE_INTEGER.
 * @param value The value to check
 * @returns True if its a valid semver number
 */ export function isValidNumber(value) {
  return typeof value === "number" && !Number.isNaN(value) && (!Number.isFinite(value) || 0 <= value && value <= Number.MAX_SAFE_INTEGER);
}
export const MAX_LENGTH = 256;
/**
 * Returns true if the value is a valid semver pre-release or build identifier.
 *
 * Must be a string. Must be between 1 and 256 characters long. Must match
 * the regular expression /[0-9A-Za-z-]+/.
 * @param value The value to check
 * @returns True if the value is a valid semver string.
 */ export function isValidString(value) {
  return typeof value === "string" && value.length > 0 && value.length <= MAX_LENGTH && /[0-9A-Za-z-]+/.test(value);
}
const NUMERIC_IDENTIFIER_REGEXP = new RegExp(`^${NUMERIC_IDENTIFIER}$`);
export function parsePrerelease(prerelease) {
  return prerelease.split(".").filter(Boolean).map((id)=>{
    if (NUMERIC_IDENTIFIER_REGEXP.test(id)) {
      const number = Number(id);
      if (isValidNumber(number)) return number;
    }
    return id;
  });
}
export function parseBuild(buildmetadata) {
  return buildmetadata.split(".").filter(Boolean);
}
export function parseNumber(input, errorMessage) {
  const number = Number(input);
  if (!isValidNumber(number)) throw new TypeError(errorMessage);
  return number;
}
export function isWildcardComparator(c) {
  return Number.isNaN(c.major) && Number.isNaN(c.minor) && Number.isNaN(c.patch) && (c.prerelease === undefined || c.prerelease.length === 0) && (c.build === undefined || c.build.length === 0);
}
//# sourceMappingURL=data:application/json;base64,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