// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { convertRowToObject, defaultReadOptions, parseRecord } from "../csv/_io.ts";
import { TextDelimiterStream } from "../streams/text_delimiter_stream.ts";
class StreamLineReader {
  #reader;
  #done = false;
  constructor(reader){
    this.#reader = reader;
  }
  async readLine() {
    const { value, done } = await this.#reader.read();
    if (done) {
      this.#done = true;
      return null;
    } else {
      // NOTE: Remove trailing CR for compatibility with golang's `encoding/csv`
      return stripLastCR(value);
    }
  }
  isEOF() {
    return this.#done;
  }
  cancel() {
    this.#reader.cancel();
  }
}
function stripLastCR(s) {
  return s.endsWith("\r") ? s.slice(0, -1) : s;
}
/**
 * Read data from a CSV-encoded stream or file. Provides an auto/custom mapper
 * for columns.
 *
 * A `CsvParseStream` expects input conforming to
 * {@link https://tools.ietf.org/html/rfc4180 | RFC 4180}.
 *
 * @example
 * ```ts
 * import { CsvParseStream } from "https://deno.land/std@$STD_VERSION/csv/csv_parse_stream.ts";
 * const res = await fetch("https://example.com/data.csv");
 * const parts = res.body!
 *   .pipeThrough(new TextDecoderStream())
 *   .pipeThrough(new CsvParseStream());
 * ```
 */ export class CsvParseStream {
  #readable;
  #options;
  #lineReader;
  #lines;
  #lineIndex = 0;
  #isFirstRow = true;
  #headers = [];
  /** Construct a new instance. */ constructor(options){
    this.#options = {
      ...defaultReadOptions,
      ...options
    };
    this.#lines = new TextDelimiterStream("\n");
    this.#lineReader = new StreamLineReader(this.#lines.readable.getReader());
    this.#readable = new ReadableStream({
      pull: (controller)=>this.#pull(controller),
      cancel: ()=>this.#lineReader.cancel()
    });
  }
  async #pull(controller) {
    const line = await this.#lineReader.readLine();
    if (line === "") {
      // Found an empty line
      this.#lineIndex++;
      return this.#pull(controller);
    }
    if (line === null) {
      // Reached to EOF
      controller.close();
      this.#lineReader.cancel();
      return;
    }
    const record = await parseRecord(line, this.#lineReader, this.#options, this.#lineIndex);
    if (record === null) {
      controller.close();
      this.#lineReader.cancel();
      return;
    }
    if (this.#isFirstRow) {
      this.#isFirstRow = false;
      if (this.#options.skipFirstRow || this.#options.columns) {
        this.#headers = [];
        if (this.#options.skipFirstRow) {
          const head = record;
          this.#headers = head;
        }
        if (this.#options.columns) {
          this.#headers = this.#options.columns;
        }
      }
      if (this.#options.skipFirstRow) {
        return this.#pull(controller);
      }
    }
    this.#lineIndex++;
    if (record.length > 0) {
      if (this.#options.skipFirstRow || this.#options.columns) {
        controller.enqueue(convertRowToObject(record, this.#headers, this.#lineIndex));
      } else {
        controller.enqueue(record);
      }
    } else {
      return this.#pull(controller);
    }
  }
  /** The instance's {@linkcode ReadableStream}. */ get readable() {
    return this.#readable;
  }
  /** The instance's {@linkcode WritableStream}. */ get writable() {
    return this.#lines.writable;
  }
}
//# sourceMappingURL=data:application/json;base64,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