// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/*!
 * Adapted directly from negotiator at https://github.com/jshttp/negotiator/
 * which is licensed as follows:
 *
 * (The MIT License)
 *
 * Copyright (c) 2012-2014 Federico Romero
 * Copyright (c) 2012-2014 Isaac Z. Schlueter
 * Copyright (c) 2014-2015 Douglas Christopher Wilson
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files (the
 * 'Software'), to deal in the Software without restriction, including
 * without limitation the rights to use, copy, modify, merge, publish,
 * distribute, sublicense, and/or sell copies of the Software, and to
 * permit persons to whom the Software is furnished to do so, subject to
 * the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED 'AS IS', WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */ import { compareSpecs, isQuality } from "./common.ts";
const simpleMediaTypeRegExp = /^\s*([^\s\/;]+)\/([^;\s]+)\s*(?:;(.*))?$/;
function splitKeyValuePair(str) {
  const [key, value] = str.split("=");
  return [
    key.toLowerCase(),
    value
  ];
}
function parseMediaType(str, i) {
  const match = simpleMediaTypeRegExp.exec(str);
  if (!match) {
    return;
  }
  const [, type, subtype, parameters] = match;
  if (!type || !subtype) {
    return;
  }
  const params = Object.create(null);
  let q = 1;
  if (parameters) {
    const kvps = parameters.split(";").map((p)=>p.trim()).map(splitKeyValuePair);
    for (const [key, val] of kvps){
      const value = val && val[0] === `"` && val[val.length - 1] === `"` ? val.slice(1, val.length - 1) : val;
      if (key === "q" && value) {
        q = parseFloat(value);
        break;
      }
      params[key] = value;
    }
  }
  return {
    type,
    subtype,
    params,
    q,
    i
  };
}
function parseAccept(accept) {
  const accepts = accept.split(",").map((p)=>p.trim());
  const mediaTypes = [];
  for (const [index, accept] of accepts.entries()){
    const mediaType = parseMediaType(accept.trim(), index);
    if (mediaType) {
      mediaTypes.push(mediaType);
    }
  }
  return mediaTypes;
}
function getFullType(spec) {
  return `${spec.type}/${spec.subtype}`;
}
function specify(type, spec, index) {
  const p = parseMediaType(type, index);
  if (!p) {
    return;
  }
  let s = 0;
  if (spec.type.toLowerCase() === p.type.toLowerCase()) {
    s |= 4;
  } else if (spec.type !== "*") {
    return;
  }
  if (spec.subtype.toLowerCase() === p.subtype.toLowerCase()) {
    s |= 2;
  } else if (spec.subtype !== "*") {
    return;
  }
  const keys = Object.keys(spec.params);
  if (keys.length) {
    if (keys.every((key)=>(spec.params[key] || "").toLowerCase() === (p.params[key] || "").toLowerCase())) {
      s |= 1;
    } else {
      return;
    }
  }
  return {
    i: index,
    o: spec.o,
    q: spec.q,
    s
  };
}
function getMediaTypePriority(type, accepted, index) {
  let priority = {
    o: -1,
    q: 0,
    s: 0,
    i: index
  };
  for (const accepts of accepted){
    const spec = specify(type, accepts, index);
    if (spec && ((priority.s || 0) - (spec.s || 0) || (priority.q || 0) - (spec.q || 0) || (priority.o || 0) - (spec.o || 0)) < 0) {
      priority = spec;
    }
  }
  return priority;
}
export function preferredMediaTypes(accept, provided) {
  const accepts = parseAccept(accept === undefined ? "*/*" : accept || "");
  if (!provided) {
    return accepts.filter(isQuality).sort(compareSpecs).map(getFullType);
  }
  const priorities = provided.map((type, index)=>{
    return getMediaTypePriority(type, accepts, index);
  });
  return priorities.filter(isQuality).sort(compareSpecs).map((priority)=>provided[priorities.indexOf(priority)]);
}
//# sourceMappingURL=data:application/json;base64,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