// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Generators and validators for UUIDs for versions v1, v3, v4 and v5.
 *
 * Use {@linkcode crypto.randomUUID} for v4 generating v4 UUIDs.
 *
 * Based on https://github.com/kelektiv/node-uuid -> https://www.ietf.org/rfc/rfc4122.txt
 *
 * Support for RFC4122 version 1, 3, 4, and 5 UUIDs
 *
 * This module is browser compatible.
 *
 * @module
 */ export * from "./constants.ts";
export * as v1 from "./v1.ts";
export * as v3 from "./v3.ts";
export * as v4 from "./v4.ts";
export * as v5 from "./v5.ts";
/**
 * The nil UUID is special form of UUID that is specified to have all 128 bits
 * set to zero.
 */ export const NIL_UUID = "00000000-0000-0000-0000-000000000000";
/**
 * Check if the passed UUID is the nil UUID.
 *
 * @example
 * ```ts
 * import { isNil } from "https://deno.land/std@$STD_VERSION/uuid/mod.ts";
 *
 * isNil("00000000-0000-0000-0000-000000000000"); // true
 * isNil(crypto.randomUUID()); // false
 * ```
 */ export function isNil(id) {
  return id === NIL_UUID;
}
/**
 * Test a string to see if it is a valid UUID.
 *
 * @example
 * ```ts
 * import { validate } from "https://deno.land/std@$STD_VERSION/uuid/mod.ts";
 *
 * validate("not a UUID"); // false
 * validate("6ec0bd7f-11c0-43da-975e-2a8ad9ebae0b"); // true
 * ```
 */ export function validate(uuid) {
  return /^(?:[0-9a-f]{8}-[0-9a-f]{4}-[1-5][0-9a-f]{3}-[89ab][0-9a-f]{3}-[0-9a-f]{12}|00000000-0000-0000-0000-000000000000)$/i.test(uuid);
}
/**
 * Detect RFC version of a UUID.
 *
 * @example
 * ```ts
 * import { version } from "https://deno.land/std@$STD_VERSION/uuid/mod.ts";
 *
 * version("d9428888-122b-11e1-b85c-61cd3cbb3210"); // 1
 * version("109156be-c4fb-41ea-b1b4-efe1671c5836"); // 4
 * ```
 */ export function version(uuid) {
  if (!validate(uuid)) {
    throw new TypeError("Invalid UUID");
  }
  return parseInt(uuid[14], 16);
}
//# sourceMappingURL=data:application/json;base64,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