// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
var _computedKey;
_computedKey = Symbol.asyncIterator;
/**
 * Multiplexes multiple async iterators into a single stream. It currently
 * makes an assumption that the final result (the value returned and not
 * yielded from the iterator) does not matter; if there is any result, it is
 * discarded.
 *
 * @example
 * ```ts
 * import { MuxAsyncIterator } from "https://deno.land/std@$STD_VERSION/async/mux_async_iterator.ts";
 *
 * async function* gen123(): AsyncIterableIterator<number> {
 *   yield 1;
 *   yield 2;
 *   yield 3;
 * }
 *
 * async function* gen456(): AsyncIterableIterator<number> {
 *   yield 4;
 *   yield 5;
 *   yield 6;
 * }
 *
 * const mux = new MuxAsyncIterator<number>();
 * mux.add(gen123());
 * mux.add(gen456());
 * for await (const value of mux) {
 *   // ...
 * }
 * // ..
 * ```
 */ export class MuxAsyncIterator {
  #iteratorCount = 0;
  #yields = [];
  // deno-lint-ignore no-explicit-any
  #throws = [];
  #signal = Promise.withResolvers();
  /** Add an async iterable to the stream. */ add(iterable) {
    ++this.#iteratorCount;
    this.#callIteratorNext(iterable[Symbol.asyncIterator]());
  }
  async #callIteratorNext(iterator) {
    try {
      const { value, done } = await iterator.next();
      if (done) {
        --this.#iteratorCount;
      } else {
        this.#yields.push({
          iterator,
          value
        });
      }
    } catch (e) {
      this.#throws.push(e);
    }
    this.#signal.resolve();
  }
  /** Returns an async iterator of the stream. */ async *iterate() {
    while(this.#iteratorCount > 0){
      // Sleep until any of the wrapped iterators yields.
      await this.#signal.promise;
      // Note that while we're looping over `yields`, new items may be added.
      for (const { iterator, value } of this.#yields){
        yield value;
        this.#callIteratorNext(iterator);
      }
      if (this.#throws.length) {
        for (const e of this.#throws){
          throw e;
        }
        this.#throws.length = 0;
      }
      // Clear the `yields` list and reset the `signal` promise.
      this.#yields.length = 0;
      this.#signal = Promise.withResolvers();
    }
  }
  /** Implements an async iterator for the stream. */ [_computedKey]() {
    return this.iterate();
  }
}
//# sourceMappingURL=data:application/json;base64,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