// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Extensions to the
 * {@link https://developer.mozilla.org/en-US/docs/Web/API/Web_Crypto_API | Web Crypto API}
 * supporting additional encryption APIs, but also delegating to the built-in
 * APIs when possible.
 *
 * Provides additional digest algorithms that are not part of the WebCrypto
 * standard as well as a `subtle.digest` and `subtle.digestSync` methods.
 *
 * The {@linkcode KeyStack} export implements the {@linkcode KeyRing} interface
 * for managing rotatable keys for signing data to prevent tampering, like with
 * HTTP cookies.
 *
 * ## Supported algorithms
 *
 * Here is a list of supported algorithms. If the algorithm name in WebCrypto
 * and Wasm/Rust is the same, this library prefers to use algorithms that are
 * supported by WebCrypto.
 *
 * WebCrypto:
 * - `SHA-384`
 * - `SHA-256`
 * - `SHA-512` (length-extendable and collidable)
 *
 * Wasm/Rust:
 * - `BLAKE2B-128`
 * - `BLAKE2B-160`
 * - `BLAKE2B-224`
 * - `BLAKE2B-256`
 * - `BLAKE2B-384`
 * - `BLAKE2B`
 * - `BLAKE2S`
 * - `BLAKE3`
 * - `FNV32` (length-extendable)
 * - `FNV32A` (length-extendable)
 * - `FNV64` (length-extendable)
 * - `FNV64A` (length-extendable)
 * - `KECCAK-224`
 * - `KECCAK-256`
 * - `KECCAK-384`
 * - `KECCAK-512`
 * - `SHA-384`
 * - `SHA3-224`
 * - `SHA3-256`
 * - `SHA3-384`
 * - `SHA3-512`
 * - `SHAKE128`
 * - `SHAKE256`
 * - `TIGER`
 * - `RIPEMD-160` (length-extendable)
 * - `SHA-224` (length-extendable)
 * - `SHA-256` (length-extendable)
 * - `SHA-512` (length-extendable)
 * - `MD4` (collidable and length-extendable)
 * - `MD5` (collidable and length-extendable)
 * - `SHA-1` (collidable and length-extendable)
 * ```
 *
 * @example
 * ```ts
 * import { crypto } from "https://deno.land/std@$STD_VERSION/crypto/mod.ts";
 *
 * // This will delegate to the runtime's WebCrypto implementation.
 * console.log(
 *   new Uint8Array(
 *     await crypto.subtle.digest(
 *       "SHA-384",
 *       new TextEncoder().encode("hello world"),
 *     ),
 *   ),
 * );
 *
 * // This will use a bundled Wasm/Rust implementation.
 * console.log(
 *   new Uint8Array(
 *     await crypto.subtle.digest(
 *       "BLAKE3",
 *       new TextEncoder().encode("hello world"),
 *     ),
 *   ),
 * );
 * ```
 *
 * @example Convert hash to a string
 *
 * ```ts
 * import {
 *   crypto,
 * } from "https://deno.land/std@$STD_VERSION/crypto/mod.ts";
 * import { encodeHex } from "https://deno.land/std@$STD_VERSION/encoding/hex.ts"
 * import { encodeBase64 } from "https://deno.land/std@$STD_VERSION/encoding/base64.ts"
 *
 * const hash = await crypto.subtle.digest(
 *   "SHA-384",
 *   new TextEncoder().encode("You hear that Mr. Anderson?"),
 * );
 *
 * // Hex encoding
 * console.log(encodeHex(hash));
 *
 * // Or with base64 encoding
 * console.log(encodeBase64(hash));
 * ```
 *
 * @module
 */ import { digestAlgorithms as wasmDigestAlgorithms, instantiateWasm } from "./_wasm/mod.ts";
import { fnv } from "./_fnv/mod.ts";
export { wasmDigestAlgorithms };
/**
 * A copy of the global WebCrypto interface, with methods bound so they're
 * safe to re-export.
 */ const webCrypto = ((crypto)=>({
    getRandomValues: crypto.getRandomValues?.bind(crypto),
    randomUUID: crypto.randomUUID?.bind(crypto),
    subtle: {
      decrypt: crypto.subtle?.decrypt?.bind(crypto.subtle),
      deriveBits: crypto.subtle?.deriveBits?.bind(crypto.subtle),
      deriveKey: crypto.subtle?.deriveKey?.bind(crypto.subtle),
      digest: crypto.subtle?.digest?.bind(crypto.subtle),
      encrypt: crypto.subtle?.encrypt?.bind(crypto.subtle),
      exportKey: crypto.subtle?.exportKey?.bind(crypto.subtle),
      generateKey: crypto.subtle?.generateKey?.bind(crypto.subtle),
      importKey: crypto.subtle?.importKey?.bind(crypto.subtle),
      sign: crypto.subtle?.sign?.bind(crypto.subtle),
      unwrapKey: crypto.subtle?.unwrapKey?.bind(crypto.subtle),
      verify: crypto.subtle?.verify?.bind(crypto.subtle),
      wrapKey: crypto.subtle?.wrapKey?.bind(crypto.subtle)
    }
  }))(globalThis.crypto);
const bufferSourceBytes = (data)=>{
  let bytes;
  if (data instanceof Uint8Array) {
    bytes = data;
  } else if (ArrayBuffer.isView(data)) {
    bytes = new Uint8Array(data.buffer, data.byteOffset, data.byteLength);
  } else if (data instanceof ArrayBuffer) {
    bytes = new Uint8Array(data);
  }
  return bytes;
};
/**
 * An wrapper for WebCrypto adding support for additional non-standard
 * algorithms, but delegating to the runtime WebCrypto implementation whenever
 * possible.
 */ const stdCrypto = ((x)=>x)({
  ...webCrypto,
  subtle: {
    ...webCrypto.subtle,
    /**
     * Polyfills stream support until the Web Crypto API does so:
     * @see {@link https://github.com/wintercg/proposal-webcrypto-streams}
     */ async digest (algorithm, data) {
      const { name, length } = normalizeAlgorithm(algorithm);
      assertValidDigestLength(length);
      const bytes = bufferSourceBytes(data);
      if (FNV_ALGORITHMS.includes(name)) {
        return fnv(name, bytes);
      }
      // We delegate to WebCrypto whenever possible,
      if (// if the algorithm is supported by the WebCrypto standard,
      webCryptoDigestAlgorithms.includes(name) && // and the data is a single buffer,
      bytes) {
        return webCrypto.subtle.digest(algorithm, bytes);
      } else if (wasmDigestAlgorithms.includes(name)) {
        if (bytes) {
          // Otherwise, we use our bundled Wasm implementation via digestSync
          // if it supports the algorithm.
          return stdCrypto.subtle.digestSync(algorithm, bytes);
        } else if (data[Symbol.iterator]) {
          return stdCrypto.subtle.digestSync(algorithm, data);
        } else if (data[Symbol.asyncIterator]) {
          const wasmCrypto = instantiateWasm();
          const context = new wasmCrypto.DigestContext(name);
          for await (const chunk of data){
            const chunkBytes = bufferSourceBytes(chunk);
            if (!chunkBytes) {
              throw new TypeError("data contained chunk of the wrong type");
            }
            context.update(chunkBytes);
          }
          return context.digestAndDrop(length).buffer;
        } else {
          throw new TypeError("data must be a BufferSource or [Async]Iterable<BufferSource>");
        }
      } else if (webCrypto.subtle?.digest) {
        // (TypeScript type definitions prohibit this case.) If they're trying
        // to call an algorithm we don't recognize, pass it along to WebCrypto
        // in case it's a non-standard algorithm supported by the the runtime
        // they're using.
        return webCrypto.subtle.digest(algorithm, data);
      } else {
        throw new TypeError(`unsupported digest algorithm: ${algorithm}`);
      }
    },
    digestSync (algorithm, data) {
      const { name, length } = normalizeAlgorithm(algorithm);
      assertValidDigestLength(length);
      const bytes = bufferSourceBytes(data);
      if (FNV_ALGORITHMS.includes(name)) {
        return fnv(name, bytes);
      }
      const wasmCrypto = instantiateWasm();
      if (bytes) {
        return wasmCrypto.digest(name, bytes, length).buffer;
      } else if (data[Symbol.iterator]) {
        const context = new wasmCrypto.DigestContext(name);
        for (const chunk of data){
          const chunkBytes = bufferSourceBytes(chunk);
          if (!chunkBytes) {
            throw new TypeError("data contained chunk of the wrong type");
          }
          context.update(chunkBytes);
        }
        return context.digestAndDrop(length).buffer;
      } else {
        throw new TypeError("data must be a BufferSource or Iterable<BufferSource>");
      }
    }
  }
});
const FNV_ALGORITHMS = [
  "FNV32",
  "FNV32A",
  "FNV64",
  "FNV64A"
];
/** Digest algorithms supported by WebCrypto. */ const webCryptoDigestAlgorithms = [
  "SHA-384",
  "SHA-256",
  "SHA-512",
  // insecure (length-extendable and collidable):
  "SHA-1"
];
/*
 * The largest digest length the current WASM implementation can support. This
 * is the value of `isize::MAX` on 32-bit platforms like WASM, which is the
 * maximum allowed capacity of a Rust `Vec<u8>`.
 */ const MAX_DIGEST_LENGTH = 0x7FFF_FFFF;
/**
 * Asserts that a number is a valid length for a digest, which must be an
 * integer that fits in a Rust `Vec<u8>`, or be undefined.
 */ function assertValidDigestLength(value) {
  if (value !== undefined && (value < 0 || value > MAX_DIGEST_LENGTH || !Number.isInteger(value))) {
    throw new RangeError(`length must be an integer between 0 and ${MAX_DIGEST_LENGTH}, inclusive`);
  }
}
function normalizeAlgorithm(algorithm) {
  return typeof algorithm === "string" ? {
    name: algorithm.toUpperCase()
  } : {
    ...algorithm,
    name: algorithm.name.toUpperCase()
  };
}
export { stdCrypto as crypto };
//# sourceMappingURL=data:application/json;base64,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