// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { concat } from "../bytes/concat.ts";
import { createLPS } from "./_common.ts";
/**
 * Divide a stream into chunks delimited by a given byte sequence.
 *
 * @example
 * Divide a CSV stream by commas, discarding the commas:
 * ```ts
 * import { DelimiterStream } from "https://deno.land/std@$STD_VERSION/streams/delimiter_stream.ts";
 * const res = await fetch("https://example.com/data.csv");
 * const parts = res.body!
 *   .pipeThrough(new DelimiterStream(new TextEncoder().encode(",")))
 *   .pipeThrough(new TextDecoderStream());
 * ```
 *
 * @example
 * Divide a stream after semi-colons, keeping the semi-colons in the output:
 * ```ts
 * import { DelimiterStream } from "https://deno.land/std@$STD_VERSION/streams/delimiter_stream.ts";
 * const res = await fetch("https://example.com/file.js");
 * const parts = res.body!
 *   .pipeThrough(
 *     new DelimiterStream(
 *       new TextEncoder().encode(";"),
 *       { disposition: "suffix" },
 *     )
 *   )
 *   .pipeThrough(new TextDecoderStream());
 * ```
 */ export class DelimiterStream extends TransformStream {
  #bufs = [];
  #delimiter;
  #matchIndex = 0;
  #delimLPS;
  #disp;
  /** Constructs a new instance. */ constructor(delimiter, options){
    super({
      transform: (chunk, controller)=>delimiter.length === 1 ? this.#handleChar(chunk, controller) : this.#handle(chunk, controller),
      flush: (controller)=>this.#flush(controller)
    });
    this.#delimiter = delimiter;
    this.#delimLPS = delimiter.length > 1 ? createLPS(delimiter) : null;
    this.#disp = options?.disposition ?? "discard";
  }
  #handle(chunk, controller) {
    const bufs = this.#bufs;
    const length = chunk.byteLength;
    const disposition = this.#disp;
    const delimiter = this.#delimiter;
    const delimLen = delimiter.length;
    const lps = this.#delimLPS;
    let chunkStart = 0;
    let matchIndex = this.#matchIndex;
    let inspectIndex = 0;
    while(inspectIndex < length){
      if (chunk[inspectIndex] === delimiter[matchIndex]) {
        // Next byte matched our next delimiter byte
        inspectIndex++;
        matchIndex++;
        if (matchIndex === delimLen) {
          // Full match
          matchIndex = 0;
          const delimiterStartIndex = inspectIndex - delimLen;
          const delimitedChunkEnd = disposition === "suffix" ? inspectIndex : delimiterStartIndex;
          if (delimitedChunkEnd <= 0 && bufs.length === 0) {
            // Our chunk started with a delimiter and no previous chunks exist:
            // Enqueue an empty chunk.
            controller.enqueue(new Uint8Array());
            chunkStart = disposition === "prefix" ? 0 : inspectIndex;
          } else if (delimitedChunkEnd > 0 && bufs.length === 0) {
            // No previous chunks, slice from current chunk.
            controller.enqueue(chunk.subarray(chunkStart, delimitedChunkEnd));
            // Our chunk may have more than one delimiter; we must remember where
            // the next delimited chunk begins.
            chunkStart = disposition === "prefix" ? delimiterStartIndex : inspectIndex;
          } else if (delimitedChunkEnd === 0 && bufs.length > 0) {
            // Our chunk started with a delimiter, previous chunks are passed as
            // they are (with concatenation).
            if (bufs.length === 1) {
              // Concat not needed when a single buffer is passed.
              controller.enqueue(bufs[0]);
            } else {
              controller.enqueue(concat(bufs));
            }
            // Drop all previous chunks.
            bufs.length = 0;
            if (disposition !== "prefix") {
              // suffix or discard: The next chunk starts where our inspection finished.
              // We should only ever end up here with a discard disposition as
              // for a suffix disposition this branch would mean that the previous
              // chunk ended with a full match but was not enqueued.
              chunkStart = inspectIndex;
            } else {
              chunkStart = 0;
            }
          } else if (delimitedChunkEnd < 0 && bufs.length > 0) {
            // Our chunk started by finishing a partial delimiter match.
            const lastIndex = bufs.length - 1;
            const last = bufs[lastIndex];
            const lastSliceIndex = last.byteLength + delimitedChunkEnd;
            const lastSliced = last.subarray(0, lastSliceIndex);
            if (lastIndex === 0) {
              controller.enqueue(lastSliced);
            } else {
              bufs[lastIndex] = lastSliced;
              controller.enqueue(concat(bufs));
            }
            bufs.length = 0;
            if (disposition === "prefix") {
              // Must keep last bytes of last chunk.
              bufs.push(last.subarray(lastSliceIndex));
              chunkStart = 0;
            } else {
              chunkStart = inspectIndex;
            }
          } else if (delimitedChunkEnd > 0 && bufs.length > 0) {
            // Previous chunks and current chunk together form a delimited chunk.
            const chunkSliced = chunk.subarray(chunkStart, delimitedChunkEnd);
            const result = concat([
              ...bufs,
              chunkSliced
            ]);
            bufs.length = 0;
            controller.enqueue(result);
            chunkStart = disposition === "prefix" ? delimitedChunkEnd : inspectIndex;
          } else {
            throw new Error("unreachable");
          }
        }
      } else if (matchIndex === 0) {
        // No match ongoing, keep going through the buffer.
        inspectIndex++;
      } else {
        // Ongoing match: Degrade to the previous possible match.
        // eg. If we're looking for 'AAB' and had matched 'AA' previously
        // but now got a new 'A', then we'll drop down to having matched
        // just 'A'. The while loop will turn around again and we'll rematch
        // to 'AA' and proceed onwards to try and match on 'B' again.
        matchIndex = lps[matchIndex - 1];
      }
    }
    // Save match index.
    this.#matchIndex = matchIndex;
    if (chunkStart === 0) {
      bufs.push(chunk);
    } else if (chunkStart < length) {
      // If we matched partially somewhere in the middle of our chunk
      // then the remnants should be pushed into buffers.
      bufs.push(chunk.subarray(chunkStart));
    }
  }
  /**
   * Optimized handler for a char delimited stream:
   *
   * For char delimited streams we do not need to keep track of
   * the match index, removing the need for a fair bit of work.
   */ #handleChar(chunk, controller) {
    const bufs = this.#bufs;
    const length = chunk.byteLength;
    const disposition = this.#disp;
    const delimiter = this.#delimiter[0];
    let chunkStart = 0;
    let inspectIndex = 0;
    while(inspectIndex < length){
      if (chunk[inspectIndex] === delimiter) {
        // Next byte matched our next delimiter
        inspectIndex++;
        /**
         * Always non-negative
         */ const delimitedChunkEnd = disposition === "suffix" ? inspectIndex : inspectIndex - 1;
        if (delimitedChunkEnd === 0 && bufs.length === 0) {
          // Our chunk started with a delimiter and no previous chunks exist:
          // Enqueue an empty chunk.
          controller.enqueue(new Uint8Array());
          chunkStart = disposition === "prefix" ? 0 : 1;
        } else if (delimitedChunkEnd > 0 && bufs.length === 0) {
          // No previous chunks, slice from current chunk.
          controller.enqueue(chunk.subarray(chunkStart, delimitedChunkEnd));
          // Our chunk may have more than one delimiter; we must remember where
          // the next delimited chunk begins.
          chunkStart = disposition === "prefix" ? inspectIndex - 1 : inspectIndex;
        } else if (delimitedChunkEnd === 0 && bufs.length > 0) {
          // Our chunk started with a delimiter, previous chunks are passed as
          // they are (with concatenation).
          if (bufs.length === 1) {
            // Concat not needed when a single buffer is passed.
            controller.enqueue(bufs[0]);
          } else {
            controller.enqueue(concat(bufs));
          }
          // Drop all previous chunks.
          bufs.length = 0;
          if (disposition !== "prefix") {
            // suffix or discard: The next chunk starts where our inspection finished.
            // We should only ever end up here with a discard disposition as
            // for a suffix disposition this branch would mean that the previous
            // chunk ended with a full match but was not enqueued.
            chunkStart = inspectIndex;
          }
        } else if (delimitedChunkEnd > 0 && bufs.length > 0) {
          // Previous chunks and current chunk together form a delimited chunk.
          const chunkSliced = chunk.subarray(chunkStart, delimitedChunkEnd);
          const result = concat([
            ...bufs,
            chunkSliced
          ]);
          bufs.length = 0;
          chunkStart = disposition === "prefix" ? delimitedChunkEnd : inspectIndex;
          controller.enqueue(result);
        } else {
          throw new Error("unreachable");
        }
      } else {
        inspectIndex++;
      }
    }
    if (chunkStart === 0) {
      bufs.push(chunk);
    } else if (chunkStart < length) {
      // If we matched partially somewhere in the middle of our chunk
      // then the remnants should be pushed into buffers.
      bufs.push(chunk.subarray(chunkStart));
    }
  }
  #flush(controller) {
    const bufs = this.#bufs;
    const length = bufs.length;
    if (length === 0) {
      controller.enqueue(new Uint8Array());
    } else if (length === 1) {
      controller.enqueue(bufs[0]);
    } else {
      controller.enqueue(concat(bufs));
    }
  }
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbImh0dHBzOi8vZGVuby5sYW5kL3N0ZEAwLjIxNy4wL3N0cmVhbXMvZGVsaW1pdGVyX3N0cmVhbS50cyJdLCJzb3VyY2VzQ29udGVudCI6WyIvLyBDb3B5cmlnaHQgMjAxOC0yMDI0IHRoZSBEZW5vIGF1dGhvcnMuIEFsbCByaWdodHMgcmVzZXJ2ZWQuIE1JVCBsaWNlbnNlLlxuLy8gVGhpcyBtb2R1bGUgaXMgYnJvd3NlciBjb21wYXRpYmxlLlxuXG5pbXBvcnQgeyBjb25jYXQgfSBmcm9tIFwiLi4vYnl0ZXMvY29uY2F0LnRzXCI7XG5pbXBvcnQgeyBjcmVhdGVMUFMgfSBmcm9tIFwiLi9fY29tbW9uLnRzXCI7XG5cbi8qKiBEaXNwb3NpdGlvbiBvZiB0aGUgZGVsaW1pdGVyIGZvciB7QGxpbmtjb2RlIERlbGltaXRlclN0cmVhbU9wdGlvbnN9LiAqL1xuZXhwb3J0IHR5cGUgRGVsaW1pdGVyRGlzcG9zaXRpb24gPVxuICAvKiogSW5jbHVkZSBkZWxpbWl0ZXIgaW4gdGhlIGZvdW5kIGNodW5rLiAqL1xuICB8IFwic3VmZml4XCJcbiAgLyoqIEluY2x1ZGUgZGVsaW1pdGVyIGluIHRoZSBzdWJzZXF1ZW50IGNodW5rLiAqL1xuICB8IFwicHJlZml4XCJcbiAgLyoqIERpc2NhcmQgdGhlIGRlbGltaXRlci4gKi9cbiAgfCBcImRpc2NhcmRcIiAvLyBkZWxpbWl0ZXIgZGlzY2FyZGVkXG47XG5cbi8qKiBPcHRpb25zIGZvciB7QGxpbmtjb2RlIERlbGltaXRlclN0cmVhbX0uICovXG5leHBvcnQgaW50ZXJmYWNlIERlbGltaXRlclN0cmVhbU9wdGlvbnMge1xuICAvKiogRGlzcG9zaXRpb24gb2YgdGhlIGRlbGltaXRlci4gKi9cbiAgZGlzcG9zaXRpb24/OiBEZWxpbWl0ZXJEaXNwb3NpdGlvbjtcbn1cblxuLyoqXG4gKiBEaXZpZGUgYSBzdHJlYW0gaW50byBjaHVua3MgZGVsaW1pdGVkIGJ5IGEgZ2l2ZW4gYnl0ZSBzZXF1ZW5jZS5cbiAqXG4gKiBAZXhhbXBsZVxuICogRGl2aWRlIGEgQ1NWIHN0cmVhbSBieSBjb21tYXMsIGRpc2NhcmRpbmcgdGhlIGNvbW1hczpcbiAqIGBgYHRzXG4gKiBpbXBvcnQgeyBEZWxpbWl0ZXJTdHJlYW0gfSBmcm9tIFwiaHR0cHM6Ly9kZW5vLmxhbmQvc3RkQCRTVERfVkVSU0lPTi9zdHJlYW1zL2RlbGltaXRlcl9zdHJlYW0udHNcIjtcbiAqIGNvbnN0IHJlcyA9IGF3YWl0IGZldGNoKFwiaHR0cHM6Ly9leGFtcGxlLmNvbS9kYXRhLmNzdlwiKTtcbiAqIGNvbnN0IHBhcnRzID0gcmVzLmJvZHkhXG4gKiAgIC5waXBlVGhyb3VnaChuZXcgRGVsaW1pdGVyU3RyZWFtKG5ldyBUZXh0RW5jb2RlcigpLmVuY29kZShcIixcIikpKVxuICogICAucGlwZVRocm91Z2gobmV3IFRleHREZWNvZGVyU3RyZWFtKCkpO1xuICogYGBgXG4gKlxuICogQGV4YW1wbGVcbiAqIERpdmlkZSBhIHN0cmVhbSBhZnRlciBzZW1pLWNvbG9ucywga2VlcGluZyB0aGUgc2VtaS1jb2xvbnMgaW4gdGhlIG91dHB1dDpcbiAqIGBgYHRzXG4gKiBpbXBvcnQgeyBEZWxpbWl0ZXJTdHJlYW0gfSBmcm9tIFwiaHR0cHM6Ly9kZW5vLmxhbmQvc3RkQCRTVERfVkVSU0lPTi9zdHJlYW1zL2RlbGltaXRlcl9zdHJlYW0udHNcIjtcbiAqIGNvbnN0IHJlcyA9IGF3YWl0IGZldGNoKFwiaHR0cHM6Ly9leGFtcGxlLmNvbS9maWxlLmpzXCIpO1xuICogY29uc3QgcGFydHMgPSByZXMuYm9keSFcbiAqICAgLnBpcGVUaHJvdWdoKFxuICogICAgIG5ldyBEZWxpbWl0ZXJTdHJlYW0oXG4gKiAgICAgICBuZXcgVGV4dEVuY29kZXIoKS5lbmNvZGUoXCI7XCIpLFxuICogICAgICAgeyBkaXNwb3NpdGlvbjogXCJzdWZmaXhcIiB9LFxuICogICAgIClcbiAqICAgKVxuICogICAucGlwZVRocm91Z2gobmV3IFRleHREZWNvZGVyU3RyZWFtKCkpO1xuICogYGBgXG4gKi9cbmV4cG9ydCBjbGFzcyBEZWxpbWl0ZXJTdHJlYW0gZXh0ZW5kcyBUcmFuc2Zvcm1TdHJlYW08VWludDhBcnJheSwgVWludDhBcnJheT4ge1xuICAjYnVmczogVWludDhBcnJheVtdID0gW107XG4gICNkZWxpbWl0ZXI6IFVpbnQ4QXJyYXk7XG4gICNtYXRjaEluZGV4ID0gMDtcbiAgI2RlbGltTFBTOiBVaW50OEFycmF5IHwgbnVsbDtcbiAgI2Rpc3A6IERlbGltaXRlckRpc3Bvc2l0aW9uO1xuXG4gIC8qKiBDb25zdHJ1Y3RzIGEgbmV3IGluc3RhbmNlLiAqL1xuICBjb25zdHJ1Y3RvcihcbiAgICBkZWxpbWl0ZXI6IFVpbnQ4QXJyYXksXG4gICAgb3B0aW9ucz86IERlbGltaXRlclN0cmVhbU9wdGlvbnMsXG4gICkge1xuICAgIHN1cGVyKHtcbiAgICAgIHRyYW5zZm9ybTogKGNodW5rLCBjb250cm9sbGVyKSA9PlxuICAgICAgICBkZWxpbWl0ZXIubGVuZ3RoID09PSAxXG4gICAgICAgICAgPyB0aGlzLiNoYW5kbGVDaGFyKGNodW5rLCBjb250cm9sbGVyKVxuICAgICAgICAgIDogdGhpcy4jaGFuZGxlKGNodW5rLCBjb250cm9sbGVyKSxcbiAgICAgIGZsdXNoOiAoY29udHJvbGxlcikgPT4gdGhpcy4jZmx1c2goY29udHJvbGxlciksXG4gICAgfSk7XG5cbiAgICB0aGlzLiNkZWxpbWl0ZXIgPSBkZWxpbWl0ZXI7XG4gICAgdGhpcy4jZGVsaW1MUFMgPSBkZWxpbWl0ZXIubGVuZ3RoID4gMSA/IGNyZWF0ZUxQUyhkZWxpbWl0ZXIpIDogbnVsbDtcbiAgICB0aGlzLiNkaXNwID0gb3B0aW9ucz8uZGlzcG9zaXRpb24gPz8gXCJkaXNjYXJkXCI7XG4gIH1cblxuICAjaGFuZGxlKFxuICAgIGNodW5rOiBVaW50OEFycmF5LFxuICAgIGNvbnRyb2xsZXI6IFRyYW5zZm9ybVN0cmVhbURlZmF1bHRDb250cm9sbGVyPFVpbnQ4QXJyYXk+LFxuICApIHtcbiAgICBjb25zdCBidWZzID0gdGhpcy4jYnVmcztcbiAgICBjb25zdCBsZW5ndGggPSBjaHVuay5ieXRlTGVuZ3RoO1xuICAgIGNvbnN0IGRpc3Bvc2l0aW9uID0gdGhpcy4jZGlzcDtcbiAgICBjb25zdCBkZWxpbWl0ZXIgPSB0aGlzLiNkZWxpbWl0ZXI7XG4gICAgY29uc3QgZGVsaW1MZW4gPSBkZWxpbWl0ZXIubGVuZ3RoO1xuICAgIGNvbnN0IGxwcyA9IHRoaXMuI2RlbGltTFBTIGFzIFVpbnQ4QXJyYXk7XG4gICAgbGV0IGNodW5rU3RhcnQgPSAwO1xuICAgIGxldCBtYXRjaEluZGV4ID0gdGhpcy4jbWF0Y2hJbmRleDtcbiAgICBsZXQgaW5zcGVjdEluZGV4ID0gMDtcbiAgICB3aGlsZSAoaW5zcGVjdEluZGV4IDwgbGVuZ3RoKSB7XG4gICAgICBpZiAoY2h1bmtbaW5zcGVjdEluZGV4XSA9PT0gZGVsaW1pdGVyW21hdGNoSW5kZXhdKSB7XG4gICAgICAgIC8vIE5leHQgYnl0ZSBtYXRjaGVkIG91ciBuZXh0IGRlbGltaXRlciBieXRlXG4gICAgICAgIGluc3BlY3RJbmRleCsrO1xuICAgICAgICBtYXRjaEluZGV4Kys7XG4gICAgICAgIGlmIChtYXRjaEluZGV4ID09PSBkZWxpbUxlbikge1xuICAgICAgICAgIC8vIEZ1bGwgbWF0Y2hcbiAgICAgICAgICBtYXRjaEluZGV4ID0gMDtcbiAgICAgICAgICBjb25zdCBkZWxpbWl0ZXJTdGFydEluZGV4ID0gaW5zcGVjdEluZGV4IC0gZGVsaW1MZW47XG4gICAgICAgICAgY29uc3QgZGVsaW1pdGVkQ2h1bmtFbmQgPSBkaXNwb3NpdGlvbiA9PT0gXCJzdWZmaXhcIlxuICAgICAgICAgICAgPyBpbnNwZWN0SW5kZXhcbiAgICAgICAgICAgIDogZGVsaW1pdGVyU3RhcnRJbmRleDtcbiAgICAgICAgICBpZiAoZGVsaW1pdGVkQ2h1bmtFbmQgPD0gMCAmJiBidWZzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgICAgLy8gT3VyIGNodW5rIHN0YXJ0ZWQgd2l0aCBhIGRlbGltaXRlciBhbmQgbm8gcHJldmlvdXMgY2h1bmtzIGV4aXN0OlxuICAgICAgICAgICAgLy8gRW5xdWV1ZSBhbiBlbXB0eSBjaHVuay5cbiAgICAgICAgICAgIGNvbnRyb2xsZXIuZW5xdWV1ZShuZXcgVWludDhBcnJheSgpKTtcbiAgICAgICAgICAgIGNodW5rU3RhcnQgPSBkaXNwb3NpdGlvbiA9PT0gXCJwcmVmaXhcIiA/IDAgOiBpbnNwZWN0SW5kZXg7XG4gICAgICAgICAgfSBlbHNlIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA+IDAgJiYgYnVmcy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAgIC8vIE5vIHByZXZpb3VzIGNodW5rcywgc2xpY2UgZnJvbSBjdXJyZW50IGNodW5rLlxuICAgICAgICAgICAgY29udHJvbGxlci5lbnF1ZXVlKGNodW5rLnN1YmFycmF5KGNodW5rU3RhcnQsIGRlbGltaXRlZENodW5rRW5kKSk7XG4gICAgICAgICAgICAvLyBPdXIgY2h1bmsgbWF5IGhhdmUgbW9yZSB0aGFuIG9uZSBkZWxpbWl0ZXI7IHdlIG11c3QgcmVtZW1iZXIgd2hlcmVcbiAgICAgICAgICAgIC8vIHRoZSBuZXh0IGRlbGltaXRlZCBjaHVuayBiZWdpbnMuXG4gICAgICAgICAgICBjaHVua1N0YXJ0ID0gZGlzcG9zaXRpb24gPT09IFwicHJlZml4XCJcbiAgICAgICAgICAgICAgPyBkZWxpbWl0ZXJTdGFydEluZGV4XG4gICAgICAgICAgICAgIDogaW5zcGVjdEluZGV4O1xuICAgICAgICAgIH0gZWxzZSBpZiAoZGVsaW1pdGVkQ2h1bmtFbmQgPT09IDAgJiYgYnVmcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAvLyBPdXIgY2h1bmsgc3RhcnRlZCB3aXRoIGEgZGVsaW1pdGVyLCBwcmV2aW91cyBjaHVua3MgYXJlIHBhc3NlZCBhc1xuICAgICAgICAgICAgLy8gdGhleSBhcmUgKHdpdGggY29uY2F0ZW5hdGlvbikuXG4gICAgICAgICAgICBpZiAoYnVmcy5sZW5ndGggPT09IDEpIHtcbiAgICAgICAgICAgICAgLy8gQ29uY2F0IG5vdCBuZWVkZWQgd2hlbiBhIHNpbmdsZSBidWZmZXIgaXMgcGFzc2VkLlxuICAgICAgICAgICAgICBjb250cm9sbGVyLmVucXVldWUoYnVmc1swXSk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICBjb250cm9sbGVyLmVucXVldWUoY29uY2F0KGJ1ZnMpKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIC8vIERyb3AgYWxsIHByZXZpb3VzIGNodW5rcy5cbiAgICAgICAgICAgIGJ1ZnMubGVuZ3RoID0gMDtcbiAgICAgICAgICAgIGlmIChkaXNwb3NpdGlvbiAhPT0gXCJwcmVmaXhcIikge1xuICAgICAgICAgICAgICAvLyBzdWZmaXggb3IgZGlzY2FyZDogVGhlIG5leHQgY2h1bmsgc3RhcnRzIHdoZXJlIG91ciBpbnNwZWN0aW9uIGZpbmlzaGVkLlxuICAgICAgICAgICAgICAvLyBXZSBzaG91bGQgb25seSBldmVyIGVuZCB1cCBoZXJlIHdpdGggYSBkaXNjYXJkIGRpc3Bvc2l0aW9uIGFzXG4gICAgICAgICAgICAgIC8vIGZvciBhIHN1ZmZpeCBkaXNwb3NpdGlvbiB0aGlzIGJyYW5jaCB3b3VsZCBtZWFuIHRoYXQgdGhlIHByZXZpb3VzXG4gICAgICAgICAgICAgIC8vIGNodW5rIGVuZGVkIHdpdGggYSBmdWxsIG1hdGNoIGJ1dCB3YXMgbm90IGVucXVldWVkLlxuICAgICAgICAgICAgICBjaHVua1N0YXJ0ID0gaW5zcGVjdEluZGV4O1xuICAgICAgICAgICAgfSBlbHNlIHtcbiAgICAgICAgICAgICAgY2h1bmtTdGFydCA9IDA7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA8IDAgJiYgYnVmcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAvLyBPdXIgY2h1bmsgc3RhcnRlZCBieSBmaW5pc2hpbmcgYSBwYXJ0aWFsIGRlbGltaXRlciBtYXRjaC5cbiAgICAgICAgICAgIGNvbnN0IGxhc3RJbmRleCA9IGJ1ZnMubGVuZ3RoIC0gMTtcbiAgICAgICAgICAgIGNvbnN0IGxhc3QgPSBidWZzW2xhc3RJbmRleF07XG4gICAgICAgICAgICBjb25zdCBsYXN0U2xpY2VJbmRleCA9IGxhc3QuYnl0ZUxlbmd0aCArIGRlbGltaXRlZENodW5rRW5kO1xuICAgICAgICAgICAgY29uc3QgbGFzdFNsaWNlZCA9IGxhc3Quc3ViYXJyYXkoMCwgbGFzdFNsaWNlSW5kZXgpO1xuICAgICAgICAgICAgaWYgKGxhc3RJbmRleCA9PT0gMCkge1xuICAgICAgICAgICAgICBjb250cm9sbGVyLmVucXVldWUobGFzdFNsaWNlZCk7XG4gICAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgICBidWZzW2xhc3RJbmRleF0gPSBsYXN0U2xpY2VkO1xuICAgICAgICAgICAgICBjb250cm9sbGVyLmVucXVldWUoY29uY2F0KGJ1ZnMpKTtcbiAgICAgICAgICAgIH1cbiAgICAgICAgICAgIGJ1ZnMubGVuZ3RoID0gMDtcbiAgICAgICAgICAgIGlmIChkaXNwb3NpdGlvbiA9PT0gXCJwcmVmaXhcIikge1xuICAgICAgICAgICAgICAvLyBNdXN0IGtlZXAgbGFzdCBieXRlcyBvZiBsYXN0IGNodW5rLlxuICAgICAgICAgICAgICBidWZzLnB1c2gobGFzdC5zdWJhcnJheShsYXN0U2xpY2VJbmRleCkpO1xuICAgICAgICAgICAgICBjaHVua1N0YXJ0ID0gMDtcbiAgICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICAgIGNodW5rU3RhcnQgPSBpbnNwZWN0SW5kZXg7XG4gICAgICAgICAgICB9XG4gICAgICAgICAgfSBlbHNlIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA+IDAgJiYgYnVmcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgICAvLyBQcmV2aW91cyBjaHVua3MgYW5kIGN1cnJlbnQgY2h1bmsgdG9nZXRoZXIgZm9ybSBhIGRlbGltaXRlZCBjaHVuay5cbiAgICAgICAgICAgIGNvbnN0IGNodW5rU2xpY2VkID0gY2h1bmsuc3ViYXJyYXkoY2h1bmtTdGFydCwgZGVsaW1pdGVkQ2h1bmtFbmQpO1xuICAgICAgICAgICAgY29uc3QgcmVzdWx0ID0gY29uY2F0KFsuLi5idWZzLCBjaHVua1NsaWNlZF0pO1xuICAgICAgICAgICAgYnVmcy5sZW5ndGggPSAwO1xuICAgICAgICAgICAgY29udHJvbGxlci5lbnF1ZXVlKHJlc3VsdCk7XG4gICAgICAgICAgICBjaHVua1N0YXJ0ID0gZGlzcG9zaXRpb24gPT09IFwicHJlZml4XCJcbiAgICAgICAgICAgICAgPyBkZWxpbWl0ZWRDaHVua0VuZFxuICAgICAgICAgICAgICA6IGluc3BlY3RJbmRleDtcbiAgICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgICAgdGhyb3cgbmV3IEVycm9yKFwidW5yZWFjaGFibGVcIik7XG4gICAgICAgICAgfVxuICAgICAgICB9XG4gICAgICB9IGVsc2UgaWYgKG1hdGNoSW5kZXggPT09IDApIHtcbiAgICAgICAgLy8gTm8gbWF0Y2ggb25nb2luZywga2VlcCBnb2luZyB0aHJvdWdoIHRoZSBidWZmZXIuXG4gICAgICAgIGluc3BlY3RJbmRleCsrO1xuICAgICAgfSBlbHNlIHtcbiAgICAgICAgLy8gT25nb2luZyBtYXRjaDogRGVncmFkZSB0byB0aGUgcHJldmlvdXMgcG9zc2libGUgbWF0Y2guXG4gICAgICAgIC8vIGVnLiBJZiB3ZSdyZSBsb29raW5nIGZvciAnQUFCJyBhbmQgaGFkIG1hdGNoZWQgJ0FBJyBwcmV2aW91c2x5XG4gICAgICAgIC8vIGJ1dCBub3cgZ290IGEgbmV3ICdBJywgdGhlbiB3ZSdsbCBkcm9wIGRvd24gdG8gaGF2aW5nIG1hdGNoZWRcbiAgICAgICAgLy8ganVzdCAnQScuIFRoZSB3aGlsZSBsb29wIHdpbGwgdHVybiBhcm91bmQgYWdhaW4gYW5kIHdlJ2xsIHJlbWF0Y2hcbiAgICAgICAgLy8gdG8gJ0FBJyBhbmQgcHJvY2VlZCBvbndhcmRzIHRvIHRyeSBhbmQgbWF0Y2ggb24gJ0InIGFnYWluLlxuICAgICAgICBtYXRjaEluZGV4ID0gbHBzW21hdGNoSW5kZXggLSAxXTtcbiAgICAgIH1cbiAgICB9XG4gICAgLy8gU2F2ZSBtYXRjaCBpbmRleC5cbiAgICB0aGlzLiNtYXRjaEluZGV4ID0gbWF0Y2hJbmRleDtcbiAgICBpZiAoY2h1bmtTdGFydCA9PT0gMCkge1xuICAgICAgYnVmcy5wdXNoKGNodW5rKTtcbiAgICB9IGVsc2UgaWYgKGNodW5rU3RhcnQgPCBsZW5ndGgpIHtcbiAgICAgIC8vIElmIHdlIG1hdGNoZWQgcGFydGlhbGx5IHNvbWV3aGVyZSBpbiB0aGUgbWlkZGxlIG9mIG91ciBjaHVua1xuICAgICAgLy8gdGhlbiB0aGUgcmVtbmFudHMgc2hvdWxkIGJlIHB1c2hlZCBpbnRvIGJ1ZmZlcnMuXG4gICAgICBidWZzLnB1c2goY2h1bmsuc3ViYXJyYXkoY2h1bmtTdGFydCkpO1xuICAgIH1cbiAgfVxuXG4gIC8qKlxuICAgKiBPcHRpbWl6ZWQgaGFuZGxlciBmb3IgYSBjaGFyIGRlbGltaXRlZCBzdHJlYW06XG4gICAqXG4gICAqIEZvciBjaGFyIGRlbGltaXRlZCBzdHJlYW1zIHdlIGRvIG5vdCBuZWVkIHRvIGtlZXAgdHJhY2sgb2ZcbiAgICogdGhlIG1hdGNoIGluZGV4LCByZW1vdmluZyB0aGUgbmVlZCBmb3IgYSBmYWlyIGJpdCBvZiB3b3JrLlxuICAgKi9cbiAgI2hhbmRsZUNoYXIoXG4gICAgY2h1bms6IFVpbnQ4QXJyYXksXG4gICAgY29udHJvbGxlcjogVHJhbnNmb3JtU3RyZWFtRGVmYXVsdENvbnRyb2xsZXI8VWludDhBcnJheT4sXG4gICkge1xuICAgIGNvbnN0IGJ1ZnMgPSB0aGlzLiNidWZzO1xuICAgIGNvbnN0IGxlbmd0aCA9IGNodW5rLmJ5dGVMZW5ndGg7XG4gICAgY29uc3QgZGlzcG9zaXRpb24gPSB0aGlzLiNkaXNwO1xuICAgIGNvbnN0IGRlbGltaXRlciA9IHRoaXMuI2RlbGltaXRlclswXTtcbiAgICBsZXQgY2h1bmtTdGFydCA9IDA7XG4gICAgbGV0IGluc3BlY3RJbmRleCA9IDA7XG4gICAgd2hpbGUgKGluc3BlY3RJbmRleCA8IGxlbmd0aCkge1xuICAgICAgaWYgKGNodW5rW2luc3BlY3RJbmRleF0gPT09IGRlbGltaXRlcikge1xuICAgICAgICAvLyBOZXh0IGJ5dGUgbWF0Y2hlZCBvdXIgbmV4dCBkZWxpbWl0ZXJcbiAgICAgICAgaW5zcGVjdEluZGV4Kys7XG4gICAgICAgIC8qKlxuICAgICAgICAgKiBBbHdheXMgbm9uLW5lZ2F0aXZlXG4gICAgICAgICAqL1xuICAgICAgICBjb25zdCBkZWxpbWl0ZWRDaHVua0VuZCA9IGRpc3Bvc2l0aW9uID09PSBcInN1ZmZpeFwiXG4gICAgICAgICAgPyBpbnNwZWN0SW5kZXhcbiAgICAgICAgICA6IGluc3BlY3RJbmRleCAtIDE7XG4gICAgICAgIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA9PT0gMCAmJiBidWZzLmxlbmd0aCA9PT0gMCkge1xuICAgICAgICAgIC8vIE91ciBjaHVuayBzdGFydGVkIHdpdGggYSBkZWxpbWl0ZXIgYW5kIG5vIHByZXZpb3VzIGNodW5rcyBleGlzdDpcbiAgICAgICAgICAvLyBFbnF1ZXVlIGFuIGVtcHR5IGNodW5rLlxuICAgICAgICAgIGNvbnRyb2xsZXIuZW5xdWV1ZShuZXcgVWludDhBcnJheSgpKTtcbiAgICAgICAgICBjaHVua1N0YXJ0ID0gZGlzcG9zaXRpb24gPT09IFwicHJlZml4XCIgPyAwIDogMTtcbiAgICAgICAgfSBlbHNlIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA+IDAgJiYgYnVmcy5sZW5ndGggPT09IDApIHtcbiAgICAgICAgICAvLyBObyBwcmV2aW91cyBjaHVua3MsIHNsaWNlIGZyb20gY3VycmVudCBjaHVuay5cbiAgICAgICAgICBjb250cm9sbGVyLmVucXVldWUoY2h1bmsuc3ViYXJyYXkoY2h1bmtTdGFydCwgZGVsaW1pdGVkQ2h1bmtFbmQpKTtcbiAgICAgICAgICAvLyBPdXIgY2h1bmsgbWF5IGhhdmUgbW9yZSB0aGFuIG9uZSBkZWxpbWl0ZXI7IHdlIG11c3QgcmVtZW1iZXIgd2hlcmVcbiAgICAgICAgICAvLyB0aGUgbmV4dCBkZWxpbWl0ZWQgY2h1bmsgYmVnaW5zLlxuICAgICAgICAgIGNodW5rU3RhcnQgPSBkaXNwb3NpdGlvbiA9PT0gXCJwcmVmaXhcIlxuICAgICAgICAgICAgPyBpbnNwZWN0SW5kZXggLSAxXG4gICAgICAgICAgICA6IGluc3BlY3RJbmRleDtcbiAgICAgICAgfSBlbHNlIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA9PT0gMCAmJiBidWZzLmxlbmd0aCA+IDApIHtcbiAgICAgICAgICAvLyBPdXIgY2h1bmsgc3RhcnRlZCB3aXRoIGEgZGVsaW1pdGVyLCBwcmV2aW91cyBjaHVua3MgYXJlIHBhc3NlZCBhc1xuICAgICAgICAgIC8vIHRoZXkgYXJlICh3aXRoIGNvbmNhdGVuYXRpb24pLlxuICAgICAgICAgIGlmIChidWZzLmxlbmd0aCA9PT0gMSkge1xuICAgICAgICAgICAgLy8gQ29uY2F0IG5vdCBuZWVkZWQgd2hlbiBhIHNpbmdsZSBidWZmZXIgaXMgcGFzc2VkLlxuICAgICAgICAgICAgY29udHJvbGxlci5lbnF1ZXVlKGJ1ZnNbMF0pO1xuICAgICAgICAgIH0gZWxzZSB7XG4gICAgICAgICAgICBjb250cm9sbGVyLmVucXVldWUoY29uY2F0KGJ1ZnMpKTtcbiAgICAgICAgICB9XG4gICAgICAgICAgLy8gRHJvcCBhbGwgcHJldmlvdXMgY2h1bmtzLlxuICAgICAgICAgIGJ1ZnMubGVuZ3RoID0gMDtcbiAgICAgICAgICBpZiAoZGlzcG9zaXRpb24gIT09IFwicHJlZml4XCIpIHtcbiAgICAgICAgICAgIC8vIHN1ZmZpeCBvciBkaXNjYXJkOiBUaGUgbmV4dCBjaHVuayBzdGFydHMgd2hlcmUgb3VyIGluc3BlY3Rpb24gZmluaXNoZWQuXG4gICAgICAgICAgICAvLyBXZSBzaG91bGQgb25seSBldmVyIGVuZCB1cCBoZXJlIHdpdGggYSBkaXNjYXJkIGRpc3Bvc2l0aW9uIGFzXG4gICAgICAgICAgICAvLyBmb3IgYSBzdWZmaXggZGlzcG9zaXRpb24gdGhpcyBicmFuY2ggd291bGQgbWVhbiB0aGF0IHRoZSBwcmV2aW91c1xuICAgICAgICAgICAgLy8gY2h1bmsgZW5kZWQgd2l0aCBhIGZ1bGwgbWF0Y2ggYnV0IHdhcyBub3QgZW5xdWV1ZWQuXG4gICAgICAgICAgICBjaHVua1N0YXJ0ID0gaW5zcGVjdEluZGV4O1xuICAgICAgICAgIH1cbiAgICAgICAgfSBlbHNlIGlmIChkZWxpbWl0ZWRDaHVua0VuZCA+IDAgJiYgYnVmcy5sZW5ndGggPiAwKSB7XG4gICAgICAgICAgLy8gUHJldmlvdXMgY2h1bmtzIGFuZCBjdXJyZW50IGNodW5rIHRvZ2V0aGVyIGZvcm0gYSBkZWxpbWl0ZWQgY2h1bmsuXG4gICAgICAgICAgY29uc3QgY2h1bmtTbGljZWQgPSBjaHVuay5zdWJhcnJheShjaHVua1N0YXJ0LCBkZWxpbWl0ZWRDaHVua0VuZCk7XG4gICAgICAgICAgY29uc3QgcmVzdWx0ID0gY29uY2F0KFsuLi5idWZzLCBjaHVua1NsaWNlZF0pO1xuICAgICAgICAgIGJ1ZnMubGVuZ3RoID0gMDtcbiAgICAgICAgICBjaHVua1N0YXJ0ID0gZGlzcG9zaXRpb24gPT09IFwicHJlZml4XCJcbiAgICAgICAgICAgID8gZGVsaW1pdGVkQ2h1bmtFbmRcbiAgICAgICAgICAgIDogaW5zcGVjdEluZGV4O1xuICAgICAgICAgIGNvbnRyb2xsZXIuZW5xdWV1ZShyZXN1bHQpO1xuICAgICAgICB9IGVsc2Uge1xuICAgICAgICAgIHRocm93IG5ldyBFcnJvcihcInVucmVhY2hhYmxlXCIpO1xuICAgICAgICB9XG4gICAgICB9IGVsc2Uge1xuICAgICAgICBpbnNwZWN0SW5kZXgrKztcbiAgICAgIH1cbiAgICB9XG4gICAgaWYgKGNodW5rU3RhcnQgPT09IDApIHtcbiAgICAgIGJ1ZnMucHVzaChjaHVuayk7XG4gICAgfSBlbHNlIGlmIChjaHVua1N0YXJ0IDwgbGVuZ3RoKSB7XG4gICAgICAvLyBJZiB3ZSBtYXRjaGVkIHBhcnRpYWxseSBzb21ld2hlcmUgaW4gdGhlIG1pZGRsZSBvZiBvdXIgY2h1bmtcbiAgICAgIC8vIHRoZW4gdGhlIHJlbW5hbnRzIHNob3VsZCBiZSBwdXNoZWQgaW50byBidWZmZXJzLlxuICAgICAgYnVmcy5wdXNoKGNodW5rLnN1YmFycmF5KGNodW5rU3RhcnQpKTtcbiAgICB9XG4gIH1cblxuICAjZmx1c2goY29udHJvbGxlcjogVHJhbnNmb3JtU3RyZWFtRGVmYXVsdENvbnRyb2xsZXI8VWludDhBcnJheT4pIHtcbiAgICBjb25zdCBidWZzID0gdGhpcy4jYnVmcztcbiAgICBjb25zdCBsZW5ndGggPSBidWZzLmxlbmd0aDtcbiAgICBpZiAobGVuZ3RoID09PSAwKSB7XG4gICAgICBjb250cm9sbGVyLmVucXVldWUobmV3IFVpbnQ4QXJyYXkoKSk7XG4gICAgfSBlbHNlIGlmIChsZW5ndGggPT09IDEpIHtcbiAgICAgIGNvbnRyb2xsZXIuZW5xdWV1ZShidWZzWzBdKTtcbiAgICB9IGVsc2Uge1xuICAgICAgY29udHJvbGxlci5lbnF1ZXVlKGNvbmNhdChidWZzKSk7XG4gICAgfVxuICB9XG59XG4iXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6IkFBQUEsMEVBQTBFO0FBQzFFLHFDQUFxQztBQUVyQyxTQUFTLE1BQU0sUUFBUSxxQkFBcUI7QUFDNUMsU0FBUyxTQUFTLFFBQVEsZUFBZTtBQWtCekM7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7Ozs7OztDQTJCQyxHQUNELE9BQU8sTUFBTSx3QkFBd0I7RUFDbkMsQ0FBQyxJQUFJLEdBQWlCLEVBQUUsQ0FBQztFQUN6QixDQUFDLFNBQVMsQ0FBYTtFQUN2QixDQUFDLFVBQVUsR0FBRyxFQUFFO0VBQ2hCLENBQUMsUUFBUSxDQUFvQjtFQUM3QixDQUFDLElBQUksQ0FBdUI7RUFFNUIsK0JBQStCLEdBQy9CLFlBQ0UsU0FBcUIsRUFDckIsT0FBZ0MsQ0FDaEM7SUFDQSxLQUFLLENBQUM7TUFDSixXQUFXLENBQUMsT0FBTyxhQUNqQixVQUFVLE1BQU0sS0FBSyxJQUNqQixJQUFJLENBQUMsQ0FBQyxVQUFVLENBQUMsT0FBTyxjQUN4QixJQUFJLENBQUMsQ0FBQyxNQUFNLENBQUMsT0FBTztNQUMxQixPQUFPLENBQUMsYUFBZSxJQUFJLENBQUMsQ0FBQyxLQUFLLENBQUM7SUFDckM7SUFFQSxJQUFJLENBQUMsQ0FBQyxTQUFTLEdBQUc7SUFDbEIsSUFBSSxDQUFDLENBQUMsUUFBUSxHQUFHLFVBQVUsTUFBTSxHQUFHLElBQUksVUFBVSxhQUFhO0lBQy9ELElBQUksQ0FBQyxDQUFDLElBQUksR0FBRyxTQUFTLGVBQWU7RUFDdkM7RUFFQSxDQUFDLE1BQU0sQ0FDTCxLQUFpQixFQUNqQixVQUF3RDtJQUV4RCxNQUFNLE9BQU8sSUFBSSxDQUFDLENBQUMsSUFBSTtJQUN2QixNQUFNLFNBQVMsTUFBTSxVQUFVO0lBQy9CLE1BQU0sY0FBYyxJQUFJLENBQUMsQ0FBQyxJQUFJO0lBQzlCLE1BQU0sWUFBWSxJQUFJLENBQUMsQ0FBQyxTQUFTO0lBQ2pDLE1BQU0sV0FBVyxVQUFVLE1BQU07SUFDakMsTUFBTSxNQUFNLElBQUksQ0FBQyxDQUFDLFFBQVE7SUFDMUIsSUFBSSxhQUFhO0lBQ2pCLElBQUksYUFBYSxJQUFJLENBQUMsQ0FBQyxVQUFVO0lBQ2pDLElBQUksZUFBZTtJQUNuQixNQUFPLGVBQWUsT0FBUTtNQUM1QixJQUFJLEtBQUssQ0FBQyxhQUFhLEtBQUssU0FBUyxDQUFDLFdBQVcsRUFBRTtRQUNqRCw0Q0FBNEM7UUFDNUM7UUFDQTtRQUNBLElBQUksZUFBZSxVQUFVO1VBQzNCLGFBQWE7VUFDYixhQUFhO1VBQ2IsTUFBTSxzQkFBc0IsZUFBZTtVQUMzQyxNQUFNLG9CQUFvQixnQkFBZ0IsV0FDdEMsZUFDQTtVQUNKLElBQUkscUJBQXFCLEtBQUssS0FBSyxNQUFNLEtBQUssR0FBRztZQUMvQyxtRUFBbUU7WUFDbkUsMEJBQTBCO1lBQzFCLFdBQVcsT0FBTyxDQUFDLElBQUk7WUFDdkIsYUFBYSxnQkFBZ0IsV0FBVyxJQUFJO1VBQzlDLE9BQU8sSUFBSSxvQkFBb0IsS0FBSyxLQUFLLE1BQU0sS0FBSyxHQUFHO1lBQ3JELGdEQUFnRDtZQUNoRCxXQUFXLE9BQU8sQ0FBQyxNQUFNLFFBQVEsQ0FBQyxZQUFZO1lBQzlDLHFFQUFxRTtZQUNyRSxtQ0FBbUM7WUFDbkMsYUFBYSxnQkFBZ0IsV0FDekIsc0JBQ0E7VUFDTixPQUFPLElBQUksc0JBQXNCLEtBQUssS0FBSyxNQUFNLEdBQUcsR0FBRztZQUNyRCxvRUFBb0U7WUFDcEUsaUNBQWlDO1lBQ2pDLElBQUksS0FBSyxNQUFNLEtBQUssR0FBRztjQUNyQixvREFBb0Q7Y0FDcEQsV0FBVyxPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUU7WUFDNUIsT0FBTztjQUNMLFdBQVcsT0FBTyxDQUFDLE9BQU87WUFDNUI7WUFDQSw0QkFBNEI7WUFDNUIsS0FBSyxNQUFNLEdBQUc7WUFDZCxJQUFJLGdCQUFnQixVQUFVO2NBQzVCLDBFQUEwRTtjQUMxRSxnRUFBZ0U7Y0FDaEUsb0VBQW9FO2NBQ3BFLHNEQUFzRDtjQUN0RCxhQUFhO1lBQ2YsT0FBTztjQUNMLGFBQWE7WUFDZjtVQUNGLE9BQU8sSUFBSSxvQkFBb0IsS0FBSyxLQUFLLE1BQU0sR0FBRyxHQUFHO1lBQ25ELDREQUE0RDtZQUM1RCxNQUFNLFlBQVksS0FBSyxNQUFNLEdBQUc7WUFDaEMsTUFBTSxPQUFPLElBQUksQ0FBQyxVQUFVO1lBQzVCLE1BQU0saUJBQWlCLEtBQUssVUFBVSxHQUFHO1lBQ3pDLE1BQU0sYUFBYSxLQUFLLFFBQVEsQ0FBQyxHQUFHO1lBQ3BDLElBQUksY0FBYyxHQUFHO2NBQ25CLFdBQVcsT0FBTyxDQUFDO1lBQ3JCLE9BQU87Y0FDTCxJQUFJLENBQUMsVUFBVSxHQUFHO2NBQ2xCLFdBQVcsT0FBTyxDQUFDLE9BQU87WUFDNUI7WUFDQSxLQUFLLE1BQU0sR0FBRztZQUNkLElBQUksZ0JBQWdCLFVBQVU7Y0FDNUIsc0NBQXNDO2NBQ3RDLEtBQUssSUFBSSxDQUFDLEtBQUssUUFBUSxDQUFDO2NBQ3hCLGFBQWE7WUFDZixPQUFPO2NBQ0wsYUFBYTtZQUNmO1VBQ0YsT0FBTyxJQUFJLG9CQUFvQixLQUFLLEtBQUssTUFBTSxHQUFHLEdBQUc7WUFDbkQscUVBQXFFO1lBQ3JFLE1BQU0sY0FBYyxNQUFNLFFBQVEsQ0FBQyxZQUFZO1lBQy9DLE1BQU0sU0FBUyxPQUFPO2lCQUFJO2NBQU07YUFBWTtZQUM1QyxLQUFLLE1BQU0sR0FBRztZQUNkLFdBQVcsT0FBTyxDQUFDO1lBQ25CLGFBQWEsZ0JBQWdCLFdBQ3pCLG9CQUNBO1VBQ04sT0FBTztZQUNMLE1BQU0sSUFBSSxNQUFNO1VBQ2xCO1FBQ0Y7TUFDRixPQUFPLElBQUksZUFBZSxHQUFHO1FBQzNCLG1EQUFtRDtRQUNuRDtNQUNGLE9BQU87UUFDTCx5REFBeUQ7UUFDekQsaUVBQWlFO1FBQ2pFLGdFQUFnRTtRQUNoRSxvRUFBb0U7UUFDcEUsNkRBQTZEO1FBQzdELGFBQWEsR0FBRyxDQUFDLGFBQWEsRUFBRTtNQUNsQztJQUNGO0lBQ0Esb0JBQW9CO0lBQ3BCLElBQUksQ0FBQyxDQUFDLFVBQVUsR0FBRztJQUNuQixJQUFJLGVBQWUsR0FBRztNQUNwQixLQUFLLElBQUksQ0FBQztJQUNaLE9BQU8sSUFBSSxhQUFhLFFBQVE7TUFDOUIsK0RBQStEO01BQy9ELG1EQUFtRDtNQUNuRCxLQUFLLElBQUksQ0FBQyxNQUFNLFFBQVEsQ0FBQztJQUMzQjtFQUNGO0VBRUE7Ozs7O0dBS0MsR0FDRCxDQUFDLFVBQVUsQ0FDVCxLQUFpQixFQUNqQixVQUF3RDtJQUV4RCxNQUFNLE9BQU8sSUFBSSxDQUFDLENBQUMsSUFBSTtJQUN2QixNQUFNLFNBQVMsTUFBTSxVQUFVO0lBQy9CLE1BQU0sY0FBYyxJQUFJLENBQUMsQ0FBQyxJQUFJO0lBQzlCLE1BQU0sWUFBWSxJQUFJLENBQUMsQ0FBQyxTQUFTLENBQUMsRUFBRTtJQUNwQyxJQUFJLGFBQWE7SUFDakIsSUFBSSxlQUFlO0lBQ25CLE1BQU8sZUFBZSxPQUFRO01BQzVCLElBQUksS0FBSyxDQUFDLGFBQWEsS0FBSyxXQUFXO1FBQ3JDLHVDQUF1QztRQUN2QztRQUNBOztTQUVDLEdBQ0QsTUFBTSxvQkFBb0IsZ0JBQWdCLFdBQ3RDLGVBQ0EsZUFBZTtRQUNuQixJQUFJLHNCQUFzQixLQUFLLEtBQUssTUFBTSxLQUFLLEdBQUc7VUFDaEQsbUVBQW1FO1VBQ25FLDBCQUEwQjtVQUMxQixXQUFXLE9BQU8sQ0FBQyxJQUFJO1VBQ3ZCLGFBQWEsZ0JBQWdCLFdBQVcsSUFBSTtRQUM5QyxPQUFPLElBQUksb0JBQW9CLEtBQUssS0FBSyxNQUFNLEtBQUssR0FBRztVQUNyRCxnREFBZ0Q7VUFDaEQsV0FBVyxPQUFPLENBQUMsTUFBTSxRQUFRLENBQUMsWUFBWTtVQUM5QyxxRUFBcUU7VUFDckUsbUNBQW1DO1VBQ25DLGFBQWEsZ0JBQWdCLFdBQ3pCLGVBQWUsSUFDZjtRQUNOLE9BQU8sSUFBSSxzQkFBc0IsS0FBSyxLQUFLLE1BQU0sR0FBRyxHQUFHO1VBQ3JELG9FQUFvRTtVQUNwRSxpQ0FBaUM7VUFDakMsSUFBSSxLQUFLLE1BQU0sS0FBSyxHQUFHO1lBQ3JCLG9EQUFvRDtZQUNwRCxXQUFXLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtVQUM1QixPQUFPO1lBQ0wsV0FBVyxPQUFPLENBQUMsT0FBTztVQUM1QjtVQUNBLDRCQUE0QjtVQUM1QixLQUFLLE1BQU0sR0FBRztVQUNkLElBQUksZ0JBQWdCLFVBQVU7WUFDNUIsMEVBQTBFO1lBQzFFLGdFQUFnRTtZQUNoRSxvRUFBb0U7WUFDcEUsc0RBQXNEO1lBQ3RELGFBQWE7VUFDZjtRQUNGLE9BQU8sSUFBSSxvQkFBb0IsS0FBSyxLQUFLLE1BQU0sR0FBRyxHQUFHO1VBQ25ELHFFQUFxRTtVQUNyRSxNQUFNLGNBQWMsTUFBTSxRQUFRLENBQUMsWUFBWTtVQUMvQyxNQUFNLFNBQVMsT0FBTztlQUFJO1lBQU07V0FBWTtVQUM1QyxLQUFLLE1BQU0sR0FBRztVQUNkLGFBQWEsZ0JBQWdCLFdBQ3pCLG9CQUNBO1VBQ0osV0FBVyxPQUFPLENBQUM7UUFDckIsT0FBTztVQUNMLE1BQU0sSUFBSSxNQUFNO1FBQ2xCO01BQ0YsT0FBTztRQUNMO01BQ0Y7SUFDRjtJQUNBLElBQUksZUFBZSxHQUFHO01BQ3BCLEtBQUssSUFBSSxDQUFDO0lBQ1osT0FBTyxJQUFJLGFBQWEsUUFBUTtNQUM5QiwrREFBK0Q7TUFDL0QsbURBQW1EO01BQ25ELEtBQUssSUFBSSxDQUFDLE1BQU0sUUFBUSxDQUFDO0lBQzNCO0VBQ0Y7RUFFQSxDQUFDLEtBQUssQ0FBQyxVQUF3RDtJQUM3RCxNQUFNLE9BQU8sSUFBSSxDQUFDLENBQUMsSUFBSTtJQUN2QixNQUFNLFNBQVMsS0FBSyxNQUFNO0lBQzFCLElBQUksV0FBVyxHQUFHO01BQ2hCLFdBQVcsT0FBTyxDQUFDLElBQUk7SUFDekIsT0FBTyxJQUFJLFdBQVcsR0FBRztNQUN2QixXQUFXLE9BQU8sQ0FBQyxJQUFJLENBQUMsRUFBRTtJQUM1QixPQUFPO01BQ0wsV0FBVyxPQUFPLENBQUMsT0FBTztJQUM1QjtFQUNGO0FBQ0YifQ==