// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Array index or record key corresponding to a value for a data object. */ const QUOTE = '"';
const LF = "\n";
const CRLF = "\r\n";
const BYTE_ORDER_MARK = "\ufeff";
function getEscapedString(value, sep) {
  if (value === undefined || value === null) return "";
  let str = "";
  if (typeof value === "object") str = JSON.stringify(value);
  else str = String(value);
  // Is regex.test more performant here? If so, how to dynamically create?
  // https://stackoverflow.com/questions/3561493/
  if (str.includes(sep) || str.includes(LF) || str.includes(QUOTE)) {
    return `${QUOTE}${str.replaceAll(QUOTE, `${QUOTE}${QUOTE}`)}${QUOTE}`;
  }
  return str;
}
function normalizeColumn(column) {
  let header, prop;
  if (typeof column === "object") {
    if (Array.isArray(column)) {
      header = String(column[column.length - 1]);
      prop = column;
    } else {
      prop = Array.isArray(column.prop) ? column.prop : [
        column.prop
      ];
      header = typeof column.header === "string" ? column.header : String(prop[prop.length - 1]);
    }
  } else {
    header = String(column);
    prop = [
      column
    ];
  }
  return {
    header,
    prop
  };
}
/** Error thrown in {@linkcode stringify}. */ export class StringifyError extends Error {
  /** Construct a new instance. */ constructor(message){
    super(message);
    this.name = "StringifyError";
  }
}
/**
 * Returns an array of values from an object using the property accessors
 * (and optional transform function) in each column
 */ function getValuesFromItem(item, normalizedColumns) {
  const values = [];
  if (normalizedColumns.length) {
    for (const column of normalizedColumns){
      let value = item;
      for (const prop of column.prop){
        if (typeof value !== "object" || value === null) continue;
        if (Array.isArray(value)) {
          if (typeof prop === "number") value = value[prop];
          else {
            throw new StringifyError('Property accessor is not of type "number"');
          }
        } else value = value[prop];
      }
      values.push(value);
    }
  } else {
    if (Array.isArray(item)) {
      values.push(...item);
    } else if (typeof item === "object") {
      throw new StringifyError("No property accessor function was provided for object");
    } else {
      values.push(item);
    }
  }
  return values;
}
/**
 * Write data using CSV encoding.
 *
 * @param data The source data to stringify. It's an array of items which are
 * plain objects or arrays.
 *
 * `DataItem: Record<string, unknown> | unknown[]`
 *
 * ```ts
 * const data = [
 *   {
 *     name: "Deno",
 *     repo: { org: "denoland", name: "deno" },
 *     runsOn: ["Rust", "TypeScript"],
 *   },
 * ];
 * ```
 *
 * @example
 * ```ts
 * import {
 *   Column,
 *   stringify,
 * } from "https://deno.land/std@$STD_VERSION/csv/stringify.ts";
 *
 * type Character = {
 *   age: number;
 *   name: {
 *     first: string;
 *     last: string;
 *   };
 * };
 *
 * const data: Character[] = [
 *   {
 *     age: 70,
 *     name: {
 *       first: "Rick",
 *       last: "Sanchez",
 *     },
 *   },
 *   {
 *     age: 14,
 *     name: {
 *       first: "Morty",
 *       last: "Smith",
 *     },
 *   },
 * ];
 *
 * let columns: Column[] = [
 *   ["name", "first"],
 *   "age",
 * ];
 *
 * console.log(stringify(data, { columns }));
 * // first,age
 * // Rick,70
 * // Morty,14
 * ```
 */ export function stringify(data, { headers = true, separator: sep = ",", columns = [], bom = false } = {}) {
  if (sep.includes(QUOTE) || sep.includes(CRLF)) {
    const message = [
      "Separator cannot include the following strings:",
      '  - U+0022: Quotation mark (")',
      "  - U+000D U+000A: Carriage Return + Line Feed (\\r\\n)"
    ].join("\n");
    throw new StringifyError(message);
  }
  const normalizedColumns = columns.map(normalizeColumn);
  let output = "";
  if (bom) {
    output += BYTE_ORDER_MARK;
  }
  if (headers) {
    output += normalizedColumns.map((column)=>getEscapedString(column.header, sep)).join(sep);
    output += CRLF;
  }
  for (const item of data){
    const values = getValuesFromItem(item, normalizedColumns);
    output += values.map((value)=>getEscapedString(value, sep)).join(sep);
    output += CRLF;
  }
  return output;
}
//# sourceMappingURL=data:application/json;base64,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