// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * A {@linkcode TransformStream} that will only read & enqueue `size` amount of
 * chunks.
 *
 * If `options.error` is set, then instead of terminating the stream,
 * an error will be thrown.
 *
 * @example
 * ```ts
 * import { LimitedTransformStream } from "https://deno.land/std@$STD_VERSION/streams/limited_transform_stream.ts";
 * const res = await fetch("https://example.com");
 * const parts = res.body!.pipeThrough(new LimitedTransformStream(50));
 * ```
 */ export class LimitedTransformStream extends TransformStream {
  #read = 0;
  /** Constructs a new instance. */ constructor(size, options = {}){
    super({
      transform: (chunk, controller)=>{
        if (this.#read + 1 > size) {
          if (options.error) {
            throw new RangeError(`Exceeded chunk limit of '${size}'`);
          } else {
            controller.terminate();
          }
        } else {
          this.#read++;
          controller.enqueue(chunk);
        }
      }
    });
  }
}
//# sourceMappingURL=data:application/json;base64,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