// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// Structured similarly to Go's cookie.go
// https://github.com/golang/go/blob/master/src/net/http/cookie.go
// This module is browser compatible.
import { assert } from "../assert/assert.ts";
const FIELD_CONTENT_REGEXP = /^(?=[\x20-\x7E]*$)[^()@<>,;:\\"\[\]?={}\s]+$/;
function toString(cookie) {
  if (!cookie.name) {
    return "";
  }
  const out = [];
  validateName(cookie.name);
  validateValue(cookie.name, cookie.value);
  out.push(`${cookie.name}=${cookie.value}`);
  // Fallback for invalid Set-Cookie
  // ref: https://tools.ietf.org/html/draft-ietf-httpbis-cookie-prefixes-00#section-3.1
  if (cookie.name.startsWith("__Secure")) {
    cookie.secure = true;
  }
  if (cookie.name.startsWith("__Host")) {
    cookie.path = "/";
    cookie.secure = true;
    delete cookie.domain;
  }
  if (cookie.secure) {
    out.push("Secure");
  }
  if (cookie.httpOnly) {
    out.push("HttpOnly");
  }
  if (typeof cookie.maxAge === "number" && Number.isInteger(cookie.maxAge)) {
    assert(cookie.maxAge >= 0, "Max-Age must be an integer superior or equal to 0");
    out.push(`Max-Age=${cookie.maxAge}`);
  }
  if (cookie.domain) {
    validateDomain(cookie.domain);
    out.push(`Domain=${cookie.domain}`);
  }
  if (cookie.sameSite) {
    out.push(`SameSite=${cookie.sameSite}`);
  }
  if (cookie.path) {
    validatePath(cookie.path);
    out.push(`Path=${cookie.path}`);
  }
  if (cookie.expires) {
    const { expires } = cookie;
    const date = typeof expires === "number" ? new Date(expires) : expires;
    out.push(`Expires=${date.toUTCString()}`);
  }
  if (cookie.unparsed) {
    out.push(cookie.unparsed.join("; "));
  }
  return out.join("; ");
}
/**
 * Validate Cookie Name.
 * @param name Cookie name.
 */ function validateName(name) {
  if (name && !FIELD_CONTENT_REGEXP.test(name)) {
    throw new TypeError(`Invalid cookie name: "${name}".`);
  }
}
/**
 * Validate Path Value.
 * See {@link https://tools.ietf.org/html/rfc6265#section-4.1.2.4}.
 * @param path Path value.
 */ function validatePath(path) {
  if (path === null) {
    return;
  }
  for(let i = 0; i < path.length; i++){
    const c = path.charAt(i);
    if (c < String.fromCharCode(0x20) || c > String.fromCharCode(0x7E) || c === ";") {
      throw new Error(path + ": Invalid cookie path char '" + c + "'");
    }
  }
}
/**
 * Validate Cookie Value.
 * See {@link https://tools.ietf.org/html/rfc6265#section-4.1}.
 * @param value Cookie value.
 */ function validateValue(name, value) {
  if (value === null) return;
  for(let i = 0; i < value.length; i++){
    const c = value.charAt(i);
    if (c < String.fromCharCode(0x21) || c === String.fromCharCode(0x22) || c === String.fromCharCode(0x2c) || c === String.fromCharCode(0x3b) || c === String.fromCharCode(0x5c) || c === String.fromCharCode(0x7f)) {
      throw new Error("RFC2616 cookie '" + name + "' cannot contain character '" + c + "'");
    }
    if (c > String.fromCharCode(0x80)) {
      throw new Error("RFC2616 cookie '" + name + "' can only have US-ASCII chars as value" + c.charCodeAt(0).toString(16));
    }
  }
}
/**
 * Validate Cookie Domain.
 * See {@link https://datatracker.ietf.org/doc/html/rfc6265#section-4.1.2.3}.
 * @param domain Cookie domain.
 */ function validateDomain(domain) {
  const char1 = domain.charAt(0);
  const charN = domain.charAt(domain.length - 1);
  if (char1 === "-" || charN === "." || charN === "-") {
    throw new Error("Invalid first/last char in cookie domain: " + domain);
  }
}
/**
 * Parse cookies of a header
 *
 * @example
 * ```ts
 * import { getCookies } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const headers = new Headers();
 * headers.set("Cookie", "full=of; tasty=chocolate");
 *
 * const cookies = getCookies(headers);
 * console.log(cookies); // { full: "of", tasty: "chocolate" }
 * ```
 *
 * @param headers The headers instance to get cookies from
 * @return Object with cookie names as keys
 */ export function getCookies(headers) {
  const cookie = headers.get("Cookie");
  if (cookie !== null) {
    const out = {};
    const c = cookie.split(";");
    for (const kv of c){
      const [cookieKey, ...cookieVal] = kv.split("=");
      assert(cookieKey !== undefined);
      const key = cookieKey.trim();
      out[key] = cookieVal.join("=");
    }
    return out;
  }
  return {};
}
/**
 * Set the cookie header properly in the headers
 *
 * @example
 * ```ts
 * import {
 *   Cookie,
 *   setCookie,
 * } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const headers = new Headers();
 * const cookie: Cookie = { name: "Space", value: "Cat" };
 * setCookie(headers, cookie);
 *
 * const cookieHeader = headers.get("set-cookie");
 * console.log(cookieHeader); // Space=Cat
 * ```
 *
 * @param headers The headers instance to set the cookie to
 * @param cookie Cookie to set
 */ export function setCookie(headers, cookie) {
  // Parsing cookie headers to make consistent set-cookie header
  // ref: https://tools.ietf.org/html/rfc6265#section-4.1.1
  const v = toString(cookie);
  if (v) {
    headers.append("Set-Cookie", v);
  }
}
/**
 * Set the cookie header with empty value in the headers to delete it
 *
 * > Note: Deleting a `Cookie` will set its expiration date before now. Forcing
 * > the browser to delete it.
 *
 * @example
 * ```ts
 * import { deleteCookie } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const headers = new Headers();
 * deleteCookie(headers, "deno");
 *
 * const cookieHeader = headers.get("set-cookie");
 * console.log(cookieHeader); // deno=; Expires=Thus, 01 Jan 1970 00:00:00 GMT
 * ```
 *
 * @param headers The headers instance to delete the cookie from
 * @param name Name of cookie
 * @param attributes Additional cookie attributes
 */ export function deleteCookie(headers, name, attributes) {
  setCookie(headers, {
    name: name,
    value: "",
    expires: new Date(0),
    ...attributes
  });
}
function parseSetCookie(value) {
  const attrs = value.split(";").map((attr)=>{
    const [key, ...values] = attr.trim().split("=");
    return [
      key,
      values.join("=")
    ];
  });
  if (!attrs[0]) {
    return null;
  }
  const cookie = {
    name: attrs[0][0],
    value: attrs[0][1]
  };
  for (const [key, value] of attrs.slice(1)){
    switch(key.toLocaleLowerCase()){
      case "expires":
        cookie.expires = new Date(value);
        break;
      case "max-age":
        cookie.maxAge = Number(value);
        if (cookie.maxAge < 0) {
          console.warn("Max-Age must be an integer superior or equal to 0. Cookie ignored.");
          return null;
        }
        break;
      case "domain":
        cookie.domain = value;
        break;
      case "path":
        cookie.path = value;
        break;
      case "secure":
        cookie.secure = true;
        break;
      case "httponly":
        cookie.httpOnly = true;
        break;
      case "samesite":
        cookie.sameSite = value;
        break;
      default:
        if (!Array.isArray(cookie.unparsed)) {
          cookie.unparsed = [];
        }
        cookie.unparsed.push([
          key,
          value
        ].join("="));
    }
  }
  if (cookie.name.startsWith("__Secure-")) {
    /** This requirement is mentioned in https://developer.mozilla.org/en-US/docs/Web/HTTP/Headers/Set-Cookie but not the RFC. */ if (!cookie.secure) {
      console.warn("Cookies with names starting with `__Secure-` must be set with the secure flag. Cookie ignored.");
      return null;
    }
  }
  if (cookie.name.startsWith("__Host-")) {
    if (!cookie.secure) {
      console.warn("Cookies with names starting with `__Host-` must be set with the secure flag. Cookie ignored.");
      return null;
    }
    if (cookie.domain !== undefined) {
      console.warn("Cookies with names starting with `__Host-` must not have a domain specified. Cookie ignored.");
      return null;
    }
    if (cookie.path !== "/") {
      console.warn("Cookies with names starting with `__Host-` must have path be `/`. Cookie has been ignored.");
      return null;
    }
  }
  return cookie;
}
/**
 * Parse set-cookies of a header
 *
 * @example
 * ```ts
 * import { getSetCookies } from "https://deno.land/std@$STD_VERSION/http/cookie.ts";
 *
 * const headers = new Headers([
 *   ["Set-Cookie", "lulu=meow; Secure; Max-Age=3600"],
 *   ["Set-Cookie", "booya=kasha; HttpOnly; Path=/"],
 * ]);
 *
 * const cookies = getSetCookies(headers);
 * console.log(cookies); // [{ name: "lulu", value: "meow", secure: true, maxAge: 3600 }, { name: "booya", value: "kahsa", httpOnly: true, path: "/ }]
 * ```
 *
 * @param headers The headers instance to get set-cookies from
 * @return List of cookies
 */ export function getSetCookies(headers) {
  return headers.getSetCookie()/** Parse each `set-cookie` header separately */ .map(parseSetCookie)/** Skip empty cookies */ .filter(Boolean);
}
//# sourceMappingURL=data:application/json;base64,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