// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/** Supporting functions for media_types that do not make part of the public
 * API.
 *
 * @module
 * @private
 */ export function consumeToken(v) {
  const notPos = indexOf(v, isNotTokenChar);
  if (notPos === -1) {
    return [
      v,
      ""
    ];
  }
  if (notPos === 0) {
    return [
      "",
      v
    ];
  }
  return [
    v.slice(0, notPos),
    v.slice(notPos)
  ];
}
export function consumeValue(v) {
  if (!v) {
    return [
      "",
      v
    ];
  }
  if (v[0] !== `"`) {
    return consumeToken(v);
  }
  let value = "";
  for(let i = 1; i < v.length; i++){
    const r = v[i];
    if (r === `"`) {
      return [
        value,
        v.slice(i + 1)
      ];
    }
    const next = v[i + 1];
    if (r === "\\" && typeof next === "string" && isTSpecial(next)) {
      value += next;
      i++;
      continue;
    }
    if (r === "\r" || r === "\n") {
      return [
        "",
        v
      ];
    }
    value += v[i];
  }
  return [
    "",
    v
  ];
}
export function consumeMediaParam(v) {
  let rest = v.trimStart();
  if (!rest.startsWith(";")) {
    return [
      "",
      "",
      v
    ];
  }
  rest = rest.slice(1);
  rest = rest.trimStart();
  let param;
  [param, rest] = consumeToken(rest);
  param = param.toLowerCase();
  if (!param) {
    return [
      "",
      "",
      v
    ];
  }
  rest = rest.slice(1);
  rest = rest.trimStart();
  const [value, rest2] = consumeValue(rest);
  if (value === "" && rest2 === rest) {
    return [
      "",
      "",
      v
    ];
  }
  rest = rest2;
  return [
    param,
    value,
    rest
  ];
}
export function decode2331Encoding(v) {
  const sv = v.split(`'`, 3);
  if (sv.length !== 3) {
    return undefined;
  }
  const [sv0, , sv2] = sv;
  const charset = sv0.toLowerCase();
  if (!charset) {
    return undefined;
  }
  if (charset !== "us-ascii" && charset !== "utf-8") {
    return undefined;
  }
  const encv = decodeURI(sv2);
  if (!encv) {
    return undefined;
  }
  return encv;
}
function indexOf(s, fn) {
  let i = -1;
  for (const v of s){
    i++;
    if (fn(v)) {
      return i;
    }
  }
  return -1;
}
export function isIterator(obj) {
  if (obj === null || obj === undefined) {
    return false;
  }
  // deno-lint-ignore no-explicit-any
  return typeof obj[Symbol.iterator] === "function";
}
export function isToken(s) {
  if (!s) {
    return false;
  }
  return indexOf(s, isNotTokenChar) < 0;
}
function isNotTokenChar(r) {
  return !isTokenChar(r);
}
function isTokenChar(r) {
  const code = r.charCodeAt(0);
  return code > 0x20 && code < 0x7f && !isTSpecial(r);
}
function isTSpecial(r) {
  return r[0] ? `()<>@,;:\\"/[]?=`.includes(r[0]) : false;
}
const CHAR_CODE_SPACE = " ".charCodeAt(0);
const CHAR_CODE_TILDE = "~".charCodeAt(0);
export function needsEncoding(s) {
  for (const b of s){
    const charCode = b.charCodeAt(0);
    if ((charCode < CHAR_CODE_SPACE || charCode > CHAR_CODE_TILDE) && b !== "\t") {
      return true;
    }
  }
  return false;
}
//# sourceMappingURL=data:application/json;base64,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