// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
// Bare keys may only contain ASCII letters,
// ASCII digits, underscores, and dashes (A-Za-z0-9_-).
function joinKeys(keys) {
  // Dotted keys are a sequence of bare or quoted keys joined with a dot.
  // This allows for grouping similar properties together:
  return keys.map((str)=>{
    return str.length === 0 || str.match(/[^A-Za-z0-9_-]/) ? JSON.stringify(str) : str;
  }).join(".");
}
class Dumper {
  maxPad = 0;
  srcObject;
  output = [];
  #arrayTypeCache = new Map();
  constructor(srcObjc){
    this.srcObject = srcObjc;
  }
  dump(fmtOptions = {}) {
    // deno-lint-ignore no-explicit-any
    this.output = this.#printObject(this.srcObject);
    this.output = this.#format(fmtOptions);
    return this.output;
  }
  #printObject(obj, keys = []) {
    const out = [];
    const props = Object.keys(obj);
    const inlineProps = [];
    const multilineProps = [];
    for (const prop of props){
      if (this.#isSimplySerializable(obj[prop])) {
        inlineProps.push(prop);
      } else {
        multilineProps.push(prop);
      }
    }
    const sortedProps = inlineProps.concat(multilineProps);
    for (const prop of sortedProps){
      const value = obj[prop];
      if (value instanceof Date) {
        out.push(this.#dateDeclaration([
          prop
        ], value));
      } else if (typeof value === "string" || value instanceof RegExp) {
        out.push(this.#strDeclaration([
          prop
        ], value.toString()));
      } else if (typeof value === "number") {
        out.push(this.#numberDeclaration([
          prop
        ], value));
      } else if (typeof value === "boolean") {
        out.push(this.#boolDeclaration([
          prop
        ], value));
      } else if (value instanceof Array) {
        const arrayType = this.#getTypeOfArray(value);
        if (arrayType === "ONLY_PRIMITIVE") {
          out.push(this.#arrayDeclaration([
            prop
          ], value));
        } else if (arrayType === "ONLY_OBJECT_EXCLUDING_ARRAY") {
          // array of objects
          for(let i = 0; i < value.length; i++){
            out.push("");
            out.push(this.#headerGroup([
              ...keys,
              prop
            ]));
            out.push(...this.#printObject(value[i], [
              ...keys,
              prop
            ]));
          }
        } else {
          // this is a complex array, use the inline format.
          const str = value.map((x)=>this.#printAsInlineValue(x)).join(",");
          out.push(`${this.#declaration([
            prop
          ])}[${str}]`);
        }
      } else if (typeof value === "object") {
        out.push("");
        out.push(this.#header([
          ...keys,
          prop
        ]));
        if (value) {
          const toParse = value;
          out.push(...this.#printObject(toParse, [
            ...keys,
            prop
          ]));
        }
      // out.push(...this._parse(value, `${path}${prop}.`));
      }
    }
    out.push("");
    return out;
  }
  #isPrimitive(value) {
    return value instanceof Date || value instanceof RegExp || [
      "string",
      "number",
      "boolean"
    ].includes(typeof value);
  }
  #getTypeOfArray(arr) {
    if (this.#arrayTypeCache.has(arr)) {
      return this.#arrayTypeCache.get(arr);
    }
    const type = this.#doGetTypeOfArray(arr);
    this.#arrayTypeCache.set(arr, type);
    return type;
  }
  #doGetTypeOfArray(arr) {
    if (!arr.length) {
      // any type should be fine
      return "ONLY_PRIMITIVE";
    }
    const onlyPrimitive = this.#isPrimitive(arr[0]);
    if (arr[0] instanceof Array) {
      return "MIXED";
    }
    for(let i = 1; i < arr.length; i++){
      if (onlyPrimitive !== this.#isPrimitive(arr[i]) || arr[i] instanceof Array) {
        return "MIXED";
      }
    }
    return onlyPrimitive ? "ONLY_PRIMITIVE" : "ONLY_OBJECT_EXCLUDING_ARRAY";
  }
  #printAsInlineValue(value) {
    if (value instanceof Date) {
      return `"${this.#printDate(value)}"`;
    } else if (typeof value === "string" || value instanceof RegExp) {
      return JSON.stringify(value.toString());
    } else if (typeof value === "number") {
      return value;
    } else if (typeof value === "boolean") {
      return value.toString();
    } else if (value instanceof Array) {
      const str = value.map((x)=>this.#printAsInlineValue(x)).join(",");
      return `[${str}]`;
    } else if (typeof value === "object") {
      if (!value) {
        throw new Error("should never reach");
      }
      const str = Object.keys(value).map((key)=>{
        return `${joinKeys([
          key
        ])} = ${// deno-lint-ignore no-explicit-any
        this.#printAsInlineValue(value[key])}`;
      }).join(",");
      return `{${str}}`;
    }
    throw new Error("should never reach");
  }
  #isSimplySerializable(value) {
    return typeof value === "string" || typeof value === "number" || typeof value === "boolean" || value instanceof RegExp || value instanceof Date || value instanceof Array && this.#getTypeOfArray(value) !== "ONLY_OBJECT_EXCLUDING_ARRAY";
  }
  #header(keys) {
    return `[${joinKeys(keys)}]`;
  }
  #headerGroup(keys) {
    return `[[${joinKeys(keys)}]]`;
  }
  #declaration(keys) {
    const title = joinKeys(keys);
    if (title.length > this.maxPad) {
      this.maxPad = title.length;
    }
    return `${title} = `;
  }
  #arrayDeclaration(keys, value) {
    return `${this.#declaration(keys)}${JSON.stringify(value)}`;
  }
  #strDeclaration(keys, value) {
    return `${this.#declaration(keys)}${JSON.stringify(value)}`;
  }
  #numberDeclaration(keys, value) {
    switch(value){
      case Infinity:
        return `${this.#declaration(keys)}inf`;
      case -Infinity:
        return `${this.#declaration(keys)}-inf`;
      default:
        return `${this.#declaration(keys)}${value}`;
    }
  }
  #boolDeclaration(keys, value) {
    return `${this.#declaration(keys)}${value}`;
  }
  #printDate(value) {
    function dtPad(v, lPad = 2) {
      return v.padStart(lPad, "0");
    }
    const m = dtPad((value.getUTCMonth() + 1).toString());
    const d = dtPad(value.getUTCDate().toString());
    const h = dtPad(value.getUTCHours().toString());
    const min = dtPad(value.getUTCMinutes().toString());
    const s = dtPad(value.getUTCSeconds().toString());
    const ms = dtPad(value.getUTCMilliseconds().toString(), 3);
    // formatted date
    const fData = `${value.getUTCFullYear()}-${m}-${d}T${h}:${min}:${s}.${ms}`;
    return fData;
  }
  #dateDeclaration(keys, value) {
    return `${this.#declaration(keys)}${this.#printDate(value)}`;
  }
  #format(options = {}) {
    const { keyAlignment = false } = options;
    const rDeclaration = /^(\".*\"|[^=]*)\s=/;
    const out = [];
    for(let i = 0; i < this.output.length; i++){
      const l = this.output[i];
      // we keep empty entry for array of objects
      if (l[0] === "[" && l[1] !== "[") {
        // non-empty object with only subobjects as properties
        if (this.output[i + 1] === "" && this.output[i + 2]?.slice(0, l.length) === l.slice(0, -1) + ".") {
          i += 1;
          continue;
        }
        out.push(l);
      } else {
        if (keyAlignment) {
          const m = rDeclaration.exec(l);
          if (m && m[1]) {
            out.push(l.replace(m[1], m[1].padEnd(this.maxPad)));
          } else {
            out.push(l);
          }
        } else {
          out.push(l);
        }
      }
    }
    // Cleaning multiple spaces
    const cleanedOutput = [];
    for(let i = 0; i < out.length; i++){
      const l = out[i];
      if (!(l === "" && out[i + 1] === "")) {
        cleanedOutput.push(l);
      }
    }
    return cleanedOutput;
  }
}
/**
 * Stringify dumps source object into TOML string and returns it.
 * @param srcObj
 * @param [fmtOptions] format options
 * @param [fmtOptions.keyAlignment] whether to align keys
 */ export function stringify(srcObj, fmtOptions) {
  return new Dumper(srcObj).dump(fmtOptions).join("\n");
}
//# sourceMappingURL=data:application/json;base64,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