// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
import { convertRowToObject, ERR_BARE_QUOTE, ERR_FIELD_COUNT, ERR_INVALID_DELIM, ERR_QUOTE, ParseError } from "./_io.ts";
import { assert } from "../assert/assert.ts";
export { ParseError };
const BYTE_ORDER_MARK = "\ufeff";
class Parser {
  #input = "";
  #cursor = 0;
  #options;
  constructor({ separator = ",", trimLeadingSpace = false, comment, lazyQuotes, fieldsPerRecord } = {}){
    this.#options = {
      separator,
      trimLeadingSpace,
      comment,
      lazyQuotes,
      fieldsPerRecord
    };
  }
  #readLine() {
    if (this.#isEOF()) return null;
    if (!this.#input.startsWith("\r\n", this.#cursor) || !this.#input.startsWith("\n", this.#cursor)) {
      let buffer = "";
      let hadNewline = false;
      while(this.#cursor < this.#input.length){
        if (this.#input.startsWith("\r\n", this.#cursor)) {
          hadNewline = true;
          this.#cursor += 2;
          break;
        }
        if (this.#input.startsWith("\n", this.#cursor)) {
          hadNewline = true;
          this.#cursor += 1;
          break;
        }
        buffer += this.#input[this.#cursor];
        this.#cursor += 1;
      }
      if (!hadNewline && buffer.endsWith("\r")) {
        buffer = buffer.slice(0, -1);
      }
      return buffer;
    }
    return null;
  }
  #isEOF() {
    return this.#cursor >= this.#input.length;
  }
  #parseRecord(startLine) {
    let line = this.#readLine();
    if (line === null) return null;
    if (line.length === 0) {
      return [];
    }
    function runeCount(s) {
      // Array.from considers the surrogate pair.
      return Array.from(s).length;
    }
    let lineIndex = startLine + 1;
    // line starting with comment character is ignored
    if (this.#options.comment && line[0] === this.#options.comment) {
      return [];
    }
    let fullLine = line;
    let quoteError = null;
    const quote = '"';
    const quoteLen = quote.length;
    const separatorLen = this.#options.separator.length;
    let recordBuffer = "";
    const fieldIndexes = [];
    parseField: for(;;){
      if (this.#options.trimLeadingSpace) {
        line = line.trimStart();
      }
      if (line.length === 0 || !line.startsWith(quote)) {
        // Non-quoted string field
        const i = line.indexOf(this.#options.separator);
        let field = line;
        if (i >= 0) {
          field = field.substring(0, i);
        }
        // Check to make sure a quote does not appear in field.
        if (!this.#options.lazyQuotes) {
          const j = field.indexOf(quote);
          if (j >= 0) {
            const col = runeCount(fullLine.slice(0, fullLine.length - line.slice(j).length));
            quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_BARE_QUOTE);
            break parseField;
          }
        }
        recordBuffer += field;
        fieldIndexes.push(recordBuffer.length);
        if (i >= 0) {
          line = line.substring(i + separatorLen);
          continue parseField;
        }
        break parseField;
      } else {
        // Quoted string field
        line = line.substring(quoteLen);
        for(;;){
          const i = line.indexOf(quote);
          if (i >= 0) {
            // Hit next quote.
            recordBuffer += line.substring(0, i);
            line = line.substring(i + quoteLen);
            if (line.startsWith(quote)) {
              // `""` sequence (append quote).
              recordBuffer += quote;
              line = line.substring(quoteLen);
            } else if (line.startsWith(this.#options.separator)) {
              // `","` sequence (end of field).
              line = line.substring(separatorLen);
              fieldIndexes.push(recordBuffer.length);
              continue parseField;
            } else if (0 === line.length) {
              // `"\n` sequence (end of line).
              fieldIndexes.push(recordBuffer.length);
              break parseField;
            } else if (this.#options.lazyQuotes) {
              // `"` sequence (bare quote).
              recordBuffer += quote;
            } else {
              // `"*` sequence (invalid non-escaped quote).
              const col = runeCount(fullLine.slice(0, fullLine.length - line.length - quoteLen));
              quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
              break parseField;
            }
          } else if (line.length > 0 || !this.#isEOF()) {
            // Hit end of line (copy all data so far).
            recordBuffer += line;
            const r = this.#readLine();
            lineIndex++;
            line = r ?? ""; // This is a workaround for making this module behave similarly to the encoding/csv/reader.go.
            fullLine = line;
            if (r === null) {
              // Abrupt end of file (EOF or error).
              if (!this.#options.lazyQuotes) {
                const col = runeCount(fullLine);
                quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
                break parseField;
              }
              fieldIndexes.push(recordBuffer.length);
              break parseField;
            }
            recordBuffer += "\n"; // preserve line feed (This is because TextProtoReader removes it.)
          } else {
            // Abrupt end of file (EOF on error).
            if (!this.#options.lazyQuotes) {
              const col = runeCount(fullLine);
              quoteError = new ParseError(startLine + 1, lineIndex, col, ERR_QUOTE);
              break parseField;
            }
            fieldIndexes.push(recordBuffer.length);
            break parseField;
          }
        }
      }
    }
    if (quoteError) {
      throw quoteError;
    }
    const result = [];
    let preIdx = 0;
    for (const i of fieldIndexes){
      result.push(recordBuffer.slice(preIdx, i));
      preIdx = i;
    }
    return result;
  }
  parse(input) {
    this.#input = input.startsWith(BYTE_ORDER_MARK) ? input.slice(1) : input;
    this.#cursor = 0;
    const result = [];
    let _nbFields;
    let lineResult;
    let first = true;
    let lineIndex = 0;
    const INVALID_RUNE = [
      "\r",
      "\n",
      '"'
    ];
    const options = this.#options;
    if (INVALID_RUNE.includes(options.separator) || typeof options.comment === "string" && INVALID_RUNE.includes(options.comment) || options.separator === options.comment) {
      throw new Error(ERR_INVALID_DELIM);
    }
    for(;;){
      const r = this.#parseRecord(lineIndex);
      if (r === null) break;
      lineResult = r;
      lineIndex++;
      // If fieldsPerRecord is 0, Read sets it to
      // the number of fields in the first record
      if (first) {
        first = false;
        if (options.fieldsPerRecord !== undefined) {
          if (options.fieldsPerRecord === 0) {
            _nbFields = lineResult.length;
          } else {
            _nbFields = options.fieldsPerRecord;
          }
        }
      }
      if (lineResult.length > 0) {
        if (_nbFields && _nbFields !== lineResult.length) {
          throw new ParseError(lineIndex, lineIndex, null, ERR_FIELD_COUNT);
        }
        result.push(lineResult);
      }
    }
    return result;
  }
}
export function parse(input, opt = {
  skipFirstRow: false
}) {
  const parser = new Parser(opt);
  const r = parser.parse(input);
  if (opt.skipFirstRow || opt.columns) {
    let headers = [];
    if (opt.skipFirstRow) {
      const head = r.shift();
      assert(head !== undefined);
      headers = head;
    }
    if (opt.columns) {
      headers = opt.columns;
    }
    const firstLineIndex = opt.skipFirstRow ? 1 : 0;
    return r.map((row, i)=>{
      return convertRowToObject(row, headers, firstLineIndex + i);
    });
  }
  return r;
}
//# sourceMappingURL=data:application/json;base64,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