// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/** Options for {@linkcode joinToString}. */ /**
 * Transforms the elements in the given array to strings using the given
 * selector. Joins the produced strings into one using the given `separator`
 * and applying the given `prefix` and `suffix` to the whole string afterwards.
 * If the array could be huge, you can specify a non-negative value of `limit`,
 * in which case only the first `limit` elements will be appended, followed by
 * the `truncated` string. Returns the resulting string.
 *
 * @example
 * ```ts
 * import { joinToString } from "https://deno.land/std@$STD_VERSION/collections/join_to_string.ts";
 * import { assertEquals } from "https://deno.land/std@$STD_VERSION/assert/assert_equals.ts";
 *
 * const users = [
 *   { name: "Kim" },
 *   { name: "Anna" },
 *   { name: "Tim" },
 * ];
 *
 * const message = joinToString(users, (it) => it.name, {
 *   suffix: " are winners",
 *   prefix: "result: ",
 *   separator: " and ",
 *   limit: 1,
 *   truncated: "others",
 * });
 *
 * assertEquals(message, "result: Kim and others are winners");
 * ```
 */ export function joinToString(array, selector, { separator = ",", prefix = "", suffix = "", limit = -1, truncated = "..." } = {}) {
  let result = "";
  let index = -1;
  for (const el of array){
    index++;
    if (index > 0) {
      result += separator;
    }
    if (limit > -1 && index >= limit) {
      result += truncated;
      break;
    }
    result += selector(el);
  }
  result = prefix + result + suffix;
  return result;
}
//# sourceMappingURL=data:application/json;base64,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