// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// Copyright (c) Jason Campbell. MIT license
/**
 * Extracts
 * {@link https://daily-dev-tips.com/posts/what-exactly-is-frontmatter/ | front matter}
 * from strings.
 *
 * {@linkcode createExtractor} and {@linkcode test} functions
 * to handle many forms of front matter.
 *
 * Adapted from
 * {@link https://github.com/jxson/front-matter/blob/36f139ef797bd9e5196a9ede03ef481d7fbca18e/index.js | jxson/front-matter}.
 *
 * Supported formats:
 *
 * - [`YAML`](./front_matter/yaml.ts)
 * - [`TOML`](./front_matter/toml.ts)
 * - [`JSON`](./front_matter/json.ts)
 *
 * ### Basic usage
 *
 * example.md
 *
 * ```markdown
 * ---
 * module: front_matter
 * tags:
 *   - yaml
 *   - toml
 *   - json
 * ---
 *
 * deno is awesome
 * ```
 *
 * example.ts
 *
 * ```ts
 * import { extract } from "https://deno.land/std@$STD_VERSION/front_matter/any.ts";
 * import { test } from "https://deno.land/std@$STD_VERSION/front_matter/test.ts";
 *
 * const str = await Deno.readTextFile("./example.md");
 *
 * if (test(str)) {
 *   console.log(extract(str));
 * } else {
 *   console.log("document doesn't contain front matter");
 * }
 * ```
 *
 * ```sh
 * $ deno run ./example.ts
 * {
 *   frontMatter: "module: front_matter\ntags:\n  - yaml\n  - toml\n  - json",
 *   body: "deno is awesome",
 *   attrs: { module: "front_matter", tags: [ "yaml", "toml", "json" ] }
 * }
 * ```
 *
 * The above example recognizes any of the supported formats, extracts metadata and
 * parses accordingly. Please note that in this case both the [YAML](#yaml) and
 * [TOML](#toml) parsers will be imported as dependencies.
 *
 * If you need only one specific format then you can import the file named
 * respectively from [here](./front_matter).
 *
 * ### Advanced usage
 *
 * ```ts
 * import { test as _test } from "https://deno.land/std@$STD_VERSION/front_matter/test.ts";
 * import {
 *   createExtractor,
 *   Parser,
 * } from "https://deno.land/std@$STD_VERSION/front_matter/mod.ts";
 * import { parse } from "https://deno.land/std@$STD_VERSION/toml/parse.ts";
 *
 * const extract = createExtractor({
 *   "toml": parse as Parser,
 *   "json": JSON.parse as Parser,
 * });
 *
 * export function test(str: string): boolean {
 *   return _test(str, ["toml", "json"]);
 * }
 * ```
 *
 * In this setup `extract()` and `test()` will work with TOML and JSON and only.
 * This way the YAML parser is not loaded if not needed. You can cherry-pick which
 * combination of formats are you supporting based on your needs.
 *
 * ### Delimiters
 *
 * #### YAML
 *
 * ```markdown
 * ---
 * these: are
 * ---
 * ```
 *
 * ```markdown
 * ---yaml
 * all: recognized
 * ---
 * ```
 *
 * ```markdown
 * = yaml =
 * as: yaml
 * = yaml =
 * ```
 *
 * #### TOML
 *
 * ```markdown
 * ---toml
 * this = 'is'
 * ---
 * ```
 *
 * ```markdown
 * = toml =
 * parsed = 'as'
 * toml = 'data'
 * = toml =
 * ```
 *
 * ```markdown
 * +++
 * is = 'that'
 * not = 'cool?'
 * +++
 * ```
 *
 * #### JSON
 *
 * ```markdown
 * ---json
 * {
 *   "and": "this"
 * }
 * ---
 * ```
 *
 * ```markdown
 * {
 *   "is": "JSON"
 * }
 * ```
 *
 * @module
 */ export * from "./create_extractor.ts";
export * from "./test.ts";
//# sourceMappingURL=data:application/json;base64,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