// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
/**
 * Logging library with the support for terminal and file outputs. Also provides
 * interfaces for building custom loggers.
 *
 * ## Loggers
 *
 * Loggers are objects that you interact with. When you use a logger method it
 * constructs a `LogRecord` and passes it down to its handlers for output. To
 * create custom loggers, specify them in `loggers` when calling `log.setup`.
 *
 * ## Custom message format
 *
 * If you want to override default format of message you can define `formatter`
 * option for handler. It can be either simple string-based format that uses
 * `LogRecord` fields or more complicated function-based one that takes `LogRecord`
 * as argument and outputs string.
 *
 * The default log format is `{levelName} {msg}`.
 *
 * ### Logging Structured JSON Lines
 *
 * To output logs in a structured JSON format you can configure most handlers
 * with a formatter that produces a JSON string. Either use the premade
 * `log.formatters.jsonFormatter` or write your own function that takes a
 * {@linkcode LogRecord} and returns a JSON.stringify'd object.
 * If you want the log to go to stdout then use {@linkcode ConsoleHandler} with
 * the configuration `useColors: false` to turn off the ANSI terminal colours.
 *
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * log.setup({
 *   handlers: {
 *     default: new log.ConsoleHandler("DEBUG", {
 *       formatter: log.formatters.jsonFormatter,
 *       useColors: false,
 *     }),
 *   },
 * });
 * ```
 *
 * The first argument passed to a log function is always treated as the
 * message and will be stringified differently. To have arguments JSON.stringify'd
 * you must pass them after the first.
 *
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * log.info("This is the message", { thisWillBe: "JSON.stringify'd"});
 * // {"level":"INFO","datetime":1702501580505,"message":"This is the message","args":{"thisWillBe":"JSON.stringify'd"}}
 *
 * log.info({ thisWontBe: "JSON.stringify'd"}, "This is an argument");
 * // {"level":"INFO","datetime":1702501580505,"message":"{\"thisWontBe\":\"JSON.stringify'd\"}","args":"This is an argument"}
 * ```
 *
 * ## Inline Logging
 *
 * Log functions return the data passed in the `msg` parameter. Data is returned
 * regardless if the logger actually logs it.
 *
 * ## Lazy Log Evaluation
 *
 * Some log statements are expensive to compute. In these cases, you can use
 * lazy log evaluation to prevent the computation taking place if the logger
 * won't log the message.
 *
 * > NOTE: When using lazy log evaluation, `undefined` will be returned if the
 * > resolver function is not called because the logger won't log it. It is an
 * > antipattern use lazy evaluation with inline logging because the return value
 * > depends on the current log level.
 *
 * ## For module authors
 *
 * The authors of public modules can let the users display the internal logs of the
 * module by using a custom logger:
 *
 * ```ts
 * import { getLogger } from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * function logger() {
 *   return getLogger("my-awesome-module");
 * }
 *
 * export function sum(a: number, b: number) {
 *   logger().debug(`running ${a} + ${b}`);
 *   return a + b;
 * }
 *
 * export function mult(a: number, b: number) {
 *   logger().debug(`running ${a} * ${b}`);
 *   return a * b;
 * }
 * ```
 *
 * The user of the module can then display the internal logs with:
 *
 * ```ts, ignore
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 * import { sum } from "<the-awesome-module>/mod.ts";
 *
 * log.setup({
 *   handlers: {
 *     console: new log.ConsoleHandler("DEBUG"),
 *   },
 *
 *   loggers: {
 *     "my-awesome-module": {
 *       level: "DEBUG",
 *       handlers: ["console"],
 *     },
 *   },
 * });
 *
 * sum(1, 2); // prints "running 1 + 2" to the console
 * ```
 *
 * Please note that, due to the order of initialization of the loggers, the
 * following won't work:
 *
 * ```ts
 * import { getLogger } from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * const logger = getLogger("my-awesome-module");
 *
 * export function sum(a: number, b: number) {
 *   logger.debug(`running ${a} + ${b}`); // no message will be logged, because getLogger() was called before log.setup()
 *   return a + b;
 * }
 * ```
 *
 * @example
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * // Simple default logger out of the box. You can customize it
 * // by overriding logger and handler named "default", or providing
 * // additional logger configurations. You can log any data type.
 * log.debug("Hello world");
 * log.info(123456);
 * log.warn(true);
 * log.error({ foo: "bar", fizz: "bazz" });
 * log.critical("500 Internal server error");
 *
 * // custom configuration with 2 loggers (the default and `tasks` loggers).
 * log.setup({
 *   handlers: {
 *     console: new log.ConsoleHandler("DEBUG"),
 *
 *     file: new log.FileHandler("WARN", {
 *       filename: "./log.txt",
 *       // you can change format of output message using any keys in `LogRecord`.
 *       formatter: (record) => `${record.levelName} ${record.msg}`,
 *     }),
 *   },
 *
 *   loggers: {
 *     // configure default logger available via short-hand methods above.
 *     default: {
 *       level: "DEBUG",
 *       handlers: ["console", "file"],
 *     },
 *
 *     tasks: {
 *       level: "ERROR",
 *       handlers: ["console"],
 *     },
 *   },
 * });
 *
 * let logger;
 *
 * // get default logger.
 * logger = log.getLogger();
 * logger.debug("fizz"); // logs to `console`, because `file` handler requires "WARN" level.
 * logger.warn(41256); // logs to both `console` and `file` handlers.
 *
 * // get custom logger
 * logger = log.getLogger("tasks");
 * logger.debug("fizz"); // won't get output because this logger has "ERROR" level.
 * logger.error({ productType: "book", value: "126.11" }); // log to `console`.
 *
 * // if you try to use a logger that hasn't been configured
 * // you're good to go, it gets created automatically with level set to 0
 * // so no message is logged.
 * const unknownLogger = log.getLogger("mystery");
 * unknownLogger.info("foobar"); // no-op
 * ```
 *
 * @example
 * Custom message format example
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * log.setup({
 *   handlers: {
 *     stringFmt: new log.ConsoleHandler("DEBUG", {
 *       formatter: (record) => `[${record.levelName}] ${record.msg}`,
 *     }),
 *
 *     functionFmt: new log.ConsoleHandler("DEBUG", {
 *       formatter: (logRecord) => {
 *         let msg = `${logRecord.level} ${logRecord.msg}`;
 *
 *         logRecord.args.forEach((arg, index) => {
 *           msg += `, arg${index}: ${arg}`;
 *         });
 *
 *         return msg;
 *       },
 *     }),
 *
 *     anotherFmt: new log.ConsoleHandler("DEBUG", {
 *       formatter: (record) => `[${record.loggerName}] - ${record.levelName} ${record.msg}`,
 *     }),
 *   },
 *
 *   loggers: {
 *     default: {
 *       level: "DEBUG",
 *       handlers: ["stringFmt", "functionFmt"],
 *     },
 *     dataLogger: {
 *       level: "INFO",
 *       handlers: ["anotherFmt"],
 *     },
 *   },
 * });
 *
 * // calling:
 * log.debug("Hello, world!", 1, "two", [3, 4, 5]);
 * // results in: [DEBUG] Hello, world!
 * // output from "stringFmt" handler.
 * // 10 Hello, world!, arg0: 1, arg1: two, arg3: [3, 4, 5] // output from "functionFmt" formatter.
 *
 * // calling:
 * log.getLogger("dataLogger").error("oh no!");
 * // results in:
 * // [dataLogger] - ERROR oh no! // output from anotherFmt handler.
 * ```

 *
 * @example
 * JSON to stdout with no color example
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * log.setup({
 *   handlers: {
 *     jsonStdout: new log.ConsoleHandler("DEBUG", {
 *       formatter: log.formatters.jsonFormatter,
 *       useColors: false,
 *     }),
 *   },
 *
 *   loggers: {
 *     default: {
 *       level: "DEBUG",
 *       handlers: ["jsonStdout"],
 *     },
 *   },
 * });
 *
 * // calling:
 * log.info("Hey");
 * // results in:
 * // {"level":"INFO","datetime":1702481922294,"message":"Hey"}
 *
 * // calling:
 * log.info("Hey", { product: "nail" });
 * // results in:
 * // {"level":"INFO","datetime":1702484111115,"message":"Hey","args":{"product":"nail"}}
 *
 * // calling:
 * log.info("Hey", 1, "two", [3, 4, 5]);
 * // results in:
 * // {"level":"INFO","datetime":1702481922294,"message":"Hey","args":[1,"two",[3,4,5]]}
 * ```
 *
 * @example
 * Custom JSON example
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * log.setup({
 *   handlers: {
 *     customJsonFmt: new log.ConsoleHandler("DEBUG", {
 *       formatter: (record) => JSON.stringify({
 *         lvl: record.level,
 *         msg: record.msg,
 *         time: record.datetime.toISOString(),
 *         name: record.loggerName,
 *       }),
 *       useColors: false,
 *     }),
 *   },
 *
 *   loggers: {
 *     default: {
 *       level: "DEBUG",
 *       handlers: ["customJsonFmt"],
 *     },
 *   },
 * });
 *
 * // calling:
 * log.info("complete");
 * // results in:
 * // {"lvl":20,"msg":"complete","time":"2023-12-13T16:38:27.328Z","name":"default"}
 * ```
 *
 * @example
 * Inline Logging
 * ```ts
 * import * as logger from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * const stringData: string = logger.debug("hello world");
 * const booleanData: boolean = logger.debug(true, 1, "abc");
 * const fn = (): number => {
 *   return 123;
 * };
 * const resolvedFunctionData: number = logger.debug(fn());
 * console.log(stringData); // 'hello world'
 * console.log(booleanData); // true
 * console.log(resolvedFunctionData); // 123
 * ```
 *
 * @example
 * Lazy Log Evaluation
 * ```ts
 * import * as log from "https://deno.land/std@$STD_VERSION/log/mod.ts";
 *
 * log.setup({
 *   handlers: {
 *     console: new log.ConsoleHandler("DEBUG"),
 *   },
 *
 *   loggers: {
 *     tasks: {
 *       level: "ERROR",
 *       handlers: ["console"],
 *     },
 *   },
 * });
 *
 * function someExpensiveFn(num: number, bool: boolean) {
 *   // do some expensive computation
 * }
 *
 * // not logged, as debug < error.
 * const data = log.debug(() => someExpensiveFn(5, true));
 * console.log(data); // undefined
 * ```
 *
 * Handlers are responsible for actual output of log messages. When a handler is
 * called by a logger, it firstly checks that {@linkcode LogRecord}'s level is
 * not lower than level of the handler. If level check passes, handlers formats
 * log record into string and outputs it to target.
 *
 * ## Custom handlers
 *
 * Custom handlers can be implemented by subclassing {@linkcode BaseHandler} or
 * {@linkcode WriterHandler}.
 *
 * {@linkcode BaseHandler} is bare-bones handler that has no output logic at all,
 *
 * {@linkcode WriterHandler} is an abstract class that supports any target with
 * `Writer` interface.
 *
 * During setup async hooks `setup` and `destroy` are called, you can use them
 * to open and close file/HTTP connection or any other action you might need.
 *
 * For examples check source code of {@linkcode FileHandler}`
 * and {@linkcode TestHandler}.
 *
 * @module
 */ export * from "./base_handler.ts";
export * from "./console_handler.ts";
export * from "./file_handler.ts";
export * from "./rotating_file_handler.ts";
export * from "./levels.ts";
export * from "./logger.ts";
export * from "./formatters.ts";
export * from "./critical.ts";
export * from "./debug.ts";
export * from "./error.ts";
export * from "./get_logger.ts";
export * from "./info.ts";
export * from "./setup.ts";
export * from "./warn.ts";
//# sourceMappingURL=data:application/json;base64,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