// deno-fmt-ignore-file
// deno-lint-ignore-file
// This code was bundled using `deno bundle` and it's not recommended to edit it manually

const LogLevels = {
    NOTSET: 0,
    DEBUG: 10,
    INFO: 20,
    WARN: 30,
    ERROR: 40,
    CRITICAL: 50
};
Object.keys(LogLevels).filter((key)=>isNaN(Number(key)));
const byLevel = {
    [LogLevels.NOTSET]: "NOTSET",
    [LogLevels.DEBUG]: "DEBUG",
    [LogLevels.INFO]: "INFO",
    [LogLevels.WARN]: "WARN",
    [LogLevels.ERROR]: "ERROR",
    [LogLevels.CRITICAL]: "CRITICAL"
};
function getLevelByName(name) {
    const level = LogLevels[name];
    if (level !== undefined) {
        return level;
    }
    throw new Error(`no log level found for name: ${name}`);
}
function getLevelName(level) {
    const levelName = byLevel[level];
    if (levelName) {
        return levelName;
    }
    throw new Error(`no level name found for level: ${level}`);
}
const DEFAULT_FORMATTER = ({ levelName, msg })=>`${levelName} ${msg}`;
class BaseHandler {
    level;
    levelName;
    formatter;
    constructor(levelName, options = {}){
        this.level = getLevelByName(levelName);
        this.levelName = levelName;
        this.formatter = options.formatter || DEFAULT_FORMATTER;
    }
    handle(logRecord) {
        if (this.level > logRecord.level) return;
        const msg = this.format(logRecord);
        this.log(msg);
    }
    format(logRecord) {
        return this.formatter(logRecord);
    }
    log(_msg) {}
    setup() {}
    destroy() {}
    [Symbol.dispose]() {
        this.destroy();
    }
}
const { Deno: Deno1 } = globalThis;
const noColor = typeof Deno1?.noColor === "boolean" ? Deno1.noColor : false;
let enabled = !noColor;
function code(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run(str, code) {
    return enabled ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function bold(str) {
    return run(str, code([
        1
    ], 22));
}
function dim(str) {
    return run(str, code([
        2
    ], 22));
}
function underline(str) {
    return run(str, code([
        4
    ], 24));
}
function red(str) {
    return run(str, code([
        31
    ], 39));
}
function green(str) {
    return run(str, code([
        32
    ], 39));
}
function yellow(str) {
    return run(str, code([
        33
    ], 39));
}
function blue(str) {
    return run(str, code([
        34
    ], 39));
}
function brightRed(str) {
    return run(str, code([
        91
    ], 39));
}
const ANSI_PATTERN = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TXZcf-nq-uy=><~]))"
].join("|"), "g");
function stripColor(string) {
    return stripAnsiCode(string);
}
function stripAnsiCode(string) {
    return string.replace(ANSI_PATTERN, "");
}
class ConsoleHandler extends BaseHandler {
    #useColors;
    constructor(levelName, options = {}){
        super(levelName, options);
        this.#useColors = options.useColors ?? true;
    }
    format(logRecord) {
        let msg = super.format(logRecord);
        if (this.#useColors) {
            msg = this.applyColors(msg, logRecord.level);
        }
        return msg;
    }
    applyColors(msg, level) {
        switch(level){
            case LogLevels.INFO:
                msg = blue(msg);
                break;
            case LogLevels.WARN:
                msg = yellow(msg);
                break;
            case LogLevels.ERROR:
                msg = red(msg);
                break;
            case LogLevels.CRITICAL:
                msg = bold(red(msg));
                break;
            default:
                break;
        }
        return msg;
    }
    log(msg) {
        console.log(msg);
    }
}
class FileHandler extends BaseHandler {
    _file;
    _buf = new Uint8Array(4096);
    _pointer = 0;
    _filename;
    _mode;
    _openOptions;
    _encoder = new TextEncoder();
    #unloadCallback = (()=>{
        this.destroy();
    }).bind(this);
    constructor(levelName, options){
        super(levelName, options);
        this._filename = options.filename;
        this._mode = options.mode ? options.mode : "a";
        this._openOptions = {
            createNew: this._mode === "x",
            create: this._mode !== "x",
            append: this._mode === "a",
            truncate: this._mode !== "a",
            write: true
        };
    }
    setup() {
        this._file = Deno.openSync(this._filename, this._openOptions);
        this.#resetBuffer();
        addEventListener("unload", this.#unloadCallback);
    }
    handle(logRecord) {
        super.handle(logRecord);
        if (logRecord.level > LogLevels.ERROR) {
            this.flush();
        }
    }
    log(msg) {
        const bytes = this._encoder.encode(msg + "\n");
        if (bytes.byteLength > this._buf.byteLength - this._pointer) {
            this.flush();
        }
        this._buf.set(bytes, this._pointer);
        this._pointer += bytes.byteLength;
    }
    flush() {
        if (this._pointer > 0 && this._file) {
            let written = 0;
            while(written < this._pointer){
                written += this._file.writeSync(this._buf.subarray(written, this._pointer));
            }
            this.#resetBuffer();
        }
    }
    #resetBuffer() {
        this._pointer = 0;
    }
    destroy() {
        this.flush();
        this._file?.close();
        this._file = undefined;
        removeEventListener("unload", this.#unloadCallback);
    }
}
function existsSync(path, options) {
    try {
        const stat = Deno.statSync(path);
        if (options && (options.isReadable || options.isDirectory || options.isFile)) {
            if (options.isDirectory && options.isFile) {
                throw new TypeError("ExistsOptions.options.isDirectory and ExistsOptions.options.isFile must not be true together.");
            }
            if (options.isDirectory && !stat.isDirectory || options.isFile && !stat.isFile) {
                return false;
            }
            if (options.isReadable) {
                if (stat.mode === null) {
                    return true;
                }
                if (Deno.uid() === stat.uid) {
                    return (stat.mode & 0o400) === 0o400;
                } else if (Deno.gid() === stat.gid) {
                    return (stat.mode & 0o040) === 0o040;
                }
                return (stat.mode & 0o004) === 0o004;
            }
        }
        return true;
    } catch (error) {
        if (error instanceof Deno.errors.NotFound) {
            return false;
        }
        if (error instanceof Deno.errors.PermissionDenied) {
            if (Deno.permissions.querySync({
                name: "read",
                path
            }).state === "granted") {
                return !options?.isReadable;
            }
        }
        throw error;
    }
}
class LogRecord {
    msg;
    #args;
    #datetime;
    level;
    levelName;
    loggerName;
    constructor(options){
        this.msg = options.msg;
        this.#args = [
            ...options.args
        ];
        this.level = options.level;
        this.loggerName = options.loggerName;
        this.#datetime = new Date();
        this.levelName = getLevelName(options.level);
    }
    get args() {
        return [
            ...this.#args
        ];
    }
    get datetime() {
        return new Date(this.#datetime.getTime());
    }
}
class Logger {
    #level;
    handlers;
    #loggerName;
    constructor(loggerName, levelName, options = {}){
        this.#loggerName = loggerName;
        this.#level = getLevelByName(levelName);
        this.handlers = options.handlers || [];
    }
    get level() {
        return this.#level;
    }
    set level(level) {
        try {
            this.#level = getLevelByName(getLevelName(level));
        } catch (_) {
            throw new TypeError(`Invalid log level: ${level}`);
        }
    }
    get levelName() {
        return getLevelName(this.#level);
    }
    set levelName(levelName) {
        this.#level = getLevelByName(levelName);
    }
    get loggerName() {
        return this.#loggerName;
    }
    #log(level, msg, ...args) {
        if (this.level > level) {
            return msg instanceof Function ? undefined : msg;
        }
        let fnResult;
        let logMessage;
        if (msg instanceof Function) {
            fnResult = msg();
            logMessage = this.asString(fnResult);
        } else {
            logMessage = this.asString(msg);
        }
        const record = new LogRecord({
            msg: logMessage,
            args: args,
            level: level,
            loggerName: this.loggerName
        });
        this.handlers.forEach((handler)=>{
            handler.handle(record);
        });
        return msg instanceof Function ? fnResult : msg;
    }
    asString(data, isProperty = false) {
        if (typeof data === "string") {
            if (isProperty) return `"${data}"`;
            return data;
        } else if (data === null || typeof data === "number" || typeof data === "bigint" || typeof data === "boolean" || typeof data === "undefined" || typeof data === "symbol") {
            return String(data);
        } else if (data instanceof Error) {
            return data.stack;
        } else if (typeof data === "object") {
            return `{${Object.entries(data).map(([k, v])=>`"${k}":${this.asString(v, true)}`).join(",")}}`;
        }
        return "undefined";
    }
    debug(msg, ...args) {
        return this.#log(LogLevels.DEBUG, msg, ...args);
    }
    info(msg, ...args) {
        return this.#log(LogLevels.INFO, msg, ...args);
    }
    warn(msg, ...args) {
        return this.#log(LogLevels.WARN, msg, ...args);
    }
    error(msg, ...args) {
        return this.#log(LogLevels.ERROR, msg, ...args);
    }
    critical(msg, ...args) {
        return this.#log(LogLevels.CRITICAL, msg, ...args);
    }
}
class AssertionError extends Error {
    constructor(message){
        super(message);
        this.name = "AssertionError";
    }
}
function assert(expr, msg = "") {
    if (!expr) {
        throw new AssertionError(msg);
    }
}
const DEFAULT_LEVEL = "INFO";
const DEFAULT_CONFIG = {
    handlers: {
        default: new ConsoleHandler(DEFAULT_LEVEL)
    },
    loggers: {
        default: {
            level: DEFAULT_LEVEL,
            handlers: [
                "default"
            ]
        }
    }
};
const state = {
    handlers: new Map(),
    loggers: new Map(),
    config: DEFAULT_CONFIG
};
function getLogger(name) {
    if (!name) {
        const d = state.loggers.get("default");
        assert(d !== undefined, `"default" logger must be set for getting logger without name`);
        return d;
    }
    const result1 = state.loggers.get(name);
    if (!result1) {
        const logger = new Logger(name, "NOTSET", {
            handlers: []
        });
        state.loggers.set(name, logger);
        return logger;
    }
    return result1;
}
function debug(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").debug(msg, ...args);
    }
    return getLogger("default").debug(msg, ...args);
}
function error(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").error(msg, ...args);
    }
    return getLogger("default").error(msg, ...args);
}
function info(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").info(msg, ...args);
    }
    return getLogger("default").info(msg, ...args);
}
function setup(config) {
    state.config = {
        handlers: {
            ...DEFAULT_CONFIG.handlers,
            ...config.handlers
        },
        loggers: {
            ...DEFAULT_CONFIG.loggers,
            ...config.loggers
        }
    };
    state.handlers.forEach((handler)=>{
        handler.destroy();
    });
    state.handlers.clear();
    const handlers = state.config.handlers || {};
    for (const [handlerName, handler] of Object.entries(handlers)){
        handler.setup();
        state.handlers.set(handlerName, handler);
    }
    state.loggers.clear();
    const loggers = state.config.loggers || {};
    for (const [loggerName, loggerConfig] of Object.entries(loggers)){
        const handlerNames = loggerConfig.handlers || [];
        const handlers = [];
        handlerNames.forEach((handlerName)=>{
            const handler = state.handlers.get(handlerName);
            if (handler) {
                handlers.push(handler);
            }
        });
        const levelName = loggerConfig.level || DEFAULT_LEVEL;
        const logger = new Logger(loggerName, levelName, {
            handlers: handlers
        });
        state.loggers.set(loggerName, logger);
    }
}
setup(DEFAULT_CONFIG);
function warn(msg, ...args) {
    if (msg instanceof Function) {
        return getLogger("default").warn(msg, ...args);
    }
    return getLogger("default").warn(msg, ...args);
}
function assertPath(path) {
    if (typeof path !== "string") {
        throw new TypeError(`Path must be a string. Received ${JSON.stringify(path)}`);
    }
}
function stripSuffix(name, suffix) {
    if (suffix.length >= name.length) {
        return name;
    }
    const lenDiff = name.length - suffix.length;
    for(let i1 = suffix.length - 1; i1 >= 0; --i1){
        if (name.charCodeAt(lenDiff + i1) !== suffix.charCodeAt(i1)) {
            return name;
        }
    }
    return name.slice(0, -suffix.length);
}
function lastPathSegment(path, isSep, start = 0) {
    let matchedNonSeparator = false;
    let end = path.length;
    for(let i1 = path.length - 1; i1 >= start; --i1){
        if (isSep(path.charCodeAt(i1))) {
            if (matchedNonSeparator) {
                start = i1 + 1;
                break;
            }
        } else if (!matchedNonSeparator) {
            matchedNonSeparator = true;
            end = i1 + 1;
        }
    }
    return path.slice(start, end);
}
function assertArgs(path, suffix) {
    assertPath(path);
    if (path.length === 0) return path;
    if (typeof suffix !== "string") {
        throw new TypeError(`Suffix must be a string. Received ${JSON.stringify(suffix)}`);
    }
}
const CHAR_FORWARD_SLASH = 47;
function stripTrailingSeparators(segment, isSep) {
    if (segment.length <= 1) {
        return segment;
    }
    let end = segment.length;
    for(let i1 = segment.length - 1; i1 > 0; i1--){
        if (isSep(segment.charCodeAt(i1))) {
            end = i1;
        } else {
            break;
        }
    }
    return segment.slice(0, end);
}
function isPosixPathSeparator(code) {
    return code === 47;
}
function isPathSeparator(code) {
    return code === 47 || code === 92;
}
function isWindowsDeviceRoot(code) {
    return code >= 97 && code <= 122 || code >= 65 && code <= 90;
}
function basename(path, suffix = "") {
    assertArgs(path, suffix);
    let start = 0;
    if (path.length >= 2) {
        const drive = path.charCodeAt(0);
        if (isWindowsDeviceRoot(drive)) {
            if (path.charCodeAt(1) === 58) start = 2;
        }
    }
    const lastSegment = lastPathSegment(path, isPathSeparator, start);
    const strippedSegment = stripTrailingSeparators(lastSegment, isPathSeparator);
    return suffix ? stripSuffix(strippedSegment, suffix) : strippedSegment;
}
const SEPARATOR = "\\";
const SEPARATOR_PATTERN = /[\\/]+/;
function assertArg(path) {
    assertPath(path);
    if (path.length === 0) return ".";
}
function dirname(path) {
    assertArg(path);
    const len = path.length;
    let rootEnd = -1;
    let end = -1;
    let matchedSlash = true;
    let offset = 0;
    const code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            rootEnd = offset = 1;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return path;
                        }
                        if (j !== last) {
                            rootEnd = offset = j + 1;
                        }
                    }
                }
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                rootEnd = offset = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) rootEnd = offset = 3;
                }
            }
        }
    } else if (isPathSeparator(code)) {
        return path;
    }
    for(let i1 = len - 1; i1 >= offset; --i1){
        if (isPathSeparator(path.charCodeAt(i1))) {
            if (!matchedSlash) {
                end = i1;
                break;
            }
        } else {
            matchedSlash = false;
        }
    }
    if (end === -1) {
        if (rootEnd === -1) return ".";
        else end = rootEnd;
    }
    return stripTrailingSeparators(path.slice(0, end), isPosixPathSeparator);
}
function extname(path) {
    assertPath(path);
    let start = 0;
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    if (path.length >= 2 && path.charCodeAt(1) === 58 && isWindowsDeviceRoot(path.charCodeAt(0))) {
        start = startPart = 2;
    }
    for(let i1 = path.length - 1; i1 >= start; --i1){
        const code = path.charCodeAt(i1);
        if (isPathSeparator(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function _format(sep, pathObject) {
    const dir = pathObject.dir || pathObject.root;
    const base = pathObject.base || (pathObject.name || "") + (pathObject.ext || "");
    if (!dir) return base;
    if (base === sep) return dir;
    if (dir === pathObject.root) return dir + base;
    return dir + sep + base;
}
function assertArg1(pathObject) {
    if (pathObject === null || typeof pathObject !== "object") {
        throw new TypeError(`The "pathObject" argument must be of type Object. Received type ${typeof pathObject}`);
    }
}
function assertArg2(url) {
    url = url instanceof URL ? url : new URL(url);
    if (url.protocol !== "file:") {
        throw new TypeError("Must be a file URL.");
    }
    return url;
}
function fromFileUrl(url) {
    url = assertArg2(url);
    let path = decodeURIComponent(url.pathname.replace(/\//g, "\\").replace(/%(?![0-9A-Fa-f]{2})/g, "%25")).replace(/^\\*([A-Za-z]:)(\\|$)/, "$1\\");
    if (url.hostname !== "") {
        path = `\\\\${url.hostname}${path}`;
    }
    return path;
}
function isAbsolute(path) {
    assertPath(path);
    const len = path.length;
    if (len === 0) return false;
    const code = path.charCodeAt(0);
    if (isPathSeparator(code)) {
        return true;
    } else if (isWindowsDeviceRoot(code)) {
        if (len > 2 && path.charCodeAt(1) === 58) {
            if (isPathSeparator(path.charCodeAt(2))) return true;
        }
    }
    return false;
}
function assertArg3(path) {
    assertPath(path);
    if (path.length === 0) return ".";
}
function normalizeString(path, allowAboveRoot, separator, isPathSeparator) {
    let res = "";
    let lastSegmentLength = 0;
    let lastSlash = -1;
    let dots = 0;
    let code;
    for(let i1 = 0, len = path.length; i1 <= len; ++i1){
        if (i1 < len) code = path.charCodeAt(i1);
        else if (isPathSeparator(code)) break;
        else code = CHAR_FORWARD_SLASH;
        if (isPathSeparator(code)) {
            if (lastSlash === i1 - 1 || dots === 1) {} else if (lastSlash !== i1 - 1 && dots === 2) {
                if (res.length < 2 || lastSegmentLength !== 2 || res.charCodeAt(res.length - 1) !== 46 || res.charCodeAt(res.length - 2) !== 46) {
                    if (res.length > 2) {
                        const lastSlashIndex = res.lastIndexOf(separator);
                        if (lastSlashIndex === -1) {
                            res = "";
                            lastSegmentLength = 0;
                        } else {
                            res = res.slice(0, lastSlashIndex);
                            lastSegmentLength = res.length - 1 - res.lastIndexOf(separator);
                        }
                        lastSlash = i1;
                        dots = 0;
                        continue;
                    } else if (res.length === 2 || res.length === 1) {
                        res = "";
                        lastSegmentLength = 0;
                        lastSlash = i1;
                        dots = 0;
                        continue;
                    }
                }
                if (allowAboveRoot) {
                    if (res.length > 0) res += `${separator}..`;
                    else res = "..";
                    lastSegmentLength = 2;
                }
            } else {
                if (res.length > 0) res += separator + path.slice(lastSlash + 1, i1);
                else res = path.slice(lastSlash + 1, i1);
                lastSegmentLength = i1 - lastSlash - 1;
            }
            lastSlash = i1;
            dots = 0;
        } else if (code === 46 && dots !== -1) {
            ++dots;
        } else {
            dots = -1;
        }
    }
    return res;
}
function normalize(path) {
    assertArg3(path);
    const len = path.length;
    let rootEnd = 0;
    let device;
    let isAbsolute = false;
    const code = path.charCodeAt(0);
    if (len > 1) {
        if (isPathSeparator(code)) {
            isAbsolute = true;
            if (isPathSeparator(path.charCodeAt(1))) {
                let j = 2;
                let last = j;
                for(; j < len; ++j){
                    if (isPathSeparator(path.charCodeAt(j))) break;
                }
                if (j < len && j !== last) {
                    const firstPart = path.slice(last, j);
                    last = j;
                    for(; j < len; ++j){
                        if (!isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        last = j;
                        for(; j < len; ++j){
                            if (isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j === len) {
                            return `\\\\${firstPart}\\${path.slice(last)}\\`;
                        } else if (j !== last) {
                            device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                            rootEnd = j;
                        }
                    }
                }
            } else {
                rootEnd = 1;
            }
        } else if (isWindowsDeviceRoot(code)) {
            if (path.charCodeAt(1) === 58) {
                device = path.slice(0, 2);
                rootEnd = 2;
                if (len > 2) {
                    if (isPathSeparator(path.charCodeAt(2))) {
                        isAbsolute = true;
                        rootEnd = 3;
                    }
                }
            }
        }
    } else if (isPathSeparator(code)) {
        return "\\";
    }
    let tail;
    if (rootEnd < len) {
        tail = normalizeString(path.slice(rootEnd), !isAbsolute, "\\", isPathSeparator);
    } else {
        tail = "";
    }
    if (tail.length === 0 && !isAbsolute) tail = ".";
    if (tail.length > 0 && isPathSeparator(path.charCodeAt(len - 1))) {
        tail += "\\";
    }
    if (device === undefined) {
        if (isAbsolute) {
            if (tail.length > 0) return `\\${tail}`;
            else return "\\";
        } else if (tail.length > 0) {
            return tail;
        } else {
            return "";
        }
    } else if (isAbsolute) {
        if (tail.length > 0) return `${device}\\${tail}`;
        else return `${device}\\`;
    } else if (tail.length > 0) {
        return device + tail;
    } else {
        return device;
    }
}
function join(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    let firstPart = null;
    for(let i1 = 0; i1 < paths.length; ++i1){
        const path = paths[i1];
        assertPath(path);
        if (path.length > 0) {
            if (joined === undefined) joined = firstPart = path;
            else joined += `\\${path}`;
        }
    }
    if (joined === undefined) return ".";
    let needsReplace = true;
    let slashCount = 0;
    assert(firstPart !== null);
    if (isPathSeparator(firstPart.charCodeAt(0))) {
        ++slashCount;
        const firstLen = firstPart.length;
        if (firstLen > 1) {
            if (isPathSeparator(firstPart.charCodeAt(1))) {
                ++slashCount;
                if (firstLen > 2) {
                    if (isPathSeparator(firstPart.charCodeAt(2))) ++slashCount;
                    else {
                        needsReplace = false;
                    }
                }
            }
        }
    }
    if (needsReplace) {
        for(; slashCount < joined.length; ++slashCount){
            if (!isPathSeparator(joined.charCodeAt(slashCount))) break;
        }
        if (slashCount >= 2) joined = `\\${joined.slice(slashCount)}`;
    }
    return normalize(joined);
}
function resolve(...pathSegments) {
    let resolvedDevice = "";
    let resolvedTail = "";
    let resolvedAbsolute = false;
    for(let i1 = pathSegments.length - 1; i1 >= -1; i1--){
        let path;
        const { Deno: Deno1 } = globalThis;
        if (i1 >= 0) {
            path = pathSegments[i1];
        } else if (!resolvedDevice) {
            if (typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a drive-letter-less path without a CWD.");
            }
            path = Deno1.cwd();
        } else {
            if (typeof Deno1?.env?.get !== "function" || typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno1.cwd();
            if (path === undefined || path.slice(0, 3).toLowerCase() !== `${resolvedDevice.toLowerCase()}\\`) {
                path = `${resolvedDevice}\\`;
            }
        }
        assertPath(path);
        const len = path.length;
        if (len === 0) continue;
        let rootEnd = 0;
        let device = "";
        let isAbsolute = false;
        const code = path.charCodeAt(0);
        if (len > 1) {
            if (isPathSeparator(code)) {
                isAbsolute = true;
                if (isPathSeparator(path.charCodeAt(1))) {
                    let j = 2;
                    let last = j;
                    for(; j < len; ++j){
                        if (isPathSeparator(path.charCodeAt(j))) break;
                    }
                    if (j < len && j !== last) {
                        const firstPart = path.slice(last, j);
                        last = j;
                        for(; j < len; ++j){
                            if (!isPathSeparator(path.charCodeAt(j))) break;
                        }
                        if (j < len && j !== last) {
                            last = j;
                            for(; j < len; ++j){
                                if (isPathSeparator(path.charCodeAt(j))) break;
                            }
                            if (j === len) {
                                device = `\\\\${firstPart}\\${path.slice(last)}`;
                                rootEnd = j;
                            } else if (j !== last) {
                                device = `\\\\${firstPart}\\${path.slice(last, j)}`;
                                rootEnd = j;
                            }
                        }
                    }
                } else {
                    rootEnd = 1;
                }
            } else if (isWindowsDeviceRoot(code)) {
                if (path.charCodeAt(1) === 58) {
                    device = path.slice(0, 2);
                    rootEnd = 2;
                    if (len > 2) {
                        if (isPathSeparator(path.charCodeAt(2))) {
                            isAbsolute = true;
                            rootEnd = 3;
                        }
                    }
                }
            }
        } else if (isPathSeparator(code)) {
            rootEnd = 1;
            isAbsolute = true;
        }
        if (device.length > 0 && resolvedDevice.length > 0 && device.toLowerCase() !== resolvedDevice.toLowerCase()) {
            continue;
        }
        if (resolvedDevice.length === 0 && device.length > 0) {
            resolvedDevice = device;
        }
        if (!resolvedAbsolute) {
            resolvedTail = `${path.slice(rootEnd)}\\${resolvedTail}`;
            resolvedAbsolute = isAbsolute;
        }
        if (resolvedAbsolute && resolvedDevice.length > 0) break;
    }
    resolvedTail = normalizeString(resolvedTail, !resolvedAbsolute, "\\", isPathSeparator);
    return resolvedDevice + (resolvedAbsolute ? "\\" : "") + resolvedTail || ".";
}
function assertArgs1(from, to) {
    assertPath(from);
    assertPath(to);
    if (from === to) return "";
}
function relative(from, to) {
    assertArgs1(from, to);
    const fromOrig = resolve(from);
    const toOrig = resolve(to);
    if (fromOrig === toOrig) return "";
    from = fromOrig.toLowerCase();
    to = toOrig.toLowerCase();
    if (from === to) return "";
    let fromStart = 0;
    let fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (from.charCodeAt(fromStart) !== 92) break;
    }
    for(; fromEnd - 1 > fromStart; --fromEnd){
        if (from.charCodeAt(fromEnd - 1) !== 92) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 0;
    let toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (to.charCodeAt(toStart) !== 92) break;
    }
    for(; toEnd - 1 > toStart; --toEnd){
        if (to.charCodeAt(toEnd - 1) !== 92) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i1 = 0;
    for(; i1 <= length; ++i1){
        if (i1 === length) {
            if (toLen > length) {
                if (to.charCodeAt(toStart + i1) === 92) {
                    return toOrig.slice(toStart + i1 + 1);
                } else if (i1 === 2) {
                    return toOrig.slice(toStart + i1);
                }
            }
            if (fromLen > length) {
                if (from.charCodeAt(fromStart + i1) === 92) {
                    lastCommonSep = i1;
                } else if (i1 === 2) {
                    lastCommonSep = 3;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i1);
        const toCode = to.charCodeAt(toStart + i1);
        if (fromCode !== toCode) break;
        else if (fromCode === 92) lastCommonSep = i1;
    }
    if (i1 !== length && lastCommonSep === -1) {
        return toOrig;
    }
    let out = "";
    if (lastCommonSep === -1) lastCommonSep = 0;
    for(i1 = fromStart + lastCommonSep + 1; i1 <= fromEnd; ++i1){
        if (i1 === fromEnd || from.charCodeAt(i1) === 92) {
            if (out.length === 0) out += "..";
            else out += "\\..";
        }
    }
    if (out.length > 0) {
        return out + toOrig.slice(toStart + lastCommonSep, toEnd);
    } else {
        toStart += lastCommonSep;
        if (toOrig.charCodeAt(toStart) === 92) ++toStart;
        return toOrig.slice(toStart, toEnd);
    }
}
const WHITESPACE_ENCODINGS = {
    "\u0009": "%09",
    "\u000A": "%0A",
    "\u000B": "%0B",
    "\u000C": "%0C",
    "\u000D": "%0D",
    "\u0020": "%20"
};
function encodeWhitespace(string) {
    return string.replaceAll(/[\s]/g, (c)=>{
        return WHITESPACE_ENCODINGS[c] ?? c;
    });
}
function toFileUrl(path) {
    if (!isAbsolute(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const [, hostname, pathname] = path.match(/^(?:[/\\]{2}([^/\\]+)(?=[/\\](?:[^/\\]|$)))?(.*)/);
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(pathname.replace(/%/g, "%25"));
    if (hostname !== undefined && hostname !== "localhost") {
        url.hostname = hostname;
        if (!url.hostname) {
            throw new TypeError("Invalid hostname.");
        }
    }
    return url;
}
function _common(paths, sep) {
    const [first = "", ...remaining] = paths;
    if (first === "" || remaining.length === 0) {
        return first.substring(0, first.lastIndexOf(sep) + 1);
    }
    const parts = first.split(sep);
    let endOfPrefix = parts.length;
    for (const path of remaining){
        const compare = path.split(sep);
        for(let i1 = 0; i1 < endOfPrefix; i1++){
            if (compare[i1] !== parts[i1]) {
                endOfPrefix = i1;
            }
        }
        if (endOfPrefix === 0) {
            return "";
        }
    }
    const prefix = parts.slice(0, endOfPrefix).join(sep);
    return prefix.endsWith(sep) ? prefix : `${prefix}${sep}`;
}
const regExpEscapeChars = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|"
];
const rangeEscapeChars = [
    "-",
    "\\",
    "]"
];
function _globToRegExp(c, glob, { extended = true, globstar: globstarOption = true, caseInsensitive = false } = {}) {
    if (glob === "") {
        return /(?!)/;
    }
    let newLength = glob.length;
    for(; newLength > 1 && c.seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i1 = j;
        for(; i1 < glob.length && !c.seps.includes(glob[i1]); i1++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars : regExpEscapeChars;
                segment += escapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
                continue;
            }
            if (glob[i1] === c.escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i1] === "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i1 + 1] === "!") {
                        i1++;
                        segment += "^";
                    } else if (glob[i1 + 1] === "^") {
                        i1++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i1 + 1] === ":") {
                    let k = i1 + 1;
                    let value = "";
                    while(glob[k + 1] !== undefined && glob[k + 1] !== ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] === ":" && glob[k + 2] === "]") {
                        i1 = k + 2;
                        if (value === "alnum") segment += "\\dA-Za-z";
                        else if (value === "alpha") segment += "A-Za-z";
                        else if (value === "ascii") segment += "\x00-\x7F";
                        else if (value === "blank") segment += "\t ";
                        else if (value === "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value === "digit") segment += "\\d";
                        else if (value === "graph") segment += "\x21-\x7E";
                        else if (value === "lower") segment += "a-z";
                        else if (value === "print") segment += "\x20-\x7E";
                        else if (value === "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value === "space") segment += "\\s\v";
                        else if (value === "upper") segment += "A-Z";
                        else if (value === "word") segment += "\\w";
                        else if (value === "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i1] === "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i1] === "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i1];
                }
                continue;
            }
            if (glob[i1] === ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] !== "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type === "!") {
                    segment += c.wildcard;
                } else if (type !== "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i1] === "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] !== "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] === "+" && extended && glob[i1 + 1] === "(") {
                i1++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i1] === "@" && extended && glob[i1 + 1] === "(") {
                i1++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i1] === "?") {
                if (extended && glob[i1 + 1] === "(") {
                    i1++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i1] === "!" && extended && glob[i1 + 1] === "(") {
                i1++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i1] === "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i1] === "}" && groupStack[groupStack.length - 1] === "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i1] === "," && groupStack[groupStack.length - 1] === "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] === "*") {
                if (extended && glob[i1 + 1] === "(") {
                    i1++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i1 - 1];
                    let numStars = 1;
                    while(glob[i1 + 1] === "*"){
                        i1++;
                        numStars++;
                    }
                    const nextChar = glob[i1 + 1];
                    if (globstarOption && numStars === 2 && [
                        ...c.seps,
                        undefined
                    ].includes(prevChar) && [
                        ...c.seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += c.globstar;
                        endsWithSep = true;
                    } else {
                        segment += c.wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i1)){
                segment += regExpEscapeChars.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i1 < glob.length ? c.sep : c.sepMaybe;
            endsWithSep = true;
        }
        while(c.seps.includes(glob[i1]))i1++;
        if (!(i1 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i1;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
const constants = {
    sep: "(?:\\\\|/)+",
    sepMaybe: "(?:\\\\|/)*",
    seps: [
        "\\",
        "/"
    ],
    globstar: "(?:[^\\\\/]*(?:\\\\|/|$)+)*",
    wildcard: "[^\\\\/]*",
    escapePrefix: "`"
};
function globToRegExp(glob, options = {}) {
    return _globToRegExp(constants, glob, options);
}
function isGlob(str) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str = str.slice(idx);
    }
    return false;
}
function normalizeGlob(glob, { globstar = false } = {}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize(glob);
    }
    const s = SEPARATOR_PATTERN.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs(globs, { extended = true, globstar = false } = {}) {
    if (!globstar || globs.length === 0) {
        return join(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path = glob;
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `${SEPARATOR}${path}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob(joined, {
        extended,
        globstar
    });
}
function isPosixPathSeparator1(code) {
    return code === 47;
}
function basename1(path, suffix = "") {
    assertArgs(path, suffix);
    const lastSegment = lastPathSegment(path, isPosixPathSeparator1);
    const strippedSegment = stripTrailingSeparators(lastSegment, isPosixPathSeparator1);
    return suffix ? stripSuffix(strippedSegment, suffix) : strippedSegment;
}
const DELIMITER = ":";
const SEPARATOR1 = "/";
const SEPARATOR_PATTERN1 = /\/+/;
function dirname1(path) {
    assertArg(path);
    let end = -1;
    let matchedNonSeparator = false;
    for(let i1 = path.length - 1; i1 >= 1; --i1){
        if (isPosixPathSeparator1(path.charCodeAt(i1))) {
            if (matchedNonSeparator) {
                end = i1;
                break;
            }
        } else {
            matchedNonSeparator = true;
        }
    }
    if (end === -1) {
        return isPosixPathSeparator1(path.charCodeAt(0)) ? "/" : ".";
    }
    return stripTrailingSeparators(path.slice(0, end), isPosixPathSeparator1);
}
function extname1(path) {
    assertPath(path);
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let preDotState = 0;
    for(let i1 = path.length - 1; i1 >= 0; --i1){
        const code = path.charCodeAt(i1);
        if (isPosixPathSeparator1(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        return "";
    }
    return path.slice(startDot, end);
}
function format(pathObject) {
    assertArg1(pathObject);
    return _format("/", pathObject);
}
function fromFileUrl1(url) {
    url = assertArg2(url);
    return decodeURIComponent(url.pathname.replace(/%(?![0-9A-Fa-f]{2})/g, "%25"));
}
function isAbsolute1(path) {
    assertPath(path);
    return path.length > 0 && isPosixPathSeparator1(path.charCodeAt(0));
}
function normalize1(path) {
    assertArg3(path);
    const isAbsolute = isPosixPathSeparator1(path.charCodeAt(0));
    const trailingSeparator = isPosixPathSeparator1(path.charCodeAt(path.length - 1));
    path = normalizeString(path, !isAbsolute, "/", isPosixPathSeparator1);
    if (path.length === 0 && !isAbsolute) path = ".";
    if (path.length > 0 && trailingSeparator) path += "/";
    if (isAbsolute) return `/${path}`;
    return path;
}
function join1(...paths) {
    if (paths.length === 0) return ".";
    let joined;
    for(let i1 = 0, len = paths.length; i1 < len; ++i1){
        const path = paths[i1];
        assertPath(path);
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `/${path}`;
        }
    }
    if (!joined) return ".";
    return normalize1(joined);
}
function parse(path) {
    assertPath(path);
    const ret = {
        root: "",
        dir: "",
        base: "",
        ext: "",
        name: ""
    };
    if (path.length === 0) return ret;
    const isAbsolute = isPosixPathSeparator1(path.charCodeAt(0));
    let start;
    if (isAbsolute) {
        ret.root = "/";
        start = 1;
    } else {
        start = 0;
    }
    let startDot = -1;
    let startPart = 0;
    let end = -1;
    let matchedSlash = true;
    let i1 = path.length - 1;
    let preDotState = 0;
    for(; i1 >= start; --i1){
        const code = path.charCodeAt(i1);
        if (isPosixPathSeparator1(code)) {
            if (!matchedSlash) {
                startPart = i1 + 1;
                break;
            }
            continue;
        }
        if (end === -1) {
            matchedSlash = false;
            end = i1 + 1;
        }
        if (code === 46) {
            if (startDot === -1) startDot = i1;
            else if (preDotState !== 1) preDotState = 1;
        } else if (startDot !== -1) {
            preDotState = -1;
        }
    }
    if (startDot === -1 || end === -1 || preDotState === 0 || preDotState === 1 && startDot === end - 1 && startDot === startPart + 1) {
        if (end !== -1) {
            if (startPart === 0 && isAbsolute) {
                ret.base = ret.name = path.slice(1, end);
            } else {
                ret.base = ret.name = path.slice(startPart, end);
            }
        }
        ret.base = ret.base || "/";
    } else {
        if (startPart === 0 && isAbsolute) {
            ret.name = path.slice(1, startDot);
            ret.base = path.slice(1, end);
        } else {
            ret.name = path.slice(startPart, startDot);
            ret.base = path.slice(startPart, end);
        }
        ret.ext = path.slice(startDot, end);
    }
    if (startPart > 0) {
        ret.dir = stripTrailingSeparators(path.slice(0, startPart - 1), isPosixPathSeparator1);
    } else if (isAbsolute) ret.dir = "/";
    return ret;
}
function resolve1(...pathSegments) {
    let resolvedPath = "";
    let resolvedAbsolute = false;
    for(let i1 = pathSegments.length - 1; i1 >= -1 && !resolvedAbsolute; i1--){
        let path;
        if (i1 >= 0) path = pathSegments[i1];
        else {
            const { Deno: Deno1 } = globalThis;
            if (typeof Deno1?.cwd !== "function") {
                throw new TypeError("Resolved a relative path without a CWD.");
            }
            path = Deno1.cwd();
        }
        assertPath(path);
        if (path.length === 0) {
            continue;
        }
        resolvedPath = `${path}/${resolvedPath}`;
        resolvedAbsolute = isPosixPathSeparator1(path.charCodeAt(0));
    }
    resolvedPath = normalizeString(resolvedPath, !resolvedAbsolute, "/", isPosixPathSeparator1);
    if (resolvedAbsolute) {
        if (resolvedPath.length > 0) return `/${resolvedPath}`;
        else return "/";
    } else if (resolvedPath.length > 0) return resolvedPath;
    else return ".";
}
function relative1(from, to) {
    assertArgs1(from, to);
    from = resolve1(from);
    to = resolve1(to);
    if (from === to) return "";
    let fromStart = 1;
    const fromEnd = from.length;
    for(; fromStart < fromEnd; ++fromStart){
        if (!isPosixPathSeparator1(from.charCodeAt(fromStart))) break;
    }
    const fromLen = fromEnd - fromStart;
    let toStart = 1;
    const toEnd = to.length;
    for(; toStart < toEnd; ++toStart){
        if (!isPosixPathSeparator1(to.charCodeAt(toStart))) break;
    }
    const toLen = toEnd - toStart;
    const length = fromLen < toLen ? fromLen : toLen;
    let lastCommonSep = -1;
    let i1 = 0;
    for(; i1 <= length; ++i1){
        if (i1 === length) {
            if (toLen > length) {
                if (isPosixPathSeparator1(to.charCodeAt(toStart + i1))) {
                    return to.slice(toStart + i1 + 1);
                } else if (i1 === 0) {
                    return to.slice(toStart + i1);
                }
            } else if (fromLen > length) {
                if (isPosixPathSeparator1(from.charCodeAt(fromStart + i1))) {
                    lastCommonSep = i1;
                } else if (i1 === 0) {
                    lastCommonSep = 0;
                }
            }
            break;
        }
        const fromCode = from.charCodeAt(fromStart + i1);
        const toCode = to.charCodeAt(toStart + i1);
        if (fromCode !== toCode) break;
        else if (isPosixPathSeparator1(fromCode)) lastCommonSep = i1;
    }
    let out = "";
    for(i1 = fromStart + lastCommonSep + 1; i1 <= fromEnd; ++i1){
        if (i1 === fromEnd || isPosixPathSeparator1(from.charCodeAt(i1))) {
            if (out.length === 0) out += "..";
            else out += "/..";
        }
    }
    if (out.length > 0) return out + to.slice(toStart + lastCommonSep);
    else {
        toStart += lastCommonSep;
        if (isPosixPathSeparator1(to.charCodeAt(toStart))) ++toStart;
        return to.slice(toStart);
    }
}
function toFileUrl1(path) {
    if (!isAbsolute1(path)) {
        throw new TypeError("Must be an absolute path.");
    }
    const url = new URL("file:///");
    url.pathname = encodeWhitespace(path.replace(/%/g, "%25").replace(/\\/g, "%5C"));
    return url;
}
function toNamespacedPath(path) {
    return path;
}
function common(paths, sep = SEPARATOR1) {
    return _common(paths, sep);
}
const constants1 = {
    sep: "/+",
    sepMaybe: "/*",
    seps: [
        "/"
    ],
    globstar: "(?:[^/]*(?:/|$)+)*",
    wildcard: "[^/]*",
    escapePrefix: "\\"
};
function globToRegExp1(glob, options = {}) {
    return _globToRegExp(constants1, glob, options);
}
function normalizeGlob1(glob, { globstar = false } = {}) {
    if (glob.match(/\0/g)) {
        throw new Error(`Glob contains invalid characters: "${glob}"`);
    }
    if (!globstar) {
        return normalize1(glob);
    }
    const s = SEPARATOR_PATTERN1.source;
    const badParentPattern = new RegExp(`(?<=(${s}|^)\\*\\*${s})\\.\\.(?=${s}|$)`, "g");
    return normalize1(glob.replace(badParentPattern, "\0")).replace(/\0/g, "..");
}
function joinGlobs1(globs, { extended = true, globstar = false } = {}) {
    if (!globstar || globs.length === 0) {
        return join1(...globs);
    }
    if (globs.length === 0) return ".";
    let joined;
    for (const glob of globs){
        const path = glob;
        if (path.length > 0) {
            if (!joined) joined = path;
            else joined += `${SEPARATOR1}${path}`;
        }
    }
    if (!joined) return ".";
    return normalizeGlob1(joined, {
        extended,
        globstar
    });
}
const mod = {
    basename: basename1,
    DELIMITER,
    SEPARATOR: SEPARATOR1,
    SEPARATOR_PATTERN: SEPARATOR_PATTERN1,
    dirname: dirname1,
    extname: extname1,
    format,
    fromFileUrl: fromFileUrl1,
    isAbsolute: isAbsolute1,
    join: join1,
    normalize: normalize1,
    parse,
    relative: relative1,
    resolve: resolve1,
    toFileUrl: toFileUrl1,
    toNamespacedPath,
    common,
    globToRegExp: globToRegExp1,
    isGlob,
    joinGlobs: joinGlobs1,
    normalizeGlob: normalizeGlob1
};
const osType = (()=>{
    const { Deno: Deno1 } = globalThis;
    if (typeof Deno1?.build?.os === "string") {
        return Deno1.build.os;
    }
    const { navigator } = globalThis;
    if (navigator?.appVersion?.includes?.("Win")) {
        return "windows";
    }
    return "linux";
})();
const isWindows = osType === "windows";
function basename2(path, suffix = "") {
    return isWindows ? basename(path, suffix) : basename1(path, suffix);
}
const SEPARATOR2 = isWindows ? "\\" : "/";
const SEPARATOR_PATTERN2 = isWindows ? /[\\/]+/ : /\/+/;
function dirname2(path) {
    return isWindows ? dirname(path) : dirname1(path);
}
function extname2(path) {
    return isWindows ? extname(path) : extname1(path);
}
function fromFileUrl2(url) {
    return isWindows ? fromFileUrl(url) : fromFileUrl1(url);
}
function isAbsolute2(path) {
    return isWindows ? isAbsolute(path) : isAbsolute1(path);
}
function join2(...paths) {
    return isWindows ? join(...paths) : join1(...paths);
}
function normalize2(path) {
    return isWindows ? normalize(path) : normalize1(path);
}
function relative2(from, to) {
    return isWindows ? relative(from, to) : relative1(from, to);
}
function resolve2(...pathSegments) {
    return isWindows ? resolve(...pathSegments) : resolve1(...pathSegments);
}
function toFileUrl2(path) {
    return isWindows ? toFileUrl(path) : toFileUrl1(path);
}
function globToRegExp2(glob, options = {}) {
    return options.os === "windows" || !options.os && isWindows ? globToRegExp(glob, options) : globToRegExp1(glob, options);
}
function joinGlobs2(globs, options = {}) {
    return isWindows ? joinGlobs(globs, options) : joinGlobs1(globs, options);
}
function toPathString(pathUrl) {
    return pathUrl instanceof URL ? fromFileUrl2(pathUrl) : pathUrl;
}
function createWalkEntrySync(path) {
    path = toPathString(path);
    path = normalize2(path);
    const name = basename2(path);
    const info = Deno.statSync(path);
    return {
        path,
        name,
        isFile: info.isFile,
        isDirectory: info.isDirectory,
        isSymlink: info.isSymlink
    };
}
class WalkError extends Error {
    root;
    constructor(cause, root){
        super(`${cause instanceof Error ? cause.message : cause} for path "${root}"`);
        this.cause = cause;
        this.name = "WalkError";
        this.root = root;
    }
}
function include(path, exts, match, skip) {
    if (exts && !exts.some((ext)=>path.endsWith(ext))) {
        return false;
    }
    if (match && !match.some((pattern)=>!!path.match(pattern))) {
        return false;
    }
    if (skip && skip.some((pattern)=>!!path.match(pattern))) {
        return false;
    }
    return true;
}
function wrapErrorWithPath(err, root) {
    if (err instanceof WalkError) return err;
    return new WalkError(err, root);
}
function* walkSync(root, { maxDepth = Infinity, includeFiles = true, includeDirs = true, includeSymlinks = true, followSymlinks = false, canonicalize = true, exts = undefined, match = undefined, skip = undefined } = {}) {
    root = toPathString(root);
    if (maxDepth < 0) {
        return;
    }
    if (includeDirs && include(root, exts, match, skip)) {
        yield createWalkEntrySync(root);
    }
    if (maxDepth < 1 || !include(root, undefined, undefined, skip)) {
        return;
    }
    let entries;
    try {
        entries = Deno.readDirSync(root);
    } catch (err) {
        throw wrapErrorWithPath(err, normalize2(root));
    }
    for (const entry of entries){
        let path = join2(root, entry.name);
        let { isSymlink, isDirectory } = entry;
        if (isSymlink) {
            if (!followSymlinks) {
                if (includeSymlinks && include(path, exts, match, skip)) {
                    yield {
                        path,
                        ...entry
                    };
                }
                continue;
            }
            const realPath = Deno.realPathSync(path);
            if (canonicalize) {
                path = realPath;
            }
            ({ isSymlink, isDirectory } = Deno.lstatSync(realPath));
        }
        if (isSymlink || isDirectory) {
            yield* walkSync(path, {
                maxDepth: maxDepth - 1,
                includeFiles,
                includeDirs,
                includeSymlinks,
                followSymlinks,
                exts,
                match,
                skip
            });
        } else if (includeFiles && include(path, exts, match, skip)) {
            yield {
                path,
                ...entry
            };
        }
    }
}
const isWindows1 = Deno.build.os === "windows";
function split(path) {
    const s = SEPARATOR_PATTERN2.source;
    const segments = path.replace(new RegExp(`^${s}|${s}$`, "g"), "").split(SEPARATOR_PATTERN2);
    const isAbsolute_ = isAbsolute2(path);
    return {
        segments,
        isAbsolute: isAbsolute_,
        hasTrailingSep: !!path.match(new RegExp(`${s}$`)),
        winRoot: isWindows1 && isAbsolute_ ? segments.shift() : undefined
    };
}
function throwUnlessNotFound(error) {
    if (!(error instanceof Deno.errors.NotFound)) {
        throw error;
    }
}
function comparePath(a, b) {
    if (a.path < b.path) return -1;
    if (a.path > b.path) return 1;
    return 0;
}
function* expandGlobSync(glob, { root, exclude = [], includeDirs = true, extended = true, globstar = true, caseInsensitive, followSymlinks, canonicalize } = {}) {
    const { segments, isAbsolute: isGlobAbsolute, hasTrailingSep, winRoot } = split(toPathString(glob));
    root ??= isGlobAbsolute ? winRoot ?? "/" : Deno.cwd();
    const globOptions = {
        extended,
        globstar,
        caseInsensitive
    };
    const absRoot = isGlobAbsolute ? root : resolve2(root);
    const resolveFromRoot = (path)=>resolve2(absRoot, path);
    const excludePatterns = exclude.map(resolveFromRoot).map((s)=>globToRegExp2(s, globOptions));
    const shouldInclude = (path)=>!excludePatterns.some((p)=>!!path.match(p));
    let fixedRoot = isGlobAbsolute ? winRoot !== undefined ? winRoot : "/" : absRoot;
    while(segments.length > 0 && !isGlob(segments[0])){
        const seg = segments.shift();
        assert(seg !== undefined);
        fixedRoot = joinGlobs2([
            fixedRoot,
            seg
        ], globOptions);
    }
    let fixedRootInfo;
    try {
        fixedRootInfo = createWalkEntrySync(fixedRoot);
    } catch (error) {
        return throwUnlessNotFound(error);
    }
    function* advanceMatch(walkInfo, globSegment) {
        if (!walkInfo.isDirectory) {
            return;
        } else if (globSegment === "..") {
            const parentPath = joinGlobs2([
                walkInfo.path,
                ".."
            ], globOptions);
            try {
                if (shouldInclude(parentPath)) {
                    return yield createWalkEntrySync(parentPath);
                }
            } catch (error) {
                throwUnlessNotFound(error);
            }
            return;
        } else if (globSegment === "**") {
            return yield* walkSync(walkInfo.path, {
                skip: excludePatterns,
                maxDepth: globstar ? Infinity : 1,
                followSymlinks,
                canonicalize
            });
        }
        const globPattern = globToRegExp2(globSegment, globOptions);
        for (const walkEntry of walkSync(walkInfo.path, {
            maxDepth: 1,
            skip: excludePatterns,
            followSymlinks
        })){
            if (walkEntry.path !== walkInfo.path && walkEntry.name.match(globPattern)) {
                yield walkEntry;
            }
        }
    }
    let currentMatches = [
        fixedRootInfo
    ];
    for (const segment of segments){
        const nextMatchMap = new Map();
        for (const currentMatch of currentMatches){
            for (const nextMatch of advanceMatch(currentMatch, segment)){
                nextMatchMap.set(nextMatch.path, nextMatch);
            }
        }
        currentMatches = [
            ...nextMatchMap.values()
        ].sort(comparePath);
    }
    if (hasTrailingSep) {
        currentMatches = currentMatches.filter((entry)=>entry.isDirectory);
    }
    if (!includeDirs) {
        currentMatches = currentMatches.filter((entry)=>!entry.isDirectory);
    }
    yield* currentMatches;
}
function listCacheClear() {
    this.__data__ = [];
    this.size = 0;
}
var __VIRTUAL_FILE = listCacheClear;
function eq(value, other) {
    return value === other || value !== value && other !== other;
}
var __VIRTUAL_FILE1 = eq;
function assocIndexOf(array, key) {
    var length = array.length;
    while(length--){
        if (__VIRTUAL_FILE1(array[length][0], key)) {
            return length;
        }
    }
    return -1;
}
var __VIRTUAL_FILE2 = assocIndexOf;
var arrayProto = Array.prototype;
var splice = arrayProto.splice;
function listCacheDelete(key) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    if (index < 0) {
        return false;
    }
    var lastIndex = data.length - 1;
    if (index == lastIndex) {
        data.pop();
    } else {
        splice.call(data, index, 1);
    }
    --this.size;
    return true;
}
var __VIRTUAL_FILE3 = listCacheDelete;
function listCacheGet(key) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    return index < 0 ? void 0 : data[index][1];
}
var __VIRTUAL_FILE4 = listCacheGet;
function listCacheHas(key) {
    return __VIRTUAL_FILE2(this.__data__, key) > -1;
}
var __VIRTUAL_FILE5 = listCacheHas;
function listCacheSet(key, value) {
    var data = this.__data__, index = __VIRTUAL_FILE2(data, key);
    if (index < 0) {
        ++this.size;
        data.push([
            key,
            value
        ]);
    } else {
        data[index][1] = value;
    }
    return this;
}
var __VIRTUAL_FILE6 = listCacheSet;
function ListCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
ListCache.prototype.clear = __VIRTUAL_FILE;
ListCache.prototype["delete"] = __VIRTUAL_FILE3;
ListCache.prototype.get = __VIRTUAL_FILE4;
ListCache.prototype.has = __VIRTUAL_FILE5;
ListCache.prototype.set = __VIRTUAL_FILE6;
var __VIRTUAL_FILE7 = ListCache;
function stackClear() {
    this.__data__ = new __VIRTUAL_FILE7();
    this.size = 0;
}
var __VIRTUAL_FILE8 = stackClear;
function stackDelete(key) {
    var data = this.__data__, result1 = data["delete"](key);
    this.size = data.size;
    return result1;
}
var __VIRTUAL_FILE9 = stackDelete;
function stackGet(key) {
    return this.__data__.get(key);
}
var __VIRTUAL_FILE10 = stackGet;
function stackHas(key) {
    return this.__data__.has(key);
}
var __VIRTUAL_FILE11 = stackHas;
var commonjsGlobal = typeof globalThis !== "undefined" ? globalThis : typeof window !== "undefined" ? window : typeof global !== "undefined" ? global : typeof self !== "undefined" ? self : {};
var freeGlobal = typeof commonjsGlobal == "object" && commonjsGlobal && commonjsGlobal.Object === Object && commonjsGlobal;
var __VIRTUAL_FILE12 = freeGlobal;
var freeSelf = typeof self == "object" && self && self.Object === Object && self;
var root = __VIRTUAL_FILE12 || freeSelf || Function("return this")();
var __VIRTUAL_FILE13 = root;
var Symbol1 = __VIRTUAL_FILE13.Symbol;
var __VIRTUAL_FILE14 = Symbol1;
var objectProto = Object.prototype;
var hasOwnProperty = objectProto.hasOwnProperty;
var nativeObjectToString = objectProto.toString;
var symToStringTag = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function getRawTag(value) {
    var isOwn = hasOwnProperty.call(value, symToStringTag), tag = value[symToStringTag];
    try {
        value[symToStringTag] = void 0;
        var unmasked = true;
    } catch (e) {}
    var result1 = nativeObjectToString.call(value);
    if (unmasked) {
        if (isOwn) {
            value[symToStringTag] = tag;
        } else {
            delete value[symToStringTag];
        }
    }
    return result1;
}
var __VIRTUAL_FILE15 = getRawTag;
var objectProto1 = Object.prototype;
var nativeObjectToString1 = objectProto1.toString;
function objectToString(value) {
    return nativeObjectToString1.call(value);
}
var __VIRTUAL_FILE16 = objectToString;
var nullTag = "[object Null]", undefinedTag = "[object Undefined]";
var symToStringTag1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.toStringTag : void 0;
function baseGetTag(value) {
    if (value == null) {
        return value === void 0 ? undefinedTag : nullTag;
    }
    return symToStringTag1 && symToStringTag1 in Object(value) ? __VIRTUAL_FILE15(value) : __VIRTUAL_FILE16(value);
}
var __VIRTUAL_FILE17 = baseGetTag;
function isObject(value) {
    var type = typeof value;
    return value != null && (type == "object" || type == "function");
}
var __VIRTUAL_FILE18 = isObject;
var asyncTag = "[object AsyncFunction]", funcTag = "[object Function]", genTag = "[object GeneratorFunction]", proxyTag = "[object Proxy]";
function isFunction(value) {
    if (!__VIRTUAL_FILE18(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == funcTag || tag == genTag || tag == asyncTag || tag == proxyTag;
}
var __VIRTUAL_FILE19 = isFunction;
var coreJsData = __VIRTUAL_FILE13["__core-js_shared__"];
var __VIRTUAL_FILE20 = coreJsData;
var maskSrcKey = function() {
    var uid = /[^.]+$/.exec(__VIRTUAL_FILE20 && __VIRTUAL_FILE20.keys && __VIRTUAL_FILE20.keys.IE_PROTO || "");
    return uid ? "Symbol(src)_1." + uid : "";
}();
function isMasked(func) {
    return !!maskSrcKey && maskSrcKey in func;
}
var __VIRTUAL_FILE21 = isMasked;
var funcProto = Function.prototype;
var funcToString = funcProto.toString;
function toSource(func) {
    if (func != null) {
        try {
            return funcToString.call(func);
        } catch (e) {}
        try {
            return func + "";
        } catch (e) {}
    }
    return "";
}
var __VIRTUAL_FILE22 = toSource;
var reRegExpChar = /[\\^$.*+?()[\]{}|]/g;
var reIsHostCtor = /^\[object .+?Constructor\]$/;
var funcProto1 = Function.prototype, objectProto2 = Object.prototype;
var funcToString1 = funcProto1.toString;
var hasOwnProperty1 = objectProto2.hasOwnProperty;
var reIsNative = RegExp("^" + funcToString1.call(hasOwnProperty1).replace(reRegExpChar, "\\$&").replace(/hasOwnProperty|(function).*?(?=\\\()| for .+?(?=\\\])/g, "$1.*?") + "$");
function baseIsNative(value) {
    if (!__VIRTUAL_FILE18(value) || __VIRTUAL_FILE21(value)) {
        return false;
    }
    var pattern = __VIRTUAL_FILE19(value) ? reIsNative : reIsHostCtor;
    return pattern.test(__VIRTUAL_FILE22(value));
}
var __VIRTUAL_FILE23 = baseIsNative;
function getValue(object, key) {
    return object == null ? void 0 : object[key];
}
var __VIRTUAL_FILE24 = getValue;
function getNative(object, key) {
    var value = __VIRTUAL_FILE24(object, key);
    return __VIRTUAL_FILE23(value) ? value : void 0;
}
var __VIRTUAL_FILE25 = getNative;
var Map1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Map");
var __VIRTUAL_FILE26 = Map1;
var nativeCreate = __VIRTUAL_FILE25(Object, "create");
var __VIRTUAL_FILE27 = nativeCreate;
function hashClear() {
    this.__data__ = __VIRTUAL_FILE27 ? __VIRTUAL_FILE27(null) : {};
    this.size = 0;
}
var __VIRTUAL_FILE28 = hashClear;
function hashDelete(key) {
    var result1 = this.has(key) && delete this.__data__[key];
    this.size -= result1 ? 1 : 0;
    return result1;
}
var __VIRTUAL_FILE29 = hashDelete;
var HASH_UNDEFINED = "__lodash_hash_undefined__";
var objectProto3 = Object.prototype;
var hasOwnProperty2 = objectProto3.hasOwnProperty;
function hashGet(key) {
    var data = this.__data__;
    if (__VIRTUAL_FILE27) {
        var result1 = data[key];
        return result1 === HASH_UNDEFINED ? void 0 : result1;
    }
    return hasOwnProperty2.call(data, key) ? data[key] : void 0;
}
var __VIRTUAL_FILE30 = hashGet;
var objectProto4 = Object.prototype;
var hasOwnProperty3 = objectProto4.hasOwnProperty;
function hashHas(key) {
    var data = this.__data__;
    return __VIRTUAL_FILE27 ? data[key] !== void 0 : hasOwnProperty3.call(data, key);
}
var __VIRTUAL_FILE31 = hashHas;
var HASH_UNDEFINED1 = "__lodash_hash_undefined__";
function hashSet(key, value) {
    var data = this.__data__;
    this.size += this.has(key) ? 0 : 1;
    data[key] = __VIRTUAL_FILE27 && value === void 0 ? HASH_UNDEFINED1 : value;
    return this;
}
var __VIRTUAL_FILE32 = hashSet;
function Hash(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
Hash.prototype.clear = __VIRTUAL_FILE28;
Hash.prototype["delete"] = __VIRTUAL_FILE29;
Hash.prototype.get = __VIRTUAL_FILE30;
Hash.prototype.has = __VIRTUAL_FILE31;
Hash.prototype.set = __VIRTUAL_FILE32;
var __VIRTUAL_FILE33 = Hash;
function mapCacheClear() {
    this.size = 0;
    this.__data__ = {
        hash: new __VIRTUAL_FILE33(),
        map: new (__VIRTUAL_FILE26 || __VIRTUAL_FILE7)(),
        string: new __VIRTUAL_FILE33()
    };
}
var __VIRTUAL_FILE34 = mapCacheClear;
function isKeyable(value) {
    var type = typeof value;
    return type == "string" || type == "number" || type == "symbol" || type == "boolean" ? value !== "__proto__" : value === null;
}
var __VIRTUAL_FILE35 = isKeyable;
function getMapData(map, key) {
    var data = map.__data__;
    return __VIRTUAL_FILE35(key) ? data[typeof key == "string" ? "string" : "hash"] : data.map;
}
var __VIRTUAL_FILE36 = getMapData;
function mapCacheDelete(key) {
    var result1 = __VIRTUAL_FILE36(this, key)["delete"](key);
    this.size -= result1 ? 1 : 0;
    return result1;
}
var __VIRTUAL_FILE37 = mapCacheDelete;
function mapCacheGet(key) {
    return __VIRTUAL_FILE36(this, key).get(key);
}
var __VIRTUAL_FILE38 = mapCacheGet;
function mapCacheHas(key) {
    return __VIRTUAL_FILE36(this, key).has(key);
}
var __VIRTUAL_FILE39 = mapCacheHas;
function mapCacheSet(key, value) {
    var data = __VIRTUAL_FILE36(this, key), size = data.size;
    data.set(key, value);
    this.size += data.size == size ? 0 : 1;
    return this;
}
var __VIRTUAL_FILE40 = mapCacheSet;
function MapCache(entries) {
    var index = -1, length = entries == null ? 0 : entries.length;
    this.clear();
    while(++index < length){
        var entry = entries[index];
        this.set(entry[0], entry[1]);
    }
}
MapCache.prototype.clear = __VIRTUAL_FILE34;
MapCache.prototype["delete"] = __VIRTUAL_FILE37;
MapCache.prototype.get = __VIRTUAL_FILE38;
MapCache.prototype.has = __VIRTUAL_FILE39;
MapCache.prototype.set = __VIRTUAL_FILE40;
var __VIRTUAL_FILE41 = MapCache;
var LARGE_ARRAY_SIZE = 200;
function stackSet(key, value) {
    var data = this.__data__;
    if (data instanceof __VIRTUAL_FILE7) {
        var pairs = data.__data__;
        if (!__VIRTUAL_FILE26 || pairs.length < LARGE_ARRAY_SIZE - 1) {
            pairs.push([
                key,
                value
            ]);
            this.size = ++data.size;
            return this;
        }
        data = this.__data__ = new __VIRTUAL_FILE41(pairs);
    }
    data.set(key, value);
    this.size = data.size;
    return this;
}
var __VIRTUAL_FILE42 = stackSet;
function Stack(entries) {
    var data = this.__data__ = new __VIRTUAL_FILE7(entries);
    this.size = data.size;
}
Stack.prototype.clear = __VIRTUAL_FILE8;
Stack.prototype["delete"] = __VIRTUAL_FILE9;
Stack.prototype.get = __VIRTUAL_FILE10;
Stack.prototype.has = __VIRTUAL_FILE11;
Stack.prototype.set = __VIRTUAL_FILE42;
var __VIRTUAL_FILE43 = Stack;
function arrayEach(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (iteratee(array[index], index, array) === false) {
            break;
        }
    }
    return array;
}
var __VIRTUAL_FILE44 = arrayEach;
var defineProperty = function() {
    try {
        var func = __VIRTUAL_FILE25(Object, "defineProperty");
        func({}, "", {});
        return func;
    } catch (e) {}
}();
var __VIRTUAL_FILE45 = defineProperty;
function baseAssignValue(object, key, value) {
    if (key == "__proto__" && __VIRTUAL_FILE45) {
        __VIRTUAL_FILE45(object, key, {
            configurable: true,
            enumerable: true,
            value,
            writable: true
        });
    } else {
        object[key] = value;
    }
}
var __VIRTUAL_FILE46 = baseAssignValue;
var objectProto5 = Object.prototype;
var hasOwnProperty4 = objectProto5.hasOwnProperty;
function assignValue(object, key, value) {
    var objValue = object[key];
    if (!(hasOwnProperty4.call(object, key) && __VIRTUAL_FILE1(objValue, value)) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE47 = assignValue;
function copyObject(source, props, object, customizer) {
    var isNew = !object;
    object || (object = {});
    var index = -1, length = props.length;
    while(++index < length){
        var key = props[index];
        var newValue = customizer ? customizer(object[key], source[key], key, object, source) : void 0;
        if (newValue === void 0) {
            newValue = source[key];
        }
        if (isNew) {
            __VIRTUAL_FILE46(object, key, newValue);
        } else {
            __VIRTUAL_FILE47(object, key, newValue);
        }
    }
    return object;
}
var __VIRTUAL_FILE48 = copyObject;
function baseTimes(n, iteratee) {
    var index = -1, result1 = Array(n);
    while(++index < n){
        result1[index] = iteratee(index);
    }
    return result1;
}
var __VIRTUAL_FILE49 = baseTimes;
function isObjectLike(value) {
    return value != null && typeof value == "object";
}
var __VIRTUAL_FILE50 = isObjectLike;
var argsTag = "[object Arguments]";
function baseIsArguments(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == argsTag;
}
var __VIRTUAL_FILE51 = baseIsArguments;
var objectProto6 = Object.prototype;
var hasOwnProperty5 = objectProto6.hasOwnProperty;
var propertyIsEnumerable = objectProto6.propertyIsEnumerable;
var isArguments = __VIRTUAL_FILE51(function() {
    return arguments;
}()) ? __VIRTUAL_FILE51 : function(value) {
    return __VIRTUAL_FILE50(value) && hasOwnProperty5.call(value, "callee") && !propertyIsEnumerable.call(value, "callee");
};
var __VIRTUAL_FILE52 = isArguments;
var isArray = Array.isArray;
var __VIRTUAL_FILE53 = isArray;
function stubFalse() {
    return false;
}
var __VIRTUAL_FILE54 = stubFalse;
function createCommonjsModule(fn) {
    var module1 = {
        exports: {}
    };
    return fn(module1, module1.exports), module1.exports;
}
var __VIRTUAL_FILE55 = createCommonjsModule(function(module1, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module1 && !module1.nodeType && module1;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0;
    var nativeIsBuffer = Buffer1 ? Buffer1.isBuffer : void 0;
    var isBuffer = nativeIsBuffer || __VIRTUAL_FILE54;
    module1.exports = isBuffer;
});
var MAX_SAFE_INTEGER = 9007199254740991;
var reIsUint = /^(?:0|[1-9]\d*)$/;
function isIndex(value, length) {
    var type = typeof value;
    length = length == null ? MAX_SAFE_INTEGER : length;
    return !!length && (type == "number" || type != "symbol" && reIsUint.test(value)) && value > -1 && value % 1 == 0 && value < length;
}
var __VIRTUAL_FILE56 = isIndex;
var MAX_SAFE_INTEGER1 = 9007199254740991;
function isLength(value) {
    return typeof value == "number" && value > -1 && value % 1 == 0 && value <= MAX_SAFE_INTEGER1;
}
var __VIRTUAL_FILE57 = isLength;
var argsTag1 = "[object Arguments]", arrayTag = "[object Array]", boolTag = "[object Boolean]", dateTag = "[object Date]", errorTag = "[object Error]", funcTag1 = "[object Function]", mapTag = "[object Map]", numberTag = "[object Number]", objectTag = "[object Object]", regexpTag = "[object RegExp]", setTag = "[object Set]", stringTag = "[object String]", weakMapTag = "[object WeakMap]";
var arrayBufferTag = "[object ArrayBuffer]", dataViewTag = "[object DataView]", float32Tag = "[object Float32Array]", float64Tag = "[object Float64Array]", int8Tag = "[object Int8Array]", int16Tag = "[object Int16Array]", int32Tag = "[object Int32Array]", uint8Tag = "[object Uint8Array]", uint8ClampedTag = "[object Uint8ClampedArray]", uint16Tag = "[object Uint16Array]", uint32Tag = "[object Uint32Array]";
var typedArrayTags = {};
typedArrayTags[float32Tag] = typedArrayTags[float64Tag] = typedArrayTags[int8Tag] = typedArrayTags[int16Tag] = typedArrayTags[int32Tag] = typedArrayTags[uint8Tag] = typedArrayTags[uint8ClampedTag] = typedArrayTags[uint16Tag] = typedArrayTags[uint32Tag] = true;
typedArrayTags[argsTag1] = typedArrayTags[arrayTag] = typedArrayTags[arrayBufferTag] = typedArrayTags[boolTag] = typedArrayTags[dataViewTag] = typedArrayTags[dateTag] = typedArrayTags[errorTag] = typedArrayTags[funcTag1] = typedArrayTags[mapTag] = typedArrayTags[numberTag] = typedArrayTags[objectTag] = typedArrayTags[regexpTag] = typedArrayTags[setTag] = typedArrayTags[stringTag] = typedArrayTags[weakMapTag] = false;
function baseIsTypedArray(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE57(value.length) && !!typedArrayTags[__VIRTUAL_FILE17(value)];
}
var __VIRTUAL_FILE58 = baseIsTypedArray;
function baseUnary(func) {
    return function(value) {
        return func(value);
    };
}
var __VIRTUAL_FILE59 = baseUnary;
function createCommonjsModule1(fn) {
    var module1 = {
        exports: {}
    };
    return fn(module1, module1.exports), module1.exports;
}
var __VIRTUAL_FILE60 = createCommonjsModule1(function(module1, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module1 && !module1.nodeType && module1;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var freeProcess = moduleExports && __VIRTUAL_FILE12.process;
    var nodeUtil = function() {
        try {
            var types = freeModule && freeModule.require && freeModule.require("util").types;
            if (types) {
                return types;
            }
            return freeProcess && freeProcess.binding && freeProcess.binding("util");
        } catch (e) {}
    }();
    module1.exports = nodeUtil;
});
var nodeIsTypedArray = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isTypedArray;
var isTypedArray = nodeIsTypedArray ? __VIRTUAL_FILE59(nodeIsTypedArray) : __VIRTUAL_FILE58;
var __VIRTUAL_FILE61 = isTypedArray;
var objectProto7 = Object.prototype;
var hasOwnProperty6 = objectProto7.hasOwnProperty;
function arrayLikeKeys(value, inherited) {
    var isArr = __VIRTUAL_FILE53(value), isArg = !isArr && __VIRTUAL_FILE52(value), isBuff = !isArr && !isArg && __VIRTUAL_FILE55(value), isType = !isArr && !isArg && !isBuff && __VIRTUAL_FILE61(value), skipIndexes = isArr || isArg || isBuff || isType, result1 = skipIndexes ? __VIRTUAL_FILE49(value.length, String) : [], length = result1.length;
    for(var key in value){
        if ((inherited || hasOwnProperty6.call(value, key)) && !(skipIndexes && (key == "length" || isBuff && (key == "offset" || key == "parent") || isType && (key == "buffer" || key == "byteLength" || key == "byteOffset") || __VIRTUAL_FILE56(key, length)))) {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE62 = arrayLikeKeys;
var objectProto8 = Object.prototype;
function isPrototype(value) {
    var Ctor = value && value.constructor, proto = typeof Ctor == "function" && Ctor.prototype || objectProto8;
    return value === proto;
}
var __VIRTUAL_FILE63 = isPrototype;
function overArg(func, transform) {
    return function(arg) {
        return func(transform(arg));
    };
}
var __VIRTUAL_FILE64 = overArg;
var nativeKeys = __VIRTUAL_FILE64(Object.keys, Object);
var __VIRTUAL_FILE65 = nativeKeys;
var objectProto9 = Object.prototype;
var hasOwnProperty7 = objectProto9.hasOwnProperty;
function baseKeys(object) {
    if (!__VIRTUAL_FILE63(object)) {
        return __VIRTUAL_FILE65(object);
    }
    var result1 = [];
    for(var key in Object(object)){
        if (hasOwnProperty7.call(object, key) && key != "constructor") {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE66 = baseKeys;
function isArrayLike(value) {
    return value != null && __VIRTUAL_FILE57(value.length) && !__VIRTUAL_FILE19(value);
}
var __VIRTUAL_FILE67 = isArrayLike;
function keys(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object) : __VIRTUAL_FILE66(object);
}
var __VIRTUAL_FILE68 = keys;
function baseAssign(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE68(source), object);
}
var __VIRTUAL_FILE69 = baseAssign;
function nativeKeysIn(object) {
    var result1 = [];
    if (object != null) {
        for(var key in Object(object)){
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE70 = nativeKeysIn;
var objectProto10 = Object.prototype;
var hasOwnProperty8 = objectProto10.hasOwnProperty;
function baseKeysIn(object) {
    if (!__VIRTUAL_FILE18(object)) {
        return __VIRTUAL_FILE70(object);
    }
    var isProto = __VIRTUAL_FILE63(object), result1 = [];
    for(var key in object){
        if (!(key == "constructor" && (isProto || !hasOwnProperty8.call(object, key)))) {
            result1.push(key);
        }
    }
    return result1;
}
var __VIRTUAL_FILE71 = baseKeysIn;
function keysIn(object) {
    return __VIRTUAL_FILE67(object) ? __VIRTUAL_FILE62(object, true) : __VIRTUAL_FILE71(object);
}
var __VIRTUAL_FILE72 = keysIn;
function baseAssignIn(object, source) {
    return object && __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object);
}
var __VIRTUAL_FILE73 = baseAssignIn;
function createCommonjsModule2(fn) {
    var module1 = {
        exports: {}
    };
    return fn(module1, module1.exports), module1.exports;
}
var __VIRTUAL_FILE74 = createCommonjsModule2(function(module1, exports) {
    var freeExports = exports && !exports.nodeType && exports;
    var freeModule = freeExports && true && module1 && !module1.nodeType && module1;
    var moduleExports = freeModule && freeModule.exports === freeExports;
    var Buffer1 = moduleExports ? __VIRTUAL_FILE13.Buffer : void 0, allocUnsafe = Buffer1 ? Buffer1.allocUnsafe : void 0;
    function cloneBuffer(buffer, isDeep) {
        if (isDeep) {
            return buffer.slice();
        }
        var length = buffer.length, result1 = allocUnsafe ? allocUnsafe(length) : new buffer.constructor(length);
        buffer.copy(result1);
        return result1;
    }
    module1.exports = cloneBuffer;
});
function copyArray(source, array) {
    var index = -1, length = source.length;
    array || (array = Array(length));
    while(++index < length){
        array[index] = source[index];
    }
    return array;
}
var __VIRTUAL_FILE75 = copyArray;
function arrayFilter(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length, resIndex = 0, result1 = [];
    while(++index < length){
        var value = array[index];
        if (predicate(value, index, array)) {
            result1[resIndex++] = value;
        }
    }
    return result1;
}
var __VIRTUAL_FILE76 = arrayFilter;
function stubArray() {
    return [];
}
var __VIRTUAL_FILE77 = stubArray;
var objectProto11 = Object.prototype;
var propertyIsEnumerable1 = objectProto11.propertyIsEnumerable;
var nativeGetSymbols = Object.getOwnPropertySymbols;
var getSymbols = !nativeGetSymbols ? __VIRTUAL_FILE77 : function(object) {
    if (object == null) {
        return [];
    }
    object = Object(object);
    return __VIRTUAL_FILE76(nativeGetSymbols(object), function(symbol) {
        return propertyIsEnumerable1.call(object, symbol);
    });
};
var __VIRTUAL_FILE78 = getSymbols;
function copySymbols(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE78(source), object);
}
var __VIRTUAL_FILE79 = copySymbols;
function arrayPush(array, values) {
    var index = -1, length = values.length, offset = array.length;
    while(++index < length){
        array[offset + index] = values[index];
    }
    return array;
}
var __VIRTUAL_FILE80 = arrayPush;
var getPrototype = __VIRTUAL_FILE64(Object.getPrototypeOf, Object);
var __VIRTUAL_FILE81 = getPrototype;
var nativeGetSymbols1 = Object.getOwnPropertySymbols;
var getSymbolsIn = !nativeGetSymbols1 ? __VIRTUAL_FILE77 : function(object) {
    var result1 = [];
    while(object){
        __VIRTUAL_FILE80(result1, __VIRTUAL_FILE78(object));
        object = __VIRTUAL_FILE81(object);
    }
    return result1;
};
var __VIRTUAL_FILE82 = getSymbolsIn;
function copySymbolsIn(source, object) {
    return __VIRTUAL_FILE48(source, __VIRTUAL_FILE82(source), object);
}
var __VIRTUAL_FILE83 = copySymbolsIn;
function baseGetAllKeys(object, keysFunc, symbolsFunc) {
    var result1 = keysFunc(object);
    return __VIRTUAL_FILE53(object) ? result1 : __VIRTUAL_FILE80(result1, symbolsFunc(object));
}
var __VIRTUAL_FILE84 = baseGetAllKeys;
function getAllKeys(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE68, __VIRTUAL_FILE78);
}
var __VIRTUAL_FILE85 = getAllKeys;
function getAllKeysIn(object) {
    return __VIRTUAL_FILE84(object, __VIRTUAL_FILE72, __VIRTUAL_FILE82);
}
var __VIRTUAL_FILE86 = getAllKeysIn;
var DataView1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "DataView");
var __VIRTUAL_FILE87 = DataView1;
var Promise1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Promise");
var __VIRTUAL_FILE88 = Promise1;
var Set1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "Set");
var __VIRTUAL_FILE89 = Set1;
var WeakMap1 = __VIRTUAL_FILE25(__VIRTUAL_FILE13, "WeakMap");
var __VIRTUAL_FILE90 = WeakMap1;
var mapTag1 = "[object Map]", objectTag1 = "[object Object]", promiseTag = "[object Promise]", setTag1 = "[object Set]", weakMapTag1 = "[object WeakMap]";
var dataViewTag1 = "[object DataView]";
var dataViewCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE87), mapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE26), promiseCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE88), setCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE89), weakMapCtorString = __VIRTUAL_FILE22(__VIRTUAL_FILE90);
var getTag = __VIRTUAL_FILE17;
if (__VIRTUAL_FILE87 && getTag(new __VIRTUAL_FILE87(new ArrayBuffer(1))) != dataViewTag1 || __VIRTUAL_FILE26 && getTag(new __VIRTUAL_FILE26()) != mapTag1 || __VIRTUAL_FILE88 && getTag(__VIRTUAL_FILE88.resolve()) != promiseTag || __VIRTUAL_FILE89 && getTag(new __VIRTUAL_FILE89()) != setTag1 || __VIRTUAL_FILE90 && getTag(new __VIRTUAL_FILE90()) != weakMapTag1) {
    getTag = function(value) {
        var result1 = __VIRTUAL_FILE17(value), Ctor = result1 == objectTag1 ? value.constructor : void 0, ctorString = Ctor ? __VIRTUAL_FILE22(Ctor) : "";
        if (ctorString) {
            switch(ctorString){
                case dataViewCtorString:
                    return dataViewTag1;
                case mapCtorString:
                    return mapTag1;
                case promiseCtorString:
                    return promiseTag;
                case setCtorString:
                    return setTag1;
                case weakMapCtorString:
                    return weakMapTag1;
            }
        }
        return result1;
    };
}
var __VIRTUAL_FILE91 = getTag;
var objectProto12 = Object.prototype;
var hasOwnProperty9 = objectProto12.hasOwnProperty;
function initCloneArray(array) {
    var length = array.length, result1 = new array.constructor(length);
    if (length && typeof array[0] == "string" && hasOwnProperty9.call(array, "index")) {
        result1.index = array.index;
        result1.input = array.input;
    }
    return result1;
}
var __VIRTUAL_FILE92 = initCloneArray;
var Uint8Array1 = __VIRTUAL_FILE13.Uint8Array;
var __VIRTUAL_FILE93 = Uint8Array1;
function cloneArrayBuffer(arrayBuffer) {
    var result1 = new arrayBuffer.constructor(arrayBuffer.byteLength);
    new __VIRTUAL_FILE93(result1).set(new __VIRTUAL_FILE93(arrayBuffer));
    return result1;
}
var __VIRTUAL_FILE94 = cloneArrayBuffer;
function cloneDataView(dataView, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(dataView.buffer) : dataView.buffer;
    return new dataView.constructor(buffer, dataView.byteOffset, dataView.byteLength);
}
var __VIRTUAL_FILE95 = cloneDataView;
var reFlags = /\w*$/;
function cloneRegExp(regexp) {
    var result1 = new regexp.constructor(regexp.source, reFlags.exec(regexp));
    result1.lastIndex = regexp.lastIndex;
    return result1;
}
var __VIRTUAL_FILE96 = cloneRegExp;
var symbolProto = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf = symbolProto ? symbolProto.valueOf : void 0;
function cloneSymbol(symbol) {
    return symbolValueOf ? Object(symbolValueOf.call(symbol)) : {};
}
var __VIRTUAL_FILE97 = cloneSymbol;
function cloneTypedArray(typedArray, isDeep) {
    var buffer = isDeep ? __VIRTUAL_FILE94(typedArray.buffer) : typedArray.buffer;
    return new typedArray.constructor(buffer, typedArray.byteOffset, typedArray.length);
}
var __VIRTUAL_FILE98 = cloneTypedArray;
var boolTag1 = "[object Boolean]", dateTag1 = "[object Date]", mapTag2 = "[object Map]", numberTag1 = "[object Number]", regexpTag1 = "[object RegExp]", setTag2 = "[object Set]", stringTag1 = "[object String]", symbolTag = "[object Symbol]";
var arrayBufferTag1 = "[object ArrayBuffer]", dataViewTag2 = "[object DataView]", float32Tag1 = "[object Float32Array]", float64Tag1 = "[object Float64Array]", int8Tag1 = "[object Int8Array]", int16Tag1 = "[object Int16Array]", int32Tag1 = "[object Int32Array]", uint8Tag1 = "[object Uint8Array]", uint8ClampedTag1 = "[object Uint8ClampedArray]", uint16Tag1 = "[object Uint16Array]", uint32Tag1 = "[object Uint32Array]";
function initCloneByTag(object, tag, isDeep) {
    var Ctor = object.constructor;
    switch(tag){
        case arrayBufferTag1:
            return __VIRTUAL_FILE94(object);
        case boolTag1:
        case dateTag1:
            return new Ctor(+object);
        case dataViewTag2:
            return __VIRTUAL_FILE95(object, isDeep);
        case float32Tag1:
        case float64Tag1:
        case int8Tag1:
        case int16Tag1:
        case int32Tag1:
        case uint8Tag1:
        case uint8ClampedTag1:
        case uint16Tag1:
        case uint32Tag1:
            return __VIRTUAL_FILE98(object, isDeep);
        case mapTag2:
            return new Ctor();
        case numberTag1:
        case stringTag1:
            return new Ctor(object);
        case regexpTag1:
            return __VIRTUAL_FILE96(object);
        case setTag2:
            return new Ctor();
        case symbolTag:
            return __VIRTUAL_FILE97(object);
    }
}
var __VIRTUAL_FILE99 = initCloneByTag;
var objectCreate = Object.create;
var baseCreate = function() {
    function object() {}
    return function(proto) {
        if (!__VIRTUAL_FILE18(proto)) {
            return {};
        }
        if (objectCreate) {
            return objectCreate(proto);
        }
        object.prototype = proto;
        var result1 = new object();
        object.prototype = void 0;
        return result1;
    };
}();
var __VIRTUAL_FILE100 = baseCreate;
function initCloneObject(object) {
    return typeof object.constructor == "function" && !__VIRTUAL_FILE63(object) ? __VIRTUAL_FILE100(__VIRTUAL_FILE81(object)) : {};
}
var __VIRTUAL_FILE101 = initCloneObject;
var mapTag3 = "[object Map]";
function baseIsMap(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == mapTag3;
}
var __VIRTUAL_FILE102 = baseIsMap;
var nodeIsMap = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isMap;
var isMap = nodeIsMap ? __VIRTUAL_FILE59(nodeIsMap) : __VIRTUAL_FILE102;
var __VIRTUAL_FILE103 = isMap;
var setTag3 = "[object Set]";
function baseIsSet(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE91(value) == setTag3;
}
var __VIRTUAL_FILE104 = baseIsSet;
var nodeIsSet = __VIRTUAL_FILE60 && __VIRTUAL_FILE60.isSet;
var isSet = nodeIsSet ? __VIRTUAL_FILE59(nodeIsSet) : __VIRTUAL_FILE104;
var __VIRTUAL_FILE105 = isSet;
var CLONE_DEEP_FLAG = 1, CLONE_FLAT_FLAG = 2, CLONE_SYMBOLS_FLAG = 4;
var argsTag2 = "[object Arguments]", arrayTag1 = "[object Array]", boolTag2 = "[object Boolean]", dateTag2 = "[object Date]", errorTag1 = "[object Error]", funcTag2 = "[object Function]", genTag1 = "[object GeneratorFunction]", mapTag4 = "[object Map]", numberTag2 = "[object Number]", objectTag2 = "[object Object]", regexpTag2 = "[object RegExp]", setTag4 = "[object Set]", stringTag2 = "[object String]", symbolTag1 = "[object Symbol]", weakMapTag2 = "[object WeakMap]";
var arrayBufferTag2 = "[object ArrayBuffer]", dataViewTag3 = "[object DataView]", float32Tag2 = "[object Float32Array]", float64Tag2 = "[object Float64Array]", int8Tag2 = "[object Int8Array]", int16Tag2 = "[object Int16Array]", int32Tag2 = "[object Int32Array]", uint8Tag2 = "[object Uint8Array]", uint8ClampedTag2 = "[object Uint8ClampedArray]", uint16Tag2 = "[object Uint16Array]", uint32Tag2 = "[object Uint32Array]";
var cloneableTags = {};
cloneableTags[argsTag2] = cloneableTags[arrayTag1] = cloneableTags[arrayBufferTag2] = cloneableTags[dataViewTag3] = cloneableTags[boolTag2] = cloneableTags[dateTag2] = cloneableTags[float32Tag2] = cloneableTags[float64Tag2] = cloneableTags[int8Tag2] = cloneableTags[int16Tag2] = cloneableTags[int32Tag2] = cloneableTags[mapTag4] = cloneableTags[numberTag2] = cloneableTags[objectTag2] = cloneableTags[regexpTag2] = cloneableTags[setTag4] = cloneableTags[stringTag2] = cloneableTags[symbolTag1] = cloneableTags[uint8Tag2] = cloneableTags[uint8ClampedTag2] = cloneableTags[uint16Tag2] = cloneableTags[uint32Tag2] = true;
cloneableTags[errorTag1] = cloneableTags[funcTag2] = cloneableTags[weakMapTag2] = false;
function baseClone(value, bitmask, customizer, key, object, stack) {
    var result1, isDeep = bitmask & CLONE_DEEP_FLAG, isFlat = bitmask & CLONE_FLAT_FLAG, isFull = bitmask & CLONE_SYMBOLS_FLAG;
    if (customizer) {
        result1 = object ? customizer(value, key, object, stack) : customizer(value);
    }
    if (result1 !== void 0) {
        return result1;
    }
    if (!__VIRTUAL_FILE18(value)) {
        return value;
    }
    var isArr = __VIRTUAL_FILE53(value);
    if (isArr) {
        result1 = __VIRTUAL_FILE92(value);
        if (!isDeep) {
            return __VIRTUAL_FILE75(value, result1);
        }
    } else {
        var tag = __VIRTUAL_FILE91(value), isFunc = tag == funcTag2 || tag == genTag1;
        if (__VIRTUAL_FILE55(value)) {
            return __VIRTUAL_FILE74(value, isDeep);
        }
        if (tag == objectTag2 || tag == argsTag2 || isFunc && !object) {
            result1 = isFlat || isFunc ? {} : __VIRTUAL_FILE101(value);
            if (!isDeep) {
                return isFlat ? __VIRTUAL_FILE83(value, __VIRTUAL_FILE73(result1, value)) : __VIRTUAL_FILE79(value, __VIRTUAL_FILE69(result1, value));
            }
        } else {
            if (!cloneableTags[tag]) {
                return object ? value : {};
            }
            result1 = __VIRTUAL_FILE99(value, tag, isDeep);
        }
    }
    stack || (stack = new __VIRTUAL_FILE43());
    var stacked = stack.get(value);
    if (stacked) {
        return stacked;
    }
    stack.set(value, result1);
    if (__VIRTUAL_FILE105(value)) {
        value.forEach(function(subValue) {
            result1.add(baseClone(subValue, bitmask, customizer, subValue, value, stack));
        });
    } else if (__VIRTUAL_FILE103(value)) {
        value.forEach(function(subValue, key2) {
            result1.set(key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
        });
    }
    var keysFunc = isFull ? isFlat ? __VIRTUAL_FILE86 : __VIRTUAL_FILE85 : isFlat ? __VIRTUAL_FILE72 : __VIRTUAL_FILE68;
    var props = isArr ? void 0 : keysFunc(value);
    __VIRTUAL_FILE44(props || value, function(subValue, key2) {
        if (props) {
            key2 = subValue;
            subValue = value[key2];
        }
        __VIRTUAL_FILE47(result1, key2, baseClone(subValue, bitmask, customizer, key2, value, stack));
    });
    return result1;
}
var __VIRTUAL_FILE106 = baseClone;
var CLONE_DEEP_FLAG1 = 1, CLONE_SYMBOLS_FLAG1 = 4;
function cloneDeep(value) {
    return __VIRTUAL_FILE106(value, CLONE_DEEP_FLAG1 | CLONE_SYMBOLS_FLAG1);
}
var __VIRTUAL_FILE107 = cloneDeep;
var now = function() {
    return __VIRTUAL_FILE13.Date.now();
};
var __VIRTUAL_FILE108 = now;
var reWhitespace = /\s/;
function trimmedEndIndex(string) {
    var index = string.length;
    while(index-- && reWhitespace.test(string.charAt(index))){}
    return index;
}
var __VIRTUAL_FILE109 = trimmedEndIndex;
var reTrimStart = /^\s+/;
function baseTrim(string) {
    return string ? string.slice(0, __VIRTUAL_FILE109(string) + 1).replace(reTrimStart, "") : string;
}
var __VIRTUAL_FILE110 = baseTrim;
var symbolTag2 = "[object Symbol]";
function isSymbol(value) {
    return typeof value == "symbol" || __VIRTUAL_FILE50(value) && __VIRTUAL_FILE17(value) == symbolTag2;
}
var __VIRTUAL_FILE111 = isSymbol;
var NAN = 0 / 0;
var reIsBadHex = /^[-+]0x[0-9a-f]+$/i;
var reIsBinary = /^0b[01]+$/i;
var reIsOctal = /^0o[0-7]+$/i;
var freeParseInt = parseInt;
function toNumber(value) {
    if (typeof value == "number") {
        return value;
    }
    if (__VIRTUAL_FILE111(value)) {
        return NAN;
    }
    if (__VIRTUAL_FILE18(value)) {
        var other = typeof value.valueOf == "function" ? value.valueOf() : value;
        value = __VIRTUAL_FILE18(other) ? other + "" : other;
    }
    if (typeof value != "string") {
        return value === 0 ? value : +value;
    }
    value = __VIRTUAL_FILE110(value);
    var isBinary = reIsBinary.test(value);
    return isBinary || reIsOctal.test(value) ? freeParseInt(value.slice(2), isBinary ? 2 : 8) : reIsBadHex.test(value) ? NAN : +value;
}
var __VIRTUAL_FILE112 = toNumber;
var FUNC_ERROR_TEXT = "Expected a function";
var nativeMax = Math.max, nativeMin = Math.min;
function debounce(func, wait, options) {
    var lastArgs, lastThis, maxWait, result1, timerId, lastCallTime, lastInvokeTime = 0, leading = false, maxing = false, trailing = true;
    if (typeof func != "function") {
        throw new TypeError(FUNC_ERROR_TEXT);
    }
    wait = __VIRTUAL_FILE112(wait) || 0;
    if (__VIRTUAL_FILE18(options)) {
        leading = !!options.leading;
        maxing = "maxWait" in options;
        maxWait = maxing ? nativeMax(__VIRTUAL_FILE112(options.maxWait) || 0, wait) : maxWait;
        trailing = "trailing" in options ? !!options.trailing : trailing;
    }
    function invokeFunc(time) {
        var args = lastArgs, thisArg = lastThis;
        lastArgs = lastThis = void 0;
        lastInvokeTime = time;
        result1 = func.apply(thisArg, args);
        return result1;
    }
    function leadingEdge(time) {
        lastInvokeTime = time;
        timerId = setTimeout(timerExpired, wait);
        return leading ? invokeFunc(time) : result1;
    }
    function remainingWait(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime, timeWaiting = wait - timeSinceLastCall;
        return maxing ? nativeMin(timeWaiting, maxWait - timeSinceLastInvoke) : timeWaiting;
    }
    function shouldInvoke(time) {
        var timeSinceLastCall = time - lastCallTime, timeSinceLastInvoke = time - lastInvokeTime;
        return lastCallTime === void 0 || timeSinceLastCall >= wait || timeSinceLastCall < 0 || maxing && timeSinceLastInvoke >= maxWait;
    }
    function timerExpired() {
        var time = __VIRTUAL_FILE108();
        if (shouldInvoke(time)) {
            return trailingEdge(time);
        }
        timerId = setTimeout(timerExpired, remainingWait(time));
    }
    function trailingEdge(time) {
        timerId = void 0;
        if (trailing && lastArgs) {
            return invokeFunc(time);
        }
        lastArgs = lastThis = void 0;
        return result1;
    }
    function cancel() {
        if (timerId !== void 0) {
            clearTimeout(timerId);
        }
        lastInvokeTime = 0;
        lastArgs = lastCallTime = lastThis = timerId = void 0;
    }
    function flush() {
        return timerId === void 0 ? result1 : trailingEdge(__VIRTUAL_FILE108());
    }
    function debounced() {
        var time = __VIRTUAL_FILE108(), isInvoking = shouldInvoke(time);
        lastArgs = arguments;
        lastThis = this;
        lastCallTime = time;
        if (isInvoking) {
            if (timerId === void 0) {
                return leadingEdge(lastCallTime);
            }
            if (maxing) {
                clearTimeout(timerId);
                timerId = setTimeout(timerExpired, wait);
                return invokeFunc(lastCallTime);
            }
        }
        if (timerId === void 0) {
            timerId = setTimeout(timerExpired, wait);
        }
        return result1;
    }
    debounced.cancel = cancel;
    debounced.flush = flush;
    return debounced;
}
var __VIRTUAL_FILE113 = debounce;
var HASH_UNDEFINED2 = "__lodash_hash_undefined__";
function setCacheAdd(value) {
    this.__data__.set(value, HASH_UNDEFINED2);
    return this;
}
var __VIRTUAL_FILE114 = setCacheAdd;
function setCacheHas(value) {
    return this.__data__.has(value);
}
var __VIRTUAL_FILE115 = setCacheHas;
function SetCache(values) {
    var index = -1, length = values == null ? 0 : values.length;
    this.__data__ = new __VIRTUAL_FILE41();
    while(++index < length){
        this.add(values[index]);
    }
}
SetCache.prototype.add = SetCache.prototype.push = __VIRTUAL_FILE114;
SetCache.prototype.has = __VIRTUAL_FILE115;
var __VIRTUAL_FILE116 = SetCache;
function baseFindIndex(array, predicate, fromIndex, fromRight) {
    var length = array.length, index = fromIndex + (fromRight ? 1 : -1);
    while(fromRight ? index-- : ++index < length){
        if (predicate(array[index], index, array)) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE117 = baseFindIndex;
function baseIsNaN(value) {
    return value !== value;
}
var __VIRTUAL_FILE118 = baseIsNaN;
function strictIndexOf(array, value, fromIndex) {
    var index = fromIndex - 1, length = array.length;
    while(++index < length){
        if (array[index] === value) {
            return index;
        }
    }
    return -1;
}
var __VIRTUAL_FILE119 = strictIndexOf;
function baseIndexOf(array, value, fromIndex) {
    return value === value ? __VIRTUAL_FILE119(array, value, fromIndex) : __VIRTUAL_FILE117(array, __VIRTUAL_FILE118, fromIndex);
}
var __VIRTUAL_FILE120 = baseIndexOf;
function arrayIncludes(array, value) {
    var length = array == null ? 0 : array.length;
    return !!length && __VIRTUAL_FILE120(array, value, 0) > -1;
}
var __VIRTUAL_FILE121 = arrayIncludes;
function arrayIncludesWith(array, value, comparator) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (comparator(value, array[index])) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE122 = arrayIncludesWith;
function arrayMap(array, iteratee) {
    var index = -1, length = array == null ? 0 : array.length, result1 = Array(length);
    while(++index < length){
        result1[index] = iteratee(array[index], index, array);
    }
    return result1;
}
var __VIRTUAL_FILE123 = arrayMap;
function cacheHas(cache, key) {
    return cache.has(key);
}
var __VIRTUAL_FILE124 = cacheHas;
var LARGE_ARRAY_SIZE1 = 200;
function baseDifference(array, values, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, isCommon = true, length = array.length, result1 = [], valuesLength = values.length;
    if (!length) {
        return result1;
    }
    if (iteratee) {
        values = __VIRTUAL_FILE123(values, __VIRTUAL_FILE59(iteratee));
    }
    if (comparator) {
        includes = __VIRTUAL_FILE122;
        isCommon = false;
    } else if (values.length >= LARGE_ARRAY_SIZE1) {
        includes = __VIRTUAL_FILE124;
        isCommon = false;
        values = new __VIRTUAL_FILE116(values);
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee == null ? value : iteratee(value);
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var valuesIndex = valuesLength;
            while(valuesIndex--){
                if (values[valuesIndex] === computed) {
                    continue outer;
                }
            }
            result1.push(value);
        } else if (!includes(values, computed, comparator)) {
            result1.push(value);
        }
    }
    return result1;
}
var __VIRTUAL_FILE125 = baseDifference;
var spreadableSymbol = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.isConcatSpreadable : void 0;
function isFlattenable(value) {
    return __VIRTUAL_FILE53(value) || __VIRTUAL_FILE52(value) || !!(spreadableSymbol && value && value[spreadableSymbol]);
}
var __VIRTUAL_FILE126 = isFlattenable;
function baseFlatten(array, depth, predicate, isStrict, result1) {
    var index = -1, length = array.length;
    predicate || (predicate = __VIRTUAL_FILE126);
    result1 || (result1 = []);
    while(++index < length){
        var value = array[index];
        if (depth > 0 && predicate(value)) {
            if (depth > 1) {
                baseFlatten(value, depth - 1, predicate, isStrict, result1);
            } else {
                __VIRTUAL_FILE80(result1, value);
            }
        } else if (!isStrict) {
            result1[result1.length] = value;
        }
    }
    return result1;
}
var __VIRTUAL_FILE127 = baseFlatten;
function identity(value) {
    return value;
}
var __VIRTUAL_FILE128 = identity;
function apply(func, thisArg, args) {
    switch(args.length){
        case 0:
            return func.call(thisArg);
        case 1:
            return func.call(thisArg, args[0]);
        case 2:
            return func.call(thisArg, args[0], args[1]);
        case 3:
            return func.call(thisArg, args[0], args[1], args[2]);
    }
    return func.apply(thisArg, args);
}
var __VIRTUAL_FILE129 = apply;
var nativeMax1 = Math.max;
function overRest(func, start, transform) {
    start = nativeMax1(start === void 0 ? func.length - 1 : start, 0);
    return function() {
        var args = arguments, index = -1, length = nativeMax1(args.length - start, 0), array = Array(length);
        while(++index < length){
            array[index] = args[start + index];
        }
        index = -1;
        var otherArgs = Array(start + 1);
        while(++index < start){
            otherArgs[index] = args[index];
        }
        otherArgs[start] = transform(array);
        return __VIRTUAL_FILE129(func, this, otherArgs);
    };
}
var __VIRTUAL_FILE130 = overRest;
function constant(value) {
    return function() {
        return value;
    };
}
var __VIRTUAL_FILE131 = constant;
var baseSetToString = !__VIRTUAL_FILE45 ? __VIRTUAL_FILE128 : function(func, string) {
    return __VIRTUAL_FILE45(func, "toString", {
        configurable: true,
        enumerable: false,
        value: __VIRTUAL_FILE131(string),
        writable: true
    });
};
var __VIRTUAL_FILE132 = baseSetToString;
var HOT_COUNT = 800, HOT_SPAN = 16;
var nativeNow = Date.now;
function shortOut(func) {
    var count = 0, lastCalled = 0;
    return function() {
        var stamp = nativeNow(), remaining = HOT_SPAN - (stamp - lastCalled);
        lastCalled = stamp;
        if (remaining > 0) {
            if (++count >= HOT_COUNT) {
                return arguments[0];
            }
        } else {
            count = 0;
        }
        return func.apply(void 0, arguments);
    };
}
var __VIRTUAL_FILE133 = shortOut;
var setToString = __VIRTUAL_FILE133(__VIRTUAL_FILE132);
var __VIRTUAL_FILE134 = setToString;
function baseRest(func, start) {
    return __VIRTUAL_FILE134(__VIRTUAL_FILE130(func, start, __VIRTUAL_FILE128), func + "");
}
var __VIRTUAL_FILE135 = baseRest;
function isArrayLikeObject(value) {
    return __VIRTUAL_FILE50(value) && __VIRTUAL_FILE67(value);
}
var __VIRTUAL_FILE136 = isArrayLikeObject;
var difference = __VIRTUAL_FILE135(function(array, values) {
    return __VIRTUAL_FILE136(array) ? __VIRTUAL_FILE125(array, __VIRTUAL_FILE127(values, 1, __VIRTUAL_FILE136, true)) : [];
});
var __VIRTUAL_FILE137 = difference;
function createBaseFor(fromRight) {
    return function(object, iteratee, keysFunc) {
        var index = -1, iterable = Object(object), props = keysFunc(object), length = props.length;
        while(length--){
            var key = props[fromRight ? length : ++index];
            if (iteratee(iterable[key], key, iterable) === false) {
                break;
            }
        }
        return object;
    };
}
var __VIRTUAL_FILE138 = createBaseFor;
var baseFor = __VIRTUAL_FILE138();
var __VIRTUAL_FILE139 = baseFor;
function baseForOwn(object, iteratee) {
    return object && __VIRTUAL_FILE139(object, iteratee, __VIRTUAL_FILE68);
}
var __VIRTUAL_FILE140 = baseForOwn;
function createBaseEach(eachFunc, fromRight) {
    return function(collection, iteratee) {
        if (collection == null) {
            return collection;
        }
        if (!__VIRTUAL_FILE67(collection)) {
            return eachFunc(collection, iteratee);
        }
        var length = collection.length, index = fromRight ? length : -1, iterable = Object(collection);
        while(fromRight ? index-- : ++index < length){
            if (iteratee(iterable[index], index, iterable) === false) {
                break;
            }
        }
        return collection;
    };
}
var __VIRTUAL_FILE141 = createBaseEach;
var baseEach = __VIRTUAL_FILE141(__VIRTUAL_FILE140);
var __VIRTUAL_FILE142 = baseEach;
function castFunction(value) {
    return typeof value == "function" ? value : __VIRTUAL_FILE128;
}
var __VIRTUAL_FILE143 = castFunction;
function forEach(collection, iteratee) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE44 : __VIRTUAL_FILE142;
    return func(collection, __VIRTUAL_FILE143(iteratee));
}
var __VIRTUAL_FILE144 = forEach;
var __VIRTUAL_FILE145 = __VIRTUAL_FILE144;
function assignMergeValue(object, key, value) {
    if (value !== void 0 && !__VIRTUAL_FILE1(object[key], value) || value === void 0 && !(key in object)) {
        __VIRTUAL_FILE46(object, key, value);
    }
}
var __VIRTUAL_FILE146 = assignMergeValue;
var objectTag3 = "[object Object]";
var funcProto2 = Function.prototype, objectProto13 = Object.prototype;
var funcToString2 = funcProto2.toString;
var hasOwnProperty10 = objectProto13.hasOwnProperty;
var objectCtorString = funcToString2.call(Object);
function isPlainObject(value) {
    if (!__VIRTUAL_FILE50(value) || __VIRTUAL_FILE17(value) != objectTag3) {
        return false;
    }
    var proto = __VIRTUAL_FILE81(value);
    if (proto === null) {
        return true;
    }
    var Ctor = hasOwnProperty10.call(proto, "constructor") && proto.constructor;
    return typeof Ctor == "function" && Ctor instanceof Ctor && funcToString2.call(Ctor) == objectCtorString;
}
var __VIRTUAL_FILE147 = isPlainObject;
function safeGet(object, key) {
    if (key === "constructor" && typeof object[key] === "function") {
        return;
    }
    if (key == "__proto__") {
        return;
    }
    return object[key];
}
var __VIRTUAL_FILE148 = safeGet;
function toPlainObject(value) {
    return __VIRTUAL_FILE48(value, __VIRTUAL_FILE72(value));
}
var __VIRTUAL_FILE149 = toPlainObject;
function baseMergeDeep(object, source, key, srcIndex, mergeFunc, customizer, stack) {
    var objValue = __VIRTUAL_FILE148(object, key), srcValue = __VIRTUAL_FILE148(source, key), stacked = stack.get(srcValue);
    if (stacked) {
        __VIRTUAL_FILE146(object, key, stacked);
        return;
    }
    var newValue = customizer ? customizer(objValue, srcValue, key + "", object, source, stack) : void 0;
    var isCommon = newValue === void 0;
    if (isCommon) {
        var isArr = __VIRTUAL_FILE53(srcValue), isBuff = !isArr && __VIRTUAL_FILE55(srcValue), isTyped = !isArr && !isBuff && __VIRTUAL_FILE61(srcValue);
        newValue = srcValue;
        if (isArr || isBuff || isTyped) {
            if (__VIRTUAL_FILE53(objValue)) {
                newValue = objValue;
            } else if (__VIRTUAL_FILE136(objValue)) {
                newValue = __VIRTUAL_FILE75(objValue);
            } else if (isBuff) {
                isCommon = false;
                newValue = __VIRTUAL_FILE74(srcValue, true);
            } else if (isTyped) {
                isCommon = false;
                newValue = __VIRTUAL_FILE98(srcValue, true);
            } else {
                newValue = [];
            }
        } else if (__VIRTUAL_FILE147(srcValue) || __VIRTUAL_FILE52(srcValue)) {
            newValue = objValue;
            if (__VIRTUAL_FILE52(objValue)) {
                newValue = __VIRTUAL_FILE149(objValue);
            } else if (!__VIRTUAL_FILE18(objValue) || __VIRTUAL_FILE19(objValue)) {
                newValue = __VIRTUAL_FILE101(srcValue);
            }
        } else {
            isCommon = false;
        }
    }
    if (isCommon) {
        stack.set(srcValue, newValue);
        mergeFunc(newValue, srcValue, srcIndex, customizer, stack);
        stack["delete"](srcValue);
    }
    __VIRTUAL_FILE146(object, key, newValue);
}
var __VIRTUAL_FILE150 = baseMergeDeep;
function baseMerge(object, source, srcIndex, customizer, stack) {
    if (object === source) {
        return;
    }
    __VIRTUAL_FILE139(source, function(srcValue, key) {
        stack || (stack = new __VIRTUAL_FILE43());
        if (__VIRTUAL_FILE18(srcValue)) {
            __VIRTUAL_FILE150(object, source, key, srcIndex, baseMerge, customizer, stack);
        } else {
            var newValue = customizer ? customizer(__VIRTUAL_FILE148(object, key), srcValue, key + "", object, source, stack) : void 0;
            if (newValue === void 0) {
                newValue = srcValue;
            }
            __VIRTUAL_FILE146(object, key, newValue);
        }
    }, __VIRTUAL_FILE72);
}
var __VIRTUAL_FILE151 = baseMerge;
function isIterateeCall(value, index, object) {
    if (!__VIRTUAL_FILE18(object)) {
        return false;
    }
    var type = typeof index;
    if (type == "number" ? __VIRTUAL_FILE67(object) && __VIRTUAL_FILE56(index, object.length) : type == "string" && index in object) {
        return __VIRTUAL_FILE1(object[index], value);
    }
    return false;
}
var __VIRTUAL_FILE152 = isIterateeCall;
function createAssigner(assigner) {
    return __VIRTUAL_FILE135(function(object, sources) {
        var index = -1, length = sources.length, customizer = length > 1 ? sources[length - 1] : void 0, guard = length > 2 ? sources[2] : void 0;
        customizer = assigner.length > 3 && typeof customizer == "function" ? (length--, customizer) : void 0;
        if (guard && __VIRTUAL_FILE152(sources[0], sources[1], guard)) {
            customizer = length < 3 ? void 0 : customizer;
            length = 1;
        }
        object = Object(object);
        while(++index < length){
            var source = sources[index];
            if (source) {
                assigner(object, source, index, customizer);
            }
        }
        return object;
    });
}
var __VIRTUAL_FILE153 = createAssigner;
var mergeWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE151(object, source, srcIndex, customizer);
});
var __VIRTUAL_FILE154 = mergeWith;
var nativeFloor = Math.floor, nativeRandom = Math.random;
function baseRandom(lower, upper) {
    return lower + nativeFloor(nativeRandom() * (upper - lower + 1));
}
var __VIRTUAL_FILE155 = baseRandom;
function shuffleSelf(array, size) {
    var index = -1, length = array.length, lastIndex = length - 1;
    size = size === void 0 ? length : size;
    while(++index < size){
        var rand = __VIRTUAL_FILE155(index, lastIndex), value = array[rand];
        array[rand] = array[index];
        array[index] = value;
    }
    array.length = size;
    return array;
}
var __VIRTUAL_FILE156 = shuffleSelf;
function arrayShuffle(array) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE75(array));
}
var __VIRTUAL_FILE157 = arrayShuffle;
function baseValues(object, props) {
    return __VIRTUAL_FILE123(props, function(key) {
        return object[key];
    });
}
var __VIRTUAL_FILE158 = baseValues;
function values(object) {
    return object == null ? [] : __VIRTUAL_FILE158(object, __VIRTUAL_FILE68(object));
}
var __VIRTUAL_FILE159 = values;
function baseShuffle(collection) {
    return __VIRTUAL_FILE156(__VIRTUAL_FILE159(collection));
}
var __VIRTUAL_FILE160 = baseShuffle;
function shuffle(collection) {
    var func = __VIRTUAL_FILE53(collection) ? __VIRTUAL_FILE157 : __VIRTUAL_FILE160;
    return func(collection);
}
var __VIRTUAL_FILE161 = shuffle;
var assignInWith = __VIRTUAL_FILE153(function(object, source, srcIndex, customizer) {
    __VIRTUAL_FILE48(source, __VIRTUAL_FILE72(source), object, customizer);
});
var __VIRTUAL_FILE162 = assignInWith;
var domExcTag = "[object DOMException]", errorTag2 = "[object Error]";
function isError(value) {
    if (!__VIRTUAL_FILE50(value)) {
        return false;
    }
    var tag = __VIRTUAL_FILE17(value);
    return tag == errorTag2 || tag == domExcTag || typeof value.message == "string" && typeof value.name == "string" && !__VIRTUAL_FILE147(value);
}
var __VIRTUAL_FILE163 = isError;
var attempt = __VIRTUAL_FILE135(function(func, args) {
    try {
        return __VIRTUAL_FILE129(func, void 0, args);
    } catch (e) {
        return __VIRTUAL_FILE163(e) ? e : new Error(e);
    }
});
var __VIRTUAL_FILE164 = attempt;
var objectProto14 = Object.prototype;
var hasOwnProperty11 = objectProto14.hasOwnProperty;
function customDefaultsAssignIn(objValue, srcValue, key, object) {
    if (objValue === void 0 || __VIRTUAL_FILE1(objValue, objectProto14[key]) && !hasOwnProperty11.call(object, key)) {
        return srcValue;
    }
    return objValue;
}
var __VIRTUAL_FILE165 = customDefaultsAssignIn;
var stringEscapes = {
    "\\": "\\",
    "'": "'",
    "\n": "n",
    "\r": "r",
    "\u2028": "u2028",
    "\u2029": "u2029"
};
function escapeStringChar(chr) {
    return "\\" + stringEscapes[chr];
}
var __VIRTUAL_FILE166 = escapeStringChar;
var reInterpolate = /<%=([\s\S]+?)%>/g;
var __VIRTUAL_FILE167 = reInterpolate;
function basePropertyOf(object) {
    return function(key) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE168 = basePropertyOf;
var htmlEscapes = {
    "&": "&amp;",
    "<": "&lt;",
    ">": "&gt;",
    '"': "&quot;",
    "'": "&#39;"
};
var escapeHtmlChar = __VIRTUAL_FILE168(htmlEscapes);
var __VIRTUAL_FILE169 = escapeHtmlChar;
var INFINITY = 1 / 0;
var symbolProto1 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolToString = symbolProto1 ? symbolProto1.toString : void 0;
function baseToString(value) {
    if (typeof value == "string") {
        return value;
    }
    if (__VIRTUAL_FILE53(value)) {
        return __VIRTUAL_FILE123(value, baseToString) + "";
    }
    if (__VIRTUAL_FILE111(value)) {
        return symbolToString ? symbolToString.call(value) : "";
    }
    var result1 = value + "";
    return result1 == "0" && 1 / value == -INFINITY ? "-0" : result1;
}
var __VIRTUAL_FILE170 = baseToString;
function toString(value) {
    return value == null ? "" : __VIRTUAL_FILE170(value);
}
var __VIRTUAL_FILE171 = toString;
var reUnescapedHtml = /[&<>"']/g, reHasUnescapedHtml = RegExp(reUnescapedHtml.source);
function escape(string) {
    string = __VIRTUAL_FILE171(string);
    return string && reHasUnescapedHtml.test(string) ? string.replace(reUnescapedHtml, __VIRTUAL_FILE169) : string;
}
var __VIRTUAL_FILE172 = escape;
var reEscape = /<%-([\s\S]+?)%>/g;
var __VIRTUAL_FILE173 = reEscape;
var reEvaluate = /<%([\s\S]+?)%>/g;
var __VIRTUAL_FILE174 = reEvaluate;
var templateSettings = {
    escape: __VIRTUAL_FILE173,
    evaluate: __VIRTUAL_FILE174,
    interpolate: __VIRTUAL_FILE167,
    variable: "",
    imports: {
        _: {
            escape: __VIRTUAL_FILE172
        }
    }
};
var __VIRTUAL_FILE175 = templateSettings;
var INVALID_TEMPL_VAR_ERROR_TEXT = "Invalid `variable` option passed into `_.template`";
var reEmptyStringLeading = /\b__p \+= '';/g, reEmptyStringMiddle = /\b(__p \+=) '' \+/g, reEmptyStringTrailing = /(__e\(.*?\)|\b__t\)) \+\n'';/g;
var reForbiddenIdentifierChars = /[()=,{}\[\]\/\s]/;
var reEsTemplate = /\$\{([^\\}]*(?:\\.[^\\}]*)*)\}/g;
var reNoMatch = /($^)/;
var reUnescapedString = /['\n\r\u2028\u2029\\]/g;
var objectProto15 = Object.prototype;
var hasOwnProperty12 = objectProto15.hasOwnProperty;
function template(string, options, guard) {
    var settings = __VIRTUAL_FILE175.imports._.templateSettings || __VIRTUAL_FILE175;
    if (guard && __VIRTUAL_FILE152(string, options, guard)) {
        options = void 0;
    }
    string = __VIRTUAL_FILE171(string);
    options = __VIRTUAL_FILE162({}, options, settings, __VIRTUAL_FILE165);
    var imports = __VIRTUAL_FILE162({}, options.imports, settings.imports, __VIRTUAL_FILE165), importsKeys = __VIRTUAL_FILE68(imports), importsValues = __VIRTUAL_FILE158(imports, importsKeys);
    var isEscaping, isEvaluating, index = 0, interpolate = options.interpolate || reNoMatch, source = "__p += '";
    var reDelimiters = RegExp((options.escape || reNoMatch).source + "|" + interpolate.source + "|" + (interpolate === __VIRTUAL_FILE167 ? reEsTemplate : reNoMatch).source + "|" + (options.evaluate || reNoMatch).source + "|$", "g");
    var sourceURL = hasOwnProperty12.call(options, "sourceURL") ? "//# sourceURL=" + (options.sourceURL + "").replace(/\s/g, " ") + "\n" : "";
    string.replace(reDelimiters, function(match, escapeValue, interpolateValue, esTemplateValue, evaluateValue, offset) {
        interpolateValue || (interpolateValue = esTemplateValue);
        source += string.slice(index, offset).replace(reUnescapedString, __VIRTUAL_FILE166);
        if (escapeValue) {
            isEscaping = true;
            source += "' +\n__e(" + escapeValue + ") +\n'";
        }
        if (evaluateValue) {
            isEvaluating = true;
            source += "';\n" + evaluateValue + ";\n__p += '";
        }
        if (interpolateValue) {
            source += "' +\n((__t = (" + interpolateValue + ")) == null ? '' : __t) +\n'";
        }
        index = offset + match.length;
        return match;
    });
    source += "';\n";
    var variable = hasOwnProperty12.call(options, "variable") && options.variable;
    if (!variable) {
        source = "with (obj) {\n" + source + "\n}\n";
    } else if (reForbiddenIdentifierChars.test(variable)) {
        throw new Error(INVALID_TEMPL_VAR_ERROR_TEXT);
    }
    source = (isEvaluating ? source.replace(reEmptyStringLeading, "") : source).replace(reEmptyStringMiddle, "$1").replace(reEmptyStringTrailing, "$1;");
    source = "function(" + (variable || "obj") + ") {\n" + (variable ? "" : "obj || (obj = {});\n") + "var __t, __p = ''" + (isEscaping ? ", __e = _.escape" : "") + (isEvaluating ? ", __j = Array.prototype.join;\nfunction print() { __p += __j.call(arguments, '') }\n" : ";\n") + source + "return __p\n}";
    var result1 = __VIRTUAL_FILE164(function() {
        return Function(importsKeys, sourceURL + "return " + source).apply(void 0, importsValues);
    });
    result1.source = source;
    if (__VIRTUAL_FILE163(result1)) {
        throw result1;
    }
    return result1;
}
var __VIRTUAL_FILE176 = template;
function noop() {}
var __VIRTUAL_FILE177 = noop;
function setToArray(set) {
    var index = -1, result1 = Array(set.size);
    set.forEach(function(value) {
        result1[++index] = value;
    });
    return result1;
}
var __VIRTUAL_FILE178 = setToArray;
var INFINITY1 = 1 / 0;
var createSet = !(__VIRTUAL_FILE89 && 1 / __VIRTUAL_FILE178(new __VIRTUAL_FILE89([
    ,
    -0
]))[1] == INFINITY1) ? __VIRTUAL_FILE177 : function(values) {
    return new __VIRTUAL_FILE89(values);
};
var __VIRTUAL_FILE179 = createSet;
var LARGE_ARRAY_SIZE2 = 200;
function baseUniq(array, iteratee, comparator) {
    var index = -1, includes = __VIRTUAL_FILE121, length = array.length, isCommon = true, result1 = [], seen = result1;
    if (comparator) {
        isCommon = false;
        includes = __VIRTUAL_FILE122;
    } else if (length >= LARGE_ARRAY_SIZE2) {
        var set = iteratee ? null : __VIRTUAL_FILE179(array);
        if (set) {
            return __VIRTUAL_FILE178(set);
        }
        isCommon = false;
        includes = __VIRTUAL_FILE124;
        seen = new __VIRTUAL_FILE116();
    } else {
        seen = iteratee ? [] : result1;
    }
    outer: while(++index < length){
        var value = array[index], computed = iteratee ? iteratee(value) : value;
        value = comparator || value !== 0 ? value : 0;
        if (isCommon && computed === computed) {
            var seenIndex = seen.length;
            while(seenIndex--){
                if (seen[seenIndex] === computed) {
                    continue outer;
                }
            }
            if (iteratee) {
                seen.push(computed);
            }
            result1.push(value);
        } else if (!includes(seen, computed, comparator)) {
            if (seen !== result1) {
                seen.push(computed);
            }
            result1.push(value);
        }
    }
    return result1;
}
var __VIRTUAL_FILE180 = baseUniq;
function uniq(array) {
    return array && array.length ? __VIRTUAL_FILE180(array) : [];
}
var __VIRTUAL_FILE181 = uniq;
function arraySome(array, predicate) {
    var index = -1, length = array == null ? 0 : array.length;
    while(++index < length){
        if (predicate(array[index], index, array)) {
            return true;
        }
    }
    return false;
}
var __VIRTUAL_FILE182 = arraySome;
var COMPARE_PARTIAL_FLAG = 1, COMPARE_UNORDERED_FLAG = 2;
function equalArrays(array, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG, arrLength = array.length, othLength = other.length;
    if (arrLength != othLength && !(isPartial && othLength > arrLength)) {
        return false;
    }
    var arrStacked = stack.get(array);
    var othStacked = stack.get(other);
    if (arrStacked && othStacked) {
        return arrStacked == other && othStacked == array;
    }
    var index = -1, result1 = true, seen = bitmask & COMPARE_UNORDERED_FLAG ? new __VIRTUAL_FILE116() : void 0;
    stack.set(array, other);
    stack.set(other, array);
    while(++index < arrLength){
        var arrValue = array[index], othValue = other[index];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, arrValue, index, other, array, stack) : customizer(arrValue, othValue, index, array, other, stack);
        }
        if (compared !== void 0) {
            if (compared) {
                continue;
            }
            result1 = false;
            break;
        }
        if (seen) {
            if (!__VIRTUAL_FILE182(other, function(othValue2, othIndex) {
                if (!__VIRTUAL_FILE124(seen, othIndex) && (arrValue === othValue2 || equalFunc(arrValue, othValue2, bitmask, customizer, stack))) {
                    return seen.push(othIndex);
                }
            })) {
                result1 = false;
                break;
            }
        } else if (!(arrValue === othValue || equalFunc(arrValue, othValue, bitmask, customizer, stack))) {
            result1 = false;
            break;
        }
    }
    stack["delete"](array);
    stack["delete"](other);
    return result1;
}
var __VIRTUAL_FILE183 = equalArrays;
function mapToArray(map) {
    var index = -1, result1 = Array(map.size);
    map.forEach(function(value, key) {
        result1[++index] = [
            key,
            value
        ];
    });
    return result1;
}
var __VIRTUAL_FILE184 = mapToArray;
var COMPARE_PARTIAL_FLAG1 = 1, COMPARE_UNORDERED_FLAG1 = 2;
var boolTag3 = "[object Boolean]", dateTag3 = "[object Date]", errorTag3 = "[object Error]", mapTag5 = "[object Map]", numberTag3 = "[object Number]", regexpTag3 = "[object RegExp]", setTag5 = "[object Set]", stringTag3 = "[object String]", symbolTag3 = "[object Symbol]";
var arrayBufferTag3 = "[object ArrayBuffer]", dataViewTag4 = "[object DataView]";
var symbolProto2 = __VIRTUAL_FILE14 ? __VIRTUAL_FILE14.prototype : void 0, symbolValueOf1 = symbolProto2 ? symbolProto2.valueOf : void 0;
function equalByTag(object, other, tag, bitmask, customizer, equalFunc, stack) {
    switch(tag){
        case dataViewTag4:
            if (object.byteLength != other.byteLength || object.byteOffset != other.byteOffset) {
                return false;
            }
            object = object.buffer;
            other = other.buffer;
        case arrayBufferTag3:
            if (object.byteLength != other.byteLength || !equalFunc(new __VIRTUAL_FILE93(object), new __VIRTUAL_FILE93(other))) {
                return false;
            }
            return true;
        case boolTag3:
        case dateTag3:
        case numberTag3:
            return __VIRTUAL_FILE1(+object, +other);
        case errorTag3:
            return object.name == other.name && object.message == other.message;
        case regexpTag3:
        case stringTag3:
            return object == other + "";
        case mapTag5:
            var convert = __VIRTUAL_FILE184;
        case setTag5:
            var isPartial = bitmask & COMPARE_PARTIAL_FLAG1;
            convert || (convert = __VIRTUAL_FILE178);
            if (object.size != other.size && !isPartial) {
                return false;
            }
            var stacked = stack.get(object);
            if (stacked) {
                return stacked == other;
            }
            bitmask |= COMPARE_UNORDERED_FLAG1;
            stack.set(object, other);
            var result1 = __VIRTUAL_FILE183(convert(object), convert(other), bitmask, customizer, equalFunc, stack);
            stack["delete"](object);
            return result1;
        case symbolTag3:
            if (symbolValueOf1) {
                return symbolValueOf1.call(object) == symbolValueOf1.call(other);
            }
    }
    return false;
}
var __VIRTUAL_FILE185 = equalByTag;
var COMPARE_PARTIAL_FLAG2 = 1;
var objectProto16 = Object.prototype;
var hasOwnProperty13 = objectProto16.hasOwnProperty;
function equalObjects(object, other, bitmask, customizer, equalFunc, stack) {
    var isPartial = bitmask & COMPARE_PARTIAL_FLAG2, objProps = __VIRTUAL_FILE85(object), objLength = objProps.length, othProps = __VIRTUAL_FILE85(other), othLength = othProps.length;
    if (objLength != othLength && !isPartial) {
        return false;
    }
    var index = objLength;
    while(index--){
        var key = objProps[index];
        if (!(isPartial ? key in other : hasOwnProperty13.call(other, key))) {
            return false;
        }
    }
    var objStacked = stack.get(object);
    var othStacked = stack.get(other);
    if (objStacked && othStacked) {
        return objStacked == other && othStacked == object;
    }
    var result1 = true;
    stack.set(object, other);
    stack.set(other, object);
    var skipCtor = isPartial;
    while(++index < objLength){
        key = objProps[index];
        var objValue = object[key], othValue = other[key];
        if (customizer) {
            var compared = isPartial ? customizer(othValue, objValue, key, other, object, stack) : customizer(objValue, othValue, key, object, other, stack);
        }
        if (!(compared === void 0 ? objValue === othValue || equalFunc(objValue, othValue, bitmask, customizer, stack) : compared)) {
            result1 = false;
            break;
        }
        skipCtor || (skipCtor = key == "constructor");
    }
    if (result1 && !skipCtor) {
        var objCtor = object.constructor, othCtor = other.constructor;
        if (objCtor != othCtor && "constructor" in object && "constructor" in other && !(typeof objCtor == "function" && objCtor instanceof objCtor && typeof othCtor == "function" && othCtor instanceof othCtor)) {
            result1 = false;
        }
    }
    stack["delete"](object);
    stack["delete"](other);
    return result1;
}
var __VIRTUAL_FILE186 = equalObjects;
var COMPARE_PARTIAL_FLAG3 = 1;
var argsTag3 = "[object Arguments]", arrayTag2 = "[object Array]", objectTag4 = "[object Object]";
var objectProto17 = Object.prototype;
var hasOwnProperty14 = objectProto17.hasOwnProperty;
function baseIsEqualDeep(object, other, bitmask, customizer, equalFunc, stack) {
    var objIsArr = __VIRTUAL_FILE53(object), othIsArr = __VIRTUAL_FILE53(other), objTag = objIsArr ? arrayTag2 : __VIRTUAL_FILE91(object), othTag = othIsArr ? arrayTag2 : __VIRTUAL_FILE91(other);
    objTag = objTag == argsTag3 ? objectTag4 : objTag;
    othTag = othTag == argsTag3 ? objectTag4 : othTag;
    var objIsObj = objTag == objectTag4, othIsObj = othTag == objectTag4, isSameTag = objTag == othTag;
    if (isSameTag && __VIRTUAL_FILE55(object)) {
        if (!__VIRTUAL_FILE55(other)) {
            return false;
        }
        objIsArr = true;
        objIsObj = false;
    }
    if (isSameTag && !objIsObj) {
        stack || (stack = new __VIRTUAL_FILE43());
        return objIsArr || __VIRTUAL_FILE61(object) ? __VIRTUAL_FILE183(object, other, bitmask, customizer, equalFunc, stack) : __VIRTUAL_FILE185(object, other, objTag, bitmask, customizer, equalFunc, stack);
    }
    if (!(bitmask & COMPARE_PARTIAL_FLAG3)) {
        var objIsWrapped = objIsObj && hasOwnProperty14.call(object, "__wrapped__"), othIsWrapped = othIsObj && hasOwnProperty14.call(other, "__wrapped__");
        if (objIsWrapped || othIsWrapped) {
            var objUnwrapped = objIsWrapped ? object.value() : object, othUnwrapped = othIsWrapped ? other.value() : other;
            stack || (stack = new __VIRTUAL_FILE43());
            return equalFunc(objUnwrapped, othUnwrapped, bitmask, customizer, stack);
        }
    }
    if (!isSameTag) {
        return false;
    }
    stack || (stack = new __VIRTUAL_FILE43());
    return __VIRTUAL_FILE186(object, other, bitmask, customizer, equalFunc, stack);
}
var __VIRTUAL_FILE187 = baseIsEqualDeep;
function baseIsEqual(value, other, bitmask, customizer, stack) {
    if (value === other) {
        return true;
    }
    if (value == null || other == null || !__VIRTUAL_FILE50(value) && !__VIRTUAL_FILE50(other)) {
        return value !== value && other !== other;
    }
    return __VIRTUAL_FILE187(value, other, bitmask, customizer, baseIsEqual, stack);
}
var __VIRTUAL_FILE188 = baseIsEqual;
var COMPARE_PARTIAL_FLAG4 = 1, COMPARE_UNORDERED_FLAG2 = 2;
function baseIsMatch(object, source, matchData, customizer) {
    var index = matchData.length, length = index, noCustomizer = !customizer;
    if (object == null) {
        return !length;
    }
    object = Object(object);
    while(index--){
        var data = matchData[index];
        if (noCustomizer && data[2] ? data[1] !== object[data[0]] : !(data[0] in object)) {
            return false;
        }
    }
    while(++index < length){
        data = matchData[index];
        var key = data[0], objValue = object[key], srcValue = data[1];
        if (noCustomizer && data[2]) {
            if (objValue === void 0 && !(key in object)) {
                return false;
            }
        } else {
            var stack = new __VIRTUAL_FILE43();
            if (customizer) {
                var result1 = customizer(objValue, srcValue, key, object, source, stack);
            }
            if (!(result1 === void 0 ? __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG4 | COMPARE_UNORDERED_FLAG2, customizer, stack) : result1)) {
                return false;
            }
        }
    }
    return true;
}
var __VIRTUAL_FILE189 = baseIsMatch;
function isStrictComparable(value) {
    return value === value && !__VIRTUAL_FILE18(value);
}
var __VIRTUAL_FILE190 = isStrictComparable;
function getMatchData(object) {
    var result1 = __VIRTUAL_FILE68(object), length = result1.length;
    while(length--){
        var key = result1[length], value = object[key];
        result1[length] = [
            key,
            value,
            __VIRTUAL_FILE190(value)
        ];
    }
    return result1;
}
var __VIRTUAL_FILE191 = getMatchData;
function matchesStrictComparable(key, srcValue) {
    return function(object) {
        if (object == null) {
            return false;
        }
        return object[key] === srcValue && (srcValue !== void 0 || key in Object(object));
    };
}
var __VIRTUAL_FILE192 = matchesStrictComparable;
function baseMatches(source) {
    var matchData = __VIRTUAL_FILE191(source);
    if (matchData.length == 1 && matchData[0][2]) {
        return __VIRTUAL_FILE192(matchData[0][0], matchData[0][1]);
    }
    return function(object) {
        return object === source || __VIRTUAL_FILE189(object, source, matchData);
    };
}
var __VIRTUAL_FILE193 = baseMatches;
var reIsDeepProp = /\.|\[(?:[^[\]]*|(["'])(?:(?!\1)[^\\]|\\.)*?\1)\]/, reIsPlainProp = /^\w*$/;
function isKey(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return false;
    }
    var type = typeof value;
    if (type == "number" || type == "symbol" || type == "boolean" || value == null || __VIRTUAL_FILE111(value)) {
        return true;
    }
    return reIsPlainProp.test(value) || !reIsDeepProp.test(value) || object != null && value in Object(object);
}
var __VIRTUAL_FILE194 = isKey;
var FUNC_ERROR_TEXT1 = "Expected a function";
function memoize(func, resolver) {
    if (typeof func != "function" || resolver != null && typeof resolver != "function") {
        throw new TypeError(FUNC_ERROR_TEXT1);
    }
    var memoized = function() {
        var args = arguments, key = resolver ? resolver.apply(this, args) : args[0], cache = memoized.cache;
        if (cache.has(key)) {
            return cache.get(key);
        }
        var result1 = func.apply(this, args);
        memoized.cache = cache.set(key, result1) || cache;
        return result1;
    };
    memoized.cache = new (memoize.Cache || __VIRTUAL_FILE41)();
    return memoized;
}
memoize.Cache = __VIRTUAL_FILE41;
var __VIRTUAL_FILE195 = memoize;
var MAX_MEMOIZE_SIZE = 500;
function memoizeCapped(func) {
    var result1 = __VIRTUAL_FILE195(func, function(key) {
        if (cache.size === MAX_MEMOIZE_SIZE) {
            cache.clear();
        }
        return key;
    });
    var cache = result1.cache;
    return result1;
}
var __VIRTUAL_FILE196 = memoizeCapped;
var rePropName = /[^.[\]]+|\[(?:(-?\d+(?:\.\d+)?)|(["'])((?:(?!\2)[^\\]|\\.)*?)\2)\]|(?=(?:\.|\[\])(?:\.|\[\]|$))/g;
var reEscapeChar = /\\(\\)?/g;
var stringToPath = __VIRTUAL_FILE196(function(string) {
    var result1 = [];
    if (string.charCodeAt(0) === 46) {
        result1.push("");
    }
    string.replace(rePropName, function(match, number, quote, subString) {
        result1.push(quote ? subString.replace(reEscapeChar, "$1") : number || match);
    });
    return result1;
});
var __VIRTUAL_FILE197 = stringToPath;
function castPath(value, object) {
    if (__VIRTUAL_FILE53(value)) {
        return value;
    }
    return __VIRTUAL_FILE194(value, object) ? [
        value
    ] : __VIRTUAL_FILE197(__VIRTUAL_FILE171(value));
}
var __VIRTUAL_FILE198 = castPath;
var INFINITY2 = 1 / 0;
function toKey(value) {
    if (typeof value == "string" || __VIRTUAL_FILE111(value)) {
        return value;
    }
    var result1 = value + "";
    return result1 == "0" && 1 / value == -INFINITY2 ? "-0" : result1;
}
var __VIRTUAL_FILE199 = toKey;
function baseGet(object, path) {
    path = __VIRTUAL_FILE198(path, object);
    var index = 0, length = path.length;
    while(object != null && index < length){
        object = object[__VIRTUAL_FILE199(path[index++])];
    }
    return index && index == length ? object : void 0;
}
var __VIRTUAL_FILE200 = baseGet;
function get(object, path, defaultValue) {
    var result1 = object == null ? void 0 : __VIRTUAL_FILE200(object, path);
    return result1 === void 0 ? defaultValue : result1;
}
var __VIRTUAL_FILE201 = get;
function baseHasIn(object, key) {
    return object != null && key in Object(object);
}
var __VIRTUAL_FILE202 = baseHasIn;
function hasPath(object, path, hasFunc) {
    path = __VIRTUAL_FILE198(path, object);
    var index = -1, length = path.length, result1 = false;
    while(++index < length){
        var key = __VIRTUAL_FILE199(path[index]);
        if (!(result1 = object != null && hasFunc(object, key))) {
            break;
        }
        object = object[key];
    }
    if (result1 || ++index != length) {
        return result1;
    }
    length = object == null ? 0 : object.length;
    return !!length && __VIRTUAL_FILE57(length) && __VIRTUAL_FILE56(key, length) && (__VIRTUAL_FILE53(object) || __VIRTUAL_FILE52(object));
}
var __VIRTUAL_FILE203 = hasPath;
function hasIn(object, path) {
    return object != null && __VIRTUAL_FILE203(object, path, __VIRTUAL_FILE202);
}
var __VIRTUAL_FILE204 = hasIn;
var COMPARE_PARTIAL_FLAG5 = 1, COMPARE_UNORDERED_FLAG3 = 2;
function baseMatchesProperty(path, srcValue) {
    if (__VIRTUAL_FILE194(path) && __VIRTUAL_FILE190(srcValue)) {
        return __VIRTUAL_FILE192(__VIRTUAL_FILE199(path), srcValue);
    }
    return function(object) {
        var objValue = __VIRTUAL_FILE201(object, path);
        return objValue === void 0 && objValue === srcValue ? __VIRTUAL_FILE204(object, path) : __VIRTUAL_FILE188(srcValue, objValue, COMPARE_PARTIAL_FLAG5 | COMPARE_UNORDERED_FLAG3);
    };
}
var __VIRTUAL_FILE205 = baseMatchesProperty;
function baseProperty(key) {
    return function(object) {
        return object == null ? void 0 : object[key];
    };
}
var __VIRTUAL_FILE206 = baseProperty;
function basePropertyDeep(path) {
    return function(object) {
        return __VIRTUAL_FILE200(object, path);
    };
}
var __VIRTUAL_FILE207 = basePropertyDeep;
function property(path) {
    return __VIRTUAL_FILE194(path) ? __VIRTUAL_FILE206(__VIRTUAL_FILE199(path)) : __VIRTUAL_FILE207(path);
}
var __VIRTUAL_FILE208 = property;
function baseIteratee(value) {
    if (typeof value == "function") {
        return value;
    }
    if (value == null) {
        return __VIRTUAL_FILE128;
    }
    if (typeof value == "object") {
        return __VIRTUAL_FILE53(value) ? __VIRTUAL_FILE205(value[0], value[1]) : __VIRTUAL_FILE193(value);
    }
    return __VIRTUAL_FILE208(value);
}
var __VIRTUAL_FILE209 = baseIteratee;
function uniqBy(array, iteratee) {
    return array && array.length ? __VIRTUAL_FILE180(array, __VIRTUAL_FILE209(iteratee, 2)) : [];
}
var __VIRTUAL_FILE210 = uniqBy;
function isEqual(value, other) {
    return __VIRTUAL_FILE188(value, other);
}
var __VIRTUAL_FILE211 = isEqual;
function baseMap(collection, iteratee) {
    var index = -1, result1 = __VIRTUAL_FILE67(collection) ? Array(collection.length) : [];
    __VIRTUAL_FILE142(collection, function(value, key, collection2) {
        result1[++index] = iteratee(value, key, collection2);
    });
    return result1;
}
var __VIRTUAL_FILE212 = baseMap;
function baseSortBy(array, comparer) {
    var length = array.length;
    array.sort(comparer);
    while(length--){
        array[length] = array[length].value;
    }
    return array;
}
var __VIRTUAL_FILE213 = baseSortBy;
function compareAscending(value, other) {
    if (value !== other) {
        var valIsDefined = value !== void 0, valIsNull = value === null, valIsReflexive = value === value, valIsSymbol = __VIRTUAL_FILE111(value);
        var othIsDefined = other !== void 0, othIsNull = other === null, othIsReflexive = other === other, othIsSymbol = __VIRTUAL_FILE111(other);
        if (!othIsNull && !othIsSymbol && !valIsSymbol && value > other || valIsSymbol && othIsDefined && othIsReflexive && !othIsNull && !othIsSymbol || valIsNull && othIsDefined && othIsReflexive || !valIsDefined && othIsReflexive || !valIsReflexive) {
            return 1;
        }
        if (!valIsNull && !valIsSymbol && !othIsSymbol && value < other || othIsSymbol && valIsDefined && valIsReflexive && !valIsNull && !valIsSymbol || othIsNull && valIsDefined && valIsReflexive || !othIsDefined && valIsReflexive || !othIsReflexive) {
            return -1;
        }
    }
    return 0;
}
var __VIRTUAL_FILE214 = compareAscending;
function compareMultiple(object, other, orders) {
    var index = -1, objCriteria = object.criteria, othCriteria = other.criteria, length = objCriteria.length, ordersLength = orders.length;
    while(++index < length){
        var result1 = __VIRTUAL_FILE214(objCriteria[index], othCriteria[index]);
        if (result1) {
            if (index >= ordersLength) {
                return result1;
            }
            var order = orders[index];
            return result1 * (order == "desc" ? -1 : 1);
        }
    }
    return object.index - other.index;
}
var __VIRTUAL_FILE215 = compareMultiple;
function baseOrderBy(collection, iteratees, orders) {
    if (iteratees.length) {
        iteratees = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            if (__VIRTUAL_FILE53(iteratee)) {
                return function(value) {
                    return __VIRTUAL_FILE200(value, iteratee.length === 1 ? iteratee[0] : iteratee);
                };
            }
            return iteratee;
        });
    } else {
        iteratees = [
            __VIRTUAL_FILE128
        ];
    }
    var index = -1;
    iteratees = __VIRTUAL_FILE123(iteratees, __VIRTUAL_FILE59(__VIRTUAL_FILE209));
    var result1 = __VIRTUAL_FILE212(collection, function(value, key, collection2) {
        var criteria = __VIRTUAL_FILE123(iteratees, function(iteratee) {
            return iteratee(value);
        });
        return {
            criteria,
            index: ++index,
            value
        };
    });
    return __VIRTUAL_FILE213(result1, function(object, other) {
        return __VIRTUAL_FILE215(object, other, orders);
    });
}
var __VIRTUAL_FILE216 = baseOrderBy;
function orderBy(collection, iteratees, orders, guard) {
    if (collection == null) {
        return [];
    }
    if (!__VIRTUAL_FILE53(iteratees)) {
        iteratees = iteratees == null ? [] : [
            iteratees
        ];
    }
    orders = guard ? void 0 : orders;
    if (!__VIRTUAL_FILE53(orders)) {
        orders = orders == null ? [] : [
            orders
        ];
    }
    return __VIRTUAL_FILE216(collection, iteratees, orders);
}
var __VIRTUAL_FILE217 = orderBy;
const cloneDeep1 = __VIRTUAL_FILE107;
const debounce1 = __VIRTUAL_FILE113;
const shuffle1 = __VIRTUAL_FILE161;
const toString1 = __VIRTUAL_FILE171;
const uniq1 = __VIRTUAL_FILE181;
const uniqBy1 = __VIRTUAL_FILE210;
const escape1 = __VIRTUAL_FILE172;
function getenv(name, defaultValue) {
    const value = Deno.env.get(name);
    if (value === undefined) {
        if (defaultValue === undefined) {
            throw new Error(`Required environment variable ${name} not specified.`);
        } else {
            return defaultValue;
        }
    } else {
        return value;
    }
}
function withPath(paths) {
    const delimiter = Deno.build.os === "windows" ? ";" : ":";
    const currentPath = Deno.env.get("PATH") || "";
    if (paths.append !== undefined && paths.prepend !== undefined) {
        return currentPath;
    } else if (paths.append?.length === 0 && paths.prepend?.length === 0) {
        return currentPath;
    } else {
        const modifiedPaths = [
            currentPath
        ];
        if (paths.append) {
            modifiedPaths.unshift(...paths.append);
        }
        if (paths.prepend) {
            modifiedPaths.push(...paths.prepend);
        }
        return modifiedPaths.join(delimiter);
    }
}
class MuxAsyncIterator {
    #iteratorCount = 0;
    #yields = [];
    #throws = [];
    #signal = Promise.withResolvers();
    add(iterable) {
        ++this.#iteratorCount;
        this.#callIteratorNext(iterable[Symbol.asyncIterator]());
    }
    async #callIteratorNext(iterator) {
        try {
            const { value, done } = await iterator.next();
            if (done) {
                --this.#iteratorCount;
            } else {
                this.#yields.push({
                    iterator,
                    value
                });
            }
        } catch (e) {
            this.#throws.push(e);
        }
        this.#signal.resolve();
    }
    async *iterate() {
        while(this.#iteratorCount > 0){
            await this.#signal.promise;
            for (const { iterator, value } of this.#yields){
                yield value;
                this.#callIteratorNext(iterator);
            }
            if (this.#throws.length) {
                for (const e of this.#throws){
                    throw e;
                }
                this.#throws.length = 0;
            }
            this.#yields.length = 0;
            this.#signal = Promise.withResolvers();
        }
    }
    [Symbol.asyncIterator]() {
        return this.iterate();
    }
}
const ERROR_WHILE_MAPPING_MESSAGE = "Threw while mapping.";
function pooledMap(poolLimit, array, iteratorFn) {
    const res = new TransformStream({
        async transform (p, controller) {
            try {
                const s = await p;
                controller.enqueue(s);
            } catch (e) {
                if (e instanceof AggregateError && e.message === ERROR_WHILE_MAPPING_MESSAGE) {
                    controller.error(e);
                }
            }
        }
    });
    (async ()=>{
        const writer = res.writable.getWriter();
        const executing = [];
        try {
            for await (const item of array){
                const p = Promise.resolve().then(()=>iteratorFn(item));
                writer.write(p);
                const e = p.then(()=>executing.splice(executing.indexOf(e), 1));
                executing.push(e);
                if (executing.length >= poolLimit) {
                    await Promise.race(executing);
                }
            }
            await Promise.all(executing);
            writer.close();
        } catch  {
            const errors = [];
            for (const result1 of (await Promise.allSettled(executing))){
                if (result1.status === "rejected") {
                    errors.push(result1.reason);
                }
            }
            writer.write(Promise.reject(new AggregateError(errors, ERROR_WHILE_MAPPING_MESSAGE))).catch(()=>{});
        }
    })();
    return Symbol.asyncIterator in res.readable && typeof res.readable[Symbol.asyncIterator] === "function" ? res.readable[Symbol.asyncIterator]() : async function*() {
        const reader = res.readable.getReader();
        while(true){
            const { done, value } = await reader.read();
            if (done) break;
            yield value;
        }
        reader.releaseLock();
    }();
}
function copy(src, dst, off = 0) {
    off = Math.max(0, Math.min(off, dst.byteLength));
    const dstBytesAvailable = dst.byteLength - off;
    if (src.byteLength > dstBytesAvailable) {
        src = src.subarray(0, dstBytesAvailable);
    }
    dst.set(src, off);
    return src.byteLength;
}
const MAX_SIZE = 2 ** 32 - 2;
class Buffer1 {
    #buf;
    #off = 0;
    #readable = new ReadableStream({
        type: "bytes",
        pull: (controller)=>{
            const view = new Uint8Array(controller.byobRequest.view.buffer);
            if (this.empty()) {
                this.reset();
                controller.close();
                controller.byobRequest.respond(0);
                return;
            }
            const nread = copy(this.#buf.subarray(this.#off), view);
            this.#off += nread;
            controller.byobRequest.respond(nread);
        },
        autoAllocateChunkSize: 16_640
    });
    get readable() {
        return this.#readable;
    }
    #writable = new WritableStream({
        write: (chunk)=>{
            const m = this.#grow(chunk.byteLength);
            copy(chunk, this.#buf, m);
        }
    });
    get writable() {
        return this.#writable;
    }
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
    #tryGrowByReslice(n) {
        const l1 = this.#buf.byteLength;
        if (n <= this.capacity - l1) {
            this.#reslice(l1 + n);
            return l1;
        }
        return -1;
    }
    #reslice(len) {
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    }
    #grow(n) {
        const m = this.length;
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        const i1 = this.#tryGrowByReslice(n);
        if (i1 >= 0) {
            return i1;
        }
        const c = this.capacity;
        if (n <= Math.floor(c / 2) - m) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n > MAX_SIZE) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf = new Uint8Array(Math.min(2 * c + n, MAX_SIZE));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        this.#off = 0;
        this.#reslice(Math.min(m + n, MAX_SIZE));
        return m;
    }
    grow(n) {
        if (n < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n);
        this.#reslice(m);
    }
}
function concat(buf) {
    let length = 0;
    for (const b of buf){
        length += b.length;
    }
    const output = new Uint8Array(length);
    let index = 0;
    for (const b of buf){
        output.set(b, index);
        index += b.length;
    }
    return output;
}
const DEFAULT_BUFFER_SIZE = 32 * 1024;
async function* iterateReader(reader, options) {
    const bufSize = options?.bufSize ?? DEFAULT_BUFFER_SIZE;
    const b = new Uint8Array(bufSize);
    while(true){
        const result1 = await reader.read(b);
        if (result1 === null) {
            break;
        }
        yield b.slice(0, result1);
    }
}
function iterateReader1(r, options) {
    return iterateReader(r, options);
}
const MIN_READ = 32 * 1024;
const MAX_SIZE1 = 2 ** 32 - 2;
class Buffer2 {
    #buf;
    #off = 0;
    constructor(ab){
        this.#buf = ab === undefined ? new Uint8Array(0) : new Uint8Array(ab);
    }
    bytes(options = {
        copy: true
    }) {
        if (options.copy === false) return this.#buf.subarray(this.#off);
        return this.#buf.slice(this.#off);
    }
    empty() {
        return this.#buf.byteLength <= this.#off;
    }
    get length() {
        return this.#buf.byteLength - this.#off;
    }
    get capacity() {
        return this.#buf.buffer.byteLength;
    }
    truncate(n) {
        if (n === 0) {
            this.reset();
            return;
        }
        if (n < 0 || n > this.length) {
            throw Error("bytes.Buffer: truncation out of range");
        }
        this.#reslice(this.#off + n);
    }
    reset() {
        this.#reslice(0);
        this.#off = 0;
    }
    #tryGrowByReslice(n) {
        const l1 = this.#buf.byteLength;
        if (n <= this.capacity - l1) {
            this.#reslice(l1 + n);
            return l1;
        }
        return -1;
    }
    #reslice(len) {
        assert(len <= this.#buf.buffer.byteLength);
        this.#buf = new Uint8Array(this.#buf.buffer, 0, len);
    }
    readSync(p) {
        if (this.empty()) {
            this.reset();
            if (p.byteLength === 0) {
                return 0;
            }
            return null;
        }
        const nread = copy(this.#buf.subarray(this.#off), p);
        this.#off += nread;
        return nread;
    }
    read(p) {
        const rr = this.readSync(p);
        return Promise.resolve(rr);
    }
    writeSync(p) {
        const m = this.#grow(p.byteLength);
        return copy(p, this.#buf, m);
    }
    write(p) {
        const n = this.writeSync(p);
        return Promise.resolve(n);
    }
    #grow(n) {
        const m = this.length;
        if (m === 0 && this.#off !== 0) {
            this.reset();
        }
        const i1 = this.#tryGrowByReslice(n);
        if (i1 >= 0) {
            return i1;
        }
        const c = this.capacity;
        if (n <= Math.floor(c / 2) - m) {
            copy(this.#buf.subarray(this.#off), this.#buf);
        } else if (c + n > MAX_SIZE1) {
            throw new Error("The buffer cannot be grown beyond the maximum size.");
        } else {
            const buf = new Uint8Array(Math.min(2 * c + n, MAX_SIZE1));
            copy(this.#buf.subarray(this.#off), buf);
            this.#buf = buf;
        }
        this.#off = 0;
        this.#reslice(Math.min(m + n, MAX_SIZE1));
        return m;
    }
    grow(n) {
        if (n < 0) {
            throw Error("Buffer.grow: negative count");
        }
        const m = this.#grow(n);
        this.#reslice(m);
    }
    async readFrom(r) {
        let n = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = await r.read(buf);
            if (nread === null) {
                return n;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n += nread;
        }
    }
    readFromSync(r) {
        let n = 0;
        const tmp = new Uint8Array(MIN_READ);
        while(true){
            const shouldGrow = this.capacity - this.length < MIN_READ;
            const buf = shouldGrow ? tmp : new Uint8Array(this.#buf.buffer, this.length);
            const nread = r.readSync(buf);
            if (nread === null) {
                return n;
            }
            if (shouldGrow) this.writeSync(buf.subarray(0, nread));
            else this.#reslice(this.length + nread);
            n += nread;
        }
    }
}
async function writeAll(writer, data) {
    let nwritten = 0;
    while(nwritten < data.length){
        nwritten += await writer.write(data.subarray(nwritten));
    }
}
function writeAllSync(writer, data) {
    let nwritten = 0;
    while(nwritten < data.length){
        nwritten += writer.writeSync(data.subarray(nwritten));
    }
}
new TextDecoder();
const cleanupHandlers = [];
function onCleanup(handler) {
    cleanupHandlers.push(handler);
}
function exitWithCleanup(code) {
    for (const handler of cleanupHandlers){
        try {
            handler();
        } catch (error) {
            info("Error occurred during cleanup: " + error);
        }
    }
    Deno.exit(code);
}
const processList = new Map();
let processCount = 0;
let cleanupRegistered = false;
function ensureCleanup() {
    if (!cleanupRegistered) {
        cleanupRegistered = true;
        onCleanup(()=>{
            for (const process of processList.values()){
                try {
                    process.kill();
                    process.close();
                } catch (error) {
                    info("Error occurred during cleanup: " + error);
                }
            }
        });
    }
}
async function execProcess(options, stdin, mergeOutput, stderrFilter, respectStreams, timeout) {
    const withTimeout = (promise)=>{
        return timeout ? Promise.race([
            promise,
            new Promise((_, reject)=>setTimeout(()=>reject(new Error("Process timed out")), timeout))
        ]) : promise;
    };
    ensureCleanup();
    try {
        debug(`[execProcess] ${options.cmd.join(" ")}`);
        const process = Deno.run({
            ...options,
            stdin: stdin !== undefined ? "piped" : options.stdin,
            stdout: typeof options.stdout === "number" ? options.stdout : "piped",
            stderr: typeof options.stderr === "number" ? options.stderr : "piped"
        });
        const thisProcessId = ++processCount;
        processList.set(thisProcessId, process);
        if (stdin !== undefined) {
            if (!process.stdin) {
                processList.delete(processCount);
                throw new Error("Process stdin not available");
            }
            const buffer = new TextEncoder().encode(stdin);
            let offset = 0;
            while(offset < buffer.length){
                const end = Math.min(offset + 4096, buffer.length);
                const window1 = buffer.subarray(offset, end);
                const written = await process.stdin.write(window1);
                offset += written;
            }
            process.stdin.close();
        }
        let stdoutText = "";
        let stderrText = "";
        if (mergeOutput) {
            const multiplexIterator = new MuxAsyncIterator();
            const addStream = (stream, filter)=>{
                if (stream !== null) {
                    const streamIter = filter ? filteredAsyncIterator(iterateReader1(stream), filter) : iterateReader1(stream);
                    multiplexIterator.add(streamIter);
                }
            };
            addStream(process.stdout);
            addStream(process.stderr, stderrFilter);
            const allOutput = await processOutput(multiplexIterator, mergeOutput === "stderr>stdout" ? options.stdout : options.stderr);
            if (mergeOutput === "stderr>stdout") {
                stdoutText = allOutput;
            } else {
                stderrText = allOutput;
            }
            const closeStream = (stream)=>{
                if (stream) {
                    stream.close();
                }
            };
            closeStream(process.stdout);
            closeStream(process.stderr);
        } else {
            const promises = [];
            if (process.stdout !== null) {
                promises.push(processOutput(iterateReader1(process.stdout), options.stdout, respectStreams ? "stdout" : undefined).then((text)=>{
                    stdoutText = text;
                    process.stdout.close();
                }));
            }
            if (process.stderr != null) {
                const iterator = stderrFilter ? filteredAsyncIterator(iterateReader1(process.stderr), stderrFilter) : iterateReader1(process.stderr);
                promises.push(processOutput(iterator, options.stderr, respectStreams ? "stderr" : undefined).then((text)=>{
                    stderrText = text;
                    process.stderr.close();
                }));
            }
            await withTimeout(Promise.all(promises));
        }
        const status = await withTimeout(process.status());
        process.close();
        processList.delete(thisProcessId);
        debug(`[execProcess] Success: ${status.success}, code: ${status.code}`);
        return {
            success: status.success,
            code: status.code,
            stdout: stdoutText,
            stderr: stderrText
        };
    } catch (e) {
        throw new Error(`Error executing '${options.cmd[0]}': ${e.message}`);
    }
}
function processSuccessResult() {
    return {
        success: true,
        code: 0
    };
}
function filteredAsyncIterator(iterator, filter) {
    const encoder = new TextEncoder();
    const decoder = new TextDecoder();
    return pooledMap(1, iterator, (data)=>{
        return Promise.resolve(encoder.encode(filter(decoder.decode(data))));
    });
}
async function processOutput(iterator, output, which) {
    const decoder = new TextDecoder();
    let outputText = "";
    for await (const chunk of iterator){
        if (output === "inherit" || output === undefined) {
            if (which === "stdout") {
                Deno.stdout.writeSync(chunk);
            } else if (which === "stderr") {
                Deno.stderr.writeSync(chunk);
            } else {
                info(decoder.decode(chunk), {
                    newline: false
                });
            }
        }
        const text = decoder.decode(chunk);
        outputText += text;
    }
    return outputText;
}
const kSkipHidden = /[/\\][\.]/;
function removeIfExists(file) {
    if (existsSync(file)) {
        Deno.removeSync(file, {
            recursive: true
        });
    }
}
function safeRemoveIfExists(file) {
    try {
        removeIfExists(file);
    } catch (error) {
        warn(`Error removing file ${file}: ${error.message}`);
    }
}
function safeRemoveSync(file, options = {}) {
    try {
        Deno.removeSync(file, options);
    } catch (e) {
        if (existsSync(file)) throw e;
    }
}
function removeIfEmptyDir(dir) {
    if (existsSync(dir)) {
        let empty = true;
        for (const _entry of Deno.readDirSync(dir)){
            empty = false;
            break;
        }
        if (empty) {
            safeRemoveSync(dir, {
                recursive: true
            });
            return true;
        }
        return false;
    } else {
        return false;
    }
}
function isModifiedAfter(file, otherFile) {
    if (!existsSync(file)) {
        throw new Error(`${file} does not exist`);
    }
    if (!existsSync(otherFile)) {
        return true;
    }
    const fileInfo = Deno.statSync(file);
    const otherfileInfo = Deno.statSync(otherFile);
    if (fileInfo.mtime === null || otherfileInfo.mtime === null) {
        return true;
    } else {
        return fileInfo.mtime > otherfileInfo.mtime;
    }
}
function dirAndStem(file) {
    return [
        dirname2(file),
        basename2(file, extname2(file))
    ];
}
function expandPath(path) {
    if (path === "~") {
        return getenv("HOME", "~");
    } else {
        return path.replace(/^~\//, getenv("HOME", "~") + "/");
    }
}
function safeExistsSync(path) {
    try {
        return existsSync(path);
    } catch  {
        return false;
    }
}
async function which(cmd) {
    const args = Deno.build.os === "windows" ? [
        "CMD",
        "/C",
        "where",
        cmd
    ] : [
        "which",
        cmd
    ];
    const result1 = await execProcess({
        cmd: args,
        stderr: "piped",
        stdout: "piped"
    });
    if (result1.code === 0) {
        return Deno.build.os === "windows" ? result1.stdout?.split("\n")[0].trim() : result1.stdout?.trim();
    } else {
        return undefined;
    }
}
function filterPaths(root, paths, globs, options) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            const needSlash = !root.endsWith("/") && !glob.startsWith("/");
            const regex = globToRegExp2(`${root}${needSlash ? "/" : ""}${glob}`);
            const matchingFiles = paths.filter((path)=>{
                return regex.test(path);
            });
            expanded.push(...matchingFiles);
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root, globs, expandGlobs, options);
}
function resolvePathGlobs(root, globs, exclude, options) {
    const expandGlobs = (targetGlobs)=>{
        const expanded = [];
        for (const glob of targetGlobs){
            for (const file of expandGlobSync(glob, {
                root,
                exclude,
                includeDirs: true,
                extended: true,
                globstar: true
            })){
                expanded.push(file.path);
            }
        }
        return __VIRTUAL_FILE181(expanded);
    };
    return resolveGlobs(root, globs, expandGlobs, options);
}
function pathWithForwardSlashes(path) {
    return path.replace(/\\/g, "/");
}
function ensureTrailingSlash(path) {
    if (path && !path.endsWith("/")) {
        return path + "/";
    } else {
        return path;
    }
}
function resolveGlobs(root, globs, expandGlobs, options) {
    const includeGlobs = [];
    const excludeGlobs = [];
    const asFullGlob = (glob, preferSmart)=>{
        const useSmartGlobs = ()=>{
            if (options?.mode === "strict") {
                return false;
            } else if (options?.mode === "always") {
                return true;
            } else if (options?.mode === "auto") {
                if (preferSmart) {
                    return true;
                } else {
                    return isGlob(glob);
                }
            } else {
                return true;
            }
        };
        const smartGlob = useSmartGlobs();
        if (glob.startsWith("\\!")) {
            glob = glob.slice(1);
        }
        if (smartGlob) {
            glob = glob.replace(/^\.([\/\\])+/, "$1");
            if (glob.endsWith("/")) {
                glob = glob + "**/*";
            } else {
                const fullPath = join2(root, glob);
                try {
                    if (Deno.statSync(fullPath).isDirectory) {
                        glob = glob + "/**/*";
                    }
                } catch  {}
            }
        }
        if (!glob.startsWith("/")) {
            if (smartGlob && (!options || !options.explicitSubfolderSearch)) {
                return "**/" + glob;
            } else {
                return glob;
            }
        } else {
            return glob.slice(1);
        }
    };
    for (const glob of globs){
        if (glob.startsWith("!")) {
            excludeGlobs.push(asFullGlob(glob.slice(1), true));
        } else {
            includeGlobs.push(asFullGlob(glob));
        }
    }
    const includeFiles = expandGlobs(includeGlobs);
    const excludeFiles = expandGlobs(excludeGlobs);
    return {
        include: includeFiles,
        exclude: excludeFiles
    };
}
function normalizePath(path) {
    let file = path instanceof URL ? fromFileUrl2(path) : path;
    if (!isAbsolute2(file)) {
        file = join2(Deno.cwd(), file);
    }
    file = normalize2(file);
    return file.replace(/^\w:\\/, (m)=>m[0].toUpperCase() + ":\\");
}
function suggestUserBinPaths() {
    if (Deno.build.os !== "windows") {
        const possiblePaths = [
            "/usr/local/bin",
            "~/.local/bin",
            "~/bin"
        ];
        const pathRaw = Deno.env.get("PATH");
        const paths = pathRaw ? pathRaw.split(":") : [];
        return possiblePaths.filter((path)=>{
            return paths.includes(path) || paths.includes(expandPath(path));
        });
    } else {
        throw new Error("suggestUserBinPaths not currently supported on Windows");
    }
}
Deno.realPathSync = normalizePath;
function withAttempts(callable) {
    for(let i1 = 0; i1 < 5; i1++){
        try {
            return callable();
        } catch (err) {
            if (err.message) {
                debug("Error attempting to create temp file: " + err.message);
                if (i1 === 5 - 1) {
                    throw err;
                } else {
                    debug(`Retrying... ${i1 + 1} of ${5 - 1}`);
                }
            }
        }
    }
    throw new Error("Shouldn't get here");
}
function withAttemptsAsync(callable) {
    const inner = (attempt)=>{
        return callable().catch((err)=>{
            if (err.message) {
                debug("Error attempting to create temp file: " + err.message);
            }
            if (attempt === 5 - 1) {
                throw err;
            } else {
                debug(`Retrying... ${attempt + 1} of ${5 - 1}`);
                return inner(attempt + 1);
            }
        });
    };
    return inner(0);
}
const oldMakeTempFile = Deno.makeTempFile;
const oldMakeTempFileSync = Deno.makeTempFileSync;
const oldMakeTempDir = Deno.makeTempDir;
const oldMakeTempDirSync = Deno.makeTempDirSync;
function makeTempFileSync(options) {
    return withAttempts(()=>oldMakeTempFileSync(options));
}
function makeTempFile(options) {
    return withAttemptsAsync(()=>oldMakeTempFile(options));
}
function makeTempDirSync(options) {
    return withAttempts(()=>oldMakeTempDirSync(options));
}
function makeTempDir(options) {
    return withAttemptsAsync(()=>oldMakeTempDir(options));
}
Deno.makeTempFile = makeTempFile;
Deno.makeTempFileSync = makeTempFileSync;
Deno.makeTempDir = makeTempDir;
Deno.makeTempDirSync = makeTempDirSync;
const oldReadTextFile = Deno.readTextFile;
const oldReadTextFileSync = Deno.readTextFileSync;
Deno.readTextFile = async (path, options)=>{
    try {
        const result1 = await oldReadTextFile(path, options);
        return result1;
    } catch (err) {
        if (err.message) {
            err.message = err.message + "\n" + "Path: " + path;
        }
        throw err;
    }
};
Deno.readTextFileSync = (path)=>{
    try {
        const result1 = oldReadTextFileSync(path);
        return result1;
    } catch (err) {
        if (err.message) {
            err.message = err.message + "\n" + "Path: " + path;
        }
        throw err;
    }
};
const { Deno: Deno2 } = globalThis;
const noColor1 = typeof Deno2?.noColor === "boolean" ? Deno2.noColor : false;
let enabled1 = !noColor1;
function setColorEnabled(value) {
    if (Deno2?.noColor) {
        return;
    }
    enabled1 = value;
}
function getColorEnabled() {
    return enabled1;
}
function code1(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run1(str, code) {
    return enabled1 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function reset(str) {
    return run1(str, code1([
        0
    ], 0));
}
function bold1(str) {
    return run1(str, code1([
        1
    ], 22));
}
function dim1(str) {
    return run1(str, code1([
        2
    ], 22));
}
function italic(str) {
    return run1(str, code1([
        3
    ], 23));
}
function underline1(str) {
    return run1(str, code1([
        4
    ], 24));
}
function inverse(str) {
    return run1(str, code1([
        7
    ], 27));
}
function hidden(str) {
    return run1(str, code1([
        8
    ], 28));
}
function strikethrough(str) {
    return run1(str, code1([
        9
    ], 29));
}
function black(str) {
    return run1(str, code1([
        30
    ], 39));
}
function red1(str) {
    return run1(str, code1([
        31
    ], 39));
}
function green1(str) {
    return run1(str, code1([
        32
    ], 39));
}
function yellow1(str) {
    return run1(str, code1([
        33
    ], 39));
}
function blue1(str) {
    return run1(str, code1([
        34
    ], 39));
}
function magenta(str) {
    return run1(str, code1([
        35
    ], 39));
}
function cyan(str) {
    return run1(str, code1([
        36
    ], 39));
}
function white(str) {
    return run1(str, code1([
        37
    ], 39));
}
function gray(str) {
    return brightBlack(str);
}
function brightBlack(str) {
    return run1(str, code1([
        90
    ], 39));
}
function brightRed1(str) {
    return run1(str, code1([
        91
    ], 39));
}
function brightGreen(str) {
    return run1(str, code1([
        92
    ], 39));
}
function brightYellow(str) {
    return run1(str, code1([
        93
    ], 39));
}
function brightBlue(str) {
    return run1(str, code1([
        94
    ], 39));
}
function brightMagenta(str) {
    return run1(str, code1([
        95
    ], 39));
}
function brightCyan(str) {
    return run1(str, code1([
        96
    ], 39));
}
function brightWhite(str) {
    return run1(str, code1([
        97
    ], 39));
}
function bgBlack(str) {
    return run1(str, code1([
        40
    ], 49));
}
function bgRed(str) {
    return run1(str, code1([
        41
    ], 49));
}
function bgGreen(str) {
    return run1(str, code1([
        42
    ], 49));
}
function bgYellow(str) {
    return run1(str, code1([
        43
    ], 49));
}
function bgBlue(str) {
    return run1(str, code1([
        44
    ], 49));
}
function bgMagenta(str) {
    return run1(str, code1([
        45
    ], 49));
}
function bgCyan(str) {
    return run1(str, code1([
        46
    ], 49));
}
function bgWhite(str) {
    return run1(str, code1([
        47
    ], 49));
}
function bgBrightBlack(str) {
    return run1(str, code1([
        100
    ], 49));
}
function bgBrightRed(str) {
    return run1(str, code1([
        101
    ], 49));
}
function bgBrightGreen(str) {
    return run1(str, code1([
        102
    ], 49));
}
function bgBrightYellow(str) {
    return run1(str, code1([
        103
    ], 49));
}
function bgBrightBlue(str) {
    return run1(str, code1([
        104
    ], 49));
}
function bgBrightMagenta(str) {
    return run1(str, code1([
        105
    ], 49));
}
function bgBrightCyan(str) {
    return run1(str, code1([
        106
    ], 49));
}
function bgBrightWhite(str) {
    return run1(str, code1([
        107
    ], 49));
}
function clampAndTruncate(n, max = 255, min = 0) {
    return Math.trunc(Math.max(Math.min(n, max), min));
}
function rgb8(str, color) {
    return run1(str, code1([
        38,
        5,
        clampAndTruncate(color)
    ], 39));
}
function bgRgb8(str, color) {
    return run1(str, code1([
        48,
        5,
        clampAndTruncate(color)
    ], 49));
}
function rgb24(str, color) {
    if (typeof color === "number") {
        return run1(str, code1([
            38,
            2,
            color >> 16 & 0xff,
            color >> 8 & 0xff,
            color & 0xff
        ], 39));
    }
    return run1(str, code1([
        38,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b)
    ], 39));
}
function bgRgb24(str, color) {
    if (typeof color === "number") {
        return run1(str, code1([
            48,
            2,
            color >> 16 & 0xff,
            color >> 8 & 0xff,
            color & 0xff
        ], 49));
    }
    return run1(str, code1([
        48,
        2,
        clampAndTruncate(color.r),
        clampAndTruncate(color.g),
        clampAndTruncate(color.b)
    ], 49));
}
const ANSI_PATTERN1 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]+)*|[a-zA-Z\\d]+(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-nq-uy=><~]))"
].join("|"), "g");
function stripColor1(string) {
    return string.replace(ANSI_PATTERN1, "");
}
const mod1 = {
    setColorEnabled: setColorEnabled,
    getColorEnabled: getColorEnabled,
    reset: reset,
    bold: bold1,
    dim: dim1,
    italic: italic,
    underline: underline1,
    inverse: inverse,
    hidden: hidden,
    strikethrough: strikethrough,
    black: black,
    red: red1,
    green: green1,
    yellow: yellow1,
    blue: blue1,
    magenta: magenta,
    cyan: cyan,
    white: white,
    gray: gray,
    brightBlack: brightBlack,
    brightRed: brightRed1,
    brightGreen: brightGreen,
    brightYellow: brightYellow,
    brightBlue: brightBlue,
    brightMagenta: brightMagenta,
    brightCyan: brightCyan,
    brightWhite: brightWhite,
    bgBlack: bgBlack,
    bgRed: bgRed,
    bgGreen: bgGreen,
    bgYellow: bgYellow,
    bgBlue: bgBlue,
    bgMagenta: bgMagenta,
    bgCyan: bgCyan,
    bgWhite: bgWhite,
    bgBrightBlack: bgBrightBlack,
    bgBrightRed: bgBrightRed,
    bgBrightGreen: bgBrightGreen,
    bgBrightYellow: bgBrightYellow,
    bgBrightBlue: bgBrightBlue,
    bgBrightMagenta: bgBrightMagenta,
    bgBrightCyan: bgBrightCyan,
    bgBrightWhite: bgBrightWhite,
    rgb8: rgb8,
    bgRgb8: bgRgb8,
    rgb24: rgb24,
    bgRgb24: bgRgb24,
    stripColor: stripColor1
};
function distance(a, b) {
    if (a.length == 0) {
        return b.length;
    }
    if (b.length == 0) {
        return a.length;
    }
    const matrix = [];
    for(let i1 = 0; i1 <= b.length; i1++){
        matrix[i1] = [
            i1
        ];
    }
    for(let j = 0; j <= a.length; j++){
        matrix[0][j] = j;
    }
    for(let i1 = 1; i1 <= b.length; i1++){
        for(let j = 1; j <= a.length; j++){
            if (b.charAt(i1 - 1) == a.charAt(j - 1)) {
                matrix[i1][j] = matrix[i1 - 1][j - 1];
            } else {
                matrix[i1][j] = Math.min(matrix[i1 - 1][j - 1] + 1, Math.min(matrix[i1][j - 1] + 1, matrix[i1 - 1][j] + 1));
            }
        }
    }
    return matrix[b.length][a.length];
}
function paramCaseToCamelCase(str) {
    return str.replace(/-([a-z])/g, (g)=>g[1].toUpperCase());
}
function underscoreToCamelCase(str) {
    return str.replace(/([a-z])([A-Z])/g, "$1_$2").toLowerCase().replace(/_([a-z])/g, (g)=>g[1].toUpperCase());
}
function getOption(flags, name) {
    while(name[0] === "-"){
        name = name.slice(1);
    }
    for (const flag of flags){
        if (isOption(flag, name)) {
            return flag;
        }
    }
    return;
}
function didYouMeanOption(option, options) {
    const optionNames = options.map((option)=>[
            option.name,
            ...option.aliases ?? []
        ]).flat().map((option)=>getFlag(option));
    return didYouMean(" Did you mean option", getFlag(option), optionNames);
}
function didYouMeanType(type, types) {
    return didYouMean(" Did you mean type", type, types);
}
function didYouMean(message, type, types) {
    const match = closest(type, types);
    return match ? `${message} "${match}"?` : "";
}
function getFlag(name) {
    if (name.startsWith("-")) {
        return name;
    }
    if (name.length > 1) {
        return `--${name}`;
    }
    return `-${name}`;
}
function isOption(option, name) {
    return option.name === name || option.aliases && option.aliases.indexOf(name) !== -1;
}
function matchWildCardOptions(name, flags) {
    for (const option of flags){
        if (option.name.indexOf("*") === -1) {
            continue;
        }
        let matched = matchWildCardOption(name, option);
        if (matched) {
            matched = {
                ...matched,
                name
            };
            flags.push(matched);
            return matched;
        }
    }
}
function matchWildCardOption(name, option) {
    const parts = option.name.split(".");
    const parts2 = name.split(".");
    if (parts.length !== parts2.length) {
        return false;
    }
    const count = Math.max(parts.length, parts2.length);
    for(let i1 = 0; i1 < count; i1++){
        if (parts[i1] !== parts2[i1] && parts[i1] !== "*") {
            return false;
        }
    }
    return option;
}
function closest(str, arr) {
    let minDistance = Infinity;
    let minIndex = 0;
    for(let i1 = 0; i1 < arr.length; i1++){
        const dist = distance(str, arr[i1]);
        if (dist < minDistance) {
            minDistance = dist;
            minIndex = i1;
        }
    }
    return arr[minIndex];
}
function getDefaultValue(option) {
    return typeof option.default === "function" ? option.default() : option.default;
}
class FlagsError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, FlagsError.prototype);
    }
}
class UnknownRequiredOptionError extends FlagsError {
    constructor(option, options){
        super(`Unknown required option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownRequiredOptionError.prototype);
    }
}
class UnknownConflictingOptionError extends FlagsError {
    constructor(option, options){
        super(`Unknown conflicting option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownConflictingOptionError.prototype);
    }
}
class UnknownTypeError extends FlagsError {
    constructor(type, types){
        super(`Unknown type "${type}".${didYouMeanType(type, types)}`);
        Object.setPrototypeOf(this, UnknownTypeError.prototype);
    }
}
class ValidationError extends FlagsError {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, ValidationError.prototype);
    }
}
class DuplicateOptionError extends ValidationError {
    constructor(name){
        super(`Option "${getFlag(name).replace(/^--no-/, "--")}" can only occur once, but was found several times.`);
        Object.setPrototypeOf(this, DuplicateOptionError.prototype);
    }
}
class InvalidOptionError extends ValidationError {
    constructor(option, options){
        super(`Invalid option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, InvalidOptionError.prototype);
    }
}
class UnknownOptionError extends ValidationError {
    constructor(option, options){
        super(`Unknown option "${getFlag(option)}".${didYouMeanOption(option, options)}`);
        Object.setPrototypeOf(this, UnknownOptionError.prototype);
    }
}
class MissingOptionValueError extends ValidationError {
    constructor(option){
        super(`Missing value for option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingOptionValueError.prototype);
    }
}
class InvalidOptionValueError extends ValidationError {
    constructor(option, expected, value){
        super(`Option "${getFlag(option)}" must be of type "${expected}", but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValueError.prototype);
    }
}
class UnexpectedOptionValueError extends ValidationError {
    constructor(option, value){
        super(`Option "${getFlag(option)}" doesn't take a value, but got "${value}".`);
        Object.setPrototypeOf(this, InvalidOptionValueError.prototype);
    }
}
class OptionNotCombinableError extends ValidationError {
    constructor(option){
        super(`Option "${getFlag(option)}" cannot be combined with other options.`);
        Object.setPrototypeOf(this, OptionNotCombinableError.prototype);
    }
}
class ConflictingOptionError extends ValidationError {
    constructor(option, conflictingOption){
        super(`Option "${getFlag(option)}" conflicts with option "${getFlag(conflictingOption)}".`);
        Object.setPrototypeOf(this, ConflictingOptionError.prototype);
    }
}
class DependingOptionError extends ValidationError {
    constructor(option, dependingOption){
        super(`Option "${getFlag(option)}" depends on option "${getFlag(dependingOption)}".`);
        Object.setPrototypeOf(this, DependingOptionError.prototype);
    }
}
class MissingRequiredOptionError extends ValidationError {
    constructor(option){
        super(`Missing required option "${getFlag(option)}".`);
        Object.setPrototypeOf(this, MissingRequiredOptionError.prototype);
    }
}
class UnexpectedRequiredArgumentError extends ValidationError {
    constructor(arg){
        super(`An required argument cannot follow an optional argument, but "${arg}"  is defined as required.`);
        Object.setPrototypeOf(this, UnexpectedRequiredArgumentError.prototype);
    }
}
class UnexpectedArgumentAfterVariadicArgumentError extends ValidationError {
    constructor(arg){
        super(`An argument cannot follow an variadic argument, but got "${arg}".`);
        Object.setPrototypeOf(this, UnexpectedArgumentAfterVariadicArgumentError.prototype);
    }
}
class InvalidTypeError extends ValidationError {
    constructor({ label, name, value, type }, expected){
        super(`${label} "${name}" must be of type "${type}", but got "${value}".` + (expected ? ` Expected values: ${expected.map((value)=>`"${value}"`).join(", ")}` : ""));
        Object.setPrototypeOf(this, MissingOptionValueError.prototype);
    }
}
var OptionType;
(function(OptionType) {
    OptionType["STRING"] = "string";
    OptionType["NUMBER"] = "number";
    OptionType["INTEGER"] = "integer";
    OptionType["BOOLEAN"] = "boolean";
})(OptionType || (OptionType = {}));
const __boolean = (type)=>{
    if (~[
        "1",
        "true"
    ].indexOf(type.value)) {
        return true;
    }
    if (~[
        "0",
        "false"
    ].indexOf(type.value)) {
        return false;
    }
    throw new InvalidTypeError(type, [
        "true",
        "false",
        "1",
        "0"
    ]);
};
const number = (type)=>{
    const value = Number(type.value);
    if (Number.isFinite(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const string = ({ value })=>{
    return value;
};
function validateFlags(ctx, opts, options = new Map()) {
    if (!opts.flags) {
        return;
    }
    setDefaultValues(ctx, opts);
    const optionNames = Object.keys(ctx.flags);
    if (!optionNames.length && opts.allowEmpty) {
        return;
    }
    if (ctx.standalone) {
        validateStandaloneOption(ctx, options, optionNames);
        return;
    }
    for (const [name, option] of options){
        validateUnknownOption(option, opts);
        validateConflictingOptions(ctx, option);
        validateDependingOptions(ctx, option);
        validateRequiredValues(ctx, option, name);
    }
    validateRequiredOptions(ctx, options, opts);
}
function validateUnknownOption(option, opts) {
    if (!getOption(opts.flags ?? [], option.name)) {
        throw new UnknownOptionError(option.name, opts.flags ?? []);
    }
}
function setDefaultValues(ctx, opts) {
    if (!opts.flags?.length) {
        return;
    }
    for (const option of opts.flags){
        let name;
        let defaultValue = undefined;
        if (option.name.startsWith("no-")) {
            const propName = option.name.replace(/^no-/, "");
            if (typeof ctx.flags[propName] !== "undefined") {
                continue;
            }
            const positiveOption = getOption(opts.flags, propName);
            if (positiveOption) {
                continue;
            }
            name = paramCaseToCamelCase(propName);
            defaultValue = true;
        }
        if (!name) {
            name = paramCaseToCamelCase(option.name);
        }
        const hasDefaultValue = (!opts.ignoreDefaults || typeof opts.ignoreDefaults[name] === "undefined") && typeof ctx.flags[name] === "undefined" && (typeof option.default !== "undefined" || typeof defaultValue !== "undefined");
        if (hasDefaultValue) {
            ctx.flags[name] = getDefaultValue(option) ?? defaultValue;
            ctx.defaults[option.name] = true;
            if (typeof option.value === "function") {
                ctx.flags[name] = option.value(ctx.flags[name]);
            }
        }
    }
}
function validateStandaloneOption(ctx, options, optionNames) {
    if (!ctx.standalone || optionNames.length === 1) {
        return;
    }
    for (const [_, opt] of options){
        if (!ctx.defaults[opt.name] && opt !== ctx.standalone) {
            throw new OptionNotCombinableError(ctx.standalone.name);
        }
    }
}
function validateConflictingOptions(ctx, option) {
    if (!option.conflicts?.length) {
        return;
    }
    for (const flag of option.conflicts){
        if (isset(flag, ctx.flags)) {
            throw new ConflictingOptionError(option.name, flag);
        }
    }
}
function validateDependingOptions(ctx, option) {
    if (!option.depends) {
        return;
    }
    for (const flag of option.depends){
        if (!isset(flag, ctx.flags) && !ctx.defaults[option.name]) {
            throw new DependingOptionError(option.name, flag);
        }
    }
}
function validateRequiredValues(ctx, option, name) {
    if (!option.args) {
        return;
    }
    const isArray = option.args.length > 1;
    for(let i1 = 0; i1 < option.args.length; i1++){
        const arg = option.args[i1];
        if (arg.optional) {
            continue;
        }
        const hasValue = isArray ? typeof ctx.flags[name][i1] !== "undefined" : typeof ctx.flags[name] !== "undefined";
        if (!hasValue) {
            throw new MissingOptionValueError(option.name);
        }
    }
}
function validateRequiredOptions(ctx, options, opts) {
    if (!opts.flags?.length) {
        return;
    }
    const optionsValues = [
        ...options.values()
    ];
    for (const option of opts.flags){
        if (!option.required || paramCaseToCamelCase(option.name) in ctx.flags) {
            continue;
        }
        const conflicts = option.conflicts ?? [];
        const hasConflict = conflicts.find((flag)=>!!ctx.flags[flag]);
        const hasConflicts = hasConflict || optionsValues.find((opt)=>opt.conflicts?.find((flag)=>flag === option.name));
        if (hasConflicts) {
            continue;
        }
        throw new MissingRequiredOptionError(option.name);
    }
}
function isset(flagName, flags) {
    const name = paramCaseToCamelCase(flagName);
    return typeof flags[name] !== "undefined";
}
const integer = (type)=>{
    const value = Number(type.value);
    if (Number.isInteger(value)) {
        return value;
    }
    throw new InvalidTypeError(type);
};
const DefaultTypes = {
    string,
    number,
    integer,
    boolean: __boolean
};
function parseFlags(argsOrCtx, opts = {}) {
    let args;
    let ctx;
    if (Array.isArray(argsOrCtx)) {
        ctx = {};
        args = argsOrCtx;
    } else {
        ctx = argsOrCtx;
        args = argsOrCtx.unknown;
        argsOrCtx.unknown = [];
    }
    args = args.slice();
    ctx.flags ??= {};
    ctx.literal ??= [];
    ctx.unknown ??= [];
    ctx.stopEarly = false;
    ctx.stopOnUnknown = false;
    ctx.defaults ??= {};
    opts.dotted ??= true;
    validateOptions(opts);
    const options = parseArgs(ctx, args, opts);
    validateFlags(ctx, opts, options);
    if (opts.dotted) {
        parseDottedOptions(ctx);
    }
    return ctx;
}
function validateOptions(opts) {
    opts.flags?.forEach((opt)=>{
        opt.depends?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownRequiredOptionError(flag, opts.flags ?? []);
            }
        });
        opt.conflicts?.forEach((flag)=>{
            if (!opts.flags || !getOption(opts.flags, flag)) {
                throw new UnknownConflictingOptionError(flag, opts.flags ?? []);
            }
        });
    });
}
function parseArgs(ctx, args, opts) {
    const optionsMap = new Map();
    let inLiteral = false;
    for(let argsIndex = 0; argsIndex < args.length; argsIndex++){
        let option;
        let current = args[argsIndex];
        let currentValue;
        let negate = false;
        if (inLiteral) {
            ctx.literal.push(current);
            continue;
        } else if (current === "--") {
            inLiteral = true;
            continue;
        } else if (ctx.stopEarly || ctx.stopOnUnknown) {
            ctx.unknown.push(current);
            continue;
        }
        const isFlag = current.length > 1 && current[0] === "-";
        if (!isFlag) {
            if (opts.stopEarly) {
                ctx.stopEarly = true;
            }
            ctx.unknown.push(current);
            continue;
        }
        const isShort = current[1] !== "-";
        const isLong = isShort ? false : current.length > 3 && current[2] !== "-";
        if (!isShort && !isLong) {
            throw new InvalidOptionError(current, opts.flags ?? []);
        }
        if (isShort && current.length > 2 && current[2] !== ".") {
            args.splice(argsIndex, 1, ...splitFlags(current));
            current = args[argsIndex];
        } else if (isLong && current.startsWith("--no-")) {
            negate = true;
        }
        const equalSignIndex = current.indexOf("=");
        if (equalSignIndex !== -1) {
            currentValue = current.slice(equalSignIndex + 1) || undefined;
            current = current.slice(0, equalSignIndex);
        }
        if (opts.flags) {
            option = getOption(opts.flags, current);
            if (!option) {
                const name = current.replace(/^-+/, "");
                option = matchWildCardOptions(name, opts.flags);
                if (!option) {
                    if (opts.stopOnUnknown) {
                        ctx.stopOnUnknown = true;
                        ctx.unknown.push(args[argsIndex]);
                        continue;
                    }
                    throw new UnknownOptionError(current, opts.flags);
                }
            }
        } else {
            option = {
                name: current.replace(/^-+/, ""),
                optionalValue: true,
                type: OptionType.STRING
            };
        }
        if (option.standalone) {
            ctx.standalone = option;
        }
        const positiveName = negate ? option.name.replace(/^no-?/, "") : option.name;
        const propName = paramCaseToCamelCase(positiveName);
        if (typeof ctx.flags[propName] !== "undefined") {
            if (!opts.flags?.length) {
                option.collect = true;
            } else if (!option.collect && !ctx.defaults[option.name]) {
                throw new DuplicateOptionError(current);
            }
        }
        if (option.type && !option.args?.length) {
            option.args = [
                {
                    type: option.type,
                    optional: option.optionalValue,
                    variadic: option.variadic,
                    list: option.list,
                    separator: option.separator
                }
            ];
        }
        if (opts.flags?.length && !option.args?.length && typeof currentValue !== "undefined") {
            throw new UnexpectedOptionValueError(option.name, currentValue);
        }
        let optionArgsIndex = 0;
        let inOptionalArg = false;
        const next = ()=>currentValue ?? args[argsIndex + 1];
        const previous = ctx.flags[propName];
        parseNext(option);
        if (typeof ctx.flags[propName] === "undefined") {
            if (option.args?.length && !option.args?.[optionArgsIndex].optional) {
                throw new MissingOptionValueError(option.name);
            } else if (typeof option.default !== "undefined" && (option.type || option.value || option.args?.length)) {
                ctx.flags[propName] = getDefaultValue(option);
            } else {
                setFlagValue(true);
            }
        }
        if (option.value) {
            const value = option.value(ctx.flags[propName], previous);
            setFlagValue(value);
        } else if (option.collect) {
            const value = typeof previous !== "undefined" ? Array.isArray(previous) ? previous : [
                previous
            ] : [];
            value.push(ctx.flags[propName]);
            setFlagValue(value);
        }
        optionsMap.set(propName, option);
        opts.option?.(option, ctx.flags[propName]);
        function parseNext(option) {
            if (negate) {
                setFlagValue(false);
                return;
            } else if (!option.args?.length) {
                setFlagValue(undefined);
                return;
            }
            const arg = option.args[optionArgsIndex];
            if (!arg) {
                const flag = next();
                throw new UnknownOptionError(flag, opts.flags ?? []);
            }
            if (!arg.type) {
                arg.type = OptionType.BOOLEAN;
            }
            if (!option.args?.length && arg.type === OptionType.BOOLEAN && arg.optional === undefined) {
                arg.optional = true;
            }
            if (arg.optional) {
                inOptionalArg = true;
            } else if (inOptionalArg) {
                throw new UnexpectedRequiredArgumentError(option.name);
            }
            let result1;
            let increase = false;
            if (arg.list && hasNext(arg)) {
                const parsed = next().split(arg.separator || ",").map((nextValue)=>{
                    const value = parseValue(option, arg, nextValue);
                    if (typeof value === "undefined") {
                        throw new InvalidOptionValueError(option.name, arg.type ?? "?", nextValue);
                    }
                    return value;
                });
                if (parsed?.length) {
                    result1 = parsed;
                }
            } else {
                if (hasNext(arg)) {
                    result1 = parseValue(option, arg, next());
                } else if (arg.optional && arg.type === OptionType.BOOLEAN) {
                    result1 = true;
                }
            }
            if (increase && typeof currentValue === "undefined") {
                argsIndex++;
                if (!arg.variadic) {
                    optionArgsIndex++;
                } else if (option.args[optionArgsIndex + 1]) {
                    throw new UnexpectedArgumentAfterVariadicArgumentError(next());
                }
            }
            if (typeof result1 !== "undefined" && (option.args.length > 1 || arg.variadic)) {
                if (!ctx.flags[propName]) {
                    setFlagValue([]);
                }
                ctx.flags[propName].push(result1);
                if (hasNext(arg)) {
                    parseNext(option);
                }
            } else {
                setFlagValue(result1);
            }
            function hasNext(arg) {
                if (!option.args?.length) {
                    return false;
                }
                const nextValue = currentValue ?? args[argsIndex + 1];
                if (!nextValue) {
                    return false;
                }
                if (option.args.length > 1 && optionArgsIndex >= option.args.length) {
                    return false;
                }
                if (!arg.optional) {
                    return true;
                }
                if (option.equalsSign && arg.optional && !arg.variadic && typeof currentValue === "undefined") {
                    return false;
                }
                if (arg.optional || arg.variadic) {
                    return nextValue[0] !== "-" || typeof currentValue !== "undefined" || arg.type === OptionType.NUMBER && !isNaN(Number(nextValue));
                }
                return false;
            }
            function parseValue(option, arg, value) {
                const result1 = opts.parse ? opts.parse({
                    label: "Option",
                    type: arg.type || OptionType.STRING,
                    name: `--${option.name}`,
                    value
                }) : parseDefaultType(option, arg, value);
                if (typeof result1 !== "undefined") {
                    increase = true;
                }
                return result1;
            }
        }
        function setFlagValue(value) {
            ctx.flags[propName] = value;
            if (ctx.defaults[propName]) {
                delete ctx.defaults[propName];
            }
        }
    }
    return optionsMap;
}
function parseDottedOptions(ctx) {
    ctx.flags = Object.keys(ctx.flags).reduce((result1, key)=>{
        if (~key.indexOf(".")) {
            key.split(".").reduce((result1, subKey, index, parts)=>{
                if (index === parts.length - 1) {
                    result1[subKey] = ctx.flags[key];
                } else {
                    result1[subKey] = result1[subKey] ?? {};
                }
                return result1[subKey];
            }, result1);
        } else {
            result1[key] = ctx.flags[key];
        }
        return result1;
    }, {});
}
function splitFlags(flag) {
    flag = flag.slice(1);
    const normalized = [];
    const index = flag.indexOf("=");
    const flags = (index !== -1 ? flag.slice(0, index) : flag).split("");
    if (isNaN(Number(flag[flag.length - 1]))) {
        flags.forEach((val)=>normalized.push(`-${val}`));
    } else {
        normalized.push(`-${flags.shift()}`);
        if (flags.length) {
            normalized.push(flags.join(""));
        }
    }
    if (index !== -1) {
        normalized[normalized.length - 1] += flag.slice(index);
    }
    return normalized;
}
function parseDefaultType(option, arg, value) {
    const type = arg.type || OptionType.STRING;
    const parseType = DefaultTypes[type];
    if (!parseType) {
        throw new UnknownTypeError(type, Object.keys(DefaultTypes));
    }
    return parseType({
        label: "Option",
        type,
        name: `--${option.name}`,
        value
    });
}
function didYouMeanCommand(command, commands, excludes = []) {
    const commandNames = commands.map((command)=>command.getName()).filter((command)=>!excludes.includes(command));
    return didYouMean(" Did you mean command", command, commandNames);
}
const ARGUMENT_REGEX = /^[<\[].+[\]>]$/;
const ARGUMENT_DETAILS_REGEX = /[<\[:>\]]/;
function splitArguments(args) {
    const parts = args.trim().split(/[, =] */g);
    const typeParts = [];
    while(parts[parts.length - 1] && ARGUMENT_REGEX.test(parts[parts.length - 1])){
        typeParts.unshift(parts.pop());
    }
    const typeDefinition = typeParts.join(" ");
    return {
        flags: parts,
        typeDefinition,
        equalsSign: args.includes("=")
    };
}
function parseArgumentsDefinition(argsDefinition, validate = true, all) {
    const argumentDetails = [];
    let hasOptional = false;
    let hasVariadic = false;
    const parts = argsDefinition.split(/ +/);
    for (const arg of parts){
        if (validate && hasVariadic) {
            throw new UnexpectedArgumentAfterVariadicArgumentError(arg);
        }
        const parts = arg.split(ARGUMENT_DETAILS_REGEX);
        if (!parts[1]) {
            if (all) {
                argumentDetails.push(parts[0]);
            }
            continue;
        }
        const type = parts[2] || OptionType.STRING;
        const details = {
            optional: arg[0] === "[",
            name: parts[1],
            action: parts[3] || type,
            variadic: false,
            list: type ? arg.indexOf(type + "[]") !== -1 : false,
            type
        };
        if (validate && !details.optional && hasOptional) {
            throw new UnexpectedRequiredArgumentError(details.name);
        }
        if (arg[0] === "[") {
            hasOptional = true;
        }
        if (details.name.length > 3) {
            const istVariadicLeft = details.name.slice(0, 3) === "...";
            const istVariadicRight = details.name.slice(-3) === "...";
            hasVariadic = details.variadic = istVariadicLeft || istVariadicRight;
            if (istVariadicLeft) {
                details.name = details.name.slice(3);
            } else if (istVariadicRight) {
                details.name = details.name.slice(0, -3);
            }
        }
        argumentDetails.push(details);
    }
    return argumentDetails;
}
function dedent(str) {
    const lines = str.split(/\r?\n|\r/g);
    let text = "";
    let indent = 0;
    for (const line of lines){
        if (text || line.trim()) {
            if (!text) {
                text = line.trimStart();
                indent = line.length - text.length;
            } else {
                text += line.slice(indent);
            }
            text += "\n";
        }
    }
    return text.trimEnd();
}
function getDescription(description, __short) {
    return __short ? description.trim().split("\n", 1)[0].trim() : dedent(description);
}
class CommandError extends Error {
    constructor(message){
        super(message);
        Object.setPrototypeOf(this, CommandError.prototype);
    }
}
class ValidationError1 extends CommandError {
    exitCode;
    cmd;
    constructor(message, { exitCode } = {}){
        super(message);
        Object.setPrototypeOf(this, ValidationError1.prototype);
        this.exitCode = exitCode ?? 2;
    }
}
class DuplicateOptionNameError extends CommandError {
    constructor(optionName, commandName){
        super(`An option with name '${bold1(getFlag(optionName))}' is already registered on command '${bold1(commandName)}'. If it is intended to override the option, set the '${bold1("override")}' option of the '${bold1("option")}' method to true.`);
        Object.setPrototypeOf(this, DuplicateOptionNameError.prototype);
    }
}
class MissingCommandNameCompletionsError extends CommandError {
    constructor(shell){
        super(`Failed to generate shell completions. Missing main command name. Use '${bold1('cmd.name("<comand-name>")')}' to set the name of the main command or use the '${bold1("--name")}' option from the '${bold1("completions")}' command to set the cli name: '${bold1(`<command> completions ${shell} --name <cli-name>`)}'.`);
        Object.setPrototypeOf(this, MissingCommandNameCompletionsError.prototype);
    }
}
class MissingCommandNameError extends CommandError {
    constructor(){
        super("Missing command name.");
        Object.setPrototypeOf(this, MissingCommandNameError.prototype);
    }
}
class DuplicateCommandNameError extends CommandError {
    constructor(name){
        super(`Duplicate command name "${name}".`);
        Object.setPrototypeOf(this, DuplicateCommandNameError.prototype);
    }
}
class DuplicateCommandAliasError extends CommandError {
    constructor(alias){
        super(`Duplicate command alias "${alias}".`);
        Object.setPrototypeOf(this, DuplicateCommandAliasError.prototype);
    }
}
class CommandNotFoundError extends CommandError {
    constructor(name, commands, excluded){
        super(`Unknown command "${name}".${didYouMeanCommand(name, commands, excluded)}`);
        Object.setPrototypeOf(this, CommandNotFoundError.prototype);
    }
}
class DuplicateTypeError extends CommandError {
    constructor(name){
        super(`Type with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateTypeError.prototype);
    }
}
class DuplicateCompletionError extends CommandError {
    constructor(name){
        super(`Completion with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateCompletionError.prototype);
    }
}
class DuplicateExampleError extends CommandError {
    constructor(name){
        super(`Example with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateExampleError.prototype);
    }
}
class DuplicateEnvVarError extends CommandError {
    constructor(name){
        super(`Environment variable with name "${name}" already exists.`);
        Object.setPrototypeOf(this, DuplicateEnvVarError.prototype);
    }
}
class MissingRequiredEnvVarError extends ValidationError1 {
    constructor(envVar){
        super(`Missing required environment variable "${envVar.names[0]}".`);
        Object.setPrototypeOf(this, MissingRequiredEnvVarError.prototype);
    }
}
class TooManyEnvVarValuesError extends CommandError {
    constructor(name){
        super(`An environment variable can only have one value, but "${name}" has more than one.`);
        Object.setPrototypeOf(this, TooManyEnvVarValuesError.prototype);
    }
}
class UnexpectedOptionalEnvVarValueError extends CommandError {
    constructor(name){
        super(`An environment variable cannot have an optional value, but "${name}" is defined as optional.`);
        Object.setPrototypeOf(this, UnexpectedOptionalEnvVarValueError.prototype);
    }
}
class UnexpectedVariadicEnvVarValueError extends CommandError {
    constructor(name){
        super(`An environment variable cannot have an variadic value, but "${name}" is defined as variadic.`);
        Object.setPrototypeOf(this, UnexpectedVariadicEnvVarValueError.prototype);
    }
}
class DefaultCommandNotFoundError extends CommandError {
    constructor(name, commands){
        super(`Default command "${name}" not found.${didYouMeanCommand(name, commands)}`);
        Object.setPrototypeOf(this, DefaultCommandNotFoundError.prototype);
    }
}
class CommandExecutableNotFoundError extends CommandError {
    constructor(name){
        super(`Command executable not found: ${name}`);
        Object.setPrototypeOf(this, CommandExecutableNotFoundError.prototype);
    }
}
class UnknownCompletionCommandError extends CommandError {
    constructor(name, commands){
        super(`Auto-completion failed. Unknown command "${name}".${didYouMeanCommand(name, commands)}`);
        Object.setPrototypeOf(this, UnknownCompletionCommandError.prototype);
    }
}
class UnknownCommandError extends ValidationError1 {
    constructor(name, commands, excluded){
        super(`Unknown command "${name}".${didYouMeanCommand(name, commands, excluded)}`);
        Object.setPrototypeOf(this, UnknownCommandError.prototype);
    }
}
class NoArgumentsAllowedError extends ValidationError1 {
    constructor(name){
        super(`No arguments allowed for command "${name}".`);
        Object.setPrototypeOf(this, NoArgumentsAllowedError.prototype);
    }
}
class MissingArgumentsError extends ValidationError1 {
    constructor(names){
        super(`Missing argument(s): ${names.join(", ")}`);
        Object.setPrototypeOf(this, MissingArgumentsError.prototype);
    }
}
class MissingArgumentError extends ValidationError1 {
    constructor(name){
        super(`Missing argument: ${name}`);
        Object.setPrototypeOf(this, MissingArgumentError.prototype);
    }
}
class TooManyArgumentsError extends ValidationError1 {
    constructor(args){
        super(`Too many arguments: ${args.join(" ")}`);
        Object.setPrototypeOf(this, TooManyArgumentsError.prototype);
    }
}
class Type {
}
class BooleanType extends Type {
    parse(type) {
        return __boolean(type);
    }
    complete() {
        return [
            "true",
            "false"
        ];
    }
}
class StringType extends Type {
    parse(type) {
        return string(type);
    }
}
class FileType extends StringType {
    constructor(){
        super();
    }
}
class IntegerType extends Type {
    parse(type) {
        return integer(type);
    }
}
class NumberType extends Type {
    parse(type) {
        return number(type);
    }
}
const border = {
    top: "─",
    topMid: "┬",
    topLeft: "┌",
    topRight: "┐",
    bottom: "─",
    bottomMid: "┴",
    bottomLeft: "└",
    bottomRight: "┘",
    left: "│",
    leftMid: "├",
    mid: "─",
    midMid: "┼",
    right: "│",
    rightMid: "┤",
    middle: "│"
};
class Cell {
    value;
    options;
    get length() {
        return this.toString().length;
    }
    static from(value) {
        let cell;
        if (value instanceof Cell) {
            cell = new this(value.getValue());
            cell.options = {
                ...value.options
            };
        } else {
            cell = new this(value);
        }
        return cell;
    }
    constructor(value){
        this.value = value;
        this.options = {};
    }
    toString() {
        return this.value.toString();
    }
    getValue() {
        return this.value;
    }
    setValue(value) {
        this.value = value;
        return this;
    }
    clone(value) {
        return Cell.from(value ?? this);
    }
    border(enable = true, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    colSpan(span, override = true) {
        if (override || typeof this.options.colSpan === "undefined") {
            this.options.colSpan = span;
        }
        return this;
    }
    rowSpan(span, override = true) {
        if (override || typeof this.options.rowSpan === "undefined") {
            this.options.rowSpan = span;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    getColSpan() {
        return typeof this.options.colSpan === "number" && this.options.colSpan > 0 ? this.options.colSpan : 1;
    }
    getRowSpan() {
        return typeof this.options.rowSpan === "number" && this.options.rowSpan > 0 ? this.options.rowSpan : 1;
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class Column {
    static from(options) {
        const opts = options instanceof Column ? options.opts : options;
        return new Column().options(opts);
    }
    opts = {};
    options(options) {
        Object.assign(this.opts, options);
        return this;
    }
    minWidth(width) {
        this.opts.minWidth = width;
        return this;
    }
    maxWidth(width) {
        this.opts.maxWidth = width;
        return this;
    }
    border(border = true) {
        this.opts.border = border;
        return this;
    }
    padding(padding) {
        this.opts.padding = padding;
        return this;
    }
    align(direction) {
        this.opts.align = direction;
        return this;
    }
    getMinWidth() {
        return this.opts.minWidth;
    }
    getMaxWidth() {
        return this.opts.maxWidth;
    }
    getBorder() {
        return this.opts.border;
    }
    getPadding() {
        return this.opts.padding;
    }
    getAlign() {
        return this.opts.align;
    }
}
const __default = JSON.parse("{\n  \"UNICODE_VERSION\": \"15.0.0\",\n  \"tables\": [\n    {\n      \"d\": \"AAECAwQFBgcICQoLDA0OAw8DDwkQCRESERIA\",\n      \"r\": \"AQEBAgEBAQEBAQEBAQEBBwEHAVABBwcBBwF4\"\n    },\n    {\n      \"d\": \"AAECAwQFBgcGCAYJCgsMDQ4PEAYREhMUBhUWFxgZGhscHR4fICEiIyIkJSYnKCkqJSssLS4vMDEyMzQ1Njc4OToGOzwKBj0GPj9AQUIGQwZEBkVGR0hJSktMTQZOBgoGT1BRUlNUVVZXWFkGWgZbBlxdXl1fYGFiY2RlZmdoBmlqBmsGAQZsBm1uO29wcXI7czt0dXZ3OwY7eHkGent8Bn0Gfn+AgYKDhIWGBoc7iAZdO4kGiosGAXGMBo0GjgaPBpAGkQaSBpMGlJUGlpcGmJmam5ydnp+gLgahLKIGo6SlpganqKmqqwasBq0Grq8GsLGyswa0BrUGtre4Brm6uwZHvAa9vga/wME7wjvDxAbFO8bHO8gGyQbKywbMzQbOBs/Q0QbSBr8GvgbT1AbUBtUG1gbXBtjZ2tsG3N0G3t/g4eLjO+Tl5ufoO+k76gbrBuztOwbu7/AGO+XxCgYKCwZd8g==\",\n      \"r\": \"AQEBAQEBAQEBAQEBAQEBAQEBAQMBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQEBAQECBQEOAQEBAQEBAQEBAwEBAQEBAQEBAQIBAwEIAQEBAQEBAQEBAQEBAQIBAQEBAQEBAQEBAQEBAQEBDQEBBQEBAQEBAgEBAwEBAQEBAQEBAQEBbQHaAQEFAQEBBAECAQEBAQEBAQEBAwGuASFkCAELAQEBAQEBAQEHAQMBAQEaAQIBCAEFAQEBAQEBAQEBAQEBAQEBAQEBAQECAQEBAQIBAQEBAQEBAwEDAQEBAQEBAQUBAQEBAQEBBAEBAVIBAdkBARABAQFfARMBAYoBBAEBBQEmAUkBAQcBAQIBHgEBARUBAQEBAQUBAQcBDwEBARoBAgEBAQEBAQECAQEBAQEBAQEBAQEBAQEBAQMBBAEBAgEBAQEUfwEBAQIDAXj/AQ==\"\n    },\n    {\n      \"d\": \"AFUVAF3Xd3X/93//VXVVV9VX9V91f1/31X93XVXdVdVV9dVV/VVX1X9X/131VfXVVXV3V1VdVV1V1/1dV1X/3VUAVf3/3/9fVf3/3/9fVV1V/11VFQBQVQEAEEEQVQBQVQBAVFUVAFVUVQUAEAAUBFBVFVFVAEBVBQBUVRUAVVFVBRAAAVBVAVVQVQBVBQBAVUVUAQBUUQEAVQVVUVVUAVRVUVUFVUVBVVRBFRRQUVVQUVUBEFRRVQVVBQBRVRQBVFVRVUFVBVVFVVRVUVVUVQRUBQRQVUFVBVVFVVBVBVVQVRVUAVRVUVUFVVFVRVUFRFVRAEBVFQBAVVEAVFUAQFVQVRFRVQEAQAAEVQEAAQBUVUVVAQQAQVVQBVRVAVRVRUFVUVVRVaoAVQFVBVRVBVUFVQVVEABQVUUBAFVRVRUAVUFVUVVAFVRVRVUBVRUUVUUAQEQBAFQVABRVAEBVAFUEQFRFVRUAVVBVBVAQUFVFUBFQVQAFVUAABABUUVVUUFUVANd/X3//BUD3XdV1VQAEAFVXVdX9V1VXVQBUVdVdVdV1VX111VXVV9V//1X/X1VdVf9fVV9VdVdV1VX31dfVXXX9193/d1X/VV9VV3VVX//1VfVVXVVdVdVVdVWlVWlVqVaWVf/f/1X/Vf/1X1Xf/19V9VVf9df1X1X1X1XVVWlVfV31VVpVd1V3VapV33/fVZVVlVX1WVWlVelV+v/v//7/31Xv/6/77/tVWaVVVlVdVWaVmlX1/1WpVVZVlVWVVlVW+V9VFVBVAKqaqlWqWlWqVaoKoKpqqapqgapVqaqpqmqqVapqqv+qVqpqVRVAAFBVBVVQVUUVVUFVVFVQVQBQVRVVBQBQVRUAUFWqVkBVFQVQVVFVAUBBVRVVVFVUVQQUVAVRVVBVRVVRVFFVqlVFVQCqWlUAqmqqaqpVqlZVqmpVAV1VUVVUVQVAVQFBVQBVQBVVQVUAVRVUVQFVBQBUVQVQVVFVAEBVFFRVFVBVFUBBUUVVUVVAVRUAAQBUVRVVUFUFAEBVARRVFVAEVUVVFQBAVVRVBQBUAFRVAAVEVUVVFQBEFQRVBVBVEFRVUFUVAEARVFUVUQAQVQEFEABVFQBBVRVEFVUABVVUVQEAQFUVABRAVRVVAUABVQUAQFBVAEAAEFUFAAUABEFVAUBFEAAQVVARVRVUVVBVBUBVRFVUFQBQVQBUVQBAVRVVFUBVqlRVWlWqVapaVapWVaqpqmmqalVlVWpZVapVqlVBAFUAUABAVRVQVRUAQAEAVQVQVQVUVQBAFQBUVVFVVFUVAAEAVQBAABQAEARAVUVVAFUAQFUAQFVWVZVV/39V/1//X1X/76uq6v9XVWpVqlWqVlVaVapaVapWVamqmqqmqlWqapWqVapWqmqmqpaqWlWVaqpVZVVpVVZVlapVqlpVVmqpVapVlVZVqlZVqlVWVapqqpqqVapWqlZVqpqqWlWlqlWqVlWqVlVRVQD/Xw==\",\n      \"r\": \"CBcBCAEBAQEBAQEBAQECAQEBAQEBAQEBAQEBAQMBAQECAQEBAQEBAQEBAQEBBAEBGAEDAQwBAwEIAQEBAQEBAQgcCAEDAQEBAQEDAQEBDQEDEAELAQEBEQEKAQEBDgEBAgIBAQoBBQQBCAEBAQEBAQEHAQEHBgEWAQIBDQECAgEFAQECAgEKAQ0BAQIKAQ0BDQEBAQEBAQEBAgEHAQ4BAQEBAQQBBgEBDgEBAQEBAQcBAQIBAQEBBAEFAQEBDgEBAQEBAQECAQcBDwECAQwCDQEBAQEBAQECAQgBAQEEAQcBDQEBAQEBAQQBBwERAQEBARYBAQECAQEBGAECAQIBARIBBgEBDQECAQEBAQECAQgBAQEZAQEBAgYBAQEDAQECAQEBAQMBCBgIBwEMAQEGAQcBBwEQAQEBAQEBAgIBCgEBDQEIAQ0BAQEBAQEBBgEBDgEBAQEBAQEBAgEMBwEMAQwBAQEBCQECAwEHAQEBAQ0BAQEBDgIBBgEDAQEBAQEBAQMBAQEBAgEBAQEBAQEBCAEBAgEBAQEBAQkBCAgBAwECAQEBAgEBAQkBAQEBAwECAQMBAQIBBwEFAQEDAQYBAQEBAgEBAQEBAQEBAQECAgEDAQECBAIDAgIBBQEEAQEBAwEPAQEBCyIBCAEJAwQBAQIBAQEBAgECAQEBAQMBAQEBAwEBAQEBAQEBAQgBAQMDAgEBAwEEAQIBAQEBBAEBAQEBAQECAQEBAQEBAQEBAQEHAQQBAwEBAQcBAgUBBgECAQYBAQwBAQEUAQELCAYBFgMFAQYDAQoBAQMBARQBAQkBAQoBBgEVAwsBCgIPAQ0BGQEBAgEHARQBAwIBBgEBAQUBBgQBAgEJAQEBBQECAQMHAQELAQECCQEQAQECAgECAQsBDAEBAQEBCgEBAQsBAQEECQ4BCAQCAQEECAEEAQEFCAEPAQEEAQEPAQgBFAEBAQEBAQEKAQEJAQ8BEAEBEwEBAQIBCwEBDgENAwEKAQEBAQELAQEBAQECAQwBCAEBAQEBDgEDAQwBAQECAQEXAQEBAQEHAgEBBQEIAQEBAQEQAgEBBQEUAQEBAQEbAQEBAQEGARQBAQEBARkBAQEBCQEBAQEQAQIBDwEBARQBAQEBBwEBAQkBAQEBAQECAQEBCwECAQEVAQEBAQQBBQEBAQEOAQEBAQEBEgEBFgEBAgEMAQEBAQ8BAQMBFgEBDgEBBQEPAQETAQECAQMOAgUBCgIBGQEBAQEIAQMBBwEBAwECEwgBAQcLAQUBFwEBAQEDAQEBBwEBBAEBDg0BAQwBAQEDAQQBAQEDBAEBBAEBAQEBEAEPAQgBAQsBAQ4BEQEMAgEBBwEOAQEHAQEBAQQBBAEDCwECAQEBAwEBBggBAgEBAREBBQMKAQEBAwQCEQEBHgEPAQIBAQYEAQYBAwEUAQUMAQEBAQEBAQECAQEBAgEIAwEBBgsBAgEODAMBAgEBCwEBAQEBAwECAQECAQEBBwgPAQ==\"\n    }\n  ]\n}");
class AssertionError1 extends Error {
    name = "AssertionError";
    constructor(message){
        super(message);
    }
}
function assert1(expr, msg = "") {
    if (!expr) {
        throw new AssertionError1(msg);
    }
}
function runLengthDecode({ d, r }) {
    const data = atob(d);
    const runLengths = atob(r);
    let out = "";
    for (const [i1, ch] of [
        ...runLengths
    ].entries()){
        out += data[i1].repeat(ch.codePointAt(0));
    }
    return Uint8Array.from([
        ...out
    ].map((x)=>x.codePointAt(0)));
}
let tables = null;
function lookupWidth(cp) {
    if (!tables) tables = __default.tables.map(runLengthDecode);
    const t1Offset = tables[0][cp >> 13 & 0xff];
    const t2Offset = tables[1][128 * t1Offset + (cp >> 6 & 0x7f)];
    const packedWidths = tables[2][16 * t2Offset + (cp >> 2 & 0xf)];
    const width = packedWidths >> 2 * (cp & 0b11) & 0b11;
    return width === 3 ? 1 : width;
}
const cache = new Map();
function charWidth(ch) {
    if (cache.has(ch)) return cache.get(ch);
    const cp = ch.codePointAt(0);
    let v = null;
    if (cp < 0x7f) {
        v = cp >= 0x20 ? 1 : cp === 0 ? 0 : null;
    } else if (cp >= 0xa0) {
        v = lookupWidth(cp);
    } else {
        v = null;
    }
    cache.set(ch, v);
    return v;
}
function unicodeWidth(str) {
    return [
        ...str
    ].map((ch)=>charWidth(ch) ?? 0).reduce((a, b)=>a + b, 0);
}
const strLength = (str)=>{
    return unicodeWidth(stripColor1(str));
};
function consumeWords(length, content) {
    let consumed = "";
    const words = content.split("\n")[0]?.split(/ /g);
    for(let i1 = 0; i1 < words.length; i1++){
        const word = words[i1];
        if (consumed) {
            const nextLength = strLength(word);
            const consumedLength = strLength(consumed);
            if (consumedLength + nextLength >= length) {
                break;
            }
        }
        consumed += (i1 > 0 ? " " : "") + word;
    }
    return consumed;
}
function longest(index, rows, maxWidth) {
    const cellLengths = rows.map((row)=>{
        const cell = row[index];
        const cellValue = cell instanceof Cell && cell.getColSpan() > 1 ? "" : cell?.toString() || "";
        return cellValue.split("\n").map((line)=>{
            const str = typeof maxWidth === "undefined" ? line : consumeWords(maxWidth, line);
            return strLength(str) || 0;
        });
    }).flat();
    return Math.max(...cellLengths);
}
class Row extends Array {
    options = {};
    static from(cells) {
        const row = new this(...cells);
        if (cells instanceof Row) {
            row.options = {
                ...cells.options
            };
        }
        return row;
    }
    clone() {
        const row = new Row(...this.map((cell)=>cell instanceof Cell ? cell.clone() : cell));
        row.options = {
            ...this.options
        };
        return row;
    }
    border(enable = true, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasBorder() {
        return this.getBorder() || this.some((cell)=>cell instanceof Cell && cell.getBorder());
    }
    getAlign() {
        return this.options.align ?? "left";
    }
}
class TableLayout {
    table;
    options;
    constructor(table, options){
        this.table = table;
        this.options = options;
    }
    toString() {
        const opts = this.createLayout();
        return opts.rows.length ? this.renderRows(opts) : "";
    }
    createLayout() {
        Object.keys(this.options.chars).forEach((key)=>{
            if (typeof this.options.chars[key] !== "string") {
                this.options.chars[key] = "";
            }
        });
        const hasBodyBorder = this.table.getBorder() || this.table.hasBodyBorder();
        const hasHeaderBorder = this.table.hasHeaderBorder();
        const hasBorder = hasHeaderBorder || hasBodyBorder;
        const rows = this.#getRows();
        const columns = Math.max(...rows.map((row)=>row.length));
        for(let rowIndex = 0; rowIndex < rows.length; rowIndex++){
            const row = rows[rowIndex];
            const length = row.length;
            if (length < columns) {
                const diff = columns - length;
                for(let i1 = 0; i1 < diff; i1++){
                    row.push(this.createCell(null, row, rowIndex, length + i1));
                }
            }
        }
        const padding = [];
        const width = [];
        for(let colIndex = 0; colIndex < columns; colIndex++){
            const column = this.options.columns.at(colIndex);
            const minColWidth = column?.getMinWidth() ?? (Array.isArray(this.options.minColWidth) ? this.options.minColWidth[colIndex] : this.options.minColWidth);
            const maxColWidth = column?.getMaxWidth() ?? (Array.isArray(this.options.maxColWidth) ? this.options.maxColWidth[colIndex] : this.options.maxColWidth);
            const colWidth = longest(colIndex, rows, maxColWidth);
            width[colIndex] = Math.min(maxColWidth, Math.max(minColWidth, colWidth));
            padding[colIndex] = column?.getPadding() ?? (Array.isArray(this.options.padding) ? this.options.padding[colIndex] : this.options.padding);
        }
        return {
            padding,
            width,
            rows,
            columns,
            hasBorder,
            hasBodyBorder,
            hasHeaderBorder
        };
    }
    #getRows() {
        const header = this.table.getHeader();
        const rows = header ? [
            header,
            ...this.table
        ] : this.table.slice();
        const hasSpan = rows.some((row)=>row.some((cell)=>cell instanceof Cell && (cell.getColSpan() > 1 || cell.getRowSpan() > 1)));
        if (hasSpan) {
            return this.spanRows(rows);
        }
        return rows.map((row, rowIndex)=>{
            const newRow = this.createRow(row);
            for(let colIndex = 0; colIndex < row.length; colIndex++){
                newRow[colIndex] = this.createCell(row[colIndex], newRow, rowIndex, colIndex);
            }
            return newRow;
        });
    }
    spanRows(rows) {
        const rowSpan = [];
        let colSpan = 1;
        let rowIndex = -1;
        while(true){
            rowIndex++;
            if (rowIndex === rows.length && rowSpan.every((span)=>span === 1)) {
                break;
            }
            const row = rows[rowIndex] = this.createRow(rows[rowIndex] || []);
            let colIndex = -1;
            while(true){
                colIndex++;
                if (colIndex === row.length && colIndex === rowSpan.length && colSpan === 1) {
                    break;
                }
                if (colSpan > 1) {
                    colSpan--;
                    rowSpan[colIndex] = rowSpan[colIndex - 1];
                    row.splice(colIndex, this.getDeleteCount(rows, rowIndex, colIndex), row[colIndex - 1]);
                    continue;
                }
                if (rowSpan[colIndex] > 1) {
                    rowSpan[colIndex]--;
                    rows[rowIndex].splice(colIndex, this.getDeleteCount(rows, rowIndex, colIndex), rows[rowIndex - 1][colIndex]);
                    continue;
                }
                const cell = row[colIndex] = this.createCell(row[colIndex] || null, row, rowIndex, colIndex);
                colSpan = cell.getColSpan();
                rowSpan[colIndex] = cell.getRowSpan();
            }
        }
        return rows;
    }
    getDeleteCount(rows, rowIndex, colIndex) {
        return colIndex <= rows[rowIndex].length - 1 && typeof rows[rowIndex][colIndex] === "undefined" ? 1 : 0;
    }
    createRow(row) {
        return Row.from(row).border(this.table.getBorder(), false).align(this.table.getAlign(), false);
    }
    createCell(cell, row, rowIndex, colIndex) {
        const column = this.options.columns.at(colIndex);
        const isHeaderRow = this.isHeaderRow(rowIndex);
        return Cell.from(cell ?? "").border((isHeaderRow ? null : column?.getBorder()) ?? row.getBorder(), false).align((isHeaderRow ? null : column?.getAlign()) ?? row.getAlign(), false);
    }
    isHeaderRow(rowIndex) {
        return rowIndex === 0 && this.table.getHeader() !== undefined;
    }
    renderRows(opts) {
        let result1 = "";
        const rowSpan = new Array(opts.columns).fill(1);
        for(let rowIndex = 0; rowIndex < opts.rows.length; rowIndex++){
            result1 += this.renderRow(rowSpan, rowIndex, opts);
        }
        return result1.slice(0, -1);
    }
    renderRow(rowSpan, rowIndex, opts, isMultiline) {
        const row = opts.rows[rowIndex];
        const prevRow = opts.rows[rowIndex - 1];
        const nextRow = opts.rows[rowIndex + 1];
        let result1 = "";
        let colSpan = 1;
        if (!isMultiline && rowIndex === 0 && row.hasBorder()) {
            result1 += this.renderBorderRow(undefined, row, rowSpan, opts);
        }
        let isMultilineRow = false;
        result1 += " ".repeat(this.options.indent || 0);
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (colSpan > 1) {
                colSpan--;
                rowSpan[colIndex] = rowSpan[colIndex - 1];
                continue;
            }
            result1 += this.renderCell(colIndex, row, opts);
            if (rowSpan[colIndex] > 1) {
                if (!isMultiline) {
                    rowSpan[colIndex]--;
                }
            } else if (!prevRow || prevRow[colIndex] !== row[colIndex]) {
                rowSpan[colIndex] = row[colIndex].getRowSpan();
            }
            colSpan = row[colIndex].getColSpan();
            if (rowSpan[colIndex] === 1 && row[colIndex].length) {
                isMultilineRow = true;
            }
        }
        if (opts.columns > 0) {
            if (row[opts.columns - 1].getBorder()) {
                result1 += this.options.chars.right;
            } else if (opts.hasBorder) {
                result1 += " ";
            }
        }
        result1 += "\n";
        if (isMultilineRow) {
            return result1 + this.renderRow(rowSpan, rowIndex, opts, isMultilineRow);
        }
        if (opts.rows.length > 1 && (rowIndex === 0 && opts.hasHeaderBorder || rowIndex < opts.rows.length - 1 && opts.hasBodyBorder)) {
            result1 += this.renderBorderRow(row, nextRow, rowSpan, opts);
        }
        if (rowIndex === opts.rows.length - 1 && row.hasBorder()) {
            result1 += this.renderBorderRow(row, undefined, rowSpan, opts);
        }
        return result1;
    }
    renderCell(colIndex, row, opts, noBorder) {
        let result1 = "";
        const prevCell = row[colIndex - 1];
        const cell = row[colIndex];
        if (!noBorder) {
            if (colIndex === 0) {
                if (cell.getBorder()) {
                    result1 += this.options.chars.left;
                } else if (opts.hasBorder) {
                    result1 += " ";
                }
            } else {
                if (cell.getBorder() || prevCell?.getBorder()) {
                    result1 += this.options.chars.middle;
                } else if (opts.hasBorder) {
                    result1 += " ";
                }
            }
        }
        let maxLength = opts.width[colIndex];
        const colSpan = cell.getColSpan();
        if (colSpan > 1) {
            for(let o = 1; o < colSpan; o++){
                maxLength += opts.width[colIndex + o] + opts.padding[colIndex + o];
                if (opts.hasBorder) {
                    maxLength += opts.padding[colIndex + o] + 1;
                }
            }
        }
        const { current, next } = this.renderCellValue(cell, maxLength);
        row[colIndex].setValue(next.getValue());
        if (opts.hasBorder) {
            result1 += " ".repeat(opts.padding[colIndex]);
        }
        result1 += current;
        if (opts.hasBorder || colIndex < opts.columns - 1) {
            result1 += " ".repeat(opts.padding[colIndex]);
        }
        return result1;
    }
    renderCellValue(cell, maxLength) {
        const length = Math.min(maxLength, strLength(cell.toString()));
        let words = consumeWords(length, cell.toString());
        const breakWord = strLength(words) > length;
        if (breakWord) {
            words = words.slice(0, length);
        }
        const next = cell.toString().slice(words.length + (breakWord ? 0 : 1));
        const fillLength = maxLength - strLength(words);
        const align = cell.getAlign();
        let current;
        if (fillLength === 0) {
            current = words;
        } else if (align === "left") {
            current = words + " ".repeat(fillLength);
        } else if (align === "center") {
            current = " ".repeat(Math.floor(fillLength / 2)) + words + " ".repeat(Math.ceil(fillLength / 2));
        } else if (align === "right") {
            current = " ".repeat(fillLength) + words;
        } else {
            throw new Error("Unknown direction: " + align);
        }
        return {
            current,
            next: cell.clone(next)
        };
    }
    renderBorderRow(prevRow, nextRow, rowSpan, opts) {
        let result1 = "";
        let colSpan = 1;
        for(let colIndex = 0; colIndex < opts.columns; colIndex++){
            if (rowSpan[colIndex] > 1) {
                if (!nextRow) {
                    throw new Error("invalid layout");
                }
                if (colSpan > 1) {
                    colSpan--;
                    continue;
                }
            }
            result1 += this.renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts);
            colSpan = nextRow?.[colIndex].getColSpan() ?? 1;
        }
        return result1.length ? " ".repeat(this.options.indent) + result1 + "\n" : "";
    }
    renderBorderCell(colIndex, prevRow, nextRow, rowSpan, opts) {
        const a1 = prevRow?.[colIndex - 1];
        const a2 = nextRow?.[colIndex - 1];
        const b1 = prevRow?.[colIndex];
        const b2 = nextRow?.[colIndex];
        const a1Border = !!a1?.getBorder();
        const a2Border = !!a2?.getBorder();
        const b1Border = !!b1?.getBorder();
        const b2Border = !!b2?.getBorder();
        const hasColSpan = (cell)=>(cell?.getColSpan() ?? 1) > 1;
        const hasRowSpan = (cell)=>(cell?.getRowSpan() ?? 1) > 1;
        let result1 = "";
        if (colIndex === 0) {
            if (rowSpan[colIndex] > 1) {
                if (b1Border) {
                    result1 += this.options.chars.left;
                } else {
                    result1 += " ";
                }
            } else if (b1Border && b2Border) {
                result1 += this.options.chars.leftMid;
            } else if (b1Border) {
                result1 += this.options.chars.bottomLeft;
            } else if (b2Border) {
                result1 += this.options.chars.topLeft;
            } else {
                result1 += " ";
            }
        } else if (colIndex < opts.columns) {
            if (a1Border && b2Border || b1Border && a2Border) {
                const a1ColSpan = hasColSpan(a1);
                const a2ColSpan = hasColSpan(a2);
                const b1ColSpan = hasColSpan(b1);
                const b2ColSpan = hasColSpan(b2);
                const a1RowSpan = hasRowSpan(a1);
                const a2RowSpan = hasRowSpan(a2);
                const b1RowSpan = hasRowSpan(b1);
                const b2RowSpan = hasRowSpan(b2);
                const hasAllBorder = a1Border && b2Border && b1Border && a2Border;
                const hasAllRowSpan = a1RowSpan && b1RowSpan && a2RowSpan && b2RowSpan;
                const hasAllColSpan = a1ColSpan && b1ColSpan && a2ColSpan && b2ColSpan;
                if (hasAllRowSpan && hasAllBorder) {
                    result1 += this.options.chars.middle;
                } else if (hasAllColSpan && hasAllBorder && a1 === b1 && a2 === b2) {
                    result1 += this.options.chars.mid;
                } else if (a1ColSpan && b1ColSpan && a1 === b1) {
                    result1 += this.options.chars.topMid;
                } else if (a2ColSpan && b2ColSpan && a2 === b2) {
                    result1 += this.options.chars.bottomMid;
                } else if (a1RowSpan && a2RowSpan && a1 === a2) {
                    result1 += this.options.chars.leftMid;
                } else if (b1RowSpan && b2RowSpan && b1 === b2) {
                    result1 += this.options.chars.rightMid;
                } else {
                    result1 += this.options.chars.midMid;
                }
            } else if (a1Border && b1Border) {
                if (hasColSpan(a1) && hasColSpan(b1) && a1 === b1) {
                    result1 += this.options.chars.bottom;
                } else {
                    result1 += this.options.chars.bottomMid;
                }
            } else if (b1Border && b2Border) {
                if (rowSpan[colIndex] > 1) {
                    result1 += this.options.chars.left;
                } else {
                    result1 += this.options.chars.leftMid;
                }
            } else if (b2Border && a2Border) {
                if (hasColSpan(a2) && hasColSpan(b2) && a2 === b2) {
                    result1 += this.options.chars.top;
                } else {
                    result1 += this.options.chars.topMid;
                }
            } else if (a1Border && a2Border) {
                if (hasRowSpan(a1) && a1 === a2) {
                    result1 += this.options.chars.right;
                } else {
                    result1 += this.options.chars.rightMid;
                }
            } else if (a1Border) {
                result1 += this.options.chars.bottomRight;
            } else if (b1Border) {
                result1 += this.options.chars.bottomLeft;
            } else if (a2Border) {
                result1 += this.options.chars.topRight;
            } else if (b2Border) {
                result1 += this.options.chars.topLeft;
            } else {
                result1 += " ";
            }
        }
        const length = opts.padding[colIndex] + opts.width[colIndex] + opts.padding[colIndex];
        if (rowSpan[colIndex] > 1 && nextRow) {
            result1 += this.renderCell(colIndex, nextRow, opts, true);
            if (nextRow[colIndex] === nextRow[nextRow.length - 1]) {
                if (b1Border) {
                    result1 += this.options.chars.right;
                } else {
                    result1 += " ";
                }
                return result1;
            }
        } else if (b1Border && b2Border) {
            result1 += this.options.chars.mid.repeat(length);
        } else if (b1Border) {
            result1 += this.options.chars.bottom.repeat(length);
        } else if (b2Border) {
            result1 += this.options.chars.top.repeat(length);
        } else {
            result1 += " ".repeat(length);
        }
        if (colIndex === opts.columns - 1) {
            if (b1Border && b2Border) {
                result1 += this.options.chars.rightMid;
            } else if (b1Border) {
                result1 += this.options.chars.bottomRight;
            } else if (b2Border) {
                result1 += this.options.chars.topRight;
            } else {
                result1 += " ";
            }
        }
        return result1;
    }
}
class Table extends Array {
    static _chars = {
        ...border
    };
    options = {
        indent: 0,
        border: false,
        maxColWidth: Infinity,
        minColWidth: 0,
        padding: 1,
        chars: {
            ...Table._chars
        },
        columns: []
    };
    headerRow;
    static from(rows) {
        const table = new this(...rows);
        if (rows instanceof Table) {
            table.options = {
                ...rows.options
            };
            table.headerRow = rows.headerRow ? Row.from(rows.headerRow) : undefined;
        }
        return table;
    }
    static fromJson(rows) {
        return new this().fromJson(rows);
    }
    static chars(chars) {
        Object.assign(this._chars, chars);
        return this;
    }
    static render(rows) {
        Table.from(rows).render();
    }
    fromJson(rows) {
        this.header(Object.keys(rows[0]));
        this.body(rows.map((row)=>Object.values(row)));
        return this;
    }
    columns(columns) {
        this.options.columns = columns.map((column)=>column instanceof Column ? column : Column.from(column));
        return this;
    }
    column(index, column) {
        if (column instanceof Column) {
            this.options.columns[index] = column;
        } else if (this.options.columns[index]) {
            this.options.columns[index].options(column);
        } else {
            this.options.columns[index] = Column.from(column);
        }
        return this;
    }
    header(header) {
        this.headerRow = header instanceof Row ? header : Row.from(header);
        return this;
    }
    body(rows) {
        this.length = 0;
        this.push(...rows);
        return this;
    }
    clone() {
        const table = new Table(...this.map((row)=>row instanceof Row ? row.clone() : Row.from(row).clone()));
        table.options = {
            ...this.options
        };
        table.headerRow = this.headerRow?.clone();
        return table;
    }
    toString() {
        return new TableLayout(this, this.options).toString();
    }
    render() {
        console.log(this.toString());
        return this;
    }
    maxColWidth(width, override = true) {
        if (override || typeof this.options.maxColWidth === "undefined") {
            this.options.maxColWidth = width;
        }
        return this;
    }
    minColWidth(width, override = true) {
        if (override || typeof this.options.minColWidth === "undefined") {
            this.options.minColWidth = width;
        }
        return this;
    }
    indent(width, override = true) {
        if (override || typeof this.options.indent === "undefined") {
            this.options.indent = width;
        }
        return this;
    }
    padding(padding, override = true) {
        if (override || typeof this.options.padding === "undefined") {
            this.options.padding = padding;
        }
        return this;
    }
    border(enable = true, override = true) {
        if (override || typeof this.options.border === "undefined") {
            this.options.border = enable;
        }
        return this;
    }
    align(direction, override = true) {
        if (override || typeof this.options.align === "undefined") {
            this.options.align = direction;
        }
        return this;
    }
    chars(chars) {
        Object.assign(this.options.chars, chars);
        return this;
    }
    getHeader() {
        return this.headerRow;
    }
    getBody() {
        return [
            ...this
        ];
    }
    getMaxColWidth() {
        return this.options.maxColWidth;
    }
    getMinColWidth() {
        return this.options.minColWidth;
    }
    getIndent() {
        return this.options.indent;
    }
    getPadding() {
        return this.options.padding;
    }
    getBorder() {
        return this.options.border === true;
    }
    hasHeaderBorder() {
        const hasBorder = this.headerRow?.hasBorder();
        return hasBorder === true || this.getBorder() && hasBorder !== false;
    }
    hasBodyBorder() {
        return this.getBorder() || this.options.columns.some((column)=>column.getBorder()) || this.some((row)=>row instanceof Row ? row.hasBorder() : row.some((cell)=>cell instanceof Cell ? cell.getBorder() : false));
    }
    hasBorder() {
        return this.hasHeaderBorder() || this.hasBodyBorder();
    }
    getAlign() {
        return this.options.align ?? "left";
    }
    getColumns() {
        return this.options.columns;
    }
    getColumn(index) {
        return this.options.columns[index] ??= new Column();
    }
}
class HelpGenerator {
    cmd;
    indent;
    options;
    static generate(cmd, options) {
        return new HelpGenerator(cmd, options).generate();
    }
    constructor(cmd, options = {}){
        this.cmd = cmd;
        this.indent = 2;
        this.options = {
            types: false,
            hints: true,
            colors: true,
            long: false,
            ...options
        };
    }
    generate() {
        const areColorsEnabled = getColorEnabled();
        setColorEnabled(this.options.colors);
        const result1 = this.generateHeader() + this.generateMeta() + this.generateDescription() + this.generateOptions() + this.generateCommands() + this.generateEnvironmentVariables() + this.generateExamples();
        setColorEnabled(areColorsEnabled);
        return result1;
    }
    generateHeader() {
        const usage = this.cmd.getUsage();
        const rows = [
            [
                bold1("Usage:"),
                brightMagenta(this.cmd.getPath() + (usage ? " " + highlightArguments(usage, this.options.types) : ""))
            ]
        ];
        const version = this.cmd.getVersion();
        if (version) {
            rows.push([
                bold1("Version:"),
                yellow1(`${this.cmd.getVersion()}`)
            ]);
        }
        return "\n" + Table.from(rows).padding(1).toString() + "\n";
    }
    generateMeta() {
        const meta = Object.entries(this.cmd.getMeta());
        if (!meta.length) {
            return "";
        }
        const rows = [];
        for (const [name, value] of meta){
            rows.push([
                bold1(`${name}: `) + value
            ]);
        }
        return "\n" + Table.from(rows).padding(1).toString() + "\n";
    }
    generateDescription() {
        if (!this.cmd.getDescription()) {
            return "";
        }
        return this.label("Description") + Table.from([
            [
                dedent(this.cmd.getDescription())
            ]
        ]).indent(this.indent).maxColWidth(140).padding(1).toString() + "\n";
    }
    generateOptions() {
        const options = this.cmd.getOptions(false);
        if (!options.length) {
            return "";
        }
        let groups = [];
        const hasGroups = options.some((option)=>option.groupName);
        if (hasGroups) {
            for (const option of options){
                let group = groups.find((group)=>group.name === option.groupName);
                if (!group) {
                    group = {
                        name: option.groupName,
                        options: []
                    };
                    groups.push(group);
                }
                group.options.push(option);
            }
        } else {
            groups = [
                {
                    name: "Options",
                    options
                }
            ];
        }
        let result1 = "";
        for (const group of groups){
            result1 += this.generateOptionGroup(group);
        }
        return result1;
    }
    generateOptionGroup(group) {
        if (!group.options.length) {
            return "";
        }
        const hasTypeDefinitions = !!group.options.find((option)=>!!option.typeDefinition);
        if (hasTypeDefinitions) {
            return this.label(group.name ?? "Options") + Table.from([
                ...group.options.map((option)=>[
                        option.flags.map((flag)=>brightBlue(flag)).join(", "),
                        highlightArguments(option.typeDefinition || "", this.options.types),
                        red1(bold1("-")),
                        getDescription(option.description, !this.options.long),
                        this.generateHints(option)
                    ])
            ]).padding([
                2,
                2,
                1,
                2
            ]).indent(this.indent).maxColWidth([
                60,
                60,
                1,
                80,
                60
            ]).toString() + "\n";
        }
        return this.label(group.name ?? "Options") + Table.from([
            ...group.options.map((option)=>[
                    option.flags.map((flag)=>brightBlue(flag)).join(", "),
                    red1(bold1("-")),
                    getDescription(option.description, !this.options.long),
                    this.generateHints(option)
                ])
        ]).indent(this.indent).maxColWidth([
            60,
            1,
            80,
            60
        ]).padding([
            2,
            1,
            2
        ]).toString() + "\n";
    }
    generateCommands() {
        const commands = this.cmd.getCommands(false);
        if (!commands.length) {
            return "";
        }
        const hasTypeDefinitions = !!commands.find((command)=>!!command.getArgsDefinition());
        if (hasTypeDefinitions) {
            return this.label("Commands") + Table.from([
                ...commands.map((command)=>[
                        [
                            command.getName(),
                            ...command.getAliases()
                        ].map((name)=>brightBlue(name)).join(", "),
                        highlightArguments(command.getArgsDefinition() || "", this.options.types),
                        red1(bold1("-")),
                        command.getShortDescription()
                    ])
            ]).indent(this.indent).maxColWidth([
                60,
                60,
                1,
                80
            ]).padding([
                2,
                2,
                1,
                2
            ]).toString() + "\n";
        }
        return this.label("Commands") + Table.from([
            ...commands.map((command)=>[
                    [
                        command.getName(),
                        ...command.getAliases()
                    ].map((name)=>brightBlue(name)).join(", "),
                    red1(bold1("-")),
                    command.getShortDescription()
                ])
        ]).maxColWidth([
            60,
            1,
            80
        ]).padding([
            2,
            1,
            2
        ]).indent(this.indent).toString() + "\n";
    }
    generateEnvironmentVariables() {
        const envVars = this.cmd.getEnvVars(false);
        if (!envVars.length) {
            return "";
        }
        return this.label("Environment variables") + Table.from([
            ...envVars.map((envVar)=>[
                    envVar.names.map((name)=>brightBlue(name)).join(", "),
                    highlightArgumentDetails(envVar.details, this.options.types),
                    red1(bold1("-")),
                    this.options.long ? dedent(envVar.description) : envVar.description.trim().split("\n", 1)[0],
                    envVar.required ? `(${yellow1(`required`)})` : ""
                ])
        ]).padding([
            2,
            2,
            1,
            2
        ]).indent(this.indent).maxColWidth([
            60,
            60,
            1,
            80,
            10
        ]).toString() + "\n";
    }
    generateExamples() {
        const examples = this.cmd.getExamples();
        if (!examples.length) {
            return "";
        }
        return this.label("Examples") + Table.from(examples.map((example)=>[
                dim1(bold1(`${capitalize(example.name)}:`)),
                dedent(example.description)
            ])).padding(1).indent(this.indent).maxColWidth(150).toString() + "\n";
    }
    generateHints(option) {
        if (!this.options.hints) {
            return "";
        }
        const hints = [];
        option.required && hints.push(yellow1(`required`));
        if (typeof option.default !== "undefined") {
            const defaultValue = getDefaultValue(option);
            if (typeof defaultValue !== "undefined") {
                hints.push(bold1(`Default: `) + inspect(defaultValue, this.options.colors));
            }
        }
        option.depends?.length && hints.push(yellow1(bold1(`Depends: `)) + italic(option.depends.map(getFlag).join(", ")));
        option.conflicts?.length && hints.push(red1(bold1(`Conflicts: `)) + italic(option.conflicts.map(getFlag).join(", ")));
        const type = this.cmd.getType(option.args[0]?.type)?.handler;
        if (type instanceof Type) {
            const possibleValues = type.values?.(this.cmd, this.cmd.getParent());
            if (possibleValues?.length) {
                hints.push(bold1(`Values: `) + possibleValues.map((value)=>inspect(value, this.options.colors)).join(", "));
            }
        }
        if (hints.length) {
            return `(${hints.join(", ")})`;
        }
        return "";
    }
    label(label) {
        return "\n" + bold1(`${label}:`) + "\n\n";
    }
}
function capitalize(string) {
    return (string?.charAt(0).toUpperCase() + string.slice(1)) ?? "";
}
function inspect(value, colors) {
    return Deno.inspect(value, {
        depth: 1,
        colors,
        trailingComma: false
    });
}
function highlightArguments(argsDefinition, types = true) {
    if (!argsDefinition) {
        return "";
    }
    return parseArgumentsDefinition(argsDefinition, false, true).map((arg)=>typeof arg === "string" ? arg : highlightArgumentDetails(arg, types)).join(" ");
}
function highlightArgumentDetails(arg, types = true) {
    let str = "";
    str += yellow1(arg.optional ? "[" : "<");
    let name = "";
    name += arg.name;
    if (arg.variadic) {
        name += "...";
    }
    name = brightMagenta(name);
    str += name;
    if (types) {
        str += yellow1(":");
        str += red1(arg.type);
        if (arg.list) {
            str += green1("[]");
        }
    }
    str += yellow1(arg.optional ? "]" : ">");
    return str;
}
class Command {
    types = new Map();
    rawArgs = [];
    literalArgs = [];
    _name = "COMMAND";
    _parent;
    _globalParent;
    ver;
    desc = "";
    _usage;
    actionHandler;
    globalActionHandler;
    options = [];
    commands = new Map();
    examples = [];
    envVars = [];
    aliases = [];
    completions = new Map();
    cmd = this;
    argsDefinition;
    isExecutable = false;
    throwOnError = false;
    _allowEmpty = false;
    _stopEarly = false;
    defaultCommand;
    _useRawArgs = false;
    args = [];
    isHidden = false;
    isGlobal = false;
    hasDefaults = false;
    _versionOptions;
    _helpOptions;
    _versionOption;
    _helpOption;
    _help;
    _shouldExit;
    _meta = {};
    _groupName = null;
    _noGlobals = false;
    errorHandler;
    versionOption(flags, desc, opts) {
        this._versionOptions = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    helpOption(flags, desc, opts) {
        this._helpOptions = flags === false ? flags : {
            flags,
            desc,
            opts: typeof opts === "function" ? {
                action: opts
            } : opts
        };
        return this;
    }
    command(nameAndArguments, cmdOrDescription, override) {
        this.reset();
        const result1 = splitArguments(nameAndArguments);
        const name = result1.flags.shift();
        const aliases = result1.flags;
        if (!name) {
            throw new MissingCommandNameError();
        }
        if (this.getBaseCommand(name, true)) {
            if (!override) {
                throw new DuplicateCommandNameError(name);
            }
            this.removeCommand(name);
        }
        let description;
        let cmd;
        if (typeof cmdOrDescription === "string") {
            description = cmdOrDescription;
        }
        if (cmdOrDescription instanceof Command) {
            cmd = cmdOrDescription.reset();
        } else {
            cmd = new Command();
        }
        cmd._name = name;
        cmd._parent = this;
        if (description) {
            cmd.description(description);
        }
        if (result1.typeDefinition) {
            cmd.arguments(result1.typeDefinition);
        }
        aliases.forEach((alias)=>cmd.alias(alias));
        this.commands.set(name, cmd);
        this.select(name);
        return this;
    }
    alias(alias) {
        if (this.cmd._name === alias || this.cmd.aliases.includes(alias)) {
            throw new DuplicateCommandAliasError(alias);
        }
        this.cmd.aliases.push(alias);
        return this;
    }
    reset() {
        this._groupName = null;
        this.cmd = this;
        return this;
    }
    select(name) {
        const cmd = this.getBaseCommand(name, true);
        if (!cmd) {
            throw new CommandNotFoundError(name, this.getBaseCommands(true));
        }
        this.cmd = cmd;
        return this;
    }
    name(name) {
        this.cmd._name = name;
        return this;
    }
    version(version) {
        if (typeof version === "string") {
            this.cmd.ver = ()=>version;
        } else if (typeof version === "function") {
            this.cmd.ver = version;
        }
        return this;
    }
    meta(name, value) {
        this.cmd._meta[name] = value;
        return this;
    }
    getMeta(name) {
        return typeof name === "undefined" ? this._meta : this._meta[name];
    }
    help(help) {
        if (typeof help === "string") {
            this.cmd._help = ()=>help;
        } else if (typeof help === "function") {
            this.cmd._help = help;
        } else {
            this.cmd._help = (cmd, options)=>HelpGenerator.generate(cmd, {
                    ...help,
                    ...options
                });
        }
        return this;
    }
    description(description) {
        this.cmd.desc = description;
        return this;
    }
    usage(usage) {
        this.cmd._usage = usage;
        return this;
    }
    hidden() {
        this.cmd.isHidden = true;
        return this;
    }
    global() {
        this.cmd.isGlobal = true;
        return this;
    }
    executable() {
        this.cmd.isExecutable = true;
        return this;
    }
    arguments(args) {
        this.cmd.argsDefinition = args;
        return this;
    }
    action(fn) {
        this.cmd.actionHandler = fn;
        return this;
    }
    globalAction(fn) {
        this.cmd.globalActionHandler = fn;
        return this;
    }
    allowEmpty(allowEmpty) {
        this.cmd._allowEmpty = allowEmpty !== false;
        return this;
    }
    stopEarly(stopEarly = true) {
        this.cmd._stopEarly = stopEarly;
        return this;
    }
    useRawArgs(useRawArgs = true) {
        this.cmd._useRawArgs = useRawArgs;
        return this;
    }
    default(name) {
        this.cmd.defaultCommand = name;
        return this;
    }
    globalType(name, handler, options) {
        return this.type(name, handler, {
            ...options,
            global: true
        });
    }
    type(name, handler, options) {
        if (this.cmd.types.get(name) && !options?.override) {
            throw new DuplicateTypeError(name);
        }
        this.cmd.types.set(name, {
            ...options,
            name,
            handler: handler
        });
        if (handler instanceof Type && (typeof handler.complete !== "undefined" || typeof handler.values !== "undefined")) {
            const completeHandler = (cmd, parent)=>handler.complete?.(cmd, parent) || [];
            this.complete(name, completeHandler, options);
        }
        return this;
    }
    globalComplete(name, complete, options) {
        return this.complete(name, complete, {
            ...options,
            global: true
        });
    }
    complete(name, complete, options) {
        if (this.cmd.completions.has(name) && !options?.override) {
            throw new DuplicateCompletionError(name);
        }
        this.cmd.completions.set(name, {
            name,
            complete,
            ...options
        });
        return this;
    }
    throwErrors() {
        this.cmd.throwOnError = true;
        return this;
    }
    error(handler) {
        this.cmd.errorHandler = handler;
        return this;
    }
    getErrorHandler() {
        return this.errorHandler ?? this._parent?.errorHandler;
    }
    noExit() {
        this.cmd._shouldExit = false;
        this.throwErrors();
        return this;
    }
    noGlobals() {
        this.cmd._noGlobals = true;
        return this;
    }
    shouldThrowErrors() {
        return this.throwOnError || !!this._parent?.shouldThrowErrors();
    }
    shouldExit() {
        return this._shouldExit ?? this._parent?.shouldExit() ?? true;
    }
    group(name) {
        this.cmd._groupName = name;
        return this;
    }
    globalOption(flags, desc, opts) {
        if (typeof opts === "function") {
            return this.option(flags, desc, {
                value: opts,
                global: true
            });
        }
        return this.option(flags, desc, {
            ...opts,
            global: true
        });
    }
    option(flags, desc, opts) {
        if (typeof opts === "function") {
            opts = {
                value: opts
            };
        }
        const result1 = splitArguments(flags);
        const args = result1.typeDefinition ? parseArgumentsDefinition(result1.typeDefinition) : [];
        const option = {
            ...opts,
            name: "",
            description: desc,
            args,
            flags: result1.flags,
            equalsSign: result1.equalsSign,
            typeDefinition: result1.typeDefinition,
            groupName: this._groupName ?? undefined
        };
        if (option.separator) {
            for (const arg of args){
                if (arg.list) {
                    arg.separator = option.separator;
                }
            }
        }
        for (const part of option.flags){
            const arg = part.trim();
            const isLong = /^--/.test(arg);
            const name = isLong ? arg.slice(2) : arg.slice(1);
            if (this.cmd.getBaseOption(name, true)) {
                if (opts?.override) {
                    this.removeOption(name);
                } else {
                    throw new DuplicateOptionNameError(name, this.getPath());
                }
            }
            if (!option.name && isLong) {
                option.name = name;
            } else if (!option.aliases) {
                option.aliases = [
                    name
                ];
            } else {
                option.aliases.push(name);
            }
        }
        if (option.prepend) {
            this.cmd.options.unshift(option);
        } else {
            this.cmd.options.push(option);
        }
        return this;
    }
    example(name, description) {
        if (this.cmd.hasExample(name)) {
            throw new DuplicateExampleError(name);
        }
        this.cmd.examples.push({
            name,
            description
        });
        return this;
    }
    globalEnv(name, description, options) {
        return this.env(name, description, {
            ...options,
            global: true
        });
    }
    env(name, description, options) {
        const result1 = splitArguments(name);
        if (!result1.typeDefinition) {
            result1.typeDefinition = "<value:boolean>";
        }
        if (result1.flags.some((envName)=>this.cmd.getBaseEnvVar(envName, true))) {
            throw new DuplicateEnvVarError(name);
        }
        const details = parseArgumentsDefinition(result1.typeDefinition);
        if (details.length > 1) {
            throw new TooManyEnvVarValuesError(name);
        } else if (details.length && details[0].optional) {
            throw new UnexpectedOptionalEnvVarValueError(name);
        } else if (details.length && details[0].variadic) {
            throw new UnexpectedVariadicEnvVarValueError(name);
        }
        this.cmd.envVars.push({
            name: result1.flags[0],
            names: result1.flags,
            description,
            type: details[0].type,
            details: details.shift(),
            ...options
        });
        return this;
    }
    parse(args = Deno.args) {
        const ctx = {
            unknown: args.slice(),
            flags: {},
            env: {},
            literal: [],
            stopEarly: false,
            stopOnUnknown: false,
            defaults: {},
            actions: []
        };
        return this.parseCommand(ctx);
    }
    async parseCommand(ctx) {
        try {
            this.reset();
            this.registerDefaults();
            this.rawArgs = ctx.unknown.slice();
            if (this.isExecutable) {
                await this.executeExecutable(ctx.unknown);
                return {
                    options: {},
                    args: [],
                    cmd: this,
                    literal: []
                };
            } else if (this._useRawArgs) {
                await this.parseEnvVars(ctx, this.envVars);
                return await this.execute(ctx.env, ctx.unknown);
            }
            let preParseGlobals = false;
            let subCommand;
            if (ctx.unknown.length > 0) {
                subCommand = this.getSubCommand(ctx);
                if (!subCommand) {
                    const optionName = ctx.unknown[0].replace(/^-+/, "");
                    const option = this.getOption(optionName, true);
                    if (option?.global) {
                        preParseGlobals = true;
                        await this.parseGlobalOptionsAndEnvVars(ctx);
                    }
                }
            }
            if (subCommand || ctx.unknown.length > 0) {
                subCommand ??= this.getSubCommand(ctx);
                if (subCommand) {
                    subCommand._globalParent = this;
                    return subCommand.parseCommand(ctx);
                }
            }
            await this.parseOptionsAndEnvVars(ctx, preParseGlobals);
            const options = {
                ...ctx.env,
                ...ctx.flags
            };
            const args = this.parseArguments(ctx, options);
            this.literalArgs = ctx.literal;
            if (ctx.actions.length) {
                await Promise.all(ctx.actions.map((action)=>action.call(this, options, ...args)));
            }
            if (ctx.standalone) {
                return {
                    options,
                    args,
                    cmd: this,
                    literal: this.literalArgs
                };
            }
            return await this.execute(options, args);
        } catch (error) {
            this.handleError(error);
        }
    }
    getSubCommand(ctx) {
        const subCommand = this.getCommand(ctx.unknown[0], true);
        if (subCommand) {
            ctx.unknown.shift();
        }
        return subCommand;
    }
    async parseGlobalOptionsAndEnvVars(ctx) {
        const isHelpOption = this.getHelpOption()?.flags.includes(ctx.unknown[0]);
        const envVars = [
            ...this.envVars.filter((envVar)=>envVar.global),
            ...this.getGlobalEnvVars(true)
        ];
        await this.parseEnvVars(ctx, envVars, !isHelpOption);
        const options = [
            ...this.options.filter((option)=>option.global),
            ...this.getGlobalOptions(true)
        ];
        this.parseOptions(ctx, options, {
            stopEarly: true,
            stopOnUnknown: true,
            dotted: false
        });
    }
    async parseOptionsAndEnvVars(ctx, preParseGlobals) {
        const helpOption = this.getHelpOption();
        const isVersionOption = this._versionOption?.flags.includes(ctx.unknown[0]);
        const isHelpOption = helpOption && ctx.flags?.[helpOption.name] === true;
        const envVars = preParseGlobals ? this.envVars.filter((envVar)=>!envVar.global) : this.getEnvVars(true);
        await this.parseEnvVars(ctx, envVars, !isHelpOption && !isVersionOption);
        const options = this.getOptions(true);
        this.parseOptions(ctx, options);
    }
    registerDefaults() {
        if (this.hasDefaults || this.getParent()) {
            return this;
        }
        this.hasDefaults = true;
        this.reset();
        !this.types.has("string") && this.type("string", new StringType(), {
            global: true
        });
        !this.types.has("number") && this.type("number", new NumberType(), {
            global: true
        });
        !this.types.has("integer") && this.type("integer", new IntegerType(), {
            global: true
        });
        !this.types.has("boolean") && this.type("boolean", new BooleanType(), {
            global: true
        });
        !this.types.has("file") && this.type("file", new FileType(), {
            global: true
        });
        if (!this._help) {
            this.help({});
        }
        if (this._versionOptions !== false && (this._versionOptions || this.ver)) {
            this.option(this._versionOptions?.flags || "-V, --version", this._versionOptions?.desc || "Show the version number for this program.", {
                standalone: true,
                prepend: true,
                action: async function() {
                    const __long = this.getRawArgs().includes(`--${this._versionOption?.name}`);
                    if (__long) {
                        await checkVersion(this);
                        this.showLongVersion();
                    } else {
                        this.showVersion();
                    }
                    this.exit();
                },
                ...this._versionOptions?.opts ?? {}
            });
            this._versionOption = this.options[0];
        }
        if (this._helpOptions !== false) {
            this.option(this._helpOptions?.flags || "-h, --help", this._helpOptions?.desc || "Show this help.", {
                standalone: true,
                global: true,
                prepend: true,
                action: async function() {
                    const __long = this.getRawArgs().includes(`--${this.getHelpOption()?.name}`);
                    await checkVersion(this);
                    this.showHelp({
                        long: __long
                    });
                    this.exit();
                },
                ...this._helpOptions?.opts ?? {}
            });
            this._helpOption = this.options[0];
        }
        return this;
    }
    async execute(options, args) {
        if (this.defaultCommand) {
            const cmd = this.getCommand(this.defaultCommand, true);
            if (!cmd) {
                throw new DefaultCommandNotFoundError(this.defaultCommand, this.getCommands());
            }
            cmd._globalParent = this;
            return cmd.execute(options, args);
        }
        await this.executeGlobalAction(options, args);
        if (this.actionHandler) {
            await this.actionHandler(options, ...args);
        }
        return {
            options,
            args,
            cmd: this,
            literal: this.literalArgs
        };
    }
    async executeGlobalAction(options, args) {
        if (!this._noGlobals) {
            await this._parent?.executeGlobalAction(options, args);
        }
        await this.globalActionHandler?.(options, ...args);
    }
    async executeExecutable(args) {
        const command = this.getPath().replace(/\s+/g, "-");
        await Deno.permissions.request({
            name: "run",
            command
        });
        try {
            const cmd = new Deno.Command(command, {
                args
            });
            const output = await cmd.output();
            if (!output.success) {
                Deno.exit(output.code);
            }
        } catch (error) {
            if (error instanceof Deno.errors.NotFound) {
                throw new CommandExecutableNotFoundError(command);
            }
            throw error;
        }
    }
    parseOptions(ctx, options, { stopEarly = this._stopEarly, stopOnUnknown = false, dotted = true } = {}) {
        parseFlags(ctx, {
            stopEarly,
            stopOnUnknown,
            dotted,
            allowEmpty: this._allowEmpty,
            flags: options,
            ignoreDefaults: ctx.env,
            parse: (type)=>this.parseType(type),
            option: (option)=>{
                if (option.action) {
                    ctx.actions.push(option.action);
                }
            }
        });
    }
    parseType(type) {
        const typeSettings = this.getType(type.type);
        if (!typeSettings) {
            throw new UnknownTypeError(type.type, this.getTypes().map((type)=>type.name));
        }
        return typeSettings.handler instanceof Type ? typeSettings.handler.parse(type) : typeSettings.handler(type);
    }
    async parseEnvVars(ctx, envVars, validate = true) {
        for (const envVar of envVars){
            const env = await this.findEnvVar(envVar.names);
            if (env) {
                const parseType = (value)=>{
                    return this.parseType({
                        label: "Environment variable",
                        type: envVar.type,
                        name: env.name,
                        value
                    });
                };
                const propertyName = underscoreToCamelCase(envVar.prefix ? envVar.names[0].replace(new RegExp(`^${envVar.prefix}`), "") : envVar.names[0]);
                if (envVar.details.list) {
                    ctx.env[propertyName] = env.value.split(envVar.details.separator ?? ",").map(parseType);
                } else {
                    ctx.env[propertyName] = parseType(env.value);
                }
                if (envVar.value && typeof ctx.env[propertyName] !== "undefined") {
                    ctx.env[propertyName] = envVar.value(ctx.env[propertyName]);
                }
            } else if (envVar.required && validate) {
                throw new MissingRequiredEnvVarError(envVar);
            }
        }
    }
    async findEnvVar(names) {
        for (const name of names){
            const status = await Deno.permissions.query({
                name: "env",
                variable: name
            });
            if (status.state === "granted") {
                const value = Deno.env.get(name);
                if (value) {
                    return {
                        name,
                        value
                    };
                }
            }
        }
        return undefined;
    }
    parseArguments(ctx, options) {
        const params = [];
        const args = ctx.unknown.slice();
        if (!this.hasArguments()) {
            if (args.length) {
                if (this.hasCommands(true)) {
                    if (this.hasCommand(args[0], true)) {
                        throw new TooManyArgumentsError(args);
                    } else {
                        throw new UnknownCommandError(args[0], this.getCommands());
                    }
                } else {
                    throw new NoArgumentsAllowedError(this.getPath());
                }
            }
        } else {
            if (!args.length) {
                const required = this.getArguments().filter((expectedArg)=>!expectedArg.optional).map((expectedArg)=>expectedArg.name);
                if (required.length) {
                    const optionNames = Object.keys(options);
                    const hasStandaloneOption = !!optionNames.find((name)=>this.getOption(name, true)?.standalone);
                    if (!hasStandaloneOption) {
                        throw new MissingArgumentsError(required);
                    }
                }
            } else {
                for (const expectedArg of this.getArguments()){
                    if (!args.length) {
                        if (expectedArg.optional) {
                            break;
                        }
                        throw new MissingArgumentError(expectedArg.name);
                    }
                    let arg;
                    const parseArgValue = (value)=>{
                        return expectedArg.list ? value.split(",").map((value)=>parseArgType(value)) : parseArgType(value);
                    };
                    const parseArgType = (value)=>{
                        return this.parseType({
                            label: "Argument",
                            type: expectedArg.type,
                            name: expectedArg.name,
                            value
                        });
                    };
                    if (expectedArg.variadic) {
                        arg = args.splice(0, args.length).map((value)=>parseArgValue(value));
                    } else {
                        arg = parseArgValue(args.shift());
                    }
                    if (expectedArg.variadic && Array.isArray(arg)) {
                        params.push(...arg);
                    } else if (typeof arg !== "undefined") {
                        params.push(arg);
                    }
                }
                if (args.length) {
                    throw new TooManyArgumentsError(args);
                }
            }
        }
        return params;
    }
    handleError(error) {
        this.throw(error instanceof ValidationError ? new ValidationError1(error.message) : error instanceof Error ? error : new Error(`[non-error-thrown] ${error}`));
    }
    throw(error) {
        if (error instanceof ValidationError1) {
            error.cmd = this;
        }
        this.getErrorHandler()?.(error, this);
        if (this.shouldThrowErrors() || !(error instanceof ValidationError1)) {
            throw error;
        }
        this.showHelp();
        console.error(red1(`  ${bold1("error")}: ${error.message}\n`));
        Deno.exit(error instanceof ValidationError1 ? error.exitCode : 1);
    }
    getName() {
        return this._name;
    }
    getParent() {
        return this._parent;
    }
    getGlobalParent() {
        return this._globalParent;
    }
    getMainCommand() {
        return this._parent?.getMainCommand() ?? this;
    }
    getAliases() {
        return this.aliases;
    }
    getPath(name) {
        return this._parent ? this._parent.getPath(name) + " " + this._name : name || this._name;
    }
    getArgsDefinition() {
        return this.argsDefinition;
    }
    getArgument(name) {
        return this.getArguments().find((arg)=>arg.name === name);
    }
    getArguments() {
        if (!this.args.length && this.argsDefinition) {
            this.args = parseArgumentsDefinition(this.argsDefinition);
        }
        return this.args;
    }
    hasArguments() {
        return !!this.argsDefinition;
    }
    getVersion() {
        return this.getVersionHandler()?.call(this, this);
    }
    getVersionHandler() {
        return this.ver ?? this._parent?.getVersionHandler();
    }
    getDescription() {
        return typeof this.desc === "function" ? this.desc = this.desc() : this.desc;
    }
    getUsage() {
        return this._usage ?? [
            this.getArgsDefinition(),
            this.getRequiredOptionsDefinition()
        ].join(" ").trim();
    }
    getRequiredOptionsDefinition() {
        return this.getOptions().filter((option)=>option.required).map((option)=>[
                findFlag(option.flags),
                option.typeDefinition
            ].filter((v)=>v).join(" ").trim()).join(" ");
    }
    getShortDescription() {
        return getDescription(this.getDescription(), true);
    }
    getRawArgs() {
        return this.rawArgs;
    }
    getLiteralArgs() {
        return this.literalArgs;
    }
    showVersion() {
        console.log(this.getVersion());
    }
    getLongVersion() {
        return `${bold1(this.getMainCommand().getName())} ${brightBlue(this.getVersion() ?? "")}` + Object.entries(this.getMeta()).map(([k, v])=>`\n${bold1(k)} ${brightBlue(v)}`).join("");
    }
    showLongVersion() {
        console.log(this.getLongVersion());
    }
    showHelp(options) {
        console.log(this.getHelp(options));
    }
    getHelp(options) {
        this.registerDefaults();
        return this.getHelpHandler().call(this, this, options ?? {});
    }
    getHelpHandler() {
        return this._help ?? this._parent?.getHelpHandler();
    }
    exit(code = 0) {
        if (this.shouldExit()) {
            Deno.exit(code);
        }
    }
    hasOptions(hidden) {
        return this.getOptions(hidden).length > 0;
    }
    getOptions(hidden) {
        return this.getGlobalOptions(hidden).concat(this.getBaseOptions(hidden));
    }
    getBaseOptions(hidden) {
        if (!this.options.length) {
            return [];
        }
        return hidden ? this.options.slice(0) : this.options.filter((opt)=>!opt.hidden);
    }
    getGlobalOptions(hidden) {
        const helpOption = this.getHelpOption();
        const getGlobals = (cmd, noGlobals, options = [], names = [])=>{
            if (cmd.options.length) {
                for (const option of cmd.options){
                    if (option.global && !this.options.find((opt)=>opt.name === option.name) && names.indexOf(option.name) === -1 && (hidden || !option.hidden)) {
                        if (noGlobals && option !== helpOption) {
                            continue;
                        }
                        names.push(option.name);
                        options.push(option);
                    }
                }
            }
            return cmd._parent ? getGlobals(cmd._parent, noGlobals || cmd._noGlobals, options, names) : options;
        };
        return this._parent ? getGlobals(this._parent, this._noGlobals) : [];
    }
    hasOption(name, hidden) {
        return !!this.getOption(name, hidden);
    }
    getOption(name, hidden) {
        return this.getBaseOption(name, hidden) ?? this.getGlobalOption(name, hidden);
    }
    getBaseOption(name, hidden) {
        const option = this.options.find((option)=>option.name === name || option.aliases?.includes(name));
        return option && (hidden || !option.hidden) ? option : undefined;
    }
    getGlobalOption(name, hidden) {
        const helpOption = this.getHelpOption();
        const getGlobalOption = (parent, noGlobals)=>{
            const option = parent.getBaseOption(name, hidden);
            if (!option?.global) {
                return parent._parent && getGlobalOption(parent._parent, noGlobals || parent._noGlobals);
            }
            if (noGlobals && option !== helpOption) {
                return;
            }
            return option;
        };
        return this._parent && getGlobalOption(this._parent, this._noGlobals);
    }
    removeOption(name) {
        const index = this.options.findIndex((option)=>option.name === name);
        if (index === -1) {
            return;
        }
        return this.options.splice(index, 1)[0];
    }
    hasCommands(hidden) {
        return this.getCommands(hidden).length > 0;
    }
    getCommands(hidden) {
        return this.getGlobalCommands(hidden).concat(this.getBaseCommands(hidden));
    }
    getBaseCommands(hidden) {
        const commands = Array.from(this.commands.values());
        return hidden ? commands : commands.filter((cmd)=>!cmd.isHidden);
    }
    getGlobalCommands(hidden) {
        const getCommands = (command, noGlobals, commands = [], names = [])=>{
            if (command.commands.size) {
                for (const [_, cmd] of command.commands){
                    if (cmd.isGlobal && this !== cmd && !this.commands.has(cmd._name) && names.indexOf(cmd._name) === -1 && (hidden || !cmd.isHidden)) {
                        if (noGlobals && cmd?.getName() !== "help") {
                            continue;
                        }
                        names.push(cmd._name);
                        commands.push(cmd);
                    }
                }
            }
            return command._parent ? getCommands(command._parent, noGlobals || command._noGlobals, commands, names) : commands;
        };
        return this._parent ? getCommands(this._parent, this._noGlobals) : [];
    }
    hasCommand(name, hidden) {
        return !!this.getCommand(name, hidden);
    }
    getCommand(name, hidden) {
        return this.getBaseCommand(name, hidden) ?? this.getGlobalCommand(name, hidden);
    }
    getBaseCommand(name, hidden) {
        for (const cmd of this.commands.values()){
            if (cmd._name === name || cmd.aliases.includes(name)) {
                return cmd && (hidden || !cmd.isHidden) ? cmd : undefined;
            }
        }
    }
    getGlobalCommand(name, hidden) {
        const getGlobalCommand = (parent, noGlobals)=>{
            const cmd = parent.getBaseCommand(name, hidden);
            if (!cmd?.isGlobal) {
                return parent._parent && getGlobalCommand(parent._parent, noGlobals || parent._noGlobals);
            }
            if (noGlobals && cmd.getName() !== "help") {
                return;
            }
            return cmd;
        };
        return this._parent && getGlobalCommand(this._parent, this._noGlobals);
    }
    removeCommand(name) {
        const command = this.getBaseCommand(name, true);
        if (command) {
            this.commands.delete(command._name);
        }
        return command;
    }
    getTypes() {
        return this.getGlobalTypes().concat(this.getBaseTypes());
    }
    getBaseTypes() {
        return Array.from(this.types.values());
    }
    getGlobalTypes() {
        const getTypes = (cmd, types = [], names = [])=>{
            if (cmd) {
                if (cmd.types.size) {
                    cmd.types.forEach((type)=>{
                        if (type.global && !this.types.has(type.name) && names.indexOf(type.name) === -1) {
                            names.push(type.name);
                            types.push(type);
                        }
                    });
                }
                return getTypes(cmd._parent, types, names);
            }
            return types;
        };
        return getTypes(this._parent);
    }
    getType(name) {
        return this.getBaseType(name) ?? this.getGlobalType(name);
    }
    getBaseType(name) {
        return this.types.get(name);
    }
    getGlobalType(name) {
        if (!this._parent) {
            return;
        }
        const cmd = this._parent.getBaseType(name);
        if (!cmd?.global) {
            return this._parent.getGlobalType(name);
        }
        return cmd;
    }
    getCompletions() {
        return this.getGlobalCompletions().concat(this.getBaseCompletions());
    }
    getBaseCompletions() {
        return Array.from(this.completions.values());
    }
    getGlobalCompletions() {
        const getCompletions = (cmd, completions = [], names = [])=>{
            if (cmd) {
                if (cmd.completions.size) {
                    cmd.completions.forEach((completion)=>{
                        if (completion.global && !this.completions.has(completion.name) && names.indexOf(completion.name) === -1) {
                            names.push(completion.name);
                            completions.push(completion);
                        }
                    });
                }
                return getCompletions(cmd._parent, completions, names);
            }
            return completions;
        };
        return getCompletions(this._parent);
    }
    getCompletion(name) {
        return this.getBaseCompletion(name) ?? this.getGlobalCompletion(name);
    }
    getBaseCompletion(name) {
        return this.completions.get(name);
    }
    getGlobalCompletion(name) {
        if (!this._parent) {
            return;
        }
        const completion = this._parent.getBaseCompletion(name);
        if (!completion?.global) {
            return this._parent.getGlobalCompletion(name);
        }
        return completion;
    }
    hasEnvVars(hidden) {
        return this.getEnvVars(hidden).length > 0;
    }
    getEnvVars(hidden) {
        return this.getGlobalEnvVars(hidden).concat(this.getBaseEnvVars(hidden));
    }
    getBaseEnvVars(hidden) {
        if (!this.envVars.length) {
            return [];
        }
        return hidden ? this.envVars.slice(0) : this.envVars.filter((env)=>!env.hidden);
    }
    getGlobalEnvVars(hidden) {
        if (this._noGlobals) {
            return [];
        }
        const getEnvVars = (cmd, envVars = [], names = [])=>{
            if (cmd) {
                if (cmd.envVars.length) {
                    cmd.envVars.forEach((envVar)=>{
                        if (envVar.global && !this.envVars.find((env)=>env.names[0] === envVar.names[0]) && names.indexOf(envVar.names[0]) === -1 && (hidden || !envVar.hidden)) {
                            names.push(envVar.names[0]);
                            envVars.push(envVar);
                        }
                    });
                }
                return getEnvVars(cmd._parent, envVars, names);
            }
            return envVars;
        };
        return getEnvVars(this._parent);
    }
    hasEnvVar(name, hidden) {
        return !!this.getEnvVar(name, hidden);
    }
    getEnvVar(name, hidden) {
        return this.getBaseEnvVar(name, hidden) ?? this.getGlobalEnvVar(name, hidden);
    }
    getBaseEnvVar(name, hidden) {
        const envVar = this.envVars.find((env)=>env.names.indexOf(name) !== -1);
        return envVar && (hidden || !envVar.hidden) ? envVar : undefined;
    }
    getGlobalEnvVar(name, hidden) {
        if (!this._parent || this._noGlobals) {
            return;
        }
        const envVar = this._parent.getBaseEnvVar(name, hidden);
        if (!envVar?.global) {
            return this._parent.getGlobalEnvVar(name, hidden);
        }
        return envVar;
    }
    hasExamples() {
        return this.examples.length > 0;
    }
    getExamples() {
        return this.examples;
    }
    hasExample(name) {
        return !!this.getExample(name);
    }
    getExample(name) {
        return this.examples.find((example)=>example.name === name);
    }
    getHelpOption() {
        return this._helpOption ?? this._parent?.getHelpOption();
    }
}
async function checkVersion(cmd) {
    const mainCommand = cmd.getMainCommand();
    const upgradeCommand = mainCommand.getCommand("upgrade");
    if (!isUpgradeCommand(upgradeCommand)) {
        return;
    }
    const latestVersion = await upgradeCommand.getLatestVersion();
    const currentVersion = mainCommand.getVersion();
    if (currentVersion === latestVersion) {
        return;
    }
    const versionHelpText = `(New version available: ${latestVersion}. Run '${mainCommand.getName()} upgrade' to upgrade to the latest version!)`;
    mainCommand.version(`${currentVersion}  ${bold1(yellow1(versionHelpText))}`);
}
function findFlag(flags) {
    for (const flag of flags){
        if (flag.startsWith("--")) {
            return flag;
        }
    }
    return flags[0];
}
function isUpgradeCommand(command) {
    return command instanceof Command && "getLatestVersion" in command;
}
class CommandType extends StringType {
    complete(_cmd, parent) {
        return parent?.getCommands(false).map((cmd)=>cmd.getName()) || [];
    }
}
class HelpCommand extends Command {
    constructor(cmd){
        super();
        return this.type("command", new CommandType()).arguments("[command:command]").description("Show this help or the help of a sub-command.").noGlobals().action(async (_, name)=>{
            if (!cmd) {
                cmd = name ? this.getGlobalParent()?.getBaseCommand(name) : this.getGlobalParent();
            }
            if (!cmd) {
                const cmds = this.getGlobalParent()?.getCommands();
                throw new UnknownCommandError(name ?? "", cmds ?? [], [
                    this.getName(),
                    ...this.getAliases()
                ]);
            }
            await checkVersion(cmd);
            cmd.showHelp();
            if (this.shouldExit()) {
                Deno.exit(0);
            }
        });
    }
}
class BashCompletionsGenerator {
    name;
    cmd;
    static generate(name, cmd) {
        if (!name || name === "COMMAND") {
            throw new MissingCommandNameCompletionsError("bash");
        }
        return new BashCompletionsGenerator(name, cmd).generate();
    }
    constructor(name, cmd){
        this.name = name;
        this.cmd = cmd;
    }
    generate() {
        const path = this.cmd.getPath(this.name);
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env bash
# bash completion support for ${path}${version}

_${replaceSpecialChars(path)}() {
  local word cur prev listFiles
  local -a opts
  COMPREPLY=()
  cur="\${COMP_WORDS[COMP_CWORD]}"
  prev="\${COMP_WORDS[COMP_CWORD-1]}"
  cmd="_"
  opts=()
  listFiles=0

  _${replaceSpecialChars(this.name)}_complete() {
    local action="$1"; shift
    mapfile -t values < <( ${this.name} completions complete "\${action}" "\${@}" )
    for i in "\${values[@]}"; do
      opts+=("$i")
    done
  }

  _${replaceSpecialChars(this.name)}_expand() {
    [ "$cur" != "\${cur%\\\\}" ] && cur="$cur\\\\"
  
    # expand ~username type directory specifications
    if [[ "$cur" == \\~*/* ]]; then
      # shellcheck disable=SC2086
      eval cur=$cur
      
    elif [[ "$cur" == \\~* ]]; then
      cur=\${cur#\\~}
      # shellcheck disable=SC2086,SC2207
      COMPREPLY=( $( compgen -P '~' -u $cur ) )
      return \${#COMPREPLY[@]}
    fi
  }

  # shellcheck disable=SC2120
  _${replaceSpecialChars(this.name)}_file_dir() {
    listFiles=1
    local IFS=$'\\t\\n' xspec #glob
    _${replaceSpecialChars(this.name)}_expand || return 0
  
    if [ "\${1:-}" = -d ]; then
      # shellcheck disable=SC2206,SC2207,SC2086
      COMPREPLY=( \${COMPREPLY[@]:-} $( compgen -d -- $cur ) )
      #eval "$glob"    # restore glob setting.
      return 0
    fi
  
    xspec=\${1:+"!*.$1"}	# set only if glob passed in as $1
    # shellcheck disable=SC2206,SC2207
    COMPREPLY=( \${COMPREPLY[@]:-} $( compgen -f -X "$xspec" -- "$cur" ) \
          $( compgen -d -- "$cur" ) )
  }

  ${this.generateCompletions(this.name, this.cmd).trim()}

  for word in "\${COMP_WORDS[@]}"; do
    case "\${word}" in
      -*) ;;
      *)
        cmd_tmp="\${cmd}_\${word//[^[:alnum:]]/_}"
        if type "\${cmd_tmp}" &>/dev/null; then
          cmd="\${cmd_tmp}"
        fi
    esac
  done

  \${cmd}

  if [[ listFiles -eq 1 ]]; then
    return 0
  fi

  if [[ \${#opts[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
    return 0
  fi

  local values
  values="$( printf "\\n%s" "\${opts[@]}" )"
  local IFS=$'\\n'
  # shellcheck disable=SC2207
  local result=($(compgen -W "\${values[@]}" -- "\${cur}"))
  if [[ \${#result[@]} -eq 0 ]]; then
    # shellcheck disable=SC2207
    COMPREPLY=($(compgen -f "\${cur}"))
  else
    # shellcheck disable=SC2207
    COMPREPLY=($(printf '%q\\n' "\${result[@]}"))
  fi

  return 0
}

complete -F _${replaceSpecialChars(path)} -o bashdefault -o default ${path}`;
    }
    generateCompletions(name, command, path = "", index = 1) {
        path = (path ? path + " " : "") + name;
        const commandCompletions = this.generateCommandCompletions(command, path, index);
        const childCommandCompletions = command.getCommands(false).filter((subCommand)=>subCommand !== command).map((subCommand)=>this.generateCompletions(subCommand.getName(), subCommand, path, index + 1)).join("");
        return `${commandCompletions}

${childCommandCompletions}`;
    }
    generateCommandCompletions(command, path, index) {
        const flags = this.getFlags(command);
        const childCommandNames = command.getCommands(false).map((childCommand)=>childCommand.getName());
        const completionsPath = ~path.indexOf(" ") ? " " + path.split(" ").slice(1).join(" ") : "";
        const optionArguments = this.generateOptionArguments(command, completionsPath);
        const completionsCmd = this.generateCommandCompletionsCommand(command, completionsPath);
        return `  __${replaceSpecialChars(path)}() {
    opts=(${[
            ...flags,
            ...childCommandNames
        ].join(" ")})
    ${completionsCmd}
    if [[ \${cur} == -* || \${COMP_CWORD} -eq ${index} ]] ; then
      return 0
    fi
    ${optionArguments}
  }`;
    }
    getFlags(command) {
        return command.getOptions(false).map((option)=>option.flags).flat();
    }
    generateOptionArguments(command, completionsPath) {
        let opts = "";
        const options = command.getOptions(false);
        if (options.length) {
            opts += 'case "${prev}" in';
            for (const option of options){
                const flags = option.flags.map((flag)=>flag.trim()).join("|");
                const completionsCmd = this.generateOptionCompletionsCommand(command, option.args, completionsPath, {
                    standalone: option.standalone
                });
                opts += `\n      ${flags}) ${completionsCmd} ;;`;
            }
            opts += "\n    esac";
        }
        return opts;
    }
    generateCommandCompletionsCommand(command, path) {
        const args = command.getArguments();
        if (args.length) {
            const type = command.getType(args[0].type);
            if (type && type.handler instanceof FileType) {
                return `_${replaceSpecialChars(this.name)}_file_dir`;
            }
            return `_${replaceSpecialChars(this.name)}_complete ${args[0].action}${path}`;
        }
        return "";
    }
    generateOptionCompletionsCommand(command, args, path, opts) {
        if (args.length) {
            const type = command.getType(args[0].type);
            if (type && type.handler instanceof FileType) {
                return `opts=(); _${replaceSpecialChars(this.name)}_file_dir`;
            }
            return `opts=(); _${replaceSpecialChars(this.name)}_complete ${args[0].action}${path}`;
        }
        if (opts?.standalone) {
            return "opts=()";
        }
        return "";
    }
}
function replaceSpecialChars(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class BashCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for bash.

To enable bash completions for this program add following line to your ${dim1(italic("~/.bashrc"))}:

    ${dim1(italic(`source <(${baseCmd.getPath()} completions bash)`))}`;
        }).noGlobals().option("-n, --name <command-name>", "The name of the main command.").action(({ name = this.getMainCommand().getName() })=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(BashCompletionsGenerator.generate(name, baseCmd));
        });
    }
}
class FishCompletionsGenerator {
    name;
    cmd;
    static generate(name, cmd) {
        if (!name || name === "COMMAND") {
            throw new MissingCommandNameCompletionsError("fish");
        }
        return new FishCompletionsGenerator(name, cmd).generate();
    }
    constructor(name, cmd){
        this.name = name;
        this.cmd = cmd;
    }
    generate() {
        const path = this.cmd.getPath(this.name);
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env fish
# fish completion support for ${path}${version}

function __fish_${replaceSpecialChars1(this.name)}_using_command
  set -l cmds ${getCommandFnNames(this.name, this.cmd).join(" ")}
  set -l words (commandline -opc)
  set -l cmd "_"
  for word in $words
    switch $word
      case '-*'
        continue
      case '*'
        set word (string replace -r -a '\\W' '_' $word)
        set -l cmd_tmp $cmd"_$word"
        if contains $cmd_tmp $cmds
          set cmd $cmd_tmp
        end
    end
  end
  if test "$cmd" = "$argv[1]"
    return 0
  end
  return 1
end

${this.generateCompletions(this.name, this.cmd).trim()}`;
    }
    generateCompletions(name, command) {
        const parent = command.getParent();
        let result1 = ``;
        if (parent) {
            result1 += "\n" + this.complete(parent, {
                description: command.getShortDescription(),
                arguments: name
            });
        }
        const commandArgs = command.getArguments();
        if (commandArgs.length) {
            result1 += "\n" + this.complete(command, {
                arguments: commandArgs.length ? this.getCompletionCommand(command, commandArgs[0]) : undefined
            });
        }
        for (const option of command.getOptions(false)){
            result1 += "\n" + this.completeOption(command, option);
        }
        for (const subCommand of command.getCommands(false)){
            result1 += this.generateCompletions(subCommand.getName(), subCommand);
        }
        return result1;
    }
    completeOption(command, option) {
        const shortOption = option.flags.find((flag)=>flag.length === 2)?.replace(/^(-)+/, "");
        const longOption = option.flags.find((flag)=>flag.length > 2)?.replace(/^(-)+/, "");
        return this.complete(command, {
            description: getDescription(option.description),
            shortOption: shortOption,
            longOption: longOption,
            required: true,
            standalone: option.standalone,
            arguments: option.args.length ? this.getCompletionCommand(command, option.args[0]) : undefined
        });
    }
    complete(command, options) {
        const cmd = [
            "complete"
        ];
        cmd.push("-c", this.name);
        cmd.push("-n", `'__fish_${replaceSpecialChars1(this.name)}_using_command __${replaceSpecialChars1(command.getPath(this.name))}'`);
        options.shortOption && cmd.push("-s", options.shortOption);
        options.longOption && cmd.push("-l", options.longOption);
        options.standalone && cmd.push("-x");
        cmd.push("-k");
        cmd.push("-f");
        if (options.arguments) {
            options.required && cmd.push("-r");
            cmd.push("-a", options.arguments);
        }
        if (options.description) {
            const description = getDescription(options.description, true).replace(/'/g, "\\'");
            cmd.push("-d", `'${description}'`);
        }
        return cmd.join(" ");
    }
    getCompletionCommand(cmd, arg) {
        const type = cmd.getType(arg.type);
        if (type && type.handler instanceof FileType) {
            return `'(__fish_complete_path)'`;
        }
        return `'(${this.name} completions complete ${arg.action + " " + getCompletionsPath(cmd)})'`;
    }
}
function getCommandFnNames(name, cmd, cmds = []) {
    cmds.push(`__${replaceSpecialChars1(cmd.getPath(name))}`);
    cmd.getCommands(false).forEach((command)=>{
        getCommandFnNames(name, command, cmds);
    });
    return cmds;
}
function getCompletionsPath(command) {
    return command.getPath().split(" ").slice(1).join(" ");
}
function replaceSpecialChars1(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class FishCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for fish.

To enable fish completions for this program add following line to your ${dim1(italic("~/.config/fish/config.fish"))}:

    ${dim1(italic(`source (${baseCmd.getPath()} completions fish | psub)`))}`;
        }).noGlobals().option("-n, --name <command-name>", "The name of the main command.").action(({ name = this.getMainCommand().getName() })=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(FishCompletionsGenerator.generate(name, baseCmd));
        });
    }
}
class ZshCompletionsGenerator {
    name;
    cmd;
    actions;
    static generate(name, cmd) {
        if (!name || name === "COMMAND") {
            throw new MissingCommandNameCompletionsError("zsh");
        }
        return new ZshCompletionsGenerator(name, cmd).generate();
    }
    constructor(name, cmd){
        this.name = name;
        this.cmd = cmd;
        this.actions = new Map();
    }
    generate() {
        const path = this.cmd.getPath(this.name);
        const version = this.cmd.getVersion() ? ` v${this.cmd.getVersion()}` : "";
        return `#!/usr/bin/env zsh
# zsh completion support for ${path}${version}

autoload -U is-at-least

# shellcheck disable=SC2154
(( $+functions[__${replaceSpecialChars2(this.name)}_complete] )) ||
function __${replaceSpecialChars2(this.name)}_complete {
  local name="$1"; shift
  local action="$1"; shift
  integer ret=1
  local -a values
  local expl lines
  _tags "$name"
  while _tags; do
    if _requested "$name"; then
      # shellcheck disable=SC2034
      lines="$(${this.name} completions complete "\${action}" "\${@}")"
      values=("\${(ps:\\n:)lines}")
      if (( \${#values[@]} )); then
        while _next_label "$name" expl "$action"; do
          compadd -S '' "\${expl[@]}" "\${values[@]}"
        done
      fi
    fi
  done
}

${this.generateCompletions(this.name, this.cmd).trim()}

# _${replaceSpecialChars2(path)} "\${@}"

compdef _${replaceSpecialChars2(path)} ${path}`;
    }
    generateCompletions(name, command, path = "") {
        if (!command.hasCommands(false) && !command.hasOptions(false) && !command.hasArguments()) {
            return "";
        }
        path = (path ? path + " " : "") + name;
        return `# shellcheck disable=SC2154
(( $+functions[_${replaceSpecialChars2(path)}] )) ||
function _${replaceSpecialChars2(path)}() {` + (!command.getParent() ? `
  local state` : "") + this.generateCommandCompletions(command, path) + this.generateSubCommandCompletions(command, path) + this.generateArgumentCompletions(command, path) + this.generateActions(command) + `\n}\n\n` + command.getCommands(false).filter((subCommand)=>subCommand !== command).map((subCommand)=>this.generateCompletions(subCommand.getName(), subCommand, path)).join("");
    }
    generateCommandCompletions(command, path) {
        const commands = command.getCommands(false);
        let completions = commands.map((subCommand)=>`'${subCommand.getName()}:${subCommand.getShortDescription().replace(/'/g, "'\"'\"'")}'`).join("\n      ");
        if (completions) {
            completions = `
    local -a commands
    # shellcheck disable=SC2034
    commands=(
      ${completions}
    )
    _describe 'command' commands`;
        }
        if (command.hasArguments()) {
            const completionsPath = path.split(" ").slice(1).join(" ");
            const arg = command.getArguments()[0];
            const action = this.addAction(arg, completionsPath);
            if (action && command.getCompletion(arg.action)) {
                completions += `\n    __${replaceSpecialChars2(this.name)}_complete ${action.arg.name} ${action.arg.action} ${action.cmd}`;
            }
        }
        if (completions) {
            completions = `\n\n  function _commands() {${completions}\n  }`;
        }
        return completions;
    }
    generateSubCommandCompletions(command, path) {
        if (command.hasCommands(false)) {
            const actions = command.getCommands(false).map((command)=>`${command.getName()}) _${replaceSpecialChars2(path + " " + command.getName())} ;;`).join("\n      ");
            return `\n
  function _command_args() {
    case "\${words[1]}" in\n      ${actions}\n    esac
  }`;
        }
        return "";
    }
    generateArgumentCompletions(command, path) {
        this.actions.clear();
        const options = this.generateOptions(command, path);
        let argIndex = 0;
        let argsCommand = "\n\n  _arguments -w -s -S -C";
        if (command.hasOptions()) {
            argsCommand += ` \\\n    ${options.join(" \\\n    ")}`;
        }
        if (command.hasCommands(false) || command.getArguments().filter((arg)=>command.getCompletion(arg.action)).length) {
            argsCommand += ` \\\n    '${++argIndex}:command:_commands'`;
        }
        if (command.hasArguments() || command.hasCommands(false)) {
            const args = [];
            for (const arg of command.getArguments().slice(1)){
                const type = command.getType(arg.type);
                if (type && type.handler instanceof FileType) {
                    const fileCompletions = this.getFileCompletions(type);
                    if (arg.variadic) {
                        argIndex++;
                        for(let i1 = 0; i1 < 5; i1++){
                            args.push(`${argIndex + i1}${arg.optional ? "::" : ":"}${arg.name}:${fileCompletions}`);
                        }
                    } else {
                        args.push(`${++argIndex}${arg.optional ? "::" : ":"}${arg.name}:${fileCompletions}`);
                    }
                } else {
                    const completionsPath = path.split(" ").slice(1).join(" ");
                    const action = this.addAction(arg, completionsPath);
                    args.push(`${++argIndex}${arg.optional ? "::" : ":"}${arg.name}:->${action.name}`);
                }
            }
            argsCommand += args.map((arg)=>`\\\n    '${arg}'`).join("");
            if (command.hasCommands(false)) {
                argsCommand += ` \\\n    '*::sub command:->command_args'`;
            }
        }
        return argsCommand;
    }
    generateOptions(command, path) {
        const options = [];
        const cmdArgs = path.split(" ");
        cmdArgs.shift();
        const completionsPath = cmdArgs.join(" ");
        const excludedFlags = command.getOptions(false).map((option)=>option.standalone ? option.flags : false).flat().filter((flag)=>typeof flag === "string");
        for (const option of command.getOptions(false)){
            options.push(this.generateOption(command, option, completionsPath, excludedFlags));
        }
        return options;
    }
    generateOption(command, option, completionsPath, excludedOptions) {
        let args = "";
        for (const arg of option.args){
            const type = command.getType(arg.type);
            const optionalValue = arg.optional ? "::" : ":";
            if (type && type.handler instanceof FileType) {
                const fileCompletions = this.getFileCompletions(type);
                args += `${optionalValue}${arg.name}:${fileCompletions}`;
            } else {
                const action = this.addAction(arg, completionsPath);
                args += `${optionalValue}${arg.name}:->${action.name}`;
            }
        }
        const description = getDescription(option.description, true).replace(/\[/g, "\\[").replace(/]/g, "\\]").replace(/"/g, '\\"').replace(/'/g, "'\"'\"'");
        const collect = option.collect ? "*" : "";
        const equalsSign = option.equalsSign ? "=" : "";
        const flags = option.flags.map((flag)=>`${flag}${equalsSign}`);
        let result1 = "";
        if (option.standalone) {
            result1 += "'(- *)'";
        } else {
            const excludedFlags = [
                ...excludedOptions
            ];
            if (option.conflicts?.length) {
                excludedFlags.push(...option.conflicts.map((opt)=>"--" + opt.replace(/^--/, "")));
            }
            if (!option.collect) {
                excludedFlags.push(...option.flags);
            }
            if (excludedFlags.length) {
                result1 += `'(${excludedFlags.join(" ")})'`;
            }
        }
        if (collect || flags.length > 1) {
            result1 += `{${collect}${flags.join(",")}}`;
        } else {
            result1 += `${flags.join(",")}`;
        }
        return `${result1}'[${description}]${args}'`;
    }
    getFileCompletions(type) {
        if (!(type.handler instanceof FileType)) {
            return "";
        }
        return "_files";
    }
    addAction(arg, cmd) {
        const action = `${arg.name}-${arg.action}`;
        if (!this.actions.has(action)) {
            this.actions.set(action, {
                arg: arg,
                label: `${arg.name}: ${arg.action}`,
                name: action,
                cmd
            });
        }
        return this.actions.get(action);
    }
    generateActions(command) {
        let actions = [];
        if (this.actions.size) {
            actions = Array.from(this.actions).map(([name, action])=>`${name}) __${replaceSpecialChars2(this.name)}_complete ${action.arg.name} ${action.arg.action} ${action.cmd} ;;`);
        }
        if (command.hasCommands(false)) {
            actions.unshift(`command_args) _command_args ;;`);
        }
        if (actions.length) {
            return `\n\n  case "$state" in\n    ${actions.join("\n    ")}\n  esac`;
        }
        return "";
    }
}
function replaceSpecialChars2(str) {
    return str.replace(/[^a-zA-Z0-9]/g, "_");
}
class ZshCompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions for zsh.

To enable zsh completions for this program add following line to your ${dim1(italic("~/.zshrc"))}:

    ${dim1(italic(`source <(${baseCmd.getPath()} completions zsh)`))}`;
        }).noGlobals().option("-n, --name <command-name>", "The name of the main command.", {
            default: ()=>this.getMainCommand().getName()
        }).action(({ name })=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            console.log(ZshCompletionsGenerator.generate(name, baseCmd));
        });
    }
}
class CompleteCommand extends Command {
    constructor(cmd){
        super();
        return this.description("Get completions for given action from given command.").noGlobals().arguments("<action:string> [command...:string]").action(async (_, action, ...commandNames)=>{
            let parent;
            const completeCommand = commandNames?.reduce((cmd, name)=>{
                parent = cmd;
                const childCmd = cmd.getCommand(name, false);
                if (!childCmd) {
                    throw new UnknownCompletionCommandError(name, cmd.getCommands());
                }
                return childCmd;
            }, cmd || this.getMainCommand()) ?? (cmd || this.getMainCommand());
            const completion = completeCommand.getCompletion(action);
            const result1 = await completion?.complete(completeCommand, parent) ?? [];
            if (result1?.length) {
                Deno.stdout.writeSync(new TextEncoder().encode(result1.join("\n")));
            }
        }).reset();
    }
}
class CompletionsCommand extends Command {
    #cmd;
    constructor(cmd){
        super();
        this.#cmd = cmd;
        return this.description(()=>{
            const baseCmd = this.#cmd || this.getMainCommand();
            return `Generate shell completions.

To enable shell completions for this program add the following line to your ${dim1(italic("~/.bashrc"))} or similar:

    ${dim1(italic(`source <(${baseCmd.getPath()} completions [shell])`))}

    For more information run ${dim1(italic(`${baseCmd.getPath()} completions [shell] --help`))}
`;
        }).noGlobals().action(()=>this.showHelp()).command("bash", new BashCompletionsCommand(this.#cmd)).command("fish", new FishCompletionsCommand(this.#cmd)).command("zsh", new ZshCompletionsCommand(this.#cmd)).command("complete", new CompleteCommand(this.#cmd)).hidden().reset();
    }
}
function getFileInfoType(fileInfo) {
    return fileInfo.isFile ? "file" : fileInfo.isDirectory ? "dir" : fileInfo.isSymlink ? "symlink" : undefined;
}
async function ensureDir(dir) {
    try {
        const fileInfo = await Deno.lstat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
        return;
    } catch (err) {
        if (!(err instanceof Deno.errors.NotFound)) {
            throw err;
        }
    }
    try {
        await Deno.mkdir(dir, {
            recursive: true
        });
    } catch (err) {
        if (!(err instanceof Deno.errors.AlreadyExists)) {
            throw err;
        }
        const fileInfo = await Deno.lstat(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    }
}
function ensureDirSync(dir) {
    try {
        const fileInfo = Deno.lstatSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
        return;
    } catch (err) {
        if (!(err instanceof Deno.errors.NotFound)) {
            throw err;
        }
    }
    try {
        Deno.mkdirSync(dir, {
            recursive: true
        });
    } catch (err) {
        if (!(err instanceof Deno.errors.AlreadyExists)) {
            throw err;
        }
        const fileInfo = Deno.lstatSync(dir);
        if (!fileInfo.isDirectory) {
            throw new Error(`Ensure path exists, expected 'dir', got '${getFileInfoType(fileInfo)}'`);
        }
    }
}
Deno.build.os === "windows";
function isSubdir(src, dest, sep = SEPARATOR2) {
    if (src === dest) {
        return false;
    }
    src = toPathString(src);
    const srcArray = src.split(sep);
    dest = toPathString(dest);
    const destArray = dest.split(sep);
    return srcArray.every((current, i1)=>destArray[i1] === current);
}
function isSamePath(src, dest) {
    src = toPathString(src);
    dest = toPathString(dest);
    return resolve2(src) === resolve2(dest);
}
const EXISTS_ERROR = new Deno.errors.AlreadyExists("dest already exists.");
class SubdirectoryMoveError extends Error {
    constructor(src, dest){
        super(`Cannot move '${src}' to a subdirectory of itself, '${dest}'.`);
    }
}
function moveSync(src, dest, { overwrite = false } = {}) {
    const srcStat = Deno.statSync(src);
    if (srcStat.isDirectory && (isSubdir(src, dest) || isSamePath(src, dest))) {
        throw new SubdirectoryMoveError(src, dest);
    }
    if (overwrite) {
        if (isSamePath(src, dest)) return;
        try {
            Deno.removeSync(dest, {
                recursive: true
            });
        } catch (error) {
            if (!(error instanceof Deno.errors.NotFound)) {
                throw error;
            }
        }
    } else {
        try {
            Deno.lstatSync(dest);
            throw EXISTS_ERROR;
        } catch (error) {
            if (error === EXISTS_ERROR) {
                throw error;
            }
        }
    }
    Deno.renameSync(src, dest);
}
const isWindows2 = Deno.build.os === "windows";
async function ensureValidCopy(src, dest, options) {
    let destStat;
    try {
        destStat = await Deno.lstat(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Deno.errors.AlreadyExists(`'${dest}' already exists.`);
    }
    return destStat;
}
function ensureValidCopySync(src, dest, options) {
    let destStat;
    try {
        destStat = Deno.lstatSync(dest);
    } catch (err) {
        if (err instanceof Deno.errors.NotFound) {
            return;
        }
        throw err;
    }
    if (options.isFolder && !destStat.isDirectory) {
        throw new Error(`Cannot overwrite non-directory '${dest}' with directory '${src}'.`);
    }
    if (!options.overwrite) {
        throw new Deno.errors.AlreadyExists(`'${dest}' already exists.`);
    }
    return destStat;
}
async function copyFile(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    await Deno.copyFile(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = await Deno.stat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copyFileSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    Deno.copyFileSync(src, dest);
    if (options.preserveTimestamps) {
        const statInfo = Deno.statSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copySymLink(src, dest, options) {
    await ensureValidCopy(src, dest, options);
    const originSrcFilePath = await Deno.readLink(src);
    const type = getFileInfoType(await Deno.lstat(src));
    if (isWindows2) {
        await Deno.symlink(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        await Deno.symlink(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = await Deno.lstat(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, statInfo.atime, statInfo.mtime);
    }
}
function copySymlinkSync(src, dest, options) {
    ensureValidCopySync(src, dest, options);
    const originSrcFilePath = Deno.readLinkSync(src);
    const type = getFileInfoType(Deno.lstatSync(src));
    if (isWindows2) {
        Deno.symlinkSync(originSrcFilePath, dest, {
            type: type === "dir" ? "dir" : "file"
        });
    } else {
        Deno.symlinkSync(originSrcFilePath, dest);
    }
    if (options.preserveTimestamps) {
        const statInfo = Deno.lstatSync(src);
        assert(statInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(statInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, statInfo.atime, statInfo.mtime);
    }
}
async function copyDir(src, dest, options) {
    const destStat = await ensureValidCopy(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        await ensureDir(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = await Deno.stat(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        await Deno.utime(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    src = toPathString(src);
    dest = toPathString(dest);
    const promises = [];
    for await (const entry of Deno.readDir(src)){
        const srcPath = join2(src, entry.name);
        const destPath = join2(dest, basename2(srcPath));
        if (entry.isSymlink) {
            promises.push(copySymLink(srcPath, destPath, options));
        } else if (entry.isDirectory) {
            promises.push(copyDir(srcPath, destPath, options));
        } else if (entry.isFile) {
            promises.push(copyFile(srcPath, destPath, options));
        }
    }
    await Promise.all(promises);
}
function copyDirSync(src, dest, options) {
    const destStat = ensureValidCopySync(src, dest, {
        ...options,
        isFolder: true
    });
    if (!destStat) {
        ensureDirSync(dest);
    }
    if (options.preserveTimestamps) {
        const srcStatInfo = Deno.statSync(src);
        assert(srcStatInfo.atime instanceof Date, `statInfo.atime is unavailable`);
        assert(srcStatInfo.mtime instanceof Date, `statInfo.mtime is unavailable`);
        Deno.utimeSync(dest, srcStatInfo.atime, srcStatInfo.mtime);
    }
    src = toPathString(src);
    dest = toPathString(dest);
    for (const entry of Deno.readDirSync(src)){
        const srcPath = join2(src, entry.name);
        const destPath = join2(dest, basename2(srcPath));
        if (entry.isSymlink) {
            copySymlinkSync(srcPath, destPath, options);
        } else if (entry.isDirectory) {
            copyDirSync(srcPath, destPath, options);
        } else if (entry.isFile) {
            copyFileSync(srcPath, destPath, options);
        }
    }
}
async function copy1(src, dest, options = {}) {
    src = resolve2(toPathString(src));
    dest = resolve2(toPathString(dest));
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = await Deno.lstat(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        await copySymLink(src, dest, options);
    } else if (srcStat.isDirectory) {
        await copyDir(src, dest, options);
    } else if (srcStat.isFile) {
        await copyFile(src, dest, options);
    }
}
function copySync(src, dest, options = {}) {
    src = resolve2(toPathString(src));
    dest = resolve2(toPathString(dest));
    if (src === dest) {
        throw new Error("Source and destination cannot be the same.");
    }
    const srcStat = Deno.lstatSync(src);
    if (srcStat.isDirectory && isSubdir(src, dest)) {
        throw new Error(`Cannot copy '${src}' to a subdirectory of itself, '${dest}'.`);
    }
    if (srcStat.isSymlink) {
        copySymlinkSync(src, dest, options);
    } else if (srcStat.isDirectory) {
        copyDirSync(src, dest, options);
    } else if (srcStat.isFile) {
        copyFileSync(src, dest, options);
    }
}
const LF = "\n";
const CRLF = "\r\n";
const EOL = Deno?.build.os === "windows" ? CRLF : LF;
const regDetect = /(?:\r?\n)/g;
function format1(content, eol) {
    return content.replace(regDetect, eol);
}
class YAMLError extends Error {
    mark;
    constructor(message = "(unknown reason)", mark = ""){
        super(`${message} ${mark}`);
        this.mark = mark;
        this.name = this.constructor.name;
    }
    toString(_compact) {
        return `${this.name}: ${this.message} ${this.mark}`;
    }
}
function isBoolean(value) {
    return typeof value === "boolean" || value instanceof Boolean;
}
function isObject1(value) {
    return value !== null && typeof value === "object";
}
function repeat(str, count) {
    let result1 = "";
    for(let cycle = 0; cycle < count; cycle++){
        result1 += str;
    }
    return result1;
}
function isNegativeZero(i1) {
    return i1 === 0 && Number.NEGATIVE_INFINITY === 1 / i1;
}
class Mark {
    name;
    buffer;
    position;
    line;
    column;
    constructor(name, buffer, position, line, column){
        this.name = name;
        this.buffer = buffer;
        this.position = position;
        this.line = line;
        this.column = column;
    }
    getSnippet(indent = 4, maxLength = 75) {
        if (!this.buffer) return null;
        let head = "";
        let start = this.position;
        while(start > 0 && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(start - 1)) === -1){
            start -= 1;
            if (this.position - start > maxLength / 2 - 1) {
                head = " ... ";
                start += 5;
                break;
            }
        }
        let tail = "";
        let end = this.position;
        while(end < this.buffer.length && "\x00\r\n\x85\u2028\u2029".indexOf(this.buffer.charAt(end)) === -1){
            end += 1;
            if (end - this.position > maxLength / 2 - 1) {
                tail = " ... ";
                end -= 5;
                break;
            }
        }
        const snippet = this.buffer.slice(start, end);
        return `${repeat(" ", indent)}${head}${snippet}${tail}\n${repeat(" ", indent + this.position - start + head.length)}^`;
    }
    toString(compact) {
        let snippet, where = "";
        if (this.name) {
            where += `in "${this.name}" `;
        }
        where += `at line ${this.line + 1}, column ${this.column + 1}`;
        if (!compact) {
            snippet = this.getSnippet();
            if (snippet) {
                where += `:\n${snippet}`;
            }
        }
        return where;
    }
}
function compileList(schema, name, result1) {
    const exclude = [];
    for (const includedSchema of schema.include){
        result1 = compileList(includedSchema, name, result1);
    }
    for (const currentType of schema[name]){
        for(let previousIndex = 0; previousIndex < result1.length; previousIndex++){
            const previousType = result1[previousIndex];
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind) {
                exclude.push(previousIndex);
            }
        }
        result1.push(currentType);
    }
    return result1.filter((_type, index)=>!exclude.includes(index));
}
function compileMap(...typesList) {
    const result1 = {
        fallback: {},
        mapping: {},
        scalar: {},
        sequence: {}
    };
    for (const types of typesList){
        for (const type of types){
            if (type.kind !== null) {
                result1[type.kind][type.tag] = result1["fallback"][type.tag] = type;
            }
        }
    }
    return result1;
}
class Schema {
    static SCHEMA_DEFAULT;
    implicit;
    explicit;
    include;
    compiledImplicit;
    compiledExplicit;
    compiledTypeMap;
    constructor(definition){
        this.explicit = definition.explicit || [];
        this.implicit = definition.implicit || [];
        this.include = definition.include || [];
        for (const type of this.implicit){
            if (type.loadKind && type.loadKind !== "scalar") {
                throw new YAMLError("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
            }
        }
        this.compiledImplicit = compileList(this, "implicit", []);
        this.compiledExplicit = compileList(this, "explicit", []);
        this.compiledTypeMap = compileMap(this.compiledImplicit, this.compiledExplicit);
    }
    extend(definition) {
        return new Schema({
            implicit: [
                ...new Set([
                    ...this.implicit,
                    ...definition?.implicit ?? []
                ])
            ],
            explicit: [
                ...new Set([
                    ...this.explicit,
                    ...definition?.explicit ?? []
                ])
            ],
            include: [
                ...new Set([
                    ...this.include,
                    ...definition?.include ?? []
                ])
            ]
        });
    }
    static create() {}
}
function checkTagFormat(tag) {
    return tag;
}
class Type1 {
    tag;
    kind = null;
    instanceOf;
    predicate;
    represent;
    defaultStyle;
    styleAliases;
    loadKind;
    constructor(tag, options){
        this.tag = checkTagFormat(tag);
        if (options) {
            this.kind = options.kind;
            this.resolve = options.resolve || (()=>true);
            this.construct = options.construct || ((data)=>data);
            this.instanceOf = options.instanceOf;
            this.predicate = options.predicate;
            this.represent = options.represent;
            this.defaultStyle = options.defaultStyle;
            this.styleAliases = options.styleAliases;
        }
    }
    resolve = ()=>true;
    construct = (data)=>data;
}
const BASE64_MAP = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary(data) {
    if (data === null) return false;
    let code;
    let bitlen = 0;
    const max = data.length;
    const map = BASE64_MAP;
    for(let idx = 0; idx < max; idx++){
        code = map.indexOf(data.charAt(idx));
        if (code > 64) continue;
        if (code < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary(data) {
    const input = data.replace(/[\r\n=]/g, "");
    const max = input.length;
    const map = BASE64_MAP;
    const result1 = [];
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result1.push(bits >> 16 & 0xff);
            result1.push(bits >> 8 & 0xff);
            result1.push(bits & 0xff);
        }
        bits = bits << 6 | map.indexOf(input.charAt(idx));
    }
    const tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result1.push(bits >> 16 & 0xff);
        result1.push(bits >> 8 & 0xff);
        result1.push(bits & 0xff);
    } else if (tailbits === 18) {
        result1.push(bits >> 10 & 0xff);
        result1.push(bits >> 2 & 0xff);
    } else if (tailbits === 12) {
        result1.push(bits >> 4 & 0xff);
    }
    return new Uint8Array(result1);
}
function representYamlBinary(object) {
    const max = object.length;
    const map = BASE64_MAP;
    let result1 = "";
    let bits = 0;
    for(let idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result1 += map[bits >> 18 & 0x3f];
            result1 += map[bits >> 12 & 0x3f];
            result1 += map[bits >> 6 & 0x3f];
            result1 += map[bits & 0x3f];
        }
        bits = (bits << 8) + object[idx];
    }
    const tail = max % 3;
    if (tail === 0) {
        result1 += map[bits >> 18 & 0x3f];
        result1 += map[bits >> 12 & 0x3f];
        result1 += map[bits >> 6 & 0x3f];
        result1 += map[bits & 0x3f];
    } else if (tail === 2) {
        result1 += map[bits >> 10 & 0x3f];
        result1 += map[bits >> 4 & 0x3f];
        result1 += map[bits << 2 & 0x3f];
        result1 += map[64];
    } else if (tail === 1) {
        result1 += map[bits >> 2 & 0x3f];
        result1 += map[bits << 4 & 0x3f];
        result1 += map[64];
        result1 += map[64];
    }
    return result1;
}
function isBinary(obj) {
    return obj instanceof Uint8Array;
}
const binary = new Type1("tag:yaml.org,2002:binary", {
    construct: constructYamlBinary,
    kind: "scalar",
    predicate: isBinary,
    represent: representYamlBinary,
    resolve: resolveYamlBinary
});
function resolveYamlBoolean(data) {
    const max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
const bool = new Type1("tag:yaml.org,2002:bool", {
    construct: constructYamlBoolean,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isBoolean,
    represent: {
        lowercase (object) {
            return object ? "true" : "false";
        },
        uppercase (object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase (object) {
            return object ? "True" : "False";
        }
    },
    resolve: resolveYamlBoolean
});
const YAML_FLOAT_PATTERN = new RegExp("^(?:[-+]?(?:0|[1-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?" + "|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?" + "|[-+]?[0-9][0-9_]*(?::[0-5]?[0-9])+\\.[0-9_]*" + "|[-+]?\\.(?:inf|Inf|INF)" + "|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat(data) {
    if (!YAML_FLOAT_PATTERN.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat(data) {
    let value = data.replace(/_/g, "").toLowerCase();
    const sign = value[0] === "-" ? -1 : 1;
    const digits = [];
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    }
    if (value === ".nan") {
        return NaN;
    }
    if (value.indexOf(":") >= 0) {
        value.split(":").forEach((v)=>{
            digits.unshift(parseFloat(v));
        });
        let valueNb = 0.0;
        let base = 1;
        digits.forEach((d)=>{
            valueNb += d * base;
            base *= 60;
        });
        return sign * valueNb;
    }
    return sign * parseFloat(value);
}
const SCIENTIFIC_WITHOUT_DOT = /^[-+]?[0-9]+e/;
function representYamlFloat(object, style) {
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (isNegativeZero(object)) {
        return "-0.0";
    }
    const res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT.test(res) ? res.replace("e", ".e") : res;
}
function isFloat(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || isNegativeZero(object));
}
const __float = new Type1("tag:yaml.org,2002:float", {
    construct: constructYamlFloat,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isFloat,
    represent: representYamlFloat,
    resolve: resolveYamlFloat
});
function reconstructFunction(code) {
    const func = new Function(`return ${code}`)();
    if (!(func instanceof Function)) {
        throw new TypeError(`Expected function but got ${typeof func}: ${code}`);
    }
    return func;
}
new Type1("tag:yaml.org,2002:js/function", {
    kind: "scalar",
    resolve (data) {
        if (data === null) {
            return false;
        }
        try {
            reconstructFunction(`${data}`);
            return true;
        } catch (_err) {
            return false;
        }
    },
    construct (data) {
        return reconstructFunction(data);
    },
    predicate (object) {
        return object instanceof Function;
    },
    represent (object) {
        return object.toString();
    }
});
function isHexCode(c) {
    return 0x30 <= c && c <= 0x39 || 0x41 <= c && c <= 0x46 || 0x61 <= c && c <= 0x66;
}
function isOctCode(c) {
    return 0x30 <= c && c <= 0x37;
}
function isDecCode(c) {
    return 0x30 <= c && c <= 0x39;
}
function resolveYamlInteger(data) {
    const max = data.length;
    let index = 0;
    let hasDigits = false;
    if (!max) return false;
    let ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        for(; index < max; index++){
            ch = data[index];
            if (ch === "_") continue;
            if (!isOctCode(data.charCodeAt(index))) return false;
            hasDigits = true;
        }
        return hasDigits && ch !== "_";
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (ch === ":") break;
        if (!isDecCode(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    if (ch !== ":") return true;
    return /^(:[0-5]?[0-9])+$/.test(data.slice(index));
}
function constructYamlInteger(data) {
    let value = data;
    const digits = [];
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    let sign = 1;
    let ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value, 16);
        return sign * parseInt(value, 8);
    }
    if (value.indexOf(":") !== -1) {
        value.split(":").forEach((v)=>{
            digits.unshift(parseInt(v, 10));
        });
        let valueInt = 0;
        let base = 1;
        digits.forEach((d)=>{
            valueInt += d * base;
            base *= 60;
        });
        return sign * valueInt;
    }
    return sign * parseInt(value, 10);
}
function isInteger(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !isNegativeZero(object);
}
const __int = new Type1("tag:yaml.org,2002:int", {
    construct: constructYamlInteger,
    defaultStyle: "decimal",
    kind: "scalar",
    predicate: isInteger,
    represent: {
        binary (obj) {
            return obj >= 0 ? `0b${obj.toString(2)}` : `-0b${obj.toString(2).slice(1)}`;
        },
        octal (obj) {
            return obj >= 0 ? `0${obj.toString(8)}` : `-0${obj.toString(8).slice(1)}`;
        },
        decimal (obj) {
            return obj.toString(10);
        },
        hexadecimal (obj) {
            return obj >= 0 ? `0x${obj.toString(16).toUpperCase()}` : `-0x${obj.toString(16).toUpperCase().slice(1)}`;
        }
    },
    resolve: resolveYamlInteger,
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ],
        octal: [
            8,
            "oct"
        ]
    }
});
const map = new Type1("tag:yaml.org,2002:map", {
    construct (data) {
        return data !== null ? data : {};
    },
    kind: "mapping"
});
function resolveYamlMerge(data) {
    return data === "<<" || data === null;
}
const merge = new Type1("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge
});
function resolveYamlNull(data) {
    const max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull() {
    return null;
}
function isNull(object) {
    return object === null;
}
const nil = new Type1("tag:yaml.org,2002:null", {
    construct: constructYamlNull,
    defaultStyle: "lowercase",
    kind: "scalar",
    predicate: isNull,
    represent: {
        canonical () {
            return "~";
        },
        lowercase () {
            return "null";
        },
        uppercase () {
            return "NULL";
        },
        camelcase () {
            return "Null";
        }
    },
    resolve: resolveYamlNull
});
const { hasOwn } = Object;
const _toString = Object.prototype.toString;
function resolveYamlOmap(data) {
    const objectKeys = [];
    let pairKey = "";
    let pairHasKey = false;
    for (const pair of data){
        pairHasKey = false;
        if (_toString.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (hasOwn(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap(data) {
    return data !== null ? data : [];
}
const omap = new Type1("tag:yaml.org,2002:omap", {
    construct: constructYamlOmap,
    kind: "sequence",
    resolve: resolveYamlOmap
});
const _toString1 = Object.prototype.toString;
function resolveYamlPairs(data) {
    const result1 = Array.from({
        length: data.length
    });
    for(let index = 0; index < data.length; index++){
        const pair = data[index];
        if (_toString1.call(pair) !== "[object Object]") return false;
        const keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs(data) {
    if (data === null) return [];
    const result1 = Array.from({
        length: data.length
    });
    for(let index = 0; index < data.length; index += 1){
        const pair = data[index];
        const keys = Object.keys(pair);
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result1;
}
const pairs = new Type1("tag:yaml.org,2002:pairs", {
    construct: constructYamlPairs,
    kind: "sequence",
    resolve: resolveYamlPairs
});
const REGEXP = /^\/(?<regexp>[\s\S]+)\/(?<modifiers>[gismuy]*)$/;
const regexp = new Type1("tag:yaml.org,2002:js/regexp", {
    kind: "scalar",
    resolve (data) {
        if (data === null || !data.length) {
            return false;
        }
        const regexp = `${data}`;
        if (regexp.charAt(0) === "/") {
            if (!REGEXP.test(data)) {
                return false;
            }
            const modifiers = [
                ...regexp.match(REGEXP)?.groups?.modifiers ?? ""
            ];
            if (new Set(modifiers).size < modifiers.length) {
                return false;
            }
        }
        return true;
    },
    construct (data) {
        const { regexp = `${data}`, modifiers = "" } = `${data}`.match(REGEXP)?.groups ?? {};
        return new RegExp(regexp, modifiers);
    },
    predicate (object) {
        return object instanceof RegExp;
    },
    represent (object) {
        return object.toString();
    }
});
const seq = new Type1("tag:yaml.org,2002:seq", {
    construct (data) {
        return data !== null ? data : [];
    },
    kind: "sequence"
});
const { hasOwn: hasOwn1 } = Object;
function resolveYamlSet(data) {
    if (data === null) return true;
    for(const key in data){
        if (hasOwn1(data, key)) {
            if (data[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet(data) {
    return data !== null ? data : {};
}
const set = new Type1("tag:yaml.org,2002:set", {
    construct: constructYamlSet,
    kind: "mapping",
    resolve: resolveYamlSet
});
const str = new Type1("tag:yaml.org,2002:str", {
    construct (data) {
        return data !== null ? data : "";
    },
    kind: "scalar"
});
const YAML_DATE_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9])" + "-([0-9][0-9])$");
const YAML_TIMESTAMP_REGEXP = new RegExp("^([0-9][0-9][0-9][0-9])" + "-([0-9][0-9]?)" + "-([0-9][0-9]?)" + "(?:[Tt]|[ \\t]+)" + "([0-9][0-9]?)" + ":([0-9][0-9])" + ":([0-9][0-9])" + "(?:\\.([0-9]*))?" + "(?:[ \\t]*(Z|([-+])([0-9][0-9]?)" + "(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp(data) {
    let match = YAML_DATE_REGEXP.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP.exec(data);
    if (match === null) throw new Error("Date resolve error");
    const year = +match[1];
    const month = +match[2] - 1;
    const day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    const hour = +match[4];
    const minute = +match[5];
    const second = +match[6];
    let fraction = 0;
    if (match[7]) {
        let partFraction = match[7].slice(0, 3);
        while(partFraction.length < 3){
            partFraction += "0";
        }
        fraction = +partFraction;
    }
    let delta = null;
    if (match[9]) {
        const tzHour = +match[10];
        const tzMinute = +(match[11] || 0);
        delta = (tzHour * 60 + tzMinute) * 60000;
        if (match[9] === "-") delta = -delta;
    }
    const date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp(date) {
    return date.toISOString();
}
const timestamp = new Type1("tag:yaml.org,2002:timestamp", {
    construct: constructYamlTimestamp,
    instanceOf: Date,
    kind: "scalar",
    represent: representYamlTimestamp,
    resolve: resolveYamlTimestamp
});
const undefinedType = new Type1("tag:yaml.org,2002:js/undefined", {
    kind: "scalar",
    resolve () {
        return true;
    },
    construct () {
        return undefined;
    },
    predicate (object) {
        return typeof object === "undefined";
    },
    represent () {
        return "";
    }
});
const failsafe = new Schema({
    explicit: [
        str,
        seq,
        map
    ]
});
const json = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ]
});
const core = new Schema({
    include: [
        json
    ]
});
const def = new Schema({
    explicit: [
        binary,
        omap,
        pairs,
        set
    ],
    implicit: [
        timestamp,
        merge
    ],
    include: [
        core
    ]
});
new Schema({
    explicit: [
        regexp,
        undefinedType
    ],
    include: [
        def
    ]
});
class State {
    schema;
    constructor(schema = def){
        this.schema = schema;
    }
}
class LoaderState extends State {
    input;
    documents;
    length;
    lineIndent;
    lineStart;
    position;
    line;
    filename;
    onWarning;
    legacy;
    json;
    listener;
    implicitTypes;
    typeMap;
    version;
    checkLineBreaks;
    tagMap;
    anchorMap;
    tag;
    anchor;
    kind;
    result;
    constructor(input, { filename, schema, onWarning, legacy = false, json = false, listener = null }){
        super(schema);
        this.input = input;
        this.documents = [];
        this.lineIndent = 0;
        this.lineStart = 0;
        this.position = 0;
        this.line = 0;
        this.result = "";
        this.filename = filename;
        this.onWarning = onWarning;
        this.legacy = legacy;
        this.json = json;
        this.listener = listener;
        this.implicitTypes = this.schema.compiledImplicit;
        this.typeMap = this.schema.compiledTypeMap;
        this.length = input.length;
    }
}
const { hasOwn: hasOwn2 } = Object;
const CONTEXT_BLOCK_IN = 3;
const CONTEXT_BLOCK_OUT = 4;
const CHOMPING_STRIP = 2;
const CHOMPING_KEEP = 3;
const PATTERN_NON_PRINTABLE = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
const PATTERN_NON_ASCII_LINE_BREAKS = /[\x85\u2028\u2029]/;
const PATTERN_FLOW_INDICATORS = /[,\[\]\{\}]/;
const PATTERN_TAG_HANDLE = /^(?:!|!!|![a-z\-]+!)$/i;
const PATTERN_TAG_URI = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class(obj) {
    return Object.prototype.toString.call(obj);
}
function isEOL(c) {
    return c === 0x0a || c === 0x0d;
}
function isWhiteSpace(c) {
    return c === 0x09 || c === 0x20;
}
function isWsOrEol(c) {
    return c === 0x09 || c === 0x20 || c === 0x0a || c === 0x0d;
}
function isFlowIndicator(c) {
    return c === 0x2c || c === 0x5b || c === 0x5d || c === 0x7b || c === 0x7d;
}
function fromHexCode(c) {
    if (0x30 <= c && c <= 0x39) {
        return c - 0x30;
    }
    const lc = c | 0x20;
    if (0x61 <= lc && lc <= 0x66) {
        return lc - 0x61 + 10;
    }
    return -1;
}
function escapedHexLen(c) {
    if (c === 0x78) {
        return 2;
    }
    if (c === 0x75) {
        return 4;
    }
    if (c === 0x55) {
        return 8;
    }
    return 0;
}
function fromDecimalCode(c) {
    if (0x30 <= c && c <= 0x39) {
        return c - 0x30;
    }
    return -1;
}
function simpleEscapeSequence(c) {
    return c === 0x30 ? "\x00" : c === 0x61 ? "\x07" : c === 0x62 ? "\x08" : c === 0x74 ? "\x09" : c === 0x09 ? "\x09" : c === 0x6e ? "\x0A" : c === 0x76 ? "\x0B" : c === 0x66 ? "\x0C" : c === 0x72 ? "\x0D" : c === 0x65 ? "\x1B" : c === 0x20 ? " " : c === 0x22 ? "\x22" : c === 0x2f ? "/" : c === 0x5c ? "\x5C" : c === 0x4e ? "\x85" : c === 0x5f ? "\xA0" : c === 0x4c ? "\u2028" : c === 0x50 ? "\u2029" : "";
}
function charFromCodepoint(c) {
    if (c <= 0xffff) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 0x010000 >> 10) + 0xd800, (c - 0x010000 & 0x03ff) + 0xdc00);
}
const simpleEscapeCheck = Array.from({
    length: 256
});
const simpleEscapeMap = Array.from({
    length: 256
});
for(let i1 = 0; i1 < 256; i1++){
    simpleEscapeCheck[i1] = simpleEscapeSequence(i1) ? 1 : 0;
    simpleEscapeMap[i1] = simpleEscapeSequence(i1);
}
function generateError(state, message) {
    return new YAMLError(message, new Mark(state.filename, state.input, state.position, state.line, state.position - state.lineStart));
}
function throwError(state, message) {
    throw generateError(state, message);
}
function throwWarning(state, message) {
    if (state.onWarning) {
        state.onWarning.call(null, generateError(state, message));
    }
}
const directiveHandlers = {
    YAML (state, _name, ...args) {
        if (state.version !== null) {
            return throwError(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            return throwError(state, "YAML directive accepts exactly one argument");
        }
        const match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            return throwError(state, "ill-formed argument of the YAML directive");
        }
        const major = parseInt(match[1], 10);
        const minor = parseInt(match[2], 10);
        if (major !== 1) {
            return throwError(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            return throwWarning(state, "unsupported YAML version of the document");
        }
    },
    TAG (state, _name, ...args) {
        if (args.length !== 2) {
            return throwError(state, "TAG directive accepts exactly two arguments");
        }
        const handle = args[0];
        const prefix = args[1];
        if (!PATTERN_TAG_HANDLE.test(handle)) {
            return throwError(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (state.tagMap && hasOwn2(state.tagMap, handle)) {
            return throwError(state, `there is a previously declared suffix for "${handle}" tag handle`);
        }
        if (!PATTERN_TAG_URI.test(prefix)) {
            return throwError(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        if (typeof state.tagMap === "undefined") {
            state.tagMap = Object.create(null);
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment(state, start, end, checkJson) {
    let result1;
    if (start < end) {
        result1 = state.input.slice(start, end);
        if (checkJson) {
            for(let position = 0, length = result1.length; position < length; position++){
                const character = result1.charCodeAt(position);
                if (!(character === 0x09 || 0x20 <= character && character <= 0x10ffff)) {
                    return throwError(state, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE.test(result1)) {
            return throwError(state, "the stream contains non-printable characters");
        }
        state.result += result1;
    }
}
function mergeMappings(state, destination, source, overridableKeys) {
    if (!isObject1(source)) {
        return throwError(state, "cannot merge mappings; the provided source object is unacceptable");
    }
    const keys = Object.keys(source);
    for(let i1 = 0, len = keys.length; i1 < len; i1++){
        const key = keys[i1];
        if (!hasOwn2(destination, key)) {
            Object.defineProperty(destination, key, {
                value: source[key],
                writable: true,
                enumerable: true,
                configurable: true
            });
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, valueNode, startLine, startPos) {
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(let index = 0, quantity = keyNode.length; index < quantity; index++){
            if (Array.isArray(keyNode[index])) {
                return throwError(state, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (result1 === null) {
        result1 = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(let index = 0, quantity = valueNode.length; index < quantity; index++){
                mergeMappings(state, result1, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings(state, result1, valueNode, overridableKeys);
        }
    } else {
        if (!state.json && !hasOwn2(overridableKeys, keyNode) && hasOwn2(result1, keyNode)) {
            state.line = startLine || state.line;
            state.position = startPos || state.position;
            return throwError(state, "duplicated mapping key");
        }
        Object.defineProperty(result1, keyNode, {
            value: valueNode,
            writable: true,
            enumerable: true,
            configurable: true
        });
        delete overridableKeys[keyNode];
    }
    return result1;
}
function readLineBreak(state) {
    const ch = state.input.charCodeAt(state.position);
    if (ch === 0x0a) {
        state.position++;
    } else if (ch === 0x0d) {
        state.position++;
        if (state.input.charCodeAt(state.position) === 0x0a) {
            state.position++;
        }
    } else {
        return throwError(state, "a line break is expected");
    }
    state.line += 1;
    state.lineStart = state.position;
}
function skipSeparationSpace(state, allowComments, checkIndent) {
    let lineBreaks = 0, ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        while(isWhiteSpace(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        if (allowComments && ch === 0x23) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (ch !== 0x0a && ch !== 0x0d && ch !== 0)
        }
        if (isEOL(ch)) {
            readLineBreak(state);
            ch = state.input.charCodeAt(state.position);
            lineBreaks++;
            state.lineIndent = 0;
            while(ch === 0x20){
                state.lineIndent++;
                ch = state.input.charCodeAt(++state.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state.lineIndent < checkIndent) {
        throwWarning(state, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator(state) {
    let _position = state.position;
    let ch = state.input.charCodeAt(_position);
    if ((ch === 0x2d || ch === 0x2e) && ch === state.input.charCodeAt(_position + 1) && ch === state.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state.input.charCodeAt(_position);
        if (ch === 0 || isWsOrEol(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines(state, count) {
    if (count === 1) {
        state.result += " ";
    } else if (count > 1) {
        state.result += repeat("\n", count - 1);
    }
}
function readPlainScalar(state, nodeIndent, withinFlowCollection) {
    const kind = state.kind;
    const result1 = state.result;
    let ch = state.input.charCodeAt(state.position);
    if (isWsOrEol(ch) || isFlowIndicator(ch) || ch === 0x23 || ch === 0x26 || ch === 0x2a || ch === 0x21 || ch === 0x7c || ch === 0x3e || ch === 0x27 || ch === 0x22 || ch === 0x25 || ch === 0x40 || ch === 0x60) {
        return false;
    }
    let following;
    if (ch === 0x3f || ch === 0x2d) {
        following = state.input.charCodeAt(state.position + 1);
        if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
            return false;
        }
    }
    state.kind = "scalar";
    state.result = "";
    let captureEnd, captureStart = captureEnd = state.position;
    let hasPendingContent = false;
    let line = 0;
    while(ch !== 0){
        if (ch === 0x3a) {
            following = state.input.charCodeAt(state.position + 1);
            if (isWsOrEol(following) || withinFlowCollection && isFlowIndicator(following)) {
                break;
            }
        } else if (ch === 0x23) {
            const preceding = state.input.charCodeAt(state.position - 1);
            if (isWsOrEol(preceding)) {
                break;
            }
        } else if (state.position === state.lineStart && testDocumentSeparator(state) || withinFlowCollection && isFlowIndicator(ch)) {
            break;
        } else if (isEOL(ch)) {
            line = state.line;
            const lineStart = state.lineStart;
            const lineIndent = state.lineIndent;
            skipSeparationSpace(state, false, -1);
            if (state.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state.input.charCodeAt(state.position);
                continue;
            } else {
                state.position = captureEnd;
                state.line = line;
                state.lineStart = lineStart;
                state.lineIndent = lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment(state, captureStart, captureEnd, false);
            writeFoldedLines(state, state.line - line);
            captureStart = captureEnd = state.position;
            hasPendingContent = false;
        }
        if (!isWhiteSpace(ch)) {
            captureEnd = state.position + 1;
        }
        ch = state.input.charCodeAt(++state.position);
    }
    captureSegment(state, captureStart, captureEnd, false);
    if (state.result) {
        return true;
    }
    state.kind = kind;
    state.result = result1;
    return false;
}
function readSingleQuotedScalar(state, nodeIndent) {
    let ch, captureStart, captureEnd;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 0x27) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 0x27) {
            captureSegment(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (ch === 0x27) {
                captureStart = state.position;
                state.position++;
                captureEnd = state.position;
            } else {
                return true;
            }
        } else if (isEOL(ch)) {
            captureSegment(state, captureStart, captureEnd, true);
            writeFoldedLines(state, skipSeparationSpace(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator(state)) {
            return throwError(state, "unexpected end of the document within a single quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    return throwError(state, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar(state, nodeIndent) {
    let ch = state.input.charCodeAt(state.position);
    if (ch !== 0x22) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    let captureEnd, captureStart = captureEnd = state.position;
    let tmp;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 0x22) {
            captureSegment(state, captureStart, state.position, true);
            state.position++;
            return true;
        }
        if (ch === 0x5c) {
            captureSegment(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (isEOL(ch)) {
                skipSeparationSpace(state, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck[ch]) {
                state.result += simpleEscapeMap[ch];
                state.position++;
            } else if ((tmp = escapedHexLen(ch)) > 0) {
                let hexLength = tmp;
                let hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state.input.charCodeAt(++state.position);
                    if ((tmp = fromHexCode(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        return throwError(state, "expected hexadecimal character");
                    }
                }
                state.result += charFromCodepoint(hexResult);
                state.position++;
            } else {
                return throwError(state, "unknown escape sequence");
            }
            captureStart = captureEnd = state.position;
        } else if (isEOL(ch)) {
            captureSegment(state, captureStart, captureEnd, true);
            writeFoldedLines(state, skipSeparationSpace(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator(state)) {
            return throwError(state, "unexpected end of the document within a double quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    return throwError(state, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection(state, nodeIndent) {
    let ch = state.input.charCodeAt(state.position);
    let terminator;
    let isMapping = true;
    let result1 = {};
    if (ch === 0x5b) {
        terminator = 0x5d;
        isMapping = false;
        result1 = [];
    } else if (ch === 0x7b) {
        terminator = 0x7d;
    } else {
        return false;
    }
    if (state.anchor !== null && typeof state.anchor !== "undefined" && typeof state.anchorMap !== "undefined") {
        state.anchorMap[state.anchor] = result1;
    }
    ch = state.input.charCodeAt(++state.position);
    const tag = state.tag, anchor = state.anchor;
    let readNext = true;
    let valueNode, keyNode, keyTag = keyNode = valueNode = null, isExplicitPair, isPair = isExplicitPair = false;
    let following = 0, line = 0;
    const overridableKeys = Object.create(null);
    while(ch !== 0){
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === terminator) {
            state.position++;
            state.tag = tag;
            state.anchor = anchor;
            state.kind = isMapping ? "mapping" : "sequence";
            state.result = result1;
            return true;
        }
        if (!readNext) {
            return throwError(state, "missed comma between flow collection entries");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 0x3f) {
            following = state.input.charCodeAt(state.position + 1);
            if (isWsOrEol(following)) {
                isPair = isExplicitPair = true;
                state.position++;
                skipSeparationSpace(state, true, nodeIndent);
            }
        }
        line = state.line;
        composeNode(state, nodeIndent, 1, false, true);
        keyTag = state.tag || null;
        keyNode = state.result;
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if ((isExplicitPair || state.line === line) && ch === 0x3a) {
            isPair = true;
            ch = state.input.charCodeAt(++state.position);
            skipSeparationSpace(state, true, nodeIndent);
            composeNode(state, nodeIndent, 1, false, true);
            valueNode = state.result;
        }
        if (isMapping) {
            storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, valueNode);
        } else if (isPair) {
            result1.push(storeMappingPair(state, null, overridableKeys, keyTag, keyNode, valueNode));
        } else {
            result1.push(keyNode);
        }
        skipSeparationSpace(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === 0x2c) {
            readNext = true;
            ch = state.input.charCodeAt(++state.position);
        } else {
            readNext = false;
        }
    }
    return throwError(state, "unexpected end of the stream within a flow collection");
}
function readBlockScalar(state, nodeIndent) {
    let chomping = 1, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false;
    let ch = state.input.charCodeAt(state.position);
    let folding = false;
    if (ch === 0x7c) {
        folding = false;
    } else if (ch === 0x3e) {
        folding = true;
    } else {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    let tmp = 0;
    while(ch !== 0){
        ch = state.input.charCodeAt(++state.position);
        if (ch === 0x2b || ch === 0x2d) {
            if (1 === chomping) {
                chomping = ch === 0x2b ? CHOMPING_KEEP : CHOMPING_STRIP;
            } else {
                return throwError(state, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode(ch)) >= 0) {
            if (tmp === 0) {
                return throwError(state, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                return throwError(state, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (isWhiteSpace(ch)) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (isWhiteSpace(ch))
        if (ch === 0x23) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (!isEOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak(state);
        state.lineIndent = 0;
        ch = state.input.charCodeAt(state.position);
        while((!detectedIndent || state.lineIndent < textIndent) && ch === 0x20){
            state.lineIndent++;
            ch = state.input.charCodeAt(++state.position);
        }
        if (!detectedIndent && state.lineIndent > textIndent) {
            textIndent = state.lineIndent;
        }
        if (isEOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state.lineIndent < textIndent) {
            if (chomping === 3) {
                state.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === 1) {
                if (didReadContent) {
                    state.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (isWhiteSpace(ch)) {
                atMoreIndented = true;
                state.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state.result += repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state.result += " ";
                }
            } else {
                state.result += repeat("\n", emptyLines);
            }
        } else {
            state.result += repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        const captureStart = state.position;
        while(!isEOL(ch) && ch !== 0){
            ch = state.input.charCodeAt(++state.position);
        }
        captureSegment(state, captureStart, state.position, false);
    }
    return true;
}
function readBlockSequence(state, nodeIndent) {
    let line, following, detected = false, ch;
    const tag = state.tag, anchor = state.anchor, result1 = [];
    if (state.anchor !== null && typeof state.anchor !== "undefined" && typeof state.anchorMap !== "undefined") {
        state.anchorMap[state.anchor] = result1;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (ch !== 0x2d) {
            break;
        }
        following = state.input.charCodeAt(state.position + 1);
        if (!isWsOrEol(following)) {
            break;
        }
        detected = true;
        state.position++;
        if (skipSeparationSpace(state, true, -1)) {
            if (state.lineIndent <= nodeIndent) {
                result1.push(null);
                ch = state.input.charCodeAt(state.position);
                continue;
            }
        }
        line = state.line;
        composeNode(state, nodeIndent, 3, false, true);
        result1.push(state.result);
        skipSeparationSpace(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if ((state.line === line || state.lineIndent > nodeIndent) && ch !== 0) {
            return throwError(state, "bad indentation of a sequence entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state.tag = tag;
        state.anchor = anchor;
        state.kind = "sequence";
        state.result = result1;
        return true;
    }
    return false;
}
function readBlockMapping(state, nodeIndent, flowIndent) {
    const tag = state.tag, anchor = state.anchor, result1 = {}, overridableKeys = Object.create(null);
    let following, allowCompact = false, line, pos, keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state.anchor !== null && typeof state.anchor !== "undefined" && typeof state.anchorMap !== "undefined") {
        state.anchorMap[state.anchor] = result1;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        following = state.input.charCodeAt(state.position + 1);
        line = state.line;
        pos = state.position;
        if ((ch === 0x3f || ch === 0x3a) && isWsOrEol(following)) {
            if (ch === 0x3f) {
                if (atExplicitKey) {
                    storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, null);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                return throwError(state, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state.position += 1;
            ch = following;
        } else if (composeNode(state, flowIndent, 2, false, true)) {
            if (state.line === line) {
                ch = state.input.charCodeAt(state.position);
                while(isWhiteSpace(ch)){
                    ch = state.input.charCodeAt(++state.position);
                }
                if (ch === 0x3a) {
                    ch = state.input.charCodeAt(++state.position);
                    if (!isWsOrEol(ch)) {
                        return throwError(state, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, null);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state.tag;
                    keyNode = state.result;
                } else if (detected) {
                    return throwError(state, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state.tag = tag;
                    state.anchor = anchor;
                    return true;
                }
            } else if (detected) {
                return throwError(state, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state.tag = tag;
                state.anchor = anchor;
                return true;
            }
        } else {
            break;
        }
        if (state.line === line || state.lineIndent > nodeIndent) {
            if (composeNode(state, nodeIndent, 4, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state.result;
                } else {
                    valueNode = state.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, valueNode, line, pos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace(state, true, -1);
            ch = state.input.charCodeAt(state.position);
        }
        if (state.lineIndent > nodeIndent && ch !== 0) {
            return throwError(state, "bad indentation of a mapping entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair(state, result1, overridableKeys, keyTag, keyNode, null);
    }
    if (detected) {
        state.tag = tag;
        state.anchor = anchor;
        state.kind = "mapping";
        state.result = result1;
    }
    return detected;
}
function readTagProperty(state) {
    let position, isVerbatim = false, isNamed = false, tagHandle = "", tagName, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 0x21) return false;
    if (state.tag !== null) {
        return throwError(state, "duplication of a tag property");
    }
    ch = state.input.charCodeAt(++state.position);
    if (ch === 0x3c) {
        isVerbatim = true;
        ch = state.input.charCodeAt(++state.position);
    } else if (ch === 0x21) {
        isNamed = true;
        tagHandle = "!!";
        ch = state.input.charCodeAt(++state.position);
    } else {
        tagHandle = "!";
    }
    position = state.position;
    if (isVerbatim) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (ch !== 0 && ch !== 0x3e)
        if (state.position < state.length) {
            tagName = state.input.slice(position, state.position);
            ch = state.input.charCodeAt(++state.position);
        } else {
            return throwError(state, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !isWsOrEol(ch)){
            if (ch === 0x21) {
                if (!isNamed) {
                    tagHandle = state.input.slice(position - 1, state.position + 1);
                    if (!PATTERN_TAG_HANDLE.test(tagHandle)) {
                        return throwError(state, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    position = state.position + 1;
                } else {
                    return throwError(state, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state.input.charCodeAt(++state.position);
        }
        tagName = state.input.slice(position, state.position);
        if (PATTERN_FLOW_INDICATORS.test(tagName)) {
            return throwError(state, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI.test(tagName)) {
        return throwError(state, `tag name cannot contain such characters: ${tagName}`);
    }
    if (isVerbatim) {
        state.tag = tagName;
    } else if (typeof state.tagMap !== "undefined" && hasOwn2(state.tagMap, tagHandle)) {
        state.tag = state.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state.tag = `!${tagName}`;
    } else if (tagHandle === "!!") {
        state.tag = `tag:yaml.org,2002:${tagName}`;
    } else {
        return throwError(state, `undeclared tag handle "${tagHandle}"`);
    }
    return true;
}
function readAnchorProperty(state) {
    let ch = state.input.charCodeAt(state.position);
    if (ch !== 0x26) return false;
    if (state.anchor !== null) {
        return throwError(state, "duplication of an anchor property");
    }
    ch = state.input.charCodeAt(++state.position);
    const position = state.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === position) {
        return throwError(state, "name of an anchor node must contain at least one character");
    }
    state.anchor = state.input.slice(position, state.position);
    return true;
}
function readAlias(state) {
    let ch = state.input.charCodeAt(state.position);
    if (ch !== 0x2a) return false;
    ch = state.input.charCodeAt(++state.position);
    const _position = state.position;
    while(ch !== 0 && !isWsOrEol(ch) && !isFlowIndicator(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        return throwError(state, "name of an alias node must contain at least one character");
    }
    const alias = state.input.slice(_position, state.position);
    if (typeof state.anchorMap !== "undefined" && !hasOwn2(state.anchorMap, alias)) {
        return throwError(state, `unidentified alias "${alias}"`);
    }
    if (typeof state.anchorMap !== "undefined") {
        state.result = state.anchorMap[alias];
    }
    skipSeparationSpace(state, true, -1);
    return true;
}
function composeNode(state, parentIndent, nodeContext, allowToSeek, allowCompact) {
    let allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, type, flowIndent, blockIndent;
    if (state.listener && state.listener !== null) {
        state.listener("open", state);
    }
    state.tag = null;
    state.anchor = null;
    state.kind = null;
    state.result = null;
    const allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT === nodeContext || CONTEXT_BLOCK_IN === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace(state, true, -1)) {
            atNewLine = true;
            if (state.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty(state) || readAnchorProperty(state)){
            if (skipSeparationSpace(state, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || 4 === nodeContext) {
        const cond = 1 === nodeContext || 2 === nodeContext;
        flowIndent = cond ? parentIndent : parentIndent + 1;
        blockIndent = state.position - state.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence(state, blockIndent) || readBlockMapping(state, blockIndent, flowIndent)) || readFlowCollection(state, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar(state, flowIndent) || readSingleQuotedScalar(state, flowIndent) || readDoubleQuotedScalar(state, flowIndent)) {
                    hasContent = true;
                } else if (readAlias(state)) {
                    hasContent = true;
                    if (state.tag !== null || state.anchor !== null) {
                        return throwError(state, "alias node should not have Any properties");
                    }
                } else if (readPlainScalar(state, flowIndent, 1 === nodeContext)) {
                    hasContent = true;
                    if (state.tag === null) {
                        state.tag = "?";
                    }
                }
                if (state.anchor !== null && typeof state.anchorMap !== "undefined") {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence(state, blockIndent);
        }
    }
    if (state.tag !== null && state.tag !== "!") {
        if (state.tag === "?") {
            for(let typeIndex = 0, typeQuantity = state.implicitTypes.length; typeIndex < typeQuantity; typeIndex++){
                type = state.implicitTypes[typeIndex];
                if (type.resolve(state.result)) {
                    state.result = type.construct(state.result);
                    state.tag = type.tag;
                    if (state.anchor !== null && typeof state.anchorMap !== "undefined") {
                        state.anchorMap[state.anchor] = state.result;
                    }
                    break;
                }
            }
        } else if (hasOwn2(state.typeMap[state.kind || "fallback"], state.tag)) {
            type = state.typeMap[state.kind || "fallback"][state.tag];
            if (state.result !== null && type.kind !== state.kind) {
                return throwError(state, `unacceptable node kind for !<${state.tag}> tag; it should be "${type.kind}", not "${state.kind}"`);
            }
            if (!type.resolve(state.result)) {
                return throwError(state, `cannot resolve a node with !<${state.tag}> explicit tag`);
            } else {
                state.result = type.construct(state.result);
                if (state.anchor !== null && typeof state.anchorMap !== "undefined") {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else {
            return throwError(state, `unknown tag !<${state.tag}>`);
        }
    }
    if (state.listener && state.listener !== null) {
        state.listener("close", state);
    }
    return state.tag !== null || state.anchor !== null || hasContent;
}
function readDocument(state) {
    const documentStart = state.position;
    let position, directiveName, directiveArgs, hasDirectives = false, ch;
    state.version = null;
    state.checkLineBreaks = state.legacy;
    state.tagMap = Object.create(null);
    state.anchorMap = Object.create(null);
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        skipSeparationSpace(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if (state.lineIndent > 0 || ch !== 0x25) {
            break;
        }
        hasDirectives = true;
        ch = state.input.charCodeAt(++state.position);
        position = state.position;
        while(ch !== 0 && !isWsOrEol(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        directiveName = state.input.slice(position, state.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            return throwError(state, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(isWhiteSpace(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            if (ch === 0x23) {
                do {
                    ch = state.input.charCodeAt(++state.position);
                }while (ch !== 0 && !isEOL(ch))
                break;
            }
            if (isEOL(ch)) break;
            position = state.position;
            while(ch !== 0 && !isWsOrEol(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            directiveArgs.push(state.input.slice(position, state.position));
        }
        if (ch !== 0) readLineBreak(state);
        if (hasOwn2(directiveHandlers, directiveName)) {
            directiveHandlers[directiveName](state, directiveName, ...directiveArgs);
        } else {
            throwWarning(state, `unknown document directive "${directiveName}"`);
        }
    }
    skipSeparationSpace(state, true, -1);
    if (state.lineIndent === 0 && state.input.charCodeAt(state.position) === 0x2d && state.input.charCodeAt(state.position + 1) === 0x2d && state.input.charCodeAt(state.position + 2) === 0x2d) {
        state.position += 3;
        skipSeparationSpace(state, true, -1);
    } else if (hasDirectives) {
        return throwError(state, "directives end mark is expected");
    }
    composeNode(state, state.lineIndent - 1, 4, false, true);
    skipSeparationSpace(state, true, -1);
    if (state.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS.test(state.input.slice(documentStart, state.position))) {
        throwWarning(state, "non-ASCII line breaks are interpreted as content");
    }
    state.documents.push(state.result);
    if (state.position === state.lineStart && testDocumentSeparator(state)) {
        if (state.input.charCodeAt(state.position) === 0x2e) {
            state.position += 3;
            skipSeparationSpace(state, true, -1);
        }
        return;
    }
    if (state.position < state.length - 1) {
        return throwError(state, "end of the stream or a document separator is expected");
    }
}
function loadDocuments(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 0x0a && input.charCodeAt(input.length - 1) !== 0x0d) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 0xfeff) {
            input = input.slice(1);
        }
    }
    const state = new LoaderState(input, options);
    state.input += "\0";
    while(state.input.charCodeAt(state.position) === 0x20){
        state.lineIndent += 1;
        state.position += 1;
    }
    while(state.position < state.length - 1){
        readDocument(state);
    }
    return state.documents;
}
function load(input, options) {
    const documents = loadDocuments(input, options);
    if (documents.length === 0) {
        return null;
    }
    if (documents.length === 1) {
        return documents[0];
    }
    throw new YAMLError("expected a single document in the stream, but found more");
}
function parse1(content, options) {
    return load(content, options);
}
const { hasOwn: hasOwn3 } = Object;
function compileStyleMap(schema, map) {
    if (typeof map === "undefined" || map === null) return {};
    let type;
    const result1 = {};
    const keys = Object.keys(map);
    let tag, style;
    for(let index = 0, length = keys.length; index < length; index += 1){
        tag = keys[index];
        style = String(map[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = `tag:yaml.org,2002:${tag.slice(2)}`;
        }
        type = schema.compiledTypeMap.fallback[tag];
        if (type && typeof type.styleAliases !== "undefined" && hasOwn3(type.styleAliases, style)) {
            style = type.styleAliases[style];
        }
        result1[tag] = style;
    }
    return result1;
}
class DumperState extends State {
    indent;
    noArrayIndent;
    skipInvalid;
    flowLevel;
    sortKeys;
    lineWidth;
    noRefs;
    noCompatMode;
    condenseFlow;
    implicitTypes;
    explicitTypes;
    tag = null;
    result = "";
    duplicates = [];
    usedDuplicates = [];
    styleMap;
    dump;
    constructor({ schema, indent = 2, noArrayIndent = false, skipInvalid = false, flowLevel = -1, styles = null, sortKeys = false, lineWidth = 80, noRefs = false, noCompatMode = false, condenseFlow = false }){
        super(schema);
        this.indent = Math.max(1, indent);
        this.noArrayIndent = noArrayIndent;
        this.skipInvalid = skipInvalid;
        this.flowLevel = flowLevel;
        this.styleMap = compileStyleMap(this.schema, styles);
        this.sortKeys = sortKeys;
        this.lineWidth = lineWidth;
        this.noRefs = noRefs;
        this.noCompatMode = noCompatMode;
        this.condenseFlow = condenseFlow;
        this.implicitTypes = this.schema.compiledImplicit;
        this.explicitTypes = this.schema.compiledExplicit;
    }
}
const _toString2 = Object.prototype.toString;
const { hasOwn: hasOwn4 } = Object;
const ESCAPE_SEQUENCES = {};
ESCAPE_SEQUENCES[0x00] = "\\0";
ESCAPE_SEQUENCES[0x07] = "\\a";
ESCAPE_SEQUENCES[0x08] = "\\b";
ESCAPE_SEQUENCES[0x09] = "\\t";
ESCAPE_SEQUENCES[0x0a] = "\\n";
ESCAPE_SEQUENCES[0x0b] = "\\v";
ESCAPE_SEQUENCES[0x0c] = "\\f";
ESCAPE_SEQUENCES[0x0d] = "\\r";
ESCAPE_SEQUENCES[0x1b] = "\\e";
ESCAPE_SEQUENCES[0x22] = '\\"';
ESCAPE_SEQUENCES[0x5c] = "\\\\";
ESCAPE_SEQUENCES[0x85] = "\\N";
ESCAPE_SEQUENCES[0xa0] = "\\_";
ESCAPE_SEQUENCES[0x2028] = "\\L";
ESCAPE_SEQUENCES[0x2029] = "\\P";
const DEPRECATED_BOOLEANS_SYNTAX = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF"
];
function encodeHex(character) {
    const string = character.toString(16).toUpperCase();
    let handle;
    let length;
    if (character <= 0xff) {
        handle = "x";
        length = 2;
    } else if (character <= 0xffff) {
        handle = "u";
        length = 4;
    } else if (character <= 0xffffffff) {
        handle = "U";
        length = 8;
    } else {
        throw new YAMLError("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return `\\${handle}${repeat("0", length - string.length)}${string}`;
}
function indentString(string, spaces) {
    const ind = repeat(" ", spaces), length = string.length;
    let position = 0, next = -1, result1 = "", line;
    while(position < length){
        next = string.indexOf("\n", position);
        if (next === -1) {
            line = string.slice(position);
            position = length;
        } else {
            line = string.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result1 += ind;
        result1 += line;
    }
    return result1;
}
function generateNextLine(state, level) {
    return `\n${repeat(" ", state.indent * level)}`;
}
function testImplicitResolving(state, str) {
    let type;
    for(let index = 0, length = state.implicitTypes.length; index < length; index += 1){
        type = state.implicitTypes[index];
        if (type.resolve(str)) {
            return true;
        }
    }
    return false;
}
function isWhitespace(c) {
    return c === 0x20 || c === 0x09;
}
function isPrintable(c) {
    return 0x00020 <= c && c <= 0x00007e || 0x000a1 <= c && c <= 0x00d7ff && c !== 0x2028 && c !== 0x2029 || 0x0e000 <= c && c <= 0x00fffd && c !== 0xfeff || 0x10000 <= c && c <= 0x10ffff;
}
function isPlainSafe(c) {
    return isPrintable(c) && c !== 0xfeff && c !== 0x2c && c !== 0x5b && c !== 0x5d && c !== 0x7b && c !== 0x7d && c !== 0x3a && c !== 0x23;
}
function isPlainSafeFirst(c) {
    return isPrintable(c) && c !== 0xfeff && !isWhitespace(c) && c !== 0x2d && c !== 0x3f && c !== 0x3a && c !== 0x2c && c !== 0x5b && c !== 0x5d && c !== 0x7b && c !== 0x7d && c !== 0x23 && c !== 0x26 && c !== 0x2a && c !== 0x21 && c !== 0x7c && c !== 0x3e && c !== 0x27 && c !== 0x22 && c !== 0x25 && c !== 0x40 && c !== 0x60;
}
function needIndentIndicator(string) {
    const leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string);
}
const STYLE_PLAIN = 1, STYLE_SINGLE = 2, STYLE_LITERAL = 3, STYLE_FOLDED = 4, STYLE_DOUBLE = 5;
function chooseScalarStyle(string, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType) {
    const shouldTrackWidth = lineWidth !== -1;
    let hasLineBreak = false, hasFoldableLine = false, previousLineBreak = -1, plain = isPlainSafeFirst(string.charCodeAt(0)) && !isWhitespace(string.charCodeAt(string.length - 1));
    let __char, i1;
    if (singleLineOnly) {
        for(i1 = 0; i1 < string.length; i1++){
            __char = string.charCodeAt(i1);
            if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
    } else {
        for(i1 = 0; i1 < string.length; i1++){
            __char = string.charCodeAt(i1);
            if (__char === 0x0a) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable(__char)) {
                return 5;
            }
            plain = plain && isPlainSafe(__char);
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        return plain && !testAmbiguousType(string) ? 1 : 2;
    }
    if (indentPerLevel > 9 && needIndentIndicator(string)) {
        return 5;
    }
    return hasFoldableLine ? 4 : 3;
}
function foldLine(line, width) {
    if (line === "" || line[0] === " ") return line;
    const breakRe = / [^ ]/g;
    let match;
    let start = 0, end, curr = 0, next = 0;
    let result1 = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result1 += `\n${line.slice(start, end)}`;
            start = end + 1;
        }
        curr = next;
    }
    result1 += "\n";
    if (line.length - start > width && curr > start) {
        result1 += `${line.slice(start, curr)}\n${line.slice(curr + 1)}`;
    } else {
        result1 += line.slice(start);
    }
    return result1.slice(1);
}
function dropEndingNewline(string) {
    return string[string.length - 1] === "\n" ? string.slice(0, -1) : string;
}
function foldString(string, width) {
    const lineRe = /(\n+)([^\n]*)/g;
    let result1 = (()=>{
        let nextLF = string.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string.length;
        lineRe.lastIndex = nextLF;
        return foldLine(string.slice(0, nextLF), width);
    })();
    let prevMoreIndented = string[0] === "\n" || string[0] === " ";
    let moreIndented;
    let match;
    while(match = lineRe.exec(string)){
        const prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result1 += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine(line, width);
        prevMoreIndented = moreIndented;
    }
    return result1;
}
function escapeString(string) {
    let result1 = "";
    let __char, nextChar;
    let escapeSeq;
    for(let i1 = 0; i1 < string.length; i1++){
        __char = string.charCodeAt(i1);
        if (__char >= 0xd800 && __char <= 0xdbff) {
            nextChar = string.charCodeAt(i1 + 1);
            if (nextChar >= 0xdc00 && nextChar <= 0xdfff) {
                result1 += encodeHex((__char - 0xd800) * 0x400 + nextChar - 0xdc00 + 0x10000);
                i1++;
                continue;
            }
        }
        escapeSeq = ESCAPE_SEQUENCES[__char];
        result1 += !escapeSeq && isPrintable(__char) ? string[i1] : escapeSeq || encodeHex(__char);
    }
    return result1;
}
function blockHeader(string, indentPerLevel) {
    const indentIndicator = needIndentIndicator(string) ? String(indentPerLevel) : "";
    const clip = string[string.length - 1] === "\n";
    const keep = clip && (string[string.length - 2] === "\n" || string === "\n");
    const chomp = keep ? "+" : clip ? "" : "-";
    return `${indentIndicator}${chomp}\n`;
}
function writeScalar(state, string, level, iskey) {
    state.dump = (()=>{
        if (string.length === 0) {
            return "''";
        }
        if (!state.noCompatMode && DEPRECATED_BOOLEANS_SYNTAX.indexOf(string) !== -1) {
            return `'${string}'`;
        }
        const indent = state.indent * Math.max(1, level);
        const lineWidth = state.lineWidth === -1 ? -1 : Math.max(Math.min(state.lineWidth, 40), state.lineWidth - indent);
        const singleLineOnly = iskey || state.flowLevel > -1 && level >= state.flowLevel;
        function testAmbiguity(str) {
            return testImplicitResolving(state, str);
        }
        switch(chooseScalarStyle(string, singleLineOnly, state.indent, lineWidth, testAmbiguity)){
            case STYLE_PLAIN:
                return string;
            case STYLE_SINGLE:
                return `'${string.replace(/'/g, "''")}'`;
            case STYLE_LITERAL:
                return `|${blockHeader(string, state.indent)}${dropEndingNewline(indentString(string, indent))}`;
            case STYLE_FOLDED:
                return `>${blockHeader(string, state.indent)}${dropEndingNewline(indentString(foldString(string, lineWidth), indent))}`;
            case STYLE_DOUBLE:
                return `"${escapeString(string)}"`;
            default:
                throw new YAMLError("impossible error: invalid scalar style");
        }
    })();
}
function writeFlowSequence(state, level, object) {
    let _result = "";
    const _tag = state.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state, level, object[index], false, false)) {
            if (index !== 0) _result += `,${!state.condenseFlow ? " " : ""}`;
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = `[${_result}]`;
}
function writeBlockSequence(state, level, object, compact = false) {
    let _result = "";
    const _tag = state.tag;
    for(let index = 0, length = object.length; index < length; index += 1){
        if (writeNode(state, level + 1, object[index], true, true)) {
            if (!compact || index !== 0) {
                _result += generateNextLine(state, level);
            }
            if (state.dump && 0x0a === state.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = _result || "[]";
}
function writeFlowMapping(state, level, object) {
    let _result = "";
    const _tag = state.tag, objectKeyList = Object.keys(object);
    let pairBuffer, objectKey, objectValue;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = state.condenseFlow ? '"' : "";
        if (index !== 0) pairBuffer += ", ";
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state, level, objectKey, false, false)) {
            continue;
        }
        if (state.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += `${state.dump}${state.condenseFlow ? '"' : ""}:${state.condenseFlow ? "" : " "}`;
        if (!writeNode(state, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = `{${_result}}`;
}
function writeBlockMapping(state, level, object, compact = false) {
    const _tag = state.tag, objectKeyList = Object.keys(object);
    let _result = "";
    if (state.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state.sortKeys === "function") {
        objectKeyList.sort(state.sortKeys);
    } else if (state.sortKeys) {
        throw new YAMLError("sortKeys must be a boolean or a function");
    }
    let pairBuffer = "", objectKey, objectValue, explicitPair;
    for(let index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || index !== 0) {
            pairBuffer += generateNextLine(state, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (!writeNode(state, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state.tag !== null && state.tag !== "?" || state.dump && state.dump.length > 1024;
        if (explicitPair) {
            if (state.dump && 0x0a === state.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine(state, level);
        }
        if (!writeNode(state, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state.dump && 0x0a === state.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = _result || "{}";
}
function detectType(state, object, explicit = false) {
    const typeList = explicit ? state.explicitTypes : state.implicitTypes;
    let type;
    let style;
    let _result;
    for(let index = 0, length = typeList.length; index < length; index += 1){
        type = typeList[index];
        if ((type.instanceOf || type.predicate) && (!type.instanceOf || typeof object === "object" && object instanceof type.instanceOf) && (!type.predicate || type.predicate(object))) {
            state.tag = explicit ? type.tag : "?";
            if (type.represent) {
                style = state.styleMap[type.tag] || type.defaultStyle;
                if (_toString2.call(type.represent) === "[object Function]") {
                    _result = type.represent(object, style);
                } else if (hasOwn4(type.represent, style)) {
                    _result = type.represent[style](object, style);
                } else {
                    throw new YAMLError(`!<${type.tag}> tag resolver accepts not "${style}" style`);
                }
                state.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode(state, level, object, block, compact, iskey = false) {
    state.tag = null;
    state.dump = object;
    if (!detectType(state, object, false)) {
        detectType(state, object, true);
    }
    const type = _toString2.call(state.dump);
    if (block) {
        block = state.flowLevel < 0 || state.flowLevel > level;
    }
    const objectOrArray = type === "[object Object]" || type === "[object Array]";
    let duplicateIndex = -1;
    let duplicate = false;
    if (objectOrArray) {
        duplicateIndex = state.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state.tag !== null && state.tag !== "?" || duplicate || state.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state.usedDuplicates[duplicateIndex]) {
        state.dump = `*ref_${duplicateIndex}`;
    } else {
        if (objectOrArray && duplicate && !state.usedDuplicates[duplicateIndex]) {
            state.usedDuplicates[duplicateIndex] = true;
        }
        if (type === "[object Object]") {
            if (block && Object.keys(state.dump).length !== 0) {
                writeBlockMapping(state, level, state.dump, compact);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex}${state.dump}`;
                }
            } else {
                writeFlowMapping(state, level, state.dump);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex} ${state.dump}`;
                }
            }
        } else if (type === "[object Array]") {
            const arrayLevel = state.noArrayIndent && level > 0 ? level - 1 : level;
            if (block && state.dump.length !== 0) {
                writeBlockSequence(state, arrayLevel, state.dump, compact);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex}${state.dump}`;
                }
            } else {
                writeFlowSequence(state, arrayLevel, state.dump);
                if (duplicate) {
                    state.dump = `&ref_${duplicateIndex} ${state.dump}`;
                }
            }
        } else if (type === "[object String]") {
            if (state.tag !== "?") {
                writeScalar(state, state.dump, level, iskey);
            }
        } else {
            if (state.skipInvalid) return false;
            throw new YAMLError(`unacceptable kind of an object to dump ${type}`);
        }
        if (state.tag !== null && state.tag !== "?") {
            state.dump = `!<${state.tag}> ${state.dump}`;
        }
    }
    return true;
}
function inspectNode(object, objects, duplicatesIndexes) {
    if (object !== null && typeof object === "object") {
        const index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(let idx = 0, length = object.length; idx < length; idx += 1){
                    inspectNode(object[idx], objects, duplicatesIndexes);
                }
            } else {
                const objectKeyList = Object.keys(object);
                for(let idx = 0, length = objectKeyList.length; idx < length; idx += 1){
                    inspectNode(object[objectKeyList[idx]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function getDuplicateReferences(object, state) {
    const objects = [], duplicatesIndexes = [];
    inspectNode(object, objects, duplicatesIndexes);
    const length = duplicatesIndexes.length;
    for(let index = 0; index < length; index += 1){
        state.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state.usedDuplicates = Array.from({
        length
    });
}
function dump(input, options) {
    options = options || {};
    const state = new DumperState(options);
    if (!state.noRefs) getDuplicateReferences(input, state);
    if (writeNode(state, 0, input, true, true)) return `${state.dump}\n`;
    return "";
}
function stringify(obj, options) {
    return dump(obj, options);
}
const State1 = {
    PASSTHROUGH: 0,
    PERCENT: 1,
    POSITIONAL: 2,
    PRECISION: 3,
    WIDTH: 4
};
const WorP = {
    WIDTH: 0,
    PRECISION: 1
};
const F = {
    sign: 1,
    mantissa: 2,
    fractional: 3,
    esign: 4,
    exponent: 5
};
class Flags {
    plus;
    dash;
    sharp;
    space;
    zero;
    lessthan;
    width = -1;
    precision = -1;
}
const min = Math.min;
const UNICODE_REPLACEMENT_CHARACTER = "\ufffd";
const FLOAT_REGEXP = /(-?)(\d)\.?(\d*)e([+-])(\d+)/;
class Printf {
    format;
    args;
    i;
    state = State1.PASSTHROUGH;
    verb = "";
    buf = "";
    argNum = 0;
    flags = new Flags();
    haveSeen;
    tmpError;
    constructor(format, ...args){
        this.format = format;
        this.args = args;
        this.haveSeen = Array.from({
            length: args.length
        });
        this.i = 0;
    }
    doPrintf() {
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State1.PASSTHROUGH:
                    if (c === "%") {
                        this.state = State1.PERCENT;
                    } else {
                        this.buf += c;
                    }
                    break;
                case State1.PERCENT:
                    if (c === "%") {
                        this.buf += c;
                        this.state = State1.PASSTHROUGH;
                    } else {
                        this.handleFormat();
                    }
                    break;
                default:
                    throw Error("Should be unreachable, certainly a bug in the lib.");
            }
        }
        let extras = false;
        let err = "%!(EXTRA";
        for(let i1 = 0; i1 !== this.haveSeen.length; ++i1){
            if (!this.haveSeen[i1]) {
                extras = true;
                err += ` '${Deno.inspect(this.args[i1])}'`;
            }
        }
        err += ")";
        if (extras) {
            this.buf += err;
        }
        return this.buf;
    }
    handleFormat() {
        this.flags = new Flags();
        const flags = this.flags;
        for(; this.i < this.format.length; ++this.i){
            const c = this.format[this.i];
            switch(this.state){
                case State1.PERCENT:
                    switch(c){
                        case "[":
                            this.handlePositional();
                            this.state = State1.POSITIONAL;
                            break;
                        case "+":
                            flags.plus = true;
                            break;
                        case "<":
                            flags.lessthan = true;
                            break;
                        case "-":
                            flags.dash = true;
                            flags.zero = false;
                            break;
                        case "#":
                            flags.sharp = true;
                            break;
                        case " ":
                            flags.space = true;
                            break;
                        case "0":
                            flags.zero = !flags.dash;
                            break;
                        default:
                            if ("1" <= c && c <= "9" || c === "." || c === "*") {
                                if (c === ".") {
                                    this.flags.precision = 0;
                                    this.state = State1.PRECISION;
                                    this.i++;
                                } else {
                                    this.state = State1.WIDTH;
                                }
                                this.handleWidthAndPrecision(flags);
                            } else {
                                this.handleVerb();
                                return;
                            }
                    }
                    break;
                case State1.POSITIONAL:
                    if (c === "*") {
                        const worp = this.flags.precision === -1 ? WorP.WIDTH : WorP.PRECISION;
                        this.handleWidthOrPrecisionRef(worp);
                        this.state = State1.PERCENT;
                        break;
                    } else {
                        this.handleVerb();
                        return;
                    }
                default:
                    throw new Error(`Should not be here ${this.state}, library bug!`);
            }
        }
    }
    handleWidthOrPrecisionRef(wOrP) {
        if (this.argNum >= this.args.length) {
            return;
        }
        const arg = this.args[this.argNum];
        this.haveSeen[this.argNum] = true;
        if (typeof arg === "number") {
            switch(wOrP){
                case WorP.WIDTH:
                    this.flags.width = arg;
                    break;
                default:
                    this.flags.precision = arg;
            }
        } else {
            const tmp = wOrP === WorP.WIDTH ? "WIDTH" : "PREC";
            this.tmpError = `%!(BAD ${tmp} '${this.args[this.argNum]}')`;
        }
        this.argNum++;
    }
    handleWidthAndPrecision(flags) {
        const fmt = this.format;
        for(; this.i !== this.format.length; ++this.i){
            const c = fmt[this.i];
            switch(this.state){
                case State1.WIDTH:
                    switch(c){
                        case ".":
                            this.flags.precision = 0;
                            this.state = State1.PRECISION;
                            break;
                        case "*":
                            this.handleWidthOrPrecisionRef(WorP.WIDTH);
                            break;
                        default:
                            {
                                const val = parseInt(c);
                                if (isNaN(val)) {
                                    this.i--;
                                    this.state = State1.PERCENT;
                                    return;
                                }
                                flags.width = flags.width === -1 ? 0 : flags.width;
                                flags.width *= 10;
                                flags.width += val;
                            }
                    }
                    break;
                case State1.PRECISION:
                    {
                        if (c === "*") {
                            this.handleWidthOrPrecisionRef(WorP.PRECISION);
                            break;
                        }
                        const val = parseInt(c);
                        if (isNaN(val)) {
                            this.i--;
                            this.state = State1.PERCENT;
                            return;
                        }
                        flags.precision *= 10;
                        flags.precision += val;
                        break;
                    }
                default:
                    throw new Error("can't be here. bug.");
            }
        }
    }
    handlePositional() {
        if (this.format[this.i] !== "[") {
            throw new Error("Can't happen? Bug.");
        }
        let positional = 0;
        const format = this.format;
        this.i++;
        let err = false;
        for(; this.i !== this.format.length; ++this.i){
            if (format[this.i] === "]") {
                break;
            }
            positional *= 10;
            const val = parseInt(format[this.i]);
            if (isNaN(val)) {
                this.tmpError = "%!(BAD INDEX)";
                err = true;
            }
            positional += val;
        }
        if (positional - 1 >= this.args.length) {
            this.tmpError = "%!(BAD INDEX)";
            err = true;
        }
        this.argNum = err ? this.argNum : positional - 1;
    }
    handleLessThan() {
        const arg = this.args[this.argNum];
        if ((arg || {}).constructor.name !== "Array") {
            throw new Error(`arg ${arg} is not an array. Todo better error handling`);
        }
        let str = "[ ";
        for(let i1 = 0; i1 !== arg.length; ++i1){
            if (i1 !== 0) str += ", ";
            str += this._handleVerb(arg[i1]);
        }
        return str + " ]";
    }
    handleVerb() {
        const verb = this.format[this.i];
        this.verb = verb;
        if (this.tmpError) {
            this.buf += this.tmpError;
            this.tmpError = undefined;
            if (this.argNum < this.haveSeen.length) {
                this.haveSeen[this.argNum] = true;
            }
        } else if (this.args.length <= this.argNum) {
            this.buf += `%!(MISSING '${verb}')`;
        } else {
            const arg = this.args[this.argNum];
            this.haveSeen[this.argNum] = true;
            if (this.flags.lessthan) {
                this.buf += this.handleLessThan();
            } else {
                this.buf += this._handleVerb(arg);
            }
        }
        this.argNum++;
        this.state = State1.PASSTHROUGH;
    }
    _handleVerb(arg) {
        switch(this.verb){
            case "t":
                return this.pad(arg.toString());
            case "b":
                return this.fmtNumber(arg, 2);
            case "c":
                return this.fmtNumberCodePoint(arg);
            case "d":
                return this.fmtNumber(arg, 10);
            case "o":
                return this.fmtNumber(arg, 8);
            case "x":
                return this.fmtHex(arg);
            case "X":
                return this.fmtHex(arg, true);
            case "e":
                return this.fmtFloatE(arg);
            case "E":
                return this.fmtFloatE(arg, true);
            case "f":
            case "F":
                return this.fmtFloatF(arg);
            case "g":
                return this.fmtFloatG(arg);
            case "G":
                return this.fmtFloatG(arg, true);
            case "s":
                return this.fmtString(arg);
            case "T":
                return this.fmtString(typeof arg);
            case "v":
                return this.fmtV(arg);
            case "j":
                return this.fmtJ(arg);
            case "i":
                return this.fmtI(arg, false);
            case "I":
                return this.fmtI(arg, true);
            default:
                return `%!(BAD VERB '${this.verb}')`;
        }
    }
    pad(s) {
        const padding = this.flags.zero ? "0" : " ";
        if (this.flags.dash) {
            return s.padEnd(this.flags.width, padding);
        }
        return s.padStart(this.flags.width, padding);
    }
    padNum(nStr, neg) {
        let sign;
        if (neg) {
            sign = "-";
        } else if (this.flags.plus || this.flags.space) {
            sign = this.flags.plus ? "+" : " ";
        } else {
            sign = "";
        }
        const zero = this.flags.zero;
        if (!zero) {
            nStr = sign + nStr;
        }
        const pad = zero ? "0" : " ";
        const len = zero ? this.flags.width - sign.length : this.flags.width;
        if (this.flags.dash) {
            nStr = nStr.padEnd(len, pad);
        } else {
            nStr = nStr.padStart(len, pad);
        }
        if (zero) {
            nStr = sign + nStr;
        }
        return nStr;
    }
    fmtNumber(n, radix, upcase = false) {
        let num = Math.abs(n).toString(radix);
        const prec = this.flags.precision;
        if (prec !== -1) {
            this.flags.zero = false;
            num = n === 0 && prec === 0 ? "" : num;
            while(num.length < prec){
                num = "0" + num;
            }
        }
        let prefix = "";
        if (this.flags.sharp) {
            switch(radix){
                case 2:
                    prefix += "0b";
                    break;
                case 8:
                    prefix += num.startsWith("0") ? "" : "0";
                    break;
                case 16:
                    prefix += "0x";
                    break;
                default:
                    throw new Error("cannot handle base: " + radix);
            }
        }
        num = num.length === 0 ? num : prefix + num;
        if (upcase) {
            num = num.toUpperCase();
        }
        return this.padNum(num, n < 0);
    }
    fmtNumberCodePoint(n) {
        let s = "";
        try {
            s = String.fromCodePoint(n);
        } catch  {
            s = UNICODE_REPLACEMENT_CHARACTER;
        }
        return this.pad(s);
    }
    fmtFloatSpecial(n) {
        if (isNaN(n)) {
            this.flags.zero = false;
            return this.padNum("NaN", false);
        }
        if (n === Number.POSITIVE_INFINITY) {
            this.flags.zero = false;
            this.flags.plus = true;
            return this.padNum("Inf", false);
        }
        if (n === Number.NEGATIVE_INFINITY) {
            this.flags.zero = false;
            return this.padNum("Inf", true);
        }
        return "";
    }
    roundFractionToPrecision(fractional, precision) {
        let round = false;
        if (fractional.length > precision) {
            fractional = "1" + fractional;
            let tmp = parseInt(fractional.slice(0, precision + 2)) / 10;
            tmp = Math.round(tmp);
            fractional = Math.floor(tmp).toString();
            round = fractional[0] === "2";
            fractional = fractional.slice(1);
        } else {
            while(fractional.length < precision){
                fractional += "0";
            }
        }
        return [
            fractional,
            round
        ];
    }
    fmtFloatE(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen, bug");
        }
        let fractional = m[F.fractional];
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        let rounding = false;
        [fractional, rounding] = this.roundFractionToPrecision(fractional, precision);
        let e = m[F.exponent];
        let esign = m[F.esign];
        let mantissa = parseInt(m[F.mantissa]);
        if (rounding) {
            mantissa += 1;
            if (10 <= mantissa) {
                mantissa = 1;
                const r = parseInt(esign + e) + 1;
                e = r.toString();
                esign = r < 0 ? "-" : "+";
            }
        }
        e = e.length === 1 ? "0" + e : e;
        const val = `${mantissa}.${fractional}${upcase ? "E" : "e"}${esign}${e}`;
        return this.padNum(val, n < 0);
    }
    fmtFloatF(n) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        function expandNumber(n) {
            if (Number.isSafeInteger(n)) {
                return n.toString() + ".";
            }
            const t = n.toExponential().split("e");
            let m = t[0].replace(".", "");
            const e = parseInt(t[1]);
            if (e < 0) {
                let nStr = "0.";
                for(let i1 = 0; i1 !== Math.abs(e) - 1; ++i1){
                    nStr += "0";
                }
                return nStr += m;
            } else {
                const splIdx = e + 1;
                while(m.length < splIdx){
                    m += "0";
                }
                return m.slice(0, splIdx) + "." + m.slice(splIdx);
            }
        }
        const val = expandNumber(Math.abs(n));
        const arr = val.split(".");
        let dig = arr[0];
        let fractional = arr[1];
        const precision = this.flags.precision !== -1 ? this.flags.precision : 6;
        let round = false;
        [fractional, round] = this.roundFractionToPrecision(fractional, precision);
        if (round) {
            dig = (parseInt(dig) + 1).toString();
        }
        return this.padNum(`${dig}.${fractional}`, n < 0);
    }
    fmtFloatG(n, upcase = false) {
        const special = this.fmtFloatSpecial(n);
        if (special !== "") {
            return special;
        }
        let P = this.flags.precision !== -1 ? this.flags.precision : 6;
        P = P === 0 ? 1 : P;
        const m = n.toExponential().match(FLOAT_REGEXP);
        if (!m) {
            throw Error("can't happen");
        }
        const X = parseInt(m[F.exponent]) * (m[F.esign] === "-" ? -1 : 1);
        let nStr = "";
        if (P > X && X >= -4) {
            this.flags.precision = P - (X + 1);
            nStr = this.fmtFloatF(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*$/, "");
            }
        } else {
            this.flags.precision = P - 1;
            nStr = this.fmtFloatE(n);
            if (!this.flags.sharp) {
                nStr = nStr.replace(/\.?0*e/, upcase ? "E" : "e");
            }
        }
        return nStr;
    }
    fmtString(s) {
        if (this.flags.precision !== -1) {
            s = s.slice(0, this.flags.precision);
        }
        return this.pad(s);
    }
    fmtHex(val, upper = false) {
        switch(typeof val){
            case "number":
                return this.fmtNumber(val, 16, upper);
            case "string":
                {
                    const sharp = this.flags.sharp && val.length !== 0;
                    let hex = sharp ? "0x" : "";
                    const prec = this.flags.precision;
                    const end = prec !== -1 ? min(prec, val.length) : val.length;
                    for(let i1 = 0; i1 !== end; ++i1){
                        if (i1 !== 0 && this.flags.space) {
                            hex += sharp ? " 0x" : " ";
                        }
                        const c = (val.charCodeAt(i1) & 0xff).toString(16);
                        hex += c.length === 1 ? `0${c}` : c;
                    }
                    if (upper) {
                        hex = hex.toUpperCase();
                    }
                    return this.pad(hex);
                }
            default:
                throw new Error("currently only number and string are implemented for hex");
        }
    }
    fmtV(val) {
        if (this.flags.sharp) {
            const options = this.flags.precision !== -1 ? {
                depth: this.flags.precision
            } : {};
            return this.pad(Deno.inspect(val, options));
        } else {
            const p = this.flags.precision;
            return p === -1 ? val.toString() : val.toString().slice(0, p);
        }
    }
    fmtJ(val) {
        return JSON.stringify(val);
    }
    fmtI(val, compact) {
        return Deno.inspect(val, {
            colors: !Deno?.noColor,
            compact,
            depth: Infinity,
            iterableLimit: Infinity
        });
    }
}
function glb(array, value, compare) {
    compare = compare || ((a, b)=>a - b);
    if (array.length === 0) {
        return -1;
    }
    if (array.length === 1) {
        if (compare(value, array[0]) < 0) {
            return -1;
        } else {
            return 0;
        }
    }
    let left = 0;
    let right = array.length - 1;
    const vLeft = array[left], vRight = array[right];
    if (compare(value, vRight) >= 0) {
        return right;
    }
    if (compare(value, vLeft) < 0) {
        return -1;
    }
    while(right - left > 1){
        const center = left + (right - left >> 1);
        const vCenter = array[center];
        const cmp = compare(value, vCenter);
        if (cmp < 0) {
            right = center;
        } else if (cmp === 0) {
            left = center;
        } else {
            left = center;
        }
    }
    return left;
}
class InternalError extends Error {
    constructor(message, printName = true, printStack = true){
        super(message);
        this.name = "Internal Error";
        this.printName = printName;
        this.printStack = printStack;
    }
    printName;
    printStack;
}
class UnreachableError extends InternalError {
    constructor(){
        super("Unreachable code was reached.", true, true);
    }
}
class ErrorEx extends Error {
    constructor(name, message, printName = true, printStack = true){
        super(message);
        this.name = name;
        this.printName = printName;
        this.printStack = printStack;
    }
    printName;
    printStack;
}
function asErrorEx(e) {
    if (e instanceof ErrorEx) {
        return e;
    } else if (e instanceof Error) {
        e.printName = e.name !== "Error";
        e.printStack = !!e.message;
        return e;
    } else {
        return new ErrorEx("Error", String(e), false, true);
    }
}
let Deno3;
try {
    Deno3 = globalThis.Deno;
} catch (_e) {}
const noColor2 = typeof (Deno3 && Deno3.noColor) === "boolean" ? Deno3.noColor : true;
let enabled2 = !noColor2;
function code2(open, close) {
    return {
        open: `\x1b[${open.join(";")}m`,
        close: `\x1b[${close}m`,
        regexp: new RegExp(`\\x1b\\[${close}m`, "g")
    };
}
function run2(str, code) {
    return enabled2 ? `${code.open}${str.replace(code.regexp, code.open)}${code.close}` : str;
}
function red2(str) {
    return run2(str, code2([
        31
    ], 39));
}
function blue2(str) {
    return run2(str, code2([
        34
    ], 39));
}
const ANSI_PATTERN2 = new RegExp([
    "[\\u001B\\u009B][[\\]()#;?]*(?:(?:(?:[a-zA-Z\\d]*(?:;[-a-zA-Z\\d\\/#&.:=?%@~_]*)*)?\\u0007)",
    "(?:(?:\\d{1,4}(?:;\\d{0,4})*)?[\\dA-PR-TZcf-ntqry=><~]))"
].join("|"), "g");
function stripColor2(string) {
    return string.replace(ANSI_PATTERN2, "");
}
function platformHasNonAsciiCharacters() {
    try {
        return Deno.build.os !== "windows";
    } catch (_e) {
        return false;
    }
}
function tidyverseInfo(msg) {
    if (platformHasNonAsciiCharacters()) {
        return `${blue2("ℹ")} ${msg}`;
    } else {
        return `${blue2("i")} ${msg}`;
    }
}
function tidyverseError(msg) {
    if (platformHasNonAsciiCharacters()) {
        return `${red2("✖")} ${msg}`;
    } else {
        return `${red2("x")} ${msg}`;
    }
}
function tidyverseFormatError(msg) {
    let { heading, error, info } = msg;
    if (msg.location) {
        heading = `${locationString(msg.location)} ${heading}`;
    }
    if (msg.fileName) {
        heading = `In file ${msg.fileName}\n${heading}`;
    }
    const strings = [
        heading,
        msg.sourceContext,
        ...error.map(tidyverseError),
        ...Object.values(info).map(tidyverseInfo),
        ""
    ];
    return strings.join("\n");
}
function quotedStringColor(msg) {
    return blue2(msg);
}
function addFileInfo(msg, src) {
    if (src.fileName !== undefined) {
        msg.fileName = src.fileName;
    }
}
function addInstancePathInfo(msg, instancePath) {
    if (instancePath.length) {
        const niceInstancePath = instancePath.map((s)=>blue2(String(s))).join(":");
        msg.info["instance-path-location"] = `The error happened in location ${niceInstancePath}.`;
    }
}
function locationString(loc) {
    const { start, end } = loc;
    if (start.line === end.line) {
        if (start.column === end.column) {
            return `(line ${start.line + 1}, column ${start.column + 1})`;
        } else {
            return `(line ${start.line + 1}, columns ${start.column + 1}--${end.column + 1})`;
        }
    } else {
        return `(line ${start.line + 1}, column ${start.column + 1} through line ${end.line + 1}, column ${end.column + 1})`;
    }
}
function errorKey(err) {
    const positionKey = (pos)=>`${pos.line}-${pos.column}`;
    return `${err.fileName || ""}-${positionKey(err.location.start)}-${positionKey(err.location.end)}`;
}
function reportOnce(reporter, reportSet) {
    const errorsReported = reportSet || new Set();
    return (err)=>{
        const key = errorKey(err);
        if (errorsReported.has(key)) {
            return;
        }
        errorsReported.add(key);
        reporter(err);
    };
}
function lines(text) {
    return text.split(/\r?\n/);
}
function normalizeNewlines(text) {
    return lines(text).join("\n");
}
function trimEmptyLines(lines, trim = "all") {
    if (trim === "all" || trim === "leading") {
        const firstNonEmpty = lines.findIndex((line)=>line.trim().length > 0);
        if (firstNonEmpty === -1) {
            return [];
        }
        lines = lines.slice(firstNonEmpty);
    }
    if (trim === "all" || trim === "trailing") {
        let lastNonEmpty = -1;
        for(let i1 = lines.length - 1; i1 >= 0; i1--){
            if (lines[i1].trim().length > 0) {
                lastNonEmpty = i1;
                break;
            }
        }
        if (lastNonEmpty > -1) {
            lines = lines.slice(0, lastNonEmpty + 1);
        }
    }
    return lines;
}
function* matchAll(text, regexp) {
    if (!regexp.global) {
        throw new Error("matchAll requires global regexps");
    }
    let match;
    while((match = regexp.exec(text)) !== null){
        yield match;
    }
}
function* lineOffsets(text) {
    yield 0;
    for (const match of matchAll(text, /\r?\n/g)){
        yield match.index + match[0].length;
    }
}
function indexToLineCol(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(offset) {
        if (offset === 0) {
            return {
                line: 0,
                column: 0
            };
        }
        const startIndex = glb(offsets, offset);
        return {
            line: startIndex,
            column: offset - offsets[startIndex]
        };
    };
}
function lineColToIndex(text) {
    const offsets = Array.from(lineOffsets(text));
    return function(position) {
        return offsets[position.line] + position.column;
    };
}
function formatLineRange(text, firstLine, lastLine) {
    const lineWidth = Math.max(String(firstLine + 1).length, String(lastLine + 1).length);
    const pad = " ".repeat(lineWidth);
    const ls = lines(text);
    const result1 = [];
    for(let i1 = firstLine; i1 <= lastLine; ++i1){
        const numberStr = `${pad}${i1 + 1}: `.slice(-(lineWidth + 2));
        const lineStr = ls[i1];
        result1.push({
            lineNumber: i1,
            content: numberStr + quotedStringColor(lineStr),
            rawLine: ls[i1]
        });
    }
    return {
        prefixWidth: lineWidth + 2,
        lines: result1
    };
}
function editDistance(w1, w2) {
    const cost = (c)=>{
        if ("_-".indexOf(c) !== -1) {
            return 1;
        }
        return 10;
    };
    const cost2 = (c1, c2)=>{
        if (c1 === c2) {
            return 0;
        }
        if ("_-".indexOf(c1) !== -1 && "_-".indexOf(c2) !== -1) {
            return 1;
        }
        if (c1.toLocaleLowerCase() === c2.toLocaleLowerCase()) {
            return 1;
        }
        const cc1 = c1.charCodeAt(0);
        const cc2 = c2.charCodeAt(0);
        if (cc1 >= 48 && cc1 <= 57 && cc2 >= 48 && cc2 <= 57) {
            return 1;
        }
        return 10;
    };
    const s1 = w1.length + 1;
    const s2 = w2.length + 1;
    const v = new Int32Array(s1 * s2);
    for(let i1 = 0; i1 < s1; ++i1){
        for(let j = 0; j < s2; ++j){
            if (i1 === 0 && j === 0) {
                continue;
            } else if (i1 === 0) {
                v[i1 * s2 + j] = v[i1 * s2 + (j - 1)] + cost(w2[j - 1]);
            } else if (j === 0) {
                v[i1 * s2 + j] = v[(i1 - 1) * s2 + j] + cost(w1[i1 - 1]);
            } else {
                v[i1 * s2 + j] = Math.min(v[(i1 - 1) * s2 + (j - 1)] + cost2(w1[i1 - 1], w2[j - 1]), v[i1 * s2 + (j - 1)] + cost(w2[j - 1]), v[(i1 - 1) * s2 + j] + cost(w1[i1 - 1]));
            }
        }
    }
    return v[(w1.length + 1) * (w2.length + 1) - 1];
}
function detectCaseConvention(key) {
    if (key.toLocaleLowerCase() !== key) {
        return "capitalizationCase";
    }
    const underscoreIndex = key.indexOf("_");
    if (underscoreIndex !== -1 && underscoreIndex !== 0 && underscoreIndex !== key.length - 1) {
        return "underscore_case";
    }
    const dashIndex = key.indexOf("-");
    if (dashIndex !== -1 && dashIndex !== 0 && dashIndex !== key.length - 1) {
        return "dash-case";
    }
    return undefined;
}
function resolveCaseConventionRegex(keys, conventions) {
    if (conventions !== undefined) {
        if (conventions.length === 0) {
            throw new InternalError("resolveCaseConventionRegex requires nonempty `conventions`");
        }
        return {
            pattern: conventions.map((c)=>`(${c})`).join("|"),
            list: conventions
        };
    }
    const disallowedNearMisses = [];
    const keySet = new Set(keys);
    const addNearMiss = (value)=>{
        if (!keySet.has(value)) {
            disallowedNearMisses.push(value);
        }
    };
    const foundConventions = new Set();
    for (const key of keys){
        const found = detectCaseConvention(key);
        if (found) {
            foundConventions.add(found);
        }
        switch(found){
            case "capitalizationCase":
                addNearMiss(toUnderscoreCase(key));
                addNearMiss(toDashCase(key));
                break;
            case "dash-case":
                addNearMiss(toUnderscoreCase(key));
                addNearMiss(toCapitalizationCase(key));
                break;
            case "underscore_case":
                addNearMiss(toDashCase(key));
                addNearMiss(toCapitalizationCase(key));
                break;
        }
    }
    if (foundConventions.size === 0) {
        return {
            pattern: undefined,
            list: []
        };
    }
    return {
        pattern: `(?!(${disallowedNearMisses.map((c)=>`^${c}$`).join("|")}))`,
        list: Array.from(foundConventions)
    };
}
function toDashCase(str) {
    return toUnderscoreCase(str).replace(/_/g, "-");
}
function toUnderscoreCase(str) {
    return str.replace(/([A-Z]+)/g, (_match, p1)=>`-${p1}`).replace(/-/g, "_").split("_").filter((x)=>x.length).join("_").toLocaleLowerCase();
}
function toCapitalizationCase(str) {
    return toUnderscoreCase(str).replace(/_(.)/g, (_match, p1)=>p1.toLocaleUpperCase());
}
function capitalizeWord(str) {
    return str.slice(0, 1).toUpperCase() + str.slice(1);
}
function capitalizeTitle(str) {
    return str.split(/\s+/).map((str, index, arr)=>{
        if (index === 0 || index === arr.length - 1 || !isNotCapitalized(str)) {
            return capitalizeWord(str);
        } else {
            return str;
        }
    }).join(" ");
}
function isNotCapitalized(str) {
    return [
        "a",
        "an",
        "the",
        "for",
        "and",
        "nor",
        "but",
        "or",
        "yet",
        "so",
        "with",
        "at",
        "by",
        "to",
        "in",
        "for",
        "from",
        "of",
        "on"
    ].includes(str);
}
const kLastPunctuationRegex = /([\S\s]*)[\.\?\!]/;
function trimSentence(text) {
    const match = text.match(kLastPunctuationRegex);
    if (match) {
        return {
            text: match[0],
            trimmed: true
        };
    } else {
        return {
            text,
            trimmed: false
        };
    }
}
function trimLength(text, length) {
    if (text.length < length) {
        return {
            text,
            trimmed: false
        };
    } else {
        return {
            text: text.substring(0, length),
            trimmed: true
        };
    }
}
function trimSpace(text) {
    const lastSpace = text.lastIndexOf(" ");
    if (lastSpace > 0) {
        return {
            text: text.substring(0, lastSpace),
            trimmed: true
        };
    } else {
        return {
            text,
            trimmed: false
        };
    }
}
function truncateText(text, length, breakAt) {
    const trimEnd = (text)=>{
        if ([
            ",",
            "/",
            ":"
        ].includes(text.charAt(text.length - 1))) {
            return text.substring(0, text.length - 1);
        } else {
            return text;
        }
    };
    const trimAtSpace = (text)=>{
        const spaceResult = trimSpace(text.substring(0, text.length - 1));
        return trimEnd(spaceResult.text) + "…";
    };
    const trimPunc = (text)=>{
        const puncResult = trimSentence(text);
        if (puncResult.trimmed) {
            return puncResult.text;
        } else {
            return trimAtSpace(puncResult.text);
        }
    };
    const lengthResult = trimLength(text, length);
    if (lengthResult.trimmed) {
        if (breakAt === "punctuation") {
            return trimPunc(lengthResult.text);
        } else {
            return trimAtSpace(lengthResult.text);
        }
    } else {
        return lengthResult.text;
    }
}
const booleanSchema = {
    "type": "boolean",
    "description": "be `true` or `false`",
    "completions": [
        "true",
        "false"
    ],
    "exhaustiveCompletions": true
};
const numberSchema = {
    "type": "number",
    "description": "be a number"
};
const stringSchema = {
    "type": "string",
    "description": "be a string"
};
const nullSchema = {
    "type": "null",
    "description": "be the null value",
    "completions": [
        "null"
    ],
    "exhaustiveCompletions": true
};
function schemaType(schema) {
    if (schema === false) {
        return "false";
    }
    if (schema === true) {
        return "true";
    }
    return schema.type;
}
function schemaDispatch(s, d) {
    const st = schemaType(s);
    if (d[st]) {
        d[st](s);
    }
}
function schemaCall(s, d, other) {
    const st = schemaType(s);
    if (d[st]) {
        return d[st](s);
    }
    if (other) {
        return other(s);
    }
    throw new Error(`Internal Error: Dispatch failed for type ${st}`);
}
function schemaDocString(d) {
    if (typeof d === "string") {
        return d;
    }
    if (d.short) {
        return d.short;
    }
    return "";
}
function schemaDescription(schema) {
    if (schema === true) {
        return `be anything`;
    } else if (schema === false) {
        return `be no possible value`;
    } else {
        return schema.description || `be ${schemaType(schema)}`;
    }
}
let globalInternalIdCounter = 0;
function internalId() {
    return {
        _internalId: ++globalInternalIdCounter
    };
}
function tagSchema(schema, tags) {
    return {
        ...schema,
        tags: {
            ...schema.tags || {},
            ...tags
        }
    };
}
function anySchema(description) {
    return {
        ...internalId(),
        description,
        "type": "any"
    };
}
function enumSchema(...args) {
    if (args.length === 0) {
        throw new InternalError("Empty enum schema not supported.");
    }
    return {
        ...internalId(),
        "type": "enum",
        "enum": args,
        "description": args.length > 1 ? `be one of: ${args.map((x)=>"`" + x + "`").join(", ")}` : `be '${args[0]}'`,
        "completions": args.map(String),
        "exhaustiveCompletions": true
    };
}
function regexSchema(arg, description) {
    const result1 = {
        ...internalId(),
        "type": "string",
        "pattern": arg
    };
    if (description) {
        result1.description = description;
    } else {
        result1.description = `be a string that satisfies regex "${arg}"`;
    }
    return result1;
}
function anyOfSchema(...args) {
    return {
        ...internalId(),
        "type": "anyOf",
        "anyOf": args,
        "description": `be at least one of: ${args.map((x)=>schemaDescription(x).slice(3)).join(", ")}`
    };
}
function allOfSchema(...args) {
    return {
        ...internalId(),
        "type": "allOf",
        "allOf": args,
        "description": `be all of: ${args.map((x)=>schemaDescription(x).slice(3)).join(", ")}`
    };
}
function objectSchema(params = {}) {
    let { properties, patternProperties, required, additionalProperties, description, baseSchema, exhaustive, completions: completionsParam, namingConvention, propertyNames: propertyNamesSchema, closed } = params;
    required = required || [];
    properties = properties || {};
    patternProperties = patternProperties || {};
    let tags = {};
    let tagsAreSet = false;
    let propertyNames = propertyNamesSchema;
    if (completionsParam) {
        tags["completions"] = completionsParam;
        tagsAreSet = true;
    }
    const createCaseConventionSchema = (props)=>{
        if (namingConvention === "ignore") {
            return undefined;
        }
        const objectKeys = Object.getOwnPropertyNames(props);
        const { pattern, list } = resolveCaseConventionRegex(objectKeys, namingConvention);
        if (pattern === undefined) {
            return undefined;
        }
        if (propertyNames !== undefined) {
            console.error("Warning: propertyNames and case convention detection are mutually exclusive.");
            console.error("Add `namingConvention: 'ignore'` to your schema definition to remove this warning.");
            return undefined;
        }
        const tags = {
            "case-convention": list,
            "error-importance": -5,
            "case-detection": true
        };
        return {
            errorMessage: "property ${value} does not match case convention " + `${objectKeys.join(",")}`,
            "type": "string",
            pattern,
            tags
        };
    };
    const hasDescription = description !== undefined;
    description = description || "be an object";
    let result1 = undefined;
    if (baseSchema) {
        if (!Array.isArray(baseSchema)) {
            baseSchema = [
                baseSchema
            ];
        }
        if (baseSchema.some((s)=>s.type !== "object")) {
            throw new InternalError("Attempted to extend a non-object schema");
        }
        if (baseSchema.length <= 0) {
            throw new InternalError("base schema cannot be empty list");
        }
        let temp = {
            ...internalId()
        };
        for (const base of baseSchema){
            temp = Object.assign(temp, base);
        }
        result1 = temp;
        if (result1 === undefined) {
            throw new UnreachableError();
        }
        if (result1.$id) {
            delete result1.$id;
        }
        for (const base of baseSchema){
            if (base.exhaustiveCompletions) {
                result1.exhaustiveCompletions = true;
            }
        }
        if (hasDescription) {
            result1.description = description;
        }
        const m = new Map();
        for (const base of baseSchema){
            for (const [k, v] of Object.entries(base.properties || {})){
                if (!m.has(k)) {
                    m.set(k, []);
                }
                m.get(k).push([
                    v,
                    base.$id
                ]);
            }
        }
        const errorMsgs = new Set();
        for (const [k, l1] of m){
            if (l1.length > 1) {
                errorMsgs.add(`Internal Error: base schemas ${l1.map((x)=>x[1]).join(", ")} share property ${k}.`);
            }
        }
        if (errorMsgs.size > 0) {
            console.error([
                ...errorMsgs
            ].toSorted((a, b)=>a.localeCompare(b)).join("\n"));
            console.error("This is a bug in quarto's schemas.");
            console.error("Note that we don't throw in order to allow build-js to finish, but the generated schemas will be invalid.");
        }
        result1.properties = Object.assign({}, ...baseSchema.map((s)=>s.properties), properties);
        result1.patternProperties = Object.assign({}, ...baseSchema.map((s)=>s.patternProperties).filter((s)=>s !== undefined), patternProperties);
        result1.required = [
            ...baseSchema.map((s)=>s.required || []),
            required || []
        ].flat();
        if (result1.required && result1.required.length === 0) {
            result1.required = undefined;
        }
        const additionalPropArray = baseSchema.map((s)=>s.additionalProperties).filter((s)=>s !== undefined);
        if (additionalProperties) {
            additionalPropArray.push(additionalProperties);
        }
        if (additionalPropArray.length) {
            result1.additionalProperties = allOfSchema(...additionalPropArray);
        }
        const propNamesArray = baseSchema.map((s)=>s.propertyNames).filter((s)=>{
            if (typeof s !== "object") return true;
            if (s.tags === undefined) return true;
            if (s.tags["case-detection"] === true) {
                return false;
            }
            return true;
        }).filter((s)=>s !== undefined);
        if (propNamesArray.length === 1) {
            result1.propertyNames = propNamesArray[0];
        } else if (propNamesArray.length > 1) {
            result1.propertyNames = anyOfSchema(...propNamesArray);
        } else {
            delete result1.propertyNames;
        }
        result1.closed = closed || baseSchema.some((s)=>s.closed);
    } else {
        const caseConventionSchema = createCaseConventionSchema(properties);
        if (caseConventionSchema !== undefined) {
            propertyNames = caseConventionSchema;
            tags = {
                ...tags,
                ...caseConventionSchema.tags
            };
            tagsAreSet = true;
        }
        result1 = {
            ...internalId(),
            "type": "object",
            description
        };
        if (exhaustive) {
            result1.exhaustiveCompletions = true;
        }
        if (properties) {
            result1.properties = properties;
        }
        if (patternProperties) {
            result1.patternProperties = patternProperties;
        }
        if (required && required.length > 0) {
            result1.required = required;
        }
        result1.closed = closed;
        if (additionalProperties !== undefined) {
            result1.additionalProperties = additionalProperties;
        }
        if (propertyNames !== undefined) {
            result1.propertyNames = propertyNames;
        }
    }
    if (tagsAreSet) {
        result1.tags = tags;
    }
    return result1;
}
function arraySchema(items) {
    if (items) {
        return {
            ...internalId(),
            "type": "array",
            "description": `be an array of values, where each element must ${schemaDescription(items)}`,
            items
        };
    } else {
        return {
            ...internalId(),
            "type": "array",
            "description": `be an array of values`
        };
    }
}
function documentSchema(schema, doc) {
    const result1 = Object.assign({}, schema);
    result1.documentation = doc;
    return result1;
}
function describeSchema(schema, description) {
    const result1 = Object.assign({}, schema);
    result1.description = `be ${description}`;
    return result1;
}
function completeSchema(schema, ...completions) {
    const result1 = Object.assign({}, schema);
    const prevCompletions = (schema.completions || []).slice();
    prevCompletions.push(...completions);
    result1.completions = prevCompletions;
    return result1;
}
function completeSchemaOverwrite(schema, ...completions) {
    const result1 = Object.assign({}, schema);
    result1.completions = completions;
    return result1;
}
function idSchema(schema, id) {
    const result1 = Object.assign({}, schema);
    result1["$id"] = id;
    return result1;
}
function errorMessageSchema(schema, errorMessage) {
    return {
        ...schema,
        errorMessage
    };
}
function refSchema($ref, description) {
    return {
        ...internalId(),
        "type": "ref",
        $ref,
        description
    };
}
function valueSchema(val, description) {
    return {
        ...internalId(),
        "type": "enum",
        "enum": [
            val
        ],
        "description": description || `be ${JSON.stringify(val)}`
    };
}
const definitionsObject = {};
function hasSchemaDefinition(key) {
    return definitionsObject[key] !== undefined;
}
function getSchemaDefinition(key) {
    if (definitionsObject[key] === undefined) {
        throw new InternalError(`Schema ${key} not found.`);
    }
    return definitionsObject[key];
}
function setSchemaDefinition(schema) {
    if (schema.$id === undefined) {
        throw new InternalError("setSchemaDefinition needs $id");
    }
    if (definitionsObject[schema.$id] === undefined) {
        definitionsObject[schema.$id] = schema;
    }
}
function getSchemaDefinitionsObject() {
    return Object.assign({}, definitionsObject);
}
function expandAliasesFrom(lst, defs) {
    const aliases = defs;
    const result1 = [];
    lst = lst.slice();
    for(let i1 = 0; i1 < lst.length; ++i1){
        const el = lst[i1];
        if (el.startsWith("$")) {
            const v = aliases[el.slice(1)];
            if (v === undefined) {
                throw new InternalError(`${el} doesn't have an entry in the aliases map`);
            }
            lst.push(...v);
        } else {
            result1.push(el);
        }
    }
    return result1;
}
function resolveSchema(schema, visit, hasRef, next) {
    if (schema === false || schema === true) {
        return schema;
    }
    if (hasRef === undefined) {
        hasRef = (cursor)=>{
            return schemaCall(cursor, {
                ref: (_s)=>true
            }, (_s)=>false);
        };
    }
    if (!hasRef(schema)) {
        return schema;
    }
    if (visit === undefined) {
        visit = (_schema)=>{};
    }
    if (next === undefined) {
        next = (cursor)=>{
            const result1 = schemaCall(cursor, {
                ref: (s)=>getSchemaDefinition(s.$ref)
            });
            if (result1 === undefined) {
                throw new InternalError("couldn't resolve schema ${JSON.stringify(cursor)}");
            }
            return result1;
        };
    }
    let cursor1 = schema;
    let cursor2 = schema;
    let stopped = false;
    do {
        cursor1 = next(cursor1);
        visit(cursor1);
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (hasRef(cursor2)) {
            cursor2 = next(cursor2);
        } else {
            stopped = true;
        }
        if (!stopped && cursor1 === cursor2) {
            throw new Error(`reference cycle detected at ${JSON.stringify(cursor1)}`);
        }
    }while (hasRef(cursor1))
    return cursor1;
}
function matchAll1(str, regex) {
    let match;
    regex = new RegExp(regex);
    const result1 = [];
    while((match = regex.exec(str)) != null){
        result1.push(match);
    }
    return result1;
}
function rangedLines(text, includeNewLines = false) {
    const regex = /\r?\n/g;
    const result1 = [];
    let startOffset = 0;
    if (!includeNewLines) {
        for (const r of matchAll1(text, regex)){
            result1.push({
                substring: text.substring(startOffset, r.index),
                range: {
                    start: startOffset,
                    end: r.index
                }
            });
            startOffset = r.index + r[0].length;
        }
        result1.push({
            substring: text.substring(startOffset, text.length),
            range: {
                start: startOffset,
                end: text.length
            }
        });
        return result1;
    } else {
        const matches = matchAll1(text, regex);
        let prevOffset = 0;
        for (const r of matches){
            const stringEnd = r.index + r[0].length;
            result1.push({
                substring: text.substring(prevOffset, stringEnd),
                range: {
                    start: prevOffset,
                    end: stringEnd
                }
            });
            prevOffset = stringEnd;
        }
        result1.push({
            substring: text.substring(prevOffset, text.length),
            range: {
                start: prevOffset,
                end: text.length
            }
        });
        return result1;
    }
}
function mappedSubstring(source, start, end) {
    if (typeof source === "string") {
        source = asMappedString(source);
    }
    const value = source.value.substring(start, end);
    const mappedSource = source;
    return {
        value,
        fileName: mappedSource.fileName,
        map: (index, closest)=>{
            if (closest) {
                index = Math.max(0, Math.min(value.length, index - 1));
            }
            if (index === 0 && index === value.length) {
                return mappedSource.map(index + start, closest);
            }
            if (index < 0 || index >= value.length) {
                return undefined;
            }
            return mappedSource.map(index + start, closest);
        }
    };
}
function mappedString(source, pieces, fileName) {
    if (typeof source === "string") {
        source = asMappedString(source, fileName);
    }
    const mappedPieces = pieces.map((piece)=>{
        if (typeof piece === "string") {
            return asMappedString(piece);
        } else if (piece.value !== undefined) {
            return piece;
        } else {
            const { start, end } = piece;
            return mappedSubstring(source, start, end);
        }
    });
    return mappedConcat(mappedPieces);
}
function asMappedString(str, fileName) {
    if (typeof str === "string") {
        return {
            value: str,
            fileName,
            map: function(index, closest) {
                if (closest) {
                    index = Math.min(str.length - 1, Math.max(0, index));
                }
                if (index < 0 || index >= str.length) {
                    return undefined;
                }
                return {
                    index,
                    originalString: this
                };
            }
        };
    } else if (fileName !== undefined) {
        throw new InternalError("can't change the fileName of an existing MappedString");
    } else {
        return str;
    }
}
function mappedConcat(strings) {
    if (strings.length === 0) {
        return {
            value: "",
            map: (_index, _closest)=>undefined
        };
    }
    if (strings.every((s)=>typeof s === "string")) {
        return asMappedString(strings.join(""));
    }
    const mappedStrings = strings.map((s)=>{
        if (typeof s === "string") {
            return asMappedString(s);
        } else return s;
    });
    let currentOffset = 0;
    const offsets = [
        0
    ];
    for (const s of mappedStrings){
        currentOffset += s.value.length;
        offsets.push(currentOffset);
    }
    const value = mappedStrings.map((s)=>s.value).join("");
    return {
        value,
        map: (offset, closest)=>{
            if (closest) {
                offset = Math.max(0, Math.min(offset, value.length - 1));
            }
            if (offset === 0 && offset == value.length && mappedStrings.length) {
                return mappedStrings[0].map(0, closest);
            }
            if (offset < 0 || offset >= value.length) {
                return undefined;
            }
            const ix = glb(offsets, offset);
            const v = mappedStrings[ix];
            return v.map(offset - offsets[ix]);
        }
    };
}
function mappedIndexToLineCol(eitherText) {
    const text = asMappedString(eitherText);
    return function(offset) {
        const mapResult = text.map(offset, true);
        if (mapResult === undefined) {
            throw new InternalError("bad offset in mappedIndexRowCol");
        }
        const { index, originalString } = mapResult;
        return indexToLineCol(originalString.value)(index);
    };
}
function join3(mappedStrs, sep) {
    const innerStrings = [];
    const mappedSep = asMappedString(sep);
    for(let i1 = 0; i1 < mappedStrs.length; ++i1){
        const mappedStr = mappedStrs[i1];
        if (typeof mappedStr === "string") {
            innerStrings.push(asMappedString(mappedStr));
        } else {
            innerStrings.push(mappedStr);
        }
        if (i1 < mappedStrs.length) {
            innerStrings.push(mappedSep);
        }
    }
    return mappedConcat(innerStrings);
}
function mappedLines(str, keepNewLines = false) {
    const lines = rangedLines(str.value, keepNewLines);
    return lines.map((v)=>mappedString(str, [
            v.range
        ]));
}
function findSpaceStart(string) {
    const result1 = string.value.match(/^\s+/);
    if (result1 === null || result1.length === 0) {
        return 0;
    }
    return result1[0].length;
}
function findSpaceEnd(string) {
    const result1 = string.value.match(/\s+$/);
    if (result1 === null || result1.length === 0) {
        return 0;
    }
    return result1[0].length;
}
function mappedTrim(string) {
    const start = findSpaceStart(string);
    const end = findSpaceEnd(string);
    if (start === 0 && end === 0) {
        return string;
    }
    if (start > string.value.length - end) {
        return mappedSubstring(string, 0, 0);
    }
    return mappedSubstring(string, start, string.value.length - end);
}
Object.freeze({
    __proto__: null
});
class DataSet {
    constructor(raw2018, raw2019, raw2020, raw2021){
        this._raw2018 = raw2018;
        this._raw2019 = raw2019;
        this._raw2020 = raw2020;
        this._raw2021 = raw2021;
    }
    get es2018() {
        return this._set2018 || (this._set2018 = new Set(this._raw2018.split(" ")));
    }
    get es2019() {
        return this._set2019 || (this._set2019 = new Set(this._raw2019.split(" ")));
    }
    get es2020() {
        return this._set2020 || (this._set2020 = new Set(this._raw2020.split(" ")));
    }
    get es2021() {
        return this._set2021 || (this._set2021 = new Set(this._raw2021.split(" ")));
    }
}
new Set([
    "General_Category",
    "gc"
]);
new Set([
    "Script",
    "Script_Extensions",
    "sc",
    "scx"
]);
new DataSet("C Cased_Letter Cc Cf Close_Punctuation Cn Co Combining_Mark Connector_Punctuation Control Cs Currency_Symbol Dash_Punctuation Decimal_Number Enclosing_Mark Final_Punctuation Format Initial_Punctuation L LC Letter Letter_Number Line_Separator Ll Lm Lo Lowercase_Letter Lt Lu M Mark Math_Symbol Mc Me Mn Modifier_Letter Modifier_Symbol N Nd Nl No Nonspacing_Mark Number Open_Punctuation Other Other_Letter Other_Number Other_Punctuation Other_Symbol P Paragraph_Separator Pc Pd Pe Pf Pi Po Private_Use Ps Punctuation S Sc Separator Sk Sm So Space_Separator Spacing_Mark Surrogate Symbol Titlecase_Letter Unassigned Uppercase_Letter Z Zl Zp Zs cntrl digit punct", "", "", "");
new DataSet("Adlam Adlm Aghb Ahom Anatolian_Hieroglyphs Arab Arabic Armenian Armi Armn Avestan Avst Bali Balinese Bamu Bamum Bass Bassa_Vah Batak Batk Beng Bengali Bhaiksuki Bhks Bopo Bopomofo Brah Brahmi Brai Braille Bugi Buginese Buhd Buhid Cakm Canadian_Aboriginal Cans Cari Carian Caucasian_Albanian Chakma Cham Cher Cherokee Common Copt Coptic Cprt Cuneiform Cypriot Cyrillic Cyrl Deseret Deva Devanagari Dsrt Dupl Duployan Egyp Egyptian_Hieroglyphs Elba Elbasan Ethi Ethiopic Geor Georgian Glag Glagolitic Gonm Goth Gothic Gran Grantha Greek Grek Gujarati Gujr Gurmukhi Guru Han Hang Hangul Hani Hano Hanunoo Hatr Hatran Hebr Hebrew Hira Hiragana Hluw Hmng Hung Imperial_Aramaic Inherited Inscriptional_Pahlavi Inscriptional_Parthian Ital Java Javanese Kaithi Kali Kana Kannada Katakana Kayah_Li Khar Kharoshthi Khmer Khmr Khoj Khojki Khudawadi Knda Kthi Lana Lao Laoo Latin Latn Lepc Lepcha Limb Limbu Lina Linb Linear_A Linear_B Lisu Lyci Lycian Lydi Lydian Mahajani Mahj Malayalam Mand Mandaic Mani Manichaean Marc Marchen Masaram_Gondi Meetei_Mayek Mend Mende_Kikakui Merc Mero Meroitic_Cursive Meroitic_Hieroglyphs Miao Mlym Modi Mong Mongolian Mro Mroo Mtei Mult Multani Myanmar Mymr Nabataean Narb Nbat New_Tai_Lue Newa Nko Nkoo Nshu Nushu Ogam Ogham Ol_Chiki Olck Old_Hungarian Old_Italic Old_North_Arabian Old_Permic Old_Persian Old_South_Arabian Old_Turkic Oriya Orkh Orya Osage Osge Osma Osmanya Pahawh_Hmong Palm Palmyrene Pau_Cin_Hau Pauc Perm Phag Phags_Pa Phli Phlp Phnx Phoenician Plrd Prti Psalter_Pahlavi Qaac Qaai Rejang Rjng Runic Runr Samaritan Samr Sarb Saur Saurashtra Sgnw Sharada Shavian Shaw Shrd Sidd Siddham SignWriting Sind Sinh Sinhala Sora Sora_Sompeng Soyo Soyombo Sund Sundanese Sylo Syloti_Nagri Syrc Syriac Tagalog Tagb Tagbanwa Tai_Le Tai_Tham Tai_Viet Takr Takri Tale Talu Tamil Taml Tang Tangut Tavt Telu Telugu Tfng Tglg Thaa Thaana Thai Tibetan Tibt Tifinagh Tirh Tirhuta Ugar Ugaritic Vai Vaii Wara Warang_Citi Xpeo Xsux Yi Yiii Zanabazar_Square Zanb Zinh Zyyy", "Dogr Dogra Gong Gunjala_Gondi Hanifi_Rohingya Maka Makasar Medefaidrin Medf Old_Sogdian Rohg Sogd Sogdian Sogo", "Elym Elymaic Hmnp Nand Nandinagari Nyiakeng_Puachue_Hmong Wancho Wcho", "Chorasmian Chrs Diak Dives_Akuru Khitan_Small_Script Kits Yezi Yezidi");
new DataSet("AHex ASCII ASCII_Hex_Digit Alpha Alphabetic Any Assigned Bidi_C Bidi_Control Bidi_M Bidi_Mirrored CI CWCF CWCM CWKCF CWL CWT CWU Case_Ignorable Cased Changes_When_Casefolded Changes_When_Casemapped Changes_When_Lowercased Changes_When_NFKC_Casefolded Changes_When_Titlecased Changes_When_Uppercased DI Dash Default_Ignorable_Code_Point Dep Deprecated Dia Diacritic Emoji Emoji_Component Emoji_Modifier Emoji_Modifier_Base Emoji_Presentation Ext Extender Gr_Base Gr_Ext Grapheme_Base Grapheme_Extend Hex Hex_Digit IDC IDS IDSB IDST IDS_Binary_Operator IDS_Trinary_Operator ID_Continue ID_Start Ideo Ideographic Join_C Join_Control LOE Logical_Order_Exception Lower Lowercase Math NChar Noncharacter_Code_Point Pat_Syn Pat_WS Pattern_Syntax Pattern_White_Space QMark Quotation_Mark RI Radical Regional_Indicator SD STerm Sentence_Terminal Soft_Dotted Term Terminal_Punctuation UIdeo Unified_Ideograph Upper Uppercase VS Variation_Selector White_Space XIDC XIDS XID_Continue XID_Start space", "Extended_Pictographic", "", "EBase EComp EMod EPres ExtPict");
function navigateSchemaBySchemaPathSingle(schema, path) {
    const ensurePathFragment = (fragment, expected)=>{
        if (fragment !== expected) {
            throw new InternalError(`navigateSchemaBySchemaPathSingle: ${fragment} !== ${expected}`);
        }
    };
    const inner = (subschema, index)=>{
        subschema = resolveSchema(subschema);
        if (subschema === undefined) {
            throw new InternalError(`navigateSchemaBySchemaPathSingle: invalid path navigation`);
        }
        if (index === path.length) {
            return subschema;
        }
        const st = schemaType(subschema);
        switch(st){
            case "anyOf":
                ensurePathFragment(path[index], "anyOf");
                return inner(subschema.anyOf[path[index + 1]], index + 2);
            case "allOf":
                ensurePathFragment(path[index], "allOf");
                return inner(subschema.allOf[path[index + 1]], index + 2);
            case "array":
                ensurePathFragment(path[index], "array");
                return inner(subschema.arrayOf.schema, index + 2);
            case "object":
                ensurePathFragment(path[index], "object");
                if (path[index + 1] === "properties") {
                    return inner(subschema.properties[path[index + 2]], index + 3);
                } else if (path[index + 1] === "patternProperties") {
                    return inner(subschema.patternProperties[path[index + 2]], index + 3);
                } else if (path[index + 1] === "additionalProperties") {
                    return inner(subschema.additionalProperties, index + 2);
                } else {
                    throw new InternalError(`navigateSchemaBySchemaPathSingle: bad path fragment ${path[index]} in object navigation`);
                }
            default:
                throw new InternalError(`navigateSchemaBySchemaPathSingle: can't navigate schema type ${st}`);
        }
    };
    return inner(schema, 0);
}
function resolveDescription(s) {
    if (typeof s === "string") {
        return s;
    }
    const valueS = resolveSchema(s);
    if (valueS === false || valueS === true) {
        return "";
    }
    if (valueS.documentation === undefined) {
        return "";
    }
    if (typeof valueS.documentation === "string") {
        return valueS.documentation;
    }
    if (valueS.documentation.short) {
        return valueS.documentation.short;
    } else {
        return "";
    }
}
function schemaCompletions(s) {
    if (s === true || s === false) {
        return [];
    }
    let schema = resolveSchema(s);
    schema = resolveSchema(schema, (_schema)=>{}, (schema)=>{
        return schema.tags !== undefined && schema.tags["complete-from"] !== undefined;
    }, (schema)=>{
        return navigateSchemaBySchemaPathSingle(schema, schema.tags["complete-from"]);
    });
    if (schema === true || schema === false) {
        return [];
    }
    const normalize = (completions)=>{
        const result1 = (completions || []).map((c)=>{
            if (typeof c === "string") {
                return {
                    type: "value",
                    display: c,
                    value: c,
                    description: "",
                    suggest_on_accept: false,
                    schema
                };
            }
            return {
                ...c,
                description: resolveDescription(c.description),
                schema
            };
        });
        return result1;
    };
    if (schema.completions && schema.completions.length) {
        return normalize(schema.completions);
    }
    if (schema.tags && schema.tags.completions) {
        if (Array.isArray(schema.tags.completions) && schema.tags.completions.length) {
            return normalize(schema.tags.completions);
        } else {
            return normalize(Object.values(schema.tags.completions));
        }
    }
    return schemaCall(schema, {
        array: (s)=>{
            if (s.items) {
                return schemaCompletions(s.items);
            } else {
                return [];
            }
        },
        anyOf: (s)=>{
            return s.anyOf.map(schemaCompletions).flat();
        },
        allOf: (s)=>{
            return s.allOf.map(schemaCompletions).flat();
        },
        "object": (s)=>{
            s.cachedCompletions = getObjectCompletions(s);
            return normalize(s.cachedCompletions);
        }
    }, (_)=>[]);
}
function getObjectCompletions(s) {
    const completionsParam = s.tags && s.tags.completions || [];
    return schemaCall(s, {
        "object": (schema)=>{
            const properties = schema.properties;
            const objectKeys = completionsParam.length ? completionsParam : Object.getOwnPropertyNames(properties);
            const completions = [];
            for (const k of objectKeys){
                const schema = properties && properties[k];
                const maybeDescriptions = [];
                let hidden = false;
                if (schema !== undefined && schema !== true && schema !== false) {
                    if (schema.documentation) {
                        maybeDescriptions.push(schemaDocString(schema.documentation));
                    } else {
                        let described = false;
                        const visitor = (schema)=>{
                            if (schema === false || schema === true) {
                                return;
                            }
                            if (schema.hidden) {
                                hidden = true;
                            }
                            if (described) {
                                return;
                            }
                            if (schema.documentation) {
                                maybeDescriptions.push(schemaDocString(schema.documentation));
                                described = true;
                            }
                        };
                        try {
                            resolveSchema(schema, visitor);
                        } catch (_e) {}
                        if (!described) {
                            schemaDispatch(schema, {
                                ref: (schema)=>maybeDescriptions.push({
                                        $ref: schema.$ref
                                    })
                            });
                        }
                    }
                }
                if (hidden) {
                    continue;
                }
                let description = "";
                for (const md of maybeDescriptions){
                    if (md !== undefined) {
                        description = md;
                        break;
                    }
                }
                completions.push({
                    type: "key",
                    display: "",
                    value: `${k}: `,
                    description,
                    suggest_on_accept: true
                });
            }
            return completions;
        }
    }, (_)=>completionsParam.map((c)=>({
                type: "value",
                display: "",
                value: c,
                description: "",
                suggest_on_accept: false
            })));
}
function possibleSchemaKeys(schema) {
    const precomputedCompletions = schemaCompletions(schema).filter((c)=>c.type === "key").map((c)=>c.value.split(":")[0]);
    if (precomputedCompletions.length) {
        return precomputedCompletions;
    }
    const results = [];
    walkSchema(schema, {
        "object": (s)=>{
            results.push(...Object.keys(s.properties || {}));
            return true;
        },
        "array": (_s)=>true
    });
    return results;
}
function possibleSchemaValues(schema) {
    const results = [];
    walkSchema(schema, {
        "enum": (s)=>{
            results.push(...s["enum"].map(String));
            return true;
        },
        "array": (_s)=>true,
        "object": (_s)=>true
    });
    return results;
}
function walkSchema(schema, f) {
    const recur = {
        "anyOf": (ss)=>{
            for (const s of ss.anyOf){
                walkSchema(s, f);
            }
        },
        "allOf": (ss)=>{
            for (const s of ss.allOf){
                walkSchema(s, f);
            }
        },
        "array": (x)=>{
            if (x.items) {
                walkSchema(x.items, f);
            }
        },
        "object": (x)=>{
            if (x.properties) {
                for (const ss of Object.values(x.properties)){
                    walkSchema(ss, f);
                }
            }
            if (x.patternProperties) {
                for (const ss of Object.values(x.patternProperties)){
                    walkSchema(ss, f);
                }
            }
            if (x.propertyNames) {
                walkSchema(x.propertyNames, f);
            }
        }
    };
    if (typeof f === "function") {
        if (f(schema) === true) {
            return;
        }
    } else {
        if (schemaCall(schema, f, (_)=>false) === true) {
            return;
        }
    }
    schemaCall(schema, recur, (_)=>false);
}
function setDefaultErrorHandlers(validator) {
    validator.addHandler(ignoreExprViolations);
    validator.addHandler(expandEmptySpan);
    validator.addHandler(improveErrorHeadingForValueErrors);
    validator.addHandler(checkForTypeMismatch);
    validator.addHandler(checkForBadBoolean);
    validator.addHandler(checkForBadColon);
    validator.addHandler(checkForBadEquals);
    validator.addHandler(identifyKeyErrors);
    validator.addHandler(checkForNearbyCorrection);
    validator.addHandler(checkForNearbyRequired);
    validator.addHandler(schemaDefinedErrors);
}
function errorKeyword(error) {
    if (error.schemaPath.length === 0) {
        return "";
    }
    return String(error.schemaPath[error.schemaPath.length - 1]);
}
function schemaPathMatches(error, strs) {
    const schemaPath = error.schemaPath.slice(-strs.length);
    if (schemaPath.length !== strs.length) {
        return false;
    }
    return strs.every((str, i1)=>str === schemaPath[i1]);
}
function getBadKey(error) {
    if (error.schemaPath.indexOf("propertyNames") === -1 && error.schemaPath.indexOf("closed") === -1) {
        return undefined;
    }
    const result1 = error.violatingObject.result;
    if (typeof result1 !== "string") {
        throw new InternalError("propertyNames error has a violating non-string.");
    }
    return result1;
}
function getVerbatimInput(error) {
    return error.source.value;
}
function navigate(path, annotation, returnKey = false, pathIndex = 0) {
    if (annotation === undefined) {
        throw new Error("Can't navigate an undefined annotation");
    }
    if (pathIndex >= path.length) {
        return annotation;
    }
    if (annotation.kind === "mapping" || annotation.kind === "block_mapping") {
        const { components } = annotation;
        const searchKey = path[pathIndex];
        const lastKeyIndex = ~~((components.length - 1) / 2) * 2;
        for(let i1 = lastKeyIndex; i1 >= 0; i1 -= 2){
            const key = components[i1].result;
            if (key === searchKey) {
                if (returnKey && pathIndex === path.length - 1) {
                    return navigate(path, components[i1], returnKey, pathIndex + 1);
                } else {
                    return navigate(path, components[i1 + 1], returnKey, pathIndex + 1);
                }
            }
        }
        return annotation;
    } else if ([
        "sequence",
        "block_sequence",
        "flow_sequence"
    ].indexOf(annotation.kind) !== -1) {
        const searchKey = Number(path[pathIndex]);
        if (isNaN(searchKey) || searchKey < 0 || searchKey >= annotation.components.length) {
            return annotation;
        }
        return navigate(path, annotation.components[searchKey], returnKey, pathIndex + 1);
    } else {
        return annotation;
    }
}
function isEmptyValue(error) {
    const rawVerbatimInput = getVerbatimInput(error);
    return rawVerbatimInput.trim().length === 0;
}
function getLastFragment(instancePath) {
    if (instancePath.length === 0) {
        return undefined;
    }
    return instancePath[instancePath.length - 1];
}
function reindent(str) {
    const s = new Set();
    const ls = lines(str);
    for (const l1 of ls){
        const r = l1.match("^[ ]+");
        if (r) {
            s.add(r[0].length);
        }
    }
    if (s.size === 0) {
        return str;
    } else if (s.size === 1) {
        const v = Array.from(s)[0];
        const oldIndent = " ".repeat(v);
        if (v <= 2) {
            return str;
        }
        return ls.map((l1)=>l1.startsWith(oldIndent) ? l1.slice(v - 2) : l1).join("\n");
    } else {
        const [first, second] = Array.from(s);
        const oldIndent = " ".repeat(first);
        const newIndent = second - first;
        if (newIndent >= first) {
            return str;
        }
        return ls.map((l1)=>l1.startsWith(oldIndent) ? l1.slice(first - newIndent) : l1).join("\n");
    }
}
function ignoreExprViolations(error, _parse, _schema) {
    const { result: result1 } = error.violatingObject;
    if (typeof result1 !== "object" || Array.isArray(result1) || result1 === null || error.schemaPath.slice(-1)[0] !== "type") {
        return error;
    }
    if (result1.tag === "!expr" && typeof result1.value === "string") {
        return null;
    } else {
        return error;
    }
}
function formatHeadingForKeyError(_error, _parse, _schema, key) {
    return `property name ${blue2(key)} is invalid`;
}
function formatHeadingForValueError(error, _parse, _schema) {
    const rawVerbatimInput = reindent(getVerbatimInput(error));
    const rawLines = lines(rawVerbatimInput);
    let verbatimInput;
    if (rawLines.length > 4) {
        verbatimInput = quotedStringColor([
            ...rawLines.slice(0, 2),
            "...",
            ...rawLines.slice(-2)
        ].join("\n"));
    } else {
        verbatimInput = quotedStringColor(rawVerbatimInput);
    }
    const empty = isEmptyValue(error);
    const lastFragment = getLastFragment(error.instancePath);
    switch(typeof lastFragment){
        case "undefined":
            if (empty) {
                return "YAML value is missing.";
            } else {
                return `YAML value ${verbatimInput} must ${schemaDescription(error.schema)}.`;
            }
        case "number":
            if (empty) {
                return `Array entry ${lastFragment + 1} is empty but it must instead ${schemaDescription(error.schema)}.`;
            } else {
                return `Array entry ${lastFragment + 1} with value ${verbatimInput} failed to ${schemaDescription(error.schema)}.`;
            }
        case "string":
            {
                const formatLastFragment = '"' + blue2(lastFragment) + '"';
                if (empty) {
                    return `Field ${formatLastFragment} has empty value but it must instead ${schemaDescription(error.schema)}`;
                } else {
                    if (verbatimInput.indexOf("\n") !== -1) {
                        return `Field ${formatLastFragment} has value

${verbatimInput}

The value must instead ${schemaDescription(error.schema)}.`;
                    } else {
                        return `Field ${formatLastFragment} has value ${verbatimInput}, which must instead ${schemaDescription(error.schema)}`;
                    }
                }
            }
    }
}
function identifyKeyErrors(error, parse, schema) {
    if (error.schemaPath.indexOf("propertyNames") === -1 && error.schemaPath.indexOf("closed") === -1) {
        return error;
    }
    const badKey = getBadKey(error);
    if (badKey) {
        if (error.instancePath.length && error.instancePath[error.instancePath.length - 1] !== badKey) {
            addInstancePathInfo(error.niceError, [
                ...error.instancePath,
                badKey
            ]);
        } else {
            addInstancePathInfo(error.niceError, error.instancePath);
        }
        error.niceError.heading = formatHeadingForKeyError(error, parse, schema, badKey);
    }
    return error;
}
function improveErrorHeadingForValueErrors(error, parse, schema) {
    if (error.schemaPath.indexOf("propertyNames") !== -1 || error.schemaPath.indexOf("closed") !== -1 || errorKeyword(error) === "required") {
        return error;
    }
    return {
        ...error,
        niceError: {
            ...error.niceError,
            heading: formatHeadingForValueError(error, parse, schema)
        }
    };
}
function expandEmptySpan(error, parse, _schema) {
    if (error.location.start.line !== error.location.end.line || error.location.start.column !== error.location.end.column || !isEmptyValue(error) || typeof getLastFragment(error.instancePath) === "undefined") {
        return error;
    }
    const lastKey = navigate(error.instancePath, parse, true);
    const locF = mappedIndexToLineCol(parse.source);
    try {
        const location = {
            start: locF(lastKey.start),
            end: locF(lastKey.end)
        };
        return {
            ...error,
            location,
            niceError: {
                ...error.niceError,
                location
            }
        };
    } catch (_e) {
        return error;
    }
}
function checkForTypeMismatch(error, parse, schema) {
    const rawVerbatimInput = getVerbatimInput(error);
    const rawLines = lines(rawVerbatimInput);
    let verbatimInput;
    if (rawLines.length > 4) {
        verbatimInput = quotedStringColor([
            ...rawLines.slice(0, 2),
            "...",
            ...rawLines.slice(-2)
        ].join("\n"));
    } else {
        verbatimInput = quotedStringColor(rawVerbatimInput);
    }
    const goodType = (obj)=>{
        if (Array.isArray(obj)) {
            return "an array";
        }
        if (obj === null) {
            return "a null value";
        }
        return typeof obj;
    };
    if (errorKeyword(error) === "type" && rawVerbatimInput.length > 0) {
        const reindented = reindent(verbatimInput);
        const subject = reindented.indexOf("\n") === -1 ? `The value ${reindented} ` : `The value

${reindented}

`;
        const newError = {
            ...error.niceError,
            heading: formatHeadingForValueError(error, parse, schema),
            error: [
                `${subject}is of type ${goodType(error.violatingObject.result)}.`
            ],
            info: {},
            location: error.niceError.location
        };
        addInstancePathInfo(newError, error.instancePath);
        addFileInfo(newError, error.source);
        return {
            ...error,
            niceError: newError
        };
    }
    return error;
}
function checkForBadBoolean(error, parse, _schema) {
    const schema = error.schema;
    if (!(typeof error.violatingObject.result === "string" && errorKeyword(error) === "type" && schemaType(schema) === "boolean")) {
        return error;
    }
    const strValue = error.violatingObject.result;
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const yesses = new Set("y|Y|yes|Yes|YES|true|True|TRUE|on|On|ON".split("|"));
    const nos = new Set("n|N|no|No|NO|false|False|FALSE|off|Off|OFF".split("|"));
    let fix;
    if (yesses.has(strValue)) {
        fix = true;
    } else if (nos.has(strValue)) {
        fix = false;
    } else {
        return error;
    }
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `Quarto uses YAML 1.2, which interprets booleans strictly.`;
    const suggestion2 = `Try using ${quotedStringColor(String(fix))} instead.`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-version-1.2"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function checkForBadColon(error, parse, schema) {
    if (typeof error.violatingObject.result !== "string") {
        return error;
    }
    if (!schemaPathMatches(error, [
        "object",
        "type"
    ])) {
        return error;
    }
    if (!error.violatingObject.result.match(/^.+:[^ ].*$/)) {
        return error;
    }
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `In YAML, key-value pairs in objects must be separated by a space.`;
    const suggestion2 = `Did you mean ${quotedStringColor(quotedStringColor(getVerbatimInput(error)).replace(/:/g, ": "))} instead?`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-key-value-pairs"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function checkForBadEquals(error, parse, schema) {
    if (typeof error.violatingObject.result !== "string") {
        return error;
    }
    if (!schemaPathMatches(error, [
        "object",
        "type"
    ]) && !schemaPathMatches(error, [
        "object",
        "propertyNames",
        "string",
        "pattern"
    ])) {
        return error;
    }
    if (!error.violatingObject.result.match(/^.+ *= *.+$/)) {
        return error;
    }
    const verbatimInput = quotedStringColor(getVerbatimInput(error));
    const errorMessage = `The value ${verbatimInput} is a string.`;
    const suggestion1 = `In YAML, key-value pairs in objects must be separated by a colon and a space.`;
    const suggestion2 = `Did you mean ${quotedStringColor(quotedStringColor(getVerbatimInput(error)).replace(/ *= */g, ": "))} instead?`;
    const newError = {
        heading: formatHeadingForValueError(error, parse, schema),
        error: [
            errorMessage
        ],
        info: {},
        location: error.niceError.location
    };
    addInstancePathInfo(newError, error.instancePath);
    addFileInfo(newError, error.source);
    newError.info["yaml-key-value-pairs"] = suggestion1;
    newError.info["suggestion-fix"] = suggestion2;
    return {
        ...error,
        niceError: newError
    };
}
function createErrorFragments(error) {
    const rawVerbatimInput = getVerbatimInput(error);
    const verbatimInput = quotedStringColor(reindent(rawVerbatimInput));
    const pathFragments = error.instancePath.map((s)=>blue2(String(s)));
    return {
        location: locationString(error.location),
        fullPath: pathFragments.join(":"),
        key: pathFragments[pathFragments.length - 1],
        value: verbatimInput
    };
}
function schemaDefinedErrors(error, _parse, _schema) {
    const schema = error.schema;
    if (schema === true || schema === false) {
        return error;
    }
    if (schema.errorMessage === undefined) {
        return error;
    }
    if (typeof schema.errorMessage !== "string") {
        return error;
    }
    let result1 = schema.errorMessage;
    for (const [k, v] of Object.entries(createErrorFragments(error))){
        result1 = result1.replace("${" + k + "}", v);
    }
    return {
        ...error,
        niceError: {
            ...error.niceError,
            heading: result1
        }
    };
}
function checkForNearbyRequired(error, _parse, _schema) {
    const schema = error.schema;
    if (errorKeyword(error) !== "required") {
        return error;
    }
    const missingKeys = [];
    const errObj = error.violatingObject.result;
    const keys = Object.keys(errObj);
    schemaCall(schema, {
        object (s) {
            if (s.required === undefined) {
                throw new InternalError("required schema error without a required field");
            }
            for (const r of s.required){
                if (keys.indexOf(r) === -1) {
                    missingKeys.push(r);
                }
            }
        }
    }, (_)=>{
        throw new InternalError("required error on a non-object schema");
    });
    for (const missingKey of missingKeys){
        let bestCorrection;
        let bestDistance = Infinity;
        for (const correction of keys){
            const d = editDistance(correction, missingKey);
            if (d < bestDistance) {
                bestCorrection = [
                    correction
                ];
                bestDistance = d;
            } else if (d === bestDistance) {
                bestCorrection.push(correction);
                bestDistance = d;
            }
        }
        if (bestDistance > missingKey.length * 10 * 0.3) {
            continue;
        }
        const suggestions = bestCorrection.map((s)=>blue2(s));
        if (suggestions.length === 1) {
            error.niceError.info[`did-you-mean-key`] = `Is ${suggestions[0]} a typo of ${blue2(missingKey)}?`;
        } else if (suggestions.length === 2) {
            error.niceError.info[`did-you-mean-key`] = `Is ${suggestions[0]} or ${suggestions[1]} a typo of ${blue2(missingKey)}?`;
        } else {
            suggestions[suggestions.length - 1] = `or ${suggestions[suggestions.length - 1]}`;
            error.niceError.info[`did-you-mean-key`] = `Is one of ${suggestions.join(", ")} a typo of ${blue2(missingKey)}?`;
        }
    }
    return error;
}
function checkForNearbyCorrection(error, parse, _schema) {
    const schema = error.schema;
    const corrections = [];
    let errVal = "";
    let keyOrValue = "";
    const key = getBadKey(error);
    if (key) {
        errVal = key;
        corrections.push(...possibleSchemaKeys(schema));
        keyOrValue = "key";
    } else {
        const val = navigate(error.instancePath, parse);
        if (typeof val.result !== "string") {
            return error;
        }
        errVal = val.result;
        corrections.push(...possibleSchemaValues(schema));
        keyOrValue = "value";
    }
    if (corrections.length === 0) {
        return error;
    }
    let bestCorrection;
    let bestDistance = Infinity;
    for (const correction of corrections){
        const d = editDistance(correction, errVal);
        if (d < bestDistance) {
            bestCorrection = [
                correction
            ];
            bestDistance = d;
        } else if (d === bestDistance) {
            bestCorrection.push(correction);
            bestDistance = d;
        }
    }
    if (bestDistance > errVal.length * 10 * 0.3) {
        return error;
    }
    const suggestions = bestCorrection.map((s)=>blue2(s));
    if (suggestions.length === 1) {
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions[0]}?`;
    } else if (suggestions.length === 2) {
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions[0]} or ${suggestions[1]}?`;
    } else {
        suggestions[suggestions.length - 1] = `or ${suggestions[suggestions.length - 1]}`;
        error.niceError.info[`did-you-mean-${keyOrValue}`] = `Did you mean ${suggestions.join(", ")}?`;
    }
    return error;
}
function createSourceContext(src, location) {
    if (src.value.length === 0) {
        return "";
    }
    const startMapResult = src.map(location.start, true);
    const endMapResult = src.map(location.end, true);
    const locF = mappedIndexToLineCol(src);
    let sourceLocation;
    try {
        sourceLocation = {
            start: locF(location.start),
            end: locF(location.end)
        };
    } catch (_e) {
        sourceLocation = {
            start: {
                line: 0,
                column: 0
            },
            end: {
                line: 0,
                column: 0
            }
        };
    }
    if (startMapResult === undefined || endMapResult === undefined) {
        throw new InternalError(`createSourceContext called with bad location ${location.start}-${location.end}.`);
    }
    if (startMapResult.originalString !== endMapResult.originalString) {
        throw new InternalError("don't know how to create source context across different source files");
    }
    const originalString = startMapResult.originalString;
    const nLines = lines(originalString.value).length;
    const { start, end } = sourceLocation;
    const { prefixWidth, lines: formattedLines } = formatLineRange(originalString.value, Math.max(0, start.line - 1), Math.min(end.line + 1, nLines - 1));
    const contextLines = [];
    let mustPrintEllipsis = true;
    for (const { lineNumber, content, rawLine } of formattedLines){
        if (lineNumber < start.line || lineNumber > end.line) {
            if (rawLine.trim().length) {
                contextLines.push(content);
            }
        } else {
            if (lineNumber >= start.line + 2 && lineNumber <= end.line - 2) {
                if (mustPrintEllipsis) {
                    mustPrintEllipsis = false;
                    contextLines.push("...");
                }
            } else {
                const startColumn = lineNumber > start.line ? 0 : start.column;
                const endColumn = lineNumber < end.line ? rawLine.length : end.column;
                contextLines.push(content);
                contextLines.push(" ".repeat(prefixWidth + startColumn - 1) + "~".repeat(endColumn - startColumn + 1));
            }
        }
    }
    return contextLines.join("\n");
}
function createLocalizedError(obj) {
    const { violatingObject, instancePath, schemaPath, source, message, schema } = obj;
    const locF = mappedIndexToLineCol(source);
    let location;
    try {
        location = {
            start: locF(violatingObject.start),
            end: locF(violatingObject.end)
        };
    } catch (_e) {
        location = {
            start: {
                line: 0,
                column: 0
            },
            end: {
                line: 0,
                column: 0
            }
        };
    }
    const mapResult = source.map(violatingObject.start);
    const fileName = mapResult ? mapResult.originalString.fileName : undefined;
    return {
        source: mappedString(source, [
            {
                start: violatingObject.start,
                end: violatingObject.end
            }
        ]),
        violatingObject: violatingObject,
        instancePath,
        schemaPath,
        schema,
        message,
        location: location,
        niceError: {
            heading: message,
            error: [],
            info: {},
            fileName,
            location: location,
            sourceContext: createSourceContext(violatingObject.source, {
                start: violatingObject.start,
                end: violatingObject.end
            })
        }
    };
}
class ValidationContext {
    instancePath;
    root;
    nodeStack;
    currentNode;
    constructor(){
        this.instancePath = [];
        this.currentNode = {
            edge: "#",
            errors: [],
            children: []
        };
        this.nodeStack = [
            this.currentNode
        ];
        this.root = this.currentNode;
    }
    error(value, schema, message) {
        this.currentNode.errors.push({
            value,
            schema,
            message,
            instancePath: this.instancePath.slice(),
            schemaPath: this.nodeStack.map((node)=>node.edge)
        });
    }
    pushSchema(schemaPath) {
        const newNode = {
            edge: schemaPath,
            errors: [],
            children: []
        };
        this.currentNode.children.push(newNode);
        this.currentNode = newNode;
        this.nodeStack.push(newNode);
    }
    popSchema(success) {
        this.nodeStack.pop();
        this.currentNode = this.nodeStack[this.nodeStack.length - 1];
        if (success) {
            this.currentNode.children.pop();
        }
        return success;
    }
    pushInstance(instance) {
        this.instancePath.push(instance);
    }
    popInstance() {
        this.instancePath.pop();
    }
    withSchemaPath(schemaPath, chunk) {
        this.pushSchema(schemaPath);
        return this.popSchema(chunk());
    }
    validate(schema, source, value, pruneErrors = true) {
        if (validateGeneric(value, schema, this)) {
            return [];
        }
        return this.collectErrors(schema, source, value, pruneErrors);
    }
    collectErrors(_schema, source, _value, pruneErrors = true) {
        const inner = (node)=>{
            const result1 = [];
            if (node.edge === "anyOf" && pruneErrors) {
                const innerResults = node.children.map(inner);
                const isRequiredError = (e)=>e.schemaPath.indexOf("required") === e.schemaPath.length - 1;
                const isPropertyNamesError = (e)=>e.schemaPath.indexOf("propertyNames") !== -1;
                if (innerResults.some((el)=>el.length && isRequiredError(el[0])) && innerResults.some((el)=>el.length && isPropertyNamesError(el[0]))) {
                    return innerResults.filter((r)=>{
                        return r.length && r[0].schemaPath.slice(-1)[0] === "required";
                    })[0];
                }
                const errorTypeQuality = (e)=>{
                    const t = e.schemaPath.slice().reverse();
                    if (typeof e.schema === "object") {
                        if (e.schema.tags && e.schema.tags["error-importance"] && typeof e.schema.tags["error-importance"] === "number") {
                            return e.schema.tags["error-importance"];
                        }
                    }
                    if (e.schemaPath.indexOf("propertyNames") !== -1) {
                        return 10;
                    }
                    if (t[0] === "required") {
                        return 0;
                    }
                    if (t[0] === "type") {
                        if (t[1] === "null") {
                            return 10;
                        }
                        return 1;
                    }
                    return 1;
                };
                const errorComparator = (a, b)=>{
                    for(let i1 = 0; i1 < a.length; ++i1){
                        if (a[i1] < b[i1]) {
                            return -1;
                        }
                        if (a[i1] > b[i1]) {
                            return 1;
                        }
                    }
                    return 0;
                };
                let bestResults = [];
                let bestError = [
                    Infinity,
                    Infinity
                ];
                for (const resultGroup of innerResults){
                    let maxQuality = -Infinity;
                    let totalSpan = 0;
                    for (const result1 of resultGroup){
                        totalSpan += result1.value.end - result1.value.start;
                        maxQuality = Math.max(maxQuality, errorTypeQuality(result1));
                    }
                    const thisError = [
                        maxQuality,
                        totalSpan
                    ];
                    if (errorComparator(thisError, bestError) === -1) {
                        bestError = thisError;
                        bestResults = resultGroup;
                    }
                }
                return bestResults;
            } else {
                result1.push(...node.errors);
                for (const child of node.children){
                    result1.push(...inner(child));
                }
                return result1;
            }
        };
        const errors = inner(this.root);
        const result1 = errors.map((validationError)=>createLocalizedError({
                violatingObject: validationError.value,
                instancePath: validationError.instancePath,
                schemaPath: validationError.schemaPath,
                schema: validationError.schema,
                message: validationError.message,
                source
            }));
        return result1;
    }
}
function validateGeneric(value, s, context) {
    s = resolveSchema(s);
    const st = schemaType(s);
    return context.withSchemaPath(st, ()=>schemaCall(s, {
            "false": (schema)=>{
                context.error(value, schema, "false");
                return false;
            },
            "true": (_)=>true,
            "any": (schema)=>validateAny(value, schema, context),
            "boolean": (schema)=>validateBoolean(value, schema, context),
            "number": (schema)=>validateNumber(value, schema, context),
            "string": (schema)=>validateString(value, schema, context),
            "null": (schema)=>validateNull(value, schema, context),
            "enum": (schema)=>validateEnum(value, schema, context),
            "anyOf": (schema)=>validateAnyOf(value, schema, context),
            "allOf": (schema)=>validateAllOf(value, schema, context),
            "array": (schema)=>validateArray(value, schema, context),
            "object": (schema)=>validateObject(value, schema, context),
            "ref": (schema)=>validateGeneric(value, resolveSchema(schema), context)
        }));
}
function typeIsValid(value, schema, context, valid) {
    if (!valid) {
        return context.withSchemaPath("type", ()=>{
            context.error(value, schema, "type mismatch");
            return false;
        });
    }
    return valid;
}
function validateAny(_value, _schema, _context) {
    return true;
}
function validateBoolean(value, schema, context) {
    return typeIsValid(value, schema, context, typeof value.result === "boolean");
}
function validateNumber(value, schema, context) {
    if (!typeIsValid(value, schema, context, typeof value.result === "number")) {
        return false;
    }
    let result1 = true;
    if (schema.minimum !== undefined) {
        result1 = context.withSchemaPath("minimum", ()=>{
            const v = value.result;
            if (!(v >= schema.minimum)) {
                context.error(value, schema, `value ${value.result} is less than required minimum ${schema.minimum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.maximum !== undefined) {
        result1 = context.withSchemaPath("maximum", ()=>{
            const v = value.result;
            if (!(v <= schema.maximum)) {
                context.error(value, schema, `value ${value.result} is greater than required maximum ${schema.maximum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.exclusiveMinimum !== undefined) {
        result1 = context.withSchemaPath("exclusiveMinimum", ()=>{
            const v = value.result;
            if (!(v > schema.exclusiveMinimum)) {
                context.error(value, schema, `value ${value.result} is less than or equal to required (exclusive) minimum ${schema.exclusiveMinimum}`);
                return false;
            }
            return true;
        });
    }
    if (schema.exclusiveMaximum !== undefined) {
        result1 = context.withSchemaPath("exclusiveMaximum", ()=>{
            const v = value.result;
            if (!(v < schema.exclusiveMaximum)) {
                context.error(value, schema, `value ${value.result} is greater than or equal to required (exclusive) maximum ${schema.exclusiveMaximum}`);
                return false;
            }
            return true;
        });
    }
    return result1;
}
function validateString(value, schema, context) {
    if (!typeIsValid(value, schema, context, typeof value.result === "string")) {
        return false;
    }
    if (schema.pattern !== undefined) {
        if (schema.compiledPattern === undefined) {
            schema.compiledPattern = new RegExp(schema.pattern);
        }
        if (!value.result.match(schema.compiledPattern)) {
            return context.withSchemaPath("pattern", ()=>{
                context.error(value, schema, `value doesn't match pattern`);
                return false;
            });
        }
    }
    return true;
}
function validateNull(value, schema, context) {
    if (!typeIsValid(value, schema, context, value.result === null)) {
        return false;
    }
    return true;
}
function validateEnum(value, schema, context) {
    for (const enumValue of schema["enum"]){
        if (enumValue === value.result) {
            return true;
        }
    }
    context.error(value, schema, `must match one of the values`);
    return false;
}
function validateAnyOf(value, schema, context) {
    let passingSchemas = 0;
    for(let i1 = 0; i1 < schema.anyOf.length; ++i1){
        const subSchema = schema.anyOf[i1];
        context.withSchemaPath(i1, ()=>{
            if (validateGeneric(value, subSchema, context)) {
                passingSchemas++;
                return true;
            }
            return false;
        });
    }
    return passingSchemas > 0;
}
function validateAllOf(value, schema, context) {
    let passingSchemas = 0;
    for(let i1 = 0; i1 < schema.allOf.length; ++i1){
        const subSchema = schema.allOf[i1];
        context.withSchemaPath(i1, ()=>{
            if (validateGeneric(value, subSchema, context)) {
                passingSchemas++;
                return true;
            }
            return false;
        });
    }
    return passingSchemas === schema.allOf.length;
}
function validateArray(value, schema, context) {
    let result1 = true;
    if (!typeIsValid(value, schema, context, Array.isArray(value.result))) {
        return false;
    }
    const length = value.result.length;
    if (schema.minItems !== undefined && length < schema.minItems) {
        context.withSchemaPath("minItems", ()=>{
            context.error(value, schema, `array should have at least ${schema.minItems} items but has ${length} items instead`);
            return false;
        });
        result1 = false;
    }
    if (schema.maxItems !== undefined && length > schema.maxItems) {
        context.withSchemaPath("maxItems", ()=>{
            context.error(value, schema, `array should have at most ${schema.maxItems} items but has ${length} items instead`);
            return false;
        });
        result1 = false;
    }
    if (schema.items !== undefined) {
        result1 = context.withSchemaPath("items", ()=>{
            let result1 = true;
            for(let i1 = 0; i1 < value.components.length; ++i1){
                context.pushInstance(i1);
                result1 = validateGeneric(value.components[i1], schema.items, context) && result1;
                context.popInstance();
            }
            return result1;
        }) && result1;
    }
    return result1;
}
function validateObject(value, schema, context) {
    const isObject = typeof value.result === "object" && !Array.isArray(value.result) && value.result !== null;
    if (!typeIsValid(value, schema, context, isObject)) {
        return false;
    }
    let result1 = true;
    const ownProperties = new Set(Object.getOwnPropertyNames(value.result));
    const objResult = value.result;
    const locate = (key, keyOrValue = "value")=>{
        for(let i1 = 0; i1 < value.components.length; i1 += 2){
            if (String(value.components[i1].result) === key) {
                if (keyOrValue === "value") {
                    return value.components[i1 + 1];
                } else {
                    return value.components[i1];
                }
            }
        }
        throw new InternalError(`Couldn't locate key ${key}`);
    };
    const inspectedProps = new Set();
    if (schema.closed) {
        result1 = context.withSchemaPath("closed", ()=>{
            if (schema.properties === undefined) {
                throw new InternalError("Closed schemas need properties");
            }
            let innerResult = true;
            for (const key of ownProperties){
                if (!schema.properties[key]) {
                    context.error(locate(key, "key"), schema, `object has invalid field ${key}`);
                    innerResult = false;
                }
            }
            return innerResult;
        }) && result1;
    }
    if (schema.properties !== undefined) {
        result1 = context.withSchemaPath("properties", ()=>{
            let result1 = true;
            for (const [key, subSchema] of Object.entries(schema.properties)){
                if (ownProperties.has(key)) {
                    inspectedProps.add(key);
                    context.pushInstance(key);
                    result1 = context.withSchemaPath(key, ()=>validateGeneric(locate(key), subSchema, context)) && result1;
                    context.popInstance();
                }
            }
            return result1;
        }) && result1;
    }
    if (schema.patternProperties !== undefined) {
        result1 = context.withSchemaPath("patternProperties", ()=>{
            let result1 = true;
            for (const [key, subSchema] of Object.entries(schema.patternProperties)){
                if (schema.compiledPatterns === undefined) {
                    schema.compiledPatterns = {};
                }
                if (schema.compiledPatterns[key] === undefined) {
                    schema.compiledPatterns[key] = new RegExp(key);
                }
                const regexp = schema.compiledPatterns[key];
                for (const [objectKey, _val] of Object.entries(objResult)){
                    if (objectKey.match(regexp)) {
                        inspectedProps.add(objectKey);
                        context.pushInstance(objectKey);
                        result1 = context.withSchemaPath(key, ()=>validateGeneric(locate(objectKey), subSchema, context)) && result1;
                        context.popInstance();
                    }
                }
            }
            return result1;
        }) && result1;
    }
    if (schema.additionalProperties !== undefined) {
        result1 = context.withSchemaPath("additionalProperties", ()=>{
            return Object.keys(objResult).filter((objectKey)=>!inspectedProps.has(objectKey)).every((objectKey)=>validateGeneric(locate(objectKey), schema.additionalProperties, context));
        }) && result1;
    }
    if (schema.propertyNames !== undefined) {
        result1 = context.withSchemaPath("propertyNames", ()=>{
            return Array.from(ownProperties).every((key)=>validateGeneric(locate(key, "key"), schema.propertyNames, context));
        }) && result1;
    }
    if (schema.required !== undefined) {
        result1 = context.withSchemaPath("required", ()=>{
            let result1 = true;
            for (const reqKey of schema.required){
                if (!ownProperties.has(reqKey)) {
                    context.error(value, schema, `object is missing required property ${reqKey}`);
                    result1 = false;
                }
            }
            return result1;
        }) && result1;
    }
    return result1;
}
function validate(value, schema, source, pruneErrors = true) {
    const context = new ValidationContext();
    return context.validate(schema, source, value, pruneErrors);
}
class YAMLSchema {
    schema;
    errorHandlers;
    constructor(schema){
        this.errorHandlers = [];
        this.schema = schema;
    }
    addHandler(handler) {
        this.errorHandlers.push(handler);
    }
    transformErrors(annotation, errors) {
        return errors.map((error)=>{
            for (const handler of this.errorHandlers){
                const localError = handler(error, annotation, this.schema);
                if (localError === null) {
                    return null;
                }
                error = localError;
            }
            return error;
        }).filter((error)=>error !== null);
    }
    async validateParse(src, annotation, pruneErrors = true) {
        const validationErrors = validate(annotation, this.schema, src, pruneErrors);
        if (validationErrors.length) {
            const localizedErrors = this.transformErrors(annotation, validationErrors);
            return {
                result: annotation.result,
                errors: localizedErrors
            };
        } else {
            return {
                result: annotation.result,
                errors: []
            };
        }
    }
    reportErrorsInSource(result1, _src, message, error, log) {
        if (result1.errors.length) {
            if (message.length) {
                error(message);
            }
            for (const err of result1.errors){
                log(err.niceError);
            }
        }
        return result1;
    }
    async validateParseWithErrors(src, annotation, message, error, log) {
        const result1 = await this.validateParse(src, annotation);
        this.reportErrorsInSource(result1, src, message, error, log);
        return result1;
    }
}
const yamlValidators = {};
function getSchemaName(schema) {
    if (schema === true || schema === false) {
        throw new Error("Expected schema to be named");
    }
    let schemaName = schema["$id"];
    if (schemaName !== undefined) {
        return schemaName;
    }
    if (schemaType(schema) === "ref") {
        schemaName = schema["$ref"];
    }
    if (schemaName !== undefined) {
        return schemaName;
    }
    throw new Error("Expected schema to be named");
}
function getValidator(schema) {
    const schemaName = getSchemaName(schema);
    if (yamlValidators[schemaName]) {
        return yamlValidators[schemaName];
    }
    const validator = new YAMLSchema(schema);
    yamlValidators[schemaName] = validator;
    setDefaultErrorHandlers(validator);
    return validator;
}
async function withValidator(schema, fun) {
    let result1;
    let error;
    try {
        const validator = getValidator(schema);
        result1 = await fun(validator);
    } catch (e) {
        error = e;
    }
    if (error !== undefined) {
        throw error;
    }
    return result1;
}
function addValidatorErrorHandler(schema, handler) {
    return withValidator(schema, async (validator)=>{
        validator.addHandler(handler);
    });
}
function memoize1(f, keyMemoizer) {
    const memo = {};
    const inner = (...args)=>{
        const key = keyMemoizer(...args);
        const v = memo[key];
        if (v !== undefined) {
            return v;
        }
        memo[key] = f(...args);
        return memo[key];
    };
    return inner;
}
const regExpEscapeChars1 = [
    "!",
    "$",
    "(",
    ")",
    "*",
    "+",
    ".",
    "=",
    "?",
    "[",
    "\\",
    "^",
    "{",
    "|"
];
const rangeEscapeChars1 = [
    "-",
    "\\",
    "]"
];
function globToRegExp3(glob, { extended = true, globstar: globstarOption = true, caseInsensitive = false } = {}) {
    if (glob == "") {
        return /(?!)/;
    }
    const sep = "/+";
    const sepMaybe = "/*";
    const seps = [
        "/"
    ];
    const globstar = "(?:[^/]*(?:/|$)+)*";
    const wildcard = "[^/]*";
    const escapePrefix = "\\";
    let newLength = glob.length;
    for(; newLength > 1 && seps.includes(glob[newLength - 1]); newLength--);
    glob = glob.slice(0, newLength);
    let regExpString = "";
    for(let j = 0; j < glob.length;){
        let segment = "";
        const groupStack = [];
        let inRange = false;
        let inEscape = false;
        let endsWithSep = false;
        let i1 = j;
        for(; i1 < glob.length && !seps.includes(glob[i1]); i1++){
            if (inEscape) {
                inEscape = false;
                const escapeChars = inRange ? rangeEscapeChars1 : regExpEscapeChars1;
                segment += escapeChars.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
                continue;
            }
            if (glob[i1] == escapePrefix) {
                inEscape = true;
                continue;
            }
            if (glob[i1] == "[") {
                if (!inRange) {
                    inRange = true;
                    segment += "[";
                    if (glob[i1 + 1] == "!") {
                        i1++;
                        segment += "^";
                    } else if (glob[i1 + 1] == "^") {
                        i1++;
                        segment += "\\^";
                    }
                    continue;
                } else if (glob[i1 + 1] == ":") {
                    let k = i1 + 1;
                    let value = "";
                    while(glob[k + 1] != null && glob[k + 1] != ":"){
                        value += glob[k + 1];
                        k++;
                    }
                    if (glob[k + 1] == ":" && glob[k + 2] == "]") {
                        i1 = k + 2;
                        if (value == "alnum") segment += "\\dA-Za-z";
                        else if (value == "alpha") segment += "A-Za-z";
                        else if (value == "ascii") segment += "\x00-\x7F";
                        else if (value == "blank") segment += "\t ";
                        else if (value == "cntrl") segment += "\x00-\x1F\x7F";
                        else if (value == "digit") segment += "\\d";
                        else if (value == "graph") segment += "\x21-\x7E";
                        else if (value == "lower") segment += "a-z";
                        else if (value == "print") segment += "\x20-\x7E";
                        else if (value == "punct") {
                            segment += "!\"#$%&'()*+,\\-./:;<=>?@[\\\\\\]^_‘{|}~";
                        } else if (value == "space") segment += "\\s\v";
                        else if (value == "upper") segment += "A-Z";
                        else if (value == "word") segment += "\\w";
                        else if (value == "xdigit") segment += "\\dA-Fa-f";
                        continue;
                    }
                }
            }
            if (glob[i1] == "]" && inRange) {
                inRange = false;
                segment += "]";
                continue;
            }
            if (inRange) {
                if (glob[i1] == "\\") {
                    segment += `\\\\`;
                } else {
                    segment += glob[i1];
                }
                continue;
            }
            if (glob[i1] == ")" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += ")";
                const type = groupStack.pop();
                if (type == "!") {
                    segment += wildcard;
                } else if (type != "@") {
                    segment += type;
                }
                continue;
            }
            if (glob[i1] == "|" && groupStack.length > 0 && groupStack[groupStack.length - 1] != "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "+" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("+");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "@" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("@");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "?") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("?");
                    segment += "(?:";
                } else {
                    segment += ".";
                }
                continue;
            }
            if (glob[i1] == "!" && extended && glob[i1 + 1] == "(") {
                i1++;
                groupStack.push("!");
                segment += "(?!";
                continue;
            }
            if (glob[i1] == "{") {
                groupStack.push("BRACE");
                segment += "(?:";
                continue;
            }
            if (glob[i1] == "}" && groupStack[groupStack.length - 1] == "BRACE") {
                groupStack.pop();
                segment += ")";
                continue;
            }
            if (glob[i1] == "," && groupStack[groupStack.length - 1] == "BRACE") {
                segment += "|";
                continue;
            }
            if (glob[i1] == "*") {
                if (extended && glob[i1 + 1] == "(") {
                    i1++;
                    groupStack.push("*");
                    segment += "(?:";
                } else {
                    const prevChar = glob[i1 - 1];
                    let numStars = 1;
                    while(glob[i1 + 1] == "*"){
                        i1++;
                        numStars++;
                    }
                    const nextChar = glob[i1 + 1];
                    if (globstarOption && numStars == 2 && [
                        ...seps,
                        undefined
                    ].includes(prevChar) && [
                        ...seps,
                        undefined
                    ].includes(nextChar)) {
                        segment += globstar;
                        endsWithSep = true;
                    } else {
                        segment += wildcard;
                    }
                }
                continue;
            }
            segment += regExpEscapeChars1.includes(glob[i1]) ? `\\${glob[i1]}` : glob[i1];
        }
        if (groupStack.length > 0 || inRange || inEscape) {
            segment = "";
            for (const c of glob.slice(j, i1)){
                segment += regExpEscapeChars1.includes(c) ? `\\${c}` : c;
                endsWithSep = false;
            }
        }
        regExpString += segment;
        if (!endsWithSep) {
            regExpString += i1 < glob.length ? sep : sepMaybe;
            endsWithSep = true;
        }
        while(seps.includes(glob[i1]))i1++;
        if (!(i1 > j)) {
            throw new Error("Assertion failure: i > j (potential infinite loop)");
        }
        j = i1;
    }
    regExpString = `^${regExpString}$`;
    return new RegExp(regExpString, caseInsensitive ? "i" : "");
}
function isGlob1(str) {
    const chars = {
        "{": "}",
        "(": ")",
        "[": "]"
    };
    const regex = /\\(.)|(^!|\*|\?|[\].+)]\?|\[[^\\\]]+\]|\{[^\\}]+\}|\(\?[:!=][^\\)]+\)|\([^|]+\|[^\\)]+\))/;
    if (str === "") {
        return false;
    }
    let match;
    while(match = regex.exec(str)){
        if (match[2]) return true;
        let idx = match.index + match[0].length;
        const open = match[1];
        const close = open ? chars[open] : null;
        if (open && close) {
            const n = str.indexOf(close, idx);
            if (n !== -1) {
                idx = n + 1;
            }
        }
        str = str.slice(idx);
    }
    return false;
}
const _resources = {};
function setYamlIntelligenceResources(resources) {
    for (const [key, value] of Object.entries(resources)){
        _resources[key] = value;
    }
}
function exportYamlIntelligenceResources(pretty = false) {
    if (pretty) {
        return JSON.stringify(_resources, null, 2);
    } else {
        return JSON.stringify(_resources);
    }
}
function getYamlIntelligenceResource(filename) {
    if (_resources[filename] === undefined) {
        throw new InternalError(`getYamlIntelligenceResource called with missing resource ${filename}`);
    }
    return _resources[filename];
}
function expandResourceGlob(glob) {
    return Object.keys(_resources).filter((key)=>key.match(globToRegExp3(glob))).map((key)=>[
            key,
            getYamlIntelligenceResource(key)
        ]);
}
function fromEntries(iterable) {
    return [
        ...iterable
    ].reduce((obj, [key, val])=>{
        obj[key] = val;
        return obj;
    }, {});
}
let _parser;
function setTreeSitter(parser) {
    _parser = parser;
}
function getTreeSitterSync() {
    if (_parser === undefined) {
        throw new Error("tree-sitter uninitialized");
    }
    return _parser;
}
function isNothing(subject) {
    return typeof subject === "undefined" || subject === null;
}
function isObject2(subject) {
    return typeof subject === "object" && subject !== null;
}
function toArray(sequence) {
    if (Array.isArray(sequence)) return sequence;
    else if (isNothing(sequence)) return [];
    return [
        sequence
    ];
}
function extend(target, source) {
    var index, length, key, sourceKeys;
    if (source) {
        sourceKeys = Object.keys(source);
        for(index = 0, length = sourceKeys.length; index < length; index += 1){
            key = sourceKeys[index];
            target[key] = source[key];
        }
    }
    return target;
}
function repeat1(string, count) {
    var result1 = "", cycle;
    for(cycle = 0; cycle < count; cycle += 1){
        result1 += string;
    }
    return result1;
}
function isNegativeZero1(number) {
    return number === 0 && Number.NEGATIVE_INFINITY === 1 / number;
}
var isNothing_1 = isNothing;
var isObject_1 = isObject2;
var toArray_1 = toArray;
var repeat_1 = repeat1;
var isNegativeZero_1 = isNegativeZero1;
var extend_1 = extend;
var common1 = {
    isNothing: isNothing_1,
    isObject: isObject_1,
    toArray: toArray_1,
    repeat: repeat_1,
    isNegativeZero: isNegativeZero_1,
    extend: extend_1
};
function formatError(exception2, compact) {
    var where = "", message = exception2.reason || "(unknown reason)";
    if (!exception2.mark) return message;
    if (exception2.mark.name) {
        where += 'in "' + exception2.mark.name + '" ';
    }
    where += "(" + (exception2.mark.line + 1) + ":" + (exception2.mark.column + 1) + ")";
    if (!compact && exception2.mark.snippet) {
        where += "\n\n" + exception2.mark.snippet;
    }
    return message + " " + where;
}
function YAMLException$1(reason, mark) {
    Error.call(this);
    this.name = "YAMLException";
    this.reason = reason;
    this.mark = mark;
    this.message = formatError(this, false);
    if (Error.captureStackTrace) {
        Error.captureStackTrace(this, this.constructor);
    } else {
        this.stack = new Error().stack || "";
    }
}
YAMLException$1.prototype = Object.create(Error.prototype);
YAMLException$1.prototype.constructor = YAMLException$1;
YAMLException$1.prototype.toString = function toString(compact) {
    return this.name + ": " + formatError(this, compact);
};
var exception = YAMLException$1;
function getLine(buffer, lineStart, lineEnd, position, maxLineLength) {
    var head = "";
    var tail = "";
    var maxHalfLength = Math.floor(maxLineLength / 2) - 1;
    if (position - lineStart > maxHalfLength) {
        head = " ... ";
        lineStart = position - maxHalfLength + head.length;
    }
    if (lineEnd - position > maxHalfLength) {
        tail = " ...";
        lineEnd = position + maxHalfLength - tail.length;
    }
    return {
        str: head + buffer.slice(lineStart, lineEnd).replace(/\t/g, "\u2192") + tail,
        pos: position - lineStart + head.length
    };
}
function padStart(string, max) {
    return common1.repeat(" ", max - string.length) + string;
}
function makeSnippet(mark, options) {
    options = Object.create(options || null);
    if (!mark.buffer) return null;
    if (!options.maxLength) options.maxLength = 79;
    if (typeof options.indent !== "number") options.indent = 1;
    if (typeof options.linesBefore !== "number") options.linesBefore = 3;
    if (typeof options.linesAfter !== "number") options.linesAfter = 2;
    var re = /\r?\n|\r|\0/g;
    var lineStarts = [
        0
    ];
    var lineEnds = [];
    var match;
    var foundLineNo = -1;
    while(match = re.exec(mark.buffer)){
        lineEnds.push(match.index);
        lineStarts.push(match.index + match[0].length);
        if (mark.position <= match.index && foundLineNo < 0) {
            foundLineNo = lineStarts.length - 2;
        }
    }
    if (foundLineNo < 0) foundLineNo = lineStarts.length - 1;
    var result1 = "", i1, line;
    var lineNoLength = Math.min(mark.line + options.linesAfter, lineEnds.length).toString().length;
    var maxLineLength = options.maxLength - (options.indent + lineNoLength + 3);
    for(i1 = 1; i1 <= options.linesBefore; i1++){
        if (foundLineNo - i1 < 0) break;
        line = getLine(mark.buffer, lineStarts[foundLineNo - i1], lineEnds[foundLineNo - i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo - i1]), maxLineLength);
        result1 = common1.repeat(" ", options.indent) + padStart((mark.line - i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n" + result1;
    }
    line = getLine(mark.buffer, lineStarts[foundLineNo], lineEnds[foundLineNo], mark.position, maxLineLength);
    result1 += common1.repeat(" ", options.indent) + padStart((mark.line + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    result1 += common1.repeat("-", options.indent + lineNoLength + 3 + line.pos) + "^\n";
    for(i1 = 1; i1 <= options.linesAfter; i1++){
        if (foundLineNo + i1 >= lineEnds.length) break;
        line = getLine(mark.buffer, lineStarts[foundLineNo + i1], lineEnds[foundLineNo + i1], mark.position - (lineStarts[foundLineNo] - lineStarts[foundLineNo + i1]), maxLineLength);
        result1 += common1.repeat(" ", options.indent) + padStart((mark.line + i1 + 1).toString(), lineNoLength) + " | " + line.str + "\n";
    }
    return result1.replace(/\n$/, "");
}
var snippet = makeSnippet;
var TYPE_CONSTRUCTOR_OPTIONS = [
    "kind",
    "multi",
    "resolve",
    "construct",
    "instanceOf",
    "predicate",
    "represent",
    "representName",
    "defaultStyle",
    "styleAliases"
];
var YAML_NODE_KINDS = [
    "scalar",
    "sequence",
    "mapping"
];
function compileStyleAliases(map2) {
    var result1 = {};
    if (map2 !== null) {
        Object.keys(map2).forEach(function(style) {
            map2[style].forEach(function(alias) {
                result1[String(alias)] = style;
            });
        });
    }
    return result1;
}
function Type$1(tag, options) {
    options = options || {};
    Object.keys(options).forEach(function(name) {
        if (TYPE_CONSTRUCTOR_OPTIONS.indexOf(name) === -1) {
            throw new exception('Unknown option "' + name + '" is met in definition of "' + tag + '" YAML type.');
        }
    });
    this.options = options;
    this.tag = tag;
    this.kind = options["kind"] || null;
    this.resolve = options["resolve"] || function() {
        return true;
    };
    this.construct = options["construct"] || function(data) {
        return data;
    };
    this.instanceOf = options["instanceOf"] || null;
    this.predicate = options["predicate"] || null;
    this.represent = options["represent"] || null;
    this.representName = options["representName"] || null;
    this.defaultStyle = options["defaultStyle"] || null;
    this.multi = options["multi"] || false;
    this.styleAliases = compileStyleAliases(options["styleAliases"] || null);
    if (YAML_NODE_KINDS.indexOf(this.kind) === -1) {
        throw new exception('Unknown kind "' + this.kind + '" is specified for "' + tag + '" YAML type.');
    }
}
var type = Type$1;
function compileList1(schema2, name) {
    var result1 = [];
    schema2[name].forEach(function(currentType) {
        var newIndex = result1.length;
        result1.forEach(function(previousType, previousIndex) {
            if (previousType.tag === currentType.tag && previousType.kind === currentType.kind && previousType.multi === currentType.multi) {
                newIndex = previousIndex;
            }
        });
        result1[newIndex] = currentType;
    });
    return result1;
}
function compileMap1() {
    var result1 = {
        scalar: {},
        sequence: {},
        mapping: {},
        fallback: {},
        multi: {
            scalar: [],
            sequence: [],
            mapping: [],
            fallback: []
        }
    }, index, length;
    function collectType(type2) {
        if (type2.multi) {
            result1.multi[type2.kind].push(type2);
            result1.multi["fallback"].push(type2);
        } else {
            result1[type2.kind][type2.tag] = result1["fallback"][type2.tag] = type2;
        }
    }
    for(index = 0, length = arguments.length; index < length; index += 1){
        arguments[index].forEach(collectType);
    }
    return result1;
}
function Schema$1(definition) {
    return this.extend(definition);
}
Schema$1.prototype.extend = function extend2(definition) {
    var implicit = [];
    var explicit = [];
    if (definition instanceof type) {
        explicit.push(definition);
    } else if (Array.isArray(definition)) {
        explicit = explicit.concat(definition);
    } else if (definition && (Array.isArray(definition.implicit) || Array.isArray(definition.explicit))) {
        if (definition.implicit) implicit = implicit.concat(definition.implicit);
        if (definition.explicit) explicit = explicit.concat(definition.explicit);
    } else {
        throw new exception("Schema.extend argument should be a Type, [ Type ], or a schema definition ({ implicit: [...], explicit: [...] })");
    }
    implicit.forEach(function(type$1) {
        if (!(type$1 instanceof type)) {
            throw new exception("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
        if (type$1.loadKind && type$1.loadKind !== "scalar") {
            throw new exception("There is a non-scalar type in the implicit list of a schema. Implicit resolving of such types is not supported.");
        }
        if (type$1.multi) {
            throw new exception("There is a multi type in the implicit list of a schema. Multi tags can only be listed as explicit.");
        }
    });
    explicit.forEach(function(type$1) {
        if (!(type$1 instanceof type)) {
            throw new exception("Specified list of YAML types (or a single Type object) contains a non-Type object.");
        }
    });
    var result1 = Object.create(Schema$1.prototype);
    result1.implicit = (this.implicit || []).concat(implicit);
    result1.explicit = (this.explicit || []).concat(explicit);
    result1.compiledImplicit = compileList1(result1, "implicit");
    result1.compiledExplicit = compileList1(result1, "explicit");
    result1.compiledTypeMap = compileMap1(result1.compiledImplicit, result1.compiledExplicit);
    return result1;
};
var schema = Schema$1;
var str1 = new type("tag:yaml.org,2002:str", {
    kind: "scalar",
    construct: function(data) {
        return data !== null ? data : "";
    }
});
var seq1 = new type("tag:yaml.org,2002:seq", {
    kind: "sequence",
    construct: function(data) {
        return data !== null ? data : [];
    }
});
var map1 = new type("tag:yaml.org,2002:map", {
    kind: "mapping",
    construct: function(data) {
        return data !== null ? data : {};
    }
});
var failsafe1 = new schema({
    explicit: [
        str1,
        seq1,
        map1
    ]
});
function resolveYamlNull1(data) {
    if (data === null) return true;
    var max = data.length;
    return max === 1 && data === "~" || max === 4 && (data === "null" || data === "Null" || data === "NULL");
}
function constructYamlNull1() {
    return null;
}
function isNull1(object) {
    return object === null;
}
var _null = new type("tag:yaml.org,2002:null", {
    kind: "scalar",
    resolve: resolveYamlNull1,
    construct: constructYamlNull1,
    predicate: isNull1,
    represent: {
        canonical: function() {
            return "~";
        },
        lowercase: function() {
            return "null";
        },
        uppercase: function() {
            return "NULL";
        },
        camelcase: function() {
            return "Null";
        },
        empty: function() {
            return "";
        }
    },
    defaultStyle: "lowercase"
});
function resolveYamlBoolean1(data) {
    if (data === null) return false;
    var max = data.length;
    return max === 4 && (data === "true" || data === "True" || data === "TRUE") || max === 5 && (data === "false" || data === "False" || data === "FALSE");
}
function constructYamlBoolean1(data) {
    return data === "true" || data === "True" || data === "TRUE";
}
function isBoolean1(object) {
    return Object.prototype.toString.call(object) === "[object Boolean]";
}
var bool1 = new type("tag:yaml.org,2002:bool", {
    kind: "scalar",
    resolve: resolveYamlBoolean1,
    construct: constructYamlBoolean1,
    predicate: isBoolean1,
    represent: {
        lowercase: function(object) {
            return object ? "true" : "false";
        },
        uppercase: function(object) {
            return object ? "TRUE" : "FALSE";
        },
        camelcase: function(object) {
            return object ? "True" : "False";
        }
    },
    defaultStyle: "lowercase"
});
function isHexCode1(c) {
    return 48 <= c && c <= 57 || 65 <= c && c <= 70 || 97 <= c && c <= 102;
}
function isOctCode1(c) {
    return 48 <= c && c <= 55;
}
function isDecCode1(c) {
    return 48 <= c && c <= 57;
}
function resolveYamlInteger1(data) {
    if (data === null) return false;
    var max = data.length, index = 0, hasDigits = false, ch;
    if (!max) return false;
    ch = data[index];
    if (ch === "-" || ch === "+") {
        ch = data[++index];
    }
    if (ch === "0") {
        if (index + 1 === max) return true;
        ch = data[++index];
        if (ch === "b") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (ch !== "0" && ch !== "1") return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "x") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isHexCode1(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
        if (ch === "o") {
            index++;
            for(; index < max; index++){
                ch = data[index];
                if (ch === "_") continue;
                if (!isOctCode1(data.charCodeAt(index))) return false;
                hasDigits = true;
            }
            return hasDigits && ch !== "_";
        }
    }
    if (ch === "_") return false;
    for(; index < max; index++){
        ch = data[index];
        if (ch === "_") continue;
        if (!isDecCode1(data.charCodeAt(index))) {
            return false;
        }
        hasDigits = true;
    }
    if (!hasDigits || ch === "_") return false;
    return true;
}
function constructYamlInteger1(data) {
    var value = data, sign = 1, ch;
    if (value.indexOf("_") !== -1) {
        value = value.replace(/_/g, "");
    }
    ch = value[0];
    if (ch === "-" || ch === "+") {
        if (ch === "-") sign = -1;
        value = value.slice(1);
        ch = value[0];
    }
    if (value === "0") return 0;
    if (ch === "0") {
        if (value[1] === "b") return sign * parseInt(value.slice(2), 2);
        if (value[1] === "x") return sign * parseInt(value.slice(2), 16);
        if (value[1] === "o") return sign * parseInt(value.slice(2), 8);
    }
    return sign * parseInt(value, 10);
}
function isInteger1(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && object % 1 === 0 && !common1.isNegativeZero(object);
}
var __int1 = new type("tag:yaml.org,2002:int", {
    kind: "scalar",
    resolve: resolveYamlInteger1,
    construct: constructYamlInteger1,
    predicate: isInteger1,
    represent: {
        binary: function(obj) {
            return obj >= 0 ? "0b" + obj.toString(2) : "-0b" + obj.toString(2).slice(1);
        },
        octal: function(obj) {
            return obj >= 0 ? "0o" + obj.toString(8) : "-0o" + obj.toString(8).slice(1);
        },
        decimal: function(obj) {
            return obj.toString(10);
        },
        hexadecimal: function(obj) {
            return obj >= 0 ? "0x" + obj.toString(16).toUpperCase() : "-0x" + obj.toString(16).toUpperCase().slice(1);
        }
    },
    defaultStyle: "decimal",
    styleAliases: {
        binary: [
            2,
            "bin"
        ],
        octal: [
            8,
            "oct"
        ],
        decimal: [
            10,
            "dec"
        ],
        hexadecimal: [
            16,
            "hex"
        ]
    }
});
var YAML_FLOAT_PATTERN1 = new RegExp("^(?:[-+]?(?:[0-9][0-9_]*)(?:\\.[0-9_]*)?(?:[eE][-+]?[0-9]+)?|\\.[0-9_]+(?:[eE][-+]?[0-9]+)?|[-+]?\\.(?:inf|Inf|INF)|\\.(?:nan|NaN|NAN))$");
function resolveYamlFloat1(data) {
    if (data === null) return false;
    if (!YAML_FLOAT_PATTERN1.test(data) || data[data.length - 1] === "_") {
        return false;
    }
    return true;
}
function constructYamlFloat1(data) {
    var value, sign;
    value = data.replace(/_/g, "").toLowerCase();
    sign = value[0] === "-" ? -1 : 1;
    if ("+-".indexOf(value[0]) >= 0) {
        value = value.slice(1);
    }
    if (value === ".inf") {
        return sign === 1 ? Number.POSITIVE_INFINITY : Number.NEGATIVE_INFINITY;
    } else if (value === ".nan") {
        return NaN;
    }
    return sign * parseFloat(value, 10);
}
var SCIENTIFIC_WITHOUT_DOT1 = /^[-+]?[0-9]+e/;
function representYamlFloat1(object, style) {
    var res;
    if (isNaN(object)) {
        switch(style){
            case "lowercase":
                return ".nan";
            case "uppercase":
                return ".NAN";
            case "camelcase":
                return ".NaN";
        }
    } else if (Number.POSITIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return ".inf";
            case "uppercase":
                return ".INF";
            case "camelcase":
                return ".Inf";
        }
    } else if (Number.NEGATIVE_INFINITY === object) {
        switch(style){
            case "lowercase":
                return "-.inf";
            case "uppercase":
                return "-.INF";
            case "camelcase":
                return "-.Inf";
        }
    } else if (common1.isNegativeZero(object)) {
        return "-0.0";
    }
    res = object.toString(10);
    return SCIENTIFIC_WITHOUT_DOT1.test(res) ? res.replace("e", ".e") : res;
}
function isFloat1(object) {
    return Object.prototype.toString.call(object) === "[object Number]" && (object % 1 !== 0 || common1.isNegativeZero(object));
}
var __float1 = new type("tag:yaml.org,2002:float", {
    kind: "scalar",
    resolve: resolveYamlFloat1,
    construct: constructYamlFloat1,
    predicate: isFloat1,
    represent: representYamlFloat1,
    defaultStyle: "lowercase"
});
var json1 = failsafe1.extend({
    implicit: [
        _null,
        bool1,
        __int1,
        __float1
    ]
});
var core1 = json1;
var YAML_DATE_REGEXP1 = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9])-([0-9][0-9])$");
var YAML_TIMESTAMP_REGEXP1 = new RegExp("^([0-9][0-9][0-9][0-9])-([0-9][0-9]?)-([0-9][0-9]?)(?:[Tt]|[ \\t]+)([0-9][0-9]?):([0-9][0-9]):([0-9][0-9])(?:\\.([0-9]*))?(?:[ \\t]*(Z|([-+])([0-9][0-9]?)(?::([0-9][0-9]))?))?$");
function resolveYamlTimestamp1(data) {
    if (data === null) return false;
    if (YAML_DATE_REGEXP1.exec(data) !== null) return true;
    if (YAML_TIMESTAMP_REGEXP1.exec(data) !== null) return true;
    return false;
}
function constructYamlTimestamp1(data) {
    var match, year, month, day, hour, minute, second, fraction = 0, delta = null, tz_hour, tz_minute, date;
    match = YAML_DATE_REGEXP1.exec(data);
    if (match === null) match = YAML_TIMESTAMP_REGEXP1.exec(data);
    if (match === null) throw new Error("Date resolve error");
    year = +match[1];
    month = +match[2] - 1;
    day = +match[3];
    if (!match[4]) {
        return new Date(Date.UTC(year, month, day));
    }
    hour = +match[4];
    minute = +match[5];
    second = +match[6];
    if (match[7]) {
        fraction = match[7].slice(0, 3);
        while(fraction.length < 3){
            fraction += "0";
        }
        fraction = +fraction;
    }
    if (match[9]) {
        tz_hour = +match[10];
        tz_minute = +(match[11] || 0);
        delta = (tz_hour * 60 + tz_minute) * 6e4;
        if (match[9] === "-") delta = -delta;
    }
    date = new Date(Date.UTC(year, month, day, hour, minute, second, fraction));
    if (delta) date.setTime(date.getTime() - delta);
    return date;
}
function representYamlTimestamp1(object) {
    return object.toISOString();
}
var timestamp1 = new type("tag:yaml.org,2002:timestamp", {
    kind: "scalar",
    resolve: resolveYamlTimestamp1,
    construct: constructYamlTimestamp1,
    instanceOf: Date,
    represent: representYamlTimestamp1
});
function resolveYamlMerge1(data) {
    return data === "<<" || data === null;
}
var merge1 = new type("tag:yaml.org,2002:merge", {
    kind: "scalar",
    resolve: resolveYamlMerge1
});
var BASE64_MAP1 = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=\n\r";
function resolveYamlBinary1(data) {
    if (data === null) return false;
    var code, idx, bitlen = 0, max = data.length, map2 = BASE64_MAP1;
    for(idx = 0; idx < max; idx++){
        code = map2.indexOf(data.charAt(idx));
        if (code > 64) continue;
        if (code < 0) return false;
        bitlen += 6;
    }
    return bitlen % 8 === 0;
}
function constructYamlBinary1(data) {
    var idx, tailbits, input = data.replace(/[\r\n=]/g, ""), max = input.length, map2 = BASE64_MAP1, bits = 0, result1 = [];
    for(idx = 0; idx < max; idx++){
        if (idx % 4 === 0 && idx) {
            result1.push(bits >> 16 & 255);
            result1.push(bits >> 8 & 255);
            result1.push(bits & 255);
        }
        bits = bits << 6 | map2.indexOf(input.charAt(idx));
    }
    tailbits = max % 4 * 6;
    if (tailbits === 0) {
        result1.push(bits >> 16 & 255);
        result1.push(bits >> 8 & 255);
        result1.push(bits & 255);
    } else if (tailbits === 18) {
        result1.push(bits >> 10 & 255);
        result1.push(bits >> 2 & 255);
    } else if (tailbits === 12) {
        result1.push(bits >> 4 & 255);
    }
    return new Uint8Array(result1);
}
function representYamlBinary1(object) {
    var result1 = "", bits = 0, idx, tail, max = object.length, map2 = BASE64_MAP1;
    for(idx = 0; idx < max; idx++){
        if (idx % 3 === 0 && idx) {
            result1 += map2[bits >> 18 & 63];
            result1 += map2[bits >> 12 & 63];
            result1 += map2[bits >> 6 & 63];
            result1 += map2[bits & 63];
        }
        bits = (bits << 8) + object[idx];
    }
    tail = max % 3;
    if (tail === 0) {
        result1 += map2[bits >> 18 & 63];
        result1 += map2[bits >> 12 & 63];
        result1 += map2[bits >> 6 & 63];
        result1 += map2[bits & 63];
    } else if (tail === 2) {
        result1 += map2[bits >> 10 & 63];
        result1 += map2[bits >> 4 & 63];
        result1 += map2[bits << 2 & 63];
        result1 += map2[64];
    } else if (tail === 1) {
        result1 += map2[bits >> 2 & 63];
        result1 += map2[bits << 4 & 63];
        result1 += map2[64];
        result1 += map2[64];
    }
    return result1;
}
function isBinary1(obj) {
    return Object.prototype.toString.call(obj) === "[object Uint8Array]";
}
var binary1 = new type("tag:yaml.org,2002:binary", {
    kind: "scalar",
    resolve: resolveYamlBinary1,
    construct: constructYamlBinary1,
    predicate: isBinary1,
    represent: representYamlBinary1
});
var _hasOwnProperty$3 = Object.prototype.hasOwnProperty;
var _toString$2 = Object.prototype.toString;
function resolveYamlOmap1(data) {
    if (data === null) return true;
    var objectKeys = [], index, length, pair, pairKey, pairHasKey, object = data;
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        pairHasKey = false;
        if (_toString$2.call(pair) !== "[object Object]") return false;
        for(pairKey in pair){
            if (_hasOwnProperty$3.call(pair, pairKey)) {
                if (!pairHasKey) pairHasKey = true;
                else return false;
            }
        }
        if (!pairHasKey) return false;
        if (objectKeys.indexOf(pairKey) === -1) objectKeys.push(pairKey);
        else return false;
    }
    return true;
}
function constructYamlOmap1(data) {
    return data !== null ? data : [];
}
var omap1 = new type("tag:yaml.org,2002:omap", {
    kind: "sequence",
    resolve: resolveYamlOmap1,
    construct: constructYamlOmap1
});
var _toString$1 = Object.prototype.toString;
function resolveYamlPairs1(data) {
    if (data === null) return true;
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        if (_toString$1.call(pair) !== "[object Object]") return false;
        keys = Object.keys(pair);
        if (keys.length !== 1) return false;
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return true;
}
function constructYamlPairs1(data) {
    if (data === null) return [];
    var index, length, pair, keys, result1, object = data;
    result1 = new Array(object.length);
    for(index = 0, length = object.length; index < length; index += 1){
        pair = object[index];
        keys = Object.keys(pair);
        result1[index] = [
            keys[0],
            pair[keys[0]]
        ];
    }
    return result1;
}
var pairs1 = new type("tag:yaml.org,2002:pairs", {
    kind: "sequence",
    resolve: resolveYamlPairs1,
    construct: constructYamlPairs1
});
var _hasOwnProperty$2 = Object.prototype.hasOwnProperty;
function resolveYamlSet1(data) {
    if (data === null) return true;
    var key, object = data;
    for(key in object){
        if (_hasOwnProperty$2.call(object, key)) {
            if (object[key] !== null) return false;
        }
    }
    return true;
}
function constructYamlSet1(data) {
    return data !== null ? data : {};
}
var set1 = new type("tag:yaml.org,2002:set", {
    kind: "mapping",
    resolve: resolveYamlSet1,
    construct: constructYamlSet1
});
var _default = core1.extend({
    implicit: [
        timestamp1,
        merge1
    ],
    explicit: [
        binary1,
        omap1,
        pairs1,
        set1
    ]
});
var _hasOwnProperty$1 = Object.prototype.hasOwnProperty;
var CONTEXT_FLOW_IN = 1;
var CONTEXT_FLOW_OUT = 2;
var CONTEXT_BLOCK_IN1 = 3;
var CONTEXT_BLOCK_OUT1 = 4;
var CHOMPING_CLIP = 1;
var CHOMPING_STRIP1 = 2;
var CHOMPING_KEEP1 = 3;
var PATTERN_NON_PRINTABLE1 = /[\x00-\x08\x0B\x0C\x0E-\x1F\x7F-\x84\x86-\x9F\uFFFE\uFFFF]|[\uD800-\uDBFF](?![\uDC00-\uDFFF])|(?:[^\uD800-\uDBFF]|^)[\uDC00-\uDFFF]/;
var PATTERN_NON_ASCII_LINE_BREAKS1 = /[\x85\u2028\u2029]/;
var PATTERN_FLOW_INDICATORS1 = /[,\[\]\{\}]/;
var PATTERN_TAG_HANDLE1 = /^(?:!|!!|![a-z\-]+!)$/i;
var PATTERN_TAG_URI1 = /^(?:!|[^,\[\]\{\}])(?:%[0-9a-f]{2}|[0-9a-z\-#;\/\?:@&=\+\$,_\.!~\*'\(\)\[\]])*$/i;
function _class1(obj) {
    return Object.prototype.toString.call(obj);
}
function is_EOL(c) {
    return c === 10 || c === 13;
}
function is_WHITE_SPACE(c) {
    return c === 9 || c === 32;
}
function is_WS_OR_EOL(c) {
    return c === 9 || c === 32 || c === 10 || c === 13;
}
function is_FLOW_INDICATOR(c) {
    return c === 44 || c === 91 || c === 93 || c === 123 || c === 125;
}
function fromHexCode1(c) {
    var lc;
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    lc = c | 32;
    if (97 <= lc && lc <= 102) {
        return lc - 97 + 10;
    }
    return -1;
}
function escapedHexLen1(c) {
    if (c === 120) {
        return 2;
    }
    if (c === 117) {
        return 4;
    }
    if (c === 85) {
        return 8;
    }
    return 0;
}
function fromDecimalCode1(c) {
    if (48 <= c && c <= 57) {
        return c - 48;
    }
    return -1;
}
function simpleEscapeSequence1(c) {
    return c === 48 ? "\0" : c === 97 ? "\x07" : c === 98 ? "\b" : c === 116 ? "	" : c === 9 ? "	" : c === 110 ? "\n" : c === 118 ? "\v" : c === 102 ? "\f" : c === 114 ? "\r" : c === 101 ? "" : c === 32 ? " " : c === 34 ? '"' : c === 47 ? "/" : c === 92 ? "\\" : c === 78 ? "\x85" : c === 95 ? "\xA0" : c === 76 ? "\u2028" : c === 80 ? "\u2029" : "";
}
function charFromCodepoint1(c) {
    if (c <= 65535) {
        return String.fromCharCode(c);
    }
    return String.fromCharCode((c - 65536 >> 10) + 55296, (c - 65536 & 1023) + 56320);
}
var simpleEscapeCheck1 = new Array(256);
var simpleEscapeMap1 = new Array(256);
for(var i1 = 0; i1 < 256; i1++){
    simpleEscapeCheck1[i1] = simpleEscapeSequence1(i1) ? 1 : 0;
    simpleEscapeMap1[i1] = simpleEscapeSequence1(i1);
}
function State$1(input, options) {
    this.input = input;
    this.filename = options["filename"] || null;
    this.schema = options["schema"] || _default;
    this.onWarning = options["onWarning"] || null;
    this.legacy = options["legacy"] || false;
    this.json = options["json"] || false;
    this.listener = options["listener"] || null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.typeMap = this.schema.compiledTypeMap;
    this.length = input.length;
    this.position = 0;
    this.line = 0;
    this.lineStart = 0;
    this.lineIndent = 0;
    this.firstTabInLine = -1;
    this.documents = [];
}
function generateError1(state, message) {
    var mark = {
        name: state.filename,
        buffer: state.input.slice(0, -1),
        position: state.position,
        line: state.line,
        column: state.position - state.lineStart
    };
    mark.snippet = snippet(mark);
    return new exception(message, mark);
}
function throwError1(state, message) {
    throw generateError1(state, message);
}
function throwWarning1(state, message) {
    if (state.onWarning) {
        state.onWarning.call(null, generateError1(state, message));
    }
}
var directiveHandlers1 = {
    YAML: function handleYamlDirective(state, name, args) {
        var match, major, minor;
        if (state.version !== null) {
            throwError1(state, "duplication of %YAML directive");
        }
        if (args.length !== 1) {
            throwError1(state, "YAML directive accepts exactly one argument");
        }
        match = /^([0-9]+)\.([0-9]+)$/.exec(args[0]);
        if (match === null) {
            throwError1(state, "ill-formed argument of the YAML directive");
        }
        major = parseInt(match[1], 10);
        minor = parseInt(match[2], 10);
        if (major !== 1) {
            throwError1(state, "unacceptable YAML version of the document");
        }
        state.version = args[0];
        state.checkLineBreaks = minor < 2;
        if (minor !== 1 && minor !== 2) {
            throwWarning1(state, "unsupported YAML version of the document");
        }
    },
    TAG: function handleTagDirective(state, name, args) {
        var handle, prefix;
        if (args.length !== 2) {
            throwError1(state, "TAG directive accepts exactly two arguments");
        }
        handle = args[0];
        prefix = args[1];
        if (!PATTERN_TAG_HANDLE1.test(handle)) {
            throwError1(state, "ill-formed tag handle (first argument) of the TAG directive");
        }
        if (_hasOwnProperty$1.call(state.tagMap, handle)) {
            throwError1(state, 'there is a previously declared suffix for "' + handle + '" tag handle');
        }
        if (!PATTERN_TAG_URI1.test(prefix)) {
            throwError1(state, "ill-formed tag prefix (second argument) of the TAG directive");
        }
        try {
            prefix = decodeURIComponent(prefix);
        } catch (err) {
            throwError1(state, "tag prefix is malformed: " + prefix);
        }
        state.tagMap[handle] = prefix;
    }
};
function captureSegment1(state, start, end, checkJson) {
    var _position, _length, _character, _result;
    if (start < end) {
        _result = state.input.slice(start, end);
        if (checkJson) {
            for(_position = 0, _length = _result.length; _position < _length; _position += 1){
                _character = _result.charCodeAt(_position);
                if (!(_character === 9 || 32 <= _character && _character <= 1114111)) {
                    throwError1(state, "expected valid JSON character");
                }
            }
        } else if (PATTERN_NON_PRINTABLE1.test(_result)) {
            throwError1(state, "the stream contains non-printable characters");
        }
        state.result += _result;
    }
}
function mergeMappings1(state, destination, source, overridableKeys) {
    var sourceKeys, key, index, quantity;
    if (!common1.isObject(source)) {
        throwError1(state, "cannot merge mappings; the provided source object is unacceptable");
    }
    sourceKeys = Object.keys(source);
    for(index = 0, quantity = sourceKeys.length; index < quantity; index += 1){
        key = sourceKeys[index];
        if (!_hasOwnProperty$1.call(destination, key)) {
            destination[key] = source[key];
            overridableKeys[key] = true;
        }
    }
}
function storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, startLine, startLineStart, startPos) {
    var index, quantity;
    if (Array.isArray(keyNode)) {
        keyNode = Array.prototype.slice.call(keyNode);
        for(index = 0, quantity = keyNode.length; index < quantity; index += 1){
            if (Array.isArray(keyNode[index])) {
                throwError1(state, "nested arrays are not supported inside keys");
            }
            if (typeof keyNode === "object" && _class1(keyNode[index]) === "[object Object]") {
                keyNode[index] = "[object Object]";
            }
        }
    }
    if (typeof keyNode === "object" && _class1(keyNode) === "[object Object]") {
        keyNode = "[object Object]";
    }
    keyNode = String(keyNode);
    if (_result === null) {
        _result = {};
    }
    if (keyTag === "tag:yaml.org,2002:merge") {
        if (Array.isArray(valueNode)) {
            for(index = 0, quantity = valueNode.length; index < quantity; index += 1){
                mergeMappings1(state, _result, valueNode[index], overridableKeys);
            }
        } else {
            mergeMappings1(state, _result, valueNode, overridableKeys);
        }
    } else {
        if (!state.json && !_hasOwnProperty$1.call(overridableKeys, keyNode) && _hasOwnProperty$1.call(_result, keyNode)) {
            state.line = startLine || state.line;
            state.lineStart = startLineStart || state.lineStart;
            state.position = startPos || state.position;
            throwError1(state, "duplicated mapping key");
        }
        if (keyNode === "__proto__") {
            Object.defineProperty(_result, keyNode, {
                configurable: true,
                enumerable: true,
                writable: true,
                value: valueNode
            });
        } else {
            _result[keyNode] = valueNode;
        }
        delete overridableKeys[keyNode];
    }
    return _result;
}
function readLineBreak1(state) {
    var ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 10) {
        state.position++;
    } else if (ch === 13) {
        state.position++;
        if (state.input.charCodeAt(state.position) === 10) {
            state.position++;
        }
    } else {
        throwError1(state, "a line break is expected");
    }
    state.line += 1;
    state.lineStart = state.position;
    state.firstTabInLine = -1;
}
function skipSeparationSpace1(state, allowComments, checkIndent) {
    var lineBreaks = 0, ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        while(is_WHITE_SPACE(ch)){
            if (ch === 9 && state.firstTabInLine === -1) {
                state.firstTabInLine = state.position;
            }
            ch = state.input.charCodeAt(++state.position);
        }
        if (allowComments && ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (ch !== 10 && ch !== 13 && ch !== 0)
        }
        if (is_EOL(ch)) {
            readLineBreak1(state);
            ch = state.input.charCodeAt(state.position);
            lineBreaks++;
            state.lineIndent = 0;
            while(ch === 32){
                state.lineIndent++;
                ch = state.input.charCodeAt(++state.position);
            }
        } else {
            break;
        }
    }
    if (checkIndent !== -1 && lineBreaks !== 0 && state.lineIndent < checkIndent) {
        throwWarning1(state, "deficient indentation");
    }
    return lineBreaks;
}
function testDocumentSeparator1(state) {
    var _position = state.position, ch;
    ch = state.input.charCodeAt(_position);
    if ((ch === 45 || ch === 46) && ch === state.input.charCodeAt(_position + 1) && ch === state.input.charCodeAt(_position + 2)) {
        _position += 3;
        ch = state.input.charCodeAt(_position);
        if (ch === 0 || is_WS_OR_EOL(ch)) {
            return true;
        }
    }
    return false;
}
function writeFoldedLines1(state, count) {
    if (count === 1) {
        state.result += " ";
    } else if (count > 1) {
        state.result += common1.repeat("\n", count - 1);
    }
}
function readPlainScalar1(state, nodeIndent, withinFlowCollection) {
    var preceding, following, captureStart, captureEnd, hasPendingContent, _line, _lineStart, _lineIndent, _kind = state.kind, _result = state.result, ch;
    ch = state.input.charCodeAt(state.position);
    if (is_WS_OR_EOL(ch) || is_FLOW_INDICATOR(ch) || ch === 35 || ch === 38 || ch === 42 || ch === 33 || ch === 124 || ch === 62 || ch === 39 || ch === 34 || ch === 37 || ch === 64 || ch === 96) {
        return false;
    }
    if (ch === 63 || ch === 45) {
        following = state.input.charCodeAt(state.position + 1);
        if (is_WS_OR_EOL(following) || withinFlowCollection && is_FLOW_INDICATOR(following)) {
            return false;
        }
    }
    state.kind = "scalar";
    state.result = "";
    captureStart = captureEnd = state.position;
    hasPendingContent = false;
    while(ch !== 0){
        if (ch === 58) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL(following) || withinFlowCollection && is_FLOW_INDICATOR(following)) {
                break;
            }
        } else if (ch === 35) {
            preceding = state.input.charCodeAt(state.position - 1);
            if (is_WS_OR_EOL(preceding)) {
                break;
            }
        } else if (state.position === state.lineStart && testDocumentSeparator1(state) || withinFlowCollection && is_FLOW_INDICATOR(ch)) {
            break;
        } else if (is_EOL(ch)) {
            _line = state.line;
            _lineStart = state.lineStart;
            _lineIndent = state.lineIndent;
            skipSeparationSpace1(state, false, -1);
            if (state.lineIndent >= nodeIndent) {
                hasPendingContent = true;
                ch = state.input.charCodeAt(state.position);
                continue;
            } else {
                state.position = captureEnd;
                state.line = _line;
                state.lineStart = _lineStart;
                state.lineIndent = _lineIndent;
                break;
            }
        }
        if (hasPendingContent) {
            captureSegment1(state, captureStart, captureEnd, false);
            writeFoldedLines1(state, state.line - _line);
            captureStart = captureEnd = state.position;
            hasPendingContent = false;
        }
        if (!is_WHITE_SPACE(ch)) {
            captureEnd = state.position + 1;
        }
        ch = state.input.charCodeAt(++state.position);
    }
    captureSegment1(state, captureStart, captureEnd, false);
    if (state.result) {
        return true;
    }
    state.kind = _kind;
    state.result = _result;
    return false;
}
function readSingleQuotedScalar1(state, nodeIndent) {
    var ch, captureStart, captureEnd;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 39) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 39) {
            captureSegment1(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (ch === 39) {
                captureStart = state.position;
                state.position++;
                captureEnd = state.position;
            } else {
                return true;
            }
        } else if (is_EOL(ch)) {
            captureSegment1(state, captureStart, captureEnd, true);
            writeFoldedLines1(state, skipSeparationSpace1(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator1(state)) {
            throwError1(state, "unexpected end of the document within a single quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError1(state, "unexpected end of the stream within a single quoted scalar");
}
function readDoubleQuotedScalar1(state, nodeIndent) {
    var captureStart, captureEnd, hexLength, hexResult, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 34) {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    state.position++;
    captureStart = captureEnd = state.position;
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        if (ch === 34) {
            captureSegment1(state, captureStart, state.position, true);
            state.position++;
            return true;
        } else if (ch === 92) {
            captureSegment1(state, captureStart, state.position, true);
            ch = state.input.charCodeAt(++state.position);
            if (is_EOL(ch)) {
                skipSeparationSpace1(state, false, nodeIndent);
            } else if (ch < 256 && simpleEscapeCheck1[ch]) {
                state.result += simpleEscapeMap1[ch];
                state.position++;
            } else if ((tmp = escapedHexLen1(ch)) > 0) {
                hexLength = tmp;
                hexResult = 0;
                for(; hexLength > 0; hexLength--){
                    ch = state.input.charCodeAt(++state.position);
                    if ((tmp = fromHexCode1(ch)) >= 0) {
                        hexResult = (hexResult << 4) + tmp;
                    } else {
                        throwError1(state, "expected hexadecimal character");
                    }
                }
                state.result += charFromCodepoint1(hexResult);
                state.position++;
            } else {
                throwError1(state, "unknown escape sequence");
            }
            captureStart = captureEnd = state.position;
        } else if (is_EOL(ch)) {
            captureSegment1(state, captureStart, captureEnd, true);
            writeFoldedLines1(state, skipSeparationSpace1(state, false, nodeIndent));
            captureStart = captureEnd = state.position;
        } else if (state.position === state.lineStart && testDocumentSeparator1(state)) {
            throwError1(state, "unexpected end of the document within a double quoted scalar");
        } else {
            state.position++;
            captureEnd = state.position;
        }
    }
    throwError1(state, "unexpected end of the stream within a double quoted scalar");
}
function readFlowCollection1(state, nodeIndent) {
    var readNext = true, _line, _lineStart, _pos, _tag = state.tag, _result, _anchor = state.anchor, following, terminator, isPair, isExplicitPair, isMapping, overridableKeys = Object.create(null), keyNode, keyTag, valueNode, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 91) {
        terminator = 93;
        isMapping = false;
        _result = [];
    } else if (ch === 123) {
        terminator = 125;
        isMapping = true;
        _result = {};
    } else {
        return false;
    }
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(++state.position);
    while(ch !== 0){
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === terminator) {
            state.position++;
            state.tag = _tag;
            state.anchor = _anchor;
            state.kind = isMapping ? "mapping" : "sequence";
            state.result = _result;
            return true;
        } else if (!readNext) {
            throwError1(state, "missed comma between flow collection entries");
        } else if (ch === 44) {
            throwError1(state, "expected the node content, but found ','");
        }
        keyTag = keyNode = valueNode = null;
        isPair = isExplicitPair = false;
        if (ch === 63) {
            following = state.input.charCodeAt(state.position + 1);
            if (is_WS_OR_EOL(following)) {
                isPair = isExplicitPair = true;
                state.position++;
                skipSeparationSpace1(state, true, nodeIndent);
            }
        }
        _line = state.line;
        _lineStart = state.lineStart;
        _pos = state.position;
        composeNode1(state, nodeIndent, CONTEXT_FLOW_IN, false, true);
        keyTag = state.tag;
        keyNode = state.result;
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if ((isExplicitPair || state.line === _line) && ch === 58) {
            isPair = true;
            ch = state.input.charCodeAt(++state.position);
            skipSeparationSpace1(state, true, nodeIndent);
            composeNode1(state, nodeIndent, CONTEXT_FLOW_IN, false, true);
            valueNode = state.result;
        }
        if (isMapping) {
            storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos);
        } else if (isPair) {
            _result.push(storeMappingPair1(state, null, overridableKeys, keyTag, keyNode, valueNode, _line, _lineStart, _pos));
        } else {
            _result.push(keyNode);
        }
        skipSeparationSpace1(state, true, nodeIndent);
        ch = state.input.charCodeAt(state.position);
        if (ch === 44) {
            readNext = true;
            ch = state.input.charCodeAt(++state.position);
        } else {
            readNext = false;
        }
    }
    throwError1(state, "unexpected end of the stream within a flow collection");
}
function readBlockScalar1(state, nodeIndent) {
    var captureStart, folding, chomping = CHOMPING_CLIP, didReadContent = false, detectedIndent = false, textIndent = nodeIndent, emptyLines = 0, atMoreIndented = false, tmp, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch === 124) {
        folding = false;
    } else if (ch === 62) {
        folding = true;
    } else {
        return false;
    }
    state.kind = "scalar";
    state.result = "";
    while(ch !== 0){
        ch = state.input.charCodeAt(++state.position);
        if (ch === 43 || ch === 45) {
            if (CHOMPING_CLIP === chomping) {
                chomping = ch === 43 ? CHOMPING_KEEP1 : CHOMPING_STRIP1;
            } else {
                throwError1(state, "repeat of a chomping mode identifier");
            }
        } else if ((tmp = fromDecimalCode1(ch)) >= 0) {
            if (tmp === 0) {
                throwError1(state, "bad explicit indentation width of a block scalar; it cannot be less than one");
            } else if (!detectedIndent) {
                textIndent = nodeIndent + tmp - 1;
                detectedIndent = true;
            } else {
                throwError1(state, "repeat of an indentation width identifier");
            }
        } else {
            break;
        }
    }
    if (is_WHITE_SPACE(ch)) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (is_WHITE_SPACE(ch))
        if (ch === 35) {
            do {
                ch = state.input.charCodeAt(++state.position);
            }while (!is_EOL(ch) && ch !== 0)
        }
    }
    while(ch !== 0){
        readLineBreak1(state);
        state.lineIndent = 0;
        ch = state.input.charCodeAt(state.position);
        while((!detectedIndent || state.lineIndent < textIndent) && ch === 32){
            state.lineIndent++;
            ch = state.input.charCodeAt(++state.position);
        }
        if (!detectedIndent && state.lineIndent > textIndent) {
            textIndent = state.lineIndent;
        }
        if (is_EOL(ch)) {
            emptyLines++;
            continue;
        }
        if (state.lineIndent < textIndent) {
            if (chomping === CHOMPING_KEEP1) {
                state.result += common1.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (chomping === CHOMPING_CLIP) {
                if (didReadContent) {
                    state.result += "\n";
                }
            }
            break;
        }
        if (folding) {
            if (is_WHITE_SPACE(ch)) {
                atMoreIndented = true;
                state.result += common1.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
            } else if (atMoreIndented) {
                atMoreIndented = false;
                state.result += common1.repeat("\n", emptyLines + 1);
            } else if (emptyLines === 0) {
                if (didReadContent) {
                    state.result += " ";
                }
            } else {
                state.result += common1.repeat("\n", emptyLines);
            }
        } else {
            state.result += common1.repeat("\n", didReadContent ? 1 + emptyLines : emptyLines);
        }
        didReadContent = true;
        detectedIndent = true;
        emptyLines = 0;
        captureStart = state.position;
        while(!is_EOL(ch) && ch !== 0){
            ch = state.input.charCodeAt(++state.position);
        }
        captureSegment1(state, captureStart, state.position, false);
    }
    return true;
}
function readBlockSequence1(state, nodeIndent) {
    var _line, _tag = state.tag, _anchor = state.anchor, _result = [], following, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError1(state, "tab characters must not be used in indentation");
        }
        if (ch !== 45) {
            break;
        }
        following = state.input.charCodeAt(state.position + 1);
        if (!is_WS_OR_EOL(following)) {
            break;
        }
        detected = true;
        state.position++;
        if (skipSeparationSpace1(state, true, -1)) {
            if (state.lineIndent <= nodeIndent) {
                _result.push(null);
                ch = state.input.charCodeAt(state.position);
                continue;
            }
        }
        _line = state.line;
        composeNode1(state, nodeIndent, CONTEXT_BLOCK_IN1, false, true);
        _result.push(state.result);
        skipSeparationSpace1(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError1(state, "bad indentation of a sequence entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "sequence";
        state.result = _result;
        return true;
    }
    return false;
}
function readBlockMapping1(state, nodeIndent, flowIndent) {
    var following, allowCompact, _line, _keyLine, _keyLineStart, _keyPos, _tag = state.tag, _anchor = state.anchor, _result = {}, overridableKeys = Object.create(null), keyTag = null, keyNode = null, valueNode = null, atExplicitKey = false, detected = false, ch;
    if (state.firstTabInLine !== -1) return false;
    if (state.anchor !== null) {
        state.anchorMap[state.anchor] = _result;
    }
    ch = state.input.charCodeAt(state.position);
    while(ch !== 0){
        if (!atExplicitKey && state.firstTabInLine !== -1) {
            state.position = state.firstTabInLine;
            throwError1(state, "tab characters must not be used in indentation");
        }
        following = state.input.charCodeAt(state.position + 1);
        _line = state.line;
        if ((ch === 63 || ch === 58) && is_WS_OR_EOL(following)) {
            if (ch === 63) {
                if (atExplicitKey) {
                    storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                    keyTag = keyNode = valueNode = null;
                }
                detected = true;
                atExplicitKey = true;
                allowCompact = true;
            } else if (atExplicitKey) {
                atExplicitKey = false;
                allowCompact = true;
            } else {
                throwError1(state, "incomplete explicit mapping pair; a key node is missed; or followed by a non-tabulated empty line");
            }
            state.position += 1;
            ch = following;
        } else {
            _keyLine = state.line;
            _keyLineStart = state.lineStart;
            _keyPos = state.position;
            if (!composeNode1(state, flowIndent, CONTEXT_FLOW_OUT, false, true)) {
                break;
            }
            if (state.line === _line) {
                ch = state.input.charCodeAt(state.position);
                while(is_WHITE_SPACE(ch)){
                    ch = state.input.charCodeAt(++state.position);
                }
                if (ch === 58) {
                    ch = state.input.charCodeAt(++state.position);
                    if (!is_WS_OR_EOL(ch)) {
                        throwError1(state, "a whitespace character is expected after the key-value separator within a block mapping");
                    }
                    if (atExplicitKey) {
                        storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
                        keyTag = keyNode = valueNode = null;
                    }
                    detected = true;
                    atExplicitKey = false;
                    allowCompact = false;
                    keyTag = state.tag;
                    keyNode = state.result;
                } else if (detected) {
                    throwError1(state, "can not read an implicit mapping pair; a colon is missed");
                } else {
                    state.tag = _tag;
                    state.anchor = _anchor;
                    return true;
                }
            } else if (detected) {
                throwError1(state, "can not read a block mapping entry; a multiline key may not be an implicit key");
            } else {
                state.tag = _tag;
                state.anchor = _anchor;
                return true;
            }
        }
        if (state.line === _line || state.lineIndent > nodeIndent) {
            if (atExplicitKey) {
                _keyLine = state.line;
                _keyLineStart = state.lineStart;
                _keyPos = state.position;
            }
            if (composeNode1(state, nodeIndent, CONTEXT_BLOCK_OUT1, true, allowCompact)) {
                if (atExplicitKey) {
                    keyNode = state.result;
                } else {
                    valueNode = state.result;
                }
            }
            if (!atExplicitKey) {
                storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, valueNode, _keyLine, _keyLineStart, _keyPos);
                keyTag = keyNode = valueNode = null;
            }
            skipSeparationSpace1(state, true, -1);
            ch = state.input.charCodeAt(state.position);
        }
        if ((state.line === _line || state.lineIndent > nodeIndent) && ch !== 0) {
            throwError1(state, "bad indentation of a mapping entry");
        } else if (state.lineIndent < nodeIndent) {
            break;
        }
    }
    if (atExplicitKey) {
        storeMappingPair1(state, _result, overridableKeys, keyTag, keyNode, null, _keyLine, _keyLineStart, _keyPos);
    }
    if (detected) {
        state.tag = _tag;
        state.anchor = _anchor;
        state.kind = "mapping";
        state.result = _result;
    }
    return detected;
}
function readTagProperty1(state) {
    var _position, isVerbatim = false, isNamed = false, tagHandle, tagName, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 33) return false;
    if (state.tag !== null) {
        throwError1(state, "duplication of a tag property");
    }
    ch = state.input.charCodeAt(++state.position);
    if (ch === 60) {
        isVerbatim = true;
        ch = state.input.charCodeAt(++state.position);
    } else if (ch === 33) {
        isNamed = true;
        tagHandle = "!!";
        ch = state.input.charCodeAt(++state.position);
    } else {
        tagHandle = "!";
    }
    _position = state.position;
    if (isVerbatim) {
        do {
            ch = state.input.charCodeAt(++state.position);
        }while (ch !== 0 && ch !== 62)
        if (state.position < state.length) {
            tagName = state.input.slice(_position, state.position);
            ch = state.input.charCodeAt(++state.position);
        } else {
            throwError1(state, "unexpected end of the stream within a verbatim tag");
        }
    } else {
        while(ch !== 0 && !is_WS_OR_EOL(ch)){
            if (ch === 33) {
                if (!isNamed) {
                    tagHandle = state.input.slice(_position - 1, state.position + 1);
                    if (!PATTERN_TAG_HANDLE1.test(tagHandle)) {
                        throwError1(state, "named tag handle cannot contain such characters");
                    }
                    isNamed = true;
                    _position = state.position + 1;
                } else {
                    throwError1(state, "tag suffix cannot contain exclamation marks");
                }
            }
            ch = state.input.charCodeAt(++state.position);
        }
        tagName = state.input.slice(_position, state.position);
        if (PATTERN_FLOW_INDICATORS1.test(tagName)) {
            throwError1(state, "tag suffix cannot contain flow indicator characters");
        }
    }
    if (tagName && !PATTERN_TAG_URI1.test(tagName)) {
        throwError1(state, "tag name cannot contain such characters: " + tagName);
    }
    try {
        tagName = decodeURIComponent(tagName);
    } catch (err) {
        throwError1(state, "tag name is malformed: " + tagName);
    }
    if (isVerbatim) {
        state.tag = tagName;
    } else if (_hasOwnProperty$1.call(state.tagMap, tagHandle)) {
        state.tag = state.tagMap[tagHandle] + tagName;
    } else if (tagHandle === "!") {
        state.tag = "!" + tagName;
    } else if (tagHandle === "!!") {
        state.tag = "tag:yaml.org,2002:" + tagName;
    } else {
        throwError1(state, 'undeclared tag handle "' + tagHandle + '"');
    }
    return true;
}
function readAnchorProperty1(state) {
    var _position, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 38) return false;
    if (state.anchor !== null) {
        throwError1(state, "duplication of an anchor property");
    }
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL(ch) && !is_FLOW_INDICATOR(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError1(state, "name of an anchor node must contain at least one character");
    }
    state.anchor = state.input.slice(_position, state.position);
    return true;
}
function readAlias1(state) {
    var _position, alias, ch;
    ch = state.input.charCodeAt(state.position);
    if (ch !== 42) return false;
    ch = state.input.charCodeAt(++state.position);
    _position = state.position;
    while(ch !== 0 && !is_WS_OR_EOL(ch) && !is_FLOW_INDICATOR(ch)){
        ch = state.input.charCodeAt(++state.position);
    }
    if (state.position === _position) {
        throwError1(state, "name of an alias node must contain at least one character");
    }
    alias = state.input.slice(_position, state.position);
    if (!_hasOwnProperty$1.call(state.anchorMap, alias)) {
        throwError1(state, 'unidentified alias "' + alias + '"');
    }
    state.result = state.anchorMap[alias];
    skipSeparationSpace1(state, true, -1);
    return true;
}
function composeNode1(state, parentIndent, nodeContext, allowToSeek, allowCompact) {
    var allowBlockStyles, allowBlockScalars, allowBlockCollections, indentStatus = 1, atNewLine = false, hasContent = false, typeIndex, typeQuantity, typeList, type2, flowIndent, blockIndent;
    if (state.listener !== null) {
        state.listener("open", state);
    }
    state.tag = null;
    state.anchor = null;
    state.kind = null;
    state.result = null;
    allowBlockStyles = allowBlockScalars = allowBlockCollections = CONTEXT_BLOCK_OUT1 === nodeContext || CONTEXT_BLOCK_IN1 === nodeContext;
    if (allowToSeek) {
        if (skipSeparationSpace1(state, true, -1)) {
            atNewLine = true;
            if (state.lineIndent > parentIndent) {
                indentStatus = 1;
            } else if (state.lineIndent === parentIndent) {
                indentStatus = 0;
            } else if (state.lineIndent < parentIndent) {
                indentStatus = -1;
            }
        }
    }
    if (indentStatus === 1) {
        while(readTagProperty1(state) || readAnchorProperty1(state)){
            if (skipSeparationSpace1(state, true, -1)) {
                atNewLine = true;
                allowBlockCollections = allowBlockStyles;
                if (state.lineIndent > parentIndent) {
                    indentStatus = 1;
                } else if (state.lineIndent === parentIndent) {
                    indentStatus = 0;
                } else if (state.lineIndent < parentIndent) {
                    indentStatus = -1;
                }
            } else {
                allowBlockCollections = false;
            }
        }
    }
    if (allowBlockCollections) {
        allowBlockCollections = atNewLine || allowCompact;
    }
    if (indentStatus === 1 || CONTEXT_BLOCK_OUT1 === nodeContext) {
        if (CONTEXT_FLOW_IN === nodeContext || CONTEXT_FLOW_OUT === nodeContext) {
            flowIndent = parentIndent;
        } else {
            flowIndent = parentIndent + 1;
        }
        blockIndent = state.position - state.lineStart;
        if (indentStatus === 1) {
            if (allowBlockCollections && (readBlockSequence1(state, blockIndent) || readBlockMapping1(state, blockIndent, flowIndent)) || readFlowCollection1(state, flowIndent)) {
                hasContent = true;
            } else {
                if (allowBlockScalars && readBlockScalar1(state, flowIndent) || readSingleQuotedScalar1(state, flowIndent) || readDoubleQuotedScalar1(state, flowIndent)) {
                    hasContent = true;
                } else if (readAlias1(state)) {
                    hasContent = true;
                    if (state.tag !== null || state.anchor !== null) {
                        throwError1(state, "alias node should not have any properties");
                    }
                } else if (readPlainScalar1(state, flowIndent, CONTEXT_FLOW_IN === nodeContext)) {
                    hasContent = true;
                    if (state.tag === null) {
                        state.tag = "?";
                    }
                }
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
            }
        } else if (indentStatus === 0) {
            hasContent = allowBlockCollections && readBlockSequence1(state, blockIndent);
        }
    }
    if (state.tag === null) {
        if (state.anchor !== null) {
            state.anchorMap[state.anchor] = state.result;
        }
    } else if (state.tag === "?") {
        if (state.result !== null && state.kind !== "scalar") {
            throwError1(state, 'unacceptable node kind for !<?> tag; it should be "scalar", not "' + state.kind + '"');
        }
        for(typeIndex = 0, typeQuantity = state.implicitTypes.length; typeIndex < typeQuantity; typeIndex += 1){
            type2 = state.implicitTypes[typeIndex];
            if (type2.resolve(state.result)) {
                state.result = type2.construct(state.result);
                state.tag = type2.tag;
                if (state.anchor !== null) {
                    state.anchorMap[state.anchor] = state.result;
                }
                break;
            }
        }
    } else if (state.tag !== "!") {
        if (_hasOwnProperty$1.call(state.typeMap[state.kind || "fallback"], state.tag)) {
            type2 = state.typeMap[state.kind || "fallback"][state.tag];
        } else {
            type2 = null;
            typeList = state.typeMap.multi[state.kind || "fallback"];
            for(typeIndex = 0, typeQuantity = typeList.length; typeIndex < typeQuantity; typeIndex += 1){
                if (state.tag.slice(0, typeList[typeIndex].tag.length) === typeList[typeIndex].tag) {
                    type2 = typeList[typeIndex];
                    break;
                }
            }
        }
        if (!type2) {
            throwError1(state, "unknown tag !<" + state.tag + ">");
        }
        if (state.result !== null && type2.kind !== state.kind) {
            throwError1(state, "unacceptable node kind for !<" + state.tag + '> tag; it should be "' + type2.kind + '", not "' + state.kind + '"');
        }
        if (!type2.resolve(state.result, state.tag)) {
            throwError1(state, "cannot resolve a node with !<" + state.tag + "> explicit tag");
        } else {
            state.result = type2.construct(state.result, state.tag);
            if (state.anchor !== null) {
                state.anchorMap[state.anchor] = state.result;
            }
        }
    }
    if (state.listener !== null) {
        state.listener("close", state);
    }
    return state.tag !== null || state.anchor !== null || hasContent;
}
function readDocument1(state) {
    var documentStart = state.position, _position, directiveName, directiveArgs, hasDirectives = false, ch;
    state.version = null;
    state.checkLineBreaks = state.legacy;
    state.tagMap = Object.create(null);
    state.anchorMap = Object.create(null);
    while((ch = state.input.charCodeAt(state.position)) !== 0){
        skipSeparationSpace1(state, true, -1);
        ch = state.input.charCodeAt(state.position);
        if (state.lineIndent > 0 || ch !== 37) {
            break;
        }
        hasDirectives = true;
        ch = state.input.charCodeAt(++state.position);
        _position = state.position;
        while(ch !== 0 && !is_WS_OR_EOL(ch)){
            ch = state.input.charCodeAt(++state.position);
        }
        directiveName = state.input.slice(_position, state.position);
        directiveArgs = [];
        if (directiveName.length < 1) {
            throwError1(state, "directive name must not be less than one character in length");
        }
        while(ch !== 0){
            while(is_WHITE_SPACE(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            if (ch === 35) {
                do {
                    ch = state.input.charCodeAt(++state.position);
                }while (ch !== 0 && !is_EOL(ch))
                break;
            }
            if (is_EOL(ch)) break;
            _position = state.position;
            while(ch !== 0 && !is_WS_OR_EOL(ch)){
                ch = state.input.charCodeAt(++state.position);
            }
            directiveArgs.push(state.input.slice(_position, state.position));
        }
        if (ch !== 0) readLineBreak1(state);
        if (_hasOwnProperty$1.call(directiveHandlers1, directiveName)) {
            directiveHandlers1[directiveName](state, directiveName, directiveArgs);
        } else {
            throwWarning1(state, 'unknown document directive "' + directiveName + '"');
        }
    }
    skipSeparationSpace1(state, true, -1);
    if (state.lineIndent === 0 && state.input.charCodeAt(state.position) === 45 && state.input.charCodeAt(state.position + 1) === 45 && state.input.charCodeAt(state.position + 2) === 45) {
        state.position += 3;
        skipSeparationSpace1(state, true, -1);
    } else if (hasDirectives) {
        throwError1(state, "directives end mark is expected");
    }
    composeNode1(state, state.lineIndent - 1, CONTEXT_BLOCK_OUT1, false, true);
    skipSeparationSpace1(state, true, -1);
    if (state.checkLineBreaks && PATTERN_NON_ASCII_LINE_BREAKS1.test(state.input.slice(documentStart, state.position))) {
        throwWarning1(state, "non-ASCII line breaks are interpreted as content");
    }
    state.documents.push(state.result);
    if (state.position === state.lineStart && testDocumentSeparator1(state)) {
        if (state.input.charCodeAt(state.position) === 46) {
            state.position += 3;
            skipSeparationSpace1(state, true, -1);
        }
        return;
    }
    if (state.position < state.length - 1) {
        throwError1(state, "end of the stream or a document separator is expected");
    } else {
        return;
    }
}
function loadDocuments1(input, options) {
    input = String(input);
    options = options || {};
    if (input.length !== 0) {
        if (input.charCodeAt(input.length - 1) !== 10 && input.charCodeAt(input.length - 1) !== 13) {
            input += "\n";
        }
        if (input.charCodeAt(0) === 65279) {
            input = input.slice(1);
        }
    }
    var state = new State$1(input, options);
    var nullpos = input.indexOf("\0");
    if (nullpos !== -1) {
        state.position = nullpos;
        throwError1(state, "null byte is not allowed in input");
    }
    state.input += "\0";
    while(state.input.charCodeAt(state.position) === 32){
        state.lineIndent += 1;
        state.position += 1;
    }
    while(state.position < state.length - 1){
        readDocument1(state);
    }
    return state.documents;
}
function loadAll$1(input, iterator, options) {
    if (iterator !== null && typeof iterator === "object" && typeof options === "undefined") {
        options = iterator;
        iterator = null;
    }
    var documents = loadDocuments1(input, options);
    if (typeof iterator !== "function") {
        return documents;
    }
    for(var index = 0, length = documents.length; index < length; index += 1){
        iterator(documents[index]);
    }
}
function load$1(input, options) {
    var documents = loadDocuments1(input, options);
    if (documents.length === 0) {
        return void 0;
    } else if (documents.length === 1) {
        return documents[0];
    }
    throw new exception("expected a single document in the stream, but found more");
}
var loadAll_1 = loadAll$1;
var load_1 = load$1;
var loader = {
    loadAll: loadAll_1,
    load: load_1
};
var _toString3 = Object.prototype.toString;
var _hasOwnProperty = Object.prototype.hasOwnProperty;
var CHAR_BOM = 65279;
var CHAR_TAB = 9;
var CHAR_LINE_FEED = 10;
var CHAR_CARRIAGE_RETURN = 13;
var CHAR_SPACE = 32;
var CHAR_EXCLAMATION = 33;
var CHAR_DOUBLE_QUOTE = 34;
var CHAR_SHARP = 35;
var CHAR_PERCENT = 37;
var CHAR_AMPERSAND = 38;
var CHAR_SINGLE_QUOTE = 39;
var CHAR_ASTERISK = 42;
var CHAR_COMMA = 44;
var CHAR_MINUS = 45;
var CHAR_COLON = 58;
var CHAR_EQUALS = 61;
var CHAR_GREATER_THAN = 62;
var CHAR_QUESTION = 63;
var CHAR_COMMERCIAL_AT = 64;
var CHAR_LEFT_SQUARE_BRACKET = 91;
var CHAR_RIGHT_SQUARE_BRACKET = 93;
var CHAR_GRAVE_ACCENT = 96;
var CHAR_LEFT_CURLY_BRACKET = 123;
var CHAR_VERTICAL_LINE = 124;
var CHAR_RIGHT_CURLY_BRACKET = 125;
var ESCAPE_SEQUENCES1 = {};
ESCAPE_SEQUENCES1[0] = "\\0";
ESCAPE_SEQUENCES1[7] = "\\a";
ESCAPE_SEQUENCES1[8] = "\\b";
ESCAPE_SEQUENCES1[9] = "\\t";
ESCAPE_SEQUENCES1[10] = "\\n";
ESCAPE_SEQUENCES1[11] = "\\v";
ESCAPE_SEQUENCES1[12] = "\\f";
ESCAPE_SEQUENCES1[13] = "\\r";
ESCAPE_SEQUENCES1[27] = "\\e";
ESCAPE_SEQUENCES1[34] = '\\"';
ESCAPE_SEQUENCES1[92] = "\\\\";
ESCAPE_SEQUENCES1[133] = "\\N";
ESCAPE_SEQUENCES1[160] = "\\_";
ESCAPE_SEQUENCES1[8232] = "\\L";
ESCAPE_SEQUENCES1[8233] = "\\P";
var DEPRECATED_BOOLEANS_SYNTAX1 = [
    "y",
    "Y",
    "yes",
    "Yes",
    "YES",
    "on",
    "On",
    "ON",
    "n",
    "N",
    "no",
    "No",
    "NO",
    "off",
    "Off",
    "OFF"
];
var DEPRECATED_BASE60_SYNTAX = /^[-+]?[0-9_]+(?::[0-9_]+)+(?:\.[0-9_]*)?$/;
function compileStyleMap1(schema2, map2) {
    var result1, keys, index, length, tag, style, type2;
    if (map2 === null) return {};
    result1 = {};
    keys = Object.keys(map2);
    for(index = 0, length = keys.length; index < length; index += 1){
        tag = keys[index];
        style = String(map2[tag]);
        if (tag.slice(0, 2) === "!!") {
            tag = "tag:yaml.org,2002:" + tag.slice(2);
        }
        type2 = schema2.compiledTypeMap["fallback"][tag];
        if (type2 && _hasOwnProperty.call(type2.styleAliases, style)) {
            style = type2.styleAliases[style];
        }
        result1[tag] = style;
    }
    return result1;
}
function encodeHex1(character) {
    var string, handle, length;
    string = character.toString(16).toUpperCase();
    if (character <= 255) {
        handle = "x";
        length = 2;
    } else if (character <= 65535) {
        handle = "u";
        length = 4;
    } else if (character <= 4294967295) {
        handle = "U";
        length = 8;
    } else {
        throw new exception("code point within a string may not be greater than 0xFFFFFFFF");
    }
    return "\\" + handle + common1.repeat("0", length - string.length) + string;
}
var QUOTING_TYPE_SINGLE = 1, QUOTING_TYPE_DOUBLE = 2;
function State2(options) {
    this.schema = options["schema"] || _default;
    this.indent = Math.max(1, options["indent"] || 2);
    this.noArrayIndent = options["noArrayIndent"] || false;
    this.skipInvalid = options["skipInvalid"] || false;
    this.flowLevel = common1.isNothing(options["flowLevel"]) ? -1 : options["flowLevel"];
    this.styleMap = compileStyleMap1(this.schema, options["styles"] || null);
    this.sortKeys = options["sortKeys"] || false;
    this.lineWidth = options["lineWidth"] || 80;
    this.noRefs = options["noRefs"] || false;
    this.noCompatMode = options["noCompatMode"] || false;
    this.condenseFlow = options["condenseFlow"] || false;
    this.quotingType = options["quotingType"] === '"' ? QUOTING_TYPE_DOUBLE : QUOTING_TYPE_SINGLE;
    this.forceQuotes = options["forceQuotes"] || false;
    this.replacer = typeof options["replacer"] === "function" ? options["replacer"] : null;
    this.implicitTypes = this.schema.compiledImplicit;
    this.explicitTypes = this.schema.compiledExplicit;
    this.tag = null;
    this.result = "";
    this.duplicates = [];
    this.usedDuplicates = null;
}
function indentString1(string, spaces) {
    var ind = common1.repeat(" ", spaces), position = 0, next = -1, result1 = "", line, length = string.length;
    while(position < length){
        next = string.indexOf("\n", position);
        if (next === -1) {
            line = string.slice(position);
            position = length;
        } else {
            line = string.slice(position, next + 1);
            position = next + 1;
        }
        if (line.length && line !== "\n") result1 += ind;
        result1 += line;
    }
    return result1;
}
function generateNextLine1(state, level) {
    return "\n" + common1.repeat(" ", state.indent * level);
}
function testImplicitResolving1(state, str2) {
    var index, length, type2;
    for(index = 0, length = state.implicitTypes.length; index < length; index += 1){
        type2 = state.implicitTypes[index];
        if (type2.resolve(str2)) {
            return true;
        }
    }
    return false;
}
function isWhitespace1(c) {
    return c === CHAR_SPACE || c === CHAR_TAB;
}
function isPrintable1(c) {
    return 32 <= c && c <= 126 || 161 <= c && c <= 55295 && c !== 8232 && c !== 8233 || 57344 <= c && c <= 65533 && c !== CHAR_BOM || 65536 <= c && c <= 1114111;
}
function isNsCharOrWhitespace(c) {
    return isPrintable1(c) && c !== CHAR_BOM && c !== CHAR_CARRIAGE_RETURN && c !== CHAR_LINE_FEED;
}
function isPlainSafe1(c, prev, inblock) {
    var cIsNsCharOrWhitespace = isNsCharOrWhitespace(c);
    var cIsNsChar = cIsNsCharOrWhitespace && !isWhitespace1(c);
    return (inblock ? cIsNsCharOrWhitespace : cIsNsCharOrWhitespace && c !== CHAR_COMMA && c !== CHAR_LEFT_SQUARE_BRACKET && c !== CHAR_RIGHT_SQUARE_BRACKET && c !== CHAR_LEFT_CURLY_BRACKET && c !== CHAR_RIGHT_CURLY_BRACKET) && c !== CHAR_SHARP && !(prev === CHAR_COLON && !cIsNsChar) || isNsCharOrWhitespace(prev) && !isWhitespace1(prev) && c === CHAR_SHARP || prev === CHAR_COLON && cIsNsChar;
}
function isPlainSafeFirst1(c) {
    return isPrintable1(c) && c !== CHAR_BOM && !isWhitespace1(c) && c !== CHAR_MINUS && c !== CHAR_QUESTION && c !== CHAR_COLON && c !== CHAR_COMMA && c !== CHAR_LEFT_SQUARE_BRACKET && c !== CHAR_RIGHT_SQUARE_BRACKET && c !== CHAR_LEFT_CURLY_BRACKET && c !== CHAR_RIGHT_CURLY_BRACKET && c !== CHAR_SHARP && c !== CHAR_AMPERSAND && c !== CHAR_ASTERISK && c !== CHAR_EXCLAMATION && c !== CHAR_VERTICAL_LINE && c !== CHAR_EQUALS && c !== CHAR_GREATER_THAN && c !== CHAR_SINGLE_QUOTE && c !== CHAR_DOUBLE_QUOTE && c !== CHAR_PERCENT && c !== CHAR_COMMERCIAL_AT && c !== CHAR_GRAVE_ACCENT;
}
function isPlainSafeLast(c) {
    return !isWhitespace1(c) && c !== CHAR_COLON;
}
function codePointAt(string, pos) {
    var first = string.charCodeAt(pos), second;
    if (first >= 55296 && first <= 56319 && pos + 1 < string.length) {
        second = string.charCodeAt(pos + 1);
        if (second >= 56320 && second <= 57343) {
            return (first - 55296) * 1024 + second - 56320 + 65536;
        }
    }
    return first;
}
function needIndentIndicator1(string) {
    var leadingSpaceRe = /^\n* /;
    return leadingSpaceRe.test(string);
}
var STYLE_PLAIN1 = 1, STYLE_SINGLE1 = 2, STYLE_LITERAL1 = 3, STYLE_FOLDED1 = 4, STYLE_DOUBLE1 = 5;
function chooseScalarStyle1(string, singleLineOnly, indentPerLevel, lineWidth, testAmbiguousType, quotingType, forceQuotes, inblock) {
    var i1;
    var __char = 0;
    var prevChar = null;
    var hasLineBreak = false;
    var hasFoldableLine = false;
    var shouldTrackWidth = lineWidth !== -1;
    var previousLineBreak = -1;
    var plain = isPlainSafeFirst1(codePointAt(string, 0)) && isPlainSafeLast(codePointAt(string, string.length - 1));
    if (singleLineOnly || forceQuotes) {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt(string, i1);
            if (!isPrintable1(__char)) {
                return STYLE_DOUBLE1;
            }
            plain = plain && isPlainSafe1(__char, prevChar, inblock);
            prevChar = __char;
        }
    } else {
        for(i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
            __char = codePointAt(string, i1);
            if (__char === CHAR_LINE_FEED) {
                hasLineBreak = true;
                if (shouldTrackWidth) {
                    hasFoldableLine = hasFoldableLine || i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
                    previousLineBreak = i1;
                }
            } else if (!isPrintable1(__char)) {
                return STYLE_DOUBLE1;
            }
            plain = plain && isPlainSafe1(__char, prevChar, inblock);
            prevChar = __char;
        }
        hasFoldableLine = hasFoldableLine || shouldTrackWidth && i1 - previousLineBreak - 1 > lineWidth && string[previousLineBreak + 1] !== " ";
    }
    if (!hasLineBreak && !hasFoldableLine) {
        if (plain && !forceQuotes && !testAmbiguousType(string)) {
            return STYLE_PLAIN1;
        }
        return quotingType === QUOTING_TYPE_DOUBLE ? STYLE_DOUBLE1 : STYLE_SINGLE1;
    }
    if (indentPerLevel > 9 && needIndentIndicator1(string)) {
        return STYLE_DOUBLE1;
    }
    if (!forceQuotes) {
        return hasFoldableLine ? STYLE_FOLDED1 : STYLE_LITERAL1;
    }
    return quotingType === QUOTING_TYPE_DOUBLE ? STYLE_DOUBLE1 : STYLE_SINGLE1;
}
function writeScalar1(state, string, level, iskey, inblock) {
    state.dump = function() {
        if (string.length === 0) {
            return state.quotingType === QUOTING_TYPE_DOUBLE ? '""' : "''";
        }
        if (!state.noCompatMode) {
            if (DEPRECATED_BOOLEANS_SYNTAX1.indexOf(string) !== -1 || DEPRECATED_BASE60_SYNTAX.test(string)) {
                return state.quotingType === QUOTING_TYPE_DOUBLE ? '"' + string + '"' : "'" + string + "'";
            }
        }
        var indent = state.indent * Math.max(1, level);
        var lineWidth = state.lineWidth === -1 ? -1 : Math.max(Math.min(state.lineWidth, 40), state.lineWidth - indent);
        var singleLineOnly = iskey || state.flowLevel > -1 && level >= state.flowLevel;
        function testAmbiguity(string2) {
            return testImplicitResolving1(state, string2);
        }
        switch(chooseScalarStyle1(string, singleLineOnly, state.indent, lineWidth, testAmbiguity, state.quotingType, state.forceQuotes && !iskey, inblock)){
            case STYLE_PLAIN1:
                return string;
            case STYLE_SINGLE1:
                return "'" + string.replace(/'/g, "''") + "'";
            case STYLE_LITERAL1:
                return "|" + blockHeader1(string, state.indent) + dropEndingNewline1(indentString1(string, indent));
            case STYLE_FOLDED1:
                return ">" + blockHeader1(string, state.indent) + dropEndingNewline1(indentString1(foldString1(string, lineWidth), indent));
            case STYLE_DOUBLE1:
                return '"' + escapeString1(string) + '"';
            default:
                throw new exception("impossible error: invalid scalar style");
        }
    }();
}
function blockHeader1(string, indentPerLevel) {
    var indentIndicator = needIndentIndicator1(string) ? String(indentPerLevel) : "";
    var clip = string[string.length - 1] === "\n";
    var keep = clip && (string[string.length - 2] === "\n" || string === "\n");
    var chomp = keep ? "+" : clip ? "" : "-";
    return indentIndicator + chomp + "\n";
}
function dropEndingNewline1(string) {
    return string[string.length - 1] === "\n" ? string.slice(0, -1) : string;
}
function foldString1(string, width) {
    var lineRe = /(\n+)([^\n]*)/g;
    var result1 = function() {
        var nextLF = string.indexOf("\n");
        nextLF = nextLF !== -1 ? nextLF : string.length;
        lineRe.lastIndex = nextLF;
        return foldLine1(string.slice(0, nextLF), width);
    }();
    var prevMoreIndented = string[0] === "\n" || string[0] === " ";
    var moreIndented;
    var match;
    while(match = lineRe.exec(string)){
        var prefix = match[1], line = match[2];
        moreIndented = line[0] === " ";
        result1 += prefix + (!prevMoreIndented && !moreIndented && line !== "" ? "\n" : "") + foldLine1(line, width);
        prevMoreIndented = moreIndented;
    }
    return result1;
}
function foldLine1(line, width) {
    if (line === "" || line[0] === " ") return line;
    var breakRe = / [^ ]/g;
    var match;
    var start = 0, end, curr = 0, next = 0;
    var result1 = "";
    while(match = breakRe.exec(line)){
        next = match.index;
        if (next - start > width) {
            end = curr > start ? curr : next;
            result1 += "\n" + line.slice(start, end);
            start = end + 1;
        }
        curr = next;
    }
    result1 += "\n";
    if (line.length - start > width && curr > start) {
        result1 += line.slice(start, curr) + "\n" + line.slice(curr + 1);
    } else {
        result1 += line.slice(start);
    }
    return result1.slice(1);
}
function escapeString1(string) {
    var result1 = "";
    var __char = 0;
    var escapeSeq;
    for(var i1 = 0; i1 < string.length; __char >= 65536 ? i1 += 2 : i1++){
        __char = codePointAt(string, i1);
        escapeSeq = ESCAPE_SEQUENCES1[__char];
        if (!escapeSeq && isPrintable1(__char)) {
            result1 += string[i1];
            if (__char >= 65536) result1 += string[i1 + 1];
        } else {
            result1 += escapeSeq || encodeHex1(__char);
        }
    }
    return result1;
}
function writeFlowSequence1(state, level, object) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode1(state, level, value, false, false) || typeof value === "undefined" && writeNode1(state, level, null, false, false)) {
            if (_result !== "") _result += "," + (!state.condenseFlow ? " " : "");
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = "[" + _result + "]";
}
function writeBlockSequence1(state, level, object, compact) {
    var _result = "", _tag = state.tag, index, length, value;
    for(index = 0, length = object.length; index < length; index += 1){
        value = object[index];
        if (state.replacer) {
            value = state.replacer.call(object, String(index), value);
        }
        if (writeNode1(state, level + 1, value, true, true, false, true) || typeof value === "undefined" && writeNode1(state, level + 1, null, true, true, false, true)) {
            if (!compact || _result !== "") {
                _result += generateNextLine1(state, level);
            }
            if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
                _result += "-";
            } else {
                _result += "- ";
            }
            _result += state.dump;
        }
    }
    state.tag = _tag;
    state.dump = _result || "[]";
}
function writeFlowMapping1(state, level, object) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, pairBuffer;
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (_result !== "") pairBuffer += ", ";
        if (state.condenseFlow) pairBuffer += '"';
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode1(state, level, objectKey, false, false)) {
            continue;
        }
        if (state.dump.length > 1024) pairBuffer += "? ";
        pairBuffer += state.dump + (state.condenseFlow ? '"' : "") + ":" + (state.condenseFlow ? "" : " ");
        if (!writeNode1(state, level, objectValue, false, false)) {
            continue;
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = "{" + _result + "}";
}
function writeBlockMapping1(state, level, object, compact) {
    var _result = "", _tag = state.tag, objectKeyList = Object.keys(object), index, length, objectKey, objectValue, explicitPair, pairBuffer;
    if (state.sortKeys === true) {
        objectKeyList.sort();
    } else if (typeof state.sortKeys === "function") {
        objectKeyList.sort(state.sortKeys);
    } else if (state.sortKeys) {
        throw new exception("sortKeys must be a boolean or a function");
    }
    for(index = 0, length = objectKeyList.length; index < length; index += 1){
        pairBuffer = "";
        if (!compact || _result !== "") {
            pairBuffer += generateNextLine1(state, level);
        }
        objectKey = objectKeyList[index];
        objectValue = object[objectKey];
        if (state.replacer) {
            objectValue = state.replacer.call(object, objectKey, objectValue);
        }
        if (!writeNode1(state, level + 1, objectKey, true, true, true)) {
            continue;
        }
        explicitPair = state.tag !== null && state.tag !== "?" || state.dump && state.dump.length > 1024;
        if (explicitPair) {
            if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
                pairBuffer += "?";
            } else {
                pairBuffer += "? ";
            }
        }
        pairBuffer += state.dump;
        if (explicitPair) {
            pairBuffer += generateNextLine1(state, level);
        }
        if (!writeNode1(state, level + 1, objectValue, true, explicitPair)) {
            continue;
        }
        if (state.dump && CHAR_LINE_FEED === state.dump.charCodeAt(0)) {
            pairBuffer += ":";
        } else {
            pairBuffer += ": ";
        }
        pairBuffer += state.dump;
        _result += pairBuffer;
    }
    state.tag = _tag;
    state.dump = _result || "{}";
}
function detectType1(state, object, explicit) {
    var _result, typeList, index, length, type2, style;
    typeList = explicit ? state.explicitTypes : state.implicitTypes;
    for(index = 0, length = typeList.length; index < length; index += 1){
        type2 = typeList[index];
        if ((type2.instanceOf || type2.predicate) && (!type2.instanceOf || typeof object === "object" && object instanceof type2.instanceOf) && (!type2.predicate || type2.predicate(object))) {
            if (explicit) {
                if (type2.multi && type2.representName) {
                    state.tag = type2.representName(object);
                } else {
                    state.tag = type2.tag;
                }
            } else {
                state.tag = "?";
            }
            if (type2.represent) {
                style = state.styleMap[type2.tag] || type2.defaultStyle;
                if (_toString3.call(type2.represent) === "[object Function]") {
                    _result = type2.represent(object, style);
                } else if (_hasOwnProperty.call(type2.represent, style)) {
                    _result = type2.represent[style](object, style);
                } else {
                    throw new exception("!<" + type2.tag + '> tag resolver accepts not "' + style + '" style');
                }
                state.dump = _result;
            }
            return true;
        }
    }
    return false;
}
function writeNode1(state, level, object, block, compact, iskey, isblockseq) {
    state.tag = null;
    state.dump = object;
    if (!detectType1(state, object, false)) {
        detectType1(state, object, true);
    }
    var type2 = _toString3.call(state.dump);
    var inblock = block;
    var tagStr;
    if (block) {
        block = state.flowLevel < 0 || state.flowLevel > level;
    }
    var objectOrArray = type2 === "[object Object]" || type2 === "[object Array]", duplicateIndex, duplicate;
    if (objectOrArray) {
        duplicateIndex = state.duplicates.indexOf(object);
        duplicate = duplicateIndex !== -1;
    }
    if (state.tag !== null && state.tag !== "?" || duplicate || state.indent !== 2 && level > 0) {
        compact = false;
    }
    if (duplicate && state.usedDuplicates[duplicateIndex]) {
        state.dump = "*ref_" + duplicateIndex;
    } else {
        if (objectOrArray && duplicate && !state.usedDuplicates[duplicateIndex]) {
            state.usedDuplicates[duplicateIndex] = true;
        }
        if (type2 === "[object Object]") {
            if (block && Object.keys(state.dump).length !== 0) {
                writeBlockMapping1(state, level, state.dump, compact);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowMapping1(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object Array]") {
            if (block && state.dump.length !== 0) {
                if (state.noArrayIndent && !isblockseq && level > 0) {
                    writeBlockSequence1(state, level - 1, state.dump, compact);
                } else {
                    writeBlockSequence1(state, level, state.dump, compact);
                }
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + state.dump;
                }
            } else {
                writeFlowSequence1(state, level, state.dump);
                if (duplicate) {
                    state.dump = "&ref_" + duplicateIndex + " " + state.dump;
                }
            }
        } else if (type2 === "[object String]") {
            if (state.tag !== "?") {
                writeScalar1(state, state.dump, level, iskey, inblock);
            }
        } else if (type2 === "[object Undefined]") {
            return false;
        } else {
            if (state.skipInvalid) return false;
            throw new exception("unacceptable kind of an object to dump " + type2);
        }
        if (state.tag !== null && state.tag !== "?") {
            tagStr = encodeURI(state.tag[0] === "!" ? state.tag.slice(1) : state.tag).replace(/!/g, "%21");
            if (state.tag[0] === "!") {
                tagStr = "!" + tagStr;
            } else if (tagStr.slice(0, 18) === "tag:yaml.org,2002:") {
                tagStr = "!!" + tagStr.slice(18);
            } else {
                tagStr = "!<" + tagStr + ">";
            }
            state.dump = tagStr + " " + state.dump;
        }
    }
    return true;
}
function getDuplicateReferences1(object, state) {
    var objects = [], duplicatesIndexes = [], index, length;
    inspectNode1(object, objects, duplicatesIndexes);
    for(index = 0, length = duplicatesIndexes.length; index < length; index += 1){
        state.duplicates.push(objects[duplicatesIndexes[index]]);
    }
    state.usedDuplicates = new Array(length);
}
function inspectNode1(object, objects, duplicatesIndexes) {
    var objectKeyList, index, length;
    if (object !== null && typeof object === "object") {
        index = objects.indexOf(object);
        if (index !== -1) {
            if (duplicatesIndexes.indexOf(index) === -1) {
                duplicatesIndexes.push(index);
            }
        } else {
            objects.push(object);
            if (Array.isArray(object)) {
                for(index = 0, length = object.length; index < length; index += 1){
                    inspectNode1(object[index], objects, duplicatesIndexes);
                }
            } else {
                objectKeyList = Object.keys(object);
                for(index = 0, length = objectKeyList.length; index < length; index += 1){
                    inspectNode1(object[objectKeyList[index]], objects, duplicatesIndexes);
                }
            }
        }
    }
}
function dump$1(input, options) {
    options = options || {};
    var state = new State2(options);
    if (!state.noRefs) getDuplicateReferences1(input, state);
    var value = input;
    if (state.replacer) {
        value = state.replacer.call({
            "": value
        }, "", value);
    }
    if (writeNode1(state, 0, value, true, true)) return state.dump + "\n";
    return "";
}
var dump_1 = dump$1;
var dumper = {
    dump: dump_1
};
function renamed(from, to) {
    return function() {
        throw new Error("Function yaml." + from + " is removed in js-yaml 4. Use yaml." + to + " instead, which is now safe by default.");
    };
}
var Type2 = type;
var Schema1 = schema;
var load1 = loader.load;
loader.loadAll;
dumper.dump;
renamed("safeLoad", "load");
renamed("safeLoadAll", "loadAll");
renamed("safeDump", "dump");
const QuartoJSONSchema = new Schema1({
    implicit: [
        _null,
        bool1,
        __int1,
        __float1
    ],
    include: [
        failsafe1
    ],
    explicit: [
        new Type2("!expr", {
            kind: "scalar",
            construct (data) {
                const result1 = data !== null ? data : "";
                return {
                    value: result1,
                    tag: "!expr"
                };
            }
        })
    ]
});
function postProcessAnnotation(parse) {
    if (parse.components.length === 1 && parse.start === parse.components[0].start && parse.end === parse.components[0].end) {
        return postProcessAnnotation(parse.components[0]);
    } else {
        parse.components = parse.components.map(postProcessAnnotation);
        return parse;
    }
}
function jsYamlParseLenient(yml) {
    try {
        return load1(yml, {
            schema: QuartoJSONSchema
        });
    } catch (_e) {
        return yml;
    }
}
function readAnnotatedYamlFromMappedString(mappedSource, lenient = false) {
    if (lenient) {
        const parser = getTreeSitterSync();
        const tree = parser.parse(mappedSource.value);
        const treeSitterAnnotation = buildTreeSitterAnnotation(tree, mappedSource);
        if (treeSitterAnnotation) {
            return treeSitterAnnotation;
        }
    }
    try {
        return buildJsYamlAnnotation(mappedSource);
    } catch (e) {
        const m = e.stack.split("\n")[0].match(/^.+ \((\d+):(\d+)\)$/);
        if (m) {
            const f = lineColToIndex(mappedSource.value);
            const location = {
                line: Number(m[1]) - 1,
                column: Number(m[2] - 1)
            };
            const offset = f(location);
            const { originalString } = mappedSource.map(offset, true);
            const filename = originalString.fileName;
            const f2 = mappedIndexToLineCol(mappedSource);
            const { line, column } = f2(offset);
            const sourceContext = createSourceContext(mappedSource, {
                start: offset,
                end: offset + 1
            });
            e.stack = `${e.reason} (${filename}, ${line + 1}:${column + 1})\n${sourceContext}`;
            e.message = e.stack;
            if (mappedLines(mappedSource)[location.line].value.indexOf("!expr") !== -1 && e.reason.match(/bad indentation of a mapping entry/)) {
                e.message = `${e.message}\n${tidyverseInfo("YAML tags like !expr must be followed by YAML strings.")}\n${tidyverseInfo("Is it possible you need to quote the value you passed to !expr ?")}`;
            }
            e.stack = "";
        }
        throw e;
    }
}
function buildJsYamlAnnotation(mappedYaml) {
    const yml = mappedYaml.value;
    const stack = [];
    const results = [];
    function listener(what, state) {
        const { result: result1, position, kind } = state;
        if (what === "close") {
            const { position: openPosition, kind: openKind } = stack.pop();
            if (results.length > 0) {
                const last = results[results.length - 1];
                if (last.start === openPosition && last.end === position) {
                    return;
                }
            }
            const components = [];
            while(results.length > 0){
                const last = results[results.length - 1];
                if (last.end <= openPosition) {
                    break;
                }
                components.push(results.pop());
            }
            components.reverse();
            const rawRange = yml.substring(openPosition, position);
            const leftTrim = rawRange.length - rawRange.trimStart().length;
            const rightTrim = rawRange.length - rawRange.trimEnd().length;
            if (openKind === null && kind === null) {} else if (rawRange.trim().length === 0) {
                results.push({
                    start: position - rightTrim,
                    end: position - rightTrim,
                    result: result1,
                    components,
                    kind,
                    source: mappedYaml
                });
            } else {
                results.push({
                    start: openPosition + leftTrim,
                    end: position - rightTrim,
                    result: result1,
                    components,
                    kind,
                    source: mappedYaml
                });
            }
        } else {
            stack.push({
                position,
                kind
            });
        }
    }
    load1(yml, {
        listener,
        schema: QuartoJSONSchema
    });
    if (results.length === 0) {
        return {
            start: 0,
            end: 0,
            result: null,
            kind: "null",
            components: [],
            source: mappedYaml
        };
    }
    if (results.length !== 1) {
        throw new InternalError(`Expected a single result, got ${results.length} instead`);
    }
    JSON.stringify(results[0]);
    return postProcessAnnotation(results[0]);
}
function buildTreeSitterAnnotation(tree, mappedSource) {
    const errors = [];
    const singletonBuild = (node)=>{
        let tag = undefined;
        for (const child of node.children){
            if (child.type === "tag") {
                tag = child;
                continue;
            }
            if (child.type !== "comment") {
                const result1 = buildNode(child, node.endIndex);
                if (tag) {
                    return annotateTag(result1, tag, node);
                } else {
                    return result1;
                }
            }
        }
        return annotateEmpty(node.endIndex);
    };
    const buildNode = (node, endIndex)=>{
        if (node === null) {
            return annotateEmpty(endIndex === undefined ? -1 : endIndex);
        }
        if (dispatch[node.type] === undefined) {
            return annotateEmpty(endIndex || node.endIndex || -1);
        }
        return dispatch[node.type](node);
    };
    const annotateEmpty = (position)=>{
        return {
            start: position,
            end: position,
            result: null,
            kind: "<<EMPTY>>",
            components: [],
            source: mappedSource
        };
    };
    const annotate = (node, result1, components)=>{
        return {
            start: node.startIndex,
            end: node.endIndex,
            result: result1,
            kind: node.type,
            components,
            source: mappedSource
        };
    };
    const annotateTag = (innerParse, tagNode, outerNode)=>{
        const tagParse = annotate(tagNode, tagNode.text, []);
        const result1 = annotate(outerNode, {
            tag: tagNode.text,
            value: innerParse.result
        }, [
            tagParse,
            innerParse
        ]);
        return result1;
    };
    const buildPair = (node)=>{
        let key, value;
        const children = node.children.filter((n)=>n.type !== "comment");
        if (children.length === 3) {
            key = annotate(children[0], children[0].text, []);
            value = buildNode(children[2], node.endIndex);
        } else if (children.length === 2) {
            key = annotate(children[0], children[0].text, []);
            value = annotateEmpty(node.endIndex);
        } else {
            key = annotateEmpty(node.endIndex);
            value = annotateEmpty(node.endIndex);
        }
        return annotate(node, {
            key: key.result,
            value: value.result
        }, [
            key,
            value
        ]);
    };
    const dispatch = {
        "stream": singletonBuild,
        "document": singletonBuild,
        "block_node": singletonBuild,
        "flow_node": singletonBuild,
        "double_quote_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "single_quote_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "plain_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "block_scalar": (node)=>{
            return annotate(node, jsYamlParseLenient(node.text), []);
        },
        "block_sequence": (node)=>{
            const result1 = [], components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type !== "block_sequence_item") {
                    continue;
                }
                const component = buildNode(child, node.endIndex);
                components.push(component);
                result1.push(component && component.result);
            }
            return annotate(node, result1, components);
        },
        "block_sequence_item": (node)=>{
            if (node.childCount < 2) {
                return annotateEmpty(node.endIndex);
            } else {
                return buildNode(node.child(1), node.endIndex);
            }
        },
        "flow_sequence": (node)=>{
            const result1 = [], components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type !== "flow_node") {
                    continue;
                }
                const component = buildNode(child, node.endIndex);
                components.push(component);
                result1.push(component.result);
            }
            return annotate(node, result1, components);
        },
        "block_mapping": (node)=>{
            const result1 = {}, components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                let component;
                if (child.type === "ERROR") {
                    result1[child.text] = "<<ERROR>>";
                    const key = annotate(child, child.text, []);
                    const value = annotateEmpty(child.endIndex);
                    component = annotate(child, {
                        key: key.result,
                        value: value.result
                    }, [
                        key,
                        value
                    ]);
                } else if (child.type !== "block_mapping_pair") {
                    continue;
                } else {
                    component = buildNode(child, node.endIndex);
                }
                const { key, value } = component.result;
                result1[String(key)] = value;
                components.push(...component.components);
            }
            return annotate(node, result1, components);
        },
        "flow_pair": buildPair,
        "flow_mapping": (node)=>{
            const result1 = {}, components = [];
            for(let i1 = 0; i1 < node.childCount; ++i1){
                const child = node.child(i1);
                if (child.type === "flow_node") {
                    continue;
                }
                if (child.type === "flow_pair") {
                    const component = buildNode(child, node.endIndex);
                    const { key, value } = component.result;
                    result1[String(key)] = value;
                    components.push(...component.components);
                }
            }
            return annotate(node, result1, components);
        },
        "block_mapping_pair": buildPair
    };
    const result1 = buildNode(tree.rootNode, tree.rootNode.endIndex);
    if (errors.length) {
        result1.errors = errors;
    }
    const parsedSize = tree.rootNode.text.trim().length;
    const codeSize = mappedSource.value.trim().length;
    const lossage = parsedSize / codeSize;
    if (lossage < 0.95) {
        return null;
    }
    return result1;
}
function locateAnnotation(annotation, position, kind) {
    const originalSource = annotation.source;
    kind = kind || "value";
    for(let i1 = 0; i1 < position.length; ++i1){
        const value = position[i1];
        if (typeof value === "number") {
            const inner = annotation.components[value];
            if (inner === undefined) {
                throw new InternalError("invalid path for locateAnnotation");
            }
            annotation = inner;
        } else {
            let found = false;
            for(let j = 0; j < annotation.components.length; j += 2){
                if (originalSource.value.substring(annotation.components[j].start, annotation.components[j].end).trim() === value) {
                    if (i1 === position.length - 1) {
                        if (kind === "key") {
                            annotation = annotation.components[j];
                        } else {
                            annotation = annotation.components[j + 1];
                        }
                    }
                    found = true;
                    break;
                }
            }
            if (!found) {
                throw new InternalError("invalid path for locateAnnotation");
            }
        }
    }
    return annotation;
}
class ValidationError2 extends Error {
    validationErrors;
    constructor(msg, validationErrors){
        super([
            msg,
            ...validationErrors.map((e)=>tidyverseFormatError(e.niceError))
        ].join("\n\n"));
        Object.setPrototypeOf(this, ValidationError2.prototype);
        this.validationErrors = validationErrors;
    }
}
const isObject3 = (value)=>{
    const type = typeof value;
    return value !== null && (type === "object" || type === "function");
};
async function readAndValidateYamlFromMappedString(mappedYaml, schema, pruneErrors = true, lenient = false) {
    const annotation = await readAnnotatedYamlFromMappedString(mappedYaml, lenient);
    if (annotation === null) {
        throw new Error("Parse error in readAnnotatedYamlFromMappedString");
    }
    const validateYaml = !isObject3(annotation.result) || annotation.result["validate-yaml"] !== false;
    if (!validateYaml) {
        return {
            yaml: annotation.result,
            yamlValidationErrors: []
        };
    }
    const validate = async (validator)=>{
        const valResult = await validator.validateParse(mappedYaml, annotation, pruneErrors);
        return {
            yaml: annotation.result,
            yamlValidationErrors: valResult.errors
        };
    };
    if (typeof annotation.result === "object" && !Array.isArray(annotation.result)) {
        const preCheckResult = await withValidator(getSchemaDefinition("bad-parse-schema"), validate);
        if (preCheckResult.yamlValidationErrors.length !== 0) {
            return preCheckResult;
        }
    }
    const result1 = await withValidator(schema, validate);
    return result1;
}
function setBaseSchemaProperties(yaml, schema) {
    if (yaml.additionalCompletions) {
        schema = completeSchema(schema, ...yaml.additionalCompletions);
    }
    if (yaml.completions) {
        schema = completeSchemaOverwrite(schema, ...yaml.completions);
    }
    if (yaml.id) {
        schema = idSchema(schema, yaml.id);
    }
    if (yaml.hidden === true) {
        schema = completeSchemaOverwrite(schema);
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    if (yaml.tags) {
        schema = tagSchema(schema, yaml.tags);
    }
    if (yaml.description) {
        schema = tagSchema(schema, {
            description: yaml.description
        });
        if (typeof yaml.description === "string") {
            schema = documentSchema(schema, yaml.description);
        } else if (typeof yaml.description === "object") {
            schema = documentSchema(schema, yaml.description.short);
        }
    }
    const result1 = Object.assign({}, schema);
    if (yaml.errorDescription) {
        result1.description = yaml.errorDescription;
    }
    if (yaml.errorMessage) {
        result1.errorMessage = yaml.errorMessage;
    }
    return result1;
}
function convertFromNull(yaml) {
    return setBaseSchemaProperties(yaml["null"], nullSchema);
}
function convertFromSchema(yaml) {
    const schema = convertFromYaml(yaml.schema);
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromString(yaml) {
    if (yaml["string"].pattern) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], regexSchema(yaml["string"].pattern)));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["string"], stringSchema));
    }
}
function convertFromPattern(yaml) {
    if (typeof yaml.pattern === "string") {
        return setBaseSchemaProperties(yaml, regexSchema(yaml.pattern));
    } else {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.pattern, regexSchema(yaml.pattern.regex)));
    }
}
function convertFromPath(yaml) {
    return setBaseSchemaProperties(yaml["path"], stringSchema);
}
function convertFromNumber(yaml) {
    return setBaseSchemaProperties(yaml["number"], numberSchema);
}
function convertFromBoolean(yaml) {
    return setBaseSchemaProperties(yaml["boolean"], booleanSchema);
}
function convertFromRef(yaml) {
    return setBaseSchemaProperties(yaml, refSchema(yaml.ref, `be ${yaml.ref}`));
}
function convertFromMaybeArrayOf(yaml) {
    const inner = convertFromYaml(yaml.maybeArrayOf);
    const schema = tagSchema(anyOfSchema(inner, arraySchema(inner)), {
        "complete-from": [
            "anyOf",
            0
        ]
    });
    return setBaseSchemaProperties(yaml, schema);
}
function convertFromArrayOf(yaml) {
    if (yaml.arrayOf.schema) {
        const result1 = arraySchema(convertFromYaml(yaml.arrayOf.schema));
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.arrayOf, result1));
    } else {
        return setBaseSchemaProperties(yaml, arraySchema(convertFromYaml(yaml.arrayOf)));
    }
}
function convertFromAllOf(yaml) {
    if (yaml.allOf.schemas) {
        const inner = yaml.allOf.schemas.map((x)=>convertFromYaml(x));
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.allOf, schema));
    } else {
        const inner = yaml.allOf.map((x)=>convertFromYaml(x));
        const schema = allOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromAnyOf(yaml) {
    if (yaml.anyOf.schemas) {
        const inner = yaml.anyOf.schemas.map((x)=>convertFromYaml(x));
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.anyOf, schema));
    } else {
        const inner = yaml.anyOf.map((x)=>convertFromYaml(x));
        const schema = anyOfSchema(...inner);
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromEnum(yaml) {
    const schema = yaml["enum"];
    if (schema.hasOwnProperty("values")) {
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml["enum"], enumSchema(...schema.values)));
    } else {
        return setBaseSchemaProperties(yaml, enumSchema(...schema));
    }
}
function convertFromRecord(yaml) {
    if (yaml.record.properties) {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record.properties,
                "closed": true,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, setBaseSchemaProperties(yaml.record, schema));
    } else {
        const schema = convertFromObject({
            "object": {
                "properties": yaml.record,
                "closed": true,
                "required": "all"
            }
        });
        return setBaseSchemaProperties(yaml, schema);
    }
}
function convertFromObject(yaml) {
    const schema = yaml["object"];
    const params = {};
    if (schema.namingConvention && typeof schema.namingConvention === "string") {
        switch(schema.namingConvention){
            case "ignore":
                params.namingConvention = "ignore";
                break;
            case "capitalizationCase":
                params.namingConvention = "capitalizationCase";
                break;
            case "capitalization-case":
                params.namingConvention = "capitalizationCase";
                break;
            case "capitalization_case":
                params.namingConvention = "capitalizationCase";
                break;
            case "underscoreCase":
                params.namingConvention = "underscore_case";
                break;
            case "underscore-case":
                params.namingConvention = "underscore_case";
                break;
            case "underscore_case":
                params.namingConvention = "underscore_case";
                break;
            case "dashCase":
                params.namingConvention = "dash-case";
                break;
            case "dash-case":
                params.namingConvention = "dash-case";
                break;
            case "dash_case":
                params.namingConvention = "dash-case";
                break;
            case "camelCase":
                params.namingConvention = "capitalizationCase";
                break;
            case "camel-case":
                params.namingConvention = "capitalizationCase";
                break;
            case "camel_case":
                params.namingConvention = "capitalizationCase";
                break;
            case "snakeCase":
                params.namingConvention = "underscore_case";
                break;
            case "snake-case":
                params.namingConvention = "underscore_case";
                break;
            case "snake_case":
                params.namingConvention = "underscore_case";
                break;
            case "kebabCase":
                params.namingConvention = "dash-case";
                break;
            case "kebab-case":
                params.namingConvention = "dash-case";
                break;
            case "kebab_case":
                params.namingConvention = "dash-case";
                break;
            default:
                throw new InternalError(`Unrecognized naming convention ${schema.namingConvention} should have failed validation`);
        }
    } else {
        params.namingConvention = schema.namingConvention;
    }
    if (schema.properties) {
        params.properties = fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]));
    }
    if (schema.patternProperties) {
        params.patternProperties = fromEntries(Object.entries(schema.properties).map(([key, value])=>[
                key,
                convertFromYaml(value)
            ]));
    }
    if (schema.propertyNames !== undefined) {
        params.propertyNames = convertFromYaml(schema.propertyNames);
    } else if (schema.closed === true) {
        const objectKeys = Object.keys(params.properties || {});
        if (objectKeys.length === 0) {
            throw new Error("object schema `closed` requires field `properties`.");
        }
        if (params.namingConvention !== undefined && params.namingConvention !== "ignore") {
            throw new Error("object schema `closed` is only supported with namingConvention: `ignore`");
        }
        params.namingConvention = "ignore";
        params.closed = true;
    }
    if (schema.additionalProperties !== undefined) {
        if (schema.additionalProperties === false) {
            params.additionalProperties = false;
        } else {
            params.additionalProperties = convertFromYaml(schema.additionalProperties);
        }
    }
    if (schema["super"]) {
        if (Array.isArray(schema["super"])) {
            params.baseSchema = schema["super"].map((s)=>convertFromYaml(s));
        } else {
            params.baseSchema = convertFromYaml(schema["super"]);
        }
    }
    if (schema["required"] === "all") {
        params.required = Object.keys(schema.properties || {});
    } else if (schema["required"]) {
        params.required = schema["required"];
    }
    if (schema["completions"]) {
        params.completions = schema["completions"];
    }
    return setBaseSchemaProperties(yaml, setBaseSchemaProperties(schema, objectSchema(params)));
}
function lookup(yaml) {
    if (!hasSchemaDefinition(yaml.resolveRef)) {
        throw new Error(`lookup of key ${yaml.resolveRef} in definitions failed`);
    }
    return getSchemaDefinition(yaml.resolveRef);
}
function convertFromYaml(yaml) {
    const literalValues = [
        {
            val: "object",
            schema: objectSchema()
        },
        {
            val: "path",
            schema: stringSchema
        },
        {
            val: "string",
            schema: stringSchema
        },
        {
            val: "number",
            schema: numberSchema
        },
        {
            val: "boolean",
            schema: booleanSchema
        },
        {
            val: "any",
            schema: anySchema()
        },
        {
            val: null,
            schema: nullSchema
        }
    ];
    for (const { val, schema } of literalValues){
        if (yaml === val) {
            return schema;
        }
    }
    if (typeof yaml !== "object") {
        return valueSchema(yaml);
    }
    const schemaObjectKeyFunctions = [
        {
            key: "anyOf",
            value: convertFromAnyOf
        },
        {
            key: "allOf",
            value: convertFromAllOf
        },
        {
            key: "boolean",
            value: convertFromBoolean
        },
        {
            key: "arrayOf",
            value: convertFromArrayOf
        },
        {
            key: "enum",
            value: convertFromEnum
        },
        {
            key: "maybeArrayOf",
            value: convertFromMaybeArrayOf
        },
        {
            key: "null",
            value: convertFromNull
        },
        {
            key: "number",
            value: convertFromNumber
        },
        {
            key: "object",
            value: convertFromObject
        },
        {
            key: "path",
            value: convertFromPath
        },
        {
            key: "record",
            value: convertFromRecord
        },
        {
            key: "ref",
            value: convertFromRef
        },
        {
            key: "resolveRef",
            value: lookup
        },
        {
            key: "string",
            value: convertFromString
        },
        {
            key: "pattern",
            value: convertFromPattern
        },
        {
            key: "schema",
            value: convertFromSchema
        }
    ];
    for (const { key: objectKey, value: fun } of schemaObjectKeyFunctions){
        if (yaml[objectKey] !== undefined) {
            return fun(yaml);
        }
    }
    throw new InternalError("Cannot convert object; this should have failed validation.");
}
function objectSchemaFromFieldsObject(fields, exclude) {
    exclude = exclude || ((_key)=>false);
    const properties = {};
    convertFromFieldsObject(fields, properties);
    for (const key of Object.keys(properties)){
        if (exclude(key)) {
            delete properties[key];
        }
    }
    return objectSchema({
        properties
    });
}
function annotateSchemaFromField(field, schema) {
    if (field.enabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.enabled
        });
    }
    if (field.disabled !== undefined) {
        schema = tagSchema(schema, {
            formats: field.disabled.map((x)=>`!${x}`)
        });
    }
    if (field.tags) {
        schema = tagSchema(schema, field.tags);
    }
    if (field.description) {
        if (typeof field.description === "string") {
            schema = documentSchema(schema, field.description);
        } else if (typeof field.description === "object") {
            schema = documentSchema(schema, field.description.short);
        }
        schema = tagSchema(schema, {
            description: field.description
        });
    }
    if (field.hidden) {
        schema = tagSchema(schema, {
            "hidden": true
        });
    }
    return schema;
}
function schemaFromField(entry) {
    const schema = convertFromYaml(entry.schema);
    return annotateSchemaFromField(entry, schema);
}
function convertFromFieldsObject(yaml, obj) {
    const result1 = obj || {};
    for (const field of yaml){
        let schema = convertFromYaml(field.schema);
        schema = annotateSchemaFromField(field, schema);
        result1[field.name] = schema;
        if (field.alias) {
            result1[field.alias] = schema;
        }
    }
    return result1;
}
function schemaFieldsFromGlob(globPath, testFun) {
    const result1 = [];
    testFun = testFun || ((_e, _p)=>true);
    for (const [file, fields] of expandResourceGlob(globPath)){
        for (const field of fields){
            const fieldName = field.name;
            const schemaId = `quarto-resource-${file.split("/").slice(-1)[0].slice(0, -4)}-${fieldName}`;
            if (testFun(field, file)) {
                result1.push({
                    schemaId,
                    field
                });
            }
        }
    }
    return result1;
}
memoize1(()=>{
    const groups = getYamlIntelligenceResource("schema/groups.yml");
    const result1 = [];
    for (const [topLevel, sub] of Object.entries(groups)){
        for (const key of Object.keys(sub)){
            result1.push(`${topLevel}-${key}`);
        }
    }
    return result1;
}, ()=>"const");
function objectRefSchemaFromContextGlob(contextGlob, testFun) {
    const regexp = globToRegExp3(contextGlob);
    return objectRefSchemaFromGlob("schema/{document,cell}-*.yml", (field, path)=>{
        if (testFun !== undefined && !testFun(field, path)) {
            return false;
        }
        const pathContext = path.split("/").slice(-1)[0].slice(0, -4);
        const schemaContexts = field !== undefined && field.tags !== undefined && field.tags.contexts || [];
        if (pathContext.match(regexp)) {
            return true;
        }
        return schemaContexts.some((c)=>c.match(regexp));
    });
}
function objectRefSchemaFromGlob(glob, testFun) {
    const properties = {};
    for (const { schemaId, field } of schemaFieldsFromGlob(glob, testFun)){
        const schema = refSchema(schemaId, schemaId);
        properties[field.name] = schema;
        if (field.alias) {
            properties[field.alias] = schema;
        }
    }
    return objectSchema({
        properties
    });
}
async function buildResourceSchemas() {
    const path = "schema/{cell-*,document-*,project}.yml";
    for (const [file, fields] of expandResourceGlob(path)){
        const yaml = fields;
        const entries = Object.entries(convertFromFieldsObject(yaml));
        for (const [fieldName, fieldSchema] of entries){
            const schemaId = `quarto-resource-${file.split("/").slice(-1)[0].slice(0, -4)}-${fieldName}`;
            const schema = idSchema(fieldSchema, schemaId);
            setSchemaDefinition(schema);
            await withValidator(schema, async (_validator)=>{});
        }
    }
}
function defineCached(thunk, schemaId) {
    let schema;
    return async ()=>{
        if (hasSchemaDefinition(schemaId)) {
            schema = getSchemaDefinition(schemaId);
            return refSchema(schema.$id, schema.description || `be a {schema['$id'] as string}`);
        }
        const result1 = await thunk();
        const { errorHandlers } = result1;
        schema = result1.schema;
        if (schemaId !== schema.$id) {
            schema = idSchema(schema, schemaId);
        }
        define1(schema);
        for (const fun of errorHandlers){
            addValidatorErrorHandler(schema, fun);
        }
        return refSchema(schema.$id, schema.description || `be a {schema['$id']}`);
    };
}
function define1(schema) {
    if (schema !== true && schema !== false && schema.$id && !hasSchemaDefinition(schema.$id)) {
        setSchemaDefinition(schema);
    }
}
async function loadDefaultSchemaDefinitions() {
    await loadSchemaDefinitions(getYamlIntelligenceResource("schema/definitions.yml"));
    await buildResourceSchemas();
}
async function loadSchemaDefinitions(yaml) {
    await Promise.all(yaml.map(async (yamlSchema)=>{
        const schema = convertFromYaml(yamlSchema);
        if (schema.$id === undefined) {
            throw new InternalError(`Unnamed schema in definitions`);
        }
        setSchemaDefinition(schema);
    }));
}
function getFormatExecuteOptionsSchema() {
    const schema = idSchema(objectRefSchemaFromContextGlob("document-execute"), "front-matter-execute");
    define1(schema);
    return refSchema("front-matter-execute", "be a front-matter-execute object");
}
let formatAliases = undefined;
function getFormatAliases() {
    if (formatAliases !== undefined) {
        return formatAliases;
    }
    formatAliases = getYamlIntelligenceResource("schema/format-aliases.yml").aliases;
    return formatAliases;
}
function expandFormatAliases(lst) {
    return expandAliasesFrom(lst, getFormatAliases());
}
function useSchema(schema, format) {
    const formats = schema && schema.tags && schema.tags.formats;
    if (formats === undefined) {
        return true;
    }
    const disabled = formats.filter((f)=>f.startsWith("!")).map((f)=>f.slice(1));
    const enabled = formats.filter((f)=>!f.startsWith("!"));
    if (disabled.length > 0 && expandFormatAliases(disabled).indexOf(format) !== -1) {
        return false;
    }
    if (enabled.length > 0 && expandFormatAliases(enabled).indexOf(format) === -1) {
        return false;
    }
    return true;
}
function getFormatSchema(format) {
    const schema = objectRefSchemaFromContextGlob("document-*", (field)=>{
        const schema = schemaFromField(field);
        return useSchema(schema, format);
    });
    return anyOfSchema(schema, enumSchema("default"));
}
function pandocFormatsResource() {
    return getYamlIntelligenceResource("pandoc/formats.yml");
}
async function makeFrontMatterFormatSchema(nonStrict = false) {
    const hideFormat = (format)=>{
        const hideList = [
            "html",
            "epub",
            "docbook"
        ];
        const hidden = hideList.some((h)=>format.startsWith(h) && format.length > h.length);
        return {
            name: format,
            hidden
        };
    };
    const formatSchemaDescriptorList = (await pandocFormatsResource()).concat("md", "hugo", "dashboard", "email").map((format)=>{
        const { name, hidden } = hideFormat(format);
        return {
            regex: `^(.+-)?${name}([-+].+)?$`,
            schema: getFormatSchema(name),
            name,
            hidden
        };
    });
    const formatSchemas = formatSchemaDescriptorList.map(({ regex, schema })=>[
            regex,
            schema
        ]);
    const plusFormatStringSchemas = formatSchemaDescriptorList.map(({ regex, name, hidden })=>{
        const schema = regexSchema(regex, `be '${name}'`);
        if (hidden) {
            return schema;
        }
        return completeSchema(schema, name);
    });
    const luaFilenameS = regexSchema("^.+\.lua$");
    plusFormatStringSchemas.push(luaFilenameS);
    const completionsObject = fromEntries(formatSchemaDescriptorList.filter(({ hidden })=>!hidden).map(({ name })=>[
            name,
            {
                type: "key",
                display: name,
                value: `${name}: `,
                description: `be '${name}'`,
                suggest_on_accept: true
            }
        ]));
    return errorMessageSchema(anyOfSchema(describeSchema(anyOfSchema(...plusFormatStringSchemas), "the name of a pandoc-supported output format"), objectSchema({
        propertyNames: luaFilenameS
    }), allOfSchema(objectSchema({
        patternProperties: fromEntries(formatSchemas),
        completions: completionsObject,
        additionalProperties: nonStrict
    }))), "${value} is not a valid output format.");
}
const getFrontMatterFormatSchema = defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(),
        errorHandlers: []
    };
}, "front-matter-format");
defineCached(async ()=>{
    return {
        schema: await makeFrontMatterFormatSchema(true),
        errorHandlers: []
    };
}, "front-matter-format-nonstrict");
const getFrontMatterSchema = defineCached(async ()=>{
    const executeObjSchema = await getFormatExecuteOptionsSchema();
    return {
        schema: anyOfSchema(nullSchema, allOfSchema(objectSchema({
            properties: {
                execute: executeObjSchema,
                format: await getFrontMatterFormatSchema()
            },
            description: "be a Quarto YAML front matter object"
        }), objectRefSchemaFromContextGlob("document-*", (field)=>field.name !== "format"), executeObjSchema, refSchema("quarto-dev-schema", ""))),
        errorHandlers: []
    };
}, "front-matter");
function Diff() {}
Diff.prototype = {
    diff: function diff(oldString, newString) {
        var options = arguments.length > 2 && arguments[2] !== undefined ? arguments[2] : {};
        var callback = options.callback;
        if (typeof options === 'function') {
            callback = options;
            options = {};
        }
        this.options = options;
        var self1 = this;
        function done(value) {
            return value;
        }
        oldString = this.castInput(oldString);
        newString = this.castInput(newString);
        oldString = this.removeEmpty(this.tokenize(oldString));
        newString = this.removeEmpty(this.tokenize(newString));
        var newLen = newString.length, oldLen = oldString.length;
        var editLength = 1;
        var maxEditLength = newLen + oldLen;
        if (options.maxEditLength) {
            maxEditLength = Math.min(maxEditLength, options.maxEditLength);
        }
        var bestPath = [
            {
                newPos: -1,
                components: []
            }
        ];
        var oldPos = this.extractCommon(bestPath[0], newString, oldString, 0);
        if (bestPath[0].newPos + 1 >= newLen && oldPos + 1 >= oldLen) {
            return done([
                {
                    value: this.join(newString),
                    count: newString.length
                }
            ]);
        }
        function execEditLength() {
            for(var diagonalPath = -1 * editLength; diagonalPath <= editLength; diagonalPath += 2){
                var basePath = void 0;
                var addPath = bestPath[diagonalPath - 1], removePath = bestPath[diagonalPath + 1], _oldPos = (removePath ? removePath.newPos : 0) - diagonalPath;
                if (addPath) {
                    bestPath[diagonalPath - 1] = undefined;
                }
                var canAdd = addPath && addPath.newPos + 1 < newLen, canRemove = removePath && 0 <= _oldPos && _oldPos < oldLen;
                if (!canAdd && !canRemove) {
                    bestPath[diagonalPath] = undefined;
                    continue;
                }
                if (!canAdd || canRemove && addPath.newPos < removePath.newPos) {
                    basePath = clonePath(removePath);
                    self1.pushComponent(basePath.components, undefined, true);
                } else {
                    basePath = addPath;
                    basePath.newPos++;
                    self1.pushComponent(basePath.components, true, undefined);
                }
                _oldPos = self1.extractCommon(basePath, newString, oldString, diagonalPath);
                if (basePath.newPos + 1 >= newLen && _oldPos + 1 >= oldLen) {
                    return done(buildValues(self1, basePath.components, newString, oldString, self1.useLongestToken));
                } else {
                    bestPath[diagonalPath] = basePath;
                }
            }
            editLength++;
        }
        while(editLength <= maxEditLength){
            var ret = execEditLength();
            if (callback && callback(ret, editLength, maxEditLength)) {
                return ret;
            }
            if (ret) {
                return ret;
            }
        }
    },
    pushComponent: function pushComponent(components, added, removed) {
        var last = components[components.length - 1];
        if (last && last.added === added && last.removed === removed) {
            components[components.length - 1] = {
                count: last.count + 1,
                added: added,
                removed: removed
            };
        } else {
            components.push({
                count: 1,
                added: added,
                removed: removed
            });
        }
    },
    extractCommon: function extractCommon(basePath, newString, oldString, diagonalPath) {
        var newLen = newString.length, oldLen = oldString.length, newPos = basePath.newPos, oldPos = newPos - diagonalPath, commonCount = 0;
        while(newPos + 1 < newLen && oldPos + 1 < oldLen && this.equals(newString[newPos + 1], oldString[oldPos + 1])){
            newPos++;
            oldPos++;
            commonCount++;
        }
        if (commonCount) {
            basePath.components.push({
                count: commonCount
            });
        }
        basePath.newPos = newPos;
        return oldPos;
    },
    equals: function equals(left, right) {
        if (this.options.comparator) {
            return this.options.comparator(left, right);
        } else {
            return left === right || this.options.ignoreCase && left.toLowerCase() === right.toLowerCase();
        }
    },
    removeEmpty: function removeEmpty(array) {
        var ret = [];
        for(var i1 = 0; i1 < array.length; i1++){
            if (array[i1]) {
                ret.push(array[i1]);
            }
        }
        return ret;
    },
    castInput: function castInput(value) {
        return value;
    },
    tokenize: function tokenize(value) {
        return value.split('');
    },
    join: function join(chars) {
        return chars.join('');
    }
};
function buildValues(diff, components, newString, oldString, useLongestToken) {
    var componentPos = 0, componentLen = components.length, newPos = 0, oldPos = 0;
    for(; componentPos < componentLen; componentPos++){
        var component = components[componentPos];
        if (!component.removed) {
            if (!component.added && useLongestToken) {
                var value = newString.slice(newPos, newPos + component.count);
                value = value.map(function(value, i1) {
                    var oldValue = oldString[oldPos + i1];
                    return oldValue.length > value.length ? oldValue : value;
                });
                component.value = diff.join(value);
            } else {
                component.value = diff.join(newString.slice(newPos, newPos + component.count));
            }
            newPos += component.count;
            if (!component.added) {
                oldPos += component.count;
            }
        } else {
            component.value = diff.join(oldString.slice(oldPos, oldPos + component.count));
            oldPos += component.count;
            if (componentPos && components[componentPos - 1].added) {
                var tmp = components[componentPos - 1];
                components[componentPos - 1] = components[componentPos];
                components[componentPos] = tmp;
            }
        }
    }
    var lastComponent = components[componentLen - 1];
    if (componentLen > 1 && typeof lastComponent.value === 'string' && (lastComponent.added || lastComponent.removed) && diff.equals('', lastComponent.value)) {
        components[componentLen - 2].value += lastComponent.value;
        components.pop();
    }
    return components;
}
function clonePath(path) {
    return {
        newPos: path.newPos,
        components: path.components.slice(0)
    };
}
new Diff();
var extendedWordChars = /^[A-Za-z\xC0-\u02C6\u02C8-\u02D7\u02DE-\u02FF\u1E00-\u1EFF]+$/;
var reWhitespace1 = /\S/;
var wordDiff = new Diff();
wordDiff.equals = function(left, right) {
    if (this.options.ignoreCase) {
        left = left.toLowerCase();
        right = right.toLowerCase();
    }
    return left === right || this.options.ignoreWhitespace && !reWhitespace1.test(left) && !reWhitespace1.test(right);
};
wordDiff.tokenize = function(value) {
    var tokens = value.split(/([^\S\r\n]+|[()[\]{}'"\r\n]|\b)/);
    for(var i1 = 0; i1 < tokens.length - 1; i1++){
        if (!tokens[i1 + 1] && tokens[i1 + 2] && extendedWordChars.test(tokens[i1]) && extendedWordChars.test(tokens[i1 + 2])) {
            tokens[i1] += tokens[i1 + 2];
            tokens.splice(i1 + 1, 2);
            i1--;
        }
    }
    return tokens;
};
var lineDiff = new Diff();
lineDiff.tokenize = function(value) {
    var retLines = [], linesAndNewlines = value.split(/(\n|\r\n)/);
    if (!linesAndNewlines[linesAndNewlines.length - 1]) {
        linesAndNewlines.pop();
    }
    for(var i1 = 0; i1 < linesAndNewlines.length; i1++){
        var line = linesAndNewlines[i1];
        if (i1 % 2 && !this.options.newlineIsToken) {
            retLines[retLines.length - 1] += line;
        } else {
            if (this.options.ignoreWhitespace) {
                line = line.trim();
            }
            retLines.push(line);
        }
    }
    return retLines;
};
function diffLines(oldStr, newStr, callback) {
    return lineDiff.diff(oldStr, newStr, callback);
}
var sentenceDiff = new Diff();
sentenceDiff.tokenize = function(value) {
    return value.split(/(\S.+?[.!?])(?=\s+|$)/);
};
var cssDiff = new Diff();
cssDiff.tokenize = function(value) {
    return value.split(/([{}:;,]|\s+)/);
};
function _typeof(obj) {
    "@babel/helpers - typeof";
    if (typeof Symbol === "function" && typeof Symbol.iterator === "symbol") {
        _typeof = function(obj) {
            return typeof obj;
        };
    } else {
        _typeof = function(obj) {
            return obj && typeof Symbol === "function" && obj.constructor === Symbol && obj !== Symbol.prototype ? "symbol" : typeof obj;
        };
    }
    return _typeof(obj);
}
var objectPrototypeToString = Object.prototype.toString;
var jsonDiff = new Diff();
jsonDiff.useLongestToken = true;
jsonDiff.tokenize = lineDiff.tokenize;
jsonDiff.castInput = function(value) {
    var _this$options = this.options, undefinedReplacement = _this$options.undefinedReplacement, _this$options$stringi = _this$options.stringifyReplacer, stringifyReplacer = _this$options$stringi === void 0 ? function(k, v) {
        return typeof v === 'undefined' ? undefinedReplacement : v;
    } : _this$options$stringi;
    return typeof value === 'string' ? value : JSON.stringify(canonicalize(value, null, null, stringifyReplacer), stringifyReplacer, '  ');
};
jsonDiff.equals = function(left, right) {
    return Diff.prototype.equals.call(jsonDiff, left.replace(/,([\r\n])/g, '$1'), right.replace(/,([\r\n])/g, '$1'));
};
function canonicalize(obj, stack, replacementStack, replacer, key) {
    stack = stack || [];
    replacementStack = replacementStack || [];
    if (replacer) {
        obj = replacer(key, obj);
    }
    var i1;
    for(i1 = 0; i1 < stack.length; i1 += 1){
        if (stack[i1] === obj) {
            return replacementStack[i1];
        }
    }
    var canonicalizedObj;
    if ('[object Array]' === objectPrototypeToString.call(obj)) {
        stack.push(obj);
        canonicalizedObj = new Array(obj.length);
        replacementStack.push(canonicalizedObj);
        for(i1 = 0; i1 < obj.length; i1 += 1){
            canonicalizedObj[i1] = canonicalize(obj[i1], stack, replacementStack, replacer, key);
        }
        stack.pop();
        replacementStack.pop();
        return canonicalizedObj;
    }
    if (obj && obj.toJSON) {
        obj = obj.toJSON();
    }
    if (_typeof(obj) === 'object' && obj !== null) {
        stack.push(obj);
        canonicalizedObj = {};
        replacementStack.push(canonicalizedObj);
        var sortedKeys = [], _key;
        for(_key in obj){
            if (obj.hasOwnProperty(_key)) {
                sortedKeys.push(_key);
            }
        }
        sortedKeys.sort();
        for(i1 = 0; i1 < sortedKeys.length; i1 += 1){
            _key = sortedKeys[i1];
            canonicalizedObj[_key] = canonicalize(obj[_key], stack, replacementStack, replacer, _key);
        }
        stack.pop();
        replacementStack.pop();
    } else {
        canonicalizedObj = obj;
    }
    return canonicalizedObj;
}
var arrayDiff = new Diff();
arrayDiff.tokenize = function(value) {
    return value.slice();
};
arrayDiff.join = arrayDiff.removeEmpty = function(value) {
    return value;
};
const nodeStack = [
    {
        name: "global",
        children: [
            {
                name: "deno-init",
                start: 0,
                end: performance.now(),
                elapsed: performance.now(),
                children: []
            }
        ],
        start: 0,
        end: 0
    }
];
function withTiming(name, fun) {
    push(name);
    try {
        const result1 = fun();
        return result1;
    } finally{
        pop();
    }
}
async function withTimingAsync(name, fun) {
    push(name);
    try {
        const result1 = await fun();
        pop();
        return result1;
    } catch (e) {
        pop();
        throw e;
    }
}
function push(name) {
    nodeStack.push({
        name,
        start: performance.now(),
        end: 0,
        children: []
    });
}
function pop(at) {
    if (nodeStack.length <= 1) {
        throw new Error("Can't pop top node");
    }
    const node = nodeStack.pop();
    node.end = at ?? performance.now();
    node.elapsed = node.end - node.start;
    nodeStack[nodeStack.length - 1].children.push(node);
}
function getData() {
    nodeStack[0].end = performance.now();
    nodeStack[0].elapsed = nodeStack[0].end - nodeStack[0].start;
    return nodeStack[0];
}
function insertExplicitTimingEntries(start, end, entries, groupName) {
    const [theirStart, ourStart] = start;
    const [theirEnd, _] = end;
    nodeStack.push({
        name: groupName,
        start: ourStart,
        end: 0,
        children: []
    });
    let prevTime = ourStart;
    if (entries.length && entries[0].name === "_start") {
        prevTime = entries[0].time - theirStart + ourStart;
        entries = entries.slice(1);
    }
    for (const entry of entries){
        nodeStack.push({
            name: entry.name,
            start: prevTime,
            end: 0,
            children: []
        });
        prevTime = entry.time - theirStart + ourStart;
        pop(prevTime);
    }
    prevTime = theirEnd - theirStart + ourStart;
    pop(prevTime);
}
async function getLuaTiming() {
    if (Deno.env.get("QUARTO_PROFILER_OUTPUT")) {
        return [
            Number((await execProcess({
                cmd: [
                    "python",
                    "-c",
                    "import time; print(time.time() * 1000)"
                ],
                stdout: "piped"
            })).stdout),
            performance.now()
        ];
    } else {
        return [
            0,
            0
        ];
    }
}
function mappedDiff(source, target) {
    return withTiming("mapped-diff", ()=>{
        const sourceLineRanges = rangedLines(source.value).map((x)=>x.range);
        let sourceCursor = 0;
        const resultChunks = [];
        const started = performance.now();
        const diffResult = diffLines(source.value, target, ()=>{
            const now = performance.now();
            if (now - started > 200) {
                return true;
            }
        });
        if (diffResult === undefined) {
            debug("The computation used to determine source line information from the engine timed out.\nLine number information will be unavailable and/or inaccurate.");
            return asMappedString(target);
        }
        for (const action of diffResult){
            if (action.removed) {
                sourceCursor += action.count;
            } else if (action.added) {
                resultChunks.push(action.value);
            } else {
                const start = sourceLineRanges[sourceCursor].start;
                const nextCursor = sourceCursor + action.count;
                const end = nextCursor < sourceLineRanges.length ? sourceLineRanges[nextCursor].start : sourceLineRanges[sourceLineRanges.length - 1].end;
                sourceCursor = nextCursor;
                resultChunks.push({
                    start,
                    end
                });
            }
        }
        return mappedString(source, resultChunks, source.fileName);
    });
}
function mappedStringFromFile(filename) {
    const value = Deno.readTextFileSync(filename);
    if (filename.startsWith("/")) {
        filename = relative2(Deno.cwd(), filename);
    }
    return asMappedString(value, filename);
}
const kRegExBeginYAML = /^---[ \t]*$/;
const kRegExEndYAML = /^(?:---|\.\.\.)([ \t]*)$/;
const kRegExYAML = /(^)(---[ \t]*[\r\n]+(?![ \t]*[\r\n]+)[\W\w]*?[\r\n]+(?:---|\.\.\.))([ \t]*)$/gm;
const kRegxHTMLComment = /<!--[\W\w]*?-->/gm;
const kRegexFencedCode = /^([\t >]*`{3,})[^`\n]*\n[\W\w]*?\n\1\s*$/gm;
function isYamlPath(file) {
    return [
        ".yaml",
        ".yml"
    ].includes(extname2(file));
}
function readYaml(file) {
    if (existsSync(file)) {
        const decoder = new TextDecoder("utf-8");
        const yml = Deno.readFileSync(file);
        const result1 = parseWithNiceErrors(decoder.decode(yml));
        try {
            JSON.stringify(result1);
            return result1;
        } catch (e) {
            throw new Error(`Circular structures not allowed.\nFile ${file}\n${e.message.split("\n").slice(1).join("\n")}`);
        }
    } else {
        throw new Error(`YAML file ${file} not found.`);
    }
}
function readYamlFromString(yml) {
    return parseWithNiceErrors(yml);
}
function readYamlFromMarkdown(markdown) {
    if (markdown) {
        markdown = normalizeNewlines(markdown);
        markdown = markdown.replaceAll(kRegxHTMLComment, "");
        markdown = markdown.replaceAll(kRegexFencedCode, "");
        let yaml = "";
        kRegExYAML.lastIndex = 0;
        let match = kRegExYAML.exec(markdown);
        while(match != null){
            let yamlBlock = removeYamlDelimiters(match[2]);
            yamlBlock = lines(yamlBlock).map((x)=>x.trimEnd()).join("\n");
            if (!yamlBlock.match(/^\n\s*\n/) && !yamlBlock.match(/^\n\s*\n---/m) && yamlBlock.trim().length > 0) {
                parseWithNiceErrors(yamlBlock, {
                    json: true,
                    schema: QuartoJSONSchema1
                });
                yaml += yamlBlock;
            }
            match = kRegExYAML.exec(markdown);
        }
        kRegExYAML.lastIndex = 0;
        const metadata = parseWithNiceErrors(yaml, {
            json: true,
            schema: QuartoJSONSchema1
        });
        return metadata || {};
    } else {
        return {};
    }
}
function partitionYamlFrontMatter(markdown) {
    const mdLines = lines(markdown.trimLeft());
    if (mdLines.length < 3 || !mdLines[0].match(kRegExBeginYAML)) {
        return null;
    } else if (mdLines[1].trim().length === 0 || mdLines[1].match(kRegExEndYAML)) {
        return null;
    } else {
        const endYamlPos = mdLines.findIndex((line, index)=>index > 0 && line.match(kRegExEndYAML));
        if (endYamlPos === -1) {
            return null;
        } else {
            return {
                yaml: mdLines.slice(0, endYamlPos + 1).join("\n"),
                markdown: "\n" + mdLines.slice(endYamlPos + 1).join("\n")
            };
        }
    }
}
function removeYamlDelimiters(yaml) {
    return yaml.replace(/^---/, "").replace(/---\s*$/, "");
}
class YAMLValidationError extends ErrorEx {
    constructor(message){
        super("YAMLValidationError", message, false, false);
    }
}
const QuartoJSONSchema1 = new Schema({
    implicit: [
        nil,
        bool,
        __int,
        __float
    ],
    include: [
        failsafe
    ],
    explicit: [
        new Type1("!expr", {
            kind: "scalar",
            construct (data) {
                const result1 = data !== null ? data : "";
                return {
                    value: result1,
                    tag: "!expr"
                };
            }
        })
    ]
});
function parseWithNiceErrors(content, options) {
    try {
        return parse1(content, options || {
            json: true,
            schema: QuartoJSONSchema1
        });
    } catch (e) {
        throw improveYamlParseErrorMessage(e);
    }
}
function improveYamlParseErrorMessage(e) {
    if (e.message.match(/unknown tag/)) {
        e.message = `${e.message}\nDid you try to use a '!' in a YAML string? If so, you need to add explicit quotes to your string.`;
    }
    return e;
}
function mergeConfigs(config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, mergeArrayCustomizer);
}
function mergeArrayCustomizer(objValue, srcValue) {
    if (__VIRTUAL_FILE53(objValue) || __VIRTUAL_FILE53(srcValue)) {
        if (!objValue) {
            return srcValue;
        } else if (!srcValue) {
            return objValue;
        } else {
            if (!__VIRTUAL_FILE53(objValue)) {
                objValue = [
                    objValue
                ];
            }
            if (!__VIRTUAL_FILE53(srcValue)) {
                srcValue = [
                    srcValue
                ];
            }
        }
        const combined = objValue.concat(srcValue);
        return __VIRTUAL_FILE210(combined, (value)=>{
            if (typeof value === "function") {
                return globalThis.crypto.randomUUID();
            } else {
                return JSON.stringify(value);
            }
        });
    }
}
function camelToKebab(camel) {
    camel = camel.replace(/([A-Z])([A-Z]+)$/g, function(_, p1, p2) {
        return p1 + p2.toLowerCase();
    });
    camel = camel.replaceAll(/([A-Z])([A-Z]+)([A-Z])/g, function(_, p1, p2, p3) {
        return p1 + p2.toLowerCase() + p3;
    });
    const kebab = [];
    for(let i1 = 0; i1 < camel.length; i1++){
        const ch = camel.charAt(i1);
        if (ch === ch.toUpperCase() && !/^\d+/.test(ch)) {
            if (i1 > 0) {
                kebab.push("-");
            }
            kebab.push(ch.toLowerCase());
        } else {
            kebab.push(ch);
        }
    }
    return kebab.join("");
}
function kebabToCamel(kebab, leadingUppercase = false) {
    const camel = [];
    for(let i1 = 0; i1 < kebab.length; i1++){
        const ch = kebab.charAt(i1);
        if (ch === "-") {
            camel.push(kebab.charAt(++i1).toUpperCase());
        } else if (i1 === 0 && leadingUppercase) {
            camel.push(ch.toUpperCase());
        } else {
            camel.push(ch);
        }
    }
    return camel.join("");
}
const kMetadataFormat = "format";
const kDisplayName = "display-name";
const kExtensionName = "extension-name";
const kTargetFormat = "target-format";
const kBaseFormat = "base-format";
const kIdentifierDefaults = "indentifier";
const kRenderDefaults = "render";
const kExecuteDefaults = "execute";
const kPandocDefaults = "pandoc";
const kLanguageDefaults = "language";
const kPandocMetadata = "metadata";
const kFigWidth = "fig-width";
const kFigHeight = "fig-height";
const kFigFormat = "fig-format";
const kFigDpi = "fig-dpi";
const kFigAsp = "fig-asp";
const kMermaidFormat = "mermaid-format";
const kDfPrint = "df-print";
const kCache = "cache";
const kFreeze = "freeze";
const kEngine = "engine";
const kEval = "eval";
const kEcho = "echo";
const kOutput = "output";
const kWarning = "warning";
const kError = "error";
const kInclude = "include";
const kResources = "resources";
const kFormatResources = "format-resources";
const kSupporting = "supporting";
const kIpynbProduceSourceNotebook = "produce-source-notebook";
const kEnableCrossRef = "enable-crossref";
const kFormatLinks = "format-links";
const kNotebookLinks = "notebook-links";
const kOtherLinks = "other-links";
const kCodeLinks = "code-links";
const kNotebookSubarticles = "notebook-subarticles";
const kNotebookView = "notebook-view";
const kNotebookViewStyle = "notebook-view-style";
const kNotebookPreserveCells = "notebook-preserve-cells";
const kClearCellOptions = "clear-cell-options";
const kDownloadUrl = "download-url";
const kLightbox = "lightbox";
const kCanonicalUrl = "canonical-url";
const kMath = "math";
const kNotebookPreviewOptions = "notebook-preview-options";
const kKeepHidden = "keep-hidden";
const kRemoveHidden = "remove-hidden";
const kClearHiddenClasses = "clear-hidden-classes";
const kUnrollMarkdownCells = "unroll-markdown-cells";
const kExecuteEnabled = "enabled";
const kExecuteIpynb = "ipynb";
const kExecuteDaemon = "daemon";
const kExecuteDaemonRestart = "daemon-restart";
const kExecuteDebug = "debug";
const kIpynbFilter = "ipynb-filter";
const kIpynbFilters = "ipynb-filters";
const kIpynbShellInteractivity = "ipynb-shell-interactivity";
const kIPynbTitleBlockTemplate = "ipynb-title-block";
const kPlotlyConnected = "plotly-connected";
const kJatsSubarticleId = "jats-subarticle-id";
const kShortcodes = "shortcodes";
const kKeepMd = "keep-md";
const kKeepTex = "keep-tex";
const kKeepTyp = "keep-typ";
const kKeepIpynb = "keep-ipynb";
const kKeepSource = "keep-source";
const kVariant = "variant";
const kOutputExt = "output-ext";
const kOutputDivs = "output-divs";
const kPageWidth = "page-width";
const kFigAlign = "fig-align";
const kFigEnv = "fig-env";
const kFigPos = "fig-pos";
const kCodeFold = "code-fold";
const kCodeLineNumbers = "code-line-numbers";
const kCodeOverflow = "code-overflow";
const kCodeLink = "code-link";
const kCodeTools = "code-tools";
const kTblCap = "tbl-cap";
const kTblColwidths = "tbl-colwidths";
const kMergeIncludes = "merge-includes";
const kInlineIncludes = "inline-includes";
const kPreserveYaml = "preserve-yaml";
const kPreferHtml = "prefer-html";
const kSelfContainedMath = "self-contained-math";
const kBodyClasses = "body-classes";
const kLatexAutoMk = "latex-auto-mk";
const kLatexAutoInstall = "latex-auto-install";
const kLatexMinRuns = "latex-min-runs";
const kLatexMaxRuns = "latex-max-runs";
const kLatexClean = "latex-clean";
const kLatexInputPaths = "latex-input-paths";
const kLatexMakeIndex = "latex-makeindex";
const kLatexMakeIndexOpts = "latex-makeindex-opts";
const kLatexTinyTex = "latex-tinytex";
const kLatexTlmgrOpts = "latex-tlmgr-opts";
const kLatexOutputDir = "latex-output-dir";
const kLinkExternalIcon = "link-external-icon";
const kLinkExternalNewwindow = "link-external-newwindow";
const kLinkExternalFilter = "link-external-filter";
const kQuartoVersion = "quarto-version";
const kQuartoRequired = "quarto-required";
const kPreviewMode = "preview-mode";
const kPreviewModeRaw = "raw";
const kFontPaths = "font-paths";
const kHtmlTableProcessing = "html-table-processing";
const kUseRsvgConvert = "use-rsvg-convert";
const kValidateYaml = "validate-yaml";
const kIdentifierDefaultsKeys = [
    kTargetFormat,
    kDisplayName,
    kExtensionName
];
const kExecuteDefaultsKeys = [
    kFigWidth,
    kFigHeight,
    kFigFormat,
    kFigDpi,
    kFigAsp,
    kMermaidFormat,
    kDfPrint,
    kError,
    kEval,
    kEngine,
    kCache,
    kFreeze,
    kEcho,
    kOutput,
    kWarning,
    kInclude,
    kKeepMd,
    kKeepIpynb,
    kExecuteEnabled,
    kExecuteIpynb,
    kExecuteDaemon,
    kExecuteDaemonRestart,
    kExecuteDebug,
    kIpynbFilter,
    kIpynbFilters,
    kIpynbShellInteractivity,
    kPlotlyConnected
];
const kRenderDefaultsKeys = [
    kKeepTex,
    kKeepTyp,
    kKeepSource,
    kKeepHidden,
    kRemoveHidden,
    kClearHiddenClasses,
    kVariant,
    kOutputExt,
    kOutputDivs,
    kPreferHtml,
    kPageWidth,
    kFigAlign,
    kFigPos,
    kFigEnv,
    kCodeFold,
    kCodeLink,
    kCodeLineNumbers,
    kCodeOverflow,
    kCodeTools,
    kShortcodes,
    kTblColwidths,
    kInlineIncludes,
    kPreserveYaml,
    kMergeIncludes,
    kSelfContainedMath,
    kLatexAutoMk,
    kLatexAutoInstall,
    kLatexMinRuns,
    kLatexMaxRuns,
    kLatexClean,
    kLatexInputPaths,
    kLatexMakeIndex,
    kLatexMakeIndexOpts,
    kLatexTlmgrOpts,
    kLatexOutputDir,
    kLatexTinyTex,
    kLinkExternalIcon,
    kLinkExternalNewwindow,
    kLinkExternalFilter,
    kFormatResources,
    kFormatLinks,
    kIpynbProduceSourceNotebook,
    kNotebookLinks,
    kNotebookSubarticles,
    kNotebookView,
    kNotebookViewStyle,
    kNotebookPreserveCells,
    kClearCellOptions,
    kHtmlTableProcessing,
    kValidateYaml,
    kCanonicalUrl,
    kBodyClasses
];
const kTocTitleDocument = "toc-title-document";
const kTocTitleWebsite = "toc-title-website";
const kRelatedFormatsTitle = "related-formats-title";
const kRelatedNotebooksTitle = "related-notebooks-title";
const kOtherLinksTitle = "other-links-title";
const kCodeLinksTitle = "code-links-title";
const kLaunchDevContainerTitle = "launch-dev-container-title";
const kLaunchBinderTitle = "launch-binder-title";
const kSourceNotebookPrefix = "source-notebooks-prefix";
const kCalloutTipCaption = "callout-tip-title";
const kCalloutNoteCaption = "callout-note-title";
const kCalloutWarningCaption = "callout-warning-title";
const kCalloutImportantCaption = "callout-important-title";
const kCalloutCautionCaption = "callout-caution-title";
const kSectionTitleAbstract = "section-title-abstract";
const kSectionTitleFootnotes = "section-title-footnotes";
const kSectionTitleReferences = "section-title-references";
const kSectionTitleAppendices = "section-title-appendices";
const kSectionTitleReuse = "section-title-reuse";
const kSectionTitleCopyright = "section-title-copyright";
const kSectionTitleCitation = "section-title-citation";
const kAppendixAttributionBibTex = "appendix-attribution-bibtex";
const kAppendixAttributionCiteAs = "appendix-attribution-cite-as";
const kAppendixViewLicense = "appendix-view-license";
const kTitleBlockAuthorSingle = "title-block-author-single";
const kTitleBlockAuthorPlural = "title-block-author-plural";
const kTitleBlockAffiliationSingle = "title-block-affiliation-single";
const kTitleBlockAffiliationPlural = "title-block-affiliation-plural";
const kTitleBlockPublished = "title-block-published";
const kTitleBlockModified = "title-block-modified";
const kTitleBlockKeywords = "title-block-keywords";
const kCodeSummary = "code-summary";
const kCodeLine = "code-line";
const kCodeLines = "code-lines";
const kCodeToolsMenuCaption = "code-tools-menu-caption";
const kCodeToolsShowAllCode = "code-tools-show-all-code";
const kCodeToolsHideAllCode = "code-tools-hide-all-code";
const kCodeToolsViewSource = "code-tools-view-source";
const kCodeToolsSourceCode = "code-tools-source-code";
const kSearchNoResultsText = "search-no-results-text";
const kSearchLabel = "search-label";
const kToolsShare = "tools-share";
const kToolsDownload = "tools-download";
const kToggleSection = "toggle-section";
const kToggleSidebar = "toggle-sidebar";
const kToggleDarkMode = "toggle-dark-mode";
const kToggleReaderMode = "toggle-reader-mode";
const kToggleNavigation = "toggle-navigation";
const kCopyButtonTooltip = "copy-button-tooltip";
const kCopyButtonTooltipSuccess = "copy-button-tooltip-success";
const kBackToTop = "back-to-top";
const kRepoActionLinksEdit = "repo-action-links-edit";
const kRepoActionLinksSource = "repo-action-links-source";
const kRepoActionLinksIssue = "repo-action-links-issue";
const kSearchMatchingDocumentsText = "search-matching-documents-text";
const kSearchCopyLinkTitle = "search-copy-link-title";
const kSearchHideMatchesText = "search-hide-matches-text";
const kSearchMoreMatchText = "search-more-match-text";
const kSearchMoreMatchesText = "search-more-matches-text";
const kSearchClearButtonTitle = "search-clear-button-title";
const kSearchTextPlaceholder = "search-text-placeholder";
const kSearchDetatchedCancelButtonTitle = "search-detached-cancel-button-title";
const kSearchSubmitButtonTitle = "search-submit-button-title";
const kCrossrefFigTitle = "crossref-fig-title";
const kCrossrefTblTitle = "crossref-tbl-title";
const kCrossrefLstTitle = "crossref-lst-title";
const kCrossrefThmTitle = "crossref-thm-title";
const kCrossrefLemTitle = "crossref-lem-title";
const kCrossrefCorTitle = "crossref-cor-title";
const kCrossrefPrfTitle = "crossref-prp-title";
const kCrossrefCnjTitle = "crossref-cnj-title";
const kCrossrefDefTitle = "crossref-def-title";
const kCrossrefExmTitle = "crossref-exm-title";
const kCrossrefExrTitle = "crossref-exr-title";
const kCrossrefFigPrefix = "crossref-fig-prefix";
const kCrossrefTblPrefix = "crossref-tbl-prefix";
const kCrossrefLstPrefix = "crossref-lst-prefix";
const kCrossrefChPrefix = "crossref-ch-prefix";
const kCrossrefApxPrefix = "crossref-apx-prefix";
const kCrossrefSecPrefix = "crossref-sec-prefix";
const kCrossrefEqPrefix = "crossref-eq-prefix";
const kCrossrefThmPrefix = "crossref-thm-prefix";
const kCrossrefLemPrefix = "crossref-lem-prefix";
const kCrossrefCorPrefix = "crossref-cor-prefix";
const kCrossrefPrpPrefix = "crossref-prp-prefix";
const kCrossrefCnjPrefix = "crossref-cnj-prefix";
const kCrossrefDefPrefix = "crossref-def-prefix";
const kCrossrefExmPrefix = "crossref-exm-prefix";
const kCrossrefExrPrefix = "crossref-exr-prefix";
const kCrossrefLofTitle = "crossref-lof-title";
const kCrossrefLotTitle = "crossref-lot-title";
const kCrossrefLolTitle = "crossref-lol-title";
const kEnvironmentProofTitle = "environment-proof-title";
const kEnvironmentRemarkTitle = "environment-remark-title";
const kEnvironmentSolutionTitle = "environment-solution-title";
const kListingPageOrderBy = "listing-page-order-by";
const kListingPageFilter = "listing-page-filter";
const kListingPageOrderByDefault = "listing-page-order-by-default";
const kListingPageOrderByDateAsc = "listing-page-order-by-date-asc";
const kListingPageOrderByDateDesc = "listing-page-order-by-date-desc";
const kListingPageOrderByNumberAsc = "listing-page-order-by-number-asc";
const kListingPageOrderByNumberDesc = "listing-page-order-by-number-desc";
const kListingPageFieldDate = "listing-page-field-date";
const kListingPageFieldTitle = "listing-page-field-title";
const kListingPageFieldDescription = "listing-page-field-description";
const kListingPageFieldAuthor = "listing-page-field-author";
const kListingPageFieldFileName = "listing-page-field-filename";
const kListingPageFieldFileModified = "listing-page-field-filemodified";
const kListingPageFieldSubtitle = "listing-page-field-subtitle";
const kListingPageFieldReadingTime = "listing-page-field-readingtime";
const kListingPageFieldWordCount = "listing-page-field-wordcount";
const kListingPageFieldCategories = "listing-page-field-categories";
const kListingPageMinutesCompact = "listing-page-minutes-compact";
const kListingPageWords = "listing-page-words";
const kListingPageCategoryAll = "listing-page-category-all";
const kListingPageNoMatches = "listing-page-no-matches";
const kNotebookPreviewDownload = "notebook-preview-download";
const kNotebookPreviewDownloadSrc = "notebook-preview-download-src";
const kNotebookPreviewBack = "notebook-preview-back";
const kArticleNotebookLabel = "article-notebook-label";
const kManuscriptMecaBundle = "manuscript-meca-bundle";
const kDraftLabel = "draft";
const kLanguageDefaultsKeys = [
    kTocTitleDocument,
    kTocTitleWebsite,
    kRelatedFormatsTitle,
    kOtherLinksTitle,
    kCodeLinksTitle,
    kLaunchDevContainerTitle,
    kLaunchBinderTitle,
    kRelatedNotebooksTitle,
    kSourceNotebookPrefix,
    kCalloutTipCaption,
    kCalloutNoteCaption,
    kCalloutWarningCaption,
    kCalloutImportantCaption,
    kCalloutCautionCaption,
    kSectionTitleAbstract,
    kSectionTitleFootnotes,
    kSectionTitleReferences,
    kSectionTitleAppendices,
    kSectionTitleReuse,
    kSectionTitleCopyright,
    kSectionTitleCitation,
    kAppendixAttributionBibTex,
    kAppendixAttributionCiteAs,
    kAppendixViewLicense,
    kTitleBlockAuthorSingle,
    kTitleBlockPublished,
    kTitleBlockModified,
    kTitleBlockKeywords,
    kCodeSummary,
    kCodeLine,
    kCodeLines,
    kCodeToolsMenuCaption,
    kCodeToolsShowAllCode,
    kCodeToolsHideAllCode,
    kCodeToolsViewSource,
    kCodeToolsSourceCode,
    kToolsShare,
    kToolsDownload,
    kSearchNoResultsText,
    kSearchLabel,
    kToggleDarkMode,
    kToggleNavigation,
    kToggleReaderMode,
    kToggleSidebar,
    kToggleSection,
    kCopyButtonTooltip,
    kCopyButtonTooltipSuccess,
    kBackToTop,
    kRepoActionLinksEdit,
    kRepoActionLinksSource,
    kRepoActionLinksIssue,
    kSearchMatchingDocumentsText,
    kSearchCopyLinkTitle,
    kSearchHideMatchesText,
    kSearchMoreMatchText,
    kSearchMoreMatchesText,
    kSearchClearButtonTitle,
    kSearchTextPlaceholder,
    kSearchDetatchedCancelButtonTitle,
    kSearchSubmitButtonTitle,
    kCrossrefFigTitle,
    kCrossrefTblTitle,
    kCrossrefLstTitle,
    kCrossrefThmTitle,
    kCrossrefLemTitle,
    kCrossrefCorTitle,
    kCrossrefPrfTitle,
    kCrossrefCnjTitle,
    kCrossrefDefTitle,
    kCrossrefExmTitle,
    kCrossrefExrTitle,
    kCrossrefFigPrefix,
    kCrossrefTblPrefix,
    kCrossrefLstPrefix,
    kCrossrefChPrefix,
    kCrossrefApxPrefix,
    kCrossrefSecPrefix,
    kCrossrefEqPrefix,
    kCrossrefThmPrefix,
    kCrossrefLemPrefix,
    kCrossrefCorPrefix,
    kCrossrefPrpPrefix,
    kCrossrefCnjPrefix,
    kCrossrefDefPrefix,
    kCrossrefExmPrefix,
    kCrossrefExrPrefix,
    kCrossrefLofTitle,
    kCrossrefLotTitle,
    kCrossrefLolTitle,
    kEnvironmentProofTitle,
    kEnvironmentRemarkTitle,
    kEnvironmentSolutionTitle,
    kListingPageFilter,
    kListingPageOrderBy,
    kListingPageOrderByDefault,
    kListingPageOrderByDateAsc,
    kListingPageOrderByDateDesc,
    kListingPageOrderByNumberAsc,
    kListingPageOrderByNumberDesc,
    kListingPageFieldDate,
    kListingPageFieldTitle,
    kListingPageFieldDescription,
    kListingPageFieldAuthor,
    kListingPageFieldFileName,
    kListingPageFieldFileModified,
    kListingPageFieldSubtitle,
    kListingPageFieldReadingTime,
    kListingPageFieldWordCount,
    kListingPageFieldCategories,
    kListingPageMinutesCompact,
    kListingPageWords,
    kListingPageCategoryAll,
    kListingPageNoMatches,
    kTitleBlockAuthorSingle,
    kTitleBlockAuthorPlural,
    kTitleBlockAffiliationSingle,
    kTitleBlockAffiliationPlural,
    kTitleBlockPublished,
    kTitleBlockModified,
    kTitleBlockKeywords,
    kNotebookPreviewDownload,
    kNotebookPreviewDownloadSrc,
    kNotebookPreviewBack,
    kArticleNotebookLabel,
    kDraftLabel,
    kManuscriptMecaBundle
];
const kTo = "to";
const kFrom = "from";
const kReader = "reader";
const kWriter = "writer";
const kOutputFile = "output-file";
const kInputFiles = "input-files";
const kMarkdownHeadings = "markdown-headings";
const kTemplate = "template";
const kWrap = "wrap";
const kColumns = "columns";
const kStandalone = "standalone";
const kSelfContained = "self-contained";
const kEmbedResources = "embed-resources";
const kIncludeBeforeBody = "include-before-body";
const kIncludeAfterBody = "include-after-body";
const kIncludeInHeader = "include-in-header";
const kResourcePath = "resource-path";
const kCiteproc = "citeproc";
const kCiteMethod = "cite-method";
const kFilters = "filters";
const kQuartoFilters = "quarto-filters";
const kFilterParams = "filter-params";
const kPdfEngine = "pdf-engine";
const kNotebooks = "notebooks";
const kPdfEngineOpts = "pdf-engine-opts";
const kPdfEngineOpt = "pdf-engine-opt";
const kListings = "listings";
const kNumberSections = "number-sections";
const kSectionNumbering = "section-numbering";
const kNumberOffset = "number-offset";
const kShiftHeadingLevelBy = "shift-heading-level-by";
const kNumberDepth = "number-depth";
const kTopLevelDivision = "top-level-division";
const kPaperSize = "papersize";
const kLogFile = "log-file";
const kHighlightStyle = "highlight-style";
const kDefaultImageExtension = "default-image-extension";
const kLinkColor = "linkcolor";
const kColorLinks = "colorlinks";
const kVariables = "variables";
const kMetadataFile = "metadata-file";
const kMetadataFiles = "metadata-files";
const kSyntaxDefinitions = "syntax-definitions";
const kSyntaxDefinition = "syntax-definition";
const kReferenceDoc = "reference-doc";
const kHtmlMathMethod = "html-math-method";
const kToc = "toc";
const kTocDepth = "toc-depth";
const kTocIndent = "toc-indent";
const kTableOfContents = "table-of-contents";
const kSectionDivs = "section-divs";
const kEPubCoverImage = "epub-cover-image";
const kReferenceLocation = "reference-location";
const kCitationLocation = "citation-location";
const kQuartoVarsKey = "_quarto-vars";
const kQuartoTemplateParams = "quarto-template-params";
const kRevealJsScripts = "reveal-jsscripts";
const kQuartoInternal = "quarto-internal";
const kTitle = "title";
const kSubtitle = "subtitle";
const kAuthor = "author";
const kDate = "date";
const kDateFormat = "date-format";
const kDateModified = "date-modified";
const kDoi = "doi";
const kAbstract = "abstract";
const kAbstractTitle = "abstract-title";
const kDescription = "description";
const kHideDescription = "hide-description";
const kTocTitle = "toc-title";
const kTocLocation = "toc-location";
const kTocExpand = "toc-expand";
const kLang = "lang";
const kOrder = "order";
const kDisableArticleLayout = "disable-article-layout";
const kAuthors = "authors";
const kInstitute = "institute";
const kInstitutes = "institutes";
const kServer = "server";
const kPageTitle = "pagetitle";
const kTitlePrefix = "title-prefix";
const kCsl = "csl";
const kNoCite = "nocite";
const kCss = "css";
const kBibliography = "bibliography";
const kReferences = "references";
const kHeaderIncludes = "header-includes";
const kIncludeBefore = "include-before";
const kIncludeAfter = "include-after";
const kLinkCitations = "link-citations";
const kDocumentClass = "documentclass";
const kClassOption = "classoption";
const kSlideLevel = "slide-level";
const kTheme = "theme";
const kCrossref = "crossref";
const kCrossrefChapters = "chapters";
const kCrossrefLabels = "labels";
const kCrossrefAppendixTitle = "appendix-title";
const kCrossrefAppendixDelim = "appendix-delim";
const kCrossrefChaptersAppendix = "chapters-appendix";
const kCrossrefChaptersAlpha = "chapters-alpha";
const kCrossrefChapterId = "chapter-id";
const kGrid = "grid";
const kContentMode = "content-mode";
const kFigResponsive = "fig-responsive";
const kOutputLocation = "output-location";
const kCapLoc = "cap-location";
const kFigCapLoc = "fig-cap-location";
const kTblCapLoc = "tbl-cap-location";
const kCapTop = "top";
const kCapBottom = "bottom";
const kPositionedRefs = "positioned-refs";
const kPandocDefaultsKeys = [
    kTo,
    kFrom,
    kReader,
    kWriter,
    kOutputFile,
    kInputFiles,
    "defaults",
    kTemplate,
    kStandalone,
    kSelfContained,
    kEmbedResources,
    kVariables,
    "metadata",
    kMetadataFiles,
    kMetadataFile,
    kIncludeBeforeBody,
    kIncludeAfterBody,
    kIncludeInHeader,
    kResourcePath,
    kCiteproc,
    kCiteMethod,
    "citation-abbreviations",
    "filters",
    "file-scope",
    "data-dir",
    "verbosity",
    kLogFile,
    kTopLevelDivision,
    "abbreviations",
    kPdfEngine,
    kPdfEngineOpts,
    kPdfEngineOpt,
    kWrap,
    kColumns,
    "dpi",
    "extract-media",
    kToc,
    kTableOfContents,
    kTocDepth,
    kNumberSections,
    kNumberOffset,
    kShiftHeadingLevelBy,
    kSectionDivs,
    "identifier-prefix",
    kTitlePrefix,
    "eol",
    "strip-comments",
    "indented-code-classes",
    "ascii",
    kDefaultImageExtension,
    kHighlightStyle,
    kSyntaxDefinitions,
    kSyntaxDefinition,
    kListings,
    kReferenceDoc,
    kHtmlMathMethod,
    "email-obfuscation",
    "tab-stop",
    "preserve-tabs",
    "incremental",
    kSlideLevel,
    "epub-subdirectory",
    "epub-metadata",
    "epub-fonts",
    "epub-chapter-level",
    kEPubCoverImage,
    "reference-links",
    kReferenceLocation,
    kMarkdownHeadings,
    "track-changes",
    "html-q-tags",
    "css",
    "ipynb-output",
    "request-headers",
    "fail-if-warnings",
    "dump-args",
    "ignore-args",
    "trace"
];
const kCellCollapsed = "collapsed";
const kCellAutoscroll = "autoscroll";
const kCellDeletable = "deletable";
const kCellFormat = "format";
const kCellName = "name";
const kCellLinesToNext = "lines_to_next_cell";
const kCellLanguage = "language";
const kCellSlideshow = "slideshow";
const kCellSlideshowSlideType = "slide_type";
const kCellRawMimeType = "raw_mimetype";
const kCellUserExpressions = "user_expressions";
const kCellId = "id";
const kCellLabel = "label";
const kCellFigCap = "fig-cap";
const kCellFigSubCap = "fig-subcap";
const kCellFigScap = "fig-scap";
const kCellFigColumn = "fig-column";
const kCellTblColumn = "tbl-column";
const kCellFigLink = "fig-link";
const kCellFigAlign = "fig-align";
const kCellFigEnv = "fig-env";
const kCellFigPos = "fig-pos";
const kCellFigAlt = "fig-alt";
const kCellLstLabel = "lst-label";
const kCellLstCap = "lst-cap";
const kCellClasses = "classes";
const kCellPanel = "panel";
const kCellColumn = "column";
const kCellOutWidth = "out-width";
const kCellOutHeight = "out-height";
const kCellMdIndent = "md-indent";
const kCellColab = "colab";
const kCellColabType = "colab_type";
const kCellColbOutputId = "outputId";
const kLayoutAlign = "layout-align";
const kLayoutVAlign = "layout-valign";
const kLayoutNcol = "layout-ncol";
const kLayoutNrow = "layout-nrow";
const kLayout = "layout";
const kCliffyImplicitCwd = "5a6d2e4f-f9a2-43bc-8019-8149fbb76c85";
const kSourceMappingRegexes = [
    /^\/\/#\s*sourceMappingURL\=.*\.map$/gm,
    /\/\*\# sourceMappingURL=.* \*\//g
];
const kFormatIdentifier = "format-identifier";
const kRenderFileLifetime = "render-file";
const kRenderServicesLifetime = "render-services";
const kQuartoPre = "pre-quarto";
const kQuartoPost = "post-render";
async function readAndValidateYamlFromFile(file, schema, errorMessage, defaultContents) {
    if (!existsSync(file)) {
        throw new Error(`YAML file ${file} not found.`);
    }
    let shortFileName = file;
    if (shortFileName.startsWith("/")) {
        shortFileName = relative2(Deno.cwd(), shortFileName);
    }
    let fileContents = Deno.readTextFileSync(file).trimEnd();
    if (fileContents.trim().length === 0 && defaultContents) {
        fileContents = defaultContents;
    }
    const contents = asMappedString(fileContents, shortFileName);
    const { yaml, yamlValidationErrors } = await readAndValidateYamlFromMappedString(contents, schema);
    if (yamlValidationErrors.length) {
        throw new ValidationError2(errorMessage, yamlValidationErrors);
    }
    return yaml;
}
const kGfmCommonmarkExtensions = [
    "+autolink_bare_uris",
    "+emoji",
    "+footnotes",
    "+gfm_auto_identifiers",
    "+pipe_tables",
    "+strikeout",
    "+task_lists",
    "+tex_math_dollars"
];
const kGfmCommonmarkVariant = kGfmCommonmarkExtensions.join("");
`commonmark${kGfmCommonmarkVariant}`;
const kQmdExtensions = [
    ".qmd"
];
const kMarkdownEngine = "markdown";
const kKnitrEngine = "knitr";
const kJupyterEngine = "jupyter";
const kJuliaEngine = "julia";
async function includedMetadata(dir, baseMetadata, schema) {
    const yamlFiles = [];
    const metadataFile = baseMetadata[kMetadataFile];
    if (metadataFile) {
        yamlFiles.push(join2(dir, metadataFile));
    }
    const metadataFiles = baseMetadata[kMetadataFiles];
    if (metadataFiles && Array.isArray(metadataFiles)) {
        metadataFiles.forEach((file)=>yamlFiles.push(join2(dir, file)));
    }
    const filesMetadata = await Promise.all(yamlFiles.map(async (yamlFile)=>{
        if (existsSync(yamlFile)) {
            try {
                const yaml = await readAndValidateYamlFromFile(yamlFile, schema, `Validation of metadata file ${yamlFile} failed.`);
                return yaml;
            } catch (e) {
                error("\nError reading metadata file from " + yamlFile + "\n");
                throw e;
            }
        } else {
            return undefined;
        }
    }));
    return {
        metadata: mergeFormatMetadata({}, ...filesMetadata),
        files: yamlFiles
    };
}
function formatFromMetadata(baseFormat, to, debug) {
    const typedFormat = {
        identifier: {},
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    let format = typedFormat;
    const configFormats = baseFormat.metadata[kMetadataFormat];
    if (configFormats instanceof Object) {
        const configFormat = configFormats[to];
        if (configFormat === "default" || configFormat === true) {
            format = metadataAsFormat({});
        } else if (configFormat instanceof Object) {
            format = metadataAsFormat(configFormat);
        }
    }
    const mergedFormat = mergeFormatMetadata(baseFormat, format);
    if (debug) {
        mergedFormat.execute[kKeepMd] = true;
        mergedFormat.render[kKeepTex] = true;
        mergedFormat.render[kKeepTyp] = true;
    }
    return mergedFormat;
}
function formatKeys(metadata) {
    if (typeof metadata[kMetadataFormat] === "string") {
        return [
            metadata[kMetadataFormat]
        ];
    } else if (metadata[kMetadataFormat] instanceof Object) {
        return Object.keys(metadata[kMetadataFormat]).filter((key)=>{
            const format = metadata[kMetadataFormat][key];
            return format !== null && format !== false;
        });
    } else {
        return [];
    }
}
function isQuartoMetadata(key) {
    return kRenderDefaultsKeys.includes(key) || kExecuteDefaultsKeys.includes(key) || kPandocDefaultsKeys.includes(key) || kLanguageDefaultsKeys.includes(key) || [
        kKnitrEngine,
        kJupyterEngine
    ].includes(key);
}
function isIncludeMetadata(key) {
    return [
        kHeaderIncludes,
        kIncludeBefore,
        kIncludeAfter
    ].includes(key);
}
function metadataAsFormat(metadata) {
    const typedFormat = {
        identifier: {},
        render: {},
        execute: {},
        pandoc: {},
        language: {},
        metadata: {}
    };
    const format = typedFormat;
    Object.keys(metadata).forEach((key)=>{
        if ([
            kIdentifierDefaults,
            kRenderDefaults,
            kExecuteDefaults,
            kPandocDefaults,
            kLanguageDefaults,
            kPandocMetadata
        ].includes(key)) {
            if (typeof metadata[key] == "boolean") {
                if (key === kExecuteDefaults) {
                    format[key] = format[key] || {};
                    format[kExecuteDefaults][kExecuteEnabled] = metadata[key];
                }
            } else {
                format[key] = {
                    ...format[key],
                    ...metadata[key]
                };
            }
        } else {
            if (kIdentifierDefaultsKeys.includes(key)) {
                format.identifier[key] = metadata[key];
            } else if (kRenderDefaultsKeys.includes(key)) {
                format.render[key] = metadata[key];
            } else if (kExecuteDefaultsKeys.includes(key)) {
                format.execute[key] = metadata[key];
            } else if (kPandocDefaultsKeys.includes(key)) {
                format.pandoc[key] = metadata[key];
            } else {
                format.metadata[key] = metadata[key];
            }
        }
    });
    if (typeof format.metadata[kServer] === "string") {
        format.metadata[kServer] = {
            type: format.metadata[kServer]
        };
    }
    const filter = format.execute[kIpynbFilter];
    if (typeof filter === "string") {
        typedFormat.execute[kIpynbFilters] = typedFormat.execute[kIpynbFilters] || [];
        typedFormat.execute[kIpynbFilters]?.push(filter);
        delete typedFormat.execute[kIpynbFilter];
    }
    if (typeof typedFormat.render.variant === "string") {
        typedFormat.render.variant = typedFormat.render.variant.replace(/^gfm/, kGfmCommonmarkVariant);
    }
    return typedFormat;
}
function metadataGetDeep(metadata, property) {
    let values = [];
    __VIRTUAL_FILE145(metadata, (value, key)=>{
        if (key === property) {
            values.push(value);
        } else if (__VIRTUAL_FILE18(value)) {
            values = values.concat(metadataGetDeep(value, property));
        }
    });
    return values;
}
function mergeFormatMetadata(config, ...configs) {
    const kUnmergeableKeys = [
        kTblColwidths
    ];
    const kBooleanDisableArrays = [
        kCodeLinks,
        kOtherLinks
    ];
    return mergeConfigsCustomized((objValue, srcValue, key)=>{
        if (kUnmergeableKeys.includes(key)) {
            return srcValue;
        } else if (key === kVariant) {
            return mergePandocVariant(objValue, srcValue);
        } else if (kBooleanDisableArrays.includes(key)) {
            return mergeDisablableArray(objValue, srcValue);
        } else {
            return undefined;
        }
    }, config, ...configs);
}
function mergeProjectMetadata(config, ...configs) {
    const kExandableStringKeys = [
        "contents"
    ];
    return mergeConfigsCustomized((objValue, srcValue, key)=>{
        if (kExandableStringKeys.includes(key) && typeof objValue === "string") {
            return srcValue;
        } else {
            return undefined;
        }
    }, config, ...configs);
}
function mergeConfigsCustomized(customizer, config, ...configs) {
    config = cloneDeep1(config);
    configs = cloneDeep1(configs);
    return __VIRTUAL_FILE154(config, ...configs, (objValue, srcValue, key)=>{
        const custom = customizer(objValue, srcValue, key);
        if (custom !== undefined) {
            return custom;
        } else {
            return mergeArrayCustomizer(objValue, srcValue);
        }
    });
}
function mergeDisablableArray(objValue, srcValue) {
    if (Array.isArray(objValue) && Array.isArray(srcValue)) {
        return mergeArrayCustomizer(objValue, srcValue);
    } else {
        if (srcValue === false) {
            return [];
        } else {
            const srcArr = srcValue !== undefined ? Array.isArray(srcValue) ? srcValue : [
                srcValue
            ] : [];
            const objArr = objValue !== undefined ? Array.isArray(objValue) ? objValue : [
                objValue
            ] : [];
            return mergeArrayCustomizer(objArr, srcArr);
        }
    }
}
function mergePandocVariant(objValue, srcValue) {
    if (typeof objValue === "string" && typeof srcValue === "string" && objValue !== srcValue) {
        const extensions = {};
        [
            ...parsePandocVariant(objValue),
            ...parsePandocVariant(srcValue)
        ].forEach((extension)=>{
            extensions[extension.name] = extension.enabled;
        });
        return Object.keys(extensions).map((name)=>`${extensions[name] ? "+" : "-"}${name}`).join("");
    } else {
        return undefined;
    }
}
function parsePandocVariant(variant) {
    variant = variant.split("\n").join();
    const extensions = [];
    const re = /([+-])([a-z_]+)/g;
    let match = re.exec(variant);
    while(match){
        extensions.push({
            name: match[2],
            enabled: match[1] === "+"
        });
        match = re.exec(variant);
    }
    return extensions;
}
function removeFlags(flags, remove) {
    let removeNext = false;
    return flags.reduce((args, arg)=>{
        if (!remove.has(arg)) {
            if (!removeNext) {
                args.push(arg);
            }
            removeNext = false;
        } else {
            removeNext = remove.get(arg);
        }
        return args;
    }, new Array());
}
const kStdOut = "-";
async function parseRenderFlags(args) {
    const flags = {};
    const argsStack = [
        ...args
    ];
    let arg = argsStack.shift();
    while(arg !== undefined){
        const equalSignIndex = arg.indexOf("=");
        if (arg.startsWith("--") && equalSignIndex > 0) {
            argsStack.unshift(arg.slice(equalSignIndex + 1));
            arg = arg.slice(0, equalSignIndex);
        }
        switch(arg){
            case "-t":
            case "--to":
                arg = argsStack.shift();
                if (arg && !arg.startsWith("-")) {
                    flags.to = arg;
                }
                break;
            case "-o":
            case "--output":
                arg = argsStack.shift();
                if (!arg || arg.startsWith("-")) {
                    flags.output = kStdOut;
                } else {
                    if (arg.match(SEPARATOR_PATTERN2)) {
                        throw new Error("--output option cannot specify a relative or absolute path");
                    }
                    flags.output = arg;
                }
                break;
            case "--output-dir":
                arg = argsStack.shift();
                flags.outputDir = arg;
                break;
            case "--site-url":
                arg = argsStack.shift();
                flags.siteUrl = arg;
                break;
            case "--standalone":
                flags[kStandalone] = true;
                arg = argsStack.shift();
                break;
            case "--self-contained":
                flags[kSelfContained] = true;
                arg = argsStack.shift();
                break;
            case "--embed-resources":
                flags[kEmbedResources] = true;
                arg = argsStack.shift();
                break;
            case "--pdf-engine":
                arg = argsStack.shift();
                flags.pdfEngine = arg;
                break;
            case "--pdf-engine-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.pdfEngineOpts = flags.pdfEngineOpts || [];
                    flags.pdfEngineOpts.push(arg);
                }
                break;
            case "--latex-makeindex-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.makeIndexOpts = flags.makeIndexOpts || [];
                    flags.makeIndexOpts.push(arg);
                }
                break;
            case "--latex-tlmgr-opt":
                arg = argsStack.shift();
                if (arg) {
                    flags.tlmgrOpts = flags.tlmgrOpts || [];
                    flags.tlmgrOpts.push(arg);
                }
                break;
            case "--natbib":
                arg = argsStack.shift();
                flags.natbib = true;
                break;
            case "--biblatex":
                arg = argsStack.shift();
                flags.biblatex = true;
                break;
            case `--${kToc}`:
            case `--${kTableOfContents}`:
                arg = argsStack.shift();
                flags.toc = true;
                break;
            case "--listings":
                arg = argsStack.shift();
                flags[kListings] = true;
                break;
            case "--number-sections":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                break;
            case "--number-offset":
                arg = argsStack.shift();
                flags[kNumberSections] = true;
                flags[kNumberOffset] = parseNumbers("--number-offset", arg);
                break;
            case "--top-level-division":
                arg = argsStack.shift();
                flags[kTopLevelDivision] = arg;
                break;
            case "--shift-heading-level-by":
                arg = argsStack.shift();
                flags[kShiftHeadingLevelBy] = arg;
                break;
            case "--include-in-header":
            case "--include-before-body":
            case "--include-after-body":
                {
                    const include = arg.replace("^--", "");
                    const includeFlags = flags;
                    includeFlags[include] = includeFlags[include] || [];
                    arg = argsStack.shift();
                    includeFlags[include].push(arg);
                    break;
                }
            case "--mathjax":
                flags.mathjax = true;
                arg = argsStack.shift();
                break;
            case "--katex":
                flags.katex = true;
                arg = argsStack.shift();
                break;
            case "--mathml":
                flags.mathml = true;
                arg = argsStack.shift();
                break;
            case "--gladtex":
                flags.gladtex = true;
                arg = argsStack.shift();
                break;
            case "--webtex":
                flags.webtex = true;
                arg = argsStack.shift();
                break;
            case "--execute":
                flags.execute = true;
                arg = argsStack.shift();
                break;
            case "--no-execute":
                flags.execute = false;
                arg = argsStack.shift();
                break;
            case "--execute-params":
                arg = argsStack.shift();
                flags.paramsFile = arg;
                break;
            case "--execute-dir":
                arg = argsStack.shift();
                if (arg) {
                    if (isAbsolute2(arg)) {
                        flags.executeDir = arg;
                    } else {
                        flags.executeDir = normalizePath(arg);
                    }
                }
                break;
            case "--execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = parseInt(arg, 10);
                if (isNaN(flags.executeDaemon)) {
                    delete flags.executeDaemon;
                }
                break;
            case "--no-execute-daemon":
                arg = argsStack.shift();
                flags.executeDaemon = 0;
                break;
            case "--execute-daemon-restart":
                arg = argsStack.shift();
                flags.executeDaemonRestart = true;
                break;
            case "--execute-debug":
                arg = argsStack.shift();
                flags.executeDebug = true;
                break;
            case "--use-freezer":
                arg = argsStack.shift();
                flags.useFreezer = true;
                break;
            case "--cache":
                arg = argsStack.shift();
                flags.executeCache = true;
                break;
            case "--no-cache":
                arg = argsStack.shift();
                flags.executeCache = false;
                break;
            case "--cache-refresh":
                arg = argsStack.shift();
                flags.executeCache = "refresh";
                break;
            case "--clean":
                arg = argsStack.shift();
                flags.clean = true;
                break;
            case "--no-clean":
                arg = argsStack.shift();
                flags.clean = false;
                break;
            case "--debug":
                flags.debug = true;
                arg = argsStack.shift();
                break;
            case "-P":
            case "--execute-param":
                arg = argsStack.shift();
                if (arg) {
                    const param = parseMetadataFlagValue(arg);
                    if (param) {
                        if (param.value !== undefined) {
                            flags.params = flags.params || {};
                            flags.params[param.name] = param.value;
                        }
                    }
                }
                break;
            case "-M":
            case "--metadata":
                arg = argsStack.shift();
                if (arg) {
                    const metadata = parseMetadataFlagValue(arg);
                    if (metadata) {
                        if (metadata.value !== undefined) {
                            if (isQuartoMetadata(metadata.name)) {
                                flags.metadata = flags.metadata || {};
                                flags.metadata[metadata.name] = metadata.value;
                            } else {
                                flags.pandocMetadata = flags.pandocMetadata || {};
                                flags.pandocMetadata[metadata.name] = metadata.value;
                            }
                        }
                    }
                }
                break;
            case "--metadata-file":
                arg = argsStack.shift();
                if (arg) {
                    if (existsSync(arg)) {
                        const metadata = await readYamlFromString(Deno.readTextFileSync(arg));
                        flags.metadata = {
                            ...flags.metadata,
                            ...metadata
                        };
                    }
                }
                break;
            case "--reference-location":
                arg = argsStack.shift();
                if (arg) {
                    flags[kReferenceLocation] = arg;
                }
                break;
            default:
                arg = argsStack.shift();
                break;
        }
    }
    if (flags.clean === undefined) {
        flags.clean = true;
    }
    return flags;
}
function havePandocArg(pandocArgs, arg) {
    return pandocArgs.indexOf(arg) !== -1;
}
function replacePandocArg(pandocArgs, arg, value) {
    const newArgs = [
        ...pandocArgs
    ];
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1) {
        newArgs[argIndex + 1] = value;
    } else {
        newArgs.push(arg);
        newArgs.push(value);
    }
    return newArgs;
}
function getPandocArg(pandocArgs, arg) {
    const argIndex = pandocArgs.indexOf(arg);
    if (argIndex !== -1 && argIndex + 1 < pandocArgs.length) {
        return pandocArgs[argIndex + 1];
    } else {
        return undefined;
    }
}
function replacePandocOutputArg(pandocArgs, output) {
    if (havePandocArg(pandocArgs, "--output")) {
        return replacePandocArg(pandocArgs, "--output", output);
    } else if (havePandocArg(pandocArgs, "-o")) {
        return replacePandocArg(pandocArgs, "-o", output);
    } else {
        return pandocArgs;
    }
}
function fixupPandocArgs(pandocArgs, flags) {
    pandocArgs = pandocArgs.reduce((args, arg, index)=>{
        args.push(arg);
        if (flags.output === kStdOut && pandocArgs[index + 1] !== kStdOut && (arg === "-o" || arg === "--output")) {
            args.push(kStdOut);
        }
        return args;
    }, new Array());
    const removeArgs = new Map();
    removeArgs.set("--output-dir", true);
    removeArgs.set("--site-url", true);
    removeArgs.set("--execute", false);
    removeArgs.set("--no-execute", false);
    removeArgs.set("-P", true);
    removeArgs.set("--execute-param", true);
    removeArgs.set("--execute-params", true);
    removeArgs.set("--execute-dir", true);
    removeArgs.set("--execute-daemon", true);
    removeArgs.set("--no-execute-daemon", false);
    removeArgs.set("--execute-daemon-restart", false);
    removeArgs.set("--execute-debug", false);
    removeArgs.set("--use-freezer", false);
    removeArgs.set("--cache", false);
    removeArgs.set("--no-cache", false);
    removeArgs.set("--cache-refresh", false);
    removeArgs.set("--clean", false);
    removeArgs.set("--no-clean", false);
    removeArgs.set("--debug", false);
    removeArgs.set("--metadata-file", true);
    removeArgs.set("--latex-makeindex-opt", true);
    removeArgs.set("--latex-tlmgr-opt", true);
    removeArgs.set("--log", true);
    removeArgs.set("--l", true);
    removeArgs.set("--log-level", true);
    removeArgs.set("--ll", true);
    removeArgs.set("--log-format", true);
    removeArgs.set("--lf", true);
    removeArgs.set("--quiet", false);
    removeArgs.set("--q", false);
    removeArgs.set("--profile", true);
    pandocArgs = removePandocArgs(pandocArgs, removeArgs);
    return removeQuartoMetadataFlags(pandocArgs);
}
function removePandocArgs(pandocArgs, removeArgs) {
    return removeFlags(pandocArgs, removeArgs);
}
function removePandocToArg(args) {
    const removeArgs = new Map();
    removeArgs.set("--to", true);
    removeArgs.set("-t", true);
    return removePandocArgs(args, removeArgs);
}
function removePandocTo(renderOptions) {
    renderOptions = cloneDeep1(renderOptions);
    delete renderOptions.flags?.to;
    if (renderOptions.pandocArgs) {
        renderOptions.pandocArgs = removePandocToArg(renderOptions.pandocArgs);
    }
    return renderOptions;
}
function removeQuartoMetadataFlags(pandocArgs) {
    const args = [];
    for(let i1 = 0; i1 < pandocArgs.length; i1++){
        const arg = pandocArgs[i1];
        if (arg === "--metadata" || arg === "-M") {
            const flagValue = parseMetadataFlagValue(pandocArgs[i1 + 1] || "");
            if (flagValue !== undefined && (isQuartoMetadata(flagValue.name) || kQuartoForwardedMetadataFields.includes(flagValue.name))) {
                i1++;
            } else {
                args.push(arg);
            }
        } else {
            args.push(arg);
        }
    }
    return args;
}
const kQuartoForwardedMetadataFields = [
    kAuthor,
    kDate
];
function parseMetadataFlagValue(arg) {
    const match = arg.match(/^([^=:]+)[=:](.*)$/);
    if (match) {
        return {
            name: match[1],
            value: readYamlFromString(match[2])
        };
    }
    return undefined;
}
function resolveParams(params, paramsFile) {
    if (params || paramsFile) {
        params = params || {};
        if (paramsFile) {
            params = mergeConfigs(readYaml(paramsFile), params);
        }
        return params;
    } else {
        return undefined;
    }
}
function parseNumbers(flag, value) {
    if (value) {
        const numbers = value.split(/,/).map((number)=>parseInt(number.trim(), 10)).filter((number)=>!isNaN(number));
        if (numbers.length > 0) {
            return numbers;
        }
    }
    throw new Error(`Invalid value for ${flag} (should be a comma separated list of numbers)`);
}
const importMeta = {
    url: "file:///D:/a/quarto-cli/quarto-cli/src/vendor/deno.land/x/deno_dom@v0.1.35-alpha/build/deno-wasm/deno-wasm.js",
    main: false
};
let wasm;
let WASM_VECTOR_LEN = 0;
let cachegetUint8Memory0 = null;
function getUint8Memory0() {
    if (cachegetUint8Memory0 === null || cachegetUint8Memory0.buffer !== wasm.memory.buffer) {
        cachegetUint8Memory0 = new Uint8Array(wasm.memory.buffer);
    }
    return cachegetUint8Memory0;
}
let cachedTextEncoder = new TextEncoder("utf-8");
const encodeString = typeof cachedTextEncoder.encodeInto === "function" ? function(arg, view) {
    return cachedTextEncoder.encodeInto(arg, view);
} : function(arg, view) {
    const buf = cachedTextEncoder.encode(arg);
    view.set(buf);
    return {
        read: arg.length,
        written: buf.length
    };
};
function passStringToWasm0(arg, malloc, realloc) {
    if (realloc === undefined) {
        const buf = cachedTextEncoder.encode(arg);
        const ptr = malloc(buf.length);
        getUint8Memory0().subarray(ptr, ptr + buf.length).set(buf);
        WASM_VECTOR_LEN = buf.length;
        return ptr;
    }
    let len = arg.length;
    let ptr = malloc(len);
    const mem = getUint8Memory0();
    let offset = 0;
    for(; offset < len; offset++){
        const code = arg.charCodeAt(offset);
        if (code > 0x7F) break;
        mem[ptr + offset] = code;
    }
    if (offset !== len) {
        if (offset !== 0) {
            arg = arg.slice(offset);
        }
        ptr = realloc(ptr, len, len = offset + arg.length * 3);
        const view = getUint8Memory0().subarray(ptr + offset, ptr + len);
        const ret = encodeString(arg, view);
        offset += ret.written;
    }
    WASM_VECTOR_LEN = offset;
    return ptr;
}
let cachegetInt32Memory0 = null;
function getInt32Memory0() {
    if (cachegetInt32Memory0 === null || cachegetInt32Memory0.buffer !== wasm.memory.buffer) {
        cachegetInt32Memory0 = new Int32Array(wasm.memory.buffer);
    }
    return cachegetInt32Memory0;
}
let cachedTextDecoder = new TextDecoder("utf-8", {
    ignoreBOM: true,
    fatal: true
});
cachedTextDecoder.decode();
function getStringFromWasm0(ptr, len) {
    return cachedTextDecoder.decode(getUint8Memory0().subarray(ptr, ptr + len));
}
function parse2(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
function parse_frag(html) {
    try {
        const retptr = wasm.__wbindgen_add_to_stack_pointer(-16);
        var ptr0 = passStringToWasm0(html, wasm.__wbindgen_malloc, wasm.__wbindgen_realloc);
        var len0 = WASM_VECTOR_LEN;
        wasm.parse_frag(retptr, ptr0, len0);
        var r0 = getInt32Memory0()[retptr / 4 + 0];
        var r1 = getInt32Memory0()[retptr / 4 + 1];
        return getStringFromWasm0(r0, r1);
    } finally{
        wasm.__wbindgen_add_to_stack_pointer(16);
        wasm.__wbindgen_free(r0, r1);
    }
}
async function load2(module1, imports) {
    if (typeof Response === "function" && module1 instanceof Response) {
        if (typeof WebAssembly.instantiateStreaming === "function") {
            try {
                return await WebAssembly.instantiateStreaming(module1, imports);
            } catch (e) {
                if (module1.headers.get("Content-Type") != "application/wasm") {
                    console.warn("`WebAssembly.instantiateStreaming` failed because your server does not serve wasm with `application/wasm` MIME type. Falling back to `WebAssembly.instantiate` which is slower. Original error:\n", e);
                } else {
                    throw e;
                }
            }
        }
        const bytes = await module1.arrayBuffer();
        return await WebAssembly.instantiate(bytes, imports);
    } else {
        const instance = await WebAssembly.instantiate(module1, imports);
        if (instance instanceof WebAssembly.Instance) {
            return {
                instance,
                module: module1
            };
        } else {
            return instance;
        }
    }
}
async function init(input) {
    if (typeof input === "undefined") {
        input = new URL("deno-wasm_bg.wasm", importMeta.url);
    }
    const imports = {};
    imports.env = {
        now () {}
    };
    input = Uint8Array.from(atob("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