// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Make {@linkcode Promise} abortable with the given signal.
 *
 * @example
 * ```ts
 * import {
 *   abortable,
 *   delay,
 * } from "https://deno.land/std@$STD_VERSION/async/mod.ts";
 *
 * const p = delay(1000);
 * const c = new AbortController();
 * setTimeout(() => c.abort(), 100);
 *
 * // Below throws `DOMException` after 100 ms
 * await abortable(p, c.signal);
 * ```
 */ export function abortable(p, signal) {
  if (p instanceof Promise) {
    return abortablePromise(p, signal);
  } else {
    return abortableAsyncIterable(p, signal);
  }
}
/**
 * Make Promise abortable with the given signal.
 *
 * @example
 * ```ts
 * import { abortablePromise } from "https://deno.land/std@$STD_VERSION/async/abortable.ts";
 *
 * const request = fetch("https://example.com");
 *
 * const c = new AbortController();
 * setTimeout(() => c.abort(), 100);
 *
 * const p = abortablePromise(request, c.signal);
 *
 * // The below throws if the request didn't resolve in 100ms
 * await p;
 * ```
 */ export function abortablePromise(p, signal) {
  if (signal.aborted) {
    return Promise.reject(createAbortError(signal.reason));
  }
  const { promise, reject } = Promise.withResolvers();
  const abort = ()=>reject(createAbortError(signal.reason));
  signal.addEventListener("abort", abort, {
    once: true
  });
  return Promise.race([
    promise,
    p
  ]).finally(()=>{
    signal.removeEventListener("abort", abort);
  });
}
/**
 * Make AsyncIterable abortable with the given signal.
 *
 * @example
 * ```ts
 * import {
 *   abortableAsyncIterable,
 *   delay,
 * } from "https://deno.land/std@$STD_VERSION/async/mod.ts";
 *
 * const p = async function* () {
 *   yield "Hello";
 *   await delay(1000);
 *   yield "World";
 * };
 * const c = new AbortController();
 * setTimeout(() => c.abort(), 100);
 *
 * // Below throws `DOMException` after 100 ms
 * // and items become `["Hello"]`
 * const items: string[] = [];
 * for await (const item of abortableAsyncIterable(p(), c.signal)) {
 *   items.push(item);
 * }
 * ```
 */ export async function* abortableAsyncIterable(p, signal) {
  if (signal.aborted) {
    throw createAbortError(signal.reason);
  }
  const { promise, reject } = Promise.withResolvers();
  const abort = ()=>reject(createAbortError(signal.reason));
  signal.addEventListener("abort", abort, {
    once: true
  });
  const it = p[Symbol.asyncIterator]();
  while(true){
    const race = Promise.race([
      promise,
      it.next()
    ]);
    race.catch(()=>{
      signal.removeEventListener("abort", abort);
    });
    const { done, value } = await race;
    if (done) {
      signal.removeEventListener("abort", abort);
      return;
    }
    yield value;
  }
}
// This `reason` comes from `AbortSignal` thus must be `any`.
// deno-lint-ignore no-explicit-any
function createAbortError(reason) {
  return new DOMException(reason ? `Aborted: ${reason}` : "Aborted", "AbortError");
}
//# sourceMappingURL=data:application/json;base64,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