// Copyright 2018-2024 the Deno authors. All rights reserved. MIT license.
// This module is browser compatible.
/**
 * Provides the {@linkcode KeyStack} class which implements the
 * {@linkcode KeyRing} interface for managing rotatable keys.
 *
 * @module
 */ var _computedKey, _computedKey1;
import { timingSafeEqual } from "./timing_safe_equal.ts";
import { encodeBase64Url } from "../encoding/base64url.ts";
const encoder = new TextEncoder();
function importKey(key) {
  if (typeof key === "string") {
    key = encoder.encode(key);
  } else if (Array.isArray(key)) {
    key = new Uint8Array(key);
  }
  return crypto.subtle.importKey("raw", key, {
    name: "HMAC",
    hash: {
      name: "SHA-256"
    }
  }, true, [
    "sign",
    "verify"
  ]);
}
function sign(data, key) {
  if (typeof data === "string") {
    data = encoder.encode(data);
  } else if (Array.isArray(data)) {
    data = Uint8Array.from(data);
  }
  return crypto.subtle.sign("HMAC", key, data);
}
/**
 * Compare two strings, Uint8Arrays, ArrayBuffers, or arrays of numbers in a
 * way that avoids timing based attacks on the comparisons on the values.
 *
 * The function will return `true` if the values match, or `false`, if they
 * do not match.
 *
 * This was inspired by https://github.com/suryagh/tsscmp which provides a
 * timing safe string comparison to avoid timing attacks as described in
 * https://codahale.com/a-lesson-in-timing-attacks/.
 */ async function compare(a, b) {
  const key = new Uint8Array(32);
  globalThis.crypto.getRandomValues(key);
  const cryptoKey = await importKey(key);
  const [ah, bh] = await Promise.all([
    sign(a, cryptoKey),
    sign(b, cryptoKey)
  ]);
  return timingSafeEqual(ah, bh);
}
_computedKey = Symbol.for("Deno.customInspect"), _computedKey1 = Symbol.for("nodejs.util.inspect.custom");
/**
 * A cryptographic key chain which allows signing of data to prevent tampering,
 * but also allows for easy key rotation without needing to re-sign the data.
 *
 * Data is signed as SHA256 HMAC.
 *
 * This was inspired by {@link https://github.com/crypto-utils/keygrip/ | keygrip}.
 *
 * @example
 * ```ts
 * import { KeyStack } from "https://deno.land/std@$STD_VERSION/crypto/unstable_keystack.ts";
 *
 * const keyStack = new KeyStack(["hello", "world"]);
 * const digest = await keyStack.sign("some data");
 *
 * const rotatedStack = new KeyStack(["deno", "says", "hello", "world"]);
 * await rotatedStack.verify("some data", digest); // true
 * ```
 */ export class KeyStack {
  #cryptoKeys = new Map();
  #keys;
  async #toCryptoKey(key) {
    if (!this.#cryptoKeys.has(key)) {
      this.#cryptoKeys.set(key, await importKey(key));
    }
    return this.#cryptoKeys.get(key);
  }
  /** Number of keys */ get length() {
    return this.#keys.length;
  }
  /**
   * A class which accepts an array of keys that are used to sign and verify
   * data and allows easy key rotation without invalidation of previously signed
   * data.
   *
   * @param keys An iterable of keys, of which the index 0 will be used to sign
   *             data, but verification can happen against any key.
   */ constructor(keys){
    const values = Array.isArray(keys) ? keys : [
      ...keys
    ];
    if (!values.length) {
      throw new TypeError("keys must contain at least one value");
    }
    this.#keys = values;
  }
  /**
   * Take `data` and return a SHA256 HMAC digest that uses the current 0 index
   * of the `keys` passed to the constructor.  This digest is in the form of a
   * URL safe base64 encoded string.
   */ async sign(data) {
    const key = await this.#toCryptoKey(this.#keys[0]);
    return encodeBase64Url(await sign(data, key));
  }
  /**
   * Given `data` and a `digest`, verify that one of the `keys` provided the
   * constructor was used to generate the `digest`.  Returns `true` if one of
   * the keys was used, otherwise `false`.
   */ async verify(data, digest) {
    return await this.indexOf(data, digest) > -1;
  }
  /**
   * Given `data` and a `digest`, return the current index of the key in the
   * `keys` passed the constructor that was used to generate the digest.  If no
   * key can be found, the method returns `-1`.
   */ async indexOf(data, digest) {
    for(let i = 0; i < this.#keys.length; i++){
      const key = this.#keys[i];
      const cryptoKey = await this.#toCryptoKey(key);
      if (await compare(digest, encodeBase64Url(await sign(data, cryptoKey)))) {
        return i;
      }
    }
    return -1;
  }
  /** Custom output for {@linkcode Deno.inspect}. */ [_computedKey](inspect) {
    const { length } = this;
    return `${this.constructor.name} ${inspect({
      length
    })}`;
  }
  /** Custom output for Node's {@linkcode https://nodejs.org/api/util.html#utilinspectobject-options|util.inspect}. */ [_computedKey1](depth, // deno-lint-ignore no-explicit-any
  options, inspect) {
    if (depth < 0) {
      return options.stylize(`[${this.constructor.name}]`, "special");
    }
    const newOptions = Object.assign({}, options, {
      depth: options.depth === null ? null : options.depth - 1
    });
    const { length } = this;
    return `${options.stylize(this.constructor.name, "special")} ${inspect({
      length
    }, newOptions)}`;
  }
}
//# sourceMappingURL=data:application/json;base64,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